// Copyright (C) 2019 The Qt Company Ltd.
// Copyright © 2004-2023 Unicode, Inc.
// SPDX-License-Identifier: Unicode-3.0

#ifndef QLOCALE_DATA_P_H
#define QLOCALE_DATA_P_H

//
//  W A R N I N G
//  -------------
//
// This file is not part of the Qt API.  It exists for the convenience
// of qapplication_*.cpp, qwidget*.cpp and qfiledialog.cpp.  This header
// file may change from version to version without notice, or even be removed.
//
// We mean it.
//

#include <QtCore/qendian.h>
#include <QtCore/private/qlocale_p.h>

#include <array>
#include <cstring> // std::memcmp

QT_BEGIN_NAMESPACE

/* This part of the file isn't generated, but written by hand. Unicode CLDR's
   information about measurement systems doesn't say which to use by default in
   each locale. Even if it did, adding another entry in every locale's row of
   locale_data[] would take up much more memory than the small table below.
*/
struct TerritoryLanguage
{
    quint16 languageId;
    quint16 territoryId;
    QLocale::MeasurementSystem system;
};

QT_WARNING_PUSH // QTBUG-128930
QT_WARNING_DISABLE_CLANG("-Wunused-const-variable")
static constexpr TerritoryLanguage ImperialMeasurementSystems[] = {
    { QLocale::English, QLocale::UnitedStates, QLocale::ImperialUSSystem },
    { QLocale::English, QLocale::UnitedStatesMinorOutlyingIslands, QLocale::ImperialUSSystem },
    { QLocale::Spanish, QLocale::UnitedStates, QLocale::ImperialUSSystem },
    { QLocale::Hawaiian, QLocale::UnitedStates, QLocale::ImperialUSSystem },
    { QLocale::English, QLocale::UnitedKingdom, QLocale::ImperialUKSystem }
};
QT_WARNING_POP // QTBUG-128930

/*
  Storage for alpha codes with length of up to 4 allowing efficient comparison.
*/
struct AlphaCode {
    constexpr AlphaCode(char c1 = 0, char c2 = 0, char c3 = 0)
        : c1(c2m(c1)), c2(c2m(c2)), c3(c2m(c3)), reserved(0) { }

    uint16_t c1: 5, c2:5, c3: 5, reserved:1;

    bool isValid() const noexcept { return c1 != 0; }

    std::array<char, 4> decode() const { return {m2c(c1), m2c(c2), m2c(c3), 0}; }

private:
    static constexpr uint16_t c2m(char c) { return c ? c - 'a' + 1 : 0; }
    static constexpr char m2c (uint16_t c) { return c ? c + 'a' - 1 : 0; }

    friend bool operator==(AlphaCode lhs, AlphaCode rhs) noexcept
    {
        static_assert(std::has_unique_object_representations_v<AlphaCode>,
            "memcmp() cannot be used to implement equality on "
            "types that don't have unique object representations");
        return std::memcmp(&lhs, &rhs, sizeof(lhs)) == 0;
    }
};

struct LanguageCodeEntry {
    AlphaCode part1;
    AlphaCode part2B;
    AlphaCode part2T;
    AlphaCode part3;
};

QT_WARNING_PUSH // QTBUG-128930
QT_WARNING_DISABLE_CLANG("-Wunused-const-variable")

// GENERATED PART STARTS HERE

/*
    This part of the file was generated on 2024-05-31 from the
    Common Locale Data Repository v45

    http://www.unicode.org/cldr/

    Do not edit this section: instead regenerate it using
    cldr2qlocalexml.py and qlocalexml2cpp.py on updated (or
    edited) CLDR data; see qtbase/util/locale_database/.
*/

static constexpr QLocaleId likely_subtags[] = {
    {   2,   0,   0 }, {   2,  27,  90 }, // ab -> ab_Cyrl_GE
    {   3,   0,   0 }, {   3,  66,  77 }, // aa -> aa_Latn_ET
    {   4,   0,   0 }, {   4,  66, 216 }, // af -> af_Latn_ZA
    {   5,   0,   0 }, {   5,  66,  40 }, // agq -> agq_Latn_CM
    {   6,   0,   0 }, {   6,  66,  92 }, // ak -> ak_Latn_GH
    {   7,   0,   0 }, {   7,  25, 113 }, // akk -> akk_Xsux_IQ
    {   8,   0,   0 }, {   8,  66,  40 }, // bss -> bss_Latn_CM
    {   9,   0,   0 }, {   9,  66,   3 }, // sq -> sq_Latn_AL
    {  10,   0,   0 }, {  10, 117, 248 }, // ase -> ase_Sgnw_US
    {  11,   0,   0 }, {  11,  33,  77 }, // am -> am_Ethi_ET
    {  12,   0,   0 }, {  12,  31,  71 }, // egy -> egy_Egyp_EG
    {  13,  70,   0 }, {  13,  70,  94 }, // grc_Linb -> grc_Linb_GR
    {  13,   0,   0 }, {  13,  26,  63 }, // grc -> grc_Cprt_CY
    {  14,   0,   0 }, {  14,   4,  71 }, // ar -> ar_Arab_EG
    {  15,   0,   0 }, {  15,  66, 220 }, // an -> an_Latn_ES
    {  16,  87,   0 }, {  16,  87, 122 }, // arc_Nbat -> arc_Nbat_JO
    {  16, 104,   0 }, {  16, 104, 227 }, // arc_Palm -> arc_Palm_SY
    {  16,   0,   0 }, {  16,  49, 112 }, // arc -> arc_Armi_IR
    {  17,   0,   0 }, {  17,   5,  12 }, // hy -> hy_Armn_AM
    {  18,   0,   0 }, {  18,   9, 110 }, // as -> as_Beng_IN
    {  19,   0,   0 }, {  19,  66, 220 }, // ast -> ast_Latn_ES
    {  20,   0,   0 }, {  20,  66, 230 }, // asa -> asa_Latn_TZ
    {  21,   0,   0 }, {  21,  66, 169 }, // cch -> cch_Latn_NG
    {  22,   0,   0 }, {  22,  27, 193 }, // av -> av_Cyrl_RU
    {  23,   0,   0 }, {  23,   6, 112 }, // ae -> ae_Avst_IR
    {  24,   0,   0 }, {  24,  66,  28 }, // ay -> ay_Latn_BO
    {  25,   0, 112 }, {  25,   4, 112 }, // az_IR -> az_Arab_IR
    {  25,   0, 113 }, {  25,   4, 113 }, // az_IQ -> az_Arab_IQ
    {  25,   0, 193 }, {  25,  27, 193 }, // az_RU -> az_Cyrl_RU
    {  25,   4,   0 }, {  25,   4, 112 }, // az_Arab -> az_Arab_IR
    {  25,   0,   0 }, {  25,  66,  17 }, // az -> az_Latn_AZ
    {  26,   0,   0 }, {  26,  66,  40 }, // ksf -> ksf_Latn_CM
    {  27,   0,   0 }, {  27,  66, 111 }, // ban -> ban_Latn_ID
    {  28,   0,   0 }, {  28,  66, 145 }, // bm -> bm_Latn_ML
    {  29,   0,   0 }, {  29,   8,  40 }, // bax -> bax_Bamu_CM
    {  30,   0,   0 }, {  30,   9,  20 }, // bn -> bn_Beng_BD
    {  31,   0,   0 }, {  31,  66,  40 }, // bas -> bas_Latn_CM
    {  32,   0,   0 }, {  32,  27, 193 }, // ba -> ba_Cyrl_RU
    {  33,   0,   0 }, {  33,  66, 220 }, // eu -> eu_Latn_ES
    {  34,   0,   0 }, {  34,  66, 111 }, // bbc -> bbc_Latn_ID
    {  35,   0,   0 }, {  35,  27,  22 }, // be -> be_Cyrl_BY
    {  36,   0,   0 }, {  36,  66, 260 }, // bem -> bem_Latn_ZM
    {  37,   0,   0 }, {  37,  66, 230 }, // bez -> bez_Latn_TZ
    {  38,   0,   0 }, {  38,  29, 110 }, // bho -> bho_Deva_IN
    {  39,   0,   0 }, {  39,  66, 252 }, // bi -> bi_Latn_VU
    {  40,   0,   0 }, {  40,  33,  74 }, // byn -> byn_Ethi_ER
    {  41,   0,   0 }, {  41,  29, 110 }, // brx -> brx_Deva_IN
    {  42,   0,   0 }, {  42,  66,  29 }, // bs -> bs_Latn_BA
    {  43,   0,   0 }, {  43,  66,  84 }, // br -> br_Latn_FR
    {  44,   0,   0 }, {  44,  66, 111 }, // bug -> bug_Latn_ID
    {  45,   0,   0 }, {  45,  27,  36 }, // bg -> bg_Cyrl_BG
    {  46,   0,   0 }, {  46,  86, 161 }, // my -> my_Mymr_MM
    {  47,   0,  50 }, {  47, 118,  50 }, // yue_CN -> yue_Hans_CN
    {  47, 118,   0 }, {  47, 118,  50 }, // yue_Hans -> yue_Hans_CN
    {  47,   0,   0 }, {  47, 137, 107 }, // yue -> yue_Hant_HK
    {  48,   0,   0 }, {  48,  66, 220 }, // ca -> ca_Latn_ES
    {  49,   0,   0 }, {  49,  66, 185 }, // ceb -> ceb_Latn_PH
    {  50,   0,   0 }, {  50,  66, 159 }, // tzm -> tzm_Latn_MA
    {  51,   0,   0 }, {  51,   4, 113 }, // ckb -> ckb_Arab_IQ
    {  52,   0,   0 }, {  52,  21,  20 }, // ccp -> ccp_Cakm_BD
    {  53,   0,   0 }, {  53,  66,  98 }, // ch -> ch_Latn_GU
    {  54,   0,   0 }, {  54,  27, 193 }, // ce -> ce_Cyrl_RU
    {  55,   0,   0 }, {  55,  23, 248 }, // chr -> chr_Cher_US
    {  56,   0,   0 }, {  56,  66, 248 }, // cic -> cic_Latn_US
    {  57,   0,   0 }, {  57,  66, 243 }, // cgg -> cgg_Latn_UG
    {  58,   0,  15 }, {  58, 137,  15 }, // zh_AU -> zh_Hant_AU
    {  58,   0,  35 }, {  58, 137,  35 }, // zh_BN -> zh_Hant_BN
    {  58,   0,  85 }, {  58, 137,  85 }, // zh_GF -> zh_Hant_GF
    {  58,   0,  86 }, {  58, 137,  86 }, // zh_PF -> zh_Hant_PF
    {  58,   0, 107 }, {  58, 137, 107 }, // zh_HK -> zh_Hant_HK
    {  58,   0, 111 }, {  58, 137, 111 }, // zh_ID -> zh_Hant_ID
    {  58,   0, 139 }, {  58, 137, 139 }, // zh_MO -> zh_Hant_MO
    {  58,   0, 181 }, {  58, 137, 181 }, // zh_PA -> zh_Hant_PA
    {  58,   0, 185 }, {  58, 137, 185 }, // zh_PH -> zh_Hant_PH
    {  58,   0, 223 }, {  58, 137, 223 }, // zh_SR -> zh_Hant_SR
    {  58,   0, 228 }, {  58, 137, 228 }, // zh_TW -> zh_Hant_TW
    {  58,   0, 231 }, {  58, 137, 231 }, // zh_TH -> zh_Hant_TH
    {  58,   0, 246 }, {  58, 137, 246 }, // zh_GB -> zh_Hant_GB
    {  58,   0, 248 }, {  58, 137, 248 }, // zh_US -> zh_Hant_US
    {  58,   0, 255 }, {  58, 137, 255 }, // zh_VN -> zh_Hant_VN
    {  58,  13,   0 }, {  58,  13, 228 }, // zh_Bopo -> zh_Bopo_TW
    {  58,  45,   0 }, {  58,  45, 228 }, // zh_Hanb -> zh_Hanb_TW
    {  58, 137,   0 }, {  58, 137, 228 }, // zh_Hant -> zh_Hant_TW
    {  58,   0,   0 }, {  58, 118,  50 }, // zh -> zh_Hans_CN
    {  59,  36,   0 }, {  59,  36,  36 }, // cu_Glag -> cu_Glag_BG
    {  59,   0,   0 }, {  59,  27, 193 }, // cu -> cu_Cyrl_RU
    {  60,   0,   0 }, {  60,  27, 193 }, // cv -> cv_Cyrl_RU
    {  61,   0,   0 }, {  61,  66,  91 }, // ksh -> ksh_Latn_DE
    {  62,   0,   0 }, {  62,  24,  71 }, // cop -> cop_Copt_EG
    {  63,   0,   0 }, {  63,  66, 246 }, // kw -> kw_Latn_GB
    {  64,   0,   0 }, {  64,  66,  84 }, // co -> co_Latn_FR
    {  65,   0,   0 }, {  65,  18,  41 }, // cr -> cr_Cans_CA
    {  66,   0,   0 }, {  66,  66,  60 }, // hr -> hr_Latn_HR
    {  67,   0,   0 }, {  67,  66,  64 }, // cs -> cs_Latn_CZ
    {  68,   0,   0 }, {  68,  66,  65 }, // da -> da_Latn_DK
    {  69,   0,   0 }, {  69, 132, 144 }, // dv -> dv_Thaa_MV
    {  70,   0,   0 }, {  70,  29, 110 }, // doi -> doi_Deva_IN
    {  71,   0,   0 }, {  71,  66,  40 }, // dua -> dua_Latn_CM
    {  72,   0,   0 }, {  72,  66, 165 }, // nl -> nl_Latn_NL
    {  73,   0,   0 }, {  73, 134,  27 }, // dz -> dz_Tibt_BT
    {  74,   0,   0 }, {  74,  66, 124 }, // ebu -> ebu_Latn_KE
    {  75, 115,   0 }, {  75, 115, 246 }, // en_Shaw -> en_Shaw_GB
    {  75,   0,   0 }, {  75,  66, 248 }, // en -> en_Latn_US
    {  76,   0,   0 }, {  76,  27, 193 }, // myv -> myv_Cyrl_RU
    {  77,   0,   0 }, {  77,  66, 258 }, // eo -> eo_Latn_001
    {  78,   0,   0 }, {  78,  66,  75 }, // et -> et_Latn_EE
    {  79,   0,   0 }, {  79,  66,  92 }, // ee -> ee_Latn_GH
    {  80,   0,   0 }, {  80,  66,  40 }, // ewo -> ewo_Latn_CM
    {  81,   0,   0 }, {  81,  66,  81 }, // fo -> fo_Latn_FO
    {  82,   0,   0 }, {  82,  66,  82 }, // fj -> fj_Latn_FJ
    {  83,   0,   0 }, {  83,  66, 185 }, // fil -> fil_Latn_PH
    {  84,   0,   0 }, {  84,  66,  83 }, // fi -> fi_Latn_FI
    {  85,   0,   0 }, {  85,  66,  84 }, // fr -> fr_Latn_FR
    {  86,   0,   0 }, {  86,  66, 117 }, // fur -> fur_Latn_IT
    {  87,   1,   0 }, {  87,   1, 102 }, // ff_Adlm -> ff_Adlm_GN
    {  87,   0,   0 }, {  87,  66, 206 }, // ff -> ff_Latn_SN
    {  88,   0,   0 }, {  88,  66, 246 }, // gd -> gd_Latn_GB
    {  89,   0,   0 }, {  89,  66,  92 }, // gaa -> gaa_Latn_GH
    {  90,   0,   0 }, {  90,  66, 220 }, // gl -> gl_Latn_ES
    {  91,   0,   0 }, {  91,  66, 243 }, // lg -> lg_Latn_UG
    {  92,   0,   0 }, {  92,  33,  77 }, // gez -> gez_Ethi_ET
    {  93,   0,   0 }, {  93,  35,  90 }, // ka -> ka_Geor_GE
    {  94,   0,   0 }, {  94,  66,  91 }, // de -> de_Latn_DE
    {  95,   0,   0 }, {  95,  37, 244 }, // got -> got_Goth_UA
    {  96,   0,   0 }, {  96,  39,  94 }, // el -> el_Grek_GR
    {  97,   0,   0 }, {  97,  66, 183 }, // gn -> gn_Latn_PY
    {  98,   0,   0 }, {  98,  40, 110 }, // gu -> gu_Gujr_IN
    {  99,   0,   0 }, {  99,  66, 124 }, // guz -> guz_Latn_KE
    { 100,   0,   0 }, { 100,  66, 104 }, // ht -> ht_Latn_HT
    { 101,   0,  40 }, { 101,   4,  40 }, // ha_CM -> ha_Arab_CM
    { 101,   0, 222 }, { 101,   4, 222 }, // ha_SD -> ha_Arab_SD
    { 101,   0,   0 }, { 101,  66, 169 }, // ha -> ha_Latn_NG
    { 102,   0,   0 }, { 102,  66, 248 }, // haw -> haw_Latn_US
    { 103,   0,   0 }, { 103,  47, 116 }, // he -> he_Hebr_IL
    { 104,   0,   0 }, { 104,  66, 162 }, // hz -> hz_Latn_NA
    { 105,   0,   0 }, { 105,  29, 110 }, // hi -> hi_Deva_IN
    { 106,   0,   0 }, { 106,  66, 182 }, // ho -> ho_Latn_PG
    { 107,   0,   0 }, { 107,  66, 108 }, // hu -> hu_Latn_HU
    { 108,   0,   0 }, { 108,  66, 109 }, // is -> is_Latn_IS
    { 109,   0,   0 }, { 109,  66, 258 }, // io -> io_Latn_001
    { 110,   0,   0 }, { 110,  66, 169 }, // ig -> ig_Latn_NG
    { 111,   0,   0 }, { 111,  66,  83 }, // smn -> smn_Latn_FI
    { 112,   0,   0 }, { 112,  66, 111 }, // id -> id_Latn_ID
    { 113,   0,   0 }, { 113,  27, 193 }, // inh -> inh_Cyrl_RU
    { 114,   0,   0 }, { 114,  66, 258 }, // ia -> ia_Latn_001
    { 115,   0,   0 }, { 115,  66,  75 }, // ie -> ie_Latn_EE
    { 116,   0,   0 }, { 116,  18,  41 }, // iu -> iu_Cans_CA
    { 117,   0,   0 }, { 117,  66, 248 }, // ik -> ik_Latn_US
    { 118,   0,   0 }, { 118,  66, 114 }, // ga -> ga_Latn_IE
    { 119,   0,   0 }, { 119,  66, 117 }, // it -> it_Latn_IT
    { 120,   0,   0 }, { 120,  53, 120 }, // ja -> ja_Jpan_JP
    { 121,   0,   0 }, { 121,  66, 111 }, // jv -> jv_Latn_ID
    { 122,   0,   0 }, { 122,  66, 169 }, // kaj -> kaj_Latn_NG
    { 123,   0,   0 }, { 123,  66, 206 }, // dyo -> dyo_Latn_SN
    { 124,   0,   0 }, { 124,  66,  43 }, // kea -> kea_Latn_CV
    { 125,   0,   0 }, { 125,  66,   4 }, // kab -> kab_Latn_DZ
    { 126,   0,   0 }, { 126,  66,  40 }, // kkj -> kkj_Latn_CM
    { 127,   0,   0 }, { 127,  66,  95 }, // kl -> kl_Latn_GL
    { 128,   0,   0 }, { 128,  66, 124 }, // kln -> kln_Latn_KE
    { 129,   0,   0 }, { 129,  66, 124 }, // kam -> kam_Latn_KE
    { 130,   0,   0 }, { 130,  56, 110 }, // kn -> kn_Knda_IN
    { 131,   4,   0 }, { 131,   4, 169 }, // kr_Arab -> kr_Arab_NG
    { 131,   0,   0 }, { 131,  66, 169 }, // kr -> kr_Latn_NG
    { 132,   0,   0 }, { 132,   4, 110 }, // ks -> ks_Arab_IN
    { 133,   0,   1 }, { 133,   4,   1 }, // kk_AF -> kk_Arab_AF
    { 133,   0,  50 }, { 133,   4,  50 }, // kk_CN -> kk_Arab_CN
    { 133,   0, 112 }, { 133,   4, 112 }, // kk_IR -> kk_Arab_IR
    { 133,   0, 156 }, { 133,   4, 156 }, // kk_MN -> kk_Arab_MN
    { 133,   4,   0 }, { 133,   4,  50 }, // kk_Arab -> kk_Arab_CN
    { 133,   0,   0 }, { 133,  27, 123 }, // kk -> kk_Cyrl_KZ
    { 134,   0,   0 }, { 134,  66,  40 }, // ken -> ken_Latn_CM
    { 135,   0,   0 }, { 135,  60,  39 }, // km -> km_Khmr_KH
    { 136,   0,   0 }, { 136,  66,  99 }, // quc -> quc_Latn_GT
    { 137,   0,   0 }, { 137,  66, 124 }, // ki -> ki_Latn_KE
    { 138,   0,   0 }, { 138,  66, 194 }, // rw -> rw_Latn_RW
    { 139,   0,   0 }, { 139,  27, 193 }, // kv -> kv_Cyrl_RU
    { 140,   0,   0 }, { 140,  66,  57 }, // kg -> kg_Latn_CD
    { 141,   0,   0 }, { 141,  29, 110 }, // kok -> kok_Deva_IN
    { 142,   0,   0 }, { 142,  63, 218 }, // ko -> ko_Kore_KR
    { 143,   0,   0 }, { 143,  66, 118 }, // kfo -> kfo_Latn_CI
    { 144,   0,   0 }, { 144,  66, 145 }, // ses -> ses_Latn_ML
    { 145,   0,   0 }, { 145,  66, 145 }, // khq -> khq_Latn_ML
    { 146,   0,   0 }, { 146,  66, 134 }, // kpe -> kpe_Latn_LR
    { 147,   0,   0 }, { 147,  66, 162 }, // kj -> kj_Latn_NA
    { 148,   0, 132 }, { 148,   4, 132 }, // ku_LB -> ku_Arab_LB
    { 148,   4,   0 }, { 148,   4, 113 }, // ku_Arab -> ku_Arab_IQ
    { 148,   0,   0 }, { 148,  66, 239 }, // ku -> ku_Latn_TR
    { 149,   0,   0 }, { 149,  66,  40 }, // nmg -> nmg_Latn_CM
    { 150,   0,  50 }, { 150,   4,  50 }, // ky_CN -> ky_Arab_CN
    { 150,   0, 239 }, { 150,  66, 239 }, // ky_TR -> ky_Latn_TR
    { 150,   4,   0 }, { 150,   4,  50 }, // ky_Arab -> ky_Arab_CN
    { 150,  66,   0 }, { 150,  66, 239 }, // ky_Latn -> ky_Latn_TR
    { 150,   0,   0 }, { 150,  27, 128 }, // ky -> ky_Cyrl_KG
    { 151,   0,   0 }, { 151,  66, 248 }, // lkt -> lkt_Latn_US
    { 152,   0,   0 }, { 152,  66, 230 }, // lag -> lag_Latn_TZ
    { 153,   0,   0 }, { 153,  65, 129 }, // lo -> lo_Laoo_LA
    { 154,   0,   0 }, { 154,  66, 253 }, // la -> la_Latn_VA
    { 155,   0,   0 }, { 155,  66, 131 }, // lv -> lv_Latn_LV
    { 156,   0,   0 }, { 156,  27, 193 }, // lez -> lez_Cyrl_RU
    { 157,   0,   0 }, { 157,  66, 165 }, // li -> li_Latn_NL
    { 158,   0,   0 }, { 158,  66,  57 }, // ln -> ln_Latn_CD
    { 159,   0,   0 }, { 159, 118,  50 }, // lzh -> lzh_Hans_CN
    { 160,   0,   0 }, { 160,  66, 137 }, // lt -> lt_Latn_LT
    { 161,   0,   0 }, { 161,  66, 258 }, // jbo -> jbo_Latn_001
    { 162,   0,   0 }, { 162,  66,  91 }, // dsb -> dsb_Latn_DE
    { 163,   0,   0 }, { 163,  66,  91 }, // nds -> nds_Latn_DE
    { 164,   0,   0 }, { 164,  66,  57 }, // lu -> lu_Latn_CD
    { 165,   0,   0 }, { 165,  66, 225 }, // smj -> smj_Latn_SE
    { 166,   0,   0 }, { 166,  66, 124 }, // luo -> luo_Latn_KE
    { 167,   0,   0 }, { 167,  66, 138 }, // lb -> lb_Latn_LU
    { 168,   0,   0 }, { 168,  66, 124 }, // luy -> luy_Latn_KE
    { 169,   0,   0 }, { 169,  27, 140 }, // mk -> mk_Cyrl_MK
    { 170,   0,   0 }, { 170,  66, 230 }, // jmc -> jmc_Latn_TZ
    { 171,   0,   0 }, { 171,  29, 110 }, // mai -> mai_Deva_IN
    { 172,   0,   0 }, { 172,  66, 160 }, // mgh -> mgh_Latn_MZ
    { 173,   0,   0 }, { 173,  66, 230 }, // kde -> kde_Latn_TZ
    { 174,   0,   0 }, { 174,  66, 141 }, // mg -> mg_Latn_MG
    { 175,   0,   0 }, { 175,  74, 110 }, // ml -> ml_Mlym_IN
    { 176,   0,  53 }, { 176,   4,  53 }, // ms_CC -> ms_Arab_CC
    { 176,   0,   0 }, { 176,  66, 143 }, // ms -> ms_Latn_MY
    { 177,   0,   0 }, { 177,  66, 146 }, // mt -> mt_Latn_MT
    { 178,   0, 102 }, { 178,  90, 102 }, // man_GN -> man_Nkoo_GN
    { 178,  90,   0 }, { 178,  90, 102 }, // man_Nkoo -> man_Nkoo_GN
    { 178,   0,   0 }, { 178,  66,  89 }, // man -> man_Latn_GM
    { 179,   0,   0 }, { 179,   9, 110 }, // mni -> mni_Beng_IN
    { 180,   0,   0 }, { 180,  66, 115 }, // gv -> gv_Latn_IM
    { 181,   0,   0 }, { 181,  66, 167 }, // mi -> mi_Latn_NZ
    { 182,   0,   0 }, { 182,  66,  49 }, // arn -> arn_Latn_CL
    { 183,   0,   0 }, { 183,  29, 110 }, // mr -> mr_Deva_IN
    { 184,   0,   0 }, { 184,  66, 147 }, // mh -> mh_Latn_MH
    { 185,   0,   0 }, { 185,  66, 124 }, // mas -> mas_Latn_KE
    { 186,   0,   0 }, { 186,   4, 112 }, // mzn -> mzn_Arab_IR
    { 187,   0,   0 }, { 187,  66, 209 }, // men -> men_Latn_SL
    { 188,   0,   0 }, { 188,  66, 124 }, // mer -> mer_Latn_KE
    { 189,   0,   0 }, { 189,  66,  40 }, // mgo -> mgo_Latn_CM
    { 190,   0,   0 }, { 190,  66,  41 }, // moh -> moh_Latn_CA
    { 191,   0,  50 }, { 191,  83,  50 }, // mn_CN -> mn_Mong_CN
    { 191,  83,   0 }, { 191,  83,  50 }, // mn_Mong -> mn_Mong_CN
    { 191,   0,   0 }, { 191,  27, 156 }, // mn -> mn_Cyrl_MN
    { 192,   0,   0 }, { 192,  66, 150 }, // mfe -> mfe_Latn_MU
    { 193,   0,   0 }, { 193,  66,  40 }, // mua -> mua_Latn_CM
    { 194,   0,   0 }, { 194,  66, 248 }, // mus -> mus_Latn_US
    { 195,   0,   0 }, { 195,  66, 162 }, // naq -> naq_Latn_NA
    { 196,   0,   0 }, { 196,  66, 163 }, // na -> na_Latn_NR
    { 197,   0,   0 }, { 197,  66, 248 }, // nv -> nv_Latn_US
    { 198,   0,   0 }, { 198,  66, 162 }, // ng -> ng_Latn_NA
    { 199,   0,   0 }, { 199,  29, 164 }, // ne -> ne_Deva_NP
    { 200,   0,   0 }, { 200,  29, 164 }, // new -> new_Deva_NP
    { 201,   0,   0 }, { 201,  66,  40 }, // nnh -> nnh_Latn_CM
    { 202,   0,   0 }, { 202,  66,  40 }, // jgo -> jgo_Latn_CM
    { 203,   0,   0 }, { 203,  66, 169 }, // pcm -> pcm_Latn_NG
    { 204,   0,   0 }, { 204,  90, 102 }, // nqo -> nqo_Nkoo_GN
    { 205,   0,   0 }, { 205,   4, 112 }, // lrc -> lrc_Arab_IR
    { 206,   0,   0 }, { 206,  66, 175 }, // se -> se_Latn_NO
    { 207,   0,   0 }, { 207,  66, 216 }, // nso -> nso_Latn_ZA
    { 208,   0,   0 }, { 208,  66, 261 }, // nd -> nd_Latn_ZW
    { 209,   0,   0 }, { 209,  66, 175 }, // nb -> nb_Latn_NO
    { 210,   0,   0 }, { 210,  66, 175 }, // nn -> nn_Latn_NO
    { 211,   0,   0 }, { 211,  66, 219 }, // nus -> nus_Latn_SS
    { 212,   0,   0 }, { 212,  66, 142 }, // ny -> ny_Latn_MW
    { 213,   0,   0 }, { 213,  66, 243 }, // nyn -> nyn_Latn_UG
    { 214,   0,   0 }, { 214,  66,  84 }, // oc -> oc_Latn_FR
    { 215,   0,   0 }, { 215,  91, 110 }, // or -> or_Orya_IN
    { 216,   0,   0 }, { 216,  18,  41 }, // oj -> oj_Cans_CA
    { 217,   0,   0 }, { 217,  92, 114 }, // sga -> sga_Ogam_IE
    { 218,   0,   0 }, { 218, 111, 225 }, // non -> non_Runr_SE
    { 219,   0,   0 }, { 219,  98, 112 }, // peo -> peo_Xpeo_IR
    { 220,   0,   0 }, { 220,  66,  77 }, // om -> om_Latn_ET
    { 221,   0,   0 }, { 221, 101, 248 }, // osa -> osa_Osge_US
    { 222,   0,   0 }, { 222,  27,  90 }, // os -> os_Cyrl_GE
    { 223, 109,   0 }, { 223, 109,  50 }, // pal_Phlp -> pal_Phlp_CN
    { 223,   0,   0 }, { 223,  50, 112 }, // pal -> pal_Phli_IR
    { 224,   0,   0 }, { 224,  66, 179 }, // pau -> pau_Latn_PW
    { 225,  14,   0 }, { 225,  14, 110 }, // pi_Brah -> pi_Brah_IN
    { 225,  29,   0 }, { 225,  29, 110 }, // pi_Deva -> pi_Deva_IN
    { 225,  59,   0 }, { 225,  59, 110 }, // pi_Khar -> pi_Khar_IN
    { 225,  60,   0 }, { 225,  60, 110 }, // pi_Khmr -> pi_Khmr_IN
    { 225,  86,   0 }, { 225,  86, 110 }, // pi_Mymr -> pi_Mymr_IN
    { 225, 133,   0 }, { 225, 133, 110 }, // pi_Thai -> pi_Thai_IN
    { 225,   0,   0 }, { 225, 119, 110 }, // pi -> pi_Sinh_IN
    { 226,   0,   0 }, { 226,  66,  62 }, // pap -> pap_Latn_CW
    { 227,   0,   0 }, { 227,   4,   1 }, // ps -> ps_Arab_AF
    { 228,   0,   0 }, { 228,   4, 112 }, // fa -> fa_Arab_IR
    { 229,   0,   0 }, { 229, 107, 132 }, // phn -> phn_Phnx_LB
    { 230,   0,   0 }, { 230,  66, 187 }, // pl -> pl_Latn_PL
    { 231,   0,   0 }, { 231,  66,  32 }, // pt -> pt_Latn_BR
    { 232,   0,   0 }, { 232,  66, 187 }, // prg -> prg_Latn_PL
    { 233,   0, 178 }, { 233,   4, 178 }, // pa_PK -> pa_Arab_PK
    { 233,   4,   0 }, { 233,   4, 178 }, // pa_Arab -> pa_Arab_PK
    { 233,   0,   0 }, { 233,  41, 110 }, // pa -> pa_Guru_IN
    { 234,   0,   0 }, { 234,  66, 184 }, // qu -> qu_Latn_PE
    { 235,   0,   0 }, { 235,  66, 192 }, // ro -> ro_Latn_RO
    { 236,   0,   0 }, { 236,  66, 226 }, // rm -> rm_Latn_CH
    { 237,   0,   0 }, { 237,  66, 230 }, // rof -> rof_Latn_TZ
    { 238,   0,   0 }, { 238,  66,  38 }, // rn -> rn_Latn_BI
    { 239,   0,   0 }, { 239,  27, 193 }, // ru -> ru_Cyrl_RU
    { 240,   0,   0 }, { 240,  66, 230 }, // rwk -> rwk_Latn_TZ
    { 241,   0,   0 }, { 241,  66,  74 }, // ssy -> ssy_Latn_ER
    { 242,   0,   0 }, { 242,  27, 193 }, // sah -> sah_Cyrl_RU
    { 243,   0,   0 }, { 243,  66, 124 }, // saq -> saq_Latn_KE
    { 244,   0,   0 }, { 244,  66, 202 }, // sm -> sm_Latn_WS
    { 245,   0,   0 }, { 245,  66,  46 }, // sg -> sg_Latn_CF
    { 246,   0,   0 }, { 246,  66, 230 }, // sbp -> sbp_Latn_TZ
    { 247,   0,   0 }, { 247,  29, 110 }, // sa -> sa_Deva_IN
    { 248,   0,   0 }, { 248,  93, 110 }, // sat -> sat_Olck_IN
    { 249,   0,   0 }, { 249,  66, 117 }, // sc -> sc_Latn_IT
    { 250,   0,   0 }, { 250, 113, 110 }, // saz -> saz_Saur_IN
    { 251,   0,   0 }, { 251,  66, 160 }, // seh -> seh_Latn_MZ
    { 252,   0, 157 }, { 252,  66, 157 }, // sr_ME -> sr_Latn_ME
    { 252,   0, 192 }, { 252,  66, 192 }, // sr_RO -> sr_Latn_RO
    { 252,   0, 193 }, { 252,  66, 193 }, // sr_RU -> sr_Latn_RU
    { 252,   0, 239 }, { 252,  66, 239 }, // sr_TR -> sr_Latn_TR
    { 252,   0,   0 }, { 252,  27, 207 }, // sr -> sr_Cyrl_RS
    { 253,   0,   0 }, { 253,  66, 230 }, // ksb -> ksb_Latn_TZ
    { 254,   0,   0 }, { 254,  66, 261 }, // sn -> sn_Latn_ZW
    { 255,   0,   0 }, { 255, 141,  50 }, // ii -> ii_Yiii_CN
    { 256,   0,   0 }, { 256,  66, 117 }, // scn -> scn_Latn_IT
    { 257,   0,   0 }, { 257,  66,  77 }, // sid -> sid_Latn_ET
    { 258,   0,   0 }, { 258,  66, 187 }, // szl -> szl_Latn_PL
    { 259,   0, 110 }, { 259,  29, 110 }, // sd_IN -> sd_Deva_IN
    { 259,  29,   0 }, { 259,  29, 110 }, // sd_Deva -> sd_Deva_IN
    { 259,  61,   0 }, { 259,  61, 110 }, // sd_Khoj -> sd_Khoj_IN
    { 259,  62,   0 }, { 259,  62, 110 }, // sd_Sind -> sd_Sind_IN
    { 259,   0,   0 }, { 259,   4, 178 }, // sd -> sd_Arab_PK
    { 260,   0,   0 }, { 260, 119, 221 }, // si -> si_Sinh_LK
    { 261,   0,   0 }, { 261,  66,  83 }, // sms -> sms_Latn_FI
    { 262,   0,   0 }, { 262,  66, 212 }, // sk -> sk_Latn_SK
    { 263,   0,   0 }, { 263,  66, 213 }, // sl -> sl_Latn_SI
    { 264,   0,   0 }, { 264,  66, 243 }, // xog -> xog_Latn_UG
    { 265,   0,   0 }, { 265,  66, 215 }, // so -> so_Latn_SO
    { 266,   0,   0 }, { 266,   4, 112 }, // sdh -> sdh_Arab_IR
    { 267,   0,   0 }, { 267,  66, 225 }, // sma -> sma_Latn_SE
    { 268,   0,   0 }, { 268,  66, 216 }, // st -> st_Latn_ZA
    { 269,   0,   0 }, { 269,  66, 216 }, // nr -> nr_Latn_ZA
    { 270,   0,   0 }, { 270,  66, 220 }, // es -> es_Latn_ES
    { 271,   0,   0 }, { 271, 135, 159 }, // zgh -> zgh_Tfng_MA
    { 272,   0,   0 }, { 272,  66, 111 }, // su -> su_Latn_ID
    { 273,   0,   0 }, { 273,  66, 230 }, // sw -> sw_Latn_TZ
    { 274,   0,   0 }, { 274,  66, 216 }, // ss -> ss_Latn_ZA
    { 275,   0,   0 }, { 275,  66, 225 }, // sv -> sv_Latn_SE
    { 276,   0,   0 }, { 276,  66, 226 }, // gsw -> gsw_Latn_CH
    { 277,   0,   0 }, { 277, 123, 113 }, // syr -> syr_Syrc_IQ
    { 278,   0,   0 }, { 278, 135, 159 }, // shi -> shi_Tfng_MA
    { 279,   0,   0 }, { 279,  66,  86 }, // ty -> ty_Latn_PF
    { 280,   0,   0 }, { 280, 127, 255 }, // blt -> blt_Tavt_VN
    { 281,   0,   0 }, { 281,  66, 124 }, // dav -> dav_Latn_KE
    { 282,   0, 178 }, { 282,   4, 178 }, // tg_PK -> tg_Arab_PK
    { 282,   4,   0 }, { 282,   4, 178 }, // tg_Arab -> tg_Arab_PK
    { 282,   0,   0 }, { 282,  27, 229 }, // tg -> tg_Cyrl_TJ
    { 283,   0,   0 }, { 283, 129, 110 }, // ta -> ta_Taml_IN
    { 284,   0,   0 }, { 284,  66, 228 }, // trv -> trv_Latn_TW
    { 285,   0,   0 }, { 285,  66, 170 }, // twq -> twq_Latn_NE
    { 286,   0,   0 }, { 286,  27, 193 }, // tt -> tt_Cyrl_RU
    { 287,   0,   0 }, { 287, 131, 110 }, // te -> te_Telu_IN
    { 288,   0,   0 }, { 288,  66, 243 }, // teo -> teo_Latn_UG
    { 289,   0,   0 }, { 289, 133, 231 }, // th -> th_Thai_TH
    { 290,   0,   0 }, { 290, 134,  50 }, // bo -> bo_Tibt_CN
    { 291,   0,   0 }, { 291,  33,  74 }, // tig -> tig_Ethi_ER
    { 292,   0,   0 }, { 292,  33,  77 }, // ti -> ti_Ethi_ET
    { 293,   0,   0 }, { 293,  66, 234 }, // tkl -> tkl_Latn_TK
    { 294,   0,   0 }, { 294,  66, 182 }, // tpi -> tpi_Latn_PG
    { 295,   0,   0 }, { 295,  66, 235 }, // to -> to_Latn_TO
    { 296,   0,   0 }, { 296,  66, 216 }, // ts -> ts_Latn_ZA
    { 297,   0,   0 }, { 297,  66, 216 }, // tn -> tn_Latn_ZA
    { 298,   0,   0 }, { 298,  66, 239 }, // tr -> tr_Latn_TR
    { 299,   0,   0 }, { 299,  66, 240 }, // tk -> tk_Latn_TM
    { 300,   0,   0 }, { 300,  66, 242 }, // tvl -> tvl_Latn_TV
    { 301,   0,   0 }, { 301,  66, 169 }, // kcg -> kcg_Latn_NG
    { 302,   0,   0 }, { 302, 138, 227 }, // uga -> uga_Ugar_SY
    { 303,   0,   0 }, { 303,  27, 244 }, // uk -> uk_Cyrl_UA
    { 304,   0,   0 }, { 304,  66,  91 }, // hsb -> hsb_Latn_DE
    { 305,   0,   0 }, { 305,   4, 178 }, // ur -> ur_Arab_PK
    { 306,   0, 123 }, { 306,  27, 123 }, // ug_KZ -> ug_Cyrl_KZ
    { 306,   0, 156 }, { 306,  27, 156 }, // ug_MN -> ug_Cyrl_MN
    { 306,  27,   0 }, { 306,  27, 123 }, // ug_Cyrl -> ug_Cyrl_KZ
    { 306,   0,   0 }, { 306,   4,  50 }, // ug -> ug_Arab_CN
    { 307,   0,   1 }, { 307,   4,   1 }, // uz_AF -> uz_Arab_AF
    { 307,   0,  50 }, { 307,  27,  50 }, // uz_CN -> uz_Cyrl_CN
    { 307,   4,   0 }, { 307,   4,   1 }, // uz_Arab -> uz_Arab_AF
    { 307,   0,   0 }, { 307,  66, 251 }, // uz -> uz_Latn_UZ
    { 308,   0,   0 }, { 308, 139, 134 }, // vai -> vai_Vaii_LR
    { 309,   0,   0 }, { 309,  66, 216 }, // ve -> ve_Latn_ZA
    { 310,   0,   0 }, { 310,  66, 255 }, // vi -> vi_Latn_VN
    { 311,   0,   0 }, { 311,  66, 258 }, // vo -> vo_Latn_001
    { 312,   0,   0 }, { 312,  66, 230 }, // vun -> vun_Latn_TZ
    { 313,   0,   0 }, { 313,  66,  23 }, // wa -> wa_Latn_BE
    { 314,   0,   0 }, { 314,  66, 226 }, // wae -> wae_Latn_CH
    { 315,   0,   0 }, { 315,  66,  15 }, // wbp -> wbp_Latn_AU
    { 316,   0,   0 }, { 316,  66, 246 }, // cy -> cy_Latn_GB
    { 317,   0,   0 }, { 317,   4, 178 }, // bgn -> bgn_Arab_PK
    { 318,   0,   0 }, { 318,  66, 165 }, // fy -> fy_Latn_NL
    { 319,   0,   0 }, { 319,  33,  77 }, // wal -> wal_Ethi_ET
    { 320,   0,   0 }, { 320,  66, 206 }, // wo -> wo_Latn_SN
    { 321,   0,   0 }, { 321,  66, 216 }, // xh -> xh_Latn_ZA
    { 322,   0,   0 }, { 322,  66,  40 }, // yav -> yav_Latn_CM
    { 323,   0,   0 }, { 323,  47, 244 }, // yi -> yi_Hebr_UA
    { 324,   0,   0 }, { 324,  66, 169 }, // yo -> yo_Latn_NG
    { 325,   0,   0 }, { 325,  66, 170 }, // dje -> dje_Latn_NE
    { 326,   0,   0 }, { 326,  66,  50 }, // za -> za_Latn_CN
    { 327,   0,   0 }, { 327,  66, 216 }, // zu -> zu_Latn_ZA
    { 328,   0,   0 }, { 328,  66,  32 }, // kgp -> kgp_Latn_BR
    { 329,   0,   0 }, { 329,  66,  32 }, // yrl -> yrl_Latn_BR
    { 330,   0,   0 }, { 330,  29, 110 }, // bgc -> bgc_Deva_IN
    { 331,   0,   0 }, { 331,  66,  91 }, // frr -> frr_Latn_DE
    { 332,   0,   0 }, { 332,  29, 110 }, // raj -> raj_Deva_IN
    { 333,   0,   0 }, { 333,  27, 193 }, // mdf -> mdf_Cyrl_RU
    { 334,   0,   0 }, { 334,  66, 258 }, // tok -> tok_Latn_001
    { 335,   0,   0 }, { 335,  66, 214 }, // pis -> pis_Latn_SB
    { 336,   0,   0 }, { 336,  66, 169 }, // ann -> ann_Latn_NG
    { 337,   0,   0 }, { 337,   4, 178 }, // bal -> bal_Arab_PK
    { 338,   0,   0 }, { 338,  66, 117 }, // lij -> lij_Latn_IT
    { 339,   0,   0 }, { 339, 142, 161 }, // rhg -> rhg_Rohg_MM
    { 340,   0,   0 }, { 340,   4, 178 }, // trw -> trw_Arab_PK
    { 341,   0,   0 }, { 341,  66,  25 }, // blo -> blo_Latn_BJ
    { 342,   0,   0 }, { 342,  29, 110 }, // xnr -> xnr_Deva_IN
    { 343,   0,   0 }, { 343,  66, 117 }, // vec -> vec_Latn_IT
    { 344,   0,   0 }, { 344,  66, 110 }, // kxv -> kxv_Latn_IN
    {   0,   4,   1 }, { 228,   4,   1 }, // und_Arab_AF -> fa_Arab_AF
    {   0,  66,   1 }, { 299,  66,   1 }, // und_Latn_AF -> tk_Latn_AF
    {   0,   0,   1 }, { 228,   4,   1 }, // und_AF -> fa_Arab_AF
    {   0,  66,   2 }, { 275,  66,   2 }, // und_Latn_AX -> sv_Latn_AX
    {   0,   0,   2 }, { 275,  66,   2 }, // und_AX -> sv_Latn_AX
    {   0,  27,   3 }, { 169,  27,   3 }, // und_Cyrl_AL -> mk_Cyrl_AL
    {   0,  66,   3 }, {   9,  66,   3 }, // und_Latn_AL -> sq_Latn_AL
    {   0,   0,   3 }, {   9,  66,   3 }, // und_AL -> sq_Latn_AL
    {   0,  66,   4 }, {  85,  66,   4 }, // und_Latn_DZ -> fr_Latn_DZ
    {   0,   0,   4 }, {  14,   4,   4 }, // und_DZ -> ar_Arab_DZ
    {   0,  66,   5 }, { 244,  66,   5 }, // und_Latn_AS -> sm_Latn_AS
    {   0,   0,   5 }, { 244,  66,   5 }, // und_AS -> sm_Latn_AS
    {   0,  66,   6 }, {  48,  66,   6 }, // und_Latn_AD -> ca_Latn_AD
    {   0,   0,   6 }, {  48,  66,   6 }, // und_AD -> ca_Latn_AD
    {   0,  66,   7 }, { 231,  66,   7 }, // und_Latn_AO -> pt_Latn_AO
    {   0,   0,   7 }, { 231,  66,   7 }, // und_AO -> pt_Latn_AO
    {   0,   0,   8 }, {  75,  66,   8 }, // und_AI -> en_Latn_AI
    {   0,   0,   9 }, {  75,  66,   9 }, // und_AQ -> en_Latn_AQ
    {   0,   0,  10 }, {  75,  66,  10 }, // und_AG -> en_Latn_AG
    {   0,  66,  11 }, { 270,  66,  11 }, // und_Latn_AR -> es_Latn_AR
    {   0,   0,  11 }, { 270,  66,  11 }, // und_AR -> es_Latn_AR
    {   0,  66,  12 }, { 148,  66,  12 }, // und_Latn_AM -> ku_Latn_AM
    {   0,   0,  12 }, {  17,   5,  12 }, // und_AM -> hy_Armn_AM
    {   0,  66,  13 }, {  72,  66,  13 }, // und_Latn_AW -> nl_Latn_AW
    {   0,   0,  13 }, {  72,  66,  13 }, // und_AW -> nl_Latn_AW
    {   0,   0,  14 }, {  75,  66,  14 }, // und_AC -> en_Latn_AC
    {   0,   0,  15 }, {  75,  66,  15 }, // und_AU -> en_Latn_AU
    {   0,  66,  16 }, {  94,  66,  16 }, // und_Latn_AT -> de_Latn_AT
    {   0,   0,  16 }, {  94,  66,  16 }, // und_AT -> de_Latn_AT
    {   0,  66,  17 }, {  25,  66,  17 }, // und_Latn_AZ -> az_Latn_AZ
    {   0,   0,  17 }, {  25,  66,  17 }, // und_AZ -> az_Latn_AZ
    {   0,   0,  18 }, {  75,  66,  18 }, // und_BS -> en_Latn_BS
    {   0,   0,  19 }, {  14,   4,  19 }, // und_BH -> ar_Arab_BH
    {   0,   0,  20 }, {  30,   9,  20 }, // und_BD -> bn_Beng_BD
    {   0,   0,  21 }, {  75,  66,  21 }, // und_BB -> en_Latn_BB
    {   0,  27,  22 }, {  35,  27,  22 }, // und_Cyrl_BY -> be_Cyrl_BY
    {   0,   0,  22 }, {  35,  27,  22 }, // und_BY -> be_Cyrl_BY
    {   0,  66,  23 }, {  72,  66,  23 }, // und_Latn_BE -> nl_Latn_BE
    {   0,   0,  23 }, {  72,  66,  23 }, // und_BE -> nl_Latn_BE
    {   0,   0,  24 }, {  75,  66,  24 }, // und_BZ -> en_Latn_BZ
    {   0,  66,  25 }, {  85,  66,  25 }, // und_Latn_BJ -> fr_Latn_BJ
    {   0,   0,  25 }, {  85,  66,  25 }, // und_BJ -> fr_Latn_BJ
    {   0,   0,  26 }, {  75,  66,  26 }, // und_BM -> en_Latn_BM
    {   0,  29,  27 }, { 199,  29,  27 }, // und_Deva_BT -> ne_Deva_BT
    {   0, 134,  27 }, {  73, 134,  27 }, // und_Tibt_BT -> dz_Tibt_BT
    {   0,   0,  27 }, {  73, 134,  27 }, // und_BT -> dz_Tibt_BT
    {   0,  66,  28 }, { 270,  66,  28 }, // und_Latn_BO -> es_Latn_BO
    {   0,   0,  28 }, { 270,  66,  28 }, // und_BO -> es_Latn_BO
    {   0,  27,  29 }, { 252,  27,  29 }, // und_Cyrl_BA -> sr_Cyrl_BA
    {   0,  66,  29 }, {  42,  66,  29 }, // und_Latn_BA -> bs_Latn_BA
    {   0,   0,  29 }, {  42,  66,  29 }, // und_BA -> bs_Latn_BA
    {   0,   0,  30 }, {  75,  66,  30 }, // und_BW -> en_Latn_BW
    {   0,   0,  31 }, {  75,  66,  31 }, // und_BV -> en_Latn_BV
    {   0,  66,  32 }, { 231,  66,  32 }, // und_Latn_BR -> pt_Latn_BR
    {   0,   0,  32 }, { 231,  66,  32 }, // und_BR -> pt_Latn_BR
    {   0,   0,  33 }, {  75,  66,  33 }, // und_IO -> en_Latn_IO
    {   0,   0,  34 }, {  75,  66,  34 }, // und_VG -> en_Latn_VG
    {   0,  66,  35 }, { 176,  66,  35 }, // und_Latn_BN -> ms_Latn_BN
    {   0,   0,  35 }, { 176,  66,  35 }, // und_BN -> ms_Latn_BN
    {   0,  27,  36 }, {  45,  27,  36 }, // und_Cyrl_BG -> bg_Cyrl_BG
    {   0,   0,  36 }, {  45,  27,  36 }, // und_BG -> bg_Cyrl_BG
    {   0,  66,  37 }, {  85,  66,  37 }, // und_Latn_BF -> fr_Latn_BF
    {   0,   0,  37 }, {  85,  66,  37 }, // und_BF -> fr_Latn_BF
    {   0,  66,  38 }, { 238,  66,  38 }, // und_Latn_BI -> rn_Latn_BI
    {   0,   0,  38 }, { 238,  66,  38 }, // und_BI -> rn_Latn_BI
    {   0,   0,  39 }, { 135,  60,  39 }, // und_KH -> km_Khmr_KH
    {   0,  66,  40 }, {  85,  66,  40 }, // und_Latn_CM -> fr_Latn_CM
    {   0,   0,  40 }, {  85,  66,  40 }, // und_CM -> fr_Latn_CM
    {   0, 137,  41 }, {  47, 137,  41 }, // und_Hant_CA -> yue_Hant_CA
    {   0,   0,  41 }, {  75,  66,  41 }, // und_CA -> en_Latn_CA
    {   0,  66,  42 }, { 270,  66,  42 }, // und_Latn_IC -> es_Latn_IC
    {   0,   0,  42 }, { 270,  66,  42 }, // und_IC -> es_Latn_IC
    {   0,  66,  43 }, { 231,  66,  43 }, // und_Latn_CV -> pt_Latn_CV
    {   0,   0,  43 }, { 231,  66,  43 }, // und_CV -> pt_Latn_CV
    {   0,  66,  44 }, { 226,  66,  44 }, // und_Latn_BQ -> pap_Latn_BQ
    {   0,   0,  44 }, { 226,  66,  44 }, // und_BQ -> pap_Latn_BQ
    {   0,   0,  45 }, {  75,  66,  45 }, // und_KY -> en_Latn_KY
    {   0,  66,  46 }, {  85,  66,  46 }, // und_Latn_CF -> fr_Latn_CF
    {   0,   0,  46 }, {  85,  66,  46 }, // und_CF -> fr_Latn_CF
    {   0,  66,  47 }, { 270,  66,  47 }, // und_Latn_EA -> es_Latn_EA
    {   0,   0,  47 }, { 270,  66,  47 }, // und_EA -> es_Latn_EA
    {   0,  66,  48 }, {  85,  66,  48 }, // und_Latn_TD -> fr_Latn_TD
    {   0,   0,  48 }, {  85,  66,  48 }, // und_TD -> fr_Latn_TD
    {   0,  66,  49 }, { 270,  66,  49 }, // und_Latn_CL -> es_Latn_CL
    {   0,   0,  49 }, { 270,  66,  49 }, // und_CL -> es_Latn_CL
    {   0,   4,  50 }, { 306,   4,  50 }, // und_Arab_CN -> ug_Arab_CN
    {   0,  66,  50 }, { 326,  66,  50 }, // und_Latn_CN -> za_Latn_CN
    {   0,   0,  50 }, {  58, 118,  50 }, // und_CN -> zh_Hans_CN
    {   0,   0,  51 }, {  75,  66,  51 }, // und_CX -> en_Latn_CX
    {   0,   0,  52 }, {  75,  66,  52 }, // und_CP -> en_Latn_CP
    {   0,   4,  53 }, { 176,   4,  53 }, // und_Arab_CC -> ms_Arab_CC
    {   0,   0,  53 }, { 176,   4,  53 }, // und_CC -> ms_Arab_CC
    {   0,  66,  54 }, { 270,  66,  54 }, // und_Latn_CO -> es_Latn_CO
    {   0,   0,  54 }, { 270,  66,  54 }, // und_CO -> es_Latn_CO
    {   0,  66,  55 }, {  85,  66,  55 }, // und_Latn_KM -> fr_Latn_KM
    {   0,   0,  55 }, {  14,   4,  55 }, // und_KM -> ar_Arab_KM
    {   0,  66,  56 }, {  85,  66,  56 }, // und_Latn_CG -> fr_Latn_CG
    {   0,   0,  56 }, {  85,  66,  56 }, // und_CG -> fr_Latn_CG
    {   0,  66,  57 }, { 273,  66,  57 }, // und_Latn_CD -> sw_Latn_CD
    {   0,   0,  57 }, { 273,  66,  57 }, // und_CD -> sw_Latn_CD
    {   0,   0,  58 }, {  75,  66,  58 }, // und_CK -> en_Latn_CK
    {   0,  66,  59 }, { 270,  66,  59 }, // und_Latn_CR -> es_Latn_CR
    {   0,   0,  59 }, { 270,  66,  59 }, // und_CR -> es_Latn_CR
    {   0,  66,  60 }, {  66,  66,  60 }, // und_Latn_HR -> hr_Latn_HR
    {   0,   0,  60 }, {  66,  66,  60 }, // und_HR -> hr_Latn_HR
    {   0,  66,  61 }, { 270,  66,  61 }, // und_Latn_CU -> es_Latn_CU
    {   0,   0,  61 }, { 270,  66,  61 }, // und_CU -> es_Latn_CU
    {   0,  66,  62 }, { 226,  66,  62 }, // und_Latn_CW -> pap_Latn_CW
    {   0,   0,  62 }, { 226,  66,  62 }, // und_CW -> pap_Latn_CW
    {   0,  66,  63 }, { 298,  66,  63 }, // und_Latn_CY -> tr_Latn_CY
    {   0,   0,  63 }, {  96,  39,  63 }, // und_CY -> el_Grek_CY
    {   0,  66,  64 }, {  67,  66,  64 }, // und_Latn_CZ -> cs_Latn_CZ
    {   0,   0,  64 }, {  67,  66,  64 }, // und_CZ -> cs_Latn_CZ
    {   0,  66,  65 }, {  68,  66,  65 }, // und_Latn_DK -> da_Latn_DK
    {   0,   0,  65 }, {  68,  66,  65 }, // und_DK -> da_Latn_DK
    {   0,   0,  66 }, {  75,  66,  66 }, // und_DG -> en_Latn_DG
    {   0,  66,  67 }, {   3,  66,  67 }, // und_Latn_DJ -> aa_Latn_DJ
    {   0,   0,  67 }, {   3,  66,  67 }, // und_DJ -> aa_Latn_DJ
    {   0,   0,  68 }, {  75,  66,  68 }, // und_DM -> en_Latn_DM
    {   0,  66,  69 }, { 270,  66,  69 }, // und_Latn_DO -> es_Latn_DO
    {   0,   0,  69 }, { 270,  66,  69 }, // und_DO -> es_Latn_DO
    {   0,  66,  70 }, { 270,  66,  70 }, // und_Latn_EC -> es_Latn_EC
    {   0,   0,  70 }, { 270,  66,  70 }, // und_EC -> es_Latn_EC
    {   0,   0,  71 }, {  14,   4,  71 }, // und_EG -> ar_Arab_EG
    {   0,  66,  72 }, { 270,  66,  72 }, // und_Latn_SV -> es_Latn_SV
    {   0,   0,  72 }, { 270,  66,  72 }, // und_SV -> es_Latn_SV
    {   0,  66,  73 }, { 270,  66,  73 }, // und_Latn_GQ -> es_Latn_GQ
    {   0,   0,  73 }, { 270,  66,  73 }, // und_GQ -> es_Latn_GQ
    {   0,  33,  74 }, { 292,  33,  74 }, // und_Ethi_ER -> ti_Ethi_ER
    {   0,   0,  74 }, { 292,  33,  74 }, // und_ER -> ti_Ethi_ER
    {   0,  66,  75 }, {  78,  66,  75 }, // und_Latn_EE -> et_Latn_EE
    {   0,   0,  75 }, {  78,  66,  75 }, // und_EE -> et_Latn_EE
    {   0,   0,  76 }, {  75,  66,  76 }, // und_SZ -> en_Latn_SZ
    {   0,  66,  77 }, {  75,  66,  77 }, // und_Latn_ET -> en_Latn_ET
    {   0,   0,  77 }, {  11,  33,  77 }, // und_ET -> am_Ethi_ET
    {   0,   0,  80 }, {  75,  66,  80 }, // und_FK -> en_Latn_FK
    {   0,  66,  81 }, {  81,  66,  81 }, // und_Latn_FO -> fo_Latn_FO
    {   0,   0,  81 }, {  81,  66,  81 }, // und_FO -> fo_Latn_FO
    {   0,   0,  82 }, {  75,  66,  82 }, // und_FJ -> en_Latn_FJ
    {   0,  66,  83 }, {  84,  66,  83 }, // und_Latn_FI -> fi_Latn_FI
    {   0,   0,  83 }, {  84,  66,  83 }, // und_FI -> fi_Latn_FI
    {   0,  66,  84 }, {  85,  66,  84 }, // und_Latn_FR -> fr_Latn_FR
    {   0,   0,  84 }, {  85,  66,  84 }, // und_FR -> fr_Latn_FR
    {   0,  66,  85 }, {  85,  66,  85 }, // und_Latn_GF -> fr_Latn_GF
    {   0,   0,  85 }, {  85,  66,  85 }, // und_GF -> fr_Latn_GF
    {   0,  66,  86 }, {  85,  66,  86 }, // und_Latn_PF -> fr_Latn_PF
    {   0,   0,  86 }, {  85,  66,  86 }, // und_PF -> fr_Latn_PF
    {   0,  66,  87 }, {  85,  66,  87 }, // und_Latn_TF -> fr_Latn_TF
    {   0,   0,  87 }, {  85,  66,  87 }, // und_TF -> fr_Latn_TF
    {   0,  66,  88 }, {  85,  66,  88 }, // und_Latn_GA -> fr_Latn_GA
    {   0,   0,  88 }, {  85,  66,  88 }, // und_GA -> fr_Latn_GA
    {   0,   0,  89 }, {  75,  66,  89 }, // und_GM -> en_Latn_GM
    {   0,  27,  90 }, {   2,  27,  90 }, // und_Cyrl_GE -> ab_Cyrl_GE
    {   0,  66,  90 }, { 148,  66,  90 }, // und_Latn_GE -> ku_Latn_GE
    {   0,   0,  90 }, {  93,  35,  90 }, // und_GE -> ka_Geor_GE
    {   0,  66,  91 }, {  94,  66,  91 }, // und_Latn_DE -> de_Latn_DE
    {   0,   0,  91 }, {  94,  66,  91 }, // und_DE -> de_Latn_DE
    {   0,  66,  92 }, {   6,  66,  92 }, // und_Latn_GH -> ak_Latn_GH
    {   0,   0,  92 }, {   6,  66,  92 }, // und_GH -> ak_Latn_GH
    {   0,   0,  93 }, {  75,  66,  93 }, // und_GI -> en_Latn_GI
    {   0,  27,  94 }, { 169,  27,  94 }, // und_Cyrl_GR -> mk_Cyrl_GR
    {   0,   0,  94 }, {  96,  39,  94 }, // und_GR -> el_Grek_GR
    {   0,  66,  95 }, { 127,  66,  95 }, // und_Latn_GL -> kl_Latn_GL
    {   0,   0,  95 }, { 127,  66,  95 }, // und_GL -> kl_Latn_GL
    {   0,   0,  96 }, {  75,  66,  96 }, // und_GD -> en_Latn_GD
    {   0,  66,  97 }, {  85,  66,  97 }, // und_Latn_GP -> fr_Latn_GP
    {   0,   0,  97 }, {  85,  66,  97 }, // und_GP -> fr_Latn_GP
    {   0,   0,  98 }, {  75,  66,  98 }, // und_GU -> en_Latn_GU
    {   0,  66,  99 }, { 270,  66,  99 }, // und_Latn_GT -> es_Latn_GT
    {   0,   0,  99 }, { 270,  66,  99 }, // und_GT -> es_Latn_GT
    {   0,   0, 100 }, {  75,  66, 100 }, // und_GG -> en_Latn_GG
    {   0,  66, 101 }, { 231,  66, 101 }, // und_Latn_GW -> pt_Latn_GW
    {   0,   0, 101 }, { 231,  66, 101 }, // und_GW -> pt_Latn_GW
    {   0,  66, 102 }, {  85,  66, 102 }, // und_Latn_GN -> fr_Latn_GN
    {   0,   0, 102 }, {  85,  66, 102 }, // und_GN -> fr_Latn_GN
    {   0,   0, 103 }, {  75,  66, 103 }, // und_GY -> en_Latn_GY
    {   0,  66, 104 }, { 100,  66, 104 }, // und_Latn_HT -> ht_Latn_HT
    {   0,   0, 104 }, { 100,  66, 104 }, // und_HT -> ht_Latn_HT
    {   0,   0, 105 }, {  75,  66, 105 }, // und_HM -> en_Latn_HM
    {   0,  66, 106 }, { 270,  66, 106 }, // und_Latn_HN -> es_Latn_HN
    {   0,   0, 106 }, { 270,  66, 106 }, // und_HN -> es_Latn_HN
    {   0,   0, 107 }, {  58, 137, 107 }, // und_HK -> zh_Hant_HK
    {   0,  66, 108 }, { 107,  66, 108 }, // und_Latn_HU -> hu_Latn_HU
    {   0,   0, 108 }, { 107,  66, 108 }, // und_HU -> hu_Latn_HU
    {   0,  66, 109 }, { 108,  66, 109 }, // und_Latn_IS -> is_Latn_IS
    {   0,   0, 109 }, { 108,  66, 109 }, // und_IS -> is_Latn_IS
    {   0,   4, 110 }, { 305,   4, 110 }, // und_Arab_IN -> ur_Arab_IN
    {   0,   0, 110 }, { 105,  29, 110 }, // und_IN -> hi_Deva_IN
    {   0,   4, 111 }, { 176,   4, 111 }, // und_Arab_ID -> ms_Arab_ID
    {   0,  66, 111 }, { 112,  66, 111 }, // und_Latn_ID -> id_Latn_ID
    {   0,   0, 111 }, { 112,  66, 111 }, // und_ID -> id_Latn_ID
    {   0,   4, 112 }, { 228,   4, 112 }, // und_Arab_IR -> fa_Arab_IR
    {   0,  66, 112 }, { 299,  66, 112 }, // und_Latn_IR -> tk_Latn_IR
    {   0,   0, 112 }, { 228,   4, 112 }, // und_IR -> fa_Arab_IR
    {   0,   0, 113 }, {  14,   4, 113 }, // und_IQ -> ar_Arab_IQ
    {   0,   0, 114 }, {  75,  66, 114 }, // und_IE -> en_Latn_IE
    {   0,   0, 115 }, {  75,  66, 115 }, // und_IM -> en_Latn_IM
    {   0,   0, 116 }, { 103,  47, 116 }, // und_IL -> he_Hebr_IL
    {   0,  66, 117 }, { 119,  66, 117 }, // und_Latn_IT -> it_Latn_IT
    {   0,   0, 117 }, { 119,  66, 117 }, // und_IT -> it_Latn_IT
    {   0,  66, 118 }, {  85,  66, 118 }, // und_Latn_CI -> fr_Latn_CI
    {   0,   0, 118 }, {  85,  66, 118 }, // und_CI -> fr_Latn_CI
    {   0,   0, 119 }, {  75,  66, 119 }, // und_JM -> en_Latn_JM
    {   0,   0, 120 }, { 120,  53, 120 }, // und_JP -> ja_Jpan_JP
    {   0,   0, 121 }, {  75,  66, 121 }, // und_JE -> en_Latn_JE
    {   0,   0, 122 }, {  14,   4, 122 }, // und_JO -> ar_Arab_JO
    {   0,   0, 123 }, { 239,  27, 123 }, // und_KZ -> ru_Cyrl_KZ
    {   0,  66, 124 }, { 273,  66, 124 }, // und_Latn_KE -> sw_Latn_KE
    {   0,   0, 124 }, { 273,  66, 124 }, // und_KE -> sw_Latn_KE
    {   0,   0, 125 }, {  75,  66, 125 }, // und_KI -> en_Latn_KI
    {   0,  27, 126 }, { 252,  27, 126 }, // und_Cyrl_XK -> sr_Cyrl_XK
    {   0,  66, 126 }, {   9,  66, 126 }, // und_Latn_XK -> sq_Latn_XK
    {   0,   0, 126 }, {   9,  66, 126 }, // und_XK -> sq_Latn_XK
    {   0,   0, 127 }, {  14,   4, 127 }, // und_KW -> ar_Arab_KW
    {   0,  27, 128 }, { 150,  27, 128 }, // und_Cyrl_KG -> ky_Cyrl_KG
    {   0,   0, 128 }, { 150,  27, 128 }, // und_KG -> ky_Cyrl_KG
    {   0,   0, 129 }, { 153,  65, 129 }, // und_LA -> lo_Laoo_LA
    {   0,  66, 130 }, { 270,  66, 130 }, // und_Latn_419 -> es_Latn_419
    {   0,   0, 130 }, { 270,  66, 130 }, // und_419 -> es_Latn_419
    {   0,  66, 131 }, { 155,  66, 131 }, // und_Latn_LV -> lv_Latn_LV
    {   0,   0, 131 }, { 155,  66, 131 }, // und_LV -> lv_Latn_LV
    {   0,   0, 132 }, {  14,   4, 132 }, // und_LB -> ar_Arab_LB
    {   0,  66, 133 }, { 268,  66, 133 }, // und_Latn_LS -> st_Latn_LS
    {   0,   0, 133 }, { 268,  66, 133 }, // und_LS -> st_Latn_LS
    {   0,   0, 134 }, {  75,  66, 134 }, // und_LR -> en_Latn_LR
    {   0,   0, 135 }, {  14,   4, 135 }, // und_LY -> ar_Arab_LY
    {   0,  66, 136 }, {  94,  66, 136 }, // und_Latn_LI -> de_Latn_LI
    {   0,   0, 136 }, {  94,  66, 136 }, // und_LI -> de_Latn_LI
    {   0,  66, 137 }, { 160,  66, 137 }, // und_Latn_LT -> lt_Latn_LT
    {   0,   0, 137 }, { 160,  66, 137 }, // und_LT -> lt_Latn_LT
    {   0,  66, 138 }, {  85,  66, 138 }, // und_Latn_LU -> fr_Latn_LU
    {   0,   0, 138 }, {  85,  66, 138 }, // und_LU -> fr_Latn_LU
    {   0,  66, 139 }, { 231,  66, 139 }, // und_Latn_MO -> pt_Latn_MO
    {   0,   0, 139 }, {  58, 137, 139 }, // und_MO -> zh_Hant_MO
    {   0,  27, 140 }, { 169,  27, 140 }, // und_Cyrl_MK -> mk_Cyrl_MK
    {   0,  66, 140 }, {   9,  66, 140 }, // und_Latn_MK -> sq_Latn_MK
    {   0,   0, 140 }, { 169,  27, 140 }, // und_MK -> mk_Cyrl_MK
    {   0,  66, 141 }, { 174,  66, 141 }, // und_Latn_MG -> mg_Latn_MG
    {   0,   0, 141 }, { 174,  66, 141 }, // und_MG -> mg_Latn_MG
    {   0,   0, 142 }, {  75,  66, 142 }, // und_MW -> en_Latn_MW
    {   0,  66, 143 }, { 176,  66, 143 }, // und_Latn_MY -> ms_Latn_MY
    {   0,   0, 143 }, { 176,  66, 143 }, // und_MY -> ms_Latn_MY
    {   0,   0, 144 }, {  69, 132, 144 }, // und_MV -> dv_Thaa_MV
    {   0,  66, 145 }, {  28,  66, 145 }, // und_Latn_ML -> bm_Latn_ML
    {   0,   0, 145 }, {  28,  66, 145 }, // und_ML -> bm_Latn_ML
    {   0,  66, 146 }, { 177,  66, 146 }, // und_Latn_MT -> mt_Latn_MT
    {   0,   0, 146 }, { 177,  66, 146 }, // und_MT -> mt_Latn_MT
    {   0,   0, 147 }, {  75,  66, 147 }, // und_MH -> en_Latn_MH
    {   0,  66, 148 }, {  85,  66, 148 }, // und_Latn_MQ -> fr_Latn_MQ
    {   0,   0, 148 }, {  85,  66, 148 }, // und_MQ -> fr_Latn_MQ
    {   0,  66, 149 }, {  85,  66, 149 }, // und_Latn_MR -> fr_Latn_MR
    {   0,   0, 149 }, {  14,   4, 149 }, // und_MR -> ar_Arab_MR
    {   0,   4, 150 }, { 305,   4, 150 }, // und_Arab_MU -> ur_Arab_MU
    {   0,  29, 150 }, {  38,  29, 150 }, // und_Deva_MU -> bho_Deva_MU
    {   0,  66, 150 }, { 192,  66, 150 }, // und_Latn_MU -> mfe_Latn_MU
    {   0,   0, 150 }, { 192,  66, 150 }, // und_MU -> mfe_Latn_MU
    {   0,  66, 151 }, {  85,  66, 151 }, // und_Latn_YT -> fr_Latn_YT
    {   0,   0, 151 }, {  85,  66, 151 }, // und_YT -> fr_Latn_YT
    {   0,  66, 152 }, { 270,  66, 152 }, // und_Latn_MX -> es_Latn_MX
    {   0,   0, 152 }, { 270,  66, 152 }, // und_MX -> es_Latn_MX
    {   0,   0, 153 }, {  75,  66, 153 }, // und_FM -> en_Latn_FM
    {   0,  27, 154 }, { 303,  27, 154 }, // und_Cyrl_MD -> uk_Cyrl_MD
    {   0,  66, 154 }, { 235,  66, 154 }, // und_Latn_MD -> ro_Latn_MD
    {   0,   0, 154 }, { 235,  66, 154 }, // und_MD -> ro_Latn_MD
    {   0,  66, 155 }, {  85,  66, 155 }, // und_Latn_MC -> fr_Latn_MC
    {   0,   0, 155 }, {  85,  66, 155 }, // und_MC -> fr_Latn_MC
    {   0,   4, 156 }, { 133,   4, 156 }, // und_Arab_MN -> kk_Arab_MN
    {   0,  27, 156 }, { 191,  27, 156 }, // und_Cyrl_MN -> mn_Cyrl_MN
    {   0,   0, 156 }, { 191,  27, 156 }, // und_MN -> mn_Cyrl_MN
    {   0,  66, 157 }, { 252,  66, 157 }, // und_Latn_ME -> sr_Latn_ME
    {   0,   0, 157 }, { 252,  66, 157 }, // und_ME -> sr_Latn_ME
    {   0,   0, 158 }, {  75,  66, 158 }, // und_MS -> en_Latn_MS
    {   0,  66, 159 }, {  85,  66, 159 }, // und_Latn_MA -> fr_Latn_MA
    {   0,   0, 159 }, {  14,   4, 159 }, // und_MA -> ar_Arab_MA
    {   0,  66, 160 }, { 231,  66, 160 }, // und_Latn_MZ -> pt_Latn_MZ
    {   0,   0, 160 }, { 231,  66, 160 }, // und_MZ -> pt_Latn_MZ
    {   0,   4, 161 }, { 339,   4, 161 }, // und_Arab_MM -> rhg_Arab_MM
    {   0,   0, 161 }, {  46,  86, 161 }, // und_MM -> my_Mymr_MM
    {   0,  66, 162 }, {   4,  66, 162 }, // und_Latn_NA -> af_Latn_NA
    {   0,   0, 162 }, {   4,  66, 162 }, // und_NA -> af_Latn_NA
    {   0,   0, 163 }, {  75,  66, 163 }, // und_NR -> en_Latn_NR
    {   0,  29, 164 }, { 199,  29, 164 }, // und_Deva_NP -> ne_Deva_NP
    {   0,   0, 164 }, { 199,  29, 164 }, // und_NP -> ne_Deva_NP
    {   0,  66, 165 }, {  72,  66, 165 }, // und_Latn_NL -> nl_Latn_NL
    {   0,   0, 165 }, {  72,  66, 165 }, // und_NL -> nl_Latn_NL
    {   0,  66, 166 }, {  85,  66, 166 }, // und_Latn_NC -> fr_Latn_NC
    {   0,   0, 166 }, {  85,  66, 166 }, // und_NC -> fr_Latn_NC
    {   0,   0, 167 }, {  75,  66, 167 }, // und_NZ -> en_Latn_NZ
    {   0,  66, 168 }, { 270,  66, 168 }, // und_Latn_NI -> es_Latn_NI
    {   0,   0, 168 }, { 270,  66, 168 }, // und_NI -> es_Latn_NI
    {   0,   4, 169 }, { 101,   4, 169 }, // und_Arab_NG -> ha_Arab_NG
    {   0,   0, 169 }, {  75,  66, 169 }, // und_NG -> en_Latn_NG
    {   0,  66, 170 }, { 101,  66, 170 }, // und_Latn_NE -> ha_Latn_NE
    {   0,   0, 170 }, { 101,  66, 170 }, // und_NE -> ha_Latn_NE
    {   0,   0, 171 }, {  75,  66, 171 }, // und_NU -> en_Latn_NU
    {   0,   0, 172 }, {  75,  66, 172 }, // und_NF -> en_Latn_NF
    {   0,   0, 173 }, {  75,  66, 173 }, // und_MP -> en_Latn_MP
    {   0,   0, 174 }, { 142,  63, 174 }, // und_KP -> ko_Kore_KP
    {   0,  66, 175 }, { 209,  66, 175 }, // und_Latn_NO -> nb_Latn_NO
    {   0,   0, 175 }, { 209,  66, 175 }, // und_NO -> nb_Latn_NO
    {   0,   0, 176 }, {  14,   4, 176 }, // und_OM -> ar_Arab_OM
    {   0,   4, 178 }, { 305,   4, 178 }, // und_Arab_PK -> ur_Arab_PK
    {   0,   0, 178 }, { 305,   4, 178 }, // und_PK -> ur_Arab_PK
    {   0,  66, 179 }, { 224,  66, 179 }, // und_Latn_PW -> pau_Latn_PW
    {   0,   0, 179 }, { 224,  66, 179 }, // und_PW -> pau_Latn_PW
    {   0,   0, 180 }, {  14,   4, 180 }, // und_PS -> ar_Arab_PS
    {   0,  66, 181 }, { 270,  66, 181 }, // und_Latn_PA -> es_Latn_PA
    {   0,   0, 181 }, { 270,  66, 181 }, // und_PA -> es_Latn_PA
    {   0,  66, 182 }, { 294,  66, 182 }, // und_Latn_PG -> tpi_Latn_PG
    {   0,   0, 182 }, { 294,  66, 182 }, // und_PG -> tpi_Latn_PG
    {   0,  66, 183 }, {  97,  66, 183 }, // und_Latn_PY -> gn_Latn_PY
    {   0,   0, 183 }, {  97,  66, 183 }, // und_PY -> gn_Latn_PY
    {   0,  66, 184 }, { 270,  66, 184 }, // und_Latn_PE -> es_Latn_PE
    {   0,   0, 184 }, { 270,  66, 184 }, // und_PE -> es_Latn_PE
    {   0,  66, 185 }, {  83,  66, 185 }, // und_Latn_PH -> fil_Latn_PH
    {   0,   0, 185 }, {  83,  66, 185 }, // und_PH -> fil_Latn_PH
    {   0,   0, 186 }, {  75,  66, 186 }, // und_PN -> en_Latn_PN
    {   0,  66, 187 }, { 230,  66, 187 }, // und_Latn_PL -> pl_Latn_PL
    {   0,   0, 187 }, { 230,  66, 187 }, // und_PL -> pl_Latn_PL
    {   0,  66, 188 }, { 231,  66, 188 }, // und_Latn_PT -> pt_Latn_PT
    {   0,   0, 188 }, { 231,  66, 188 }, // und_PT -> pt_Latn_PT
    {   0,  66, 189 }, { 270,  66, 189 }, // und_Latn_PR -> es_Latn_PR
    {   0,   0, 189 }, { 270,  66, 189 }, // und_PR -> es_Latn_PR
    {   0,   0, 190 }, {  14,   4, 190 }, // und_QA -> ar_Arab_QA
    {   0,  66, 191 }, {  85,  66, 191 }, // und_Latn_RE -> fr_Latn_RE
    {   0,   0, 191 }, {  85,  66, 191 }, // und_RE -> fr_Latn_RE
    {   0,  27, 192 }, {  45,  27, 192 }, // und_Cyrl_RO -> bg_Cyrl_RO
    {   0,  66, 192 }, { 235,  66, 192 }, // und_Latn_RO -> ro_Latn_RO
    {   0,   0, 192 }, { 235,  66, 192 }, // und_RO -> ro_Latn_RO
    {   0,   0, 193 }, { 239,  27, 193 }, // und_RU -> ru_Cyrl_RU
    {   0,  66, 194 }, { 138,  66, 194 }, // und_Latn_RW -> rw_Latn_RW
    {   0,   0, 194 }, { 138,  66, 194 }, // und_RW -> rw_Latn_RW
    {   0,  66, 195 }, {  85,  66, 195 }, // und_Latn_BL -> fr_Latn_BL
    {   0,   0, 195 }, {  85,  66, 195 }, // und_BL -> fr_Latn_BL
    {   0,   0, 196 }, {  75,  66, 196 }, // und_SH -> en_Latn_SH
    {   0,   0, 197 }, {  75,  66, 197 }, // und_KN -> en_Latn_KN
    {   0,   0, 198 }, {  75,  66, 198 }, // und_LC -> en_Latn_LC
    {   0,  66, 199 }, {  85,  66, 199 }, // und_Latn_MF -> fr_Latn_MF
    {   0,   0, 199 }, {  85,  66, 199 }, // und_MF -> fr_Latn_MF
    {   0,  66, 200 }, {  85,  66, 200 }, // und_Latn_PM -> fr_Latn_PM
    {   0,   0, 200 }, {  85,  66, 200 }, // und_PM -> fr_Latn_PM
    {   0,   0, 201 }, {  75,  66, 201 }, // und_VC -> en_Latn_VC
    {   0,  66, 202 }, { 244,  66, 202 }, // und_Latn_WS -> sm_Latn_WS
    {   0,   0, 202 }, { 244,  66, 202 }, // und_WS -> sm_Latn_WS
    {   0,  66, 203 }, { 119,  66, 203 }, // und_Latn_SM -> it_Latn_SM
    {   0,   0, 203 }, { 119,  66, 203 }, // und_SM -> it_Latn_SM
    {   0,  66, 204 }, { 231,  66, 204 }, // und_Latn_ST -> pt_Latn_ST
    {   0,   0, 204 }, { 231,  66, 204 }, // und_ST -> pt_Latn_ST
    {   0,   0, 205 }, {  14,   4, 205 }, // und_SA -> ar_Arab_SA
    {   0,  66, 206 }, {  85,  66, 206 }, // und_Latn_SN -> fr_Latn_SN
    {   0,   0, 206 }, {  85,  66, 206 }, // und_SN -> fr_Latn_SN
    {   0,  27, 207 }, { 252,  27, 207 }, // und_Cyrl_RS -> sr_Cyrl_RS
    {   0,  66, 207 }, { 252,  66, 207 }, // und_Latn_RS -> sr_Latn_RS
    {   0,   0, 207 }, { 252,  27, 207 }, // und_RS -> sr_Cyrl_RS
    {   0,  66, 208 }, {  85,  66, 208 }, // und_Latn_SC -> fr_Latn_SC
    {   0,   0, 208 }, {  85,  66, 208 }, // und_SC -> fr_Latn_SC
    {   0,   0, 210 }, {  75,  66, 210 }, // und_SG -> en_Latn_SG
    {   0,   0, 211 }, {  75,  66, 211 }, // und_SX -> en_Latn_SX
    {   0,  27, 212 }, { 303,  27, 212 }, // und_Cyrl_SK -> uk_Cyrl_SK
    {   0,  66, 212 }, { 262,  66, 212 }, // und_Latn_SK -> sk_Latn_SK
    {   0,   0, 212 }, { 262,  66, 212 }, // und_SK -> sk_Latn_SK
    {   0,  66, 213 }, { 263,  66, 213 }, // und_Latn_SI -> sl_Latn_SI
    {   0,   0, 213 }, { 263,  66, 213 }, // und_SI -> sl_Latn_SI
    {   0,   0, 214 }, {  75,  66, 214 }, // und_SB -> en_Latn_SB
    {   0,  66, 215 }, { 265,  66, 215 }, // und_Latn_SO -> so_Latn_SO
    {   0,   0, 215 }, { 265,  66, 215 }, // und_SO -> so_Latn_SO
    {   0,   0, 216 }, {  75,  66, 216 }, // und_ZA -> en_Latn_ZA
    {   0,   0, 217 }, {  75,  66, 217 }, // und_GS -> en_Latn_GS
    {   0,   0, 218 }, { 142,  63, 218 }, // und_KR -> ko_Kore_KR
    {   0,   0, 219 }, {  14,   4, 219 }, // und_SS -> ar_Arab_SS
    {   0,  66, 220 }, { 270,  66, 220 }, // und_Latn_ES -> es_Latn_ES
    {   0,   0, 220 }, { 270,  66, 220 }, // und_ES -> es_Latn_ES
    {   0,   0, 221 }, { 260, 119, 221 }, // und_LK -> si_Sinh_LK
    {   0,   0, 222 }, {  14,   4, 222 }, // und_SD -> ar_Arab_SD
    {   0,  66, 223 }, {  72,  66, 223 }, // und_Latn_SR -> nl_Latn_SR
    {   0,   0, 223 }, {  72,  66, 223 }, // und_SR -> nl_Latn_SR
    {   0,  66, 224 }, { 209,  66, 224 }, // und_Latn_SJ -> nb_Latn_SJ
    {   0,   0, 224 }, { 209,  66, 224 }, // und_SJ -> nb_Latn_SJ
    {   0,  47, 225 }, { 323,  47, 225 }, // und_Hebr_SE -> yi_Hebr_SE
    {   0,  66, 225 }, { 275,  66, 225 }, // und_Latn_SE -> sv_Latn_SE
    {   0,   0, 225 }, { 275,  66, 225 }, // und_SE -> sv_Latn_SE
    {   0,  66, 226 }, {  94,  66, 226 }, // und_Latn_CH -> de_Latn_CH
    {   0,   0, 226 }, {  94,  66, 226 }, // und_CH -> de_Latn_CH
    {   0,  66, 227 }, {  85,  66, 227 }, // und_Latn_SY -> fr_Latn_SY
    {   0,   0, 227 }, {  14,   4, 227 }, // und_SY -> ar_Arab_SY
    {   0,  66, 228 }, { 284,  66, 228 }, // und_Latn_TW -> trv_Latn_TW
    {   0,   0, 228 }, {  58, 137, 228 }, // und_TW -> zh_Hant_TW
    {   0,   4, 229 }, { 228,   4, 229 }, // und_Arab_TJ -> fa_Arab_TJ
    {   0,  27, 229 }, { 282,  27, 229 }, // und_Cyrl_TJ -> tg_Cyrl_TJ
    {   0,   0, 229 }, { 282,  27, 229 }, // und_TJ -> tg_Cyrl_TJ
    {   0,  66, 230 }, { 273,  66, 230 }, // und_Latn_TZ -> sw_Latn_TZ
    {   0,   0, 230 }, { 273,  66, 230 }, // und_TZ -> sw_Latn_TZ
    {   0,   0, 231 }, { 289, 133, 231 }, // und_TH -> th_Thai_TH
    {   0,  66, 232 }, { 231,  66, 232 }, // und_Latn_TL -> pt_Latn_TL
    {   0,   0, 232 }, { 231,  66, 232 }, // und_TL -> pt_Latn_TL
    {   0,  66, 233 }, {  85,  66, 233 }, // und_Latn_TG -> fr_Latn_TG
    {   0,   0, 233 }, {  85,  66, 233 }, // und_TG -> fr_Latn_TG
    {   0,  66, 234 }, { 293,  66, 234 }, // und_Latn_TK -> tkl_Latn_TK
    {   0,   0, 234 }, { 293,  66, 234 }, // und_TK -> tkl_Latn_TK
    {   0,  66, 235 }, { 295,  66, 235 }, // und_Latn_TO -> to_Latn_TO
    {   0,   0, 235 }, { 295,  66, 235 }, // und_TO -> to_Latn_TO
    {   0,   0, 236 }, {  75,  66, 236 }, // und_TT -> en_Latn_TT
    {   0,   0, 237 }, {  75,  66, 237 }, // und_TA -> en_Latn_TA
    {   0,  66, 238 }, {  85,  66, 238 }, // und_Latn_TN -> fr_Latn_TN
    {   0,   0, 238 }, {  14,   4, 238 }, // und_TN -> ar_Arab_TN
    {   0,  66, 239 }, { 298,  66, 239 }, // und_Latn_TR -> tr_Latn_TR
    {   0,   0, 239 }, { 298,  66, 239 }, // und_TR -> tr_Latn_TR
    {   0,  66, 240 }, { 299,  66, 240 }, // und_Latn_TM -> tk_Latn_TM
    {   0,   0, 240 }, { 299,  66, 240 }, // und_TM -> tk_Latn_TM
    {   0,   0, 241 }, {  75,  66, 241 }, // und_TC -> en_Latn_TC
    {   0,  66, 242 }, { 300,  66, 242 }, // und_Latn_TV -> tvl_Latn_TV
    {   0,   0, 242 }, { 300,  66, 242 }, // und_TV -> tvl_Latn_TV
    {   0,  66, 243 }, { 273,  66, 243 }, // und_Latn_UG -> sw_Latn_UG
    {   0,   0, 243 }, { 273,  66, 243 }, // und_UG -> sw_Latn_UG
    {   0,  27, 244 }, { 303,  27, 244 }, // und_Cyrl_UA -> uk_Cyrl_UA
    {   0,  47, 244 }, { 323,  47, 244 }, // und_Hebr_UA -> yi_Hebr_UA
    {   0,  66, 244 }, { 230,  66, 244 }, // und_Latn_UA -> pl_Latn_UA
    {   0,   0, 244 }, { 303,  27, 244 }, // und_UA -> uk_Cyrl_UA
    {   0,   0, 245 }, {  14,   4, 245 }, // und_AE -> ar_Arab_AE
    {   0,   4, 246 }, { 305,   4, 246 }, // und_Arab_GB -> ur_Arab_GB
    {   0,   0, 246 }, {  75,  66, 246 }, // und_GB -> en_Latn_GB
    {   0,   0, 247 }, {  75,  66, 247 }, // und_UM -> en_Latn_UM
    {   0,  47, 248 }, { 323,  47, 248 }, // und_Hebr_US -> yi_Hebr_US
    {   0,   0, 248 }, {  75,  66, 248 }, // und_US -> en_Latn_US
    {   0,   0, 249 }, {  75,  66, 249 }, // und_VI -> en_Latn_VI
    {   0,  66, 250 }, { 270,  66, 250 }, // und_Latn_UY -> es_Latn_UY
    {   0,   0, 250 }, { 270,  66, 250 }, // und_UY -> es_Latn_UY
    {   0,  66, 251 }, { 307,  66, 251 }, // und_Latn_UZ -> uz_Latn_UZ
    {   0,   0, 251 }, { 307,  66, 251 }, // und_UZ -> uz_Latn_UZ
    {   0,  66, 252 }, {  39,  66, 252 }, // und_Latn_VU -> bi_Latn_VU
    {   0,   0, 252 }, {  39,  66, 252 }, // und_VU -> bi_Latn_VU
    {   0,  66, 253 }, { 119,  66, 253 }, // und_Latn_VA -> it_Latn_VA
    {   0,   0, 253 }, { 119,  66, 253 }, // und_VA -> it_Latn_VA
    {   0,  66, 254 }, { 270,  66, 254 }, // und_Latn_VE -> es_Latn_VE
    {   0,   0, 254 }, { 270,  66, 254 }, // und_VE -> es_Latn_VE
    {   0,  66, 255 }, { 310,  66, 255 }, // und_Latn_VN -> vi_Latn_VN
    {   0,   0, 255 }, { 310,  66, 255 }, // und_VN -> vi_Latn_VN
    {   0,  66, 256 }, {  85,  66, 256 }, // und_Latn_WF -> fr_Latn_WF
    {   0,   0, 256 }, {  85,  66, 256 }, // und_WF -> fr_Latn_WF
    {   0,   0, 257 }, {  14,   4, 257 }, // und_EH -> ar_Arab_EH
    {   0,   0, 259 }, {  14,   4, 259 }, // und_YE -> ar_Arab_YE
    {   0,  66, 260 }, {  36,  66, 260 }, // und_Latn_ZM -> bem_Latn_ZM
    {   0,   0, 260 }, {  36,  66, 260 }, // und_ZM -> bem_Latn_ZM
    {   0,  66, 261 }, { 254,  66, 261 }, // und_Latn_ZW -> sn_Latn_ZW
    {   0,   0, 261 }, { 254,  66, 261 }, // und_ZW -> sn_Latn_ZW
    {   0,   1,   0 }, {  87,   1, 102 }, // und_Adlm -> ff_Adlm_GN
    {   0,   4,   0 }, {  14,   4,  71 }, // und_Arab -> ar_Arab_EG
    {   0,   5,   0 }, {  17,   5,  12 }, // und_Armn -> hy_Armn_AM
    {   0,   6,   0 }, {  23,   6, 112 }, // und_Avst -> ae_Avst_IR
    {   0,   7,   0 }, {  27,   7, 111 }, // und_Bali -> ban_Bali_ID
    {   0,   8,   0 }, {  29,   8,  40 }, // und_Bamu -> bax_Bamu_CM
    {   0,   9,   0 }, {  30,   9,  20 }, // und_Beng -> bn_Beng_BD
    {   0,  11,   0 }, {  34,  11, 111 }, // und_Batk -> bbc_Batk_ID
    {   0,  12,   0 }, { 247,  12, 110 }, // und_Bhks -> sa_Bhks_IN
    {   0,  13,   0 }, {  58,  13, 228 }, // und_Bopo -> zh_Bopo_TW
    {   0,  15,   0 }, {  85,  15,  84 }, // und_Brai -> fr_Brai_FR
    {   0,  16,   0 }, {  44,  16, 111 }, // und_Bugi -> bug_Bugi_ID
    {   0,  18,   0 }, { 116,  18,  41 }, // und_Cans -> iu_Cans_CA
    {   0,  21,   0 }, {  52,  21,  20 }, // und_Cakm -> ccp_Cakm_BD
    {   0,  23,   0 }, {  55,  23, 248 }, // und_Cher -> chr_Cher_US
    {   0,  24,   0 }, {  62,  24,  71 }, // und_Copt -> cop_Copt_EG
    {   0,  25,   0 }, {   7,  25, 113 }, // und_Xsux -> akk_Xsux_IQ
    {   0,  26,   0 }, {  13,  26,  63 }, // und_Cprt -> grc_Cprt_CY
    {   0,  27,   0 }, { 239,  27, 193 }, // und_Cyrl -> ru_Cyrl_RU
    {   0,  29,   0 }, { 105,  29, 110 }, // und_Deva -> hi_Deva_IN
    {   0,  30,   0 }, {  85,  30,  84 }, // und_Dupl -> fr_Dupl_FR
    {   0,  31,   0 }, {  12,  31,  71 }, // und_Egyp -> egy_Egyp_EG
    {   0,  32,   0 }, {   9,  32,   3 }, // und_Elba -> sq_Elba_AL
    {   0,  33,   0 }, {  11,  33,  77 }, // und_Ethi -> am_Ethi_ET
    {   0,  35,   0 }, {  93,  35,  90 }, // und_Geor -> ka_Geor_GE
    {   0,  36,   0 }, {  59,  36,  36 }, // und_Glag -> cu_Glag_BG
    {   0,  37,   0 }, {  95,  37, 244 }, // und_Goth -> got_Goth_UA
    {   0,  38,   0 }, { 247,  38, 110 }, // und_Gran -> sa_Gran_IN
    {   0,  39,   0 }, {  96,  39,  94 }, // und_Grek -> el_Grek_GR
    {   0,  40,   0 }, {  98,  40, 110 }, // und_Gujr -> gu_Gujr_IN
    {   0,  41,   0 }, { 233,  41, 110 }, // und_Guru -> pa_Guru_IN
    {   0,  42,   0 }, { 142,  42, 218 }, // und_Hang -> ko_Hang_KR
    {   0,  43,   0 }, {  58,  43,  50 }, // und_Hani -> zh_Hani_CN
    {   0,  45,   0 }, {  58,  45, 228 }, // und_Hanb -> zh_Hanb_TW
    {   0,  47,   0 }, { 103,  47, 116 }, // und_Hebr -> he_Hebr_IL
    {   0,  48,   0 }, { 120,  48, 120 }, // und_Hira -> ja_Hira_JP
    {   0,  49,   0 }, {  16,  49, 112 }, // und_Armi -> arc_Armi_IR
    {   0,  50,   0 }, { 223,  50, 112 }, // und_Phli -> pal_Phli_IR
    {   0,  52,   0 }, { 142,  52, 218 }, // und_Jamo -> ko_Jamo_KR
    {   0,  53,   0 }, { 120,  53, 120 }, // und_Jpan -> ja_Jpan_JP
    {   0,  54,   0 }, { 121,  54, 111 }, // und_Java -> jv_Java_ID
    {   0,  55,   0 }, {  38,  55, 110 }, // und_Kthi -> bho_Kthi_IN
    {   0,  56,   0 }, { 130,  56, 110 }, // und_Knda -> kn_Knda_IN
    {   0,  57,   0 }, { 120,  57, 120 }, // und_Kana -> ja_Kana_JP
    {   0,  60,   0 }, { 135,  60,  39 }, // und_Khmr -> km_Khmr_KH
    {   0,  61,   0 }, { 259,  61, 110 }, // und_Khoj -> sd_Khoj_IN
    {   0,  62,   0 }, { 259,  62, 110 }, // und_Sind -> sd_Sind_IN
    {   0,  63,   0 }, { 142,  63, 218 }, // und_Kore -> ko_Kore_KR
    {   0,  65,   0 }, { 153,  65, 129 }, // und_Laoo -> lo_Laoo_LA
    {   0,  66,   0 }, {  75,  66, 248 }, // und_Latn -> en_Latn_US
    {   0,  70,   0 }, {  13,  70,  94 }, // und_Linb -> grc_Linb_GR
    {   0,  73,   0 }, { 105,  73, 110 }, // und_Mahj -> hi_Mahj_IN
    {   0,  74,   0 }, { 175,  74, 110 }, // und_Mlym -> ml_Mlym_IN
    {   0,  77,   0 }, { 290,  77,  50 }, // und_Marc -> bo_Marc_CN
    {   0,  78,   0 }, { 179,  78, 110 }, // und_Mtei -> mni_Mtei_IN
    {   0,  79,   0 }, { 187,  79, 209 }, // und_Mend -> men_Mend_SL
    {   0,  82,   0 }, { 183,  82, 110 }, // und_Modi -> mr_Modi_IN
    {   0,  83,   0 }, { 191,  83,  50 }, // und_Mong -> mn_Mong_CN
    {   0,  86,   0 }, {  46,  86, 161 }, // und_Mymr -> my_Mymr_MM
    {   0,  87,   0 }, {  16,  87, 122 }, // und_Nbat -> arc_Nbat_JO
    {   0,  88,   0 }, { 200,  88, 164 }, // und_Newa -> new_Newa_NP
    {   0,  90,   0 }, { 178,  90, 102 }, // und_Nkoo -> man_Nkoo_GN
    {   0,  91,   0 }, { 215,  91, 110 }, // und_Orya -> or_Orya_IN
    {   0,  92,   0 }, { 217,  92, 114 }, // und_Ogam -> sga_Ogam_IE
    {   0,  93,   0 }, { 248,  93, 110 }, // und_Olck -> sat_Olck_IN
    {   0,  94,   0 }, { 107,  94, 108 }, // und_Hung -> hu_Hung_HU
    {   0,  97,   0 }, { 139,  97, 193 }, // und_Perm -> kv_Perm_RU
    {   0,  98,   0 }, { 219,  98, 112 }, // und_Xpeo -> peo_Xpeo_IR
    {   0, 101,   0 }, { 221, 101, 248 }, // und_Osge -> osa_Osge_US
    {   0, 102,   0 }, { 265, 102, 215 }, // und_Osma -> so_Osma_SO
    {   0, 104,   0 }, {  16, 104, 227 }, // und_Palm -> arc_Palm_SY
    {   0, 106,   0 }, { 159, 106,  50 }, // und_Phag -> lzh_Phag_CN
    {   0, 107,   0 }, { 229, 107, 132 }, // und_Phnx -> phn_Phnx_LB
    {   0, 109,   0 }, { 223, 109,  50 }, // und_Phlp -> pal_Phlp_CN
    {   0, 111,   0 }, { 218, 111, 225 }, // und_Runr -> non_Runr_SE
    {   0, 113,   0 }, { 250, 113, 110 }, // und_Saur -> saz_Saur_IN
    {   0, 114,   0 }, { 247, 114, 110 }, // und_Shrd -> sa_Shrd_IN
    {   0, 115,   0 }, {  75, 115, 246 }, // und_Shaw -> en_Shaw_GB
    {   0, 116,   0 }, { 247, 116, 110 }, // und_Sidd -> sa_Sidd_IN
    {   0, 117,   0 }, {  10, 117, 248 }, // und_Sgnw -> ase_Sgnw_US
    {   0, 118,   0 }, {  58, 118,  50 }, // und_Hans -> zh_Hans_CN
    {   0, 119,   0 }, { 260, 119, 221 }, // und_Sinh -> si_Sinh_LK
    {   0, 121,   0 }, { 272, 121, 111 }, // und_Sund -> su_Sund_ID
    {   0, 123,   0 }, { 277, 123, 113 }, // und_Syrc -> syr_Syrc_IQ
    {   0, 124,   0 }, {  83, 124, 185 }, // und_Tglg -> fil_Tglg_PH
    {   0, 127,   0 }, { 280, 127, 255 }, // und_Tavt -> blt_Tavt_VN
    {   0, 128,   0 }, {  70, 128, 110 }, // und_Takr -> doi_Takr_IN
    {   0, 129,   0 }, { 283, 129, 110 }, // und_Taml -> ta_Taml_IN
    {   0, 131,   0 }, { 287, 131, 110 }, // und_Telu -> te_Telu_IN
    {   0, 132,   0 }, {  69, 132, 144 }, // und_Thaa -> dv_Thaa_MV
    {   0, 133,   0 }, { 289, 133, 231 }, // und_Thai -> th_Thai_TH
    {   0, 134,   0 }, { 290, 134,  50 }, // und_Tibt -> bo_Tibt_CN
    {   0, 135,   0 }, { 271, 135, 159 }, // und_Tfng -> zgh_Tfng_MA
    {   0, 136,   0 }, { 171, 136, 110 }, // und_Tirh -> mai_Tirh_IN
    {   0, 137,   0 }, {  58, 137, 228 }, // und_Hant -> zh_Hant_TW
    {   0, 138,   0 }, { 302, 138, 227 }, // und_Ugar -> uga_Ugar_SY
    {   0, 139,   0 }, { 308, 139, 134 }, // und_Vaii -> vai_Vaii_LR
    {   0, 141,   0 }, { 255, 141,  50 }, // und_Yiii -> ii_Yiii_CN
    {   0, 142,   0 }, { 339, 142, 161 }  // und_Rohg -> rhg_Rohg_MM
};

static constexpr quint16 locale_index[] = {
     0, // AnyLanguage
     0, // C
     1, // Abkhazian
     2, // Afar
     5, // Afrikaans
     7, // Aghem
     8, // Akan
     0, // Akkadian
     9, // Akoose
    10, // Albanian
     0, // American Sign Language
    13, // Amharic
     0, // Ancient Egyptian
     0, // Ancient Greek
    14, // Arabic
    42, // Aragonese
     0, // Aramaic
    43, // Armenian
    44, // Assamese
    45, // Asturian
    46, // Asu
    47, // Atsam
     0, // Avaric
     0, // Avestan
     0, // Aymara
    48, // Azerbaijani
    53, // Bafia
     0, // Balinese
    54, // Bambara
     0, // Bamun
    56, // Bangla
    58, // Basaa
    59, // Bashkir
    60, // Basque
     0, // Batak Toba
    61, // Belarusian
    62, // Bemba
    63, // Bena
    64, // Bhojpuri
     0, // Bislama
    65, // Blin
    66, // Bodo
    67, // Bosnian
    69, // Breton
     0, // Buginese
    70, // Bulgarian
    71, // Burmese
    72, // Cantonese
    74, // Catalan
    78, // Cebuano
    79, // Central Atlas Tamazight
    80, // Central Kurdish
    82, // Chakma
     0, // Chamorro
    84, // Chechen
    85, // Cherokee
    86, // Chickasaw
    87, // Chiga
    88, // Chinese
    95, // Church
    96, // Chuvash
    97, // Colognian
     0, // Coptic
    98, // Cornish
    99, // Corsican
     0, // Cree
   100, // Croatian
   102, // Czech
   103, // Danish
   105, // Divehi
   106, // Dogri
   107, // Duala
   108, // Dutch
   115, // Dzongkha
   116, // Embu
   117, // English
   226, // Erzya
   227, // Esperanto
   228, // Estonian
   229, // Ewe
   231, // Ewondo
   232, // Faroese
     0, // Fijian
   234, // Filipino
   235, // Finnish
   236, // French
   282, // Friulian
   283, // Fulah
   307, // Gaelic
   308, // Ga
   309, // Galician
   310, // Ganda
   311, // Geez
   313, // Georgian
   314, // German
     0, // Gothic
   321, // Greek
   323, // Guarani
   324, // Gujarati
   325, // Gusii
     0, // Haitian
   326, // Hausa
   331, // Hawaiian
   332, // Hebrew
     0, // Herero
   333, // Hindi
     0, // Hiri Motu
   335, // Hungarian
   336, // Icelandic
   337, // Ido
   338, // Igbo
   339, // Inari Sami
   340, // Indonesian
     0, // Ingush
   341, // Interlingua
   342, // Interlingue
   343, // Inuktitut
     0, // Inupiaq
   345, // Irish
   347, // Italian
   351, // Japanese
   352, // Javanese
   353, // Jju
   354, // Jola-Fonyi
   355, // Kabuverdianu
   356, // Kabyle
   357, // Kako
   358, // Kalaallisut
   359, // Kalenjin
   360, // Kamba
   361, // Kannada
     0, // Kanuri
   362, // Kashmiri
   364, // Kazakh
   365, // Kenyang
   366, // Khmer
   367, // Kiche
   368, // Kikuyu
   369, // Kinyarwanda
     0, // Komi
     0, // Kongo
   370, // Konkani
   371, // Korean
     0, // Koro
   374, // Koyraboro Senni
   375, // Koyra Chiini
   376, // Kpelle
     0, // Kuanyama
   378, // Kurdish
   379, // Kwasio
   380, // Kyrgyz
   381, // Lakota
   382, // Langi
   383, // Lao
   384, // Latin
   385, // Latvian
     0, // Lezghian
     0, // Limburgish
   386, // Lingala
     0, // Literary Chinese
   390, // Lithuanian
   391, // Lojban
   392, // Lower Sorbian
   393, // Low German
   395, // Luba-Katanga
   396, // Lule Sami
   398, // Luo
   399, // Luxembourgish
   400, // Luyia
   401, // Macedonian
   402, // Machame
   403, // Maithili
   404, // Makhuwa-Meetto
   405, // Makonde
   406, // Malagasy
   407, // Malayalam
   408, // Malay
   414, // Maltese
     0, // Mandingo
   415, // Manipuri
   417, // Manx
   418, // Maori
   419, // Mapuche
   420, // Marathi
     0, // Marshallese
   421, // Masai
   423, // Mazanderani
     0, // Mende
   424, // Meru
   425, // Meta
   426, // Mohawk
   427, // Mongolian
   430, // Morisyen
   431, // Mundang
   432, // Muscogee
   433, // Nama
     0, // Nauru
   434, // Navajo
     0, // Ndonga
   435, // Nepali
     0, // Newari
   437, // Ngiemboon
   438, // Ngomba
   439, // Nigerian Pidgin
   440, // Nko
   441, // Northern Luri
   443, // Northern Sami
   446, // Northern Sotho
   447, // North Ndebele
   448, // Norwegian Bokmal
   450, // Norwegian Nynorsk
   451, // Nuer
   452, // Nyanja
   453, // Nyankole
   454, // Occitan
   456, // Odia
     0, // Ojibwa
     0, // Old Irish
     0, // Old Norse
     0, // Old Persian
   457, // Oromo
   459, // Osage
   460, // Ossetic
     0, // Pahlavi
     0, // Palauan
     0, // Pali
   462, // Papiamento
   464, // Pashto
   466, // Persian
     0, // Phoenician
   468, // Polish
   469, // Portuguese
   481, // Prussian
   482, // Punjabi
   484, // Quechua
   487, // Romanian
   489, // Romansh
   490, // Rombo
   491, // Rundi
   492, // Russian
   498, // Rwa
   499, // Saho
   500, // Sakha
   501, // Samburu
     0, // Samoan
   502, // Sango
   503, // Sangu
   504, // Sanskrit
   505, // Santali
   507, // Sardinian
     0, // Saurashtra
   508, // Sena
   509, // Serbian
   517, // Shambala
   518, // Shona
   519, // Sichuan Yi
   520, // Sicilian
   521, // Sidamo
   522, // Silesian
   523, // Sindhi
   525, // Sinhala
   526, // Skolt Sami
   527, // Slovak
   528, // Slovenian
   529, // Soga
   530, // Somali
   534, // Southern Kurdish
   536, // Southern Sami
   538, // Southern Sotho
   540, // South Ndebele
   541, // Spanish
   569, // Standard Moroccan Tamazight
   570, // Sundanese
   571, // Swahili
   575, // Swati
   577, // Swedish
   580, // Swiss German
   583, // Syriac
   585, // Tachelhit
     0, // Tahitian
   587, // Tai Dam
   588, // Taita
   589, // Tajik
   590, // Tamil
   594, // Taroko
   595, // Tasawaq
   596, // Tatar
   597, // Telugu
   598, // Teso
   600, // Thai
   601, // Tibetan
   603, // Tigre
   604, // Tigrinya
     0, // Tokelau
   606, // Tok Pisin
   607, // Tongan
   608, // Tsonga
   609, // Tswana
   611, // Turkish
   613, // Turkmen
     0, // Tuvalu
   614, // Tyap
     0, // Ugaritic
   615, // Ukrainian
   616, // Upper Sorbian
   617, // Urdu
   619, // Uyghur
   620, // Uzbek
   623, // Vai
   625, // Venda
   626, // Vietnamese
   627, // Volapuk
   628, // Vunjo
   629, // Walloon
   630, // Walser
   631, // Warlpiri
   632, // Welsh
   633, // Western Balochi
   638, // Western Frisian
   639, // Wolaytta
   640, // Wolof
   641, // Xhosa
   642, // Yangben
   643, // Yiddish
   644, // Yoruba
   646, // Zarma
   647, // Zhuang
   648, // Zulu
   649, // Kaingang
   650, // Nheengatu
   653, // Haryanvi
   654, // Northern Frisian
   655, // Rajasthani
   656, // Moksha
   657, // Toki Pona
   658, // Pijin
   659, // Obolo
   660, // Baluchi
   662, // Ligurian
   663, // Rohingya
   665, // Torwali
   666, // Anii
   667, // Kangri
   668, // Venetian
   669, // Kuvi
     0 // trailing 0
};

static constexpr QLocaleData locale_data[] = {
   //  lang  script   terr lStrt lpMid lpEnd lPair lDelm  dec  group prcnt  zero minus plus   exp  qtOpn qtEnd altQO altQE lDFmt sDFmt lTFmt sTFmt slDay lDays ssDys sDays snDay nDays   am    pm   byte siQnt iecQn crSym crDsp crFmt crFNg ntLng ntTer                                                                                                                                                        currISO   curDgt curRnd dow1st  wknd+  wknd- grpTop grpMid grpEnd
    {      1,     0,     0,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,    6,    7,    7,    8,    8,    0,   17,    0,    0,    0,    0,   56,   56,   83,   96,    0,    0,    0,    5,   22,    0,    0,    0,    0,    0,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 10,  8, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  0,  0,  4,  0,  0,  0,    {0,0,0},      2,     1,     1,     6,     7,     1,     3,     3 }, // C/AnyScript/AnyTerritory
    {      2,    27,    90,    0,    0,    7,    7,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14,   27,   49,   10,    0,  109,  109,  157,  157,  179,  179,    0,    0,    0,    5,   22,    0,    0,    4,    0,    0,    6,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22, 10, 13,  5, 48, 48, 22, 22, 15, 15,  2,  2,  4, 17, 23,  1,  0,  5,  0,  6,  9, {71,69,76},      2,     1,     1,     6,     7,     1,     3,     3 }, // Abkhazian/Cyrillic/Georgia
    {      3,    66,    77,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,   59,   78,   23,   38,  194,  194,  245,  245,  272,  272,    0,    0,    0,    5,   22,    1,    0,    2,    0,   15,   20,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  8, 15,  7, 51, 51, 27, 27, 13, 13,  2,  2,  4, 17, 23,  2,  0,  4,  0,  5,  7, {69,84,66},      2,     1,     7,     6,     7,     1,     3,     3 }, // Afar/Latin/Ethiopia
    {      3,    66,    67,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,   59,   78,   23,   38,  194,  194,  245,  245,  272,  272,    0,    0,    0,    5,   22,    3,    0,    2,    0,   15,   27,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  8, 15,  7, 51, 51, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3,  0,  4,  0,  5,  7, {68,74,70},      0,     0,     6,     6,     7,     1,     3,     3 }, // Afar/Latin/Djibouti
    {      3,    66,    74,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,   59,   78,   23,   38,  194,  194,  245,  245,  272,  272,    0,    0,    0,    5,   22,    6,    0,    2,    0,   15,   34,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  8, 15,  7, 51, 51, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3,  0,  4,  0,  5,  7, {69,82,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Afar/Latin/Eritrea
    {      4,    66,   216,    0,    0,   16,   16,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17,   86,  103,   10,    0,  285,  285,  342,  342,  369,  369,    2,    2,   45,    5,   22,    9,    0,    2,    9,   41,   50,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 57, 57, 27, 27, 13, 13,  3,  3,  5, 17, 23,  1, 20,  4,  6,  9, 11, {90,65,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // Afrikaans/Latin/South Africa
    {      4,    66,   162,    0,    0,   16,   16,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  103,   23,   38,  285,  285,  342,  342,  369,  369,    2,    2,   45,    5,   22,   10,   20,    2,    9,   41,   61,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 15,  7, 57, 57, 27, 27, 13, 13,  3,  3,  5, 17, 23,  1, 16,  4,  6,  9,  7, {78,65,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Afrikaans/Latin/Namibia
    {      5,    66,    40,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   18,   17,  113,  129,   10,    0,  382,  382,  453,  453,  480,  480,    5,    5,    0,    5,   22,   11,   36,    0,    0,   68,   73,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 71, 71, 27, 27, 13, 13,  3,  3,  4, 17, 23,  4, 14,  4,  0,  5,  7, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Aghem/Latin/Cameroon
    {      6,    66,    92,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  137,  155,   23,   38,  493,  493,  541,  541,  568,  568,    8,    8,    0,    5,   22,   15,   50,    2,    0,   80,   84,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 15,  7, 48, 48, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3, 10,  4,  0,  4,  5, {71,72,83},      2,     1,     1,     6,     7,     1,     3,     3 }, // Akan/Latin/Ghana
    {      8,    66,    40,    0,    0,   24,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,   11,   60,   15,    0,   89,   95,  6,  6,  5,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  4, 10,  5,  0,  6,  7, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Akoose/Latin/Cameroon
    {      9,    66,     3,    0,    0,   29,   29,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,    0,  180,   45,   38,  581,  581,  638,  638,  665,  665,   10,   10,   50,    5,   22,   18,   70,    4,   20,  102,  107,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 16,  7, 57, 57, 27, 27, 14, 14, 11, 10,  4, 17, 23,  4, 13,  5,  7,  5,  8, {65,76,76},      0,     0,     1,     6,     7,     2,     3,     3 }, // Albanian/Latin/Albania
    {      9,    66,   126,    0,    0,   29,   29,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,    0,  180,   10,    0,  581,  581,  638,  638,  665,  665,   10,   10,   50,    5,   22,   22,   83,    4,   20,  102,  115,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 13,  5, 57, 57, 27, 27, 14, 14, 11, 10,  4, 17, 23,  1,  6,  5,  7,  5,  6, {69,85,82},      2,     1,     1,     6,     7,     2,     3,     3 }, // Albanian/Latin/Kosovo
    {      9,    66,   140,    0,    0,   29,   29,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,    0,  180,   10,    0,  581,  581,  638,  638,  665,  665,   10,   10,   50,    5,   22,   23,   89,    4,   20,  102,  121,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 13,  5, 57, 57, 27, 27, 14, 14, 11, 10,  4, 17, 23,  3, 16,  5,  7,  5, 18, {77,75,68},      2,     1,     1,     6,     7,     2,     3,     3 }, // Albanian/Latin/Macedonia
    {     11,    33,    77,   38,   38,   44,   53,    6,    0,    1,    2,    3,    4,    5,   10,   11,   12,   19,   20,  163,  186,   61,   76,  679,  679,  706,  706,  732,  732,   21,   20,   54,   57,   22,   26,  105,    2,    9,  139,  143,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 27, 27, 26, 26, 13, 13,  3,  4,  3, 23, 23,  2,  9,  4,  6,  4,  5, {69,84,66},      2,     1,     7,     6,     7,     1,     3,     3 }, // Amharic/Ethiopic/Ethiopia
    {     14,     4,    71,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   28,  114,   27,    0,  148,  155,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5,  9,  6,  0,  7,  3, {69,71,80},      2,     1,     6,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Egypt
    {     14,     4,     4,   61,   61,   61,   61,    6,    1,    0,   32,    3,   35,   37,   10,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   33,  123,   33,   38,  148,  158,  6,  6,  6,  6,  1,  1,  1,  3,  1,  2,  2,  1,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 12,  5,  7,  7,  7, {68,90,68},      2,     1,     6,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Algeria
    {     14,     4,    19,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   38,  135,   27,    0,  148,  165,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 12,  6,  0,  7,  7, {66,72,68},      3,     0,     6,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Bahrain
    {     14,     4,    48,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   11,  147,   27,    0,  148,  172,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  4, 15,  6,  0,  7,  4, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Arabic/Arabic/Chad
    {     14,     4,    55,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   10,    0,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   12,  162,   27,    0,  148,  176,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 13,  5, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  2, 14,  6,  0,  7,  9, {75,77,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Arabic/Arabic/Comoros
    {     14,     4,    67,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,    3,  176,   27,    0,  148,  185,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  3, 11,  6,  0,  7,  6, {68,74,70},      0,     0,     6,     6,     7,     1,     3,     3 }, // Arabic/Arabic/Djibouti
    {     14,     4,    74,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,    6,  187,   27,    0,  148,  191,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  3, 12,  6,  0,  7,  7, {69,82,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Arabic/Arabic/Eritrea
    {     14,     4,   113,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   43,  199,   27,    0,  148,  198,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 11,  6,  0,  7,  6, {73,81,68},      0,     0,     6,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Iraq
    {     14,     4,   116,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   11,    1,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   48,  210,   27,    0,  148,  204,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 12,  4, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  1, 18,  6,  0,  7,  7, {73,76,83},      2,     1,     7,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Israel
    {     14,     4,   122,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   49,  228,   27,    0,  148,  211,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 11,  6,  0,  7,  6, {74,79,68},      3,     0,     6,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Jordan
    {     14,     4,   127,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   54,  239,   27,    0,  148,  217,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 11,  6,  0,  7,  6, {75,87,68},      3,     0,     6,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Kuwait
    {     14,     4,   132,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   59,  250,   27,    0,  148,  223,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 11,  6,  0,  7,  5, {76,66,80},      0,     0,     1,     6,     7,     1,     3,     3 }, // Arabic/Arabic/Lebanon
    {     14,     4,   135,   61,   61,   61,   61,    6,    1,    0,   32,    3,   35,   37,   10,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   64,  261,   33,   38,  148,  228,  6,  6,  6,  6,  1,  1,  1,  3,  1,  2,  2,  1,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 10,  5,  7,  7,  5, {76,89,68},      3,     0,     6,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Libya
    {     14,     4,   149,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   69,  271,   27,    0,  148,  233,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  4, 15,  6,  0,  7,  9, {77,82,85},      2,     1,     1,     6,     7,     1,     3,     3 }, // Arabic/Arabic/Mauritania
    {     14,     4,   159,   61,   61,   61,   61,    6,    1,    0,   32,    3,   35,   37,   10,   15,   14,   17,   16,  196,  213,   10,    0,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   73,  286,   33,   38,  148,  242,  6,  6,  6,  6,  1,  1,  1,  3,  1,  2,  2,  1,  1,  1,  1,  1, 17, 10, 13,  5, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 10,  5,  7,  7,  6, {77,65,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Arabic/Arabic/Morocco
    {     14,     4,   176,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   78,  296,   27,    0,  148,  248,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 10,  6,  0,  7,  5, {79,77,82},      3,     0,     6,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Oman
    {     14,     4,   180,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   48,  210,   27,    0,  148,  253,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  1, 18,  6,  0,  7, 18, {73,76,83},      2,     1,     1,     6,     7,     1,     3,     3 }, // Arabic/Arabic/Palestinian Territories
    {     14,     4,   190,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   83,  306,   27,    0,  148,  271,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5,  9,  6,  0,  7,  3, {81,65,82},      2,     1,     6,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Qatar
    {     14,     4,   205,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   88,  315,   27,    0,  148,  274,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 10,  6,  0,  7, 24, {83,65,82},      2,     1,     7,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Saudi Arabia
    {     14,     4,   215,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   93,  325,   27,    0,  148,  298,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  1, 10,  6,  0,  7,  7, {83,79,83},      0,     0,     1,     6,     7,     1,     3,     3 }, // Arabic/Arabic/Somalia
    {     14,     4,   219,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   94,  335,   27,    0,  148,  305,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  1, 17,  6,  0,  7, 12, {83,83,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // Arabic/Arabic/South Sudan
    {     14,     4,   222,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   95,  352,   27,    0,  148,  317,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  4, 11,  6,  0,  7,  7, {83,68,71},      2,     1,     6,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Sudan
    {     14,     4,   227,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   99,  363,   27,    0,  148,  324,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 10,  6,  0,  7,  5, {83,89,80},      0,     0,     6,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Syria
    {     14,     4,   238,   61,   61,   61,   61,    6,    1,    0,   32,    3,   35,   37,   10,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,  104,  373,   33,   38,  148,  329,  6,  6,  6,  6,  1,  1,  1,  3,  1,  2,  2,  1,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 11,  5,  7,  7,  4, {84,78,68},      3,     0,     1,     6,     7,     1,     3,     3 }, // Arabic/Arabic/Tunisia
    {     14,     4,   245,   61,   61,   61,   61,    6,    0,    1,   32,    3,   35,   37,   10,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,  109,  384,   33,   38,  148,  333,  6,  6,  6,  6,  1,  1,  1,  3,  1,  2,  2,  1,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 12,  5,  7,  7, 24, {65,69,68},      2,     1,     6,     6,     7,     1,     3,     3 }, // Arabic/Arabic/United Arab Emirates
    {     14,     4,   257,   61,   61,   61,   61,    6,    0,    1,   32,    3,   35,   37,   10,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,   73,  286,   33,   38,  148,  357,  6,  6,  6,  6,  1,  1,  1,  3,  1,  2,  2,  1,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5, 10,  5,  7,  7, 15, {77,65,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Arabic/Arabic/Western Sahara
    {     14,     4,   258,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,    0,    0,   27,    0,  372,  394,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  0,  0,  6,  0, 22,  6,    {0,0,0},      2,     1,     1,     6,     7,     1,     3,     3 }, // Arabic/Arabic/world
    {     14,     4,   259,   61,   61,   61,   61,   67,   21,   22,   23,   25,   26,   28,   30,   15,   14,   17,   16,  196,  213,   61,   76,  745,  745,  745,  745,  796,  796,   24,   24,   80,   84,   22,  114,  396,   27,    0,  148,  400,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 51, 51, 51, 51, 13, 13,  1,  1,  4, 37, 23,  5,  9,  6,  0,  7,  5, {89,69,82},      0,     0,     7,     5,     6,     1,     3,     3 }, // Arabic/Arabic/Yemen
    {     15,    66,   220,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   15,   15,  223,  129,   11,    1,  809,  809,  860,  860,  887,  887,    0,    0,    0,    5,   22,   22,  405,    2,    9,  405,  413,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22,  6, 12,  4, 51, 51, 27, 27, 16, 16,  2,  2,  4, 17, 23,  1,  4,  4,  6,  8,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Aragonese/Latin/Spain
    {     17,     5,    12,    0,    0,   75,   75,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   11,   12,  245,   49,   10,    0,  903,  903,  964,  964,  991,  991,    0,    0,  121,  127,   22,  119,  409,    4,    0,  420,  427,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 20,  8, 13,  5, 61, 61, 27, 27, 13, 13,  2,  2,  6, 17, 23,  1, 13,  5,  0,  7,  8, {65,77,68},      2,     0,     1,     6,     7,     1,     3,     3 }, // Armenian/Armenian/Armenia
    {     18,     9,   110,    0,    0,   82,   82,    6,    0,    1,    2,   39,    4,    5,   10,   14,   15,   16,   17,  265,  283,   83,   83, 1004, 1004, 1061, 1061, 1092, 1092,   25,   25,  144,  148,   22,  120,  422,    2,    9,  435,  442,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 15,  7, 57, 57, 31, 31, 13, 13,  9,  7,  4, 37, 23,  1, 12,  4,  6,  7,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Assamese/Bangla/India
    {     19,    66,   220,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   14,   15,  223,  129,   10,    0, 1105, 1105, 1158, 1158, 1185, 1185,   34,   32,    0,    5,   22,   22,  405,    4,    0,  446,  455,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22,  6, 13,  5, 53, 53, 27, 27, 13, 13, 12, 11,  5, 17, 23,  1,  4,  5,  0,  9,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Asturian/Latin/Spain
    {     20,    66,   230,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0, 1198, 1198, 1257, 1257, 1284, 1284,   46,   43,    0,    5,   22,  121,  434,    4,    0,  461,  467,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 59, 59, 27, 27, 13, 13,  9,  8,  4, 17, 23,  3, 21,  5,  0,  6,  8, {84,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Asu/Latin/Tanzania
    {     21,    66,   169,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  137,  155,   10,    0, 1297, 1297, 1383, 1383,   83,   83,    0,    0,    0,    5,   22,  124,  455,   15,    0,  475,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 86, 86, 33, 33, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  0,  5,  0, {78,71,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Atsam/Latin/Nigeria
    {     25,    66,    17,    0,    0,   91,   91,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  291,   49,   10,    0, 1416, 1416, 1482, 1508,   96,   96,    0,    0,  185,    5,   22,  125,  459,    4,    0,  480,  490,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 13,  5, 66, 66, 26, 26, 13, 13,  2,  2,  4, 17, 23,  1, 17,  5,  0, 10, 10, {65,90,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Azerbaijani/Latin/Azerbaijan
    {     25,     4,   112,    0,    0,    0,    0,   67,   21,   22,   23,   40,   35,   41,   44,   11,   12,   19,   20,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,    0,    0,   15,    0,  500,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  0,  0,  5,  0,  6,  0, {73,82,82},      0,     0,     6,     5,     5,     1,     3,     3 }, // Azerbaijani/Arabic/Iran
    {     25,     4,   113,    0,    0,    0,    0,   67,   21,   22,   23,   40,   35,   41,   44,   11,   12,   19,   20,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,    0,    0,   15,    0,  500,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  0,  0,  5,  0,  6,  0, {73,81,68},      0,     0,     6,     5,     6,     1,     3,     3 }, // Azerbaijani/Arabic/Iraq
    {     25,     4,   239,    0,    0,    0,    0,   67,   21,   22,   23,   40,   35,   41,   44,   11,   12,   19,   20,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  126,    0,   15,    0,  500,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  6,  0, {84,82,89},      2,     1,     1,     6,     7,     1,     3,     3 }, // Azerbaijani/Arabic/Turkey
    {     25,    27,    17,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   19,   20,  291,   49,   10,    0, 1534, 1534, 1600, 1600,   96,   96,   55,   51,    0,    5,   22,  125,  476,    4,    0,  506,  516,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 13,  5, 66, 66, 26, 26, 13, 13,  2,  2,  4, 17, 23,  1,  5,  5,  0, 10, 10, {65,90,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Azerbaijani/Cyrillic/Azerbaijan
    {     26,    66,    40,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   16,   17,  113,  129,   10,    0, 1626, 1626, 1670, 1670, 1698, 1698,   57,   53,    0,    5,   22,   11,  481,    4,    0,  526,  531,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 44, 44, 28, 28, 13, 13,  6,  7,  4, 17, 23,  4,  4,  5,  0,  5,  7, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Bafia/Latin/Cameroon
    {     28,    66,   145,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   11,   12,   14,   15,  113,  129,   10,    0, 1711, 1711, 1754, 1754, 1781, 1781,    0,    0,    0,    5,   22,  127,  485,    2,    9,  538,  547,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 43, 43, 27, 27, 13, 13,  2,  2,  4, 17, 23,  5, 17,  4,  6,  9,  4, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Bambara/Latin/Mali
    {     28,    90,   145,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  129,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  127,    0,    2,    9,    0,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  5,  0,  4,  6,  0,  0, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Bambara/Nko/Mali
    {     30,     9,    20,    0,    0,   99,   99,    6,    0,    1,    2,   39,    4,    5,   10,   14,   15,   16,   17,  265,  129,   61,   76, 1794, 1794, 1851, 1851, 1887, 1887,    0,    0,  144,    5,   22,  132,  502,    0,   45,  551,  556,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 57, 57, 36, 36, 17, 17,  2,  2,  4, 17, 23,  1, 14,  4,  6,  5,  8, {66,68,84},      2,     1,     7,     6,     7,     1,     2,     3 }, // Bangla/Bangla/Bangladesh
    {     30,     9,   110,    0,    0,   99,   99,    6,    0,    1,    2,   39,    4,    5,   10,   14,   15,   16,   17,  265,  129,   61,   76, 1794, 1794, 1851, 1851, 1887, 1887,    0,    0,  144,    5,   22,  120,  516,    2,    9,  551,  564,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 57, 57, 36, 36, 17, 17,  2,  2,  4, 17, 23,  1, 12,  4,  6,  5,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Bangla/Bangla/India
    {     31,    66,    40,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14,  113,  129,   10,    0, 1904, 1904, 1973, 1973, 2000, 2000,   63,   60,    0,    5,   22,   11,  528,    4,    0,  568,  573,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 69, 69, 27, 27, 13, 13, 10,  9,  4, 17, 23,  4, 15,  5,  0,  5,  8, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Basaa/Latin/Cameroon
    {     32,    27,   193,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  133,    0,   15,    0,  581,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0, 12,  0, {82,85,66},      2,     1,     1,     6,     7,     1,     3,     3 }, // Bashkir/Cyrillic/Russia
    {     33,    66,   220,    0,    0,  108,  108,    6,    1,    0,    2,    3,   48,    5,   10,   11,   12,   14,   15,  308,  344,   98,    0, 2013, 2013, 2080, 2080, 2107, 2107,    0,    0,  189,    5,   22,   22,  543,    4,   20,  593,  600,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 36,  6, 15,  5, 67, 67, 27, 27, 13, 13,  2,  2,  7, 17, 23,  1,  5,  5,  7,  7,  8, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Basque/Latin/Spain
    {     35,    27,    22,    0,    0,  117,  117,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14,  350,   50,  113,    0, 2120, 2120, 2175, 2175, 2195, 2195,    0,    0,  196,  201,   22,    1,  548,    4,    0,  608,  618,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22,  7, 14,  5, 55, 55, 20, 20, 13, 13,  2,  2,  5, 17, 23,  2, 16,  5,  0, 10,  8, {66,89,78},      2,     0,     1,     6,     7,     2,     3,     3 }, // Belarusian/Cyrillic/Belarus
    {     36,    66,   260,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38, 2208, 2208, 2208, 2208,   83,   83,   73,   69,    0,    5,   22,  134,    0,    2,    9,  626,  635,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 79, 79, 79, 79, 13, 13,  8,  7,  4, 17, 23,  1,  0,  4,  6,  9,  6, {90,77,87},      2,     1,     1,     6,     7,     1,     3,     3 }, // Bemba/Latin/Zambia
    {     37,    66,   230,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0, 2287, 2287, 2368, 2368, 2395, 2395,   81,   76,    0,    5,   22,  121,  564,    0,    0,  641,  647,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 81, 81, 27, 27, 13, 13,  7,  7,  4, 17, 23,  3, 22,  4,  0,  6, 10, {84,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Bena/Latin/Tanzania
    {     38,    29,   110,    0,    0,    0,    0,    6,    0,    1,    2,   49,    4,    5,   10,   14,   15,   16,   17,  163,  103,   61,   76, 2408, 2408, 2408, 2408,   83,   83,   88,   83,    0,    5,   22,  120,    0,    2,    0,  657,  664,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 55, 55, 55, 55, 13, 13,  3,  4,  4, 17, 23,  1,  0,  4,  0,  7,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Bhojpuri/Devanagari/India
    {     40,    33,    74,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  372,   78,   61,   76, 2463, 2463, 2505, 2505, 2530, 2530,    0,    0,    0,    5,   22,    6,    0,    2,    0,  668,  671,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22,  8, 15,  7, 42, 42, 25, 25, 13, 13,  2,  2,  4, 17, 23,  3,  0,  4,  0,  3,  4, {69,82,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Blin/Ethiopic/Eritrea
    {     41,    29,   110,    0,    0,  124,  134,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  291,  394,  127,  142, 2543, 2597, 2650, 2650, 2682, 2682,   91,   87,    0,    5,   22,  120,  586,    2,    9,  675,  664,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15, 10, 54, 53, 32, 32, 17, 17,  3,  6,  4, 17, 23,  1, 11,  4,  6,  3,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Bodo/Devanagari/India
    {     42,    66,    29,    0,    0,  143,  143,    6,    1,    0,    2,    3,    4,    5,   10,   13,   15,   16,   17,  404,  423,   10,    0, 2699, 2699, 2756, 2756, 2783, 2796,   94,   93,  218,    5,   22,  135,  597,    4,    0,  678,  686,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19, 11, 13,  5, 57, 57, 27, 27, 13, 13, 10,  7,  7, 17, 23,  2, 40,  5,  0,  8, 19, {66,65,77},      2,     1,     1,     6,     7,     1,     3,     3 }, // Bosnian/Latin/Bosnia and Herzegovina
    {     42,    27,    29,    0,    0,  150,  150,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   18,   16,  434,  454,   10,    0, 2809, 2809, 2864, 2864, 2891, 2891,  104,  100,    0,    5,   22,  137,  637,    4,    0,  705,  713,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 20,  7, 13,  5, 55, 55, 27, 27, 13, 13, 11, 13,  4, 17, 23,  2, 19,  5,  0,  8, 19, {66,65,77},      2,     1,     1,     6,     7,     1,     3,     3 }, // Bosnian/Cyrillic/Bosnia and Herzegovina
    {     43,    66,    84,    0,    0,  157,  157,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  113,  186,   10,    0, 2904, 2904, 2946, 2946, 2978, 2978,  115,  113,  225,  232,  249,   22,  405,    4,    0,  732,  741,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 42, 42, 32, 32, 17, 17,  4,  4,  7, 17, 23,  1,  4,  5,  0,  9,  5, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Breton/Latin/France
    {     45,    27,    36,    0,    0,  150,  150,    6,    1,    9,    2,    3,    4,    5,   10,   13,   14,   13,   14,  350,  461,  152,    1, 2995, 2995, 3049, 3049, 3069, 3069,  119,  117,  272,    5,   22,  139,  656,    4,   20,  746,  755,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22, 12, 17,  4, 54, 54, 20, 20, 13, 13,  6,  6,  7, 17, 23,  3, 13,  5,  7,  9,  8, {66,71,78},      2,     1,     1,     6,     7,     2,     3,     3 }, // Bulgarian/Cyrillic/Bulgaria
    {     46,    86,   161,  165,  165,  172,  172,  182,    0,    1,    2,   50,    4,    5,   10,   14,   15,   16,   17,  473,  129,  169,    1, 3082, 3082, 3082, 3082, 3135, 3135,  125,  123,  279,    5,   22,  134,  669,   15,    0,  763,  763,  7,  7, 10, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 13,  4, 53, 53, 53, 53, 13, 13,  5,  3,  5, 17, 23,  1, 11,  5,  0,  6,  6, {77,77,75},      0,     0,     7,     6,     7,     1,     3,     3 }, // Burmese/Myanmar/Myanmar
    {     47,   137,   107,  183,  183,  188,  188,    6,    0,    1,    2,    3,    4,    5,   10,   51,   52,   53,   54,  491,  505,  182,   43, 3148, 3148, 3148, 3148, 3175, 3175,  130,  126,    0,    5,   22,  142,  680,    2,    9,  769,  771,  5,  5,  5,  5,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 14,  8, 16,  6, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3,  2,  4,  6,  2, 14, {72,75,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Cantonese/Traditional Han/Hong Kong
    {     47,   118,    50,  183,  183,  188,  188,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  513,  505,  169,    0, 3148, 3148, 3188, 3188, 3175, 3175,  130,  126,    0,    5,   22,  145,  682,    2,    9,  785,  787,  5,  5,  5,  5,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 13,  8, 13,  5, 27, 27, 20, 20, 13, 13,  2,  2,  4, 17, 23,  1,  3,  4,  6,  2,  7, {67,78,89},      2,     1,     1,     6,     7,     1,     3,     3 }, // Cantonese/Simplified Han/China
    {     48,    66,   220,    0,    0,  143,  143,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   14,   15,  526,  129,   99,    1, 3208, 3208, 3267, 3267, 3267, 3267,  132,  128,    0,    5,   22,   22,  405,    4,   20,  794,  413,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  6, 14,  4, 59, 59, 27, 27, 27, 27,  5,  5,  5, 17, 23,  1,  4,  5,  7,  6,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Catalan/Latin/Spain
    {     48,    66,     6,    0,    0,  143,  143,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   14,   15,  526,  129,   99,    1, 3208, 3208, 3267, 3267, 3267, 3267,  132,  128,    0,    5,   22,   22,  405,    4,   20,  794,  800,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  6, 14,  4, 59, 59, 27, 27, 27, 27,  5,  5,  5, 17, 23,  1,  4,  5,  7,  6,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Catalan/Latin/Andorra
    {     48,    66,    84,    0,    0,  143,  143,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   14,   15,  526,  129,   99,    1, 3208, 3208, 3267, 3267, 3267, 3267,  132,  128,    0,    5,   22,   22,  405,    4,   20,  794,  807,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  6, 14,  4, 59, 59, 27, 27, 27, 27,  5,  5,  5, 17, 23,  1,  4,  5,  7,  6,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Catalan/Latin/France
    {     48,    66,   117,    0,    0,  143,  143,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   14,   15,  526,  129,   99,    1, 3208, 3208, 3267, 3267, 3267, 3267,  132,  128,    0,    5,   22,   22,  405,    4,   20,  794,  813,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  6, 14,  4, 59, 59, 27, 27, 27, 27,  5,  5,  5, 17, 23,  1,  4,  5,  7,  6,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Catalan/Latin/Italy
    {     49,    66,   185,    0,    0,  193,  202,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38, 3294, 3294, 3349, 3349, 3376, 3376,    0,    0,  284,    5,   22,  146,  685,    2,    9,  819,  826,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 55, 55, 27, 27, 13, 13,  2,  2,  8, 17, 23,  1, 15,  4,  6,  7,  9, {80,72,80},      2,     1,     7,     6,     7,     1,     3,     3 }, // Cebuano/Latin/Philippines
    {     50,    66,   159,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0, 3389, 3389, 3436, 3436, 3463, 3463,  137,  133,    0,    5,   22,    0,  700,    4,    0,  835,  852,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 47, 47, 27, 27, 13, 13,  9, 10,  4, 17, 23,  0, 15,  5,  0, 17,  6, {77,65,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Central Atlas Tamazight/Latin/Morocco
    {     51,     4,   113,    0,    0,    0,    0,   67,   21,   22,   23,   25,   55,   57,   59,   14,   15,   16,   17,  163,  103,   61,   76, 3476, 3476, 3476, 3476, 3533, 3533,  146,  143,    0,    5,   22,   43,  715,    4,    0,  858,  872,  6,  6,  6,  6,  1,  1,  1,  1,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 15,  7, 57, 57, 57, 57, 13, 13,  3,  3,  4, 17, 23,  5, 13,  5,  0, 14,  5, {73,81,68},      0,     0,     6,     5,     6,     1,     3,     3 }, // Central Kurdish/Arabic/Iraq
    {     51,     4,   112,    0,    0,    0,    0,   67,   21,   22,   23,   25,   55,   57,   59,   14,   15,   16,   17,  163,  103,   10,    0, 3476, 3476, 3476, 3476, 3533, 3533,  146,  143,    0,    5,   22,    0,  728,    4,    0,  858,  877,  6,  6,  6,  6,  1,  1,  1,  1,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 13,  5, 57, 57, 57, 57, 13, 13,  3,  3,  4, 17, 23,  0, 12,  5,  0, 14,  5, {73,82,82},      0,     0,     6,     5,     5,     1,     3,     3 }, // Central Kurdish/Arabic/Iran
    {     52,    21,    20,    0,    0,  210,  210,    6,    0,    1,    2,   61,    4,    5,   10,   14,   15,   16,   17,  265,  129,   61,   76, 3546, 3546, 3672, 3672, 3756, 3756,    0,    0,  292,    5,   22,  132,  740,    0,   45,  882,  894,  6,  6, 12, 12,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7,126,126, 84, 84, 38, 38,  2,  2,  8, 17, 23,  1, 21,  4,  6, 12, 14, {66,68,84},      2,     1,     7,     6,     7,     1,     2,     3 }, // Chakma/Chakma/Bangladesh
    {     52,    21,   110,    0,    0,  210,  210,    6,    0,    1,    2,   61,    4,    5,   10,   14,   15,   16,   17,  265,  129,   61,   76, 3546, 3546, 3672, 3672, 3756, 3756,    0,    0,  292,    5,   22,  120,  761,    0,   45,  882,  908,  6,  6, 12, 12,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7,126,126, 84, 84, 38, 38,  2,  2,  8, 17, 23,  1, 27,  4,  6, 12, 10, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Chakma/Chakma/India
    {     54,    27,   193,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0, 3794, 3794, 3838, 3838, 3862, 3838,    0,    0,    0,    5,   22,  133,  788,    4,    0,  918,  925,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 44, 44, 24, 24, 16, 24,  2,  2,  4, 17, 23,  1, 11,  5,  0,  7,  5, {82,85,66},      2,     1,     1,     6,     7,     1,     3,     3 }, // Chechen/Cyrillic/Russia
    {     55,    23,   248,    0,    0,  222,  231,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   61,   76, 3878, 3878, 3926, 3926, 3953, 3953,  149,  146,  300,    5,   22,   10,  799,    2,    9,  930,  933,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 48, 48, 27, 27, 13, 13,  3,  6,  6, 17, 23,  1,  6,  4,  6,  3, 15, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Cherokee/Cherokee/United States
    {     56,    66,   248,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38, 3966, 3966, 3966, 3966,   83,   83,    0,    0,    0,    5,   22,   10,    0,   15,    0,  948,  964,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 87, 87, 87, 87, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0, 16, 13, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Chickasaw/Latin/United States
    {     57,    66,   243,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0, 4053, 4053, 4126, 4126, 4153, 4153,    0,    0,    0,    5,   22,  147,  805,    2,    0,  977,  983,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 73, 73, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3, 19,  4,  0,  6,  6, {85,71,88},      0,     0,     1,     7,     7,     1,     3,     3 }, // Chiga/Latin/Uganda
    {     58,   118,    50,  183,  183,  239,  239,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  513,  505,  169,    0, 3148, 3148, 3188, 3188, 3175, 3175,  130,  126,  306,    5,   22,  150,  682,    2,    9,  989,  993,  5,  5,  5,  5,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 13,  8, 13,  5, 27, 27, 20, 20, 13, 13,  2,  2,  2, 17, 23,  1,  3,  4,  6,  4,  2, {67,78,89},      2,     1,     1,     6,     7,     1,     3,     3 }, // Chinese/Simplified Han/China
    {     58,   118,   107,  183,  183,  239,  239,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  513,  129,  198,   43, 3148, 3148, 3188, 3188, 3175, 3175,  130,  126,  306,    5,   22,  142,  824,    2,    9,  989,  995,  5,  5,  5,  5,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 13,  6, 14,  6, 27, 27, 20, 20, 13, 13,  2,  2,  2, 17, 23,  3,  2,  4,  6,  4,  9, {72,75,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Chinese/Simplified Han/Hong Kong
    {     58,   118,   139,  183,  183,  239,  239,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  513,  129,  198,   43, 3148, 3148, 3188, 3188, 3175, 3175,  130,  126,  306,    5,   22,  151,  826,    2,    9,  989, 1004,  5,  5,  5,  5,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 13,  6, 14,  6, 27, 27, 20, 20, 13, 13,  2,  2,  2, 17, 23,  4,  3,  4,  6,  4,  9, {77,79,80},      2,     1,     7,     6,     7,     1,     3,     3 }, // Chinese/Simplified Han/Macao
    {     58,   118,   210,  183,  183,  239,  239,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  513,   78,  198,   43, 3148, 3148, 3188, 3188, 3175, 3175,  130,  126,  306,    5,   22,   10,  829,    2,    9,  989, 1013,  5,  5,  5,  5,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 13,  8, 14,  6, 27, 27, 20, 20, 13, 13,  2,  2,  2, 17, 23,  1,  4,  4,  6,  4,  3, {83,71,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Chinese/Simplified Han/Singapore
    {     58,   137,   107,  183,  183,  244,  244,    6,    0,    1,    2,    3,    4,    5,   10,   51,   52,   53,   54,  513,  129,  182,   43, 3148, 3148, 4166, 4166, 3175, 3175,  130,  126,  308,    5,   22,  142,  824,    2,    9, 1016, 1020,  5,  5,  5,  5,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 13,  8, 16,  6, 27, 27, 20, 20, 13, 13,  2,  2,  3, 17, 23,  3,  2,  4,  6,  4,  9, {72,75,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Chinese/Traditional Han/Hong Kong
    {     58,   137,   139,  183,  183,  244,  244,    6,    0,    1,    2,    3,    4,    5,   10,   51,   52,   53,   54,  513,  129,  182,   43, 3148, 3148, 4166, 4166, 3175, 3175,  130,  126,  308,    5,   22,  151,  833,    2,    9, 1016, 1029,  5,  5,  5,  5,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 13,  8, 16,  6, 27, 27, 20, 20, 13, 13,  2,  2,  3, 17, 23,  4,  3,  4,  6,  4,  9, {77,79,80},      2,     1,     7,     6,     7,     1,     3,     3 }, // Chinese/Traditional Han/Macao
    {     58,   137,   228,  183,  183,  239,  239,    6,    0,    1,    2,    3,    4,    5,   10,   51,   52,   53,   54,  491,  505,  182,   43, 3148, 3148, 4166, 4166, 3175, 3175,  130,  126,    0,    5,   22,   10,  836,    2,    9, 1016, 1038,  5,  5,  5,  5,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 14,  8, 16,  6, 27, 27, 20, 20, 13, 13,  2,  2,  4, 17, 23,  1,  3,  4,  6,  4,  2, {84,87,68},      2,     0,     7,     6,     7,     1,     3,     3 }, // Chinese/Traditional Han/Taiwan
    {     59,    27,   193,    0,    0,  249,  249,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14,  573,  596,   10,    0, 4186, 4186, 4253, 4253, 4289, 4289,    0,    0,    0,    5,   22,  133,  839,    4,    0, 1040, 1059,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23, 10, 13,  5, 67, 67, 36, 36, 13, 13,  2,  2,  4, 17, 23,  1, 18,  5,  0, 19,  7, {82,85,66},      2,     1,     1,     6,     7,     1,     3,     3 }, // Church/Cyrillic/Russia
    {     60,    27,   193,    0,    0,  257,  257,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14,  606,   49,   10,    0, 4302, 4302, 4367, 4367, 4399, 4399,    0,    0,    0,    5,   22,  133,  857,    4,    0, 1066, 1071,  6,  6, 10, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22, 10, 13,  5, 65, 65, 32, 32, 13, 13,  2,  2,  4, 17, 23,  1, 12,  5,  0,  5,  6, {82,85,66},      2,     1,     1,     6,     7,     1,     3,     3 }, // Chuvash/Cyrillic/Russia
    {     61,    66,    91,    0,    0,  267,  267,    6,    1,    9,    2,    3,   48,    5,   63,   13,   14,   18,   16,  628,  423,   10,    0, 4412, 4412, 4483, 4483, 4510, 4510,  152,  152,    0,    5,   22,   22,   83,    4,    0, 1077, 1083,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  4,  1,  1,  1,  1, 23, 10, 13,  5, 71, 71, 27, 27, 13, 13, 16, 16,  4, 17, 23,  1,  4,  5,  0,  6, 11, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Colognian/Latin/Germany
    {     63,    66,   246,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  186,   10,    0, 4523, 4523, 4583, 4583,   83,   83,  168,  168,    0,    5,   22,   94,    0,    2,    0, 1094, 1102,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 60, 60, 27, 27, 13, 13,  4,  4,  4, 17, 23,  1,  0,  4,  0,  8, 14, {71,66,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // Cornish/Latin/United Kingdom
    {     64,    66,    84,    0,    0,  275,  275,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   11,   12,  651,  186,   10,    0, 4610, 4610, 4660, 4660, 4694, 4694,    0,    0,    0,    5,   22,  155,  405,    4,   51, 1116, 1121,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 25, 10, 13,  5, 50, 50, 34, 34, 13, 13,  2,  2,  4, 17, 23,  3,  4,  5,  7,  5,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Corsican/Latin/France
    {     66,    66,    60,    0,    0,  143,  143,    6,    1,    0,    2,    3,   48,    5,   10,   13,   14,   18,   16,  404,  676,   98,    0, 2699, 2699, 2756, 2756, 2783, 2796,    0,    0,  218,    5,   22,   22,  405,    4,    0, 1128, 1136,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19, 13, 15,  5, 57, 57, 27, 27, 13, 13,  2,  2,  7, 17, 23,  1,  4,  5,  0,  8,  8, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Croatian/Latin/Croatia
    {     66,    66,    29,    0,    0,  143,  143,    6,    1,    0,    2,    3,   48,    5,   10,   13,   14,   18,   16,  404,  689,   98,    0, 2699, 2699, 2756, 2756, 2796, 2796,    0,    0,  218,    5,   22,  135,  618,    4,    0, 1128,  686,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  9, 15,  5, 57, 57, 27, 27, 13, 13,  2,  2,  7, 17, 23,  2, 19,  5,  0,  8, 19, {66,65,77},      2,     1,     1,     6,     7,     1,     3,     3 }, // Croatian/Latin/Bosnia and Herzegovina
    {     67,    66,    64,    0,    0,  282,  282,    6,    1,    9,    2,    3,    4,    5,   10,   13,   14,   18,   16,  698,   49,  114,    1, 4707, 4707, 4755, 4755, 4775, 4775,  172,  172,  311,    5,   22,  158,  869,    4,    0, 1144, 1151,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 13,  4, 48, 48, 20, 20, 13, 13,  4,  4,  5, 17, 23,  2, 12,  5,  0,  7,  5, {67,90,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Czech/Latin/Czechia
    {     68,    66,    65,    0,    0,  289,  289,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  715,   49,  212,  212, 4788, 4788, 4838, 4838, 4874, 4874,    0,    0,    0,    5,   22,  160,  881,    4,    0, 1156, 1161,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23, 10, 13,  5, 50, 50, 36, 36, 13, 13,  2,  2,  5, 17, 23,  3, 11,  5,  0,  5,  7, {68,75,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Danish/Latin/Denmark
    {     68,    66,    95,    0,    0,  289,  289,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  715,   49,  212,  212, 4788, 4788, 4838, 4838, 4874, 4874,    0,    0,    0,    5,   22,  160,  881,    4,    0, 1156, 1168,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23, 10, 13,  5, 50, 50, 36, 36, 13, 13,  2,  2,  5, 17, 23,  3, 11,  5,  0,  5,  8, {68,75,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Danish/Latin/Greenland
    {     69,   132,   144,    0,    0,    0,    0,    2,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  283,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  163,    0,   15,    0, 1176, 1186,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  6, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  2,  0,  5,  0, 10, 13, {77,86,82},      2,     1,     5,     6,     7,     1,     3,     3 }, // Divehi/Thaana/Maldives
    {     70,    29,   110,    0,    0,  297,  306,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  738,  129,   61,   76, 4887, 4887, 4937, 4937, 4966, 4988,  176,  176,    0,    5,   22,  120,  892,    2,    0, 1199,  664,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 50, 50, 29, 29, 22, 24,  4,  9,  4, 17, 23,  1, 10,  4,  0,  5,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Dogri/Devanagari/India
    {     71,    66,    40,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   16,   17,  113,  129,   10,    0, 5012, 5012, 5056, 5056, 5083, 5083,  180,  185,    0,    5,   22,   11,    0,    4,    0, 1204, 1209,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 44, 44, 27, 27, 13, 13,  5,  6,  4, 17, 23,  4,  0,  5,  0,  5,  8, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Duala/Latin/Cameroon
    {     72,    66,   165,    0,    0,   16,   16,    6,    1,    0,    2,    3,    4,    5,   10,   16,   17,   16,   17,  113,  394,   10,    0, 5096, 5096, 5154, 5154, 5174, 5174,  168,  168,    0,    5,   22,   22,   83,   15,   58, 1217, 1217,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 58, 58, 20, 20, 13, 13,  4,  4,  4, 17, 23,  1,  4,  5,  7, 10,  9, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Dutch/Latin/Netherlands
    {     72,    66,    13,    0,    0,   16,   16,    6,    1,    0,    2,    3,    4,    5,   10,   16,   17,   16,   17,  113,  394,   10,    0, 5096, 5096, 5154, 5154, 5174, 5174,  168,  168,    0,    5,   22,  165,  902,   15,   58, 1217, 1227,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 58, 58, 20, 20, 13, 13,  4,  4,  4, 17, 23,  4, 16,  5,  7, 10,  5, {65,87,71},      2,     1,     1,     6,     7,     1,     3,     3 }, // Dutch/Latin/Aruba
    {     72,    66,    23,    0,    0,   16,   16,    6,    1,    0,    2,    3,    4,    5,   10,   16,   17,   16,   17,  113,  187,   10,    0, 5096, 5096, 5154, 5154, 5174, 5174,  168,  168,    0,    5,   22,   22,   83,   15,   58, 1232, 1238,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  9, 13,  5, 58, 58, 20, 20, 13, 13,  4,  4,  4, 17, 23,  1,  4,  5,  7,  6,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Dutch/Latin/Belgium
    {     72,    66,    44,    0,    0,   16,   16,    6,    1,    0,    2,    3,    4,    5,   10,   16,   17,   16,   17,  113,  394,   10,    0, 5096, 5096, 5154, 5154, 5174, 5174,  168,  168,    0,    5,   22,   10,  918,   15,   58, 1217, 1244,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 58, 58, 20, 20, 13, 13,  4,  4,  4, 17, 23,  1, 18,  5,  7, 10, 19, {85,83,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Dutch/Latin/Caribbean Netherlands
    {     72,    66,    62,    0,    0,   16,   16,    6,    1,    0,    2,    3,    4,    5,   10,   16,   17,   16,   17,  113,  394,   10,    0, 5096, 5096, 5154, 5154, 5174, 5174,  168,  168,    0,    5,   22,  169,  936,   15,   58, 1217, 1263,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 58, 58, 20, 20, 13, 13,  4,  4,  4, 17, 23,  3, 17,  5,  7, 10,  7, {88,67,71},      2,     1,     1,     6,     7,     1,     3,     3 }, // Dutch/Latin/Curacao
    {     72,    66,   211,    0,    0,   16,   16,    6,    1,    0,    2,    3,    4,    5,   10,   16,   17,   16,   17,  113,  394,   10,    0, 5096, 5096, 5154, 5154, 5174, 5174,  168,  168,    0,    5,   22,  169,  936,   15,   58, 1217, 1270,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 58, 58, 20, 20, 13, 13,  4,  4,  4, 17, 23,  3, 17,  5,  7, 10, 12, {88,67,71},      2,     1,     1,     6,     7,     1,     3,     3 }, // Dutch/Latin/Sint Maarten
    {     72,    66,   223,    0,    0,   16,   16,    6,    1,    0,    2,    3,    4,    5,   10,   16,   17,   16,   17,  113,  394,   10,    0, 5096, 5096, 5154, 5154, 5174, 5174,  168,  168,    0,    5,   22,   10,  953,   15,   58, 1217, 1282,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 58, 58, 20, 20, 13, 13,  4,  4,  4, 17, 23,  1, 17,  5,  7, 10,  8, {83,82,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Dutch/Latin/Suriname
    {     73,   134,    27,  314,  314,  314,  314,    6,    0,    1,    2,   67,    4,    5,   10,   14,   15,   16,   17,  756,  103,  225,  255, 5187, 5187, 5265, 5265, 5298, 5298,  185,  191,    0,    5,   22,  172,  970,    2,    0, 1290, 1296,  9,  9,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 30, 10, 30, 22, 78, 78, 33, 33, 26, 26,  5,  6,  4, 17, 23,  3,  8,  4,  0,  6,  5, {66,84,78},      2,     1,     7,     6,     7,     1,     2,     3 }, // Dzongkha/Tibetan/Bhutan
    {     74,    66,   124,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0, 5324, 5324, 5387, 5387, 5414, 5414,  190,  197,    0,    5,   22,  175,  978,    2,    9, 1301, 1307,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 63, 63, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3, 17,  4,  6,  6,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Embu/Latin/Kenya
    {     75,    66,   248,    0,    0,  323,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38,    0,    0,   56,   56,   83,   83,    0,    0,    0,    5,   22,   10,  995,    2,    9, 1312,  964,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  9,  4,  6, 16, 13, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/United States
    {     75,    28,   248,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0, 5427, 5427, 5511, 5511, 5559, 5559,  192,  199,    0,    5,   22,   10,    0,   15,    0, 1328, 1338,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 84, 84, 48, 48, 20, 20,  4,  4,  4, 17, 23,  1,  0,  5,  0, 10, 25, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Deseret/United States
    {     75,    66,     5,    0,    0,  323,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38,    0,    0,   56,   56,   83,   83,    0,    0,    0,    5,   22,   10,  995,    2,    9, 1321, 1363,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  9,  4,  6,  7, 14, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/American Samoa
    {     75,    66,     8,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1004,    2,    9, 1321, 1377,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 21,  4,  6,  7,  8, {88,67,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Anguilla
    {     75,    66,    10,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1004,    2,    9, 1321, 1385,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 21,  4,  6,  7, 17, {88,67,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Antigua and Barbuda
    {     75,    66,    15,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,    6,   14,   15,   16,   17,  113,  129,   23,   38,    0,    0,   56,   56,   83, 5579,   82,  203,    0,    5,   22,   10, 1025,    2,    9, 1402, 1402,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  6, 15,  7, 56, 56, 27, 27, 13, 24,  2,  2,  5, 17, 23,  1, 17,  4,  6, 18,  9, {65,85,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Australia
    {     75,    66,    16,    0,    0,  333,  333,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   22,   83,   15,    0, 1321, 1420,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  5,  0,  7,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Austria
    {     75,    66,    18,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1042,    2,    9, 1321, 1427,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 15,  4,  6,  7,  7, {66,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Bahamas
    {     75,    66,    21,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1057,    2,    9, 1321, 1434,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 16,  4,  6,  7,  8, {66,66,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Barbados
    {     75,    66,    23,    0,    0,  333,  333,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,   78,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   22,   83,    4,    0, 1321, 1442,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  5,  0,  7,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Belgium
    {     75,    66,    24,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  786,   78,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1073,    2,    9, 1321, 1449,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 13,  4,  6,  7,  6, {66,90,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Belize
    {     75,    66,    26,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1086,    2,    9, 1321, 1455,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 16,  4,  6,  7,  7, {66,77,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Bermuda
    {     75,    66,    30,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  786,   78,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  153, 1102,    2,    9, 1321, 1462,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 14,  4,  6,  7,  8, {66,87,80},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Botswana
    {     75,    66,    33,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  178,  995,    2,    9, 1321, 1470,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3,  9,  4,  6,  7, 30, {85,83,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/British Indian Ocean Territory
    {     75,    66,    34,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  178,  995,    2,    9, 1321, 1500,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3,  9,  4,  6,  7, 22, {85,83,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/British Virgin Islands
    {     75,    66,    38,    0,    0,  323,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   10,    0,    0,    0,   56,   56,   83,   83,    0,    0,    0,    5,   22,  181, 1116,    2,    9, 1321, 1522,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3, 15,  4,  6,  7,  7, {66,73,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // English/Latin/Burundi
    {     75,    66,    40,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   11, 1131,    2,    9, 1321, 1529,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  4, 25,  4,  6,  7,  8, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // English/Latin/Cameroon
    {     75,    66,    41,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  103,   23,   38,    0,    0,   56,   56,   83,   83,  168,  168,    0,    5,   22,   10, 1156,    2,    9, 1537, 1553,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18, 10, 15,  7, 56, 56, 27, 27, 13, 13,  4,  4,  5, 17, 23,  1, 15,  4,  6, 16,  6, {67,65,68},      2,     0,     7,     6,     7,     1,     3,     3 }, // English/Latin/Canada
    {     75,    66,    45,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1171,    2,    9, 1321, 1559,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 21,  4,  6,  7, 14, {75,89,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Cayman Islands
    {     75,    66,    51,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1025,    2,    9, 1321, 1573,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 17,  4,  6,  7, 16, {65,85,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Christmas Island
    {     75,    66,    53,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1025,    2,    9, 1321, 1589,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 17,  4,  6,  7, 23, {65,85,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Cocos Islands
    {     75,    66,    58,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1192,    2,    9, 1321, 1612,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 18,  4,  6,  7, 12, {78,90,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Cook Islands
    {     75,    66,    63,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   22,   83,    2,    9, 1321, 1624,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  4,  6,  7,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Cyprus
    {     75,    66,    65,    0,    0,  333,  333,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,  212,  212,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  160, 1210,    4,    0, 1321, 1630,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3, 12,  5,  0,  7,  7, {68,75,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // English/Latin/Denmark
    {     75,    66,    66,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  178,  995,    2,    9, 1321, 1637,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3,  9,  4,  6,  7, 12, {85,83,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Diego Garcia
    {     75,    66,    68,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1004,    2,    9, 1321, 1649,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 21,  4,  6,  7,  8, {88,67,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Dominica
    {     75,    66,    74,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,    6, 1222,    2,    9, 1321, 1657,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3, 14,  4,  6,  7,  7, {69,82,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Eritrea
    {     75,    66,    76,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  155, 1236,    2,    9, 1321, 1664,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 15,  4,  6,  7,  8, {83,90,76},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Eswatini
    {     75,    66,    78,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,    0,    0,    4,    0, 1321, 1672,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  0,  0,  5,  0,  7,  6,    {0,0,0},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Europe
    {     75,    66,    80,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   94, 1251,    2,    9, 1321, 1678,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 22,  4,  6,  7, 16, {70,75,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Falkland Islands
    {     75,    66,    82,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1273,    2,    9, 1321, 1694,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 13,  4,  6,  7,  4, {70,74,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Fiji
    {     75,    66,    83,    0,    0,  333,  333,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,  213,  213,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   22,   83,    4,    0, 1321, 1698,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 12,  4, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  5,  0,  7,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Finland
    {     75,    66,    89,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  184, 1286,    2,    9, 1321, 1705,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 14,  4,  6,  7,  6, {71,77,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Gambia
    {     75,    66,    91,    0,    0,  333,  333,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   22,   83,    4,    0, 1321, 1711,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  5,  0,  7,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Germany
    {     75,    66,    92,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   15, 1300,    2,    9, 1321, 1718,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3, 13,  4,  6,  7,  5, {71,72,83},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Ghana
    {     75,    66,    93,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   94, 1313,    2,    9, 1321, 1723,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 15,  4,  6,  7,  9, {71,73,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Gibraltar
    {     75,    66,    96,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1004,    2,    9, 1321, 1732,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 21,  4,  6,  7,  7, {88,67,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Grenada
    {     75,    66,    98,    0,    0,  323,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38,    0,    0,   56,   56,   83,   83,    0,    0,    0,    5,   22,   10,  995,    2,    9, 1321, 1739,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  9,  4,  6,  7,  4, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Guam
    {     75,    66,   100,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   94, 1328,    2,    9, 1321, 1743,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  8,  4,  6,  7,  8, {71,66,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Guernsey
    {     75,    66,   103,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1336,    2,    9, 1321, 1751,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 16,  4,  6,  7,  6, {71,89,68},      2,     0,     1,     6,     7,     1,     3,     3 }, // English/Latin/Guyana
    {     75,    66,   107,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  129,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  142, 1352,    2,    9, 1321, 1757,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3, 16,  4,  6,  7, 19, {72,75,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Hong Kong
    {     75,    66,   110,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  804,   78,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  120, 1368,    2,    9, 1321, 1478,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 12,  4,  6,  7,  5, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // English/Latin/India
    {     75,    66,   111,    0,    0,  333,  333,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,  212,  212,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  185, 1380,    2,    9, 1321, 1776,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  2, 17,  4,  6,  7,  9, {73,68,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // English/Latin/Indonesia
    {     75,    66,   114,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  186,   10,    0,    0,    0,   56,   56,   83,   83,  168,  168,    0,    5,   22,   22,   83,    2,    9, 1321, 1785,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 56, 56, 27, 27, 13, 13,  4,  4,  5, 17, 23,  1,  4,  4,  6,  7,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Ireland
    {     75,    66,   115,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   94, 1328,    2,    9, 1321, 1792,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  8,  4,  6,  7, 11, {71,66,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Isle of Man
    {     75,    66,   116,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   11,    1,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   48, 1397,    2,    9, 1321, 1803,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 12,  4, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 18,  4,  6,  7,  6, {73,76,83},      2,     1,     7,     5,     6,     1,     3,     3 }, // English/Latin/Israel
    {     75,    66,   119,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  129,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1415,    2,    9, 1321, 1809,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 15,  4,  6,  7,  7, {74,77,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Jamaica
    {     75,    66,   121,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   94, 1328,    2,    9, 1321, 1816,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  8,  4,  6,  7,  6, {71,66,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Jersey
    {     75,    66,   124,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  175, 1430,    2,    9, 1321, 1307,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3, 15,  4,  6,  7,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Kenya
    {     75,    66,   125,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1025,    2,    9, 1321, 1822,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 17,  4,  6,  7,  8, {65,85,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Kiribati
    {     75,    66,   133,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,    9, 1445,    2,    9, 1321, 1830,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 18,  4,  6,  7,  7, {90,65,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Lesotho
    {     75,    66,   134,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1463,    2,    9, 1321, 1837,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 15,  4,  6,  7,  7, {76,82,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Liberia
    {     75,    66,   139,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  151, 1478,    2,    9, 1321, 1844,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  4, 15,  4,  6,  7, 15, {77,79,80},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Macao
    {     75,    66,   141,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  187, 1493,    2,    9, 1321, 1859,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  2, 15,  4,  6,  7, 10, {77,71,65},      0,     0,     1,     6,     7,     1,     3,     3 }, // English/Latin/Madagascar
    {     75,    66,   142,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  189, 1508,    2,    9, 1321, 1869,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  2, 15,  4,  6,  7,  6, {77,87,75},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Malawi
    {     75,    66,   143,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  191, 1523,    2,    9, 1321, 1875,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  2, 17,  4,  6,  7,  8, {77,89,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Malaysia
    {     75,    66,   144,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  283,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  193, 1540,   15,    0, 1321, 1883,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  6, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  2, 17,  5,  0,  7,  8, {77,86,82},      2,     1,     5,     6,     7,     1,     3,     3 }, // English/Latin/Maldives
    {     75,    66,   146,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   22,   83,    2,    9, 1321, 1891,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  4,  6,  7,  5, {69,85,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Malta
    {     75,    66,   147,    0,    0,  323,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38,    0,    0,   56,   56,   83,   83,    0,    0,    0,    5,   22,   10,  995,    2,    9, 1321, 1896,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  9,  4,  6,  7, 16, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Marshall Islands
    {     75,    66,   150,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  195, 1557,    2,    9, 1321, 1912,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  2, 15,  4,  6,  7,  9, {77,85,82},      2,     0,     1,     6,     7,     1,     3,     3 }, // English/Latin/Mauritius
    {     75,    66,   153,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  178,  995,    2,    9, 1321, 1921,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3,  9,  4,  6,  7, 10, {85,83,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Micronesia
    {     75,    66,   158,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1004,    2,    9, 1321, 1931,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 21,  4,  6,  7, 10, {88,67,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Montserrat
    {     75,    66,   162,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1572,    2,    9, 1321, 1941,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 15,  4,  6,  7,  7, {78,65,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Namibia
    {     75,    66,   163,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1025,    2,    9, 1321, 1948,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 17,  4,  6,  7,  5, {65,85,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Nauru
    {     75,    66,   165,    0,    0,  333,  333,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   22,   83,   15,   58, 1321, 1953,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  5,  7,  7, 11, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Netherlands
    {     75,    66,   167,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1192,    2,    9, 1321, 1964,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 18,  4,  6,  7, 11, {78,90,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/New Zealand
    {     75,    66,   169,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  124, 1587,    2,    9, 1321, 1975,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 14,  4,  6,  7,  7, {78,71,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Nigeria
    {     75,    66,   171,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1192,    2,    9, 1321, 1982,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 18,  4,  6,  7,  4, {78,90,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Niue
    {     75,    66,   172,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1025,    2,    9, 1321, 1986,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 17,  4,  6,  7, 14, {65,85,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Norfolk Island
    {     75,    66,   173,    0,    0,  323,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38,    0,    0,   56,   56,   83,   83,    0,    0,    0,    5,   22,   10,  995,    2,    9, 1321, 2000,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  9,  4,  6,  7, 24, {85,83,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Northern Mariana Islands
    {     75,    66,   178,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  195, 1601,    2,    9, 1321, 2024,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  2, 15,  4,  6,  7,  8, {80,75,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // English/Latin/Pakistan
    {     75,    66,   179,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  178,  995,    2,    9, 1321, 2032,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3,  9,  4,  6,  7,  5, {85,83,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Palau
    {     75,    66,   182,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  134, 1616,    2,    9, 1321, 2037,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 22,  4,  6,  7, 16, {80,71,75},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Papua New Guinea
    {     75,    66,   185,    0,    0,  323,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38,    0,    0,   56,   56,   83,   83,    0,    0,    0,    5,   22,  146,  685,    2,    9, 1321, 2053,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 15,  4,  6,  7, 11, {80,72,80},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Philippines
    {     75,    66,   186,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1192,    2,    9, 1321, 2064,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 18,  4,  6,  7, 16, {78,90,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Pitcairn
    {     75,    66,   189,    0,    0,  323,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38,    0,    0,   56,   56,   83,   83,    0,    0,    0,    5,   22,   10,  995,    2,    9, 1321, 2080,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  9,  4,  6,  7, 11, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Puerto Rico
    {     75,    66,   194,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  197, 1638,    2,    9, 1321, 2091,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  2, 13,  4,  6,  7,  6, {82,87,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // English/Latin/Rwanda
    {     75,    66,   196,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   94, 1651,    2,    9, 1321, 2097,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 15,  4,  6,  7,  9, {83,72,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Saint Helena
    {     75,    66,   197,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1004,    2,    9, 1321, 2106,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 21,  4,  6,  7, 16, {88,67,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Saint Kitts and Nevis
    {     75,    66,   198,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1004,    2,    9, 1321, 2122,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 21,  4,  6,  7,  8, {88,67,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Saint Lucia
    {     75,    66,   201,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1004,    2,    9, 1321, 2130,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 21,  4,  6,  7, 27, {88,67,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Saint Vincent and Grenadines
    {     75,    66,   202,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  199, 1666,    2,    9, 1321, 1372,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3, 11,  4,  6,  7,  5, {87,83,84},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Samoa
    {     75,    66,   208,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  202, 1677,    2,    9, 1321, 2157,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  2, 17,  4,  6,  7, 10, {83,67,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Seychelles
    {     75,    66,   209,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   18, 1694,    2,    9, 1321, 2167,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  2, 20,  4,  6,  7, 12, {83,76,69},      2,     0,     1,     6,     7,     1,     3,     3 }, // English/Latin/Sierra Leone
    {     75,    66,   210,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  129,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1714,    2,    9, 1321, 2179,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 16,  4,  6,  7,  9, {83,71,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Singapore
    {     75,    66,   211,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  169, 1730,    2,    9, 1321, 2188,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3, 17,  4,  6,  7, 12, {88,67,71},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Sint Maarten
    {     75,    66,   213,    0,    0,  333,  333,    6,    1,    0,    2,    3,    4,    5,    6,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   22,   83,    4,   20, 1321, 2200,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  5,  7,  7,  8, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Slovenia
    {     75,    66,   214,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1747,    2,    9, 1321, 2208,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 22,  4,  6,  7, 15, {83,66,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Solomon Islands
    {     75,    66,   216,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  786,  821,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,    9, 1445,    2,    9, 1321, 2223,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 18,  4,  6,  7, 12, {90,65,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/South Africa
    {     75,    66,   219,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   94, 1769,    2,    9, 1321, 2235,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 20,  4,  6,  7, 11, {83,83,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/South Sudan
    {     75,    66,   222,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,    0, 1789,    2,    9, 1321, 2246,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  0, 14,  4,  6,  7,  5, {83,68,71},      2,     1,     6,     5,     6,     1,     3,     3 }, // English/Latin/Sudan
    {     75,    66,   225,    0,    0,  333,  333,    6,    1,    9,    2,    3,    4,    5,   63,   14,   15,   16,   17,    0,  103,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  160, 1803,    4,    0, 1321, 2251,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  4,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  2, 13,  5,  0,  7,  6, {83,69,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // English/Latin/Sweden
    {     75,    66,   226,    0,    0,  333,  333,    6,    0,   17,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,   49,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,    0, 1816,   15,   65, 1321, 2257,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  0, 11,  5,  5,  7, 11, {67,72,70},      2,     0,     1,     6,     7,     1,     3,     3 }, // English/Latin/Switzerland
    {     75,    66,   230,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  121, 1827,    2,    9, 1321, 2268,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3, 18,  4,  6,  7,  8, {84,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // English/Latin/Tanzania
    {     75,    66,   234,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1192,    2,    9, 1321, 2276,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 18,  4,  6,  7,  7, {78,90,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Tokelau
    {     75,    66,   235,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  204, 1845,    2,    9, 1321, 2283,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  2, 14,  4,  6,  7,  5, {84,79,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Tonga
    {     75,    66,   236,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1859,    2,    9, 1321, 2288,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 24,  4,  6,  7, 17, {84,84,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Trinidad and Tobago
    {     75,    66,   241,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  178,  995,    2,    9, 1321, 2305,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3,  9,  4,  6,  7, 22, {85,83,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Turks and Caicos Islands
    {     75,    66,   242,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   10, 1025,    2,    9, 1321, 2327,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 17,  4,  6,  7,  6, {65,85,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Tuvalu
    {     75,    66,   243,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  147, 1883,    2,    9, 1321,  983,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3, 16,  4,  6,  7,  6, {85,71,88},      0,     0,     1,     7,     7,     1,     3,     3 }, // English/Latin/Uganda
    {     75,    66,   245,    0,    0,  323,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,    0,    0,    0,    5,   22,    0, 1899,    2,    9, 1321, 2333,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  0, 27,  4,  6,  7, 20, {65,69,68},      2,     1,     6,     6,     7,     1,     3,     3 }, // English/Latin/United Arab Emirates
    {     75,    66,   246,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  186,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,   94, 1926,    2,    9, 2353, 2368,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 13,  4,  6, 15, 14, {71,66,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/United Kingdom
    {     75,    66,   247,    0,    0,  323,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38,    0,    0,   56,   56,   83,   83,    0,    0,    0,    5,   22,   10,  995,    2,    9, 1321, 2382,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  9,  4,  6,  7, 21, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/United States Outlying Islands
    {     75,    66,   249,    0,    0,  323,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38,    0,    0,   56,   56,   83,   83,    0,    0,    0,    5,   22,   10,  995,    2,    9, 1321, 2403,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  9,  4,  6,  7, 19, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/United States Virgin Islands
    {     75,    66,   252,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  206, 1939,    2,    9, 1321, 2422,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  2, 12,  4,  6,  7,  7, {86,85,86},      0,     0,     1,     6,     7,     1,     3,     3 }, // English/Latin/Vanuatu
    {     75,    66,   258,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,    0,    0,    2,    9, 1321, 2429,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  0,  0,  4,  6,  7,  5,    {0,0,0},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/world
    {     75,    66,   260,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  134, 1951,    2,    9, 1321,  635,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 14,  4,  6,  7,  6, {90,77,87},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Latin/Zambia
    {     75,    66,   261,    0,    0,  333,  333,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  786,  129,   10,    0,    0,    0,   56,   56,   83,   83,   82,  203,    0,    5,   22,  178,  995,    2,    9, 1321, 2434,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  3,  9,  4,  6,  7,  8, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // English/Latin/Zimbabwe
    {     75,   115,   246,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0, 5603, 5603, 5690, 5690, 5731, 5731,  196,  205,    0,    5,   22,   94,    0,   15,    0,    0,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 87, 87, 41, 41, 20, 20,  4,  4,  4, 17, 23,  1,  0,  5,  0,  0,  0, {71,66,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // English/Shavian/United Kingdom
    {     76,    27,   193,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0, 5751, 5811, 5892, 5892,   83,   83,    0,    0,    0,    5,   22,  133,    0,   15,    0, 2442, 2453,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 60, 81, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0, 11, 13, {82,85,66},      2,     1,     1,     6,     7,     1,     3,     3 }, // Erzya/Cyrillic/Russia
    {     77,    66,   258,    0,    0,  342,  342,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17,  831,  105,   10,    0, 5919, 5919, 5969, 5969, 5989, 5989,  200,  209,  316,    5,   22,    0,    0,    4,    0, 2466, 2475,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 31,  8, 13,  5, 50, 50, 20, 20, 13, 13,  3,  3,  6, 17, 23,  0,  0,  5,  0,  9,  5,    {0,0,0},      2,     1,     1,     6,     7,     1,     3,     3 }, // Esperanto/Latin/world
    {     78,    66,    75,    0,    0,  351,  351,    6,    1,    9,    2,    3,   48,    5,   63,   13,   14,   18,   16,  404,   49,   10,    0, 6002, 6002, 6064, 6064, 6064, 6064,    0,    0,  322,    5,   22,   22,  405,    4,   20, 2480, 2485,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  4,  1,  1,  1,  1, 18,  8, 13,  5, 62, 62, 13, 13, 13, 13,  2,  2,  6, 17, 23,  1,  4,  5,  7,  5,  5, {69,85,82},      2,     1,     1,     6,     7,     2,     3,     3 }, // Estonian/Latin/Estonia
    {     79,    66,    92,    0,    0,  359,  370,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  862,  567,  277,  277, 6077, 6077, 6120, 6120, 6147, 6147,  203,  212,    0,    5,   22,   15, 1965,    2,    9, 2490, 2496,  6,  6, 11, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  6, 20, 12, 43, 43, 27, 27, 13, 13,  3,  5,  4, 17, 23,  3, 10,  4,  6,  6, 12, {71,72,83},      2,     1,     1,     6,     7,     3,     3,     3 }, // Ewe/Latin/Ghana
    {     79,    66,   233,    0,    0,  359,  370,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  862,  567,   10,    0, 6077, 6077, 6120, 6120, 6147, 6147,  203,  212,    0,    5,   22,  127, 1975,    2,    9, 2490, 2508,  6,  6, 11, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  6, 13,  5, 43, 43, 27, 27, 13, 13,  3,  5,  4, 17, 23,  5, 33,  4,  6,  6, 11, {88,79,70},      0,     0,     1,     6,     7,     3,     3,     3 }, // Ewe/Latin/Togo
    {     80,    66,    40,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  113,  129,   10,    0, 6160, 6160, 6244, 6244, 6273, 6273,  206,  217,    0,    5,   22,   11, 2008,    4,    0, 2519, 2525,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 84, 84, 29, 29, 13, 13,  7,  9,  4, 17, 23,  4, 16,  5,  0,  6,  7, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Ewondo/Latin/Cameroon
    {     81,    66,    81,    0,    0,  380,  289,    6,    1,    0,    2,    3,   48,    5,   10,   14,   15,   16,   17,  404,   49,   10,    0, 6286, 6286, 6359, 6386, 6420, 6420,    0,    0,  328,    5,   22,  160, 2024,    4,   20, 2532, 2540,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 73, 73, 27, 34, 13, 13,  2,  2,  3, 17, 23,  2, 11,  5,  7,  8,  7, {68,75,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Faroese/Latin/Faroe Islands
    {     81,    66,    65,    0,    0,  380,  289,    6,    1,    0,    2,    3,   48,    5,   10,   14,   15,   16,   17,  404,   49,   10,    0, 6286, 6286, 6359, 6386, 6420, 6420,    0,    0,  328,    5,   22,  160, 2024,    4,   20, 2532, 1161,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 73, 73, 27, 34, 13, 13,  2,  2,  3, 17, 23,  3, 11,  5,  7,  8,  7, {68,75,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Faroese/Latin/Denmark
    {     83,    66,   185,    0,    0,  389,  398,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38, 6433, 6433, 6487, 6487, 6487, 6487,    0,    0,    0,    5,   22,  146, 2035,    2,    9, 2547,  826,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 54, 54, 27, 27, 27, 27,  2,  2,  5, 17, 23,  1, 17,  4,  6,  8,  9, {80,72,80},      2,     1,     7,     6,     7,     1,     3,     3 }, // Filipino/Latin/Philippines
    {     84,    66,    83,    0,    0,  351,  351,    6,    1,    9,    2,    3,   48,    5,   10,   15,   15,   17,   17,  698,  885,  213,  213, 6514, 6580, 6660, 6660, 6680, 6680,  213,  226,  331,  336,  353,   22,  405,    4,    0, 2555, 2560,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 12,  4, 66, 80, 20, 20, 13, 13,  3,  3,  5, 17, 23,  1,  4,  5,  0,  5,  5, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Finnish/Latin/Finland
    {     85,    66,    84,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   22,  405,    4,   20, 2565, 2573,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  1,  4,  5,  7,  8,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/France
    {     85,    66,     4,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   23,   38, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  208, 2052,    4,   20, 2565, 2579,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 15,  7, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  2, 14,  5,  7,  8,  7, {68,90,68},      2,     1,     6,     5,     6,     1,     3,     3 }, // French/Latin/Algeria
    {     85,    66,    23,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,   79,  297,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   22,  405,    4,   20, 2565, 2586,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  7, 26,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  1,  4,  5,  7,  8,  8, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Belgium
    {     85,    66,    25,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  127, 2066,    4,   20, 2565, 2594,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  5, 17,  5,  7,  8,  5, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Benin
    {     85,    66,    37,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  127, 2066,    4,   20, 2565, 2599,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  5, 17,  5,  7,  8, 12, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Burkina Faso
    {     85,    66,    38,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  181, 2083,    4,   20, 2565, 1522,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  3, 15,  5,  7,  8,  7, {66,73,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Burundi
    {     85,    66,    40,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,  216,  229,  376,  232,  249,   11, 2098,    4,   20, 2565, 1209,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  5,  4,  6, 17, 23,  4, 16,  5,  7,  8,  8, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Cameroon
    {     85,    66,    41,    0,    0,  406,  406,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   15,   14,  113,  103,  323,  323, 6693, 6693, 6744, 6744, 6778, 6778,  168,  168,  376,  232,  249,   10, 2114,    4,   20, 2611, 1553,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 27,  9, 51, 51, 34, 34, 13, 13,  4,  4,  6, 17, 23,  1, 15,  5,  7, 17,  6, {67,65,68},      2,     0,     7,     6,     7,     1,     3,     3 }, // French/Latin/Canada
    {     85,    66,    46,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   11, 2098,    4,   20, 2565, 2628,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  4, 16,  5,  7,  8, 25, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Central African Republic
    {     85,    66,    48,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   23,   38, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   11, 2098,    4,   20, 2565, 2653,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 15,  7, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  4, 16,  5,  7,  8,  5, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Chad
    {     85,    66,    55,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   12, 2129,    4,   20, 2565, 2658,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  2, 14,  5,  7,  8,  7, {75,77,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Comoros
    {     85,    66,    56,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   11, 2098,    4,   20, 2565, 2665,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  4, 16,  5,  7,  8, 17, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Congo - Brazzaville
    {     85,    66,    57,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   11, 2143,    4,   20, 2565, 2682,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  2, 15,  5,  7,  8, 14, {67,68,70},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Congo - Kinshasa
    {     85,    66,    67,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   23,   38, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,    3, 2158,    4,   20, 2565, 2696,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 15,  7, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  3, 16,  5,  7,  8,  8, {68,74,70},      0,     0,     6,     6,     7,     1,     3,     3 }, // French/Latin/Djibouti
    {     85,    66,    73,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   11, 2098,    4,   20, 2565, 2704,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  4, 16,  5,  7,  8, 18, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Equatorial Guinea
    {     85,    66,    85,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   22,  405,    4,   20, 2565, 2722,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  1,  4,  5,  7,  8, 16, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/French Guiana
    {     85,    66,    86,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  210, 2174,    4,   20, 2565, 2738,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  4,  9,  5,  7,  8, 19, {88,80,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/French Polynesia
    {     85,    66,    88,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   11, 2098,    4,   20, 2565, 2757,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  4, 16,  5,  7,  8,  5, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Gabon
    {     85,    66,    97,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   22,  405,    4,   20, 2565, 2762,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  1,  4,  5,  7,  8, 10, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Guadeloupe
    {     85,    66,   102,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  214, 2183,    4,   20, 2565, 2704,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  2, 13,  5,  7,  8,  6, {71,78,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Guinea
    {     85,    66,   104,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   15, 2196,    4,   20, 2565, 2772,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  1, 16,  5,  7,  8,  5, {72,84,71},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Haiti
    {     85,    66,   118,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  127, 2066,    4,   20, 2565, 2777,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  5, 17,  5,  7,  8, 13, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Ivory Coast
    {     85,    66,   138,    0,    0,  406,  406,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   22,  405,    4,   20, 2565, 2790,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  1,  4,  5,  7,  8, 10, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Luxembourg
    {     85,    66,   141,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  187, 2212,    4,   20, 2565, 1859,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  2, 15,  5,  7,  8, 10, {77,71,65},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Madagascar
    {     85,    66,   145,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  127, 2066,    4,   20, 2565,  547,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  5, 17,  5,  7,  8,  4, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Mali
    {     85,    66,   148,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   22,  405,    4,   20, 2565, 2800,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  1,  4,  5,  7,  8, 10, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Martinique
    {     85,    66,   149,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   23,   38, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  216, 2227,    4,   20, 2565, 2810,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 15,  7, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  2, 19,  5,  7,  8, 10, {77,82,85},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Mauritania
    {     85,    66,   150,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  195, 2246,    4,   20, 2565, 2820,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  2, 18,  5,  7,  8,  7, {77,85,82},      2,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Mauritius
    {     85,    66,   151,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   22,  405,    4,   20, 2565, 2827,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  1,  4,  5,  7,  8,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Mayotte
    {     85,    66,   155,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   22,  405,    4,   20, 2565, 2834,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  1,  4,  5,  7,  8,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Monaco
    {     85,    66,   159,    0,    0,  406,  406,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,  168,  168,  376,  232,  249,    0, 2264,    4,   20, 2565, 2840,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  4,  4,  6, 17, 23,  0, 15,  5,  7,  8,  5, {77,65,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Morocco
    {     85,    66,   166,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  210, 2174,    4,   20, 2565, 2845,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  4,  9,  5,  7,  8, 18, {88,80,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/New Caledonia
    {     85,    66,   170,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  127, 2066,    4,   20, 2565, 1975,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  5, 17,  5,  7,  8,  5, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Niger
    {     85,    66,   191,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   22,  405,    4,   20, 2565, 2863,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  1,  4,  5,  7,  8, 10, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Reunion
    {     85,    66,   194,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  197, 2279,    4,   20, 2565, 2091,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  2, 14,  5,  7,  8,  6, {82,87,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Rwanda
    {     85,    66,   195,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   22,  405,    4,   20, 2565, 2873,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  1,  4,  5,  7,  8, 16, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Saint Barthelemy
    {     85,    66,   199,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   22,  405,    4,   20, 2565, 2889,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  1,  4,  5,  7,  8, 12, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Saint Martin
    {     85,    66,   200,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,   22,  405,    4,   20, 2565, 2901,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  1,  4,  5,  7,  8, 24, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Saint Pierre and Miquelon
    {     85,    66,   206,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  127, 2066,    4,   20, 2565, 2925,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  5, 17,  5,  7,  8,  7, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Senegal
    {     85,    66,   208,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  202, 2293,    4,   20, 2565, 2157,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  2, 21,  5,  7,  8, 10, {83,67,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // French/Latin/Seychelles
    {     85,    66,   226,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   19,   20,    0,   49,  350,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,    0, 2314,    4,   20, 2932, 2947,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 17,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  0, 12,  5,  7, 15,  6, {67,72,70},      2,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Switzerland
    {     85,    66,   227,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   23,   38, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  218, 2326,    4,   20, 2565, 2953,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 15,  7, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  2, 14,  5,  7,  8,  5, {83,89,80},      0,     0,     6,     5,     6,     1,     3,     3 }, // French/Latin/Syria
    {     85,    66,   233,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  127, 2066,    4,   20, 2565, 2508,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  5, 17,  5,  7,  8,  4, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Togo
    {     85,    66,   238,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   23,   38, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  220, 2340,    4,   20, 2565, 2958,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 15,  7, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  2, 14,  5,  7,  8,  7, {84,78,68},      3,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Tunisia
    {     85,    66,   252,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   23,   38, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  206, 2354,    4,   20, 2565, 2422,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 15,  7, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  2, 14,  5,  7,  8,  7, {86,85,86},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Vanuatu
    {     85,    66,   256,    0,    0,  406,  406,    6,    1,   68,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  186,   10,    0, 6693, 6693, 6744, 6744, 6778, 6778,    0,    0,  376,  232,  249,  210, 2174,    4,   20, 2565, 2965,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 51, 51, 34, 34, 13, 13,  2,  2,  6, 17, 23,  4,  9,  5,  7,  8, 16, {88,80,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // French/Latin/Wallis and Futuna
    {     86,    66,   117,    0,    0,  414,  414,    6,    1,    0,    2,    3,    4,    5,   10,   16,   17,   14,   15,  893,   78,   10,    0, 6791, 6791, 6840, 6840, 6778, 6778,    5,  128,    0,    5,   22,   22,  405,   15,    0, 2981, 2987,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 49, 49, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  0,  6,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Friulian/Latin/Italy
    {     87,    66,   206,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   18,   17,  113,  129,   10,    0, 6867, 6867, 6925, 6925, 6952, 6952,  221,  233,    0,    5,   22,  127, 2368,    4,    0, 2993, 2999,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 58, 58, 27, 27, 13, 13,  6,  7,  4, 17, 23,  5, 19,  5,  0,  6,  8, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Latin/Senegal
    {     87,     1,    37,  421,  421,  427,  436,  444,    0,   69,    2,   70,    4,    5,   72,   14,   15,   16,   17,  920,  283,   10,    0, 6965, 6965, 7083, 7083, 7135, 7135,  227,  240,  382,  394,   22,  222, 2387,   15,    0, 3007, 3017,  6,  6,  9,  8,  1,  1,  1,  1,  2,  1,  1,  2,  1,  1,  1,  1, 17,  8, 13,  5,118,118, 52, 52, 22, 22,  4,  4, 12, 27, 23,  6, 51,  5,  0, 10, 25, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Adlam/Burkina Faso
    {     87,     1,    40,  421,  421,  427,  436,  444,    0,   69,    2,   70,    4,    5,   72,   14,   15,   16,   17,  920,  283,   10,    0, 6965, 6965, 7083, 7083, 7135, 7135,  227,  240,  382,  394,   22,  228, 2438,   15,    0, 3007, 3042,  6,  6,  9,  8,  1,  1,  1,  1,  2,  1,  1,  2,  1,  1,  1,  1, 17,  8, 13,  5,118,118, 52, 52, 22, 22,  4,  4, 12, 27, 23,  8, 44,  5,  0, 10, 16, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Adlam/Cameroon
    {     87,     1,    89,  421,  421,  427,  436,  444,    0,   69,    2,   70,    4,    5,   72,   14,   15,   16,   17,  920,  283,   61,   76, 6965, 6965, 7083, 7083, 7135, 7135,  227,  240,  382,  394,   22,  184, 2482,   15,    0, 3007, 3058,  6,  6,  9,  8,  1,  1,  1,  1,  2,  1,  1,  2,  1,  1,  1,  1, 17,  8, 15,  7,118,118, 52, 52, 22, 22,  4,  4, 12, 27, 23,  1, 29,  5,  0, 10, 14, {71,77,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Fulah/Adlam/Gambia
    {     87,     1,    92,  421,  421,  427,  436,  444,    0,   69,    2,   70,    4,    5,   72,   14,   15,   16,   17,  920,  283,   61,   76, 6965, 6965, 7083, 7083, 7135, 7135,  227,  240,  382,  394,   22,   15, 2511,   15,    0, 3007, 3072,  6,  6,  9,  8,  1,  1,  1,  1,  2,  1,  1,  2,  1,  1,  1,  1, 17,  8, 15,  7,118,118, 52, 52, 22, 22,  4,  4, 12, 27, 23,  3, 23,  5,  0, 10,  8, {71,72,83},      2,     1,     1,     6,     7,     1,     3,     3 }, // Fulah/Adlam/Ghana
    {     87,     1,   101,  421,  421,  427,  436,  444,    0,   69,    2,   70,    4,    5,   72,   14,   15,   16,   17,  920,  283,   10,    0, 6965, 6965, 7083, 7083, 7135, 7135,  227,  240,  382,  394,   22,  222, 2387,   15,    0, 3007, 3080,  6,  6,  9,  8,  1,  1,  1,  1,  2,  1,  1,  2,  1,  1,  1,  1, 17,  8, 13,  5,118,118, 52, 52, 22, 22,  4,  4, 12, 27, 23,  6, 51,  5,  0, 10, 23, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Adlam/Guinea-Bissau
    {     87,     1,   102,  421,  421,  427,  436,  444,    0,   69,    2,   70,    4,    5,   72,   14,   15,   16,   17,  920,  283,   10,    0, 6965, 6965, 7083, 7083, 7135, 7135,  227,  240,  382,  394,   22,  214, 2534,   15,    0, 3007, 3080,  6,  6,  9,  8,  1,  1,  1,  1,  2,  1,  1,  2,  1,  1,  1,  1, 17,  8, 13,  5,118,118, 52, 52, 22, 22,  4,  4, 12, 27, 23,  2, 25,  5,  0, 10,  8, {71,78,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Adlam/Guinea
    {     87,     1,   134,  421,  421,  427,  436,  444,    0,   69,    2,   70,    4,    5,   72,   14,   15,   16,   17,  920,  283,   61,   76, 6965, 6965, 7083, 7083, 7135, 7135,  227,  240,  382,  394,   22,   10, 2559,   15,    0, 3007, 3103,  6,  6,  9,  8,  1,  1,  1,  1,  2,  1,  1,  2,  1,  1,  1,  1, 17,  8, 15,  7,118,118, 52, 52, 22, 22,  4,  4, 12, 27, 23,  1, 31,  5,  0, 10, 18, {76,82,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Fulah/Adlam/Liberia
    {     87,     1,   149,  421,  421,  427,  436,  444,    0,   69,    2,   70,    4,    5,   72,   14,   15,   16,   17,  920,  283,   61,   76, 6965, 6965, 7083, 7083, 7135, 7135,  227,  240,  382,  394,   22,  216, 2590,   15,    0, 3007, 3121,  6,  6,  9,  8,  1,  1,  1,  1,  2,  1,  1,  2,  1,  1,  1,  1, 17,  8, 15,  7,118,118, 52, 52, 22, 22,  4,  4, 12, 27, 23,  2, 37,  5,  0, 10, 16, {77,82,85},      2,     1,     1,     6,     7,     1,     3,     3 }, // Fulah/Adlam/Mauritania
    {     87,     1,   169,  421,  421,  427,  436,  444,    0,   69,    2,   70,    4,    5,   72,   14,   15,   16,   17,  920,  283,   10,    0, 6965, 6965, 7083, 7083, 7135, 7135,  227,  240,  382,  394,   22,  124, 2627,   15,    0, 3007, 3137,  6,  6,  9,  8,  1,  1,  1,  1,  2,  1,  1,  2,  1,  1,  1,  1, 17,  8, 13,  5,118,118, 52, 52, 22, 22,  4,  4, 12, 27, 23,  1, 33,  5,  0, 10, 18, {78,71,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Fulah/Adlam/Nigeria
    {     87,     1,   170,  421,  421,  427,  436,  444,    0,   69,    2,   70,    4,    5,   72,   14,   15,   16,   17,  920,  283,   10,    0, 6965, 6965, 7083, 7083, 7135, 7135,  227,  240,  382,  394,   22,  222, 2387,   15,    0, 3007, 3155,  6,  6,  9,  8,  1,  1,  1,  1,  2,  1,  1,  2,  1,  1,  1,  1, 17,  8, 13,  5,118,118, 52, 52, 22, 22,  4,  4, 12, 27, 23,  6, 51,  5,  0, 10, 12, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Adlam/Niger
    {     87,     1,   206,  421,  421,  427,  436,  444,    0,   69,    2,   70,    4,    5,   72,   14,   15,   16,   17,  920,  283,   10,    0, 6965, 6965, 7083, 7083, 7135, 7135,  227,  240,  382,  394,   22,  222, 2387,   15,    0, 3007, 3167,  6,  6,  9,  8,  1,  1,  1,  1,  2,  1,  1,  2,  1,  1,  1,  1, 17,  8, 13,  5,118,118, 52, 52, 22, 22,  4,  4, 12, 27, 23,  6, 51,  5,  0, 10, 16, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Adlam/Senegal
    {     87,     1,   209,  421,  421,  427,  436,  444,    0,   69,    2,   70,    4,    5,   72,   14,   15,   16,   17,  920,  283,   61,   76, 6965, 6965, 7083, 7083, 7135, 7135,  227,  240,  382,  394,   22,   18, 2660,   15,    0, 3007, 3183,  6,  6,  9,  8,  1,  1,  1,  1,  2,  1,  1,  2,  1,  1,  1,  1, 17,  8, 15,  7,118,118, 52, 52, 22, 22,  4,  4, 12, 27, 23,  2, 33,  5,  0, 10, 14, {83,76,69},      2,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Adlam/Sierra Leone
    {     87,    66,    37,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   18,   17,  113,  129,   10,    0, 6867, 6867, 6925, 6925, 6952, 6952,  221,  233,    0,    5,   22,  127, 2368,    4,    0, 2993, 3197,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 58, 58, 27, 27, 13, 13,  6,  7,  4, 17, 23,  5, 19,  5,  0,  6, 14, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Latin/Burkina Faso
    {     87,    66,    40,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   18,   17,  113,  129,   10,    0, 6867, 6867, 6925, 6925, 6952, 6952,  221,  233,    0,    5,   22,   11, 2693,    4,    0, 2993, 3211,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 58, 58, 27, 27, 13, 13,  6,  7,  4, 17, 23,  4, 18,  5,  0,  6,  8, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Latin/Cameroon
    {     87,    66,    89,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   18,   17,  113,  129,   23,   38, 6867, 6867, 6925, 6925, 6952, 6952,  221,  233,    0,    5,   22,  184, 2711,    4,    0, 2993, 3219,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 15,  7, 58, 58, 27, 27, 13, 13,  6,  7,  4, 17, 23,  1, 13,  5,  0,  6,  6, {71,77,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Fulah/Latin/Gambia
    {     87,    66,    92,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   18,   17,  113,  129,   23,   38, 6867, 6867, 6925, 6925, 6952, 6952,  221,  233,    0,    5,   22,   15,    0,    4,    0, 2993, 3225,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 15,  7, 58, 58, 27, 27, 13, 13,  6,  7,  4, 17, 23,  3,  0,  5,  0,  6,  5, {71,72,83},      2,     1,     1,     6,     7,     1,     3,     3 }, // Fulah/Latin/Ghana
    {     87,    66,   101,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   18,   17,  113,  129,   10,    0, 6867, 6867, 6925, 6925, 6952, 6952,  221,  233,    0,    5,   22,  127, 2368,    4,    0, 2993, 3230,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 58, 58, 27, 27, 13, 13,  6,  7,  4, 17, 23,  5, 19,  5,  0,  6, 12, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Latin/Guinea-Bissau
    {     87,    66,   102,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   18,   17,  113,  129,   10,    0, 6867, 6867, 6925, 6925, 6952, 6952,  221,  233,    0,    5,   22,  214,    0,    4,    0, 2993, 3230,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 58, 58, 27, 27, 13, 13,  6,  7,  4, 17, 23,  2,  0,  5,  0,  6,  4, {71,78,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Latin/Guinea
    {     87,    66,   134,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   18,   17,  113,  129,   23,   38, 6867, 6867, 6925, 6925, 6952, 6952,  221,  233,    0,    5,   22,   10, 2724,    4,    0, 2993, 3242,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 15,  7, 58, 58, 27, 27, 13, 13,  6,  7,  4, 17, 23,  1, 16,  5,  0,  6,  9, {76,82,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Fulah/Latin/Liberia
    {     87,    66,   149,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   18,   17,  113,  129,   23,   38, 6867, 6867, 6925, 6925, 6952, 6952,  221,  233,    0,    5,   22,  216, 2740,    4,    0, 2993, 3251,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 15,  7, 58, 58, 27, 27, 13, 13,  6,  7,  4, 17, 23,  2, 15,  5,  0,  6,  8, {77,82,85},      2,     1,     1,     6,     7,     1,     3,     3 }, // Fulah/Latin/Mauritania
    {     87,    66,   169,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   18,   17,  113,  129,   10,    0, 6867, 6867, 6925, 6925, 6952, 6952,  221,  233,    0,    5,   22,  124, 2755,    4,    0, 2993, 3259,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 58, 58, 27, 27, 13, 13,  6,  7,  4, 17, 23,  1, 16,  5,  0,  6,  9, {78,71,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Fulah/Latin/Nigeria
    {     87,    66,   170,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   18,   17,  113,  129,   10,    0, 6867, 6867, 6925, 6925, 6952, 6952,  221,  233,    0,    5,   22,  127, 2368,    4,    0, 2993, 3268,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 58, 58, 27, 27, 13, 13,  6,  7,  4, 17, 23,  5, 19,  5,  0,  6,  6, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Latin/Niger
    {     87,    66,   209,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   18,   17,  113,  129,   23,   38, 6867, 6867, 6925, 6925, 6952, 6952,  221,  233,    0,    5,   22,   18, 2771,    4,    0, 2993, 3274,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 15,  7, 58, 58, 27, 27, 13, 13,  6,  7,  4, 17, 23,  2, 18,  5,  0,  6, 11, {83,76,69},      2,     0,     1,     6,     7,     1,     3,     3 }, // Fulah/Latin/Sierra Leone
    {     88,    66,   246,    0,    0,  445,  445,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  937,  186,   10,    0, 7157, 7157, 7225, 7225, 7252, 7252,    3,  135,  421,    5,   22,   94, 2789,    2,    9, 3285, 3293,  6,  6, 10, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 21, 10, 13,  5, 68, 68, 27, 27, 13, 13,  1,  1,  6, 17, 23,  1, 15,  4,  6,  8, 22, {71,66,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // Gaelic/Latin/United Kingdom
    {     89,    66,    92,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38, 7265, 7265, 7297, 7297, 7323, 7323,    0,    0,    0,    5,   22,   15,   50,    2,    9, 3315, 1718,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 32, 32, 26, 26, 13, 13,  2,  2,  4, 17, 23,  3, 10,  4,  6,  2,  5, {71,72,83},      2,     1,     1,     6,     7,     1,     3,     3 }, // Ga/Latin/Ghana
    {     90,    66,   220,    0,    0,  414,  414,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,   78,   10,    0, 7336, 7336, 7384, 7384, 1185, 7418,  168,  168,    0,    5,   22,   22,  405,    4,    0, 3317,  455,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 48, 48, 34, 34, 13, 20,  4,  4,  5, 17, 23,  1,  4,  5,  0,  6,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Galician/Latin/Spain
    {     91,    66,   243,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0, 7438, 7438, 7503, 7503, 7530, 7530,    0,    0,    0,    5,   22,  147, 2804,    0,    0, 3323, 3330,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 65, 65, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3, 19,  4,  0,  7,  7, {85,71,88},      0,     0,     1,     7,     7,     1,     3,     3 }, // Ganda/Latin/Uganda
    {     92,    33,    77,    0,    0,    0,    0,    6,    0,   74,    2,    3,    4,    5,   10,   14,   15,   16,   17,  985,   78,   61,   76, 7543, 7543, 7543, 7543, 7571, 7571,    0,    0,    0,    5,   22,    0,  105,   15,    0, 3337,  143,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  8, 15,  7, 28, 28, 28, 28, 13, 13,  2,  2,  4, 17, 23,  0,  9,  5,  0,  4,  5, {69,84,66},      2,     1,     7,     6,     7,     1,     3,     3 }, // Geez/Ethiopic/Ethiopia
    {     92,    33,    74,    0,    0,    0,    0,    6,    0,   74,    2,    3,    4,    5,   10,   14,   15,   16,   17,  985,   78,   61,   76, 7543, 7543, 7543, 7543, 7571, 7571,    0,    0,    0,    5,   22,    6,    0,   15,    0, 3337,  671,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  8, 15,  7, 28, 28, 28, 28, 13, 13,  2,  2,  4, 17, 23,  3,  0,  5,  0,  4,  4, {69,82,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Geez/Ethiopic/Eritrea
    {     93,    35,    90,    0,    0,  455,  455,    6,    1,    9,    2,    3,    4,    5,   10,   13,   14,   11,   12, 1008,   49,   10,    0, 7584, 7584, 7645, 7645, 7672, 7672,    0,    0,  427,  432,   22,    0, 2823,    4,    0, 3341, 3348,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  8, 13,  5, 61, 61, 27, 27, 13, 13,  2,  2,  5, 29, 23,  1, 12,  5,  0,  7, 10, {71,69,76},      2,     1,     1,     6,     7,     2,     3,     3 }, // Georgian/Georgian/Georgia
    {     94,    66,    91,    0,    0,  463,  463,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   18,   16,  404,   49,   10,    0, 7685, 7685, 7744, 7764, 4510, 4510,    0,    0,  461,    5,   22,   22,   83,    4,    0, 3358, 3365,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 59, 59, 20, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  5,  0,  7, 11, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // German/Latin/Germany
    {     94,    66,    16,    0,    0,  463,  463,    6,    1,    9,    2,    3,    4,    5,   10,   13,   14,   18,   16,  404,   49,   10,    0, 7685, 7685, 7744, 7764, 4510, 4510,    0,    0,  461,    5,   22,   22,   83,   15,    0, 3376, 3376,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 59, 59, 20, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  5,  0, 24, 10, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // German/Latin/Austria
    {     94,    66,    23,    0,    0,  463,  463,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   18,   16,  404,   49,   10,    0, 7685, 7685, 7744, 7764, 4510, 4510,    0,    0,  461,    5,   22,   22,   83,    4,    0, 3358, 3400,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 59, 59, 20, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  5,  0,  7,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // German/Latin/Belgium
    {     94,    66,   117,    0,    0,  463,  463,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   18,   16,  404,   49,   10,    0, 7685, 7685, 7744, 7764, 4510, 4510,    0,    0,  461,    5,   22,   22,   83,    4,    0, 3358, 3407,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 59, 59, 20, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  5,  0,  7,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // German/Latin/Italy
    {     94,    66,   136,    0,    0,  463,  463,    6,    0,   17,    2,    3,    4,    5,   10,   13,   14,   18,   16,  404,   49,   10,    0, 7685, 7685, 7744, 7764, 4510, 4510,    0,    0,  461,    5,   22,    0, 2835,   15,    0, 3358, 3414,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 59, 59, 20, 27, 13, 13,  2,  2,  5, 17, 23,  0, 17,  5,  0,  7, 13, {67,72,70},      2,     0,     1,     6,     7,     1,     3,     3 }, // German/Latin/Liechtenstein
    {     94,    66,   138,    0,    0,  463,  463,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   18,   16,  404,   49,   10,    0, 7685, 7685, 7744, 7764, 4510, 4510,    0,    0,  461,    5,   22,   22,   83,    4,    0, 3358, 3427,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 59, 59, 20, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  5,  0,  7,  9, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // German/Latin/Luxembourg
    {     94,    66,   226,    0,    0,  463,  463,    6,    0,   17,    2,    3,    4,    5,   10,   13,   14,   18,   16,  404,   49,   10,    0, 7685, 7685, 7744, 7764, 4510, 4510,    0,    0,  461,    5,   22,    0, 2835,   15,   65, 3436, 3436,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 59, 59, 20, 27, 13, 13,  2,  2,  5, 17, 23,  0, 17,  5,  5, 21,  7, {67,72,70},      2,     0,     1,     6,     7,     1,     3,     3 }, // German/Latin/Switzerland
    {     96,    39,    94,    0,    0,  472,  472,    6,    1,    0,    2,    3,    4,    5,    6,   11,   12,   14,   15,  113,  129,   23,   38, 7791, 7791, 7845, 7845, 7872, 7872,  231,  244,    0,    5,   22,   22, 2852,    4,    0, 3457, 3465,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  6, 15,  7, 54, 54, 27, 27, 13, 13,  4,  4,  4, 17, 23,  1,  4,  5,  0,  8,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Greek/Greek/Greece
    {     96,    39,    63,    0,    0,  472,  472,    6,    1,    0,    2,    3,    4,    5,    6,   11,   12,   14,   15,  113,  129,   23,   38, 7791, 7791, 7845, 7845, 7872, 7872,  231,  244,    0,    5,   22,   22, 2852,    4,    0, 3457, 3471,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  6, 15,  7, 54, 54, 27, 27, 13, 13,  4,  4,  4, 17, 23,  1,  4,  5,  0,  8,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Greek/Greek/Cyprus
    {     97,    66,   183,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0, 7885, 7885, 7885, 7885,   83,   83,    0,    0,    0,    5,   22,  236,    0,   15,    0, 3477, 3484,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 53, 53, 53, 53, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  7,  8, {80,89,71},      0,     0,     7,     6,     7,     1,     3,     3 }, // Guarani/Latin/Paraguay
    {     98,    40,   110,    0,    0,  481,  481,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  265,  129,  367,  383, 7938, 7938, 7990, 7990, 8021, 8021,    0,    0,  466,    5,   22,  120, 2856,    2,    9, 3492, 3499,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 16,  8, 52, 52, 31, 31, 18, 18,  2,  2,  4, 17, 23,  1, 13,  4,  6,  7,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Gujarati/Gujarati/India
    {     99,    66,   124,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0, 8039, 8039, 8100, 8100, 8127, 8127,  235,  248,    0,    5,   22,  175,  978,    2,    9, 3503, 1307,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 61, 61, 27, 27, 13, 13,  6,  3,  4, 17, 23,  3, 17,  4,  6,  8,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Gusii/Latin/Kenya
    {    101,    66,   169,    0,    0,  490,  499,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  804,  129,   10,    0, 8140, 8140, 8191, 8191, 8218, 8218,  241,  251,    0,  470,  511,  124, 2869,   15,    0, 3511, 3259,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 13,  5, 51, 51, 27, 27, 13, 13,  6,  5,  5, 41, 47,  1, 15,  5,  0,  5,  8, {78,71,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Hausa/Latin/Nigeria
    {    101,     4,   169,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0, 8231, 8231, 8287, 8287,   83,   83,    0,    0,    0,    5,   22,  124, 2884,   15,    0,    0,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 30, 30, 13, 13,  2,  2,  4, 17, 23,  1,  6,  5,  0,  0,  0, {78,71,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Hausa/Arabic/Nigeria
    {    101,     4,   222,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0, 8231, 8231, 8287, 8287,   83,   83,    0,    0,    0,    5,   22,    0,    0,   15,    0,    0,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 56, 56, 30, 30, 13, 13,  2,  2,  4, 17, 23,  0,  0,  5,  0,  0,  0, {83,68,71},      2,     1,     6,     5,     6,     1,     3,     3 }, // Hausa/Arabic/Sudan
    {    101,    66,    92,    0,    0,  490,  499,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  804,  129,   23,   38, 8140, 8140, 8191, 8191, 8218, 8218,  241,  251,    0,  470,  511,   15, 2890,   15,    0, 3511, 3225,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 51, 51, 27, 27, 13, 13,  6,  5,  5, 41, 47,  3, 13,  5,  0,  5,  4, {71,72,83},      2,     1,     1,     6,     7,     1,     3,     3 }, // Hausa/Latin/Ghana
    {    101,    66,   170,    0,    0,  490,  499,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  804,  129,   10,    0, 8140, 8140, 8191, 8191, 8218, 8218,  241,  251,    0,  470,  511,  127, 2903,   15,    0, 3511, 3516,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 13,  5, 51, 51, 27, 27, 13, 13,  6,  5,  5, 41, 47,  5, 29,  5,  0,  5,  5, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Hausa/Latin/Niger
    {    102,    66,   248,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  129,   23,   38, 8317, 8317, 8373, 8373,   83,   83,    0,    0,    0,    5,   22,   10,    0,    2,    9, 3521, 3535,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 56, 56, 20, 20, 13, 13,  2,  2,  4, 17, 23,  1,  0,  4,  6, 14, 19, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Hawaiian/Latin/United States
    {    103,    47,   116,    0,    0,  507,  507,    6,    0,    1,    2,    3,   35,   37,   10,   15,   15,   17,   17, 1027,  885,   11,    1, 8393, 8393, 8457, 8457, 8502, 8502,  247,  256,  558,    5,   22,   48, 2932,   70,   77, 3554, 3559,  6,  6,  6,  6,  1,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1, 18,  8, 12,  4, 64, 64, 45, 45, 20, 20,  6,  5,  4, 17, 23,  1,  7,  7,  9,  5,  5, {73,76,83},      2,     1,     7,     5,     6,     1,     3,     3 }, // Hebrew/Hebrew/Israel
    {    105,    29,   110,    0,    0,  513,  522,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  129,   61,   76, 8522, 8522, 8574, 8574, 8605, 8605,   82,  203,  562,    5,   22,  120, 2939,    2,    0, 3564,  664,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 52, 52, 31, 31, 18, 18,  2,  2,  4, 17, 23,  1, 12,  4,  0,  6,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Hindi/Devanagari/India
    {    105,    66,   110,    0,    0,  530,  540,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  804,  186,   23,   38, 8623, 8623, 8689, 8689, 8727, 8727,    0,    0,    0,    5,   22,  120, 1368,    2,    0, 3570, 3583,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 66, 66, 38, 38, 21, 21,  2,  2,  5, 17, 23,  1, 12,  4,  0, 13,  6, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Hindi/Latin/India
    {    107,    66,   108,    0,    0,  549,  549,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   12,   11, 1045, 1064,   11,    1, 8748, 8748, 8799, 8799, 8817, 8817,  253,  261,  566,    5,   22,  237, 2951,    4,    0, 3589, 3595,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19, 13, 12,  4, 51, 51, 18, 18, 16, 16,  3,  3,  4, 17, 23,  2, 13,  5,  0,  6, 12, {72,85,70},      2,     0,     1,     6,     7,     2,     3,     3 }, // Hungarian/Latin/Hungary
    {    108,    66,   109,    0,    0,  289,  289,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   18,   16,  404,  885,   10,    0, 8833, 8833, 8913, 8913, 8947, 8947,  256,  264,  570,    5,   22,  160, 2964,    4,    0, 3607, 3615,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 80, 80, 34, 34, 13, 13,  4,  4,  4, 17, 23,  3, 13,  5,  0,  8,  6, {73,83,75},      0,     0,     1,     6,     7,     1,     3,     3 }, // Icelandic/Latin/Iceland
    {    109,    66,   258,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,    0,    0,   15,    0, 3621,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  0,  0,  5,  0,  3,  0,    {0,0,0},      2,     1,     1,     6,     7,     1,     3,     3 }, // Ido/Latin/world
    {    110,    66,   169,    0,    0,  557,  566,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  129,   10,    0, 8960, 8960, 9013, 9013,   83,   83,  260,  268,    0,    5,   22,  124, 2977,    2,    9, 3624, 3628,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 13,  5, 53, 53, 28, 28, 13, 13,  7,  7,  4, 17, 23,  1,  5,  4,  6,  4,  8, {78,71,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Igbo/Latin/Nigeria
    {    111,    66,    83,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1077,  885,  213,  213, 9041, 9110, 9182, 9182,   83, 9209,  267,  275,    0,    5,   22,   22,  405,    4,    0, 3636, 3647,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 12,  4, 69, 72, 27, 27, 13, 13,  3,  3,  4, 17, 23,  1,  4,  5,  0, 11,  5, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Inari Sami/Latin/Finland
    {    112,    66,   111,    0,    0,  574,  584,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  786,   78,  212,  212, 9222, 9222, 9264, 9264, 9291, 9291,    0,    0,    0,    5,   22,  185, 2982,    2,    0, 1776, 1776,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 42, 42, 27, 27, 13, 13,  2,  2,  4, 17, 23,  2, 16,  4,  0,  9,  9, {73,68,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // Indonesian/Latin/Indonesia
    {    114,    66,   258,    0,    0,  414,  414,    6,    1,    0,    2,    3,    4,    5,   10,   16,   17,   14,   15, 1095,  394,   10,    0, 9304, 9304, 9360, 9360, 9387, 9387,    0,    0,    0,    5,   22,    0,    0,   15,   58, 3652, 3663,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 26, 10, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  5, 17, 23,  0,  0,  5,  7, 11,  5,    {0,0,0},      2,     1,     1,     6,     7,     2,     3,     3 }, // Interlingua/Latin/world
    {    115,    66,    75,    0,    0,    0,  414,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  113,  180,   10,    0, 9400, 9400, 9451, 9451, 9485, 9485,  270,  278,  574,  232,  249,   22,  405,   15,   86, 3668, 3679,  6,  6,  6,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  6, 13,  5, 51, 51, 34, 34, 13, 13,  9,  8,  7, 17, 23,  1,  4,  5,  6, 11,  7, {69,85,82},      2,     1,     1,     6,     7,     2,     3,     3 }, // Interlingue/Latin/Estonia
    {    116,    18,    41,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549, 1121,   61,   76, 9498, 9498, 9498, 9498,   83,   83,    0,    0,    0,    5,   22,  239,    0,   15,    0, 3686, 3692,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18, 10, 15,  7, 54, 54, 54, 54, 13, 13,  2,  2,  4, 17, 23,  3,  0,  5,  0,  6,  4, {67,65,68},      2,     0,     7,     6,     7,     1,     3,     3 }, // Inuktitut/Canadian Aboriginal/Canada
    {    116,    66,    41,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  239,    0,   15,    0,    0,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3,  0,  5,  0,  0,  0, {67,65,68},      2,     0,     7,     6,     7,     1,     3,     3 }, // Inuktitut/Latin/Canada
    {    118,    66,   114,    0,    0,  445,  445,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  186,   10,    0, 9552, 9552, 9626, 9626, 9662, 9662,  279,  286,  581,    5,   22,   22,   83,    2,    9, 3696, 3703,  6,  6, 10, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 74, 74, 36, 36, 13, 13,  4,  4,  6, 17, 23,  1,  4,  4,  6,  7,  4, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Irish/Latin/Ireland
    {    118,    66,   246,    0,    0,  445,  445,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  186,   10,    0, 9552, 9552, 9626, 9626, 9662, 9662,  279,  286,  581,    5,   22,   94, 2998,    2,    9, 3696, 3707,  6,  6, 10, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 74, 74, 36, 36, 13, 13,  4,  4,  6, 17, 23,  1, 14,  4,  6,  7, 19, {71,66,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // Irish/Latin/United Kingdom
    {    119,    66,   117,    0,    0,  414,  414,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   14,   15,  113,   78,   10,    0, 9675, 9675, 9731, 9731, 4694, 4694,    0,    0,    0,    5,   22,   22,  405,    4,    0, 3726, 3734,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  0,  8,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Italian/Latin/Italy
    {    119,    66,   203,    0,    0,  414,  414,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   14,   15,  113,   78,   10,    0, 9675, 9675, 9731, 9731, 4694, 4694,    0,    0,    0,    5,   22,   22,  405,    4,    0, 3726, 3740,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  0,  8, 10, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Italian/Latin/San Marino
    {    119,    66,   226,    0,    0,  414,  414,    6,    0,   17,    2,    3,    4,    5,   10,   11,   12,   19,   20,    0,   49,   10,    0, 9675, 9675, 9731, 9731, 4694, 4694,    0,    0,    0,    5,   22,    0, 3012,   15,   65, 3726, 3750,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  4, 17, 23,  0, 15,  5,  5,  8,  8, {67,72,70},      2,     0,     1,     6,     7,     1,     3,     3 }, // Italian/Latin/Switzerland
    {    119,    66,   253,    0,    0,  414,  414,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   14,   15,  113,   78,   10,    0, 9675, 9675, 9731, 9731, 4694, 4694,    0,    0,    0,    5,   22,   22,  405,    4,    0, 3726, 3758,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  0,  8, 18, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Italian/Latin/Vatican City
    {    120,    53,   120,  183,  183,  183,  183,    6,    0,    1,    2,    3,    4,    5,   10,   51,   52,   53,   54,  513,  821,  391,    1, 9758, 9758, 9785, 9785, 9785, 9785,  283,  290,  587,  590,   22,  145, 3027,    2,    9, 3776, 3776,  5,  5,  5,  5,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 13, 10, 13,  4, 27, 27, 13, 13, 13, 13,  2,  2,  3, 17, 23,  1,  3,  4,  6,  3,  2, {74,80,89},      0,     0,     7,     6,     7,     1,     3,     3 }, // Japanese/Japanese/Japan
    {    121,    66,   111,    0,    0,  593,  603,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  394,   10,    0, 9798, 9798, 9838, 9838, 9866, 9866,  285,  292,  607,    5,   22,  185, 2982,   15,    0, 3779, 3783,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 40, 40, 28, 28, 13, 13,  4,  5,  4, 17, 23,  2, 16,  5,  0,  4,  9, {73,68,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // Javanese/Latin/Indonesia
    {    122,    66,   169,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  137,  155,   10,    0, 9879, 9879, 9922, 9922,   83,   83,    0,    0,    0,    5,   22,  124, 3030,   15,    0, 3792,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 43, 43, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  7,  5,  0,  4,  0, {78,71,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Jju/Latin/Nigeria
    {    123,    66,   206,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  113,  129,   10,    0, 9949, 9949, 9998, 9998,10025,10025,    0,    0,    0,    5,   22,  127, 3037,    4,    0, 3796, 3801,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 49, 49, 27, 27, 13, 13,  2,  2,  4, 17, 23,  5, 16,  5,  0,  5,  7, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Jola-Fonyi/Latin/Senegal
    {    124,    66,    43,    0,    0,  143,  143,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1131,  186,   10,    0,10038,10038,10110,10110,10137,10137,   82,  203,    0,    5,   22,  242, 3053,    4,   20, 3808, 3820,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27, 10, 13,  5, 72, 72, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1, 18,  5,  7, 12, 10, {67,86,69},      2,     1,     1,     6,     7,     1,     3,     3 }, // Kabuverdianu/Latin/Cape Verde
    {    125,    66,     4,    0,    0,  612,  620,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  113,  129,   23,   38,10150,10183,10233,10260,10289,10302,  289,  297,  611,  618,   22,  208, 3071,    0,    0, 3830, 3839,  6,  6,  8,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 15,  7, 33, 50, 27, 29, 13, 13,  7,  9,  7, 21, 23,  2, 14,  4,  0,  9,  8, {68,90,68},      2,     1,     6,     5,     6,     1,     3,     3 }, // Kabyle/Latin/Algeria
    {    126,    66,    40,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   19,   20,   86, 1158,   10,    0,10315,10315,10315,10315,10368,10368,    0,    0,    0,    5,   22,   11, 3085,   15,    0, 3847, 3851,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 53, 53, 53, 53, 20, 20,  2,  2,  4, 17, 23,  4,  9,  5,  0,  4,  7, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Kako/Latin/Cameroon
    {    127,    66,    95,    0,    0,  627,  627,    6,    1,    0,    2,    3,   48,    5,   63,   12,   11,   20,   19,   86,  103,  212,  212,10388,10388,10485,10485,10512,10512,    0,    0,    0,    5,   22,  160, 3094,    2,   92, 3858, 3869,  6,  6, 11, 11,  1,  1,  1,  1,  1,  1,  1,  4,  1,  1,  1,  1, 17, 10, 13,  5, 97, 97, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3, 19,  4,  5, 11, 16, {68,75,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Kalaallisut/Latin/Greenland
    {    128,    66,   124,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,10525,10525,10577,10577,10604,10604,  296,  306,    0,    5,   22,  175, 3113,    2,    9, 3885, 3893,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 52, 52, 27, 27, 13, 13,  6, 10,  4, 17, 23,  3, 19,  4,  6,  8, 12, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Kalenjin/Latin/Kenya
    {    129,    66,   124,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,10617,10617,10690,10690,10717,10717,  302,  316,    0,    5,   22,  175, 3132,    2,    9, 3905, 1307,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 73, 73, 27, 27, 13, 13,  9,  7,  4, 17, 23,  3, 16,  4,  6,  7,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Kamba/Latin/Kenya
    {    130,    56,   110,    0,    0,  638,  650,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  129,  367,  383,10730,10730,10783,10783,10815,10815,  311,  323,  639,  647,   22,  120, 3148,    2,    9, 3912, 3917,  6,  6, 12, 11,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 16,  8, 53, 53, 32, 32, 19, 19,  9,  7,  8, 35, 23,  1, 13,  4,  6,  5,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Kannada/Kannada/India
    {    132,     4,   110,  661,  661,  667,  677,   67,   21,   22,   23,   40,   35,   41,   44,   14,   15,   16,   17,  549,  567,   61,   76,10834,10834,10885,10885,10934,10934,  320,  330,    0,    5,   22,  120, 3161,    2,    0, 3921, 3926,  6,  6, 10,  9,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 18,  6, 15,  7, 51, 51, 49, 49, 13, 13,  6,  6,  4, 17, 23,  1, 16,  4,  0,  5,  9, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Kashmiri/Arabic/India
    {    132,    29,   110,    0,    0,  686,  695,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  129,  127,  127,10947,10996,10947,11045,11092,11092,  326,  336,    0,    5,   22,  120, 3177,   15,    0, 3935, 3940,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 49, 49, 49, 47, 13, 13,  5,  5,  4, 17, 23,  1, 11,  5,  0,  5, 10, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Kashmiri/Devanagari/India
    {    133,    27,   123,    0,    0,    0,  703,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15, 1168,   49,   10,    0,11105,11105,11160,11160,11180,11180,    0,    0,  196,  682,  699,  243, 3188,    4,    0, 3950, 3960,  6,  6,  6, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22,  8, 13,  5, 55, 55, 20, 20, 13, 13,  2,  2,  4, 17, 23,  1, 17,  5,  0, 10,  9, {75,90,84},      2,     1,     1,     6,     7,     1,     3,     3 }, // Kazakh/Cyrillic/Kazakhstan
    {    134,    66,    40,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,   11,    0,   15,    0, 3969,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  4,  0,  5,  0,  6,  0, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Kenyang/Latin/Cameroon
    {    135,    60,    39,    0,    0,  713,  722,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  129,   61,   76,11193,11238,11284,11284,11323,11323,    0,    0,  722,    5,   22,  244, 3205,    0,   45, 3975, 3980,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  6, 15,  7, 45, 46, 39, 39, 13, 13,  2,  2,  2, 17, 23,  1, 11,  4,  6,  5,  7, {75,72,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // Khmer/Khmer/Cambodia
    {    136,    66,    99,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  245,    0,   15,    0, 3987,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  7,  0, {71,84,81},      2,     1,     7,     6,     7,     1,     3,     3 }, // Kiche/Latin/Guatemala
    {    137,    66,   124,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,11336,11336,11398,11398,11425,11425,  331,  341,    0,    5,   22,  175, 3216,    2,    9, 3994, 1307,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 62, 62, 27, 27, 13, 13,  6,  8,  4, 17, 23,  3, 16,  4,  6,  6,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Kikuyu/Latin/Kenya
    {    138,    66,   194,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   16,   17,  163,  103,   10,    0,11438,11438,11521,11521,   83,   83,    0,    0,    0,    5,   22,  197,    0,   15,    0, 4000, 4011,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 83, 83, 34, 34, 13, 13,  2,  2,  4, 17, 23,  2,  0,  5,  0, 11,  8, {82,87,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Kinyarwanda/Latin/Rwanda
    {    141,    29,   110,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  283,   61,   76,11555,11555,11555,11555,11605,11623,  337,  349,  724,    5,   22,  120, 2939,    2,    9, 4019,  664,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  6, 15,  7, 50, 50, 50, 50, 18, 19,  4,  4,  4, 17, 23,  1, 12,  4,  6,  6,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Konkani/Devanagari/India
    {    142,    63,   218,    0,    0,  731,  731,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1190, 1208,  404,  127,11642,11642,11669,11669,11669,11669,  341,  353,  728,    5,   22,  246, 3232,    2,    9, 4025, 4028,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  9, 16,  7, 27, 27, 13, 13, 13, 13,  2,  2,  3, 17, 23,  1,  6,  4,  6,  3,  4, {75,82,87},      0,     0,     7,     6,     7,     1,     3,     3 }, // Korean/Korean/South Korea
    {    142,    63,    50,    0,    0,  731,  731,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1190, 1208,  169,    0,11642,11642,11669,11669,11669,11669,  341,  353,  728,    5,   22,  247, 3238,    2,    9, 4025, 4032,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  9, 13,  5, 27, 27, 13, 13, 13, 13,  2,  2,  3, 17, 23,  3,  6,  4,  6,  3,  2, {67,78,89},      2,     1,     1,     6,     7,     1,     3,     3 }, // Korean/Korean/China
    {    142,    63,   174,    0,    0,  731,  731,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1190, 1208,  404,  127,11642,11642,11669,11669,11669,11669,  341,  353,  728,    5,   22,  246, 3244,    2,    9, 4025, 4034,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  9, 16,  7, 27, 27, 13, 13, 13, 13,  2,  2,  3, 17, 23,  1, 16,  4,  6,  3, 11, {75,80,87},      0,     0,     1,     6,     7,     1,     3,     3 }, // Korean/Korean/North Korea
    {    144,    66,   145,    0,    0,    0,    0,    6,    0,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  129,   10,    0,11682,11682,11735,11735,11762,11762,  343,  355,    0,    5,   22,  127, 3260,    0,    0, 4045, 4060,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 53, 53, 27, 27, 13, 13,  6,  6,  4, 17, 23,  5, 16,  4,  0, 15,  5, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Koyraboro Senni/Latin/Mali
    {    145,    66,   145,    0,    0,    0,    0,    6,    0,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  129,   10,    0,11775,11775,11827,11827,11762,11762,  343,  355,    0,    5,   22,  127, 3260,    0,    0, 4065, 4060,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 52, 52, 27, 27, 13, 13,  6,  6,  4, 17, 23,  5, 16,  4,  0, 11,  5, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Koyra Chiini/Latin/Mali
    {    146,    66,   134,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   23,   38,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,   10,    0,   15,    0, 4076,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  6,  0, {76,82,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Kpelle/Latin/Liberia
    {    146,    66,   102,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  214,    0,   15,    0, 4076,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  2,  0,  5,  0,  6,  0, {71,78,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Kpelle/Latin/Guinea
    {    148,    66,   239,    0,    0,  738,  738,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1217,   49,   10,    0,11854,11854,11896,11896,11923,11923,  349,  361,    0,    5,   22,  126, 3276,    4,   20, 4082, 4098,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27, 10, 13,  5, 42, 42, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1, 12,  5,  7, 16,  7, {84,82,89},      2,     1,     1,     6,     7,     1,     3,     3 }, // Kurdish/Latin/Turkey
    {    149,    66,    40,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   11,   12,  113,  129,   10,    0,11936,11936,12024,12024,12053,12053,  351,  363,    0,    5,   22,   11, 3288,    4,    0, 4105, 4111,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 88, 88, 29, 29, 13, 13,  4,  4,  4, 17, 23,  4, 13,  5,  0,  6,  7, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Kwasio/Latin/Cameroon
    {    150,    27,   128,    0,    0,  745,  745,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14, 1244,  129,   10,    0,12066,12066,12122,12122,12159,12159,  355,  367,  196,  731,   22,  250, 3301,    4,    0, 4118, 4126,  6,  6, 10, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  6, 13,  5, 56, 56, 37, 37, 13, 13,  5, 14,  4, 18, 23,  3, 15,  5,  0,  8, 10, {75,71,83},      2,     1,     1,     6,     7,     1,     3,     3 }, // Kyrgyz/Cyrillic/Kyrgyzstan
    {    151,    66,   248,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38,12172,12172,12172,12172,   83,12258,    0,    0,    0,    5,   22,   10,    0,   15,    0, 4136, 4148,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 86, 86, 86, 86, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0, 12, 22, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Lakota/Latin/United States
    {    152,    66,   230,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   15,   15,   17,   17,    0,  186,   10,    0,12271,12271,12333,12333,12368,12368,  360,  381,    0,    5,   22,  121, 3316,   15,    0, 4170, 4178,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 62, 62, 35, 35, 13, 13,  3,  3,  4, 17, 23,  3, 22,  5,  0,  8,  9, {84,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Langi/Latin/Tanzania
    {    153,    65,   129,    0,    0,    0,  755,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1267,  129,  420,    1,12381,12381,12437,12437,12472,12472,  363,  384,    0,    5,   22,  253, 3338,    2,   65, 4187, 4187,  6,  6,  6,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  8, 27,  4, 56, 56, 35, 35, 16, 16,  8,  8,  4, 17, 23,  1,  7,  4,  5,  3,  3, {76,65,75},      0,     0,     7,     6,     7,     1,     3,     3 }, // Lao/Lao/Laos
    {    154,    66,   253,    0,    0,  406,  406,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1286, 1309,   10,    0,12488,12488,12572,12572,   83,   83,    0,    0,    0,    5,   22,   22,   83,   15,    0, 4190, 4196,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  8, 13,  5, 84, 84, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  0,  6, 16, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Latin/Latin/Vatican City
    {    155,    66,   131,    0,    0,  267,  267,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1317,   49,   10,    0,12599,12670,12741,12791,12841,12841,  371,  392,  749,    5,   22,   22, 3345,    4,    0, 4212, 4220,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 26,  8, 13,  5, 71, 71, 50, 50, 13, 13, 14, 11,  5, 17, 23,  1,  4,  5,  0,  8,  7, {69,85,82},      2,     1,     1,     6,     7,     2,     3,     3 }, // Latvian/Latin/Latvia
    {    158,    66,    57,    0,    0,  764,  764,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  129,   10,    0,12854,12854,12953,12953,12980,12980,  385,  403,    0,    5,   22,   11, 3349,    4,    0, 4227, 4234,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 99, 99, 27, 27, 13, 13,  8,  6,  4, 17, 23,  2, 16,  5,  0,  7, 30, {67,68,70},      2,     1,     1,     6,     7,     1,     3,     3 }, // Lingala/Latin/Congo - Kinshasa
    {    158,    66,     7,    0,    0,  764,  764,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  129,   10,    0,12854,12854,12953,12953,12980,12980,  385,  403,    0,    5,   22,  254, 3365,    4,    0, 4227, 4264,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 99, 99, 27, 27, 13, 13,  8,  6,  4, 17, 23,  2, 16,  5,  0,  7,  6, {65,79,65},      2,     1,     1,     6,     7,     1,     3,     3 }, // Lingala/Latin/Angola
    {    158,    66,    46,    0,    0,  764,  764,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  129,   10,    0,12854,12854,12953,12953,12980,12980,  385,  403,    0,    5,   22,   11, 3381,    4,    0, 4227, 4270,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 99, 99, 27, 27, 13, 13,  8,  6,  4, 17, 23,  4, 16,  5,  0,  7, 26, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Lingala/Latin/Central African Republic
    {    158,    66,    56,    0,    0,  764,  764,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  129,   10,    0,12854,12854,12953,12953,12980,12980,  385,  403,    0,    5,   22,   11, 3381,    4,    0, 4227, 4296,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 99, 99, 27, 27, 13, 13,  8,  6,  4, 17, 23,  4, 16,  5,  0,  7,  5, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Lingala/Latin/Congo - Brazzaville
    {    160,    66,   137,    0,    0,  773,  773,    6,    1,    9,    2,    3,   48,    5,   63,   13,   14,   13,   14, 1343,  103,   10,    0,12993,12993,13081,13081,13101,13101,  393,  409,  754,    5,   22,   22, 3397,    4,    0, 4301, 4309,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  4,  1,  1,  1,  1, 27, 10, 13,  5, 88, 88, 20, 20, 13, 13,  9,  6,  6, 17, 23,  1,  5,  5,  0,  8,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Lithuanian/Latin/Lithuania
    {    161,    66,   258,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,    0,    0,   15,    0, 4316,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  0,  0,  5,  0, 11,  0,    {0,0,0},      2,     1,     1,     6,     7,     1,     3,     3 }, // Lojban/Latin/world
    {    162,    66,    91,    0,    0,  781,  781,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   18,   16,  404,  180,   11,    1,13114,13114,13166,13166,13193,13193,  402,  415,    0,    5,   22,   22,  405,    4,    0, 4327, 4341,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 12,  4, 52, 52, 27, 27, 13, 13,  9, 10,  4, 17, 23,  1,  4,  5,  0, 14,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Lower Sorbian/Latin/Germany
    {    163,    66,    91,    0,    0,  267,  267,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   18,   16, 1370,   50,  447,  469,13206,13206,13270,13270, 4510, 4510,    0,    0,    0,    5,   22,   22,   83,    4,    0, 4347, 4361,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  7, 22, 10, 64, 64, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  0, 14, 11, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Low German/Latin/Germany
    {    163,    66,   165,    0,    0,  267,  267,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   18,   16, 1370,   50,  447,  469,13206,13206,13270,13270, 4510, 4510,    0,    0,    0,    5,   22,   22,   83,    4,    0, 4347, 4372,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  7, 22, 10, 64, 64, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  0, 14, 12, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Low German/Latin/Netherlands
    {    164,    66,    57,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  129,   10,    0,13297,13297,13346,13346,13373,13373,  411,  425,    0,    5,   22,   11, 3402,    0,    0, 4384, 4392,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 49, 49, 27, 27, 13, 13,  5,  6,  4, 17, 23,  2, 17,  4,  0,  8, 16, {67,68,70},      2,     1,     1,     6,     7,     1,     3,     3 }, // Luba-Katanga/Latin/Congo - Kinshasa
    {    165,    66,   225,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  160,    0,   15,    0, 4408,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  2,  0,  5,  0, 15,  0, {83,69,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Lule Sami/Latin/Sweden
    {    165,    66,   175,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  160,    0,   15,    0, 4408,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  2,  0,  5,  0, 15,  0, {78,79,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Lule Sami/Latin/Norway
    {    166,    66,   124,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,13386,13386,13454,13454,13481,13481,  416,  431,    0,    5,   22,  175, 3419,    0,    0, 4423, 1307,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 68, 68, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3, 16,  4,  0,  6,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Luo/Latin/Kenya
    {    167,    66,   138,    0,    0,  788,  788,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   18,   16,  404,   49,   10,    0,13494,13494,13558,13585, 4510, 4510,  418,  433,  461,    5,   22,   22,   83,    4,    0, 4429, 4429,  6,  6, 10, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 64, 64, 27, 34, 13, 13,  5,  8,  5, 17, 23,  1,  4,  5,  0, 14, 10, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Luxembourgish/Latin/Luxembourg
    {    168,    66,   124,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   13,   14,   18,   16,    0,  186,   10,    0,13619,13619,13693,13693,   83,   83,  168,  168,    0,    5,   22,  175, 3435,    2,   97, 4443, 1307,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 74, 74, 20, 20, 13, 13,  4,  4,  4, 17, 23,  3, 16,  4,  6,  7,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Luyia/Latin/Kenya
    {    169,    27,   140,    0,    0,  150,  150,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   18,   16,    0,  180,   10,    0,13713,13713,13766,13766, 3069, 3069,  423,  441,  760,    5,   22,  256, 3451,    4,    0, 4450, 4460,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 13,  5, 53, 53, 34, 34, 13, 13,  7,  5,  5, 17, 23,  4, 16,  5,  0, 10, 18, {77,75,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Macedonian/Cyrillic/Macedonia
    {    170,    66,   230,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,13800,13800,13861,13861, 1284, 1284,  430,  446,    0,    5,   22,  121, 3467,    2,    0, 4478, 2268,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 61, 61, 27, 27, 13, 13,  5,  9,  4, 17, 23,  3, 20,  4,  0,  9,  8, {84,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Machame/Latin/Tanzania
    {    171,    29,   110,    0,    0,  513,  522,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  129,   61,   76,13888,13888, 8574, 8574, 8605, 8605,   88,   83,    0,    5,   22,  120, 2939,   15,    0, 4487,  664,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 63, 63, 31, 31, 18, 18,  3,  4,  4, 17, 23,  1, 12,  5,  0,  6,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Maithili/Devanagari/India
    {    172,    66,   160,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,13951,13951,14009,14009,14036,14036,  435,  455,    0,    5,   22,  260,    0,   15,    0, 4493, 4498,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 58, 58, 27, 27, 13, 13,  8, 10,  4, 17, 23,  3,  0,  5,  0,  5, 10, {77,90,78},      2,     1,     7,     6,     7,     1,     3,     3 }, // Makhuwa-Meetto/Latin/Mozambique
    {    173,    66,   230,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,14049,14049,14181,14181,14208,14208,  443,  465,    0,    5,   22,  121, 3467,    2,    9, 4508, 2268,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5,132,132, 27, 27, 13, 13,  4,  5,  4, 17, 23,  3, 20,  4,  6, 10,  8, {84,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Makonde/Latin/Tanzania
    {    174,    66,   141,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   11,   12,   14,   15,  113,  103,   10,    0,14221,14221,14280,14280,14313,14313,    0,    0,    0,    5,   22,  187, 1502,    2,    0, 4518, 4526,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 59, 59, 33, 33, 13, 13,  2,  2,  4, 17, 23,  2,  6,  4,  0,  8, 12, {77,71,65},      0,     0,     1,     6,     7,     1,     3,     3 }, // Malagasy/Latin/Madagascar
    {    175,    74,   110,    0,    0,  798,  811,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1393,  129,   61,   76,14326,14402,14477,14477,14517,14538,    0,    0,  765,  771,   22,  120, 3487,    2,    9, 4538, 4544,  6,  6, 13, 12,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 76, 75, 40, 40, 21, 20,  2,  2,  6, 27, 23,  1, 11,  4,  6,  6,  6, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Malayalam/Malayalam/India
    {    176,    66,   143,    0,    0,  584,  584,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,   79,   23,   38,14558,14558,14600,14600,14627,14627,  447,  470,  749,    5,   22,  191, 3498,    2,    9, 4550, 1875,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  7, 15,  7, 42, 42, 27, 27, 13, 13,  2,  3,  4, 17, 23,  2, 16,  4,  6,  6,  8, {77,89,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Malay/Latin/Malaysia
    {    176,     4,    35,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   15,   14,   17,   16,   91,   79,   61,   76,14640,14640,14640,14640,   83,   83,    0,    0,    0,    5,   22,   10, 3514,    2,    9, 4556, 4566,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 12,  7, 15,  7, 34, 34, 34, 34, 13, 13,  2,  2,  4, 17, 23,  1, 10,  4,  6, 10,  5, {66,78,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Malay/Arabic/Brunei
    {    176,     4,   143,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   15,   14,   17,   16,  196,   79,   61,   76,14640,14640,14640,14640,   83,   83,    0,    0,    0,    5,   22,  191, 3524,    2,    9, 4556, 4571,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  7, 15,  7, 34, 34, 34, 34, 13, 13,  2,  2,  4, 17, 23,  2, 13,  4,  6, 10,  6, {77,89,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Malay/Arabic/Malaysia
    {    176,    66,    35,    0,    0,  584,  584,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,   91,   79,   23,   38,14558,14558,14600,14600,14627,14627,  447,  470,  749,    5,   22,   10, 3537,    2,    9, 4550, 4577,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 12,  7, 15,  7, 42, 42, 27, 27, 13, 13,  2,  3,  4, 17, 23,  1, 12,  4,  6,  6,  6, {66,78,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Malay/Latin/Brunei
    {    176,    66,   111,    0,    0,  584,  584,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  786,   78,  212,  212,14558,14558,14600,14600,14627,14627,  447,  470,  749,    5,   22,  185, 2982,    2,    0, 4550, 1776,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 42, 42, 27, 27, 13, 13,  2,  3,  4, 17, 23,  2, 16,  4,  0,  6,  9, {73,68,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // Malay/Latin/Indonesia
    {    176,    66,   210,    0,    0,  584,  584,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,   79,   23,   38,14558,14558,14600,14600,14627,14627,  447,  470,  749,    5,   22,   10, 3549,    2,    9, 4550, 4583,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  7, 15,  7, 42, 42, 27, 27, 13, 13,  2,  3,  4, 17, 23,  1, 15,  4,  6,  6,  9, {83,71,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Malay/Latin/Singapore
    {    177,    66,   146,    0,    0,  823,  831,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1411,  186,   10,    0,14674,14674,14736,14736,14763,14783,    0,    0,    0,    5,   22,   22, 3564,    2,    0, 4592, 1891,  6,  6,  8,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23, 10, 13,  5, 62, 62, 27, 27, 20, 19,  2,  2,  4, 17, 23,  1,  4,  4,  0,  5,  5, {69,85,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // Maltese/Latin/Malta
    {    179,     9,   110,    0,    0,  838,  838,    6,    0,    1,    2,   39,    4,    5,   10,   14,   15,   16,   17, 1434,  129,   61,   76,14802,14802,14802,14802,14860,14885,  449,  473,    0,    5,   22,  120, 3568,   15,    0, 4597, 4605,  6,  6, 11, 11,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 58, 58, 58, 58, 25, 29,  4,  5,  4, 17, 23,  1, 14,  5,  0,  8,  8, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Manipuri/Bangla/India
    {    179,    78,   110,    0,    0,    0,    0,    6,    0,    1,    2,   75,    4,    5,   10,   14,   15,   16,   17,  265,  283,  479,  494,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  120,    0,   15,    0, 4613,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 15,  8, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  7,  0, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Manipuri/Meitei Mayek/India
    {    180,    66,   115,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,   86,   78,   10,    0,14914,14914,14970,14970,   83,   83,  168,  168,    0,    5,   22,   94,    0,    2,    0, 4620, 4625,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 13,  5, 56, 56, 29, 29, 13, 13,  4,  4,  4, 17, 23,  1,  0,  4,  0,  5, 12, {71,66,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // Manx/Latin/Isle of Man
    {    181,    66,   167,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  394,   23,   38,14999,14999,15046,15046,15073,15073,    0,    0,    0,    5,   22,   10, 3582,   15,    0, 4637, 4642,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 47, 47, 27, 27, 15, 15,  2,  2,  4, 17, 23,  1, 15,  5,  0,  5,  8, {78,90,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Maori/Latin/New Zealand
    {    182,    66,    49,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,   10,    0,   15,    0, 4650,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0, 10,  0, {67,76,80},      0,     0,     1,     6,     7,     1,     3,     3 }, // Mapuche/Latin/Chile
    {    183,    29,   110,    0,    0,  849,  849,    6,    0,    1,    2,   49,    4,    5,   10,   14,   15,   16,   17,  265,  129,   61,   76,15088,15088,15140,15140, 8605, 8605,    0,    0,  562,    5,   22,  120, 2939,    2,    9, 4660,  664,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 52, 52, 31, 31, 18, 18,  2,  2,  4, 17, 23,  1, 12,  4,  6,  5,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Marathi/Devanagari/India
    {    185,    66,   124,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,15171,15171,13861,13861,14208,14208,  453,  478,    0,    5,   22,  175, 3597,    2,    9, 1275, 1307,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 57, 57, 27, 27, 13, 13,  9,  6,  4, 17, 23,  3, 18,  4,  6,  3,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Masai/Latin/Kenya
    {    185,    66,   230,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,15171,15171,13861,13861,14208,14208,  453,  478,    0,    5,   22,  121, 3615,    2,    9, 1275, 4665,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 57, 57, 27, 27, 13, 13,  9,  6,  4, 17, 23,  3, 21,  4,  6,  3,  8, {84,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Masai/Latin/Tanzania
    {    186,     4,   112,    0,    0,    0,    0,   67,   21,   22,   23,   40,   35,   41,   44,   11,   12,   19,   20,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,  798,  802,   22,    0, 3636,   15,    0, 4673, 4680,  6,  6,  6,  6,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 39, 23,  0, 10,  5,  0,  7,  5, {73,82,82},      0,     0,     6,     5,     5,     1,     3,     3 }, // Mazanderani/Arabic/Iran
    {    188,    66,   124,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,15228,15228,15278,15278,15305,15305,  462,  484,    0,    5,   22,  175,  978,    2,    9, 4685, 1307,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 50, 50, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3, 17,  4,  6,  6,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Meru/Latin/Kenya
    {    189,    66,    40,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  137,  103,   10,    0,15318,15318,15318,15318,15366,15366,    0,    0,    0,    5,   22,   11, 3646,   15,    0, 4691, 4696,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18, 10, 13,  5, 48, 48, 48, 48, 20, 20,  2,  2,  4, 17, 23,  4,  5,  5,  0,  5,  7, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Meta/Latin/Cameroon
    {    190,    66,    41,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   23,   38,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  239,    0,   15,    0, 4703,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3,  0,  5,  0, 11,  0, {67,65,68},      2,     0,     7,     6,     7,     1,     3,     3 }, // Mohawk/Latin/Canada
    {    191,    27,   156,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1452,  596,   98,    0,15386,15428,15470,15470,15470,15470,  464,  486,  196,  841,   22,  263, 3651,   15,    0, 4714, 4720,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 35, 10, 15,  5, 42, 42, 20, 20, 20, 20,  4,  4,  4, 17, 23,  1, 13,  5,  0,  6,  6, {77,78,84},      2,     0,     1,     6,     7,     1,     3,     3 }, // Mongolian/Cyrillic/Mongolia
    {    191,    83,    50,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  247, 3664,   15,    0,    0,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3,  4,  5,  0,  0,  0, {67,78,89},      2,     1,     1,     6,     7,     1,     3,     3 }, // Mongolian/Mongolian/China
    {    191,    83,   156,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1487,  596,   98,    0,15490,15490,15532,15555,15578,15578,  468,  490,    0,    5,   22,  263, 3668,    2,    0, 4726, 4726,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 36, 10, 15,  5, 42, 42, 23, 23, 23, 22,  4,  5,  4, 17, 23,  1,  8,  4,  0,  6,  6, {77,78,84},      2,     0,     1,     6,     7,     1,     3,     3 }, // Mongolian/Mongolian/Mongolia
    {    192,    66,   150,    0,    0,    0,    0,    6,    0,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  129,   10,    0,15601,15601,15648,15648,15674,15674,    0,    0,    0,    5,   22,  195, 3676,   15,    0, 4732, 4746,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 47, 47, 26, 26, 13, 13,  2,  2,  4, 17, 23,  2, 14,  5,  0, 14,  5, {77,85,82},      2,     0,     1,     6,     7,     1,     3,     3 }, // Morisyen/Latin/Mauritius
    {    193,    66,    40,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   14,   15,  113,  129,   10,    0,15687,15687,15760,15760,15787,15787,  472,  495,    0,    5,   22,   11, 3690,    2,    9, 4751, 4757,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 73, 73, 27, 27, 13, 13,  5,  5,  4, 17, 23,  4, 10,  4,  6,  6,  7, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Mundang/Latin/Cameroon
    {    194,    66,   248,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   23,   38,15800,15800,15800,15800,   83,   83,    0,    0,    0,    5,   22,   10,    0,   15,    0, 4764,  964,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7,107,107,107,107, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  7, 13, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Muscogee/Latin/United States
    {    195,    66,   162,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,15907,15907,15998,15998,16020,16020,  477,  500,    0,    5,   22,   10, 3700,    2,    0, 4771, 4784,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 91, 91, 22, 22, 13, 13,  7,  5,  4, 17, 23,  1, 15,  4,  0, 13,  8, {78,65,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Nama/Latin/Namibia
    {    197,    66,   248,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  178,    0,   15,    0, 4792,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3,  0,  5,  0, 11,  0, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Navajo/Latin/United States
    {    199,    29,   164,  858,    0,  863,  863,    6,    0,    1,    2,   49,    4,    5,   10,   14,   15,   16,   17,  163,  344,   10,    0,16033,16033,16086,16086,16118,16118,  484,  505,  562,  858,   22,  264, 3715,   15,    0, 4803, 4803,  5,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 13,  5, 53, 53, 32, 32, 17, 17,  9,  7,  4, 19, 23,  4, 14,  5,  0,  6,  5, {78,80,82},      2,     1,     7,     6,     7,     1,     2,     3 }, // Nepali/Devanagari/Nepal
    {    199,    29,   110,  858,    0,  863,  863,    6,    0,    1,    2,   49,    4,    5,   10,   14,   15,   16,   17,  163,  344,   61,   76,16033,16033,16086,16086,16118,16118,  484,  505,  562,  858,   22,  120, 3729,   15,    0, 4803,  664,  5,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 53, 53, 32, 32, 17, 17,  9,  7,  4, 19, 23,  1, 14,  5,  0,  6,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Nepali/Devanagari/India
    {    201,    66,    40,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   14,   15, 1523,   78,   10,    0,16135,16135,16135,16135,   83,   83,  493,  512,    0,    5,   22,   11, 3743,   15,    0, 4809, 4825,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 32,  8, 13,  5,110,110,110,110, 13, 13,  9,  8,  4, 17, 23,  4,  9,  5,  0, 16,  7, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Ngiemboon/Latin/Cameroon
    {    202,    66,    40,  870,  870,  881,  897,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   19,   20,  137,  103,   10,    0,16245,16245,16245,16245,16304,16304,  502,  520,    0,    5,   22,   11, 3752,   15,    0, 4832, 4837, 11, 11, 16,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18, 10, 13,  5, 59, 59, 59, 59, 24, 24,  8, 13,  4, 17, 23,  4,  5,  5,  0,  5,  7, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Ngomba/Latin/Cameroon
    {    203,    66,   169,    0,    0,  906,  915,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,16328,16328,16379,16379,   83,   83,  510,  533,  877,    5,   22,  124, 3757,    2,    0, 4844, 4858,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 51, 51, 32, 32, 13, 13,  9,  8,  8, 17, 23,  1, 14,  4,  0, 14,  8, {78,71,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Nigerian Pidgin/Latin/Nigeria
    {    204,    90,   102,    0,    0,    0,    0,    6,    0,   76,    2,   77,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,16411,16411,16471,16503,16537,16537,  519,  541,    0,    5,   22,  268, 3771,   15,    0, 4866, 4869,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 60, 60, 32, 34, 13, 13,  1,  1,  4, 17, 23,  1, 22,  5,  0,  3,  6, {71,78,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Nko/Nko/Guinea
    {    205,     4,   112,    0,    0,    0,    0,   67,   21,   22,   23,   40,   35,   41,   44,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,    0,    0,   15,    0, 4875,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  0,  0,  5,  0, 11,  0, {73,82,82},      0,     0,     6,     5,     5,     1,     3,     3 }, // Northern Luri/Arabic/Iran
    {    205,     4,   113,    0,    0,    0,    0,   67,   21,   22,   23,   40,   35,   41,   44,   14,   15,   16,   17,  163,  103,   61,   76,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,   43,    0,   15,    0, 4875,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 17, 10, 15,  7, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  5,  0,  5,  0, 11,  0, {73,81,68},      0,     0,     6,     5,     6,     1,     3,     3 }, // Northern Luri/Arabic/Iraq
    {    206,    66,   175,    0,    0,  351,  351,    6,    1,    9,    2,    3,   48,    5,   78,   15,   15,   17,   17,  163,  103,   10,    0,16550,16550,16624,16624,16656,16656,  520,  542,    0,    5,   22,  160, 3793,    4,    0, 4886, 4901,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  4,  1,  1,  1,  1, 17, 10, 13,  5, 74, 74, 32, 32, 13, 13, 11, 13,  4, 17, 23,  2, 14,  5,  0, 15,  5, {78,79,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Northern Sami/Latin/Norway
    {    206,    66,    83,    0,    0,  351,  351,    6,    1,    9,    2,    3,   48,    5,   78,   15,   15,   17,   17,  113,   49,   10,    0,16669,16669,16738,16738,16758,16758,  531,  185,    0,    5,   22,   22,  405,    4,    0, 4886, 4906,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  4,  1,  1,  1,  1, 16, 10, 13,  5, 69, 69, 20, 20, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  0, 15,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Northern Sami/Latin/Finland
    {    206,    66,   225,    0,    0,  351,  351,    6,    1,    9,    2,    3,   48,    5,   78,   15,   15,   17,   17,  163,  103,   10,    0,16550,16550,16624,16624,16656,16656,  520,  542,    0,    5,   22,  160, 3807,    4,    0, 4886, 4912,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  4,  1,  1,  1,  1, 17, 10, 13,  5, 74, 74, 32, 32, 13, 13, 11, 13,  4, 17, 23,  2, 14,  5,  0, 15,  6, {83,69,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Northern Sami/Latin/Sweden
    {    207,    66,   216,    0,    0,    0,    0,    6,    0,    9,    2,    3,    4,    5,   10,   16,   17,   14,   15,  163,  103,   10,    0,16771,16771,16834,16834,16860,16860,    0,    0,    0,    5,   22,    9,    0,    2,    0, 4918, 4934,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 63, 63, 26, 26, 13, 13,  2,  2,  4, 17, 23,  1,  0,  4,  0, 16, 12, {90,65,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // Northern Sotho/Latin/South Africa
    {    208,    66,   261,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,16873,16873,16922,16922,16949,16949,    0,    0,    0,    5,   22,  178, 3821,    2,    9, 4946, 2434,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 49, 49, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3, 17,  4,  6, 10,  8, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // North Ndebele/Latin/Zimbabwe
    {    209,    66,   175,    0,    0,  289,  289,    6,    1,    9,    2,    3,   48,    5,   10,   11,   12,   16,   17,  698,   49,   10,    0, 4788, 4788,16962,16962, 4874, 4874,  168,  168,    0,    5,   22,  160, 3838,   15,   58, 4956, 4968,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 50, 50, 34, 34, 13, 13,  4,  4,  4, 17, 23,  2, 13,  5,  7, 12,  5, {78,79,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Norwegian Bokmal/Latin/Norway
    {    209,    66,   224,    0,    0,  289,  289,    6,    1,    9,    2,    3,   48,    5,   10,   11,   12,   16,   17,  698,   49,   10,    0, 4788, 4788,16962,16962, 4874, 4874,  168,  168,    0,    5,   22,  160, 3838,   15,   58, 4956, 4973,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 50, 50, 34, 34, 13, 13,  4,  4,  4, 17, 23,  2, 13,  5,  7, 12, 21, {78,79,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Norwegian Bokmal/Latin/Svalbard and Jan Mayen
    {    210,    66,   175,    0,    0,  289,  289,    6,    1,    9,    2,    3,   48,    5,   10,   11,   12,   16,   17,  698,   49,  502,    0,16996,16996,17046,17073, 4874, 4874,  533,  555,    0,    5,   22,  160, 3838,    4,    0, 4994, 5007,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 19,  5, 50, 50, 27, 27, 13, 13,  4,  4,  4, 17, 23,  2, 13,  5,  0, 13,  5, {78,79,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Norwegian Nynorsk/Latin/Norway
    {    211,    66,   219,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  187,  521,   38,17100,17100,17178,17178,17215,17215,  537,  559,    0,    5,   22,   94,    0,    2,    9, 5012,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  9, 15,  7, 78, 78, 37, 37, 13, 13,  2,  2,  4, 17, 23,  1,  0,  4,  6,  9,  0, {83,83,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // Nuer/Latin/South Sudan
    {    212,    66,   142,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   23,   38,17228,17228,17294,17294,   83,   83,    0,    0,    0,    5,   22,    0, 1508,   15,    0, 5021,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 66, 66, 27, 27, 13, 13,  2,  2,  4, 17, 23,  0, 15,  5,  0,  6,  0, {77,87,75},      2,     1,     1,     6,     7,     1,     3,     3 }, // Nyanja/Latin/Malawi
    {    213,    66,   243,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0, 4053, 4053, 4126, 4126, 4153, 4153,    0,    0,    0,    5,   22,  147,  805,    2,    0, 5027,  983,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 73, 73, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3, 19,  4,  0, 10,  6, {85,71,88},      0,     0,     1,     7,     7,     1,     3,     3 }, // Nyankole/Latin/Uganda
    {    214,    66,    84,    0,    0,  414,  414,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   11,   12, 1555,   79,   10,  536,17321,17321,17321,17321,17377,17377,    0,    0,  376,  232,  249,   22,  405,    0,   45, 5037,  807,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 21,  7, 13,  6, 56, 56, 56, 56, 20, 20,  2,  2,  6, 17, 23,  1,  4,  4,  6,  7,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Occitan/Latin/France
    {    214,    66,   220,    0,    0,  414,  414,    6,    0,    1,    2,    3,    4,    5,   10,   11,   12,   14,   15,  958,   79,   99,    1,17397,17397,17454,17454,17481,17481,    0,    0,  376,  232,  249,   22,  405,    0,    0, 5037, 5044,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  7, 14,  4, 57, 57, 27, 27, 13, 13,  2,  2,  6, 17, 23,  1,  4,  4,  0,  7,  7, {69,85,82},      2,     1,     1,     6,     7,     2,     3,     3 }, // Occitan/Latin/Spain
    {    215,    91,   110,    0,    0,  923,  931,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   61,   76,17494,17494,17547,17547,17579,17579,    0,    0,  885,    5,   22,  120, 3851,    2,    9, 5051, 5056,  6,  6,  8,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 53, 53, 32, 32, 17, 17,  2,  2,  5, 17, 23,  1, 12,  4,  6,  5,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Odia/Odia/India
    {    220,    66,    77,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,   78,   23,   38,17596,17596,17650,17650,   83,   83,  539,  561,    0,    5,   22,    1, 3863,    2,    0, 5060, 5066,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 15,  7, 54, 54, 27, 27, 13, 13,  2,  2,  4, 17, 23,  2, 17,  4,  0,  6, 10, {69,84,66},      2,     1,     7,     6,     7,     1,     3,     3 }, // Oromo/Latin/Ethiopia
    {    220,    66,   124,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,   78,   10,    0,17596,17596,17650,17650,17677,17677,  539,  561,    0,    5,   22,  175,    0,    2,    0, 5060, 5076,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 54, 54, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3,  0,  4,  0,  6,  8, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Oromo/Latin/Kenya
    {    221,   101,   248,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   61,   76,17690,17690,17690,17690,   83,17870,    0,    0,    0,    5,   22,   10,    0,   15,    0, 5084,  964,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7,180,180,180,180, 13, 20,  2,  2,  4, 17, 23,  1,  0,  5,  0, 12, 13, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Osage/Osage/United States
    {    222,    27,    90,    0,    0,  938,  938,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14, 1576,   49,   10,    0,17890,17950,18010,18037,18064,18064,  541,  563,    0,    5,   22,    0, 3880,   15,    0, 5096, 5100,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  8, 13,  5, 60, 60, 27, 27, 13, 13, 15, 15,  4, 17, 23,  1,  3,  5,  0,  4, 11, {71,69,76},      2,     1,     1,     6,     7,     1,     3,     3 }, // Ossetic/Cyrillic/Georgia
    {    222,    27,   193,    0,    0,  938,  938,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14, 1576,   49,   10,    0,17890,17950,18010,18037,18064,18064,  541,  563,    0,    5,   22,  133, 3883,   15,    0, 5096, 5111,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  8, 13,  5, 60, 60, 27, 27, 13, 13, 15, 15,  4, 17, 23,  1,  3,  5,  0,  4,  6, {82,85,66},      2,     1,     1,     6,     7,     1,     3,     3 }, // Ossetic/Cyrillic/Russia
    {    226,    66,    62,    0,    0,  143,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  394,   10,    0,18077,18077,18077,18077,18139,18139,    0,    0,    0,    5,   22,  169,    0,   15,    0, 5117, 5127,  6,  6,  7,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 62, 62, 62, 62, 20, 20,  2,  2,  4, 17, 23,  3,  0,  5,  0, 10,  6, {88,67,71},      2,     1,     1,     6,     7,     1,     3,     3 }, // Papiamento/Latin/Curacao
    {    226,    66,    13,    0,    0,  143,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  394,   10,    0,18077,18077,18077,18077,18139,18139,    0,    0,    0,    5,   22,    0, 3886,   15,    0, 5117, 1227,  6,  6,  7,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 62, 62, 62, 62, 20, 20,  2,  2,  4, 17, 23,  0, 15,  5,  0, 10,  5, {65,87,71},      2,     1,     1,     6,     7,     1,     3,     3 }, // Papiamento/Latin/Aruba
    {    227,     4,     1,  661,  661,  947,  956,   67,   21,   22,   23,   40,   35,   41,   44,   14,   15,   16,   17, 1599,  505,   99,    1,18159,18159,18159,18159,   83,   83,  556,  578,  890,    5,   22,  269, 3901,    2,    9, 5133, 5137,  6,  6,  9,  8,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 20,  8, 14,  4, 38, 38, 38, 38, 13, 13,  4,  4,  5, 17, 23,  1,  6,  4,  6,  4,  9, {65,70,78},      0,     0,     6,     4,     5,     1,     3,     3 }, // Pashto/Arabic/Afghanistan
    {    227,     4,   178,  661,  661,  947,  956,   67,   21,   22,   23,   40,   35,   41,   44,   14,   15,   16,   17, 1599,  505,   61,   76,18159,18159,18159,18159,   83,   83,  556,  578,  890,    5,   22,  195, 3907,    2,    9, 5133, 5146,  6,  6,  9,  8,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 20,  8, 15,  7, 38, 38, 38, 38, 13, 13,  4,  4,  5, 17, 23,  2, 15,  4,  6,  4,  7, {80,75,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // Pashto/Arabic/Pakistan
    {    228,     4,   112,  964,  964,  971,  979,   67,   21,   22,   23,   40,   82,   37,   44,   11,   12,   19,   20,  113,  505,   99,    1,18197,18197,18197,18197,18245,18245,  560,  582,  798,    5,   22,  270, 3922,  103,  109, 5153, 4680,  7,  7,  8,  7,  1,  1,  1,  1,  1,  2,  2,  4,  1,  1,  1,  1, 16,  8, 14,  4, 48, 48, 48, 48, 13, 13,  9,  8,  4, 17, 23,  4, 10,  6,  8,  5,  5, {73,82,82},      0,     0,     6,     5,     5,     1,     3,     3 }, // Persian/Arabic/Iran
    {    228,     4,     1,  964,  964,  971,  979,   67,   21,   22,   23,   40,   82,   37,   44,   11,   12,   19,   20,  113,  505,   99,    1,18197,18197,18197,18197,18245,18245,  560,  582,  798,    5,   22,  269, 3932,   15,  109, 5158, 5137,  7,  7,  8,  7,  1,  1,  1,  1,  1,  2,  2,  4,  1,  1,  1,  1, 16,  8, 14,  4, 48, 48, 48, 48, 13, 13,  9,  8,  4, 17, 23,  1, 16,  5,  8,  3,  9, {65,70,78},      0,     0,     6,     4,     5,     1,     3,     3 }, // Persian/Arabic/Afghanistan
    {    230,    66,   187,    0,    0,  143,  143,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   11,   12,    0,   50,   10,    0,18258,18258,18316,18316,18349,18362,    0,    0,  311,    5,   22,  274, 3948,    4,   20, 5161, 5167,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  9, 13,  5, 58, 58, 33, 33, 13, 13,  2,  2,  5, 17, 23,  2, 12,  5,  7,  6,  6, {80,76,78},      2,     1,     1,     6,     7,     2,     3,     3 }, // Polish/Latin/Poland
    {    231,    66,    32,    0,    0,  414,  414,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  186,   10,    0,18375,18375,18453,18453,18487,18487,    0,    0,    0,    5,   22,    9, 3960,   15,    0, 5173, 5182,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27, 10, 13,  5, 78, 78, 34, 34, 13, 13,  2,  2,  5, 17, 23,  2, 15,  5,  0,  9,  6, {66,82,76},      2,     1,     7,     6,     7,     1,     3,     3 }, // Portuguese/Latin/Brazil
    {    231,    66,     7,    0,    0,  414,  414,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  958,   78,   10,    0,18375,18375,18500,18500,18487,18487,  569,  590,    0,    5,   22,  254, 3975,    4,   20, 5173, 5188,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 78, 78, 48, 48, 13, 13,  8,  8,  5, 17, 23,  2, 15,  5,  7,  9,  6, {65,79,65},      2,     1,     1,     6,     7,     1,     3,     3 }, // Portuguese/Latin/Angola
    {    231,    66,    43,    0,    0,  414,  414,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  958,   78,   10,    0,18375,18375,18500,18500,18487,18487,  569,  590,    0,    5,   22,  242, 3990,    4,   20, 5173, 5194,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 78, 78, 48, 48, 13, 13,  8,  8,  5, 17, 23,  1, 20,  5,  7,  9, 10, {67,86,69},      2,     1,     1,     6,     7,     2,     3,     3 }, // Portuguese/Latin/Cape Verde
    {    231,    66,    73,    0,    0,  414,  414,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  958,   78,   10,    0,18375,18375,18500,18500,18487,18487,  569,  590,    0,    5,   22,   11, 4010,    4,   20, 5173, 5204,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 78, 78, 48, 48, 13, 13,  8,  8,  5, 17, 23,  4, 17,  5,  7,  9, 16, {88,65,70},      0,     0,     1,     6,     7,     2,     3,     3 }, // Portuguese/Latin/Equatorial Guinea
    {    231,    66,   101,    0,    0,  414,  414,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  958,   78,   10,    0,18375,18375,18500,18500,18487,18487,  569,  590,    0,    5,   22,  127, 4027,    4,   20, 5173, 5220,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 78, 78, 48, 48, 13, 13,  8,  8,  5, 17, 23,  5, 18,  5,  7,  9, 12, {88,79,70},      0,     0,     1,     6,     7,     2,     3,     3 }, // Portuguese/Latin/Guinea-Bissau
    {    231,    66,   138,    0,    0,  414,  414,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  958,   78,   10,    0,18375,18375,18500,18500,18487,18487,  569,  590,    0,    5,   22,   22,  405,    4,   20, 5173, 5232,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 78, 78, 48, 48, 13, 13,  8,  8,  5, 17, 23,  1,  4,  5,  7,  9, 10, {69,85,82},      2,     1,     1,     6,     7,     2,     3,     3 }, // Portuguese/Latin/Luxembourg
    {    231,    66,   139,    0,    0,  414,  414,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  958,   78,   23,   38,18375,18375,18500,18500,18487,18487,  569,  590,    0,    5,   22,  151, 4045,    4,   20, 5173, 5242,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 15,  7, 78, 78, 48, 48, 13, 13,  8,  8,  5, 17, 23,  4, 15,  5,  7,  9, 19, {77,79,80},      2,     1,     7,     6,     7,     2,     3,     3 }, // Portuguese/Latin/Macao
    {    231,    66,   160,    0,    0,  414,  414,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  958,   78,   10,    0,18375,18375,18500,18500,18487,18487,  569,  590,    0,    5,   22,  260, 4060,    4,   20, 5173, 5261,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 78, 78, 48, 48, 13, 13,  8,  8,  5, 17, 23,  3, 19,  5,  7,  9, 10, {77,90,78},      2,     1,     7,     6,     7,     2,     3,     3 }, // Portuguese/Latin/Mozambique
    {    231,    66,   188,    0,    0,  414,  414,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  958,   78,   10,    0,18375,18375,18500,18500,18487,18487,  569,  590,    0,    5,   22,   22,  405,    4,   20, 5271, 5288,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 78, 78, 48, 48, 13, 13,  8,  8,  5, 17, 23,  1,  4,  5,  7, 17,  8, {69,85,82},      2,     1,     7,     6,     7,     2,     3,     3 }, // Portuguese/Latin/Portugal
    {    231,    66,   204,    0,    0,  414,  414,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  958,   78,   10,    0,18375,18375,18500,18500,18487,18487,  569,  590,    0,    5,   22,  276, 4079,    4,   20, 5173, 5296,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 78, 78, 48, 48, 13, 13,  8,  8,  5, 17, 23,  2, 28,  5,  7,  9, 19, {83,84,78},      2,     1,     1,     6,     7,     2,     3,     3 }, // Portuguese/Latin/Sao Tome and Principe
    {    231,    66,   226,    0,    0,  414,  414,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  958,   78,   10,    0,18375,18375,18500,18500,18487,18487,  569,  590,    0,    5,   22,    0, 4107,    4,   20, 5173, 5315,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 78, 78, 48, 48, 13, 13,  8,  8,  5, 17, 23,  0, 12,  5,  7,  9,  5, {67,72,70},      2,     0,     1,     6,     7,     2,     3,     3 }, // Portuguese/Latin/Switzerland
    {    231,    66,   232,    0,    0,  414,  414,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15,  958,   78,   10,    0,18375,18375,18500,18500,18487,18487,  569,  590,    0,    5,   22,  178, 4119,    4,   20, 5173, 5320,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 78, 78, 48, 48, 13, 13,  8,  8,  5, 17, 23,  3, 24,  5,  7,  9, 11, {85,83,68},      2,     1,     1,     6,     7,     2,     3,     3 }, // Portuguese/Latin/Timor-Leste
    {    232,    66,   187,    0,    0,  986,  986,    6,    1,    9,    2,    3,    4,    5,   10,   13,   14,   13,   14, 1619,   49,   10,    0,18548,18548,18616,18616,18643,18643,  577,  598,    0,    5,   22,  274,    0,    4,    0, 5331, 5340,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 68, 68, 27, 27, 13, 13, 10, 14,  4, 17, 23,  2,  0,  5,  0,  9,  4, {80,76,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Prussian/Latin/Poland
    {    233,    41,   110,    0,    0,  994,  994,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  129,   61,   76,18656,18656,18712,18712,18747,18747,  587,  612,  895,    5,   22,  120, 4143,   15,    0, 5344, 5350,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 56, 56, 35, 35, 22, 22,  6,  6,  4, 17, 23,  1, 11,  5,  0,  6,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Punjabi/Gurmukhi/India
    {    233,     4,   178,    0,    0,    0,    0,   67,   21,   22,   23,   40,   35,   41,   44,   14,   15,   16,   17,  786,  186,   61,   76,18769,18769,18769,18769,   83,   83,    0,    0,    0,    5,   22,   78, 4154,   15,    0, 5354, 5146,  6,  6,  6,  6,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 18, 10, 15,  7, 36, 36, 36, 36, 13, 13,  2,  2,  4, 17, 23,  1,  6,  5,  0,  6,  7, {80,75,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // Punjabi/Arabic/Pakistan
    {    234,    66,   184,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  265,  129,   23,   38,18805,18805,18857,18857,18884,18884,  168,  168,    0,    5,   22,  278, 4160,   15,    0, 5360, 5368,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 52, 52, 27, 27, 13, 13,  4,  4,  4, 17, 23,  2, 11,  5,  0,  8,  4, {80,69,78},      2,     1,     7,     6,     7,     1,     3,     3 }, // Quechua/Latin/Peru
    {    234,    66,    28,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  265,  129,   23,   38,18805,18805,18857,18857,18884,18884,  168,  168,    0,    5,   22,  280, 4171,   15,    0, 5360, 5372,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 52, 52, 27, 27, 13, 13,  4,  4,  4, 17, 23,  2,  9,  5,  0,  8,  7, {66,79,66},      2,     1,     1,     6,     7,     1,     3,     3 }, // Quechua/Latin/Bolivia
    {    234,    66,    70,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  265,  129,   23,   38,18805,18805,18857,18857,18884,18884,  168,  168,    0,    5,   22,   10, 4180,   15,    0, 5360, 5379,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 52, 52, 27, 27, 13, 13,  4,  4,  4, 17, 23,  1, 15,  5,  0,  8,  7, {85,83,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Quechua/Latin/Ecuador
    {    235,    66,   192,    0,    0, 1003, 1003,    6,    1,    0,    2,    3,    4,    5,   10,   13,   15,   11,   12,    0,   49,   10,    0,18897,18897,18944,18944, 6778, 6778,  168,  168,  899,    5,   22,  282, 4195,    4,   20, 5386, 5392,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 47, 47, 33, 33, 13, 13,  4,  4,  4, 17, 23,  3, 12,  5,  7,  6,  7, {82,79,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Romanian/Latin/Romania
    {    235,    66,   154,    0,    0, 1003, 1003,    6,    1,    0,    2,    3,    4,    5,   10,   13,   15,   11,   12,    0,   49,   10,    0,18897,18897,18977,18977,19004,19004,  168,  168,  899,    5,   22,   18, 4207,    4,   20, 5386, 5399,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 47, 47, 27, 27, 15, 15,  4,  4,  4, 17, 23,  1, 15,  5,  7,  6, 17, {77,68,76},      2,     1,     1,     6,     7,     1,     3,     3 }, // Romanian/Latin/Moldova
    {    236,    66,   226,    0,    0,  414,  414,    6,    0,   17,    2,    3,   48,    5,   10,   11,   12,   19,   20, 1646,  394,   10,    0,19019,19019,19074,19074,19096,19096,    0,    0,    0,    5,   22,    0, 4222,    4,    0, 5416, 5425,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23,  8, 13,  5, 55, 55, 22, 22, 13, 13,  2,  2,  5, 17, 23,  0, 13,  5,  0,  9,  6, {67,72,70},      2,     0,     1,     6,     7,     1,     3,     3 }, // Romansh/Latin/Switzerland
    {    237,    66,   230,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,19109,19109,19173,19173,14208,14208,  593,  618,    0,    5,   22,  121, 4235,    2,    0, 5431, 2268,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 64, 64, 28, 28, 13, 13,  8,  7,  4, 17, 23,  3, 18,  4,  0,  9,  8, {84,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Rombo/Latin/Tanzania
    {    238,    66,    38,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   15,   15,   17,   17,  113,  129,   10,    0,19201,19201,19289,19289,   83,   83,  601,  625,    0,    5,   22,  181, 4253,    0,    0, 5440, 5448,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 88, 88, 33, 33, 13, 13,  5,  5,  4, 17, 23,  3, 20,  4,  0,  8,  8, {66,73,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Rundi/Latin/Burundi
    {    239,    27,   193,    0,    0,  150,  150,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14,  350,   49,   10,    0,19322,19322,19383,19383,19403,19403,    0,    0,  196,  841,   22,  133, 4273,    4,    0, 5456, 5463,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22, 10, 13,  5, 61, 61, 20, 20, 13, 13,  2,  2,  5, 17, 23,  1, 16,  5,  0,  7,  6, {82,85,66},      2,     1,     1,     6,     7,     1,     3,     3 }, // Russian/Cyrillic/Russia
    {    239,    27,    22,    0,    0,  150,  150,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14,  350,   49,   10,    0,19322,19322,19383,19383,19403,19403,    0,    0,  196,  841,   22,    1, 4289,    4,    0, 5456,  618,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22, 10, 13,  5, 61, 61, 20, 20, 13, 13,  2,  2,  5, 17, 23,  2, 17,  5,  0,  7,  8, {66,89,78},      2,     0,     1,     6,     7,     1,     3,     3 }, // Russian/Cyrillic/Belarus
    {    239,    27,   123,    0,    0,  150,  150,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14,  350,   49,   10,    0,19322,19322,19383,19383,19403,19403,    0,    0,  196,  841,   22,  243, 4306,    4,    0, 5456, 5469,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22, 10, 13,  5, 61, 61, 20, 20, 13, 13,  2,  2,  5, 17, 23,  1, 15,  5,  0,  7,  9, {75,90,84},      2,     1,     1,     6,     7,     1,     3,     3 }, // Russian/Cyrillic/Kazakhstan
    {    239,    27,   128,    0,    0,  150,  150,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14,  350,   49,   10,    0,19322,19322,19383,19383,19403,19403,    0,    0,  196,  841,   22,  250, 4321,    4,    0, 5456, 5478,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22, 10, 13,  5, 61, 61, 20, 20, 13, 13,  2,  2,  5, 17, 23,  3, 14,  5,  0,  7,  8, {75,71,83},      2,     1,     1,     6,     7,     1,     3,     3 }, // Russian/Cyrillic/Kyrgyzstan
    {    239,    27,   154,    0,    0,  150,  150,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14,  350,   49,   10,    0,19322,19322,19383,19383,19403,19403,    0,    0,  196,  841,   22,   18, 4335,    4,    0, 5456, 5486,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22, 10, 13,  5, 61, 61, 20, 20, 13, 13,  2,  2,  5, 17, 23,  1, 14,  5,  0,  7,  7, {77,68,76},      2,     1,     1,     6,     7,     1,     3,     3 }, // Russian/Cyrillic/Moldova
    {    239,    27,   244,    0,    0,  150,  150,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14,  350,   49,   10,    0,19322,19322,19383,19383,19403,19403,    0,    0,  196,  841,   22,  285, 4349,    4,    0, 5456, 5493,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22, 10, 13,  5, 61, 61, 20, 20, 13, 13,  2,  2,  5, 17, 23,  1, 17,  5,  0,  7,  7, {85,65,72},      2,     1,     1,     6,     7,     2,     3,     3 }, // Russian/Cyrillic/Ukraine
    {    240,    66,   230,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,13800,13800,13861,13861, 1284, 1284,  430,  446,    0,    5,   22,  121, 3467,    0,    0, 5500, 2268,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 61, 61, 27, 27, 13, 13,  5,  9,  4, 17, 23,  3, 20,  4,  0,  6,  8, {84,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Rwa/Latin/Tanzania
    {    241,    66,    74,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,   59,   78,   23,   38,19416,19416,19471,19471,19498,19498,    0,    0,    0,    5,   22,    6,    0,    2,    0, 5506,   34,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  8, 15,  7, 55, 55, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3,  0,  4,  0,  4,  7, {69,82,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Saho/Latin/Eritrea
    {    242,    27,   193,    0,    0, 1011, 1011,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   14, 1669,  344,   10,    0,19511,19511,19581,19581,19601,19601,  606,  630,  903,  908,   22,  133, 4366,    4,    0, 5510, 5519,  6,  6, 11, 11,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 30,  6, 13,  5, 70, 70, 20, 20, 13, 13,  2,  2,  5, 17, 23,  1, 20,  5,  0,  9,  9, {82,85,66},      2,     1,     1,     6,     7,     1,     3,     3 }, // Sakha/Cyrillic/Russia
    {    243,    66,   124,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,19614,19614,19718,19718,19745,19745,  608,  632,    0,    5,   22,  175, 4386,    2,    9, 5528, 1307,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5,104,104, 27, 27, 13, 13,  7,  5,  4, 17, 23,  3, 18,  4,  6,  8,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Samburu/Latin/Kenya
    {    245,    66,    46,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   14,   15,  113,  129,   10,    0,19758,19758,19823,19823,19850,19850,  615,  637,    0,    5,   22,   11, 4404,    2,   65, 5536, 5541,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 65, 65, 27, 27, 13, 13,  2,  2,  4, 17, 23,  4, 18,  4,  5,  5, 22, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Sango/Latin/Central African Republic
    {    246,    66,   230,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,19863,19863,19922,19922,19949,19949,  617,  639,    0,    5,   22,  121, 4422,    0,    0, 5563, 5572,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 59, 59, 27, 27, 13, 13,  9,  9,  4, 17, 23,  3, 18,  4,  0,  9,  9, {84,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Sangu/Latin/Tanzania
    {    247,    29,   110,    0,    0, 1022, 1032,    6,    0,    1,    2,   49,    4,    5,   10,   14,   15,   16,   17,    0,  129,   61,   76,19962,19962, 8574, 8574, 8605, 8605,  484,  505,    0,    5,   22,  120, 4440,   15,    0, 5581, 5593,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 66, 66, 31, 31, 18, 18,  9,  7,  4, 17, 23,  1, 15,  5,  0, 12,  5, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Sanskrit/Devanagari/India
    {    248,    93,   110,    0,    0,    0,    0,    6,    0,    1,    2,   84,    4,    5,   10,   14,   15,   16,   17,    0,  129,   61,   76,20028,20028,20069,20069,20094,20094,  626,  648,    0,    5,   22,  120, 4455,   15,    0, 5598, 5605,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 41, 41, 25, 25, 13, 13,  5,  5,  4, 17, 23,  1, 16,  5,  0,  7,  6, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Santali/Ol Chiki/India
    {    248,    29,   110,    0,    0,    0,    0,    6,    0,    1,    2,   49,    4,    5,   10,   14,   15,   16,   17,    0,  129,   61,   76,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  120,    0,   15,    0, 5611,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  8,  0, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Santali/Devanagari/India
    {    249,    66,   117,    0,    0,  414,  414,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   14,   15, 1699,  186,   10,    0,20107,20107,20161,20161,20188,20188,    0,    0,    0,    5,   22,   22, 4471,    4,    0, 5619,  813,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 31, 10, 13,  5, 54, 54, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1,  4,  5,  0,  5,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Sardinian/Latin/Italy
    {    251,    66,   160,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   10,    0,20201,20201,20255,20255,20282,20282,    0,    0,    0,    5,   22,  260, 4475,    0,    0, 5624, 5261,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 13,  5, 54, 54, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3, 21,  4,  0,  4, 10, {77,90,78},      2,     1,     7,     6,     7,     1,     3,     3 }, // Sena/Latin/Mozambique
    {    252,    27,   207,    0,    0,  150,  150,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   16,   16,  404,  454,   10,    0,20295,20295,20346,20346, 2891, 2891,    0,    0,  925,    5,   22,    0, 4496,    4,   20, 5628, 5634,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  7, 13,  5, 51, 51, 27, 27, 13, 13,  2,  2,  7, 17, 23,  0, 12,  5,  7,  6,  6, {82,83,68},      0,     0,     1,     6,     7,     1,     3,     3 }, // Serbian/Cyrillic/Serbia
    {    252,    27,    29,    0,    0,  150,  150,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   16,   16,  404,  454,   10,    0, 2809, 2809, 2864, 2864, 2891, 2891,  104,  653,  925,    5,   22,  137, 4508,    4,   20, 5628,  713,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  7, 13,  5, 55, 55, 27, 27, 13, 13, 11,  8,  7, 17, 23,  2, 40,  5,  7,  6, 19, {66,65,77},      2,     1,     1,     6,     7,     1,     3,     3 }, // Serbian/Cyrillic/Bosnia and Herzegovina
    {    252,    27,   126,    0,    0,  150,  150,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   16,   16,  404,  454,   10,    0,20295,20295,20346,20346, 2891, 2891,    0,    0,  925,    5,   22,   22, 4548,    4,   20, 5628, 5640,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  7, 13,  5, 51, 51, 27, 27, 13, 13,  2,  2,  7, 17, 23,  1,  4,  5,  7,  6,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Serbian/Cyrillic/Kosovo
    {    252,    27,   157,    0,    0,  150,  150,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   16,   16,  404,  454,   10,    0,20373,20373,20346,20346, 2891, 2891,  104,  653,  925,    5,   22,   22, 4548,    4,   20, 5628, 5646,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  7, 13,  5, 54, 54, 27, 27, 13, 13, 11,  8,  7, 17, 23,  1,  4,  5,  7,  6,  9, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Serbian/Cyrillic/Montenegro
    {    252,    66,    29,    0,    0,  143,  143,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   16,   16,  404,  454,   10,    0, 2699, 2699, 2756, 2756, 2783, 2783,  631,  661,  218,    5,   22,  135,  597,    4,   20, 5655,  686,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  7, 13,  5, 57, 57, 27, 27, 13, 13, 11,  8,  7, 17, 23,  2, 40,  5,  7,  6, 19, {66,65,77},      2,     1,     1,     6,     7,     1,     3,     3 }, // Serbian/Latin/Bosnia and Herzegovina
    {    252,    66,   126,    0,    0,  143,  143,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   16,   16,  404,  454,   10,    0,20427,20427,20480,20480, 2783, 2783,    0,    0,  218,    5,   22,   22, 4552,    4,   20, 5655, 5661,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  7, 13,  5, 53, 53, 27, 27, 13, 13,  2,  2,  7, 17, 23,  1,  4,  5,  7,  6,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Serbian/Latin/Kosovo
    {    252,    66,   157,    0,    0,  143,  143,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   16,   16,  404,  454,   10,    0,20507,20507,20480,20480, 2783, 2783,  631,  661,  218,    5,   22,   22, 4552,    4,   20, 5655, 5667,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  7, 13,  5, 56, 56, 27, 27, 13, 13, 11,  8,  7, 17, 23,  1,  4,  5,  7,  6,  9, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Serbian/Latin/Montenegro
    {    252,    66,   207,    0,    0,  143,  143,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   16,   16,  404,  454,   10,    0,20427,20427,20480,20480, 2783, 2783,    0,    0,  218,    5,   22,    0, 4556,    4,   20, 5655, 5676,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  7, 13,  5, 53, 53, 27, 27, 13, 13,  2,  2,  7, 17, 23,  0, 12,  5,  7,  6,  6, {82,83,68},      0,     0,     1,     6,     7,     1,     3,     3 }, // Serbian/Latin/Serbia
    {    253,    66,   230,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,20563,20563,20625,20625,20652,20652,  642,  669,    0,    5,   22,  121, 4568,    0,    0, 5682, 2268,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 62, 62, 27, 27, 13, 13,  5,  8,  4, 17, 23,  3, 20,  4,  0,  9,  8, {84,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Shambala/Latin/Tanzania
    {    254,    66,   261,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   15,   15,   17,   17,  163,  103,   10,    0,20665,20665,20719,20719,20746,20746,    0,    0,    0,    5,   22,  178, 4588,    2,    9, 5691, 2434,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 54, 54, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3, 15,  4,  6,  8,  8, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Shona/Latin/Zimbabwe
    {    255,   141,    50,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,20759,20759,20786,20786,20806,20806,  647,  677,    0,    5,   22,  150,    0,   15,    0, 5699, 5702,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 20, 20, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  3,  2, {67,78,89},      2,     1,     1,     6,     7,     1,     3,     3 }, // Sichuan Yi/Yi/China
    {    256,    66,   117,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,20819,20819,20819,20819,   83,   83,    0,    0,    0,    5,   22,   22,    0,   15,    0, 5704, 3734,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 62, 62, 62, 62, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  9,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Sicilian/Latin/Italy
    {    257,    66,    77,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,   59,   78,   23,   38,20881,20881,20931,20931,20958,20958,    0,    0,    0,    5,   22,    1,    0,    2,    0, 5713, 5724,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  8, 15,  7, 50, 50, 27, 27, 13, 13,  2,  2,  4, 17, 23,  2,  0,  4,  0, 11, 11, {69,84,66},      2,     1,     7,     6,     7,     1,     3,     3 }, // Sidamo/Latin/Ethiopia
    {    258,    66,   187,    0,    0,  143,  143,    6,    1,    9,    2,    3,    4,    5,   10,   13,   15,   12,   11,    0,   49,   10,    0,20971,20971,21031,21031,13193,13193,  649,  679,  311,    5,   22,  274,    0,   15,    0, 5735, 5167,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 60, 60, 27, 27, 13, 13, 11, 11,  5, 17, 23,  2,  0,  5,  0,  7,  6, {80,76,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Silesian/Latin/Poland
    {    259,     4,   178,    0,    0, 1041, 1049,   67,   21,   22,   23,   25,   26,   28,   59,   14,   15,   16,   17,  549,  103,   61,   76,21058,21058,21058,21058,21092,21092,  660,  690,  932,  938,   22,  195, 4603,    4,    0, 5742, 5746,  6,  6,  8,  7,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 18, 10, 15,  7, 34, 34, 34, 34, 30, 30, 11, 11,  6, 25, 23,  2, 12,  5,  0,  4,  7, {80,75,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // Sindhi/Arabic/Pakistan
    {    259,    29,   110,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   61,   76,21122,21149,21190,21212,21240,21240,  671,  701,    0,    5,   22,  120, 4615,   15,    0, 5753,  664,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 27, 41, 22, 28, 20, 20,  8,  6,  4, 17, 23,  1, 17,  5,  0,  6,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Sindhi/Devanagari/India
    {    260,   119,   221,    0,    0, 1056, 1065,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,  212,  212,21260,21260,21321,21321,21359,21359,  679,  707,  963,  968,   22,  286, 4632,    2,    9, 5759, 5764,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 61, 61, 38, 38, 18, 18,  5,  4,  5, 42, 23,  3, 17,  4,  6,  5, 11, {76,75,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Sinhala/Sinhala/Sri Lanka
    {    261,    66,    83,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,   22,    0,   15,    0, 5775, 5785,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0, 10, 12, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Skolt Sami/Latin/Finland
    {    262,    66,   212,    0,    0,  781,  282,    6,    1,    9,    2,    3,    4,    5,    6,   13,   14,   18,   16,  698,  423,   11,    1,21377,21377,21428,21428,21448,21448,    0,    0,  311,    5,   22,   22,  405,    4,   20, 5797, 5807,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 12,  4, 51, 51, 20, 20, 13, 13,  2,  2,  5, 17, 23,  1,  4,  5,  7, 10,  9, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Slovak/Latin/Slovakia
    {    263,    66,   213,    0,    0, 1073, 1073,    6,    1,    0,    2,    3,   48,    5,    6,   13,   14,   18,   16,  404,  423,   10,    0,21461,21461,21512,21512,21546,21546,  172,  711,   50,    5,   22,   22, 4649,    4,   20, 5816, 5827,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 51, 51, 34, 34, 13, 13,  4,  4,  4, 17, 23,  1,  4,  5,  7, 11,  9, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Slovenian/Latin/Slovenia
    {    264,    66,   243,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,21559,21559,21623,21623,21657,21657,  684,  715,    0,    5,   22,  147, 2804,    4,    0, 5836, 3330,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 64, 64, 34, 34, 13, 13,  6,  6,  4, 17, 23,  3, 19,  5,  0,  7,  7, {85,71,88},      0,     0,     1,     7,     7,     1,     3,     3 }, // Soga/Latin/Uganda
    {    265,    66,   215,    0,    0, 1081, 1081,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,   78,   23,   38,21670,21670,21716,21716,21747,21747,  690,  721, 1010, 1016,   22,   93, 4653,    2,    9, 5843, 5851,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 15,  7, 46, 46, 31, 31, 14, 14,  2,  2,  6, 17, 23,  1, 20,  4,  6,  8, 10, {83,79,83},      0,     0,     1,     6,     7,     1,     3,     3 }, // Somali/Latin/Somalia
    {    265,    66,    67,    0,    0, 1081, 1081,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,   78,   23,   38,21670,21670,21716,21716,21747,21747,  690,  721, 1010, 1016,   22,    3, 4673,    2,    9, 5843, 5861,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 15,  7, 46, 46, 31, 31, 14, 14,  2,  2,  6, 17, 23,  3, 13,  4,  6,  8,  7, {68,74,70},      0,     0,     6,     6,     7,     1,     3,     3 }, // Somali/Latin/Djibouti
    {    265,    66,    77,    0,    0, 1081, 1081,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,   78,   23,   38,21670,21670,21716,21716,21747,21747,  690,  721, 1010, 1016,   22,    1, 4686,    2,    9, 5843, 5868,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 15,  7, 46, 46, 31, 31, 14, 14,  2,  2,  6, 17, 23,  2, 15,  4,  6,  8,  8, {69,84,66},      2,     1,     7,     6,     7,     1,     3,     3 }, // Somali/Latin/Ethiopia
    {    265,    66,   124,    0,    0, 1081, 1081,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,   78,   10,    0,21670,21670,21716,21716,21747,21747,  690,  721, 1010, 1016,   22,  175, 4701,    2,    9, 5843, 1307,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 46, 46, 31, 31, 14, 14,  2,  2,  6, 17, 23,  3, 15,  4,  6,  8,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Somali/Latin/Kenya
    {    266,     4,   112,    0,    0,    0,    0,   67,   21,   22,   23,   25,   26,   28,   59,   11,   12,   19,   20,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,    0,    0,    4,    0, 5876,    0,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  0,  0,  5,  0, 11,  0, {73,82,82},      0,     0,     6,     5,     5,     1,     3,     3 }, // Southern Kurdish/Arabic/Iran
    {    266,     4,   113,    0,    0,    0,    0,   67,   21,   22,   23,   25,   26,   28,   59,   11,   12,   19,   20,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,    0,    0,    4,    0, 5876,    0,  6,  6,  6,  6,  1,  1,  1,  2,  1,  2,  2,  2,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  0,  0,  5,  0, 11,  0, {73,81,68},      0,     0,     6,     5,     6,     1,     3,     3 }, // Southern Kurdish/Arabic/Iraq
    {    267,    66,   225,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  160,    0,   15,    0, 5887,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  2,  0,  5,  0, 19,  0, {83,69,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Southern Sami/Latin/Sweden
    {    267,    66,   175,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  160,    0,   15,    0, 5887,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  2,  0,  5,  0, 19,  0, {78,79,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Southern Sami/Latin/Norway
    {    268,    66,   216,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   16,   17,   14,   15,  163,  103,   10,    0,21761,21761,21821,21821,   83,   83,    0,    0,    0,    5,   22,    9,    0,    2,    0, 4918,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 60, 60, 26, 26, 13, 13,  2,  2,  4, 17, 23,  1,  0,  4,  0,  7,  0, {90,65,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // Southern Sotho/Latin/South Africa
    {    268,    66,   133,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   16,   17,   14,   15,  163,  103,   23,   38,21761,21761,21821,21821,   83,   83,    0,    0,    0,    5,   22,    9,    0,    2,    0, 4918,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 60, 60, 26, 26, 13, 13,  2,  2,  4, 17, 23,  1,  0,  4,  0,  7,  0, {90,65,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Southern Sotho/Latin/Lesotho
    {    269,    66,   216,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   16,   17,   14,   15,  163,  103,   10,    0,21847,21847,21912,21912,   83,   83,    0,    0,    0,    5,   22,    9,    0,    2,    0, 4946,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 65, 65, 26, 26, 13, 13,  2,  2,  4, 17, 23,  1,  0,  4,  0, 10,  0, {90,65,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // South Ndebele/Latin/South Africa
    {    270,    66,   220,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   99,    1,21938,21938,21990,21990,18884,18884,  132,  128,    0,    5,   22,   22,  405,    4,    0, 5906,  455,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 14,  4, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1,  4,  5,  0, 17,  6, {69,85,82},      2,     1,     1,     6,     7,     2,     3,     3 }, // Spanish/Latin/Spain
    {    270,    66,    11,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,   10, 4716,   15,   58, 5906, 5923,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1, 14,  5,  7,  7,  9, {65,82,83},      2,     1,     1,     6,     7,     1,     3,     3 }, // Spanish/Latin/Argentina
    {    270,    66,    24,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   10,    0,21938,21938,21990,21990, 6778, 6778,  168,  168,    0,    5,   22,   10, 4730,    2,    0, 5906, 5932,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 13,  5, 52, 52, 27, 27, 13, 13,  4,  4,  5, 17, 23,  1, 14,  4,  0,  7,  6, {66,90,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/Belize
    {    270,    66,    28,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,  280, 4744,    2,    0, 5906, 5372,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  2,  9,  4,  0,  7,  7, {66,79,66},      2,     1,     1,     6,     7,     1,     3,     3 }, // Spanish/Latin/Bolivia
    {    270,    66,    32,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   10,    0,21938,21938,21990,21990, 6778, 6778,  168,  168,    0,    5,   22,    9, 4753,    2,    0, 5906, 5182,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 13,  5, 52, 52, 27, 27, 13, 13,  4,  4,  5, 17, 23,  2, 14,  4,  0,  7,  6, {66,82,76},      2,     1,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/Brazil
    {    270,    66,    42,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   99,    1,21938,21938,21990,21990,18884,18884,  132,  128,    0,    5,   22,   22,  405,    4,    0, 5906, 5938,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 14,  4, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1,  4,  5,  0,  7,  8, {69,85,82},      2,     1,     1,     6,     7,     2,     3,     3 }, // Spanish/Latin/Canary Islands
    {    270,    66,    47,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   99,    1,21938,21938,21990,21990,18884,18884,  132,  128,    0,    5,   22,   22,  405,    4,    0, 5906, 5946,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 14,  4, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1,  4,  5,  0,  7, 15, {69,85,82},      2,     1,     1,     6,     7,     2,     3,     3 }, // Spanish/Latin/Ceuta and Melilla
    {    270,    66,    49,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  394,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,   10, 4767,    2,   65, 5906, 5961,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1, 12,  4,  5,  7,  5, {67,76,80},      0,     0,     1,     6,     7,     1,     3,     3 }, // Spanish/Latin/Chile
    {    270,    66,    54,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,   79,   23,   38,21938,21938,21990,21990, 9387,22017,  132,  128,    0,    5,   22,   10, 4779,   15,    0, 5906, 5966,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  7, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1, 15,  5,  0,  7,  8, {67,79,80},      2,     0,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/Colombia
    {    270,    66,    59,    0,    0,   68,   68,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,  289, 4794,    2,    0, 5906, 5974,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1, 19,  4,  0,  7, 10, {67,82,67},      2,     0,     1,     6,     7,     1,     3,     3 }, // Spanish/Latin/Costa Rica
    {    270,    66,    61,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  168,  168,    0,    5,   22,   10, 4813,    2,    0, 5906, 5984,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  4,  4,  5, 17, 23,  1, 11,  4,  0,  7,  4, {67,85,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // Spanish/Latin/Cuba
    {    270,    66,    69,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,  290, 4824,    2,    9, 5906, 5988,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  3, 15,  4,  6,  7, 20, {68,79,80},      2,     1,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/Dominican Republic
    {    270,    66,    70,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,   10, 4839,    2,   65, 5906, 5379,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1, 20,  4,  5,  7,  7, {85,83,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Spanish/Latin/Ecuador
    {    270,    66,    72,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,   10, 4839,    2,    0, 5906, 6008,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1, 20,  4,  0,  7, 11, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/El Salvador
    {    270,    66,    73,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   99,    1,21938,21938,21990,21990,18884,18884,  132,  128,    0,    5,   22,   11, 4859,    2,    0, 5906, 6019,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 14,  4, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  4, 28,  4,  0,  7, 17, {88,65,70},      0,     0,     1,     6,     7,     2,     3,     3 }, // Spanish/Latin/Equatorial Guinea
    {    270,    66,    99,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,   79,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,  245, 4887,    2,    0, 5906, 6036,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  7, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1,  7,  4,  0,  7,  9, {71,84,81},      2,     1,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/Guatemala
    {    270,    66,   106,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1730,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,   18, 4894,    2,    0, 5906, 6045,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1, 17,  4,  0,  7,  8, {72,78,76},      2,     1,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/Honduras
    {    270,    66,   130,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  168,  168,    0,    5,   22,    0,    0,    2,    0, 6053, 6076,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  4,  4,  5, 17, 23,  0,  0,  4,  0, 23, 13,    {0,0,0},      2,     1,     1,     6,     7,     1,     3,     3 }, // Spanish/Latin/Latin America
    {    270,    66,   152,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,   78,   23,   38,21938,21938,21990,21990, 6778, 6778,  168,  168,    0,    5,   22,   10, 4911,    2,    0, 6089, 6106,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 15,  7, 52, 52, 27, 27, 13, 13,  4,  4,  5, 17, 23,  1, 13,  4,  0, 17,  6, {77,88,78},      2,     1,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/Mexico
    {    270,    66,   168,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,  293, 4924,    2,    0, 5906, 6112,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  2, 20,  4,  0,  7,  9, {78,73,79},      2,     1,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/Nicaragua
    {    270,    66,   181,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958, 1121,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,  295, 4944,    2,    0, 5906, 6121,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  3, 15,  4,  0,  7,  6, {80,65,66},      2,     1,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/Panama
    {    270,    66,   183,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,  298, 4959,   15,   86, 5906, 6127,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  3, 17,  5,  6,  7,  8, {80,89,71},      0,     0,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/Paraguay
    {    270,    66,   184,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,   79,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,  278, 4976,   15,    0, 5906, 5368,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  7, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  2, 11,  5,  0,  7,  4, {80,69,78},      2,     1,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/Peru
    {    270,    66,   185,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   23,   38,21938,21938,21990,21990,18884,18884,  132,  128,    0,    5,   22,  146, 4987,    4,    0, 5906, 6135,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1, 13,  5,  0,  7,  9, {80,72,80},      2,     1,     7,     6,     7,     2,     3,     3 }, // Spanish/Latin/Philippines
    {    270,    66,   189,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958, 1121,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,   10, 4839,    2,    0, 5906, 2080,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1, 20,  4,  0,  7, 11, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/Puerto Rico
    {    270,    66,   248,    0,    0,   68,   68,    6,    0,    1,    2,    3,    4,    5,   10,   11,   12,   14,   15,  958,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  168,  168,    0,    5,   22,   10, 4839,    2,    0, 5906, 6144,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  8, 15,  7, 52, 52, 27, 27, 13, 13,  4,  4,  5, 17, 23,  1, 20,  4,  0,  7, 14, {85,83,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/United States
    {    270,    66,   250,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,   10, 5000,   15,   58, 5906, 6158,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  1, 13,  5,  7,  7,  7, {85,89,85},      2,     1,     1,     6,     7,     1,     3,     3 }, // Spanish/Latin/Uruguay
    {    270,    66,   254,    0,    0,   68,   68,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  958,  129,   23,   38,21938,21938,21990,21990, 6778, 6778,  132,  128,    0,    5,   22,  301, 5013,    2,   65, 5906, 6165,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 27,  6, 15,  7, 52, 52, 27, 27, 13, 13,  5,  5,  5, 17, 23,  4, 16,  4,  5,  7,  9, {86,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Spanish/Latin/Venezuela
    {    271,   135,   159,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   15,  113,  129,   10,    0,22030,22030,22077,22077,   83,   83,  692,  723,    0,    5,   22,    0, 5029,    0,    0, 6174, 6182,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 47, 47, 29, 29, 13, 13,  6,  8,  4, 17, 23,  0, 14,  4,  0,  8,  6, {77,65,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Standard Moroccan Tamazight/Tifinagh/Morocco
    {    272,    66,   111,    0,    0, 1090, 1103,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  129,  213,  213,22106,22106,22149,22149, 9291, 9291,    0,    0,    0,    5,   22,  185, 5043,    2,    0, 6188, 1776,  6,  6, 13, 12,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 12,  4, 43, 43, 27, 27, 13, 13,  2,  2,  4, 17, 23,  2, 15,  4,  0, 10,  9, {73,68,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // Sundanese/Latin/Indonesia
    {    273,    66,   230,    0,    0,  566,  566,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0, 1198, 1198, 1198, 1198,   83,   83,    0,    0,  749, 1033,   22,  121, 3467,   15,    0, 6198, 2268,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 59, 59, 59, 59, 13, 13,  2,  2,  5, 51, 23,  3, 20,  5,  0,  9,  8, {84,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Swahili/Latin/Tanzania
    {    273,    66,    57,    0,    0,  566,  566,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0, 1198, 1198, 1198, 1198,   83,   83,    0,    0,  749, 1033,   22,   11, 5058,   15,    0, 6198, 6207,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 59, 59, 59, 59, 13, 13,  2,  2,  5, 51, 23,  2, 16,  5,  0,  9, 32, {67,68,70},      2,     1,     1,     6,     7,     1,     3,     3 }, // Swahili/Latin/Congo - Kinshasa
    {    273,    66,   124,    0,    0,  566,  566,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0, 1198, 1198, 1198, 1198,   83,   83,    0,    0,  749, 1033,   22,  175,  978,   15,    0, 6198, 1307,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 59, 59, 59, 59, 13, 13,  2,  2,  5, 51, 23,  3, 17,  5,  0,  9,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Swahili/Latin/Kenya
    {    273,    66,   243,    0,    0,  566,  566,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0, 1198, 1198, 1198, 1198,   83,   83,    0,    0,  749, 1033,   22,  147, 5074,   15,    0, 6198,  983,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 59, 59, 59, 59, 13, 13,  2,  2,  5, 51, 23,  3, 18,  5,  0,  9,  6, {85,71,88},      0,     0,     1,     7,     7,     1,     3,     3 }, // Swahili/Latin/Uganda
    {    274,    66,   216,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   16,   17,   14,   15,  163,  103,   10,    0,22176,22176,22243,22243,   83,   83,    0,    0,    0,    5,   22,    9,    0,    2,    0, 6239,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 67, 67, 26, 26, 13, 13,  2,  2,  4, 17, 23,  1,  0,  4,  0,  7,  0, {90,65,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // Swati/Latin/South Africa
    {    274,    66,    76,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   16,   17,   14,   15,  163,  103,   23,   38,22176,22176,22243,22243,   83,   83,    0,    0,    0,    5,   22,  155,    0,    2,    0, 6239, 6246,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 67, 67, 26, 26, 13, 13,  2,  2,  4, 17, 23,  1,  0,  4,  0,  7,  8, {83,90,76},      2,     1,     1,     6,     7,     1,     3,     3 }, // Swati/Latin/Eswatini
    {    275,    66,   225,    0,    0, 1115, 1115,    6,    1,    9,    2,    3,   48,    5,   63,   15,   15,   17,   17,  113,  103,   10,    0,22269,22269,22318,22318, 4874, 4874,  698,  731,    0,    5,   22,  160, 5092,    4,    0, 6254, 6261,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  4,  1,  1,  1,  1, 16, 10, 13,  5, 49, 49, 28, 28, 13, 13,  2,  2,  4, 17, 23,  2, 12,  5,  0,  7,  7, {83,69,75},      2,     0,     1,     6,     7,     1,     3,     3 }, // Swedish/Latin/Sweden
    {    275,    66,     2,    0,    0, 1115, 1115,    6,    1,    9,    2,    3,   48,    5,   63,   15,   15,   17,   17,  113,  103,   10,    0,22269,22269,22318,22318, 4874, 4874,  698,  731,    0,    5,   22,   22,  405,    4,    0, 6254, 6268,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  4,  1,  1,  1,  1, 16, 10, 13,  5, 49, 49, 28, 28, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  0,  7,  5, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Swedish/Latin/Aland Islands
    {    275,    66,    83,    0,    0, 1115, 1115,    6,    1,    9,    2,    3,   48,    5,   63,   15,   15,   17,   17,  113,  103,   10,    0,22269,22269,22318,22318, 4874, 4874,  698,  731,    0,    5,   22,   22,  405,    4,    0, 6254, 1698,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  4,  1,  1,  1,  1, 16, 10, 13,  5, 49, 49, 28, 28, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  0,  7,  7, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Swedish/Latin/Finland
    {    276,    66,   226,    0,    0,  463,  463,    6,    0,   17,    2,    3,   48,    5,   10,   11,   12,   19,   20,  404,   49,   10,    0,22346,22346,22408,22408, 4510, 4510,  700,  733,    0,    5,   22,    0, 5104,    4,    0, 6273, 6273,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 62, 62, 27, 27, 13, 13, 12, 11,  4, 17, 23,  0, 16,  5,  0, 16,  7, {67,72,70},      2,     0,     1,     6,     7,     1,     3,     3 }, // Swiss German/Latin/Switzerland
    {    276,    66,    84,    0,    0,  463,  463,    6,    0,   17,    2,    3,   48,    5,   10,   11,   12,   19,   20,  404,   49,   10,    0,22346,22346,22408,22408, 4510, 4510,  700,  733,    0,    5,   22,   22,   83,    4,    0, 6273, 6289,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 62, 62, 27, 27, 13, 13, 12, 11,  4, 17, 23,  1,  4,  5,  0, 16, 10, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Swiss German/Latin/France
    {    276,    66,   136,    0,    0,  463,  463,    6,    0,   17,    2,    3,   48,    5,   10,   11,   12,   19,   20,  404,   49,   10,    0,22346,22346,22408,22408, 4510, 4510,  700,  733,    0,    5,   22,    0, 5104,    4,    0, 6273, 6299,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 62, 62, 27, 27, 13, 13, 12, 11,  4, 17, 23,  0, 16,  5,  0, 16, 13, {67,72,70},      2,     0,     1,     6,     7,     1,     3,     3 }, // Swiss German/Latin/Liechtenstein
    {    277,   123,   113, 1124, 1124, 1124, 1124,    6,    0,    1,    2,    3,    4,    5,   10,   15,   14,   17,   16, 1757,  395,   61,   76,22435,22435,22487,22487,22516,22516,  712,  744, 1084,    5,   22,    0,    0,   15,    0, 6312, 6318,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  9, 15,  7, 52, 52, 29, 29, 13, 13,  4,  4,  4, 17, 23,  0,  0,  5,  0,  6,  4, {73,81,68},      0,     0,     6,     5,     6,     1,     3,     3 }, // Syriac/Syriac/Iraq
    {    277,   123,   227, 1124, 1124, 1124, 1124,    6,    0,    1,    2,    3,    4,    5,   10,   15,   14,   17,   16, 1757,  395,   61,   76,22435,22435,22487,22487,22516,22516,  712,  744, 1084,    5,   22,   99,    0,   15,    0, 6312, 6322,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  9, 15,  7, 52, 52, 29, 29, 13, 13,  4,  4,  4, 17, 23,  5,  0,  5,  0,  6,  5, {83,89,80},      0,     0,     6,     5,     6,     1,     3,     3 }, // Syriac/Syriac/Syria
    {    278,   135,   159,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   15,  113,  129,   10,    0,22529,22529,22077,22077,   83,   83,  692,  723,    0,    5,   22,    0, 5029,    0,    0, 6327, 6182,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 46, 46, 29, 29, 13, 13,  6,  8,  4, 17, 23,  0, 14,  4,  0,  7,  6, {77,65,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Tachelhit/Tifinagh/Morocco
    {    278,    66,   159,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   13,   15,  113,  129,   10,    0,22575,22575,22622,22622,   83,   83,  716,  748,    0,    5,   22,    0, 5120,    0,    0, 6334, 6344,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 47, 47, 29, 29, 13, 13,  6,  8,  4, 17, 23,  0, 14,  4,  0, 10,  6, {77,65,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Tachelhit/Latin/Morocco
    {    280,   127,   255,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  305,    0,   15,    0, 6350,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  4,  0, {86,78,68},      0,     0,     1,     6,     7,     1,     3,     3 }, // Tai Dam/Tai Viet/Vietnam
    {    281,    66,   124,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,22651,22651,22755,22755,22782,22782,  722,  756,    0,    5,   22,  175,  978,    2,    9, 6354, 1307,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5,104,104, 27, 27, 13, 13, 10, 10,  4, 17, 23,  3, 17,  4,  6,  7,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Taita/Latin/Kenya
    {    282,    27,   229,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   12,   11,   16,   17,  786,   78,   10,    0,22795,22795,22849,22849,22876,22876,    0,    0,    0,    5,   22,  306, 5134,    4,    0, 6361, 6367,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 54, 54, 27, 27, 13, 13,  2,  2,  4, 17, 23,  4,  6,  5,  0,  6, 10, {84,74,83},      2,     1,     1,     6,     7,     1,     3,     3 }, // Tajik/Cyrillic/Tajikistan
    {    283,   129,   110,    0,    0, 1130, 1130,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  265,  129,   61,   76,22889,22889,22937,22937,22975,22975,    0,    0, 1088,    5,   22,  120, 5140,    2,    9, 6377, 6382,  6,  6, 13, 13,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 48, 48, 38, 38, 19, 19,  2,  2,  7, 17, 23,  1, 13,  4,  6,  5,  7, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Tamil/Tamil/India
    {    283,   129,   143,    0,    0, 1130, 1130,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  265,  129,   61,   76,22889,22889,22937,22937,22975,22975,    0,    0, 1088,    5,   22,  191, 5153,    2,    9, 6377, 6389,  6,  6, 13, 13,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 48, 48, 38, 38, 19, 19,  2,  2,  7, 17, 23,  2, 17,  4,  6,  5,  7, {77,89,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Tamil/Tamil/Malaysia
    {    283,   129,   210,    0,    0, 1130, 1130,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  265,  129,   61,   76,22889,22889,22937,22937,22975,22975,    0,    0, 1088,    5,   22,   10, 5170,    2,    9, 6377, 6396,  6,  6, 13, 13,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 48, 48, 38, 38, 19, 19,  2,  2,  7, 17, 23,  1, 17,  4,  6,  5, 11, {83,71,68},      2,     1,     7,     6,     7,     1,     3,     3 }, // Tamil/Tamil/Singapore
    {    283,   129,   221,    0,    0, 1130, 1130,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  265,  129,   10,    0,22889,22889,22937,22937,22975,22975,    0,    0, 1088,    5,   22,  310, 5187,    2,    9, 6377, 6407,  6,  6, 13, 13,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 13,  5, 48, 48, 38, 38, 19, 19,  2,  2,  7, 17, 23,  3, 13,  4,  6,  5,  6, {76,75,82},      2,     1,     1,     6,     7,     1,     2,     3 }, // Tamil/Tamil/Sri Lanka
    {    284,    66,   228,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  137,  103,   23,   38,22994,22994,23165,23165,23192,23192,    0,    0,    0,    5,   22,  313, 5200,   15,    0, 6413,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18, 10, 15,  7,171,171, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3, 11,  5,  0, 12,  0, {84,87,68},      2,     0,     7,     6,     7,     1,     3,     3 }, // Taroko/Latin/Taiwan
    {    285,    66,   170,    0,    0,    0,    0,    6,    0,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  129,   10,    0,11682,11682,11735,11735,11762,11762,  732,  766,    0,    5,   22,  127, 3260,    0,    0, 6425, 6438,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 53, 53, 27, 27, 13, 13,  8, 10,  4, 17, 23,  5, 16,  4,  0, 13,  5, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Tasawaq/Latin/Niger
    {    286,    27,   193,    0,    0, 1143, 1143,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1775,   49,   11,    1,23205,23205,23260,23260,23295,23295,    0,    0,    0,    5,   22,  133, 5211,    4,    0, 6443, 5463,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23, 10, 12,  4, 55, 55, 35, 35, 13, 13,  2,  2,  4, 17, 23,  1, 11,  5,  0,  5,  6, {82,85,66},      2,     1,     1,     6,     7,     1,     3,     3 }, // Tatar/Cyrillic/Russia
    {    287,   131,   110,    0,    0, 1152, 1152,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1798,  394,   61,   76,23308,23308,23367,23367,23398,23398,    0,    0, 1095, 1102,   22,  120, 5222,    2,    9, 6448, 6454,  6,  6, 11, 11,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 15,  7, 59, 59, 31, 31, 17, 17,  2,  2,  7, 29, 23,  1, 14,  4,  6,  6,  8, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Telugu/Telugu/India
    {    288,    66,   243,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,23415,23415,23483,23483,23510,23510,  740,  776,    0,    5,   22,  147, 5236,    2,    9, 6462,  983,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 68, 68, 27, 27, 13, 13,  9,  6,  4, 17, 23,  3, 21,  4,  6,  6,  6, {85,71,88},      0,     0,     1,     7,     7,     1,     3,     3 }, // Teso/Latin/Uganda
    {    288,    66,   124,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,23415,23415,23483,23483,23510,23510,  740,  776,    0,    5,   22,  175, 5257,    2,    9, 6462, 6468,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 68, 68, 27, 27, 13, 13,  9,  6,  4, 17, 23,  3, 20,  4,  6,  6,  5, {75,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Teso/Latin/Kenya
    {    289,   133,   231,   24,   24, 1163, 1171,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1816,  129,  542,    0,23523,23523,23590,23590,23612,23612,  749,  782, 1131,    5,   22,  316, 5277,    2,    9, 6473, 6473,  5,  5,  8,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  6, 31,  5, 67, 67, 22, 22, 15, 15, 10, 10,  4, 17, 23,  1,  3,  4,  6,  3,  3, {84,72,66},      2,     1,     7,     6,     7,     1,     3,     3 }, // Thai/Thai/Thailand
    {    290,   134,    50,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1835,  103,   10,    0,23627,23627,23705,23705,23755,23755,  759,  792,    0,    5,   22,  150, 5280,   15,    0, 6476, 6484,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23, 10, 13,  5, 78, 78, 50, 50, 26, 26,  7,  8,  4, 17, 23,  1,  6,  5,  0,  8,  6, {67,78,89},      2,     1,     1,     6,     7,     1,     3,     3 }, // Tibetan/Tibetan/China
    {    290,   134,   110,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1835,  103,   61,   76,23627,23627,23705,23705,23755,23755,  759,  792,    0,    5,   22,  120, 5286,   15,    0, 6476, 6490,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23, 10, 15,  7, 78, 78, 50, 50, 26, 26,  7,  8,  4, 17, 23,  1, 12,  5,  0,  8,  7, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Tibetan/Tibetan/India
    {    291,    33,    74,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1858,   78,   61,   76,23781,23781,23821,23821,23847,23847,    0,    0,    0,    5,   22,    6,    0,    2,    0, 6497,  671,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 21,  8, 15,  7, 40, 40, 26, 26, 13, 13,  2,  2,  4, 17, 23,  3,  0,  4,  0,  3,  4, {69,82,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Tigre/Ethiopic/Eritrea
    {    292,    33,    77,   38,   38, 1178, 1178,    6,    0,    1,    2,    3,    4,    5,   10,   11,   12,   14,   15, 1879,   78,   61,   76,23860,23860,23888,23888,23908,23908,  766,  800,    0,    5,   22,    1,  112,    2,    0, 6500,  143,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 15,  7, 28, 28, 20, 20, 13, 13,  4,  4,  4, 17, 23,  2,  2,  4,  0,  4,  5, {69,84,66},      2,     1,     7,     6,     7,     1,     3,     3 }, // Tigrinya/Ethiopic/Ethiopia
    {    292,    33,    74,   38,   38, 1178, 1178,    6,    0,    1,    2,    3,    4,    5,   10,   16,   17,   14,   15, 1879,   78,   61,   76,23860,23860,23888,23888,23908,23908,  766,  800,    0,    5,   22,    6, 5298,    2,    0, 6500,  671,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 15,  7, 28, 28, 20, 20, 13, 13,  4,  4,  4, 17, 23,  3,  3,  4,  0,  4,  4, {69,82,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Tigrinya/Ethiopic/Eritrea
    {    294,    66,   182,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  787,   78,  573,  589,23921,23921,23965,23965,   83,   83,    0,    0,    0,    5,   22,    0,    0,    4,    0, 6504, 6513,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 16,  8, 44, 44, 27, 27, 13, 13,  2,  2,  4, 17, 23,  0,  0,  5,  0,  9, 13, {80,71,75},      2,     1,     1,     6,     7,     1,     3,     3 }, // Tok Pisin/Latin/Papua New Guinea
    {    295,    66,   235, 1185, 1185, 1185, 1185,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  129,   23,   38,23992,23992,24051,24051,24079,24079,  770,  804, 1135, 1140, 1199,  204, 5301,   15,    0, 6526, 2283,  8,  8,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  6, 15,  7, 59, 59, 28, 28, 13, 13, 10,  6,  5, 59, 65,  2, 17,  5,  0, 13,  5, {84,79,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // Tongan/Latin/Tonga
    {    296,    66,   216,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   16,   17,   14,   15,  163,  103,   10,    0,24092,24092,24163,24163,   83,   83,    0,    0,    0,    5,   22,    9,    0,   15,    0, 6539,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 71, 71, 26, 26, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  8,  0, {90,65,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // Tsonga/Latin/South Africa
    {    297,    66,   216,    0,    0,    0,    0,    6,    0,    9,    2,    3,    4,    5,   10,   16,   17,   14,   15,  163,  103,   10,    0,24189,24189,24252,24252,   83,   83,    0,    0,    0,    5,   22,    9,    0,    2,    0, 6547, 6555,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 63, 63, 31, 31, 13, 13,  2,  2,  4, 17, 23,  1,  0,  4,  0,  8, 13, {90,65,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // Tswana/Latin/South Africa
    {    297,    66,    30,    0,    0,    0,    0,    6,    0,    9,    2,    3,    4,    5,   10,   16,   17,   14,   15,  163,  103,   10,    0,24189,24189,24252,24252,   83,   83,    0,    0,    0,    5,   22,  153,    0,    2,    0, 6547,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 63, 63, 31, 31, 13, 13,  2,  2,  4, 17, 23,  1,  0,  4,  0,  8,  0, {66,87,80},      2,     1,     7,     6,     7,     1,     3,     3 }, // Tswana/Latin/Botswana
    {    298,    66,   239,    0,    0, 1193, 1193,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1896,   50,   10,    0,24283,24283,24336,24336,24363,24363,  780,  810,  185,    5,   22,  126, 5318,    2,    9, 6568, 6574,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  9, 13,  5, 53, 53, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1, 11,  4,  6,  6,  7, {84,82,89},      2,     1,     1,     6,     7,     1,     3,     3 }, // Turkish/Latin/Turkey
    {    298,    66,    63,    0,    0, 1193, 1193,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1896,   50,   23,   38,24283,24283,24336,24336,24363,24363,  780,  810,  185,    5,   22,   22,   83,    2,    9, 6568, 6581,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  9, 15,  7, 53, 53, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  4,  4,  6,  6,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Turkish/Latin/Cyprus
    {    299,    66,   240,    0,    0, 1201, 1201,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   14,   15, 1896,   49,   10,    0,24376,24429,24482,24509,24536,24536,  782,  812, 1264,    5,   22,    0, 5329,    4,    0, 6587, 6599,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16, 10, 13,  5, 53, 53, 27, 27, 13, 13, 13, 14,  4, 17, 23,  0, 14,  5,  0, 12, 12, {84,77,84},      2,     1,     1,     6,     7,     1,     3,     3 }, // Turkmen/Latin/Turkmenistan
    {    301,    66,   169,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  137,  155,   10,    0,24549,24549,24590,24590,   83,   83,    0,    0,    0,    5,   22,  124, 5343,   15,    0, 6611,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 41, 41, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  0,  5,  0, {78,71,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Tyap/Latin/Nigeria
    {    303,    27,   244,    0,    0,  117,  117,    6,    1,    9,    2,    3,    4,    5,   85,   11,   12,   13,   14, 1912,   49,   10,    0,24617,24672, 3049, 3049, 4289, 4289,  795,  826, 1268,  841,   22,  285, 5347,    4,    0, 6616, 6626,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22,  8, 13,  5, 55, 55, 20, 20, 13, 13,  2,  2,  5, 17, 23,  1, 17,  5,  0, 10,  7, {85,65,72},      2,     1,     1,     6,     7,     1,     3,     3 }, // Ukrainian/Cyrillic/Ukraine
    {    304,    66,    91,    0,    0,  781,  781,    6,    1,    0,    2,    3,    4,    5,   10,   13,   14,   18,   16,  404,  180,   11,  597,24727,24727,24779,24779,24806,24806,  402,  828, 1273,    5,   22,   22,  405,    4,    0, 6633, 6648,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 12, 12, 52, 52, 27, 27, 13, 13,  9,  9,  5, 17, 23,  1,  4,  5,  0, 15,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Upper Sorbian/Latin/Germany
    {    305,     4,   178,  661,  661, 1209, 1219,    6,    0,    1,    2,    3,   35,   37,   10,   15,   14,   17,   16, 1934,  129,   61,   76,24819,24819,24819,24819,   83,   83,    0,    0, 1278, 1282,   22,  195, 5364,    2,    9, 6654, 5146,  6,  6, 10,  9,  1,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1, 18,  6, 15,  7, 35, 35, 35, 35, 13, 13,  2,  2,  4, 20, 23,  2, 14,  4,  6,  4,  7, {80,75,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // Urdu/Arabic/Pakistan
    {    305,     4,   110,  661,  661, 1209, 1219,    6,   21,   22,    2,   40,   35,   41,   44,   15,   14,   17,   16, 1934,  129,   61,   76,24819,24819,24819,24819,   83,   83,    0,    0, 1278, 1282,   22,  120, 5378,    2,    9, 6654, 6658,  6,  6, 10,  9,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 18,  6, 15,  7, 35, 35, 35, 35, 13, 13,  2,  2,  4, 20, 23,  1, 12,  4,  6,  4,  5, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Urdu/Arabic/India
    {    306,     4,    50,    0,    0,  323,  333,    6,    0,    1,    2,    3,    4,    5,   10,   12,   11,   20,   19, 1952,  103,   10,    0,24854,24854,24908,24908,24928,24928,  797,  837,    0,    5,   22,  145, 5390,    2,    9, 6663, 6671,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 54, 54, 20, 20, 13, 13, 12, 12,  4, 17, 23,  1, 11,  4,  6,  8,  5, {67,78,89},      2,     1,     1,     6,     7,     1,     3,     3 }, // Uyghur/Arabic/China
    {    307,    66,   251,    0,    0, 1228, 1228,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   17,   16, 1969,   78,   99,    0,24941,24941,25001,25001,25032,25032,  360,  849,  185,    5,   22,  317, 5401,    2,    9, 6676, 6682,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 14,  5, 60, 60, 31, 31, 13, 13,  2,  2,  4, 17, 23,  4, 17,  4,  6,  6, 11, {85,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Uzbek/Latin/Uzbekistan
    {    307,     4,     1,    0,    0,    0,    0,   67,   21,   22,   23,   40,   35,   41,   44,   14,   15,   16,   17, 1987,  505,   99,    1,18197,18197,25045,25045,   83,   83,    0,    0,    0,    5,   22,  269, 3932,    4,    0, 6693, 5137,  6,  6,  6,  6,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 33,  8, 14,  4, 48, 48, 20, 20, 13, 13,  2,  2,  4, 17, 23,  1,  6,  5,  0,  6,  9, {65,70,78},      0,     0,     6,     4,     5,     1,     3,     3 }, // Uzbek/Arabic/Afghanistan
    {    307,    27,   251,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1008,   78,   98,    0,25065,25065,25117,25117,25144,25144,  809,  851,    0,    5,   22,  321, 5418,    4,    0, 6699, 6706,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  8, 15,  5, 52, 52, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3, 14,  5,  0,  7, 10, {85,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Uzbek/Cyrillic/Uzbekistan
    {    308,   139,   134,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   61,   76,25157,25157,25157,25157,   83,   83,    0,    0,    0,    5,   22,   10, 5432,    2,    9, 6716, 6718,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 29, 29, 29, 29, 13, 13,  2,  2,  4, 17, 23,  1,  8,  4,  6,  2,  4, {76,82,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Vai/Vai/Liberia
    {    308,    66,   134,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,25186,25186,25186,25186,   83,   83,    0,    0,    0,    5,   22,   10, 5440,    2,    9, 6722, 6725,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 47, 47, 47, 47, 13, 13,  2,  2,  4, 17, 23,  1, 13,  4,  6,  3,  8, {76,82,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Vai/Latin/Liberia
    {    309,    66,   216,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   16,   17,   14,   15,  163,  103,   10,    0,25233,25233,25302,25302,   83,   83,    0,    0,    0,    5,   22,    9,    0,    2,    0, 6733,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 69, 69, 26, 26, 13, 13,  2,  2,  4, 17, 23,  1,  0,  4,  0,  9,  0, {90,65,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // Venda/Latin/South Africa
    {    310,    66,   255,    0,    0, 1236, 1236,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  265,  129,   10,    0,25328,25328,25382,25382,25414,25414,  811,  853,    0,    5,   22,  305, 5453,    4,    0, 6742, 6752,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 13,  5, 54, 54, 32, 32, 20, 20,  2,  2,  4, 17, 23,  1, 13,  5,  0, 10,  8, {86,78,68},      0,     0,     1,     6,     7,     1,     3,     3 }, // Vietnamese/Latin/Vietnam
    {    311,    66,   258,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 2020,  103,   10,    0,25434,25434,25476,25496,25523,25523,    0,    0,    0,    5,   22,    0,    0,   15,    0, 6760,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23, 10, 13,  5, 42, 42, 20, 27, 13, 13,  2,  2,  4, 17, 23,  0,  0,  5,  0,  7,  0,    {0,0,0},      2,     1,     1,     6,     7,     1,     3,     3 }, // Volapuk/Latin/world
    {    312,    66,   230,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,13800,13800,13861,13861, 1284, 1284,  430,  446,    0,    5,   22,  121, 3467,    2,    0, 6767, 2268,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 61, 61, 27, 27, 13, 13,  5,  9,  4, 17, 23,  3, 20,  4,  0,  8,  8, {84,90,83},      2,     0,     1,     6,     7,     1,     3,     3 }, // Vunjo/Latin/Tanzania
    {    313,    66,    23,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,   22,    0,   15,    0, 6775,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  5,  0, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Walloon/Latin/Belgium
    {    314,    66,   226,    0,    0,  463,  463,    6,    1,   17,    2,    3,    4,    5,   10,   11,   12,   19,   20,  404,  103,   10,    0,25536,25536,25588,25588,25615,25615,    0,    0,    0,    5,   22,    0,    0,   15,    0, 6780, 6786,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18, 10, 13,  5, 52, 52, 27, 27, 13, 13,  2,  2,  4, 17, 23,  0,  0,  5,  0,  6,  6, {67,72,70},      2,     0,     1,     6,     7,     1,     3,     3 }, // Walser/Latin/Switzerland
    {    315,    66,    15,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  240,    0,   15,    0, 6792,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  2,  0,  5,  0,  8,  0, {65,85,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Warlpiri/Latin/Australia
    {    316,    66,   246,    0,    0, 1244, 1255,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,   78,   10,    0,25628,25628,25704,25732,25761,25761,  813,  855, 1302,    5,   22,   94, 5466,    2,    9, 6800, 6807,  6,  6, 11, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 13,  5, 76, 76, 28, 29, 14, 14,  2,  2,  7, 17, 23,  1, 12,  4,  6,  7, 16, {71,66,80},      2,     1,     1,     6,     7,     1,     3,     3 }, // Welsh/Latin/United Kingdom
    {    317,     4,   178,  661,  661,  971,  979,   67,   21,   22,   23,   40,   35,   41,   44,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  195, 5478,   15,    0, 6823,    0,  6,  6,  8,  7,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  2, 13,  5,  0, 14,  0, {80,75,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // Western Balochi/Arabic/Pakistan
    {    317,     4,     1,  661,  661,  971,  979,   67,   21,   22,   23,   40,   35,   41,   44,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  269, 5491,   15,    0, 6823,    0,  6,  6,  8,  7,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1, 17,  5,  0, 14,  0, {65,70,78},      0,     0,     6,     4,     5,     1,     3,     3 }, // Western Balochi/Arabic/Afghanistan
    {    317,     4,   112,  661,  661,  971,  979,   67,   21,   22,   23,   40,   35,   41,   44,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  270, 5508,   15,    0, 6823,    0,  6,  6,  8,  7,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  4, 11,  5,  0, 14,  0, {73,82,82},      0,     0,     6,     5,     5,     1,     3,     3 }, // Western Balochi/Arabic/Iran
    {    317,     4,   176,  661,  661,  971,  979,   67,   21,   22,   23,   40,   35,   41,   44,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,    0,    0,   15,    0, 6823, 6837,  6,  6,  8,  7,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  0,  0,  5,  0, 14,  5, {79,77,82},      3,     0,     6,     5,     6,     1,     3,     3 }, // Western Balochi/Arabic/Oman
    {    317,     4,   245,  661,  661,  971,  979,   67,   21,   22,   23,   40,   35,   41,   44,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,    0,    0,   15,    0, 6823, 6842,  6,  6,  8,  7,  1,  1,  1,  1,  1,  3,  3,  4,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  0,  0,  5,  0, 14, 19, {65,69,68},      2,     1,     6,     6,     7,     1,     3,     3 }, // Western Balochi/Arabic/United Arab Emirates
    {    318,    66,   165,    0,    0,   16,   16,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  394,   10,    0,25775,25775,25828,25828,   83,   83,    0,    0,    0,    5,   22,   22,   83,   15,   58, 6861, 6866,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 53, 53, 20, 20, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  7,  5,  8, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Western Frisian/Latin/Netherlands
    {    319,    33,    77,    0,    0,    0,    0,    6,    0,   17,    2,    3,    4,    5,   10,   14,   15,   16,   17, 2043,   78,   61,   76,25848,25848,25848,25848,25874,25874,    0,    0,    0,    5,   22,    1,  105,    2,    0, 6874,  143,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 22,  8, 15,  7, 26, 26, 26, 26, 13, 13,  2,  2,  4, 17, 23,  2,  9,  4,  0,  5,  5, {69,84,66},      2,     1,     7,     6,     7,     1,     3,     3 }, // Wolaytta/Ethiopic/Ethiopia
    {    320,    66,   206,    0,    0,    0,    0,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17, 2065,  394,   10,    0,25887,25887,25936,25936,25936,25936,  732,  857,    0,    5,   22,  127, 5519,   15,    0, 6879, 2999,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 49, 49, 27, 27, 27, 27,  3,  3,  4, 17, 23,  5, 29,  5,  0,  5,  8, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Wolof/Latin/Senegal
    {    321,    66,   216,    0,    0,    0,    0,    6,    0,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   10,    0,25963,25963,26023,26050,26079,26099,    0,    0,    0,    5,   22,    9, 5548,    2,    0, 6884, 6892,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 13,  5, 60, 60, 27, 29, 20, 21,  2,  2,  4, 17, 23,  1, 25,  4,  0,  8, 15, {90,65,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // Xhosa/Latin/South Africa
    {    322,    66,    40,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   11,   12,  113,  129,   10,    0,26120,26120,26190,26190,26210,26210,  815,  860,    0,    5,   22,   11,    0,    4,   20, 6907, 6913,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 70, 70, 20, 20, 13, 13,  8,  8,  4, 17, 23,  4,  0,  5,  7,  6,  7, {88,65,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Yangben/Latin/Cameroon
    {    323,    47,   244,    0,    0, 1265, 1265,    6,    0,    1,    2,    3,    4,    5,   10,   15,   15,   17,   17, 2082,   78,   10,    0,26223,26223,26223,26223,   83,   83,  823,  868,    0,    5,   22,  285,    0,   15,    0, 6920, 6926,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  8, 13,  5, 53, 53, 53, 53, 13, 13, 11, 10,  4, 17, 23,  1,  0,  5,  0,  6,  9, {85,65,72},      2,     1,     1,     6,     7,     1,     3,     3 }, // Yiddish/Hebrew/Ukraine
    {    324,    66,   169,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 2101,  129,   10,    1,26276,26319,26387,26387,26419,26419,  834,  878, 1309, 1320,   22,  124, 5573,    2,    9, 6935, 6945,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  3, 43, 68, 32, 32, 13, 13,  5,  5, 11, 37, 23,  1, 14,  4,  6, 10,  8, {78,71,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Yoruba/Latin/Nigeria
    {    324,    66,    25,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 2101,  129,   10,    1,26432,26475,26543,26543,26575,26575,  839,  883, 1357, 1320,   22,  127, 5587,    2,    9, 6935, 6953,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  3, 43, 68, 32, 32, 13, 13,  5,  5, 11, 37, 23,  5, 26,  4,  6, 10,  6, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Yoruba/Latin/Benin
    {    325,    66,   170,    0,    0,    0,    0,    6,    0,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,  129,   10,    0,26588,26588,11735,11735,26640,26640,  732,  766,    0,    5,   22,  127, 3260,    0,    0, 6959, 6438,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 52, 52, 27, 27, 13, 13,  8, 10,  4, 17, 23,  5, 16,  4,  0, 10,  5, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Zarma/Latin/Niger
    {    326,    66,    50,    0,    0, 1274, 1274,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,26653,26653,26653,26653,   83,   83,  844,  888,    0,    5,   22,  150, 5613,   15,    0, 6969, 6978,  6,  6, 11, 11,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 89, 89, 89, 89, 13, 13,  7, 12,  4, 17, 23,  1, 10,  5,  0,  9,  8, {67,78,89},      2,     1,     1,     6,     7,     1,     3,     3 }, // Zhuang/Latin/China
    {    327,    66,   216,    0,    0, 1285, 1294,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  549,  567,   10,    0,26742,26742,26815,26815,26842,26842,    0,    0,    0,    5,   22,    9, 5623,    2,    9, 6986, 6993,  6,  6,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 13,  5, 73, 73, 27, 27, 13, 13,  2,  2,  5, 17, 23,  1, 20,  4,  6,  7, 17, {90,65,82},      2,     1,     7,     6,     7,     1,     3,     3 }, // Zulu/Latin/South Africa
    {    328,    66,    32,    0,    0, 1302, 1302,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17, 2117,  186,   10,    0,26855,26855,26941,26941,26975,26975,    0,    0, 1368,    5,   22,    9, 5643,   15,    0, 7010, 7017,  6,  6,  9,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 23, 10, 13,  5, 86, 86, 34, 34, 20, 20,  2,  2,  7, 17, 23,  2, 12,  5,  0,  7,  6, {66,82,76},      2,     1,     7,     6,     7,     1,     3,     3 }, // Kaingang/Latin/Brazil
    {    329,    66,    32,    0,    0, 1311, 1311,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   10,    0,26995,26995,27059,27059,27086,27086,    0,    0, 1375,    5,   22,    9, 5655,   15,    0, 7023, 7031,  6,  6, 10, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 64, 64, 27, 27, 13, 13,  2,  2,  8, 17, 23,  2, 15,  5,  0,  8,  6, {66,82,76},      2,     1,     7,     6,     7,     1,     3,     3 }, // Nheengatu/Latin/Brazil
    {    329,    66,    54,    0,    0, 1311, 1311,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,26995,26995,27059,27059,27086,27086,  132,  128, 1375,    5,   22,   10, 5670,   15,    0, 7037, 7044,  6,  6, 10, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 64, 64, 27, 27, 13, 13,  5,  5,  8, 17, 23,  1, 17,  5,  0,  7,  8, {67,79,80},      2,     0,     7,     6,     7,     1,     3,     3 }, // Nheengatu/Latin/Colombia
    {    329,    66,   254,    0,    0, 1311, 1311,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  186,   23,   38,26995,26995,27059,27059,27086,27086,  132,  128, 1375,    5,   22,  301, 5687,   15,    0, 7037, 7052,  6,  6, 10, 10,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 64, 64, 27, 27, 13, 13,  5,  5,  8, 17, 23,  4, 22,  5,  0,  7,  9, {86,69,83},      2,     1,     7,     6,     7,     1,     3,     3 }, // Nheengatu/Latin/Venezuela
    {    330,    29,   110,    0,    0,    0,    0,    6,    0,    1,    2,   49,    4,    5,   10,   14,   15,   16,   17,  163,  103,   61,   76,27099,27099,27099,27099,   83,   83,  851,   83,    0,    5,   22,  120,    0,   15,    0, 7061,  664,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 54, 54, 54, 54, 13, 13,  4,  4,  4, 17, 23,  1,  0,  5,  0,  8,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Haryanvi/Devanagari/India
    {    331,    66,    91,    0,    0,  915,  915,    6,    1,    0,    2,    3,    4,    5,   10,   14,   15,   16,   17,  404,   78,   10,    0,27153,27153,27209,27209,   83,   83,    0,    0,    0,    5,   22,   22,   83,   15,    0, 7069, 7079,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  8, 13,  5, 56, 56, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  0, 10,  9, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Northern Frisian/Latin/Germany
    {    332,    29,   110,    0,    0,    0,    0,    6,    0,    1,    2,   49,    4,    5,   10,   14,   15,   16,   17,  163,  103,   61,   76, 8522, 8522, 8522, 8522,   83,   83,  855,  900,    0,    5,   22,  120,    0,   15,    0, 7088,  664,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 52, 52, 52, 52, 13, 13,  5,  4,  4, 17, 23,  1,  0,  5,  0,  9,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     3,     3 }, // Rajasthani/Devanagari/India
    {    333,    27,   193,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  133,    0,   15,    0, 7097,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0, 12,  0, {82,85,66},      2,     1,     1,     6,     7,     1,     3,     3 }, // Moksha/Cyrillic/Russia
    {    334,    66,   258,    0,    0,    0,    0,    6,    1,    9,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,27236,27236,27236,27236,   83,   83,  860,  904,    0,    5,   22,    0,    0,    2,    0, 7109, 7118,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 90, 90, 90, 90, 13, 13, 12, 12,  4, 17, 23,  0,  0,  4,  0,  9,  6,    {0,0,0},      2,     1,     1,     6,     7,     1,     2,     2 }, // Toki Pona/Latin/world
    {    335,    66,   214,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,27326,27326,27326,27326,   83,   83,    0,    0,    0,    5,   22,   10,    0,   15,    0, 7124, 7129,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 46, 46, 46, 46, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  5, 13, {83,66,68},      2,     1,     1,     6,     7,     1,     3,     3 }, // Pijin/Latin/Solomon Islands
    {    336,    66,   169,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  124,    0,   15,    0, 7142,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0,  5,  0, {78,71,78},      2,     1,     1,     6,     7,     1,     3,     3 }, // Obolo/Latin/Nigeria
    {    337,     4,   178,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  265,  129,  367,  383,27372,27372,27418,27418,   83,   83,    0,    0,    0,    5,   22,  195, 5364,   15,    0, 7147, 5146,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 16,  8, 46, 46, 24, 24, 13, 13,  2,  2,  4, 17, 23,  2, 13,  5,  0,  5,  7, {80,75,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // Baluchi/Arabic/Pakistan
    {    337,    66,   178,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  265,  129,  573,  589,27442,27442,27512,27512,   83,   83,    0,    0,    0,    5,   22,  195, 5709,   15,    0, 7152, 7159,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 16,  8, 70, 70, 26, 26, 13, 13,  2,  2,  4, 17, 23,  2, 14,  5,  0,  7,  8, {80,75,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // Baluchi/Latin/Pakistan
    {    338,    66,   117,    0,    0,  414,  414,    6,    1,    0,    2,    3,    4,    5,   10,   11,   12,   14,   15, 2140,   78,   10,    0,27538,27538,27592,27592,27626,27626,    0,    0,    0,    5,   22,   22,  405,    4,   20, 7167, 3734,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 21,  8, 13,  5, 54, 54, 34, 34, 13, 13,  2,  2,  4, 17, 23,  1,  4,  5,  7,  6,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Ligurian/Latin/Italy
    {    339,   142,   161,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    1,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  134,    0,   15,    0, 7173,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  4, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0, 18,  0, {77,77,75},      0,     0,     7,     6,     7,     1,     3,     3 }, // Rohingya/Hanifi/Myanmar
    {    339,   142,    20,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   61,   76,   56,   56,   56,   56,   83,   83,    0,    0,    0,    5,   22,  132,    0,   15,    0, 7173,    0,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 15,  7, 27, 27, 27, 27, 13, 13,  2,  2,  4, 17, 23,  1,  0,  5,  0, 18,  0, {66,68,84},      2,     1,     7,     6,     7,     1,     3,     3 }, // Rohingya/Hanifi/Bangladesh
    {    340,     4,   178, 1321, 1321, 1326, 1335,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17, 1934,  129,   61,   76,27639,27639,27639,27639,27696,27696,    0,    0, 1278, 1282,   22,  195, 5364,   15,    0, 7191, 5146,  5,  5,  9,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 18,  6, 15,  7, 57, 57, 57, 57, 13, 13,  2,  2,  4, 20, 23,  2, 14,  5,  0,  7,  7, {80,75,82},      2,     0,     7,     6,     7,     1,     3,     3 }, // Torwali/Arabic/Pakistan
    {    341,    66,    25,    0,    0,  566,  566,    6,    1,    9,    2,    3,    4,    5,   10,   11,   12,   14,   15, 2161, 2178,   10,    0,27709,27709,27767,27767,27801,27801,  872,  916,    0,    5,   22,  127, 5723,   15,   86, 7198, 7209,  6,  6,  8,  8,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  8, 13,  5, 58, 58, 34, 34, 20, 20, 13, 13,  4, 17, 23,  5, 33,  5,  6, 11,  5, {88,79,70},      0,     0,     1,     6,     7,     1,     3,     3 }, // Anii/Latin/Benin
    {    342,    29,   110,    0,    0, 1343, 1353,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  129,   61,   76,27821,27821,27873,27873,27906,27906,  885,  929,    0,    5,   22,  120, 5756,    2,    0, 7214,  664,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 52, 52, 33, 33, 18, 18,  6, 11,  4, 17, 23,  1, 14,  4,  0,  7,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Kangri/Devanagari/India
    {    343,    66,   117,    0,    0,  414,  414,    6,    1,   68,    2,    3,    4,    5,   10,   14,   15,   16,   17,  113,   78,   10,    0,27924,27924,27968,27968,27626,27626,    0,    0,    0,    5,   22,  155,  405,  117,    0, 7221, 3734,  6,  6,  7,  7,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 16,  8, 13,  5, 44, 44, 27, 27, 13, 13,  2,  2,  4, 17, 23,  3,  4,  5,  0,  6,  6, {69,85,82},      2,     1,     1,     6,     7,     1,     3,     3 }, // Venetian/Latin/Italy
    {    344,    66,   110,    0,    0, 1362, 1372,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,    0,  129,   23,   38,27995,27995,28057,28057,28093,28093,   82,  203,    0,    5,   22,  120, 5770,    2,    9, 7227, 7231,  6,  6, 10,  9,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17,  6, 15,  7, 62, 62, 36, 36, 19, 19,  2,  2,  4, 17, 23,  1, 13,  4,  6,  4,  5, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Kuvi/Latin/India
    {    344,    29,   110,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,28112,28112,28112,28112,   83,   83,  891,  336,    0,    5,   22,  120,    0,    2,    0, 7236, 7240,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 71, 71, 71, 71, 13, 13,  4,  5,  4, 17, 23,  1,  0,  4,  0,  4,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Kuvi/Devanagari/India
    {    344,    91,   110,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,28183,28183,28183,28183,   83,   83,  895,  940,    0,    5,   22,  120,    0,    2,    0, 7244, 7248,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 65, 65, 65, 65, 13, 13,  4,  5,  4, 17, 23,  1,  0,  4,  0,  4,  4, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Kuvi/Odia/India
    {    344,   131,   110,    0,    0,    0,    0,    6,    0,    1,    2,    3,    4,    5,   10,   14,   15,   16,   17,  163,  103,   10,    0,28248,28248,28248,28248,   83,   83,  899,  945,    0,    5,   22,  120,    0,    2,    0, 7252, 7256,  6,  6,  6,  6,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 17, 10, 13,  5, 51, 51, 51, 51, 13, 13,  5,  6,  4, 17, 23,  1,  0,  4,  0,  4,  9, {73,78,82},      2,     1,     7,     7,     7,     1,     2,     3 }, // Kuvi/Telugu/India
    {      0,     0,     0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,    {0,0,0},      0,     0,     0,     0,     0,     0,     0,     0 } // trailing zeros
};

static constexpr char16_t list_pattern_part_data[] = {
0x25, 0x31, 0x2c, 0x20, 0x25, 0x32, 0x3b, 0x25, 0x31, 0x2d, 0x438, 0x20,
0x25, 0x32, 0x2d, 0x438, 0x25, 0x31, 0x20, 0x65, 0x6e, 0x20, 0x25, 0x32,
0x25, 0x31, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x64, 0x68, 0x65, 0x20,
0x25, 0x32, 0x25, 0x31, 0x1363, 0x20, 0x25, 0x32, 0x25, 0x31, 0x1363, 0x20,
0x12a5, 0x1293, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x12a5, 0x1293, 0x20, 0x25,
0x32, 0x25, 0x31, 0x20, 0x648, 0x25, 0x32, 0x61b, 0x25, 0x31, 0x20, 0x79,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x587, 0x20, 0x25, 0x32, 0x25, 0x31,
0x20, 0x986, 0x9f0, 0x9c1, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x76, 0x259,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x98f, 0x9ac, 0x982, 0x20, 0x25, 0x32,
0x25, 0x31, 0x20, 0x65, 0x74, 0x61, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20,
0x456, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x906, 0x930, 0x94b, 0x20,
0x25, 0x32, 0x25, 0x31, 0x20, 0x906, 0x930, 0x94b, 0x20, 0x25, 0x32, 0x25,
0x31, 0x20, 0x69, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x438, 0x20, 0x25,
0x32, 0x25, 0x31, 0x20, 0x68, 0x61, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20,
0x2d, 0x20, 0x25, 0x32, 0x25, 0x31, 0x1014, 0x103e, 0x1004, 0x1037, 0x103a, 0x20,
0x25, 0x32, 0x104a, 0x25, 0x31, 0x3001, 0x25, 0x32, 0x25, 0x31, 0x540c, 0x25,
0x32, 0x25, 0x31, 0x2c, 0x20, 0x75, 0x67, 0x20, 0x25, 0x32, 0x25, 0x31,
0x20, 0x75, 0x67, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0xd804, 0xdd03, 0xd804,
0xdd33, 0xd804, 0xdd03, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x13a0, 0x13b4,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x13a0, 0x13b4, 0x20, 0x25, 0x32, 0x25,
0x31, 0x548c, 0x25, 0x32, 0x25, 0x31, 0x53ca, 0x25, 0x32, 0x25, 0x31, 0x20,
0x438, 0x486, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x442, 0x430, 0x442, 0x430,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x75, 0x6e, 0x20, 0x25, 0x32, 0x25,
0x31, 0x20, 0xe8, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x61, 0xa0, 0x25,
0x32, 0x25, 0x31, 0x20, 0x6f, 0x67, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c,
0x20, 0x924, 0x947, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x924, 0x947, 0x20,
0x25, 0x32, 0x25, 0x31, 0x20, 0xf51, 0xf44, 0xf0b, 0x20, 0x25, 0x32, 0x25,
0x31, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20,
0x61, 0x6e, 0x64, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x6b, 0x61, 0x6a,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x6a, 0x61, 0x20, 0x25, 0x32, 0x25,
0x31, 0x2c, 0x20, 0x6b, 0x70, 0x6c, 0x65, 0x20, 0x25, 0x32, 0x25, 0x31,
0x20, 0x6b, 0x70, 0x6c, 0x65, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x20,
0x6f, 0x67, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x61, 0x74, 0x20,
0x25, 0x32, 0x25, 0x31, 0x20, 0x61, 0x74, 0x20, 0x25, 0x32, 0x25, 0x31,
0x20, 0x65, 0x74, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x65, 0x20, 0x25,
0x32, 0x25, 0x31, 0x2e41, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2e41, 0x20, 0xd83a,
0xdd2b, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0xd83a, 0xdd2b, 0x20, 0x25, 0x32,
0x204f, 0x25, 0x31, 0x20, 0x61, 0x67, 0x75, 0x73, 0x20, 0x25, 0x32, 0x25,
0x31, 0x20, 0x10d3, 0x10d0, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x75, 0x6e,
0x64, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x3ba, 0x3b1, 0x3b9, 0x20, 0x25,
0x32, 0x25, 0x31, 0x20, 0xa85, 0xaa8, 0xac7, 0x20, 0x25, 0x32, 0x25, 0x31,
0x2c, 0x20, 0x64, 0x61, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x64, 0x61,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x5d5, 0x25, 0x32, 0x25, 0x31, 0x2c,
0x20, 0x914, 0x930, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x914, 0x930, 0x20,
0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x61, 0x75, 0x72, 0x20, 0x25, 0x32,
0x25, 0x31, 0x20, 0x61, 0x75, 0x72, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20,
0xe9, 0x73, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x6e, 0x61, 0x20,
0x25, 0x32, 0x25, 0x31, 0x20, 0x6e, 0x61, 0x20, 0x25, 0x32, 0x25, 0x31,
0x2c, 0x20, 0x64, 0x61, 0x6e, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x64,
0x61, 0x6e, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x6c, 0x61, 0x6e,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x6c, 0x61, 0x6e, 0x20, 0x25, 0x32,
0x25, 0x31, 0x2c, 0x20, 0x64, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x64,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x61, 0x61, 0x6d, 0x6d, 0x61, 0x20,
0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0xcae, 0xca4, 0xccd, 0xca4, 0xcc1, 0x20,
0x25, 0x32, 0x25, 0x31, 0x20, 0xcae, 0xca4, 0xccd, 0xca4, 0xcc1, 0x20, 0x25,
0x32, 0x25, 0x31, 0x60c, 0x20, 0x25, 0x32, 0x25, 0x31, 0x60c, 0x20, 0x62a,
0x655, 0x6c1, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x62a, 0x655, 0x6c1, 0x20,
0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x924, 0x93f, 0x20, 0x25, 0x32, 0x25,
0x31, 0x20, 0x924, 0x93f, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x436, 0x4d9,
0x43d, 0x435, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x1793, 0x17b7, 0x1784, 0x20,
0x25, 0x32, 0x25, 0x31, 0x20, 0x1793, 0x17b7, 0x1784, 0x200b, 0x25, 0x32, 0x25,
0x31, 0x20, 0xbc0f, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0xfb, 0x20, 0x25,
0x32, 0x25, 0x31, 0x20, 0x436, 0x430, 0x43d, 0x430, 0x20, 0x25, 0x32, 0x25,
0x31, 0x20, 0xec1, 0xea5, 0xeb0, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x6d,
0x70, 0xe9, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x69, 0x72, 0x20, 0x25,
0x32, 0x25, 0x31, 0x20, 0x61, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x61,
0x28, 0x6e, 0x29, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x25, 0x32,
0x20, 0xd0e, 0xd28, 0xd4d, 0xd28, 0xd3f, 0xd35, 0x25, 0x31, 0x20, 0xd15, 0xd42,
0xd1f, 0xd3e, 0xd24, 0xd46, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x75,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x75, 0x20, 0x25, 0x32, 0x25, 0x31,
0x20, 0x985, 0x9ae, 0x9b8, 0x9c1, 0x982, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20,
0x906, 0x923, 0x93f, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x25, 0x32, 0x25,
0x31, 0x20, 0x930, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x14b, 0x301,
0x67, 0x25b, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x1e3f, 0x62, 0x25b,
0x6e, 0x20, 0x14b, 0x301, 0x67, 0x25b, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20,
0x70, 0x254, 0x70, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x61, 0x6e,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x61, 0x6e, 0x20, 0x25, 0x32, 0x25,
0x31, 0x2c, 0x20, 0xb13, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0xb13, 0x20,
0x25, 0x32, 0x25, 0x31, 0x20, 0x4d5, 0x43c, 0x4d5, 0x20, 0x25, 0x32, 0x25,
0x31, 0x60c, 0x20, 0x627, 0x648, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x627,
0x648, 0x20, 0x25, 0x32, 0x25, 0x31, 0x60c, 0x200f, 0x20, 0x25, 0x32, 0x25,
0x31, 0x60c, 0x20, 0x648, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x648, 0x20,
0x25, 0x32, 0x25, 0x31, 0x20, 0x62, 0x65, 0x20, 0x25, 0x32, 0x25, 0x31,
0x20, 0xa05, 0xa24, 0xa47, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x219, 0x69,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x443, 0x43e, 0x43d, 0x43d, 0x430, 0x20,
0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x924, 0x925, 0x93e, 0x20, 0x25, 0x32,
0x25, 0x31, 0x20, 0x924, 0x925, 0x93e, 0x20, 0x25, 0x32, 0x25, 0x31, 0x60c,
0x20, 0x6fd, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x6fd, 0x20, 0x25, 0x32,
0x25, 0x31, 0x2c, 0x20, 0xdc3, 0xdc4, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20,
0xdc3, 0xdc4, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x69, 0x6e, 0x20, 0x25,
0x32, 0x25, 0x31, 0x20, 0x69, 0x79, 0x6f, 0x20, 0x25, 0x32, 0x25, 0x31,
0x2c, 0x20, 0x73, 0x61, 0x72, 0x65, 0x6e, 0x67, 0x20, 0x25, 0x32, 0x25,
0x31, 0x20, 0x73, 0x61, 0x72, 0x65, 0x6e, 0x67, 0x20, 0x25, 0x32, 0x25,
0x31, 0x20, 0x6f, 0x63, 0x68, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x718,
0x25, 0x32, 0x25, 0x31, 0x20, 0xbae, 0xbb1, 0xbcd, 0xbb1, 0xbc1, 0xbae, 0xbcd,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x4bb, 0x4d9, 0x43c, 0x20, 0x25, 0x32,
0x25, 0x31, 0x20, 0xc2e, 0xc30, 0xc3f, 0xc2f, 0xc41, 0x20, 0x25, 0x32, 0x25,
0x31, 0x20, 0xe41, 0xe25, 0xe30, 0x25, 0x32, 0x25, 0x31, 0xe41, 0xe25, 0xe30,
0x25, 0x32, 0x25, 0x31, 0x1295, 0x20, 0x25, 0x32, 0x1295, 0x25, 0x31, 0x20,
0x6d, 0x6f, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x76, 0x65, 0x20, 0x25,
0x32, 0x25, 0x31, 0x20, 0x77, 0x65, 0x20, 0x25, 0x32, 0x25, 0x31, 0x60c,
0x20, 0x627, 0x648, 0x631, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x627, 0x648,
0x631, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x76, 0x61, 0x20, 0x25, 0x32,
0x25, 0x31, 0x20, 0x76, 0xe0, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x20,
0x61, 0x28, 0x63, 0x29, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x61, 0x28,
0x63, 0x29, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x5d0, 0x5d5, 0x5df, 0x20,
0x25, 0x32, 0x25, 0x31, 0x20, 0x63, 0x61, 0x65, 0x75, 0x71, 0x20, 0x25,
0x32, 0x25, 0x31, 0x2c, 0x20, 0x6e, 0x65, 0x2d, 0x25, 0x32, 0x25, 0x31,
0x20, 0x6e, 0x65, 0x2d, 0x25, 0x32, 0x25, 0x31, 0x20, 0x6b, 0x61, 0x72,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x61, 0x73, 0x75, 0xed, 0x20, 0x25,
0x32, 0x25, 0x31, 0x60c, 0x25, 0x32, 0x25, 0x31, 0x20, 0x60c, 0x622, 0x6ba,
0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x622, 0x6ba, 0x20, 0x25, 0x32, 0x25,
0x31, 0x2c, 0x20, 0x915, 0x928, 0x947, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20,
0x915, 0x928, 0x947, 0x20, 0x25, 0x32, 0x25, 0x31, 0x2c, 0x20, 0x61, 0x1e0d,
0x65, 0x20, 0x25, 0x32, 0x25, 0x31, 0x20, 0x61, 0x1e0d, 0x65, 0x20, 0x25,
0x32
};

static constexpr char16_t single_character_data[] = {
0x2e, 0x2c, 0x25, 0x30, 0x2d, 0x2b, 0x65, 0x22, 0x27, 0xa0, 0x45, 0xab,
0xbb, 0x201e, 0x201c, 0x201d, 0x2018, 0x2019, 0x201a, 0x2039, 0x203a, 0x66b, 0x66c, 0x66a,
0x61c, 0x660, 0x61c, 0x2d, 0x61c, 0x2b, 0x623, 0x633, 0x200e, 0x25, 0x200e, 0x200e,
0x2d, 0x200e, 0x2b, 0x9e6, 0x6f0, 0x200e, 0x2b, 0x200e, 0xd7, 0x6f1, 0x6f0, 0x5e,
0x2212, 0x966, 0x1040, 0x300c, 0x300d, 0x300e, 0x300f, 0x200f, 0x2d, 0x200f, 0x2b, 0x627,
0x633, 0xd804, 0xdd36, 0xd7, 0x31, 0x30, 0x5e, 0xf20, 0x202f, 0x2e41, 0xd83a, 0xdd50,
0xd83a, 0xdd09, 0x12c8, 0xabf0, 0x60c, 0x7c0, 0xb7, 0x31, 0x30, 0x5e, 0x200e, 0x2212,
0x1c50, 0x415
};

static constexpr char16_t date_format_data[] = {
0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d,
0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x20, 0x79,
0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64, 0x20, 0x4d,
0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x20, 0x27, 0x448, 0x27,
0x2e, 0x64, 0x64, 0x2e, 0x4d, 0x4d, 0x2e, 0x79, 0x79, 0x79, 0x79, 0x64,
0x64, 0x64, 0x64, 0x2c, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x64, 0x64,
0x2c, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x2f, 0x4d, 0x4d, 0x2f,
0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x20, 0x64, 0x64, 0x20, 0x4d, 0x4d,
0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x2d,
0x4d, 0x4d, 0x2d, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x20, 0x64, 0x20,
0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x2f, 0x4d,
0x2f, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x79,
0x79, 0x79, 0x79, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x64, 0x64, 0x79,
0x79, 0x2f, 0x4d, 0x4d, 0x2f, 0x64, 0x64, 0x79, 0x79, 0x79, 0x79, 0x20,
0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x64, 0x2c, 0x20, 0x64, 0x64, 0x64, 0x64,
0x64, 0x2e, 0x4d, 0x2e, 0x79, 0x79, 0x64, 0x64, 0x2f, 0x4d, 0x4d, 0x2f,
0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x60c, 0x20, 0x64, 0x20,
0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x200f, 0x2f,
0x4d, 0x200f, 0x2f, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x2c,
0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x27, 0x64, 0x65, 0x27,
0x20, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x20, 0x569, 0x2e,
0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x64, 0x2c, 0x20, 0x64, 0x64, 0x64,
0x64, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d,
0x4d, 0x2c, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x2d, 0x4d, 0x2d, 0x79,
0x79, 0x79, 0x79, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79,
0x79, 0x79, 0x2c, 0x20, 0x64, 0x64, 0x64, 0x64, 0x79, 0x79, 0x79, 0x79,
0x28, 0x27, 0x65, 0x27, 0x29, 0x27, 0x6b, 0x6f, 0x27, 0x20, 0x4d, 0x4d,
0x4d, 0x4d, 0x27, 0x72, 0x65, 0x6e, 0x27, 0x20, 0x64, 0x28, 0x27, 0x61,
0x27, 0x29, 0x2c, 0x20, 0x64, 0x64, 0x64, 0x64, 0x79, 0x79, 0x2f, 0x4d,
0x2f, 0x64, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64, 0x20, 0x4d, 0x4d,
0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x202f, 0x27, 0x433, 0x27, 0x2e,
0x64, 0x64, 0x64, 0x64, 0x1361, 0x20, 0x64, 0x64, 0x20, 0x4d, 0x4d, 0x4d,
0x4d, 0x20, 0x130d, 0x122d, 0x130b, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64,
0x2d, 0x4d, 0x4d, 0x2d, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64,
0x2c, 0x20, 0x64, 0x2e, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79,
0x79, 0x79, 0x2e, 0x64, 0x2e, 0x20, 0x4d, 0x2e, 0x20, 0x79, 0x79, 0x79,
0x79, 0x2e, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64, 0x64, 0x2e, 0x20,
0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x2e, 0x64, 0x2e,
0x4d, 0x2e, 0x79, 0x79, 0x2e, 0x64, 0x2e, 0x4d, 0x4d, 0x2e, 0x79, 0x79,
0x202f, 0x27, 0x433, 0x27, 0x2e, 0x79, 0x79, 0x79, 0x79, 0x104a, 0x20, 0x4d,
0x4d, 0x4d, 0x4d, 0x20, 0x64, 0x104a, 0x20, 0x64, 0x64, 0x64, 0x64, 0x79,
0x79, 0x79, 0x79, 0x5e74, 0x4d, 0x6708, 0x64, 0x65e5, 0x20, 0x64, 0x64, 0x64,
0x64, 0x79, 0x79, 0x79, 0x79, 0x2f, 0x4d, 0x2f, 0x64, 0x79, 0x79, 0x79,
0x79, 0x5e74, 0x4d, 0x6708, 0x64, 0x65e5, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64,
0x64, 0x64, 0x2c, 0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x27,
0x64, 0x65, 0x6c, 0x27, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64,
0x64, 0x2c, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x64, 0x2c, 0x20, 0x79,
0x79, 0x79, 0x79, 0x4d, 0x2f, 0x64, 0x2f, 0x79, 0x79, 0x64, 0x64, 0x64,
0x64, 0x2c, 0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x27, 0x43b,
0x27, 0x2e, 0x20, 0x79, 0x79, 0x79, 0x79, 0x2e, 0x79, 0x79, 0x79, 0x79,
0x2e, 0x4d, 0x4d, 0x2e, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20,
0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x202f,
0x27, 0x4ab, 0x27, 0x2e, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x27, 0x64,
0xe4, 0x27, 0x20, 0x64, 0x2e, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79,
0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x20, 0x64, 0x20, 0x4d, 0x4d,
0x4d, 0x4d, 0x20, 0x27, 0x64, 0x69, 0x27, 0x20, 0x27, 0x75, 0x27, 0x20,
0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x2e, 0x20, 0x4d, 0x4d, 0x2e, 0x20,
0x79, 0x79, 0x79, 0x79, 0x2e, 0x64, 0x2e, 0x20, 0x4d, 0x2e, 0x20, 0x79,
0x79, 0x2e, 0x64, 0x64, 0x64, 0x64, 0x20, 0x64, 0x2e, 0x20, 0x4d, 0x4d,
0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x20,
0x27, 0x64, 0x65, 0x6e, 0x27, 0x20, 0x64, 0x2e, 0x20, 0x4d, 0x4d, 0x4d,
0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20,
0x64, 0x2c, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79,
0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0xf66, 0xfa4, 0xfb1, 0xf72, 0xf0b, 0xf63,
0xf7c, 0xf0b, 0x79, 0x79, 0x79, 0x79, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20,
0xf5a, 0xf7a, 0xf66, 0xf0b, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20,
0x64, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79,
0x64, 0x64, 0x64, 0x64, 0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x2c,
0x20, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x2f, 0x4d, 0x4d,
0x2f, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x27, 0x6c, 0x61,
0x27, 0x20, 0x64, 0x2d, 0x27, 0x61, 0x27, 0x20, 0x27, 0x64, 0x65, 0x27,
0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64,
0x64, 0x64, 0x2c, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x64, 0x20, 0x27,
0x6c, 0x69, 0x61, 0x27, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x2e, 0x4d,
0x2e, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x20, 0x64, 0x20,
0x27, 0x64, 0x69, 0x27, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x27, 0x64,
0x61, 0x6c, 0x27, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64,
0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x2e41, 0x20, 0x79, 0x79, 0x79,
0x79, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64, 0x27, 0x6d, 0x68, 0x27,
0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64,
0x64, 0x64, 0x2c, 0x20, 0x64, 0x20, 0x27, 0x64, 0x65, 0x27, 0x20, 0x4d,
0x4d, 0x4d, 0x4d, 0x20, 0x27, 0x64, 0x65, 0x27, 0x20, 0x79, 0x79, 0x79,
0x79, 0x64, 0x64, 0x64, 0x64, 0x1365, 0x20, 0x64, 0x64, 0x20, 0x4d, 0x4d,
0x4d, 0x4d, 0x20, 0x1218, 0x12d3, 0x120d, 0x1275, 0x20, 0x79, 0x79, 0x79, 0x79,
0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64, 0x64, 0x20, 0x4d, 0x4d, 0x4d,
0x4d, 0x2c, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x2c,
0x20, 0x64, 0x20, 0x5d1, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79,
0x79, 0x79, 0x79, 0x79, 0x79, 0x2e, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20,
0x64, 0x2e, 0x2c, 0x20, 0x64, 0x64, 0x64, 0x64, 0x79, 0x79, 0x79, 0x79,
0x2e, 0x20, 0x4d, 0x4d, 0x2e, 0x20, 0x64, 0x64, 0x2e, 0x64, 0x64, 0x64,
0x64, 0x2c, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x64, 0x2e, 0x20, 0x79,
0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x20, 0x27, 0x6c, 0x65, 0x27,
0x20, 0x64, 0x20, 0x27, 0x64, 0x65, 0x27, 0x20, 0x4d, 0x4d, 0x4d, 0x4d,
0x20, 0x79, 0x79, 0x79, 0x79, 0x4d, 0x4d, 0x2f, 0x64, 0x64, 0x2f, 0x79,
0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64, 0x20, 0x27,
0x64, 0x69, 0x27, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x27, 0x64, 0x69,
0x27, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x2f, 0x4d, 0x4d, 0x20,
0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x202f, 0x27, 0x436, 0x27,
0x2e, 0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x2c, 0x20, 0x64, 0x64,
0x64, 0x64, 0x79, 0x79, 0x79, 0x79, 0xb144, 0x20, 0x4d, 0x4d, 0x4d, 0x4d,
0x20, 0x64, 0xc77c, 0x20, 0x64, 0x64, 0x64, 0x64, 0x79, 0x79, 0x2e, 0x20,
0x4d, 0x2e, 0x20, 0x64, 0x2e, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64,
0x27, 0xea, 0x27, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x27, 0x61, 0x27, 0x20,
0x79, 0x79, 0x79, 0x79, 0x27, 0x61, 0x6e, 0x27, 0x79, 0x79, 0x79, 0x79,
0x2d, 0x27, 0x436, 0x27, 0x2e, 0x2c, 0x20, 0x64, 0x2d, 0x4d, 0x4d, 0x4d,
0x4d, 0x2c, 0x20, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x20,
0xe97, 0xeb5, 0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79,
0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x27, 0x64, 0x69, 0x65,
0x27, 0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79,
0x79, 0x64, 0x20, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64,
0x64, 0x2c, 0x20, 0x79, 0x79, 0x79, 0x79, 0x2e, 0x20, 0x27, 0x67, 0x61,
0x64, 0x61, 0x27, 0x20, 0x64, 0x2e, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x79,
0x79, 0x79, 0x79, 0x20, 0x27, 0x6d, 0x27, 0x2e, 0x20, 0x4d, 0x4d, 0x4d,
0x4d, 0x20, 0x64, 0x20, 0x27, 0x64, 0x27, 0x2e, 0x2c, 0x20, 0x64, 0x64,
0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x27, 0x64, 0x65, 0x27,
0x20, 0x64, 0x2e, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79,
0x79, 0x79, 0x79, 0x79, 0x79, 0x2c, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20,
0x64, 0x2c, 0x20, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x2c,
0x20, 0x64, 0x20, 0x27, 0x74, 0x61, 0x27, 0x2019, 0x20, 0x4d, 0x4d, 0x4d,
0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x64,
0x2c, 0x20, 0x79, 0x79, 0x79, 0x79, 0x2c, 0x20, 0x64, 0x64, 0x64, 0x64,
0x79, 0x79, 0x79, 0x79, 0x202f, 0x27, 0x43e, 0x43d, 0x44b, 0x27, 0x20, 0x4d,
0x4d, 0x4d, 0x4d, 0x27, 0x44b, 0x43d, 0x27, 0x20, 0x64, 0x2c, 0x20, 0x64,
0x64, 0x64, 0x64, 0x20, 0x27, 0x433, 0x430, 0x440, 0x430, 0x433, 0x27, 0x79,
0x79, 0x79, 0x79, 0x20, 0x1823, 0x1828, 0x20, 0x180e, 0x180e, 0x180e, 0x1824, 0x20,
0x4d, 0x4d, 0x4d, 0x4d, 0x180e, 0x180e, 0x1822, 0x1822, 0x1828, 0x64, 0x2e, 0x20,
0x64, 0x64, 0x64, 0x64, 0x20, 0x180b, 0x182d, 0x1820, 0x1837, 0x1820, 0x182d, 0x64,
0x64, 0x64, 0x64, 0x20, 0x2c, 0x20, 0x27, 0x6c, 0x79, 0x25b, 0x27, 0x30c,
0x2bc, 0x20, 0x64, 0x20, 0x27, 0x6e, 0x61, 0x27, 0x20, 0x4d, 0x4d, 0x4d,
0x4d, 0x2c, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x20,
0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x27, 0x64, 0x65, 0x27, 0x20,
0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64, 0x20,
0x4d, 0x4d, 0x4d, 0x4d, 0x2c, 0x20, 0x79, 0x79, 0x79, 0x79, 0x202f, 0x27,
0x430, 0x437, 0x27, 0x64, 0x64, 0x64, 0x64, 0x20, 0x62f, 0x20, 0x79, 0x79,
0x79, 0x79, 0x20, 0x62f, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x64, 0x64,
0x64, 0x64, 0x64, 0x2c, 0x20, 0x79, 0x79, 0x79, 0x79, 0x20, 0x27, 0x6d,
0x65, 0x74, 0x74, 0x61, 0x73, 0x27, 0x20, 0x64, 0x2e, 0x20, 0x4d, 0x4d,
0x4d, 0x4d, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x27, 0x69, 0x6c, 0x73,
0x27, 0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79,
0x79, 0x79, 0x79, 0x79, 0x79, 0x202f, 0x27, 0x441, 0x44b, 0x43b, 0x27, 0x20,
0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x64, 0x20, 0x27, 0x43a, 0x4af, 0x43d, 0x44d,
0x27, 0x2c, 0x20, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x20,
0x64, 0x20, 0x27, 0x64, 0x65, 0x27, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20,
0x27, 0x64, 0x65, 0x27, 0x20, 0x27, 0x73, 0x75, 0x27, 0x20, 0x79, 0x79,
0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x20, 0x64, 0x64, 0x20, 0x27, 0x64,
0x65, 0x27, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x27, 0x64, 0x65, 0x27,
0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x60c, 0x20, 0x64,
0x20, 0x712, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64,
0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x2c, 0x20, 0x79, 0x79, 0x79, 0x79, 0x202f,
0x27, 0x435, 0x43b, 0x27, 0x2c, 0x20, 0x64, 0x64, 0x64, 0x64, 0x64, 0x2c,
0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x2c, 0x20,
0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0xe17, 0xe35, 0xe48, 0x20,
0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x79,
0x79, 0x79, 0x79, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0xf60, 0xf72, 0xf0b, 0xf5a,
0xf7a, 0xf66, 0xf0b, 0x64, 0x2c, 0x20, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64,
0x64, 0x64, 0x1361, 0x20, 0x64, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20,
0x12ee, 0x121d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x1363,
0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79,
0x64, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x20,
0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64, 0x20,
0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79, 0x79, 0x202f, 0x27, 0x440,
0x27, 0x2e, 0x64, 0x64, 0x64, 0x64, 0x60c, 0x20, 0x64, 0x20, 0x4d, 0x4d,
0x4d, 0x4d, 0x60c, 0x20, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79,
0x20, 0x64, 0x2d, 0x4d, 0x4d, 0x4d, 0x4d, 0x60c, 0x20, 0x64, 0x64, 0x64,
0x64, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64, 0x2d, 0x4d, 0x4d, 0x4d,
0x4d, 0x2c, 0x20, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x20,
0x646, 0x686, 0x6cc, 0x20, 0x6cc, 0x6cc, 0x644, 0x20, 0x64, 0x20, 0x646, 0x686,
0x6cc, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x64, 0x64, 0x64, 0x64, 0x20,
0x6a9, 0x648, 0x646, 0x6cc, 0x79, 0x79, 0x79, 0x79, 0x20, 0x4d, 0x4d, 0x4d,
0x4d, 0x27, 0x61, 0x27, 0x20, 0x27, 0x64, 0x27, 0x2e, 0x20, 0x64, 0x27,
0x69, 0x64, 0x27, 0x64, 0x64, 0x64, 0x64, 0x1365, 0x20, 0x64, 0x64, 0x20,
0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x130b, 0x120b, 0x1233, 0x20, 0x79, 0x79, 0x79,
0x79, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d,
0x2c, 0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20,
0x64, 0x5d8, 0x5df, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20, 0x79, 0x79, 0x79,
0x79, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64, 0x20, 0x4d, 0x4d, 0x4d,
0x20, 0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x64,
0x20, 0x27, 0x6e, 0x65, 0x27, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x2c, 0x20,
0x79, 0x79, 0x79, 0x79, 0x64, 0x64, 0x64, 0x64, 0x20, 0x64, 0x20, 0x4d,
0x4d, 0x4d, 0x4d, 0x20, 0x27, 0x64, 0x6f, 0x27, 0x20, 0x79, 0x79, 0x79,
0x79, 0x64, 0x64, 0x64, 0x64, 0x2c, 0x20, 0x4d, 0x4d, 0x4d, 0x4d, 0x20,
0x64, 0x20, 0x79, 0x79, 0x79, 0x79, 0x4d, 0x2f, 0x64, 0x2f, 0x79, 0x79,
0x79, 0x79
};

static constexpr char16_t time_format_data[] = {
0x48, 0x48, 0x3a, 0x6d, 0x6d, 0x3a, 0x73, 0x73, 0x20, 0x74, 0x48, 0x48,
0x3a, 0x6d, 0x6d, 0x3a, 0x73, 0x73, 0x20, 0x74, 0x74, 0x74, 0x74, 0x68,
0x3a, 0x6d, 0x6d, 0x3a, 0x73, 0x73, 0x202f, 0x41, 0x70, 0x20, 0x74, 0x74,
0x74, 0x74, 0x68, 0x3a, 0x6d, 0x6d, 0x202f, 0x41, 0x70, 0x68, 0x3a, 0x6d,
0x6d, 0x3a, 0x73, 0x73, 0x202f, 0x41, 0x70, 0x2c, 0x20, 0x74, 0x74, 0x74,
0x74, 0x68, 0x3a, 0x6d, 0x6d, 0x3a, 0x73, 0x73, 0x20, 0x41, 0x70, 0x20,
0x74, 0x74, 0x74, 0x74, 0x68, 0x3a, 0x6d, 0x6d, 0x20, 0x41, 0x70, 0x41,
0x70, 0x20, 0x68, 0x2e, 0x6d, 0x6d, 0x2e, 0x73, 0x73, 0x20, 0x74, 0x74,
0x74, 0x74, 0x48, 0x48, 0x3a, 0x6d, 0x6d, 0x3a, 0x73, 0x73, 0x20, 0x28,
0x74, 0x74, 0x74, 0x74, 0x29, 0x48, 0x48, 0x3a, 0x6d, 0x6d, 0x3a, 0x73,
0x73, 0x2c, 0x20, 0x74, 0x74, 0x74, 0x74, 0x41, 0x70, 0x20, 0x68, 0x3a,
0x6d, 0x6d, 0x3a, 0x73, 0x73, 0x20, 0x74, 0x74, 0x74, 0x74, 0x41, 0x70,
0x20, 0x928, 0x93f, 0x20, 0x68, 0x3a, 0x6d, 0x6d, 0x48, 0x3a, 0x6d, 0x6d,
0x3a, 0x73, 0x73, 0x20, 0x27, 0x447, 0x27, 0x2e, 0x20, 0x74, 0x74, 0x74,
0x74, 0x74, 0x74, 0x74, 0x74, 0x20, 0x48, 0x48, 0x3a, 0x6d, 0x6d, 0x3a,
0x73, 0x73, 0x41, 0x70, 0x68, 0x3a, 0x6d, 0x6d, 0x3a, 0x73, 0x73, 0x20,
0x5b, 0x74, 0x74, 0x74, 0x74, 0x5d, 0x74, 0x74, 0x74, 0x74, 0x20, 0x41,
0x70, 0x68, 0x3a, 0x6d, 0x6d, 0x3a, 0x73, 0x73, 0x48, 0x48, 0x2e, 0x6d,
0x6d, 0x2e, 0x73, 0x73, 0x20, 0x74, 0x74, 0x74, 0x74, 0xf46, 0xf74, 0xf0b,
0xf5a, 0xf7c, 0xf51, 0xf0b, 0x20, 0x68, 0x20, 0xf66, 0xf90, 0xf62, 0xf0b, 0xf58,
0xf0b, 0x20, 0x6d, 0x6d, 0x3a, 0x73, 0x73, 0x20, 0x41, 0x70, 0x20, 0x74,
0x74, 0x74, 0x74, 0xf46, 0xf74, 0xf0b, 0xf5a, 0xf7c, 0xf51, 0xf0b, 0x20, 0x68,
0x20, 0xf66, 0xf90, 0xf62, 0xf0b, 0xf58, 0xf0b, 0x20, 0x6d, 0x6d, 0x20, 0x41,
0x70, 0x41, 0x70, 0x20, 0x27, 0x67, 0x61, 0x27, 0x20, 0x68, 0x3a, 0x6d,
0x6d, 0x3a, 0x73, 0x73, 0x20, 0x74, 0x74, 0x74, 0x74, 0x48, 0x20, 0x27,
0x68, 0x27, 0x20, 0x6d, 0x6d, 0x20, 0x27, 0x6d, 0x69, 0x6e, 0x27, 0x20,
0x73, 0x73, 0x20, 0x27, 0x73, 0x27, 0x20, 0x74, 0x74, 0x74, 0x74, 0x48,
0x48, 0x20, 0x27, 0x68, 0x27, 0x20, 0x6d, 0x6d, 0x20, 0x27, 0x6d, 0x69,
0x6e, 0x27, 0x20, 0x73, 0x73, 0x20, 0x27, 0x73, 0x27, 0x20, 0x74, 0x74,
0x74, 0x74, 0x48, 0x48, 0x2e, 0x6d, 0x6d, 0x3a, 0x73, 0x73, 0x20, 0x27,
0x68, 0x27, 0x20, 0x74, 0x74, 0x74, 0x74, 0x68, 0x68, 0x3a, 0x6d, 0x6d,
0x3a, 0x73, 0x73, 0x20, 0x41, 0x70, 0x20, 0x74, 0x74, 0x74, 0x74, 0x68,
0x68, 0x3a, 0x6d, 0x6d, 0x20, 0x41, 0x70, 0x48, 0x6642, 0x6d, 0x6d, 0x5206,
0x73, 0x73, 0x79d2, 0x20, 0x74, 0x74, 0x74, 0x74, 0x41, 0x70, 0x20, 0x68,
0xc2dc, 0x20, 0x6d, 0xbd84, 0x20, 0x73, 0xcd08, 0x20, 0x74, 0x74, 0x74, 0x74,
0x48, 0x20, 0xec2, 0xea1, 0xe87, 0x20, 0x6d, 0x20, 0xe99, 0xeb2, 0xe97, 0xeb5,
0x20, 0x73, 0x73, 0x20, 0xea7, 0xeb4, 0xe99, 0xeb2, 0xe97, 0xeb5, 0x20, 0x74,
0x74, 0x74, 0x74, 0x27, 0x4b, 0x6c, 0x6f, 0x63, 0x6b, 0x27, 0x20, 0x48,
0x2e, 0x6d, 0x6d, 0x3a, 0x73, 0x73, 0x20, 0x28, 0x74, 0x74, 0x74, 0x74,
0x29, 0x27, 0x4b, 0x6c, 0x27, 0x2e, 0x20, 0x48, 0x2e, 0x6d, 0x6d, 0x68,
0x2e, 0x6d, 0x6d, 0x2e, 0x73, 0x73, 0x20, 0x41, 0x70, 0x20, 0x74, 0x74,
0x74, 0x74, 0x68, 0x2e, 0x6d, 0x6d, 0x2e, 0x20, 0x41, 0x70, 0x27, 0x6b,
0x6c, 0x27, 0x2e, 0x20, 0x48, 0x48, 0x3a, 0x6d, 0x6d, 0x3a, 0x73, 0x73,
0x20, 0x74, 0x74, 0x74, 0x74, 0x74, 0x74, 0x74, 0x74, 0x20, 0x68, 0x3a,
0x6d, 0x6d, 0x3a, 0x73, 0x73, 0x202f, 0x41, 0x70, 0x48, 0x27, 0x68, 0x27,
0x6d, 0x6d, 0x48, 0x20, 0xe19, 0xe32, 0xe2c, 0xe34, 0xe01, 0xe32, 0x20, 0x6d,
0x6d, 0x20, 0xe19, 0xe32, 0xe17, 0xe35, 0x20, 0x73, 0x73, 0x20, 0xe27, 0xe34,
0xe19, 0xe32, 0xe17, 0xe35, 0x20, 0x74, 0x74, 0x74, 0x74, 0x68, 0x68, 0x3a,
0x6d, 0x6d, 0x3a, 0x73, 0x73, 0x202f, 0x41, 0x70, 0x20, 0x74, 0x74, 0x74,
0x74, 0x68, 0x68, 0x3a, 0x6d, 0x6d, 0x202f, 0x41, 0x70, 0x48, 0x3a, 0x6d,
0x6d, 0x20, 0x27, 0x68, 0x6f, 0x64, 0x17a, 0x27, 0x2e
};

static constexpr char16_t days_data[] = {
0x53, 0x75, 0x6e, 0x64, 0x61, 0x79, 0x3b, 0x4d, 0x6f, 0x6e, 0x64, 0x61,
0x79, 0x3b, 0x54, 0x75, 0x65, 0x73, 0x64, 0x61, 0x79, 0x3b, 0x57, 0x65,
0x64, 0x6e, 0x65, 0x73, 0x64, 0x61, 0x79, 0x3b, 0x54, 0x68, 0x75, 0x72,
0x73, 0x64, 0x61, 0x79, 0x3b, 0x46, 0x72, 0x69, 0x64, 0x61, 0x79, 0x3b,
0x53, 0x61, 0x74, 0x75, 0x72, 0x64, 0x61, 0x79, 0x53, 0x75, 0x6e, 0x3b,
0x4d, 0x6f, 0x6e, 0x3b, 0x54, 0x75, 0x65, 0x3b, 0x57, 0x65, 0x64, 0x3b,
0x54, 0x68, 0x75, 0x3b, 0x46, 0x72, 0x69, 0x3b, 0x53, 0x61, 0x74, 0x53,
0x3b, 0x4d, 0x3b, 0x54, 0x3b, 0x57, 0x3b, 0x54, 0x3b, 0x46, 0x3b, 0x53,
0x37, 0x3b, 0x31, 0x3b, 0x32, 0x3b, 0x33, 0x3b, 0x34, 0x3b, 0x35, 0x3b,
0x36, 0x410, 0x43c, 0x4bd, 0x44b, 0x448, 0x430, 0x3b, 0x410, 0x448, 0x4d9, 0x430,
0x445, 0x44c, 0x430, 0x3b, 0x410, 0x4a9, 0x430, 0x448, 0x430, 0x3b, 0x410, 0x445,
0x430, 0x448, 0x430, 0x3b, 0x410, 0x525, 0x448, 0x44c, 0x430, 0x448, 0x430, 0x3b,
0x410, 0x445, 0x4d9, 0x430, 0x448, 0x430, 0x3b, 0x410, 0x441, 0x430, 0x431, 0x448,
0x430, 0x410, 0x43c, 0x3b, 0x410, 0x448, 0x4d9, 0x3b, 0x410, 0x4a9, 0x3b, 0x410,
0x445, 0x3b, 0x410, 0x525, 0x3b, 0x410, 0x445, 0x4d9, 0x3b, 0x410, 0x441, 0x41c,
0x3b, 0x428, 0x4d9, 0x3b, 0x4a8, 0x3b, 0x425, 0x3b, 0x524, 0x3b, 0x425, 0x4d9,
0x3b, 0x421, 0x41, 0x63, 0x61, 0x61, 0x64, 0x61, 0x3b, 0x45, 0x74, 0x6c,
0x65, 0x65, 0x6e, 0x69, 0x3b, 0x54, 0x61, 0x6c, 0x61, 0x61, 0x74, 0x61,
0x3b, 0x41, 0x72, 0x62, 0x61, 0x71, 0x61, 0x3b, 0x4b, 0x61, 0x6d, 0x69,
0x69, 0x73, 0x69, 0x3b, 0x47, 0x75, 0x6d, 0x71, 0x61, 0x74, 0x61, 0x3b,
0x53, 0x61, 0x62, 0x74, 0x69, 0x41, 0x63, 0x61, 0x3b, 0x45, 0x74, 0x6c,
0x3b, 0x54, 0x61, 0x6c, 0x3b, 0x41, 0x72, 0x62, 0x3b, 0x4b, 0x61, 0x6d,
0x3b, 0x47, 0x75, 0x6d, 0x3b, 0x53, 0x61, 0x62, 0x41, 0x3b, 0x45, 0x3b,
0x54, 0x3b, 0x41, 0x3b, 0x4b, 0x3b, 0x47, 0x3b, 0x53, 0x53, 0x6f, 0x6e,
0x64, 0x61, 0x67, 0x3b, 0x4d, 0x61, 0x61, 0x6e, 0x64, 0x61, 0x67, 0x3b,
0x44, 0x69, 0x6e, 0x73, 0x64, 0x61, 0x67, 0x3b, 0x57, 0x6f, 0x65, 0x6e,
0x73, 0x64, 0x61, 0x67, 0x3b, 0x44, 0x6f, 0x6e, 0x64, 0x65, 0x72, 0x64,
0x61, 0x67, 0x3b, 0x56, 0x72, 0x79, 0x64, 0x61, 0x67, 0x3b, 0x53, 0x61,
0x74, 0x65, 0x72, 0x64, 0x61, 0x67, 0x53, 0x6f, 0x2e, 0x3b, 0x4d, 0x61,
0x2e, 0x3b, 0x44, 0x69, 0x2e, 0x3b, 0x57, 0x6f, 0x2e, 0x3b, 0x44, 0x6f,
0x2e, 0x3b, 0x56, 0x72, 0x2e, 0x3b, 0x53, 0x61, 0x2e, 0x53, 0x3b, 0x4d,
0x3b, 0x44, 0x3b, 0x57, 0x3b, 0x44, 0x3b, 0x56, 0x3b, 0x53, 0x74, 0x73,
0x75, 0x294, 0x6e, 0x74, 0x73, 0x268, 0x3b, 0x74, 0x73, 0x75, 0x294, 0x75,
0x6b, 0x70, 0xe0, 0x3b, 0x74, 0x73, 0x75, 0x294, 0x75, 0x67, 0x68, 0x254,
0x65, 0x3b, 0x74, 0x73, 0x75, 0x294, 0x75, 0x74, 0x254, 0x300, 0x6d, 0x6c,
0xf2, 0x3b, 0x74, 0x73, 0x75, 0x294, 0x75, 0x6d, 0xe8, 0x3b, 0x74, 0x73,
0x75, 0x294, 0x75, 0x67, 0x68, 0x268, 0x302, 0x6d, 0x3b, 0x74, 0x73, 0x75,
0x294, 0x6e, 0x64, 0x7a, 0x268, 0x6b, 0x254, 0x294, 0x254, 0x6e, 0x74, 0x73,
0x3b, 0x6b, 0x70, 0x61, 0x3b, 0x67, 0x68, 0x254, 0x3b, 0x74, 0x254, 0x6d,
0x3b, 0x75, 0x6d, 0x65, 0x3b, 0x67, 0x68, 0x268, 0x3b, 0x64, 0x7a, 0x6b,
0x6e, 0x3b, 0x6b, 0x3b, 0x67, 0x3b, 0x74, 0x3b, 0x75, 0x3b, 0x67, 0x3b,
0x64, 0x4b, 0x77, 0x65, 0x73, 0x69, 0x64, 0x61, 0x3b, 0x44, 0x77, 0x6f,
0x77, 0x64, 0x61, 0x3b, 0x42, 0x65, 0x6e, 0x61, 0x64, 0x61, 0x3b, 0x57,
0x75, 0x6b, 0x75, 0x64, 0x61, 0x3b, 0x59, 0x61, 0x77, 0x64, 0x61, 0x3b,
0x46, 0x69, 0x64, 0x61, 0x3b, 0x4d, 0x65, 0x6d, 0x65, 0x6e, 0x65, 0x64,
0x61, 0x4b, 0x77, 0x65, 0x3b, 0x44, 0x77, 0x6f, 0x3b, 0x42, 0x65, 0x6e,
0x3b, 0x57, 0x75, 0x6b, 0x3b, 0x59, 0x61, 0x77, 0x3b, 0x46, 0x69, 0x61,
0x3b, 0x4d, 0x65, 0x6d, 0x4b, 0x3b, 0x44, 0x3b, 0x42, 0x3b, 0x57, 0x3b,
0x59, 0x3b, 0x46, 0x3b, 0x4d, 0x65, 0x20, 0x64, 0x69, 0x65, 0x6c, 0x3b,
0x65, 0x20, 0x68, 0xeb, 0x6e, 0xeb, 0x3b, 0x65, 0x20, 0x6d, 0x61, 0x72,
0x74, 0xeb, 0x3b, 0x65, 0x20, 0x6d, 0xeb, 0x72, 0x6b, 0x75, 0x72, 0xeb,
0x3b, 0x65, 0x20, 0x65, 0x6e, 0x6a, 0x74, 0x65, 0x3b, 0x65, 0x20, 0x70,
0x72, 0x65, 0x6d, 0x74, 0x65, 0x3b, 0x65, 0x20, 0x73, 0x68, 0x74, 0x75,
0x6e, 0xeb, 0x64, 0x69, 0x65, 0x3b, 0x68, 0xeb, 0x6e, 0x3b, 0x6d, 0x61,
0x72, 0x3b, 0x6d, 0xeb, 0x72, 0x3b, 0x65, 0x6e, 0x6a, 0x3b, 0x70, 0x72,
0x65, 0x3b, 0x73, 0x68, 0x74, 0x64, 0x3b, 0x68, 0x3b, 0x6d, 0x3b, 0x6d,
0x3b, 0x65, 0x3b, 0x70, 0x3b, 0x73, 0x68, 0x12a5, 0x1211, 0x12f5, 0x3b, 0x1230,
0x129e, 0x3b, 0x121b, 0x12ad, 0x1230, 0x129e, 0x3b, 0x1228, 0x1261, 0x12d5, 0x3b, 0x1210,
0x1219, 0x1235, 0x3b, 0x12d3, 0x122d, 0x1265, 0x3b, 0x1245, 0x12f3, 0x121c, 0x12a5, 0x1211,
0x12f5, 0x3b, 0x1230, 0x129e, 0x3b, 0x121b, 0x12ad, 0x1230, 0x3b, 0x1228, 0x1261, 0x12d5,
0x3b, 0x1210, 0x1219, 0x1235, 0x3b, 0x12d3, 0x122d, 0x1265, 0x3b, 0x1245, 0x12f3, 0x121c,
0x12a5, 0x3b, 0x1230, 0x3b, 0x121b, 0x3b, 0x1228, 0x3b, 0x1210, 0x3b, 0x12d3, 0x3b,
0x1245, 0x627, 0x644, 0x623, 0x62d, 0x62f, 0x3b, 0x627, 0x644, 0x627, 0x62b, 0x646,
0x64a, 0x646, 0x3b, 0x627, 0x644, 0x62b, 0x644, 0x627, 0x62b, 0x627, 0x621, 0x3b,
0x627, 0x644, 0x623, 0x631, 0x628, 0x639, 0x627, 0x621, 0x3b, 0x627, 0x644, 0x62e,
0x645, 0x64a, 0x633, 0x3b, 0x627, 0x644, 0x62c, 0x645, 0x639, 0x629, 0x3b, 0x627,
0x644, 0x633, 0x628, 0x62a, 0x62d, 0x3b, 0x646, 0x3b, 0x62b, 0x3b, 0x631, 0x3b,
0x62e, 0x3b, 0x62c, 0x3b, 0x633, 0x64, 0x6f, 0x6d, 0x69, 0x6e, 0x63, 0x68,
0x65, 0x3b, 0x6c, 0x75, 0x6e, 0x73, 0x3b, 0x6d, 0x61, 0x72, 0x74, 0x7a,
0x3b, 0x6d, 0x69, 0x65, 0x72, 0x63, 0x72, 0x65, 0x73, 0x3b, 0x63, 0x68,
0x75, 0x65, 0x76, 0x65, 0x73, 0x3b, 0x76, 0x69, 0x65, 0x72, 0x6e, 0x65,
0x73, 0x3b, 0x73, 0x61, 0x62, 0x61, 0x64, 0x6f, 0x64, 0x6f, 0x6d, 0x3b,
0x6c, 0x75, 0x6e, 0x3b, 0x6d, 0x61, 0x72, 0x3b, 0x6d, 0x69, 0x65, 0x3b,
0x63, 0x68, 0x75, 0x3b, 0x76, 0x69, 0x65, 0x3b, 0x73, 0x61, 0x62, 0x44,
0x3b, 0x4c, 0x3b, 0x4d, 0x61, 0x3b, 0x4d, 0x69, 0x3b, 0x43, 0x68, 0x3b,
0x56, 0x3b, 0x53, 0x56f, 0x56b, 0x580, 0x561, 0x56f, 0x56b, 0x3b, 0x565, 0x580,
0x56f, 0x578, 0x582, 0x577, 0x561, 0x562, 0x569, 0x56b, 0x3b, 0x565, 0x580, 0x565,
0x584, 0x577, 0x561, 0x562, 0x569, 0x56b, 0x3b, 0x579, 0x578, 0x580, 0x565, 0x584,
0x577, 0x561, 0x562, 0x569, 0x56b, 0x3b, 0x570, 0x56b, 0x576, 0x563, 0x577, 0x561,
0x562, 0x569, 0x56b, 0x3b, 0x578, 0x582, 0x580, 0x562, 0x561, 0x569, 0x3b, 0x577,
0x561, 0x562, 0x561, 0x569, 0x56f, 0x56b, 0x580, 0x3b, 0x565, 0x580, 0x56f, 0x3b,
0x565, 0x580, 0x584, 0x3b, 0x579, 0x580, 0x584, 0x3b, 0x570, 0x576, 0x563, 0x3b,
0x578, 0x582, 0x580, 0x3b, 0x577, 0x562, 0x569, 0x53f, 0x3b, 0x535, 0x3b, 0x535,
0x3b, 0x549, 0x3b, 0x540, 0x3b, 0x548, 0x3b, 0x547, 0x9a6, 0x9c7, 0x993, 0x9ac,
0x9be, 0x9f0, 0x3b, 0x9b8, 0x9cb, 0x9ae, 0x9ac, 0x9be, 0x9f0, 0x3b, 0x9ae, 0x999,
0x9cd, 0x997, 0x9b2, 0x9ac, 0x9be, 0x9f0, 0x3b, 0x9ac, 0x9c1, 0x9a7, 0x9ac, 0x9be,
0x9f0, 0x3b, 0x9ac, 0x9c3, 0x9b9, 0x9b8, 0x9cd, 0x9aa, 0x9a4, 0x9bf, 0x9ac, 0x9be,
0x9f0, 0x3b, 0x9b6, 0x9c1, 0x995, 0x9cd, 0x9f0, 0x9ac, 0x9be, 0x9f0, 0x3b, 0x9b6,
0x9a8, 0x9bf, 0x9ac, 0x9be, 0x9f0, 0x9a6, 0x9c7, 0x993, 0x3b, 0x9b8, 0x9cb, 0x9ae,
0x3b, 0x9ae, 0x999, 0x9cd, 0x997, 0x9b2, 0x3b, 0x9ac, 0x9c1, 0x9a7, 0x3b, 0x9ac,
0x9c3, 0x9b9, 0x3b, 0x9b6, 0x9c1, 0x995, 0x9cd, 0x9f0, 0x3b, 0x9b6, 0x9a8, 0x9bf,
0x9a6, 0x3b, 0x9b8, 0x3b, 0x9ae, 0x3b, 0x9ac, 0x3b, 0x9ac, 0x3b, 0x9b6, 0x3b,
0x9b6, 0x64, 0x6f, 0x6d, 0x69, 0x6e, 0x67, 0x75, 0x3b, 0x6c, 0x6c, 0x75,
0x6e, 0x65, 0x73, 0x3b, 0x6d, 0x61, 0x72, 0x74, 0x65, 0x73, 0x3b, 0x6d,
0x69, 0xe9, 0x72, 0x63, 0x6f, 0x6c, 0x65, 0x73, 0x3b, 0x78, 0x75, 0x65,
0x76, 0x65, 0x73, 0x3b, 0x76, 0x69, 0x65, 0x6e, 0x72, 0x65, 0x73, 0x3b,
0x73, 0xe1, 0x62, 0x61, 0x64, 0x75, 0x64, 0x6f, 0x6d, 0x3b, 0x6c, 0x6c,
0x75, 0x3b, 0x6d, 0x61, 0x72, 0x3b, 0x6d, 0x69, 0xe9, 0x3b, 0x78, 0x75,
0x65, 0x3b, 0x76, 0x69, 0x65, 0x3b, 0x73, 0xe1, 0x62, 0x44, 0x3b, 0x4c,
0x3b, 0x4d, 0x3b, 0x4d, 0x3b, 0x58, 0x3b, 0x56, 0x3b, 0x53, 0x4a, 0x75,
0x6d, 0x61, 0x70, 0x69, 0x6c, 0x69, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x74,
0x61, 0x74, 0x75, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x6e, 0x6e, 0x65, 0x3b,
0x4a, 0x75, 0x6d, 0x61, 0x74, 0x61, 0x6e, 0x6f, 0x3b, 0x41, 0x6c, 0x68,
0x61, 0x6d, 0x69, 0x73, 0x69, 0x3b, 0x49, 0x6a, 0x75, 0x6d, 0x61, 0x61,
0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x6d, 0x6f, 0x73, 0x69, 0x4a, 0x70, 0x69,
0x3b, 0x4a, 0x74, 0x74, 0x3b, 0x4a, 0x6e, 0x6e, 0x3b, 0x4a, 0x74, 0x6e,
0x3b, 0x41, 0x6c, 0x68, 0x3b, 0x49, 0x6a, 0x6d, 0x3b, 0x4a, 0x6d, 0x6f,
0x4a, 0x3b, 0x4a, 0x3b, 0x4a, 0x3b, 0x4a, 0x3b, 0x41, 0x3b, 0x49, 0x3b,
0x4a, 0x57, 0x61, 0x69, 0x20, 0x59, 0x6f, 0x6b, 0x61, 0x20, 0x42, 0x61,
0x77, 0x61, 0x69, 0x3b, 0x57, 0x61, 0x69, 0x20, 0x54, 0x75, 0x6e, 0x67,
0x61, 0x3b, 0x54, 0x6f, 0x6b, 0x69, 0x20, 0x47, 0x69, 0x74, 0x75, 0x6e,
0x67, 0x3b, 0x54, 0x73, 0x61, 0x6d, 0x20, 0x4b, 0x61, 0x73, 0x75, 0x77,
0x61, 0x3b, 0x57, 0x61, 0x69, 0x20, 0x4e, 0x61, 0x20, 0x4e, 0x61, 0x73,
0x3b, 0x57, 0x61, 0x69, 0x20, 0x4e, 0x61, 0x20, 0x54, 0x69, 0x79, 0x6f,
0x6e, 0x3b, 0x57, 0x61, 0x69, 0x20, 0x4e, 0x61, 0x20, 0x43, 0x68, 0x69,
0x72, 0x69, 0x6d, 0x59, 0x6f, 0x6b, 0x3b, 0x54, 0x75, 0x6e, 0x67, 0x3b,
0x47, 0x69, 0x74, 0x75, 0x6e, 0x67, 0x3b, 0x54, 0x73, 0x61, 0x6e, 0x3b,
0x4e, 0x61, 0x73, 0x3b, 0x4e, 0x61, 0x74, 0x3b, 0x43, 0x68, 0x69, 0x72,
0x62, 0x61, 0x7a, 0x61, 0x72, 0x3b, 0x62, 0x61, 0x7a, 0x61, 0x72, 0x20,
0x65, 0x72, 0x74, 0x259, 0x73, 0x69, 0x3b, 0xe7, 0x259, 0x72, 0x15f, 0x259,
0x6e, 0x62, 0x259, 0x20, 0x61, 0x78, 0x15f, 0x61, 0x6d, 0x131, 0x3b, 0xe7,
0x259, 0x72, 0x15f, 0x259, 0x6e, 0x62, 0x259, 0x3b, 0x63, 0xfc, 0x6d, 0x259,
0x20, 0x61, 0x78, 0x15f, 0x61, 0x6d, 0x131, 0x3b, 0x63, 0xfc, 0x6d, 0x259,
0x3b, 0x15f, 0x259, 0x6e, 0x62, 0x259, 0x42, 0x2e, 0x3b, 0x42, 0x2e, 0x45,
0x2e, 0x3b, 0xc7, 0x2e, 0x41, 0x2e, 0x3b, 0xc7, 0x2e, 0x3b, 0x43, 0x2e,
0x41, 0x2e, 0x3b, 0x43, 0x2e, 0x3b, 0x15e, 0x2e, 0x42, 0x2e, 0x3b, 0x42,
0x2e, 0x65, 0x2e, 0x3b, 0xc7, 0x2e, 0x61, 0x2e, 0x3b, 0xc7, 0x2e, 0x3b,
0x43, 0x2e, 0x61, 0x2e, 0x3b, 0x43, 0x2e, 0x3b, 0x15e, 0x2e, 0x431, 0x430,
0x437, 0x430, 0x440, 0x3b, 0x431, 0x430, 0x437, 0x430, 0x440, 0x20, 0x435, 0x440,
0x442, 0x4d9, 0x441, 0x438, 0x3b, 0x447, 0x4d9, 0x440, 0x448, 0x4d9, 0x43d, 0x431,
0x4d9, 0x20, 0x430, 0x445, 0x448, 0x430, 0x43c, 0x44b, 0x3b, 0x447, 0x4d9, 0x440,
0x448, 0x4d9, 0x43d, 0x431, 0x4d9, 0x3b, 0x4b9, 0x4af, 0x43c, 0x4d9, 0x20, 0x430,
0x445, 0x448, 0x430, 0x43c, 0x44b, 0x3b, 0x4b9, 0x4af, 0x43c, 0x4d9, 0x3b, 0x448,
0x4d9, 0x43d, 0x431, 0x4d9, 0x411, 0x2e, 0x3b, 0x411, 0x2e, 0x415, 0x2e, 0x3b,
0x427, 0x2e, 0x410, 0x2e, 0x3b, 0x427, 0x2e, 0x3b, 0x4b8, 0x2e, 0x410, 0x2e,
0x3b, 0x4b8, 0x2e, 0x3b, 0x428, 0x2e, 0x73, 0x254, 0x301, 0x6e, 0x64, 0x1dd,
0x3b, 0x6c, 0x1dd, 0x6e, 0x64, 0xed, 0x3b, 0x6d, 0x61, 0x61, 0x64, 0xed,
0x3b, 0x6d, 0x25b, 0x6b, 0x72, 0x25b, 0x64, 0xed, 0x3b, 0x6a, 0x1dd, 0x1dd,
0x64, 0xed, 0x3b, 0x6a, 0xfa, 0x6d, 0x62, 0xe1, 0x3b, 0x73, 0x61, 0x6d,
0x64, 0xed, 0x73, 0x254, 0x301, 0x6e, 0x3b, 0x6c, 0x1dd, 0x6e, 0x3b, 0x6d,
0x61, 0x61, 0x3b, 0x6d, 0x25b, 0x6b, 0x3b, 0x6a, 0x1dd, 0x1dd, 0x3b, 0x6a,
0xfa, 0x6d, 0x3b, 0x73, 0x61, 0x6d, 0x73, 0x3b, 0x6c, 0x3b, 0x6d, 0x3b,
0x6d, 0x3b, 0x6a, 0x3b, 0x6a, 0x3b, 0x73, 0x6b, 0x61, 0x72, 0x69, 0x3b,
0x6e, 0x74, 0x25b, 0x6e, 0x25b, 0x3b, 0x74, 0x61, 0x72, 0x61, 0x74, 0x61,
0x3b, 0x61, 0x72, 0x61, 0x62, 0x61, 0x3b, 0x61, 0x6c, 0x61, 0x6d, 0x69,
0x73, 0x61, 0x3b, 0x6a, 0x75, 0x6d, 0x61, 0x3b, 0x73, 0x69, 0x62, 0x69,
0x72, 0x69, 0x6b, 0x61, 0x72, 0x3b, 0x6e, 0x74, 0x25b, 0x3b, 0x74, 0x61,
0x72, 0x3b, 0x61, 0x72, 0x61, 0x3b, 0x61, 0x6c, 0x61, 0x3b, 0x6a, 0x75,
0x6d, 0x3b, 0x73, 0x69, 0x62, 0x4b, 0x3b, 0x4e, 0x3b, 0x54, 0x3b, 0x41,
0x3b, 0x41, 0x3b, 0x4a, 0x3b, 0x53, 0x9b0, 0x9ac, 0x9bf, 0x9ac, 0x9be, 0x9b0,
0x3b, 0x9b8, 0x9cb, 0x9ae, 0x9ac, 0x9be, 0x9b0, 0x3b, 0x9ae, 0x999, 0x9cd, 0x997,
0x9b2, 0x9ac, 0x9be, 0x9b0, 0x3b, 0x9ac, 0x9c1, 0x9a7, 0x9ac, 0x9be, 0x9b0, 0x3b,
0x9ac, 0x9c3, 0x9b9, 0x9b8, 0x9cd, 0x9aa, 0x9a4, 0x9bf, 0x9ac, 0x9be, 0x9b0, 0x3b,
0x9b6, 0x9c1, 0x995, 0x9cd, 0x9b0, 0x9ac, 0x9be, 0x9b0, 0x3b, 0x9b6, 0x9a8, 0x9bf,
0x9ac, 0x9be, 0x9b0, 0x9b0, 0x9ac, 0x9bf, 0x3b, 0x9b8, 0x9cb, 0x9ae, 0x3b, 0x9ae,
0x999, 0x9cd, 0x997, 0x9b2, 0x3b, 0x9ac, 0x9c1, 0x9a7, 0x3b, 0x9ac, 0x9c3, 0x9b9,
0x9b8, 0x9cd, 0x9aa, 0x9a4, 0x9bf, 0x3b, 0x9b6, 0x9c1, 0x995, 0x9cd, 0x9b0, 0x3b,
0x9b6, 0x9a8, 0x9bf, 0x9b0, 0x3b, 0x9b8, 0x9cb, 0x3b, 0x9ae, 0x3b, 0x9ac, 0x9c1,
0x3b, 0x9ac, 0x9c3, 0x3b, 0x9b6, 0x9c1, 0x3b, 0x9b6, 0x14b, 0x67, 0x77, 0xe0,
0x20, 0x6e, 0x254, 0x302, 0x79, 0x3b, 0x14b, 0x67, 0x77, 0xe0, 0x20, 0x6e,
0x6a, 0x61, 0x14b, 0x67, 0x75, 0x6d, 0x62, 0x61, 0x3b, 0x14b, 0x67, 0x77,
0xe0, 0x20, 0xfb, 0x6d, 0x3b, 0x14b, 0x67, 0x77, 0xe0, 0x20, 0x14b, 0x67,
0xea, 0x3b, 0x14b, 0x67, 0x77, 0xe0, 0x20, 0x6d, 0x62, 0x254, 0x6b, 0x3b,
0x14b, 0x67, 0x77, 0xe0, 0x20, 0x6b, 0x254, 0x254, 0x3b, 0x14b, 0x67, 0x77,
0xe0, 0x20, 0x6a, 0xf4, 0x6e, 0x6e, 0x254, 0x79, 0x3b, 0x6e, 0x6a, 0x61,
0x3b, 0x75, 0x75, 0x6d, 0x3b, 0x14b, 0x67, 0x65, 0x3b, 0x6d, 0x62, 0x254,
0x3b, 0x6b, 0x254, 0x254, 0x3b, 0x6a, 0x6f, 0x6e, 0x6e, 0x3b, 0x6e, 0x3b,
0x75, 0x3b, 0x14b, 0x3b, 0x6d, 0x3b, 0x6b, 0x3b, 0x6a, 0x69, 0x67, 0x61,
0x6e, 0x64, 0x65, 0x61, 0x3b, 0x61, 0x73, 0x74, 0x65, 0x6c, 0x65, 0x68,
0x65, 0x6e, 0x61, 0x3b, 0x61, 0x73, 0x74, 0x65, 0x61, 0x72, 0x74, 0x65,
0x61, 0x3b, 0x61, 0x73, 0x74, 0x65, 0x61, 0x7a, 0x6b, 0x65, 0x6e, 0x61,
0x3b, 0x6f, 0x73, 0x74, 0x65, 0x67, 0x75, 0x6e, 0x61, 0x3b, 0x6f, 0x73,
0x74, 0x69, 0x72, 0x61, 0x6c, 0x61, 0x3b, 0x6c, 0x61, 0x72, 0x75, 0x6e,
0x62, 0x61, 0x74, 0x61, 0x69, 0x67, 0x2e, 0x3b, 0x61, 0x6c, 0x2e, 0x3b,
0x61, 0x72, 0x2e, 0x3b, 0x61, 0x7a, 0x2e, 0x3b, 0x6f, 0x67, 0x2e, 0x3b,
0x6f, 0x72, 0x2e, 0x3b, 0x6c, 0x72, 0x2e, 0x49, 0x3b, 0x41, 0x3b, 0x41,
0x3b, 0x41, 0x3b, 0x4f, 0x3b, 0x4f, 0x3b, 0x4c, 0x43d, 0x44f, 0x434, 0x437,
0x435, 0x43b, 0x44f, 0x3b, 0x43f, 0x430, 0x43d, 0x44f, 0x434, 0x437, 0x435, 0x43b,
0x430, 0x43a, 0x3b, 0x430, 0x45e, 0x442, 0x43e, 0x440, 0x430, 0x43a, 0x3b, 0x441,
0x435, 0x440, 0x430, 0x434, 0x430, 0x3b, 0x447, 0x430, 0x446, 0x432, 0x435, 0x440,
0x3b, 0x43f, 0x44f, 0x442, 0x43d, 0x456, 0x446, 0x430, 0x3b, 0x441, 0x443, 0x431,
0x43e, 0x442, 0x430, 0x43d, 0x434, 0x3b, 0x43f, 0x43d, 0x3b, 0x430, 0x45e, 0x3b,
0x441, 0x440, 0x3b, 0x447, 0x446, 0x3b, 0x43f, 0x442, 0x3b, 0x441, 0x431, 0x43d,
0x3b, 0x43f, 0x3b, 0x430, 0x3b, 0x441, 0x3b, 0x447, 0x3b, 0x43f, 0x3b, 0x441,
0x50, 0x61, 0x20, 0x4d, 0x75, 0x6c, 0x75, 0x6e, 0x67, 0x75, 0x3b, 0x50,
0x61, 0x6c, 0x69, 0x63, 0x68, 0x69, 0x6d, 0x6f, 0x3b, 0x50, 0x61, 0x6c,
0x69, 0x63, 0x68, 0x69, 0x62, 0x75, 0x6c, 0x69, 0x3b, 0x50, 0x61, 0x6c,
0x69, 0x63, 0x68, 0x69, 0x74, 0x61, 0x74, 0x75, 0x3b, 0x50, 0x61, 0x6c,
0x69, 0x63, 0x68, 0x69, 0x6e, 0x65, 0x3b, 0x50, 0x61, 0x6c, 0x69, 0x63,
0x68, 0x69, 0x73, 0x61, 0x6e, 0x6f, 0x3b, 0x50, 0x61, 0x63, 0x68, 0x69,
0x62, 0x65, 0x6c, 0x75, 0x73, 0x68, 0x69, 0x70, 0x61, 0x20, 0x6d, 0x75,
0x6c, 0x75, 0x6e, 0x67, 0x75, 0x3b, 0x70, 0x61, 0x20, 0x73, 0x68, 0x61,
0x68, 0x75, 0x76, 0x69, 0x6c, 0x75, 0x68, 0x61, 0x3b, 0x70, 0x61, 0x20,
0x68, 0x69, 0x76, 0x69, 0x6c, 0x69, 0x3b, 0x70, 0x61, 0x20, 0x68, 0x69,
0x64, 0x61, 0x74, 0x75, 0x3b, 0x70, 0x61, 0x20, 0x68, 0x69, 0x74, 0x61,
0x79, 0x69, 0x3b, 0x70, 0x61, 0x20, 0x68, 0x69, 0x68, 0x61, 0x6e, 0x75,
0x3b, 0x70, 0x61, 0x20, 0x73, 0x68, 0x61, 0x68, 0x75, 0x6c, 0x65, 0x6d,
0x62, 0x65, 0x6c, 0x61, 0x4d, 0x75, 0x6c, 0x3b, 0x56, 0x69, 0x6c, 0x3b,
0x48, 0x69, 0x76, 0x3b, 0x48, 0x69, 0x64, 0x3b, 0x48, 0x69, 0x74, 0x3b,
0x48, 0x69, 0x68, 0x3b, 0x4c, 0x65, 0x6d, 0x4d, 0x3b, 0x4a, 0x3b, 0x48,
0x3b, 0x48, 0x3b, 0x48, 0x3b, 0x57, 0x3b, 0x4a, 0x930, 0x92c, 0x940, 0x92c,
0x93e, 0x930, 0x3b, 0x938, 0x94b, 0x92e, 0x92c, 0x93e, 0x930, 0x3b, 0x92e, 0x902,
0x917, 0x932, 0x92c, 0x93e, 0x930, 0x3b, 0x92c, 0x941, 0x927, 0x92c, 0x93e, 0x930,
0x3b, 0x92c, 0x943, 0x939, 0x938, 0x94d, 0x92a, 0x924, 0x93f, 0x92c, 0x93e, 0x930,
0x3b, 0x936, 0x941, 0x915, 0x94d, 0x930, 0x92c, 0x93e, 0x930, 0x3b, 0x938, 0x928,
0x940, 0x91a, 0x930, 0x1230, 0x1295, 0x1260, 0x122d, 0x20, 0x1245, 0x12f3, 0x12c5, 0x3b,
0x1230, 0x1291, 0x3b, 0x1230, 0x120a, 0x131d, 0x3b, 0x1208, 0x1313, 0x20, 0x12c8, 0x122a,
0x20, 0x1208, 0x1265, 0x12cb, 0x3b, 0x12a3, 0x121d, 0x12f5, 0x3b, 0x12a3, 0x122d, 0x1265,
0x3b, 0x1230, 0x1295, 0x1260, 0x122d, 0x20, 0x123d, 0x1313, 0x12c5, 0x1230, 0x2f, 0x1245,
0x3b, 0x1230, 0x1291, 0x3b, 0x1230, 0x120a, 0x131d, 0x3b, 0x1208, 0x1313, 0x3b, 0x12a3,
0x121d, 0x12f5, 0x3b, 0x12a3, 0x122d, 0x1265, 0x3b, 0x1230, 0x2f, 0x123d, 0x1230, 0x3b,
0x1230, 0x3b, 0x1230, 0x3b, 0x1208, 0x3b, 0x12a3, 0x3b, 0x12a3, 0x3b, 0x1230, 0x930,
0x92c, 0x93f, 0x92c, 0x93e, 0x930, 0x3b, 0x938, 0x92e, 0x92c, 0x93e, 0x930, 0x3b,
0x92e, 0x902, 0x917, 0x932, 0x92c, 0x93e, 0x930, 0x3b, 0x92c, 0x941, 0x927, 0x92c,
0x93e, 0x930, 0x3b, 0x92c, 0x93f, 0x938, 0x94d, 0x925, 0x93f, 0x92c, 0x93e, 0x930,
0x3b, 0x938, 0x941, 0x941, 0x916, 0x941, 0x930, 0x92c, 0x93e, 0x930, 0x3b, 0x938,
0x928, 0x93f, 0x92c, 0x93e, 0x930, 0x930, 0x92c, 0x93f, 0x92c, 0x93e, 0x930, 0x3b,
0x938, 0x92e, 0x92c, 0x93e, 0x930, 0x3b, 0x92e, 0x902, 0x917, 0x932, 0x92c, 0x93e,
0x930, 0x3b, 0x92c, 0x941, 0x927, 0x92c, 0x93e, 0x930, 0x3b, 0x92c, 0x93f, 0x938,
0x94d, 0x925, 0x93f, 0x92c, 0x93e, 0x930, 0x3b, 0x938, 0x941, 0x916, 0x941, 0x930,
0x92c, 0x93e, 0x930, 0x3b, 0x938, 0x928, 0x93f, 0x92c, 0x93e, 0x930, 0x930, 0x92c,
0x93f, 0x3b, 0x938, 0x92e, 0x3b, 0x92e, 0x902, 0x917, 0x932, 0x3b, 0x92c, 0x941,
0x927, 0x3b, 0x92c, 0x93f, 0x938, 0x94d, 0x925, 0x93f, 0x3b, 0x938, 0x941, 0x916,
0x941, 0x930, 0x3b, 0x938, 0x928, 0x93f, 0x930, 0x3b, 0x938, 0x3b, 0x92e, 0x902,
0x3b, 0x92c, 0x941, 0x3b, 0x92c, 0x93f, 0x3b, 0x938, 0x941, 0x3b, 0x938, 0x6e,
0x65, 0x64, 0x6a, 0x65, 0x6c, 0x6a, 0x61, 0x3b, 0x70, 0x6f, 0x6e, 0x65,
0x64, 0x6a, 0x65, 0x6c, 0x6a, 0x61, 0x6b, 0x3b, 0x75, 0x74, 0x6f, 0x72,
0x61, 0x6b, 0x3b, 0x73, 0x72, 0x69, 0x6a, 0x65, 0x64, 0x61, 0x3b, 0x10d,
0x65, 0x74, 0x76, 0x72, 0x74, 0x61, 0x6b, 0x3b, 0x70, 0x65, 0x74, 0x61,
0x6b, 0x3b, 0x73, 0x75, 0x62, 0x6f, 0x74, 0x61, 0x6e, 0x65, 0x64, 0x3b,
0x70, 0x6f, 0x6e, 0x3b, 0x75, 0x74, 0x6f, 0x3b, 0x73, 0x72, 0x69, 0x3b,
0x10d, 0x65, 0x74, 0x3b, 0x70, 0x65, 0x74, 0x3b, 0x73, 0x75, 0x62, 0x6e,
0x3b, 0x70, 0x3b, 0x75, 0x3b, 0x73, 0x3b, 0x10d, 0x3b, 0x70, 0x3b, 0x73,
0x4e, 0x3b, 0x50, 0x3b, 0x55, 0x3b, 0x53, 0x3b, 0x10c, 0x3b, 0x50, 0x3b,
0x53, 0x43d, 0x435, 0x434, 0x458, 0x435, 0x459, 0x430, 0x3b, 0x43f, 0x43e, 0x43d,
0x435, 0x434, 0x458, 0x435, 0x459, 0x430, 0x43a, 0x3b, 0x443, 0x442, 0x43e, 0x440,
0x430, 0x43a, 0x3b, 0x441, 0x440, 0x438, 0x458, 0x435, 0x434, 0x430, 0x3b, 0x447,
0x435, 0x442, 0x432, 0x440, 0x442, 0x430, 0x43a, 0x3b, 0x43f, 0x435, 0x442, 0x430,
0x43a, 0x3b, 0x441, 0x443, 0x431, 0x43e, 0x442, 0x430, 0x43d, 0x435, 0x434, 0x3b,
0x43f, 0x43e, 0x43d, 0x3b, 0x443, 0x442, 0x43e, 0x3b, 0x441, 0x440, 0x438, 0x3b,
0x447, 0x435, 0x442, 0x3b, 0x43f, 0x435, 0x442, 0x3b, 0x441, 0x443, 0x431, 0x43d,
0x3b, 0x43f, 0x3b, 0x443, 0x3b, 0x441, 0x3b, 0x447, 0x3b, 0x43f, 0x3b, 0x441,
0x53, 0x75, 0x6c, 0x3b, 0x4c, 0x75, 0x6e, 0x3b, 0x4d, 0x65, 0x75, 0x72,
0x7a, 0x68, 0x3b, 0x4d, 0x65, 0x72, 0x63, 0x2bc, 0x68, 0x65, 0x72, 0x3b,
0x59, 0x61, 0x6f, 0x75, 0x3b, 0x47, 0x77, 0x65, 0x6e, 0x65, 0x72, 0x3b,
0x53, 0x61, 0x64, 0x6f, 0x72, 0x6e, 0x53, 0x75, 0x6c, 0x3b, 0x4c, 0x75,
0x6e, 0x3b, 0x4d, 0x65, 0x75, 0x2e, 0x3b, 0x4d, 0x65, 0x72, 0x2e, 0x3b,
0x59, 0x61, 0x6f, 0x75, 0x3b, 0x47, 0x77, 0x65, 0x2e, 0x3b, 0x53, 0x61,
0x64, 0x2e, 0x53, 0x75, 0x3b, 0x4c, 0x3b, 0x4d, 0x7a, 0x3b, 0x4d, 0x63,
0x3b, 0x59, 0x3b, 0x47, 0x3b, 0x53, 0x61, 0x43d, 0x435, 0x434, 0x435, 0x43b,
0x44f, 0x3b, 0x43f, 0x43e, 0x43d, 0x435, 0x434, 0x435, 0x43b, 0x43d, 0x438, 0x43a,
0x3b, 0x432, 0x442, 0x43e, 0x440, 0x43d, 0x438, 0x43a, 0x3b, 0x441, 0x440, 0x44f,
0x434, 0x430, 0x3b, 0x447, 0x435, 0x442, 0x432, 0x44a, 0x440, 0x442, 0x44a, 0x43a,
0x3b, 0x43f, 0x435, 0x442, 0x44a, 0x43a, 0x3b, 0x441, 0x44a, 0x431, 0x43e, 0x442,
0x430, 0x43d, 0x434, 0x3b, 0x43f, 0x43d, 0x3b, 0x432, 0x442, 0x3b, 0x441, 0x440,
0x3b, 0x447, 0x442, 0x3b, 0x43f, 0x442, 0x3b, 0x441, 0x431, 0x43d, 0x3b, 0x43f,
0x3b, 0x432, 0x3b, 0x441, 0x3b, 0x447, 0x3b, 0x43f, 0x3b, 0x441, 0x1010, 0x1014,
0x1004, 0x103a, 0x1039, 0x1002, 0x1014, 0x103d, 0x1031, 0x3b, 0x1010, 0x1014, 0x1004, 0x103a,
0x1039, 0x101c, 0x102c, 0x3b, 0x1021, 0x1004, 0x103a, 0x1039, 0x1002, 0x102b, 0x3b, 0x1017,
0x102f, 0x1012, 0x1039, 0x1013, 0x101f, 0x1030, 0x1038, 0x3b, 0x1000, 0x103c, 0x102c, 0x101e,
0x1015, 0x1010, 0x1031, 0x1038, 0x3b, 0x101e, 0x1031, 0x102c, 0x1000, 0x103c, 0x102c, 0x3b,
0x1005, 0x1014, 0x1031, 0x1010, 0x3b, 0x1010, 0x3b, 0x1021, 0x3b, 0x1017, 0x3b, 0x1000,
0x3b, 0x101e, 0x3b, 0x1005, 0x661f, 0x671f, 0x65e5, 0x3b, 0x661f, 0x671f, 0x4e00, 0x3b,
0x661f, 0x671f, 0x4e8c, 0x3b, 0x661f, 0x671f, 0x4e09, 0x3b, 0x661f, 0x671f, 0x56db, 0x3b,
0x661f, 0x671f, 0x4e94, 0x3b, 0x661f, 0x671f, 0x516d, 0x65e5, 0x3b, 0x4e00, 0x3b, 0x4e8c,
0x3b, 0x4e09, 0x3b, 0x56db, 0x3b, 0x4e94, 0x3b, 0x516d, 0x5468, 0x65e5, 0x3b, 0x5468,
0x4e00, 0x3b, 0x5468, 0x4e8c, 0x3b, 0x5468, 0x4e09, 0x3b, 0x5468, 0x56db, 0x3b, 0x5468,
0x4e94, 0x3b, 0x5468, 0x516d, 0x64, 0x69, 0x75, 0x6d, 0x65, 0x6e, 0x67, 0x65,
0x3b, 0x64, 0x69, 0x6c, 0x6c, 0x75, 0x6e, 0x73, 0x3b, 0x64, 0x69, 0x6d,
0x61, 0x72, 0x74, 0x73, 0x3b, 0x64, 0x69, 0x6d, 0x65, 0x63, 0x72, 0x65,
0x73, 0x3b, 0x64, 0x69, 0x6a, 0x6f, 0x75, 0x73, 0x3b, 0x64, 0x69, 0x76,
0x65, 0x6e, 0x64, 0x72, 0x65, 0x73, 0x3b, 0x64, 0x69, 0x73, 0x73, 0x61,
0x62, 0x74, 0x65, 0x64, 0x67, 0x2e, 0x3b, 0x64, 0x6c, 0x2e, 0x3b, 0x64,
0x74, 0x2e, 0x3b, 0x64, 0x63, 0x2e, 0x3b, 0x64, 0x6a, 0x2e, 0x3b, 0x64,
0x76, 0x2e, 0x3b, 0x64, 0x73, 0x2e, 0x44, 0x6f, 0x6d, 0x69, 0x6e, 0x67,
0x6f, 0x3b, 0x4c, 0x75, 0x6e, 0x65, 0x73, 0x3b, 0x4d, 0x61, 0x72, 0x74,
0x65, 0x73, 0x3b, 0x4d, 0x69, 0x79, 0x65, 0x72, 0x6b, 0x75, 0x6c, 0x65,
0x73, 0x3b, 0x48, 0x75, 0x77, 0x65, 0x62, 0x65, 0x73, 0x3b, 0x42, 0x69,
0x79, 0x65, 0x72, 0x6e, 0x65, 0x73, 0x3b, 0x53, 0x61, 0x62, 0x61, 0x64,
0x6f, 0x44, 0x6f, 0x6d, 0x3b, 0x4c, 0x75, 0x6e, 0x3b, 0x4d, 0x61, 0x72,
0x3b, 0x4d, 0x69, 0x79, 0x3b, 0x48, 0x75, 0x77, 0x3b, 0x42, 0x69, 0x79,
0x3b, 0x53, 0x61, 0x62, 0x44, 0x3b, 0x4c, 0x3b, 0x4d, 0x3b, 0x4d, 0x3b,
0x48, 0x3b, 0x42, 0x3b, 0x53, 0x41, 0x73, 0x61, 0x6d, 0x61, 0x73, 0x3b,
0x41, 0x79, 0x6e, 0x61, 0x73, 0x3b, 0x41, 0x73, 0x69, 0x6e, 0x61, 0x73,
0x3b, 0x41, 0x6b, 0x72, 0x61, 0x73, 0x3b, 0x41, 0x6b, 0x77, 0x61, 0x73,
0x3b, 0x41, 0x73, 0x69, 0x6d, 0x77, 0x61, 0x73, 0x3b, 0x41, 0x73, 0x69,
0x1e0d, 0x79, 0x61, 0x73, 0x41, 0x73, 0x61, 0x3b, 0x41, 0x79, 0x6e, 0x3b,
0x41, 0x73, 0x6e, 0x3b, 0x41, 0x6b, 0x72, 0x3b, 0x41, 0x6b, 0x77, 0x3b,
0x41, 0x73, 0x6d, 0x3b, 0x41, 0x73, 0x1e0d, 0x41, 0x3b, 0x41, 0x3b, 0x41,
0x3b, 0x41, 0x3b, 0x41, 0x3b, 0x41, 0x3b, 0x41, 0x6cc, 0x6d5, 0x6a9, 0x634,
0x6d5, 0x645, 0x645, 0x6d5, 0x3b, 0x62f, 0x648, 0x648, 0x634, 0x6d5, 0x645, 0x645,
0x6d5, 0x3b, 0x633, 0x6ce, 0x634, 0x6d5, 0x645, 0x645, 0x6d5, 0x3b, 0x686, 0x648,
0x627, 0x631, 0x634, 0x6d5, 0x645, 0x645, 0x6d5, 0x3b, 0x67e, 0x6ce, 0x646, 0x62c,
0x634, 0x6d5, 0x645, 0x645, 0x6d5, 0x3b, 0x6be, 0x6d5, 0x6cc, 0x646, 0x6cc, 0x3b,
0x634, 0x6d5, 0x645, 0x645, 0x6d5, 0x6cc, 0x3b, 0x62f, 0x3b, 0x633, 0x3b, 0x686,
0x3b, 0x67e, 0x3b, 0x6be, 0x3b, 0x634, 0xd804, 0xdd22, 0xd804, 0xdd27, 0xd804, 0xdd1d,
0xd804, 0xdd28, 0xd804, 0xdd1d, 0xd804, 0xdd22, 0xd804, 0xdd34, 0x3b, 0xd804, 0xdd25, 0xd804,
0xdd27, 0xd804, 0xdd1f, 0xd804, 0xdd34, 0xd804, 0xdd1d, 0xd804, 0xdd22, 0xd804, 0xdd34, 0x3b,
0xd804, 0xdd1f, 0xd804, 0xdd27, 0xd804, 0xdd01, 0xd804, 0xdd09, 0xd804, 0xdd27, 0xd804, 0xdd23,
0xd804, 0xdd34, 0xd804, 0xdd1d, 0xd804, 0xdd22, 0xd804, 0xdd34, 0x3b, 0xd804, 0xdd1d, 0xd804,
0xdd2a, 0xd804, 0xdd16, 0xd804, 0xdd34, 0xd804, 0xdd1d, 0xd804, 0xdd22, 0xd804, 0xdd34, 0x3b,
0xd804, 0xdd1d, 0xd804, 0xdd33, 0xd804, 0xdd22, 0xd804, 0xdd28, 0xd804, 0xdd25, 0xd804, 0xdd2a,
0xd804, 0xdd1b, 0xd804, 0xdd34, 0xd804, 0xdd1d, 0xd804, 0xdd22, 0xd804, 0xdd34, 0x3b, 0xd804,
0xdd25, 0xd804, 0xdd2a, 0xd804, 0xdd07, 0xd804, 0xdd34, 0xd804, 0xdd07, 0xd804, 0xdd2e, 0xd804,
0xdd22, 0xd804, 0xdd34, 0xd804, 0xdd1d, 0xd804, 0xdd22, 0xd804, 0xdd34, 0x3b, 0xd804, 0xdd25,
0xd804, 0xdd27, 0xd804, 0xdd1a, 0xd804, 0xdd28, 0xd804, 0xdd1d, 0xd804, 0xdd22, 0xd804, 0xdd34,
0xd804, 0xdd22, 0xd804, 0xdd27, 0xd804, 0xdd1d, 0xd804, 0xdd28, 0x3b, 0xd804, 0xdd25, 0xd804,
0xdd27, 0xd804, 0xdd1f, 0xd804, 0xdd34, 0x3b, 0xd804, 0xdd1f, 0xd804, 0xdd27, 0xd804, 0xdd01,
0xd804, 0xdd09, 0xd804, 0xdd27, 0xd804, 0xdd23, 0xd804, 0xdd34, 0x3b, 0xd804, 0xdd1d, 0xd804,
0xdd2a, 0xd804, 0xdd16, 0xd804, 0xdd34, 0x3b, 0xd804, 0xdd1d, 0xd804, 0xdd33, 0xd804, 0xdd22,
0xd804, 0xdd28, 0xd804, 0xdd25, 0xd804, 0xdd2a, 0xd804, 0xdd1b, 0xd804, 0xdd34, 0x3b, 0xd804,
0xdd25, 0xd804, 0xdd2a, 0xd804, 0xdd07, 0xd804, 0xdd34, 0xd804, 0xdd07, 0xd804, 0xdd2e, 0xd804,
0xdd22, 0xd804, 0xdd34, 0x3b, 0xd804, 0xdd25, 0xd804, 0xdd27, 0xd804, 0xdd1a, 0xd804, 0xdd28,
0xd804, 0xdd22, 0xd804, 0xdd27, 0x3b, 0xd804, 0xdd25, 0xd804, 0xdd27, 0x3b, 0xd804, 0xdd1f,
0xd804, 0xdd27, 0x3b, 0xd804, 0xdd1d, 0xd804, 0xdd2a, 0x3b, 0xd804, 0xdd1d, 0xd804, 0xdd33,
0xd804, 0xdd22, 0xd804, 0xdd28, 0x3b, 0xd804, 0xdd25, 0xd804, 0xdd2a, 0x3b, 0xd804, 0xdd25,
0xd804, 0xdd27, 0x43a, 0x4c0, 0x438, 0x440, 0x430, 0x3b, 0x43e, 0x440, 0x448, 0x43e,
0x442, 0x3b, 0x448, 0x438, 0x43d, 0x430, 0x440, 0x430, 0x3b, 0x43a, 0x445, 0x430,
0x430, 0x440, 0x430, 0x3b, 0x435, 0x430, 0x440, 0x430, 0x3b, 0x43f, 0x4c0, 0x435,
0x440, 0x430, 0x441, 0x43a, 0x430, 0x3b, 0x448, 0x443, 0x43e, 0x442, 0x43a, 0x4c0,
0x438, 0x3b, 0x43e, 0x440, 0x3b, 0x448, 0x438, 0x3b, 0x43a, 0x445, 0x430, 0x3b,
0x435, 0x430, 0x3b, 0x43f, 0x4c0, 0x435, 0x3b, 0x448, 0x443, 0x43e, 0x43a, 0x4c0,
0x3b, 0x43e, 0x3b, 0x448, 0x3b, 0x43a, 0x445, 0x3b, 0x435, 0x3b, 0x43f, 0x4c0,
0x3b, 0x448, 0x13a4, 0x13be, 0x13d9, 0x13d3, 0x13c6, 0x13cd, 0x13ac, 0x3b, 0x13a4, 0x13be,
0x13d9, 0x13d3, 0x13c9, 0x13c5, 0x13af, 0x3b, 0x13d4, 0x13b5, 0x13c1, 0x13a2, 0x13a6, 0x3b,
0x13e6, 0x13a2, 0x13c1, 0x13a2, 0x13a6, 0x3b, 0x13c5, 0x13a9, 0x13c1, 0x13a2, 0x13a6, 0x3b,
0x13e7, 0x13be, 0x13a9, 0x13b6, 0x13cd, 0x13d7, 0x3b, 0x13a4, 0x13be, 0x13d9, 0x13d3, 0x13c8,
0x13d5, 0x13be, 0x13c6, 0x13cd, 0x13ac, 0x3b, 0x13c9, 0x13c5, 0x13af, 0x3b, 0x13d4, 0x13b5,
0x13c1, 0x3b, 0x13e6, 0x13a2, 0x13c1, 0x3b, 0x13c5, 0x13a9, 0x13c1, 0x3b, 0x13e7, 0x13be,
0x13a9, 0x3b, 0x13c8, 0x13d5, 0x13be, 0x13c6, 0x3b, 0x13c9, 0x3b, 0x13d4, 0x3b, 0x13e6,
0x3b, 0x13c5, 0x3b, 0x13e7, 0x3b, 0x13a4, 0x4e, 0x69, 0x74, 0x74, 0x61, 0x6b,
0x20, 0x48, 0x6f, 0x6c, 0x6c, 0x6f, 0x2bc, 0x3b, 0x4d, 0x61, 0x6e, 0x74,
0x69, 0x2bc, 0x3b, 0x43, 0x68, 0x6f, 0x73, 0x74, 0x69, 0x2bc, 0x3b, 0x57,
0x69, 0x6e, 0x73, 0x74, 0x69, 0x2bc, 0x3b, 0x53, 0x6f, 0x69, 0x73, 0x74,
0x69, 0x2bc, 0x3b, 0x4e, 0x61, 0x6e, 0x6e, 0x61, 0x6c, 0x68, 0x63, 0x68,
0x69, 0x66, 0x61, 0x2bc, 0x20, 0x4e, 0x69, 0x74, 0x74, 0x61, 0x6b, 0x3b,
0x4e, 0x69, 0x74, 0x74, 0x61, 0x6b, 0x20, 0x48, 0x6f, 0x6c, 0x6c, 0x6f,
0x2bc, 0x20, 0x4e, 0x61, 0x6b, 0x66, 0x69, 0x73, 0x68, 0x53, 0x61, 0x6e,
0x64, 0x65, 0x3b, 0x4f, 0x72, 0x77, 0x6f, 0x6b, 0x75, 0x62, 0x61, 0x6e,
0x7a, 0x61, 0x3b, 0x4f, 0x72, 0x77, 0x61, 0x6b, 0x61, 0x62, 0x69, 0x72,
0x69, 0x3b, 0x4f, 0x72, 0x77, 0x61, 0x6b, 0x61, 0x73, 0x68, 0x61, 0x74,
0x75, 0x3b, 0x4f, 0x72, 0x77, 0x61, 0x6b, 0x61, 0x6e, 0x61, 0x3b, 0x4f,
0x72, 0x77, 0x61, 0x6b, 0x61, 0x74, 0x61, 0x61, 0x6e, 0x6f, 0x3b, 0x4f,
0x72, 0x77, 0x61, 0x6d, 0x75, 0x6b, 0x61, 0x61, 0x67, 0x61, 0x53, 0x41,
0x4e, 0x3b, 0x4f, 0x52, 0x4b, 0x3b, 0x4f, 0x4b, 0x42, 0x3b, 0x4f, 0x4b,
0x53, 0x3b, 0x4f, 0x4b, 0x4e, 0x3b, 0x4f, 0x4b, 0x54, 0x3b, 0x4f, 0x4d,
0x4b, 0x53, 0x3b, 0x4b, 0x3b, 0x52, 0x3b, 0x53, 0x3b, 0x4e, 0x3b, 0x54,
0x3b, 0x4d, 0x9031, 0x65e5, 0x3b, 0x9031, 0x4e00, 0x3b, 0x9031, 0x4e8c, 0x3b, 0x9031,
0x4e09, 0x3b, 0x9031, 0x56db, 0x3b, 0x9031, 0x4e94, 0x3b, 0x9031, 0x516d, 0x43d, 0x435,
0x434, 0x463, 0x301, 0x43b, 0x467, 0x3b, 0x43f, 0x43e, 0x43d, 0x435, 0x434, 0x463,
0x301, 0x43b, 0x44c, 0x43d, 0x438, 0x43a, 0x44a, 0x3b, 0x432, 0x442, 0x43e, 0x301,
0x440, 0x43d, 0x438, 0x43a, 0x44a, 0x3b, 0x441, 0x440, 0x435, 0x434, 0x430, 0x300,
0x3b, 0x447, 0x435, 0x442, 0x432, 0x435, 0x440, 0x442, 0x43e, 0x301, 0x43a, 0x44a,
0x3b, 0x43f, 0x467, 0x442, 0x43e, 0x301, 0x43a, 0x44a, 0x3b, 0x441, 0xa64b, 0x431,
0x431, 0x461, 0x301, 0x442, 0x430, 0x43d, 0x434, 0x2de7, 0x487, 0x467, 0x3b, 0x43f,
0x43d, 0x2de3, 0x435, 0x3b, 0x432, 0x442, 0x43e, 0x2dec, 0x487, 0x3b, 0x441, 0x440,
0x2de3, 0x435, 0x3b, 0x447, 0x435, 0x2de6, 0x487, 0x3b, 0x43f, 0x467, 0x2de6, 0x487,
0x3b, 0x441, 0xa64b, 0x2de0, 0x487, 0x41d, 0x3b, 0x41f, 0x3b, 0x412, 0x3b, 0x421,
0x3b, 0x427, 0x3b, 0x41f, 0x3b, 0x421, 0x432, 0x44b, 0x440, 0x441, 0x430, 0x440,
0x43d, 0x438, 0x43a, 0x443, 0x43d, 0x3b, 0x442, 0x443, 0x43d, 0x442, 0x438, 0x43a,
0x443, 0x43d, 0x3b, 0x44b, 0x442, 0x43b, 0x430, 0x440, 0x438, 0x43a, 0x443, 0x43d,
0x3b, 0x44e, 0x43d, 0x43a, 0x443, 0x43d, 0x3b, 0x43a, 0x4d7, 0x4ab, 0x43d, 0x435,
0x440, 0x43d, 0x438, 0x43a, 0x443, 0x43d, 0x3b, 0x44d, 0x440, 0x43d, 0x435, 0x43a,
0x443, 0x43d, 0x3b, 0x448, 0x4d1, 0x43c, 0x430, 0x442, 0x43a, 0x443, 0x43d, 0x432,
0x44b, 0x440, 0x2e, 0x3b, 0x442, 0x443, 0x43d, 0x2e, 0x3b, 0x44b, 0x442, 0x43b,
0x2e, 0x3b, 0x44e, 0x43d, 0x2e, 0x3b, 0x43a, 0x4d7, 0x4ab, 0x2e, 0x3b, 0x44d,
0x440, 0x2e, 0x3b, 0x448, 0x4d1, 0x43c, 0x2e, 0x412, 0x3b, 0x422, 0x3b, 0x42b,
0x3b, 0x42e, 0x3b, 0x41a, 0x3b, 0x42d, 0x3b, 0x428, 0x53, 0x75, 0x6e, 0x6e,
0x64, 0x61, 0x61, 0x63, 0x68, 0x3b, 0x4d, 0x6f, 0x68, 0x6e, 0x64, 0x61,
0x61, 0x63, 0x68, 0x3b, 0x44, 0x69, 0x6e, 0x6e, 0x73, 0x64, 0x61, 0x61,
0x63, 0x68, 0x3b, 0x4d, 0x65, 0x74, 0x77, 0x6f, 0x63, 0x68, 0x3b, 0x44,
0x75, 0x6e, 0x6e, 0x65, 0x72, 0x73, 0x64, 0x61, 0x61, 0x63, 0x68, 0x3b,
0x46, 0x72, 0x69, 0x69, 0x64, 0x61, 0x61, 0x63, 0x68, 0x3b, 0x53, 0x61,
0x6d, 0x73, 0x64, 0x61, 0x61, 0x63, 0x68, 0x53, 0x75, 0x2e, 0x3b, 0x4d,
0x6f, 0x2e, 0x3b, 0x44, 0x69, 0x2e, 0x3b, 0x4d, 0x65, 0x2e, 0x3b, 0x44,
0x75, 0x2e, 0x3b, 0x46, 0x72, 0x2e, 0x3b, 0x53, 0x61, 0x2e, 0x53, 0x3b,
0x4d, 0x3b, 0x44, 0x3b, 0x4d, 0x3b, 0x44, 0x3b, 0x46, 0x3b, 0x53, 0x64,
0x79, 0x20, 0x53, 0x75, 0x6c, 0x3b, 0x64, 0x79, 0x20, 0x4c, 0x75, 0x6e,
0x3b, 0x64, 0x79, 0x20, 0x4d, 0x65, 0x75, 0x72, 0x74, 0x68, 0x3b, 0x64,
0x79, 0x20, 0x4d, 0x65, 0x72, 0x68, 0x65, 0x72, 0x3b, 0x64, 0x79, 0x20,
0x59, 0x6f, 0x77, 0x3b, 0x64, 0x79, 0x20, 0x47, 0x77, 0x65, 0x6e, 0x65,
0x72, 0x3b, 0x64, 0x79, 0x20, 0x53, 0x61, 0x64, 0x6f, 0x72, 0x6e, 0x53,
0x75, 0x6c, 0x3b, 0x4c, 0x75, 0x6e, 0x3b, 0x4d, 0x74, 0x68, 0x3b, 0x4d,
0x68, 0x72, 0x3b, 0x59, 0x6f, 0x77, 0x3b, 0x47, 0x77, 0x65, 0x3b, 0x53,
0x61, 0x64, 0x64, 0x75, 0x6d, 0x65, 0x6e, 0x69, 0x63, 0x61, 0x3b, 0x6c,
0x75, 0x6e, 0x69, 0x3b, 0x6d, 0x61, 0x72, 0x74, 0x69, 0x3b, 0x6d, 0x65,
0x72, 0x63, 0x75, 0x72, 0x69, 0x3b, 0x67, 0x68, 0x6a, 0x6f, 0x76, 0x69,
0x3b, 0x76, 0x65, 0x6e, 0x6e, 0x65, 0x72, 0x69, 0x3b, 0x73, 0x61, 0x62,
0x62, 0x61, 0x74, 0x75, 0x64, 0x75, 0x6d, 0x2e, 0x3b, 0x6c, 0x75, 0x6e,
0x2e, 0x3b, 0x6d, 0x61, 0x72, 0x2e, 0x3b, 0x6d, 0x65, 0x72, 0x2e, 0x3b,
0x67, 0x68, 0x6a, 0x2e, 0x3b, 0x76, 0x65, 0x6e, 0x2e, 0x3b, 0x73, 0x61,
0x62, 0x2e, 0x44, 0x3b, 0x4c, 0x3b, 0x4d, 0x3b, 0x4d, 0x3b, 0x47, 0x3b,
0x56, 0x3b, 0x53, 0x6e, 0x65, 0x64, 0x11b, 0x6c, 0x65, 0x3b, 0x70, 0x6f,
0x6e, 0x64, 0x11b, 0x6c, 0xed, 0x3b, 0xfa, 0x74, 0x65, 0x72, 0xfd, 0x3b,
0x73, 0x74, 0x159, 0x65, 0x64, 0x61, 0x3b, 0x10d, 0x74, 0x76, 0x72, 0x74,
0x65, 0x6b, 0x3b, 0x70, 0xe1, 0x74, 0x65, 0x6b, 0x3b, 0x73, 0x6f, 0x62,
0x6f, 0x74, 0x61, 0x6e, 0x65, 0x3b, 0x70, 0x6f, 0x3b, 0xfa, 0x74, 0x3b,
0x73, 0x74, 0x3b, 0x10d, 0x74, 0x3b, 0x70, 0xe1, 0x3b, 0x73, 0x6f, 0x4e,
0x3b, 0x50, 0x3b, 0xda, 0x3b, 0x53, 0x3b, 0x10c, 0x3b, 0x50, 0x3b, 0x53,
0x73, 0xf8, 0x6e, 0x64, 0x61, 0x67, 0x3b, 0x6d, 0x61, 0x6e, 0x64, 0x61,
0x67, 0x3b, 0x74, 0x69, 0x72, 0x73, 0x64, 0x61, 0x67, 0x3b, 0x6f, 0x6e,
0x73, 0x64, 0x61, 0x67, 0x3b, 0x74, 0x6f, 0x72, 0x73, 0x64, 0x61, 0x67,
0x3b, 0x66, 0x72, 0x65, 0x64, 0x61, 0x67, 0x3b, 0x6c, 0xf8, 0x72, 0x64,
0x61, 0x67, 0x73, 0xf8, 0x6e, 0x2e, 0x3b, 0x6d, 0x61, 0x6e, 0x2e, 0x3b,
0x74, 0x69, 0x72, 0x73, 0x2e, 0x3b, 0x6f, 0x6e, 0x73, 0x2e, 0x3b, 0x74,
0x6f, 0x72, 0x73, 0x2e, 0x3b, 0x66, 0x72, 0x65, 0x2e, 0x3b, 0x6c, 0xf8,
0x72, 0x2e, 0x53, 0x3b, 0x4d, 0x3b, 0x54, 0x3b, 0x4f, 0x3b, 0x54, 0x3b,
0x46, 0x3b, 0x4c, 0x910, 0x924, 0x92c, 0x93e, 0x930, 0x3b, 0x938, 0x94b, 0x92e,
0x92c, 0x93e, 0x930, 0x3b, 0x92e, 0x902, 0x917, 0x932, 0x92c, 0x93e, 0x930, 0x3b,
0x92c, 0x941, 0x927, 0x92c, 0x93e, 0x930, 0x3b, 0x92c, 0x940, 0x930, 0x92c, 0x93e,
0x930, 0x3b, 0x936, 0x941, 0x915, 0x94d, 0x930, 0x92c, 0x93e, 0x930, 0x3b, 0x936,
0x928, 0x93f, 0x92c, 0x93e, 0x930, 0x910, 0x924, 0x3b, 0x938, 0x94b, 0x92e, 0x3b,
0x92e, 0x902, 0x917, 0x932, 0x3b, 0x92c, 0x941, 0x927, 0x3b, 0x92c, 0x940, 0x930,
0x3b, 0x936, 0x941, 0x915, 0x94d, 0x930, 0x3b, 0x936, 0x928, 0x93f, 0x910, 0x3b,
0x938, 0x94b, 0x3b, 0x92e, 0x2e, 0x3b, 0x92c, 0x941, 0x2e, 0x3b, 0x92c, 0x940,
0x2e, 0x3b, 0x936, 0x941, 0x2e, 0x3b, 0x936, 0x2e, 0x910, 0x2e, 0x3b, 0x938,
0x94b, 0x2e, 0x3b, 0x92e, 0x2e, 0x3b, 0x92c, 0x941, 0x2e, 0x3b, 0x92c, 0x940,
0x2e, 0x3b, 0x936, 0x941, 0x2e, 0x3b, 0x936, 0x2e, 0xe9, 0x74, 0x69, 0x3b,
0x6d, 0x254, 0x301, 0x73, 0xfa, 0x3b, 0x6b, 0x77, 0x61, 0x73, 0xfa, 0x3b,
0x6d, 0x75, 0x6b, 0x254, 0x301, 0x73, 0xfa, 0x3b, 0x14b, 0x67, 0x69, 0x73,
0xfa, 0x3b, 0x257, 0xf3, 0x6e, 0x25b, 0x73, 0xfa, 0x3b, 0x65, 0x73, 0x61,
0x253, 0x61, 0x73, 0xfa, 0xe9, 0x74, 0x3b, 0x6d, 0x254, 0x301, 0x73, 0x3b,
0x6b, 0x77, 0x61, 0x3b, 0x6d, 0x75, 0x6b, 0x3b, 0x14b, 0x67, 0x69, 0x3b,
0x257, 0xf3, 0x6e, 0x3b, 0x65, 0x73, 0x61, 0x65, 0x3b, 0x6d, 0x3b, 0x6b,
0x3b, 0x6d, 0x3b, 0x14b, 0x3b, 0x257, 0x3b, 0x65, 0x7a, 0x6f, 0x6e, 0x64,
0x61, 0x67, 0x3b, 0x6d, 0x61, 0x61, 0x6e, 0x64, 0x61, 0x67, 0x3b, 0x64,
0x69, 0x6e, 0x73, 0x64, 0x61, 0x67, 0x3b, 0x77, 0x6f, 0x65, 0x6e, 0x73,
0x64, 0x61, 0x67, 0x3b, 0x64, 0x6f, 0x6e, 0x64, 0x65, 0x72, 0x64, 0x61,
0x67, 0x3b, 0x76, 0x72, 0x69, 0x6a, 0x64, 0x61, 0x67, 0x3b, 0x7a, 0x61,
0x74, 0x65, 0x72, 0x64, 0x61, 0x67, 0x7a, 0x6f, 0x3b, 0x6d, 0x61, 0x3b,
0x64, 0x69, 0x3b, 0x77, 0x6f, 0x3b, 0x64, 0x6f, 0x3b, 0x76, 0x72, 0x3b,
0x7a, 0x61, 0x5a, 0x3b, 0x4d, 0x3b, 0x44, 0x3b, 0x57, 0x3b, 0x44, 0x3b,
0x56, 0x3b, 0x5a, 0xf42, 0xf5f, 0xf60, 0xf0b, 0xf5f, 0xfb3, 0xf0b, 0xf56, 0xf0b,
0x3b, 0xf42, 0xf5f, 0xf60, 0xf0b, 0xf58, 0xf72, 0xf42, 0xf0b, 0xf51, 0xf58, 0xf62,
0xf0b, 0x3b, 0xf42, 0xf5f, 0xf60, 0xf0b, 0xf63, 0xfb7, 0xf42, 0xf0b, 0xf54, 0xf0b,
0x3b, 0xf42, 0xf5f, 0xf60, 0xf0b, 0xf55, 0xf74, 0xf62, 0xf0b, 0xf56, 0xf74, 0xf0b,
0x3b, 0xf42, 0xf5f, 0xf60, 0xf0b, 0xf54, 0xf0b, 0xf66, 0xf44, 0xf66, 0xf0b, 0x3b,
0xf42, 0xf5f, 0xf60, 0xf0b, 0xf66, 0xfa4, 0xf7a, 0xf53, 0xf0b, 0xf54, 0xf0b, 0x3b,
0xf42, 0xf5f, 0xf60, 0xf0b, 0xf49, 0xf72, 0xf0b, 0xf58, 0xf0b, 0xf5f, 0xfb3, 0xf0b,
0x3b, 0xf58, 0xf72, 0xf62, 0xf0b, 0x3b, 0xf63, 0xfb7, 0xf42, 0xf0b, 0x3b, 0xf55,
0xf74, 0xf62, 0xf0b, 0x3b, 0xf66, 0xf44, 0xf66, 0xf0b, 0x3b, 0xf66, 0xfa4, 0xf7a,
0xf53, 0xf0b, 0x3b, 0xf49, 0xf72, 0xf0b, 0xf5f, 0xfb3, 0x3b, 0xf58, 0xf72, 0xf62,
0x3b, 0xf63, 0xfb7, 0xf42, 0x3b, 0xf55, 0xf74, 0xf62, 0x3b, 0xf66, 0xf44, 0xfb6,
0x3b, 0xf66, 0xfa4, 0xf7a, 0xf53, 0x3b, 0xf49, 0xf72, 0x4b, 0x69, 0x75, 0x6d,
0x69, 0x61, 0x3b, 0x4e, 0x6a, 0x75, 0x6d, 0x61, 0x74, 0x61, 0x74, 0x75,
0x3b, 0x4e, 0x6a, 0x75, 0x6d, 0x61, 0x69, 0x6e, 0x65, 0x3b, 0x4e, 0x6a,
0x75, 0x6d, 0x61, 0x74, 0x61, 0x6e, 0x6f, 0x3b, 0x41, 0x72, 0x61, 0x6d,
0x69, 0x74, 0x68, 0x69, 0x3b, 0x4e, 0x6a, 0x75, 0x6d, 0x61, 0x61, 0x3b,
0x4e, 0x4a, 0x75, 0x6d, 0x61, 0x6d, 0x6f, 0x74, 0x68, 0x69, 0x69, 0x4b,
0x6d, 0x61, 0x3b, 0x54, 0x61, 0x74, 0x3b, 0x49, 0x6e, 0x65, 0x3b, 0x54,
0x61, 0x6e, 0x3b, 0x41, 0x72, 0x6d, 0x3b, 0x4d, 0x61, 0x61, 0x3b, 0x4e,
0x4d, 0x4d, 0x4b, 0x3b, 0x4e, 0x3b, 0x4e, 0x3b, 0x4e, 0x3b, 0x41, 0x3b,
0x4d, 0x3b, 0x4e, 0xd801, 0xdc1d, 0xd801, 0xdc32, 0xd801, 0xdc4c, 0xd801, 0xdc3c, 0xd801,
0xdc29, 0x3b, 0xd801, 0xdc23, 0xd801, 0xdc32, 0xd801, 0xdc4c, 0xd801, 0xdc3c, 0xd801, 0xdc29,
0x3b, 0xd801, 0xdc13, 0xd801, 0xdc2d, 0xd801, 0xdc46, 0xd801, 0xdc3c, 0xd801, 0xdc29, 0x3b,
0xd801, 0xdc0e, 0xd801, 0xdc2f, 0xd801, 0xdc4c, 0xd801, 0xdc46, 0xd801, 0xdc3c, 0xd801, 0xdc29,
0x3b, 0xd801, 0xdc1b, 0xd801, 0xdc32, 0xd801, 0xdc49, 0xd801, 0xdc46, 0xd801, 0xdc3c, 0xd801,
0xdc29, 0x3b, 0xd801, 0xdc19, 0xd801, 0xdc49, 0xd801, 0xdc34, 0xd801, 0xdc3c, 0xd801, 0xdc29,
0x3b, 0xd801, 0xdc1d, 0xd801, 0xdc30, 0xd801, 0xdc3b, 0xd801, 0xdc32, 0xd801, 0xdc49, 0xd801,
0xdc3c, 0xd801, 0xdc29, 0xd801, 0xdc1d, 0xd801, 0xdc32, 0xd801, 0xdc4c, 0x3b, 0xd801, 0xdc23,
0xd801, 0xdc32, 0xd801, 0xdc4c, 0x3b, 0xd801, 0xdc13, 0xd801, 0xdc2d, 0xd801, 0xdc46, 0x3b,
0xd801, 0xdc0e, 0xd801, 0xdc2f, 0xd801, 0xdc4c, 0x3b, 0xd801, 0xdc1b, 0xd801, 0xdc32, 0xd801,
0xdc49, 0x3b, 0xd801, 0xdc19, 0xd801, 0xdc49, 0xd801, 0xdc34, 0x3b, 0xd801, 0xdc1d, 0xd801,
0xdc30, 0xd801, 0xdc3b, 0xd801, 0xdc1d, 0x3b, 0xd801, 0xdc23, 0x3b, 0xd801, 0xdc13, 0x3b,
0xd801, 0xdc0e, 0x3b, 0xd801, 0xdc1b, 0x3b, 0xd801, 0xdc19, 0x3b, 0xd801, 0xdc1d, 0x53,
0x75, 0x2e, 0x3b, 0x4d, 0x2e, 0x3b, 0x54, 0x75, 0x2e, 0x3b, 0x57, 0x2e,
0x3b, 0x54, 0x68, 0x2e, 0x3b, 0x46, 0x2e, 0x3b, 0x53, 0x61, 0x2e, 0xb7,
0xd801, 0xdc55, 0xd801, 0xdc6d, 0xd801, 0xdc59, 0xd801, 0xdc5b, 0xd801, 0xdc71, 0x3b, 0xb7,
0xd801, 0xdc65, 0xd801, 0xdc6d, 0xd801, 0xdc59, 0xd801, 0xdc5b, 0xd801, 0xdc71, 0x3b, 0xb7,
0xd801, 0xdc51, 0xd801, 0xdc75, 0xd801, 0xdc5f, 0xd801, 0xdc5b, 0xd801, 0xdc71, 0x3b, 0xb7,
0xd801, 0xdc62, 0xd801, 0xdc67, 0xd801, 0xdc59, 0xd801, 0xdc5f, 0xd801, 0xdc5b, 0xd801, 0xdc71,
0x3b, 0xb7, 0xd801, 0xdc54, 0xd801, 0xdc7b, 0xd801, 0xdc5f, 0xd801, 0xdc5b, 0xd801, 0xdc71,
0x3b, 0xb7, 0xd801, 0xdc53, 0xd801, 0xdc6e, 0xd801, 0xdc72, 0xd801, 0xdc5b, 0xd801, 0xdc71,
0x3b, 0xb7, 0xd801, 0xdc55, 0xd801, 0xdc68, 0xd801, 0xdc5b, 0xd801, 0xdc7b, 0xd801, 0xdc5b,
0xd801, 0xdc71, 0xb7, 0xd801, 0xdc55, 0xd801, 0xdc6d, 0x3b, 0xb7, 0xd801, 0xdc65, 0xd801,
0xdc6d, 0x3b, 0xb7, 0xd801, 0xdc51, 0xd801, 0xdc75, 0x3b, 0xb7, 0xd801, 0xdc62, 0xd801,
0xdc67, 0x3b, 0xb7, 0xd801, 0xdc54, 0xd801, 0xdc7b, 0x3b, 0xb7, 0xd801, 0xdc53, 0xd801,
0xdc6e, 0x3b, 0xb7, 0xd801, 0xdc55, 0xd801, 0xdc68, 0xd801, 0xdc55, 0x3b, 0xd801, 0xdc65,
0x3b, 0xd801, 0xdc51, 0x3b, 0xd801, 0xdc62, 0x3b, 0xd801, 0xdc54, 0x3b, 0xd801, 0xdc53,
0x3b, 0xd801, 0xdc55, 0x442, 0x430, 0x440, 0x433, 0x43e, 0x447, 0x438, 0x3b, 0x430,
0x442, 0x44f, 0x43d, 0x44c, 0x447, 0x438, 0x3b, 0x432, 0x430, 0x441, 0x442, 0x430,
0x43d, 0x44c, 0x447, 0x438, 0x3b, 0x43a, 0x443, 0x43d, 0x448, 0x43a, 0x430, 0x447,
0x438, 0x3b, 0x43a, 0x430, 0x43b, 0x43e, 0x43d, 0x44c, 0x447, 0x438, 0x3b, 0x441,
0x44e, 0x43a, 0x43e, 0x43d, 0x44c, 0x447, 0x438, 0x3b, 0x448, 0x43b, 0x44f, 0x43c,
0x43e, 0x447, 0x438, 0x442, 0x430, 0x440, 0x433, 0x43e, 0x447, 0x438, 0x441, 0x442,
0x44d, 0x3b, 0x430, 0x442, 0x44f, 0x43d, 0x44c, 0x447, 0x438, 0x441, 0x442, 0x44d,
0x3b, 0x432, 0x430, 0x441, 0x442, 0x430, 0x43d, 0x44c, 0x447, 0x438, 0x441, 0x442,
0x44d, 0x3b, 0x43a, 0x443, 0x43d, 0x448, 0x43a, 0x430, 0x447, 0x438, 0x441, 0x442,
0x44d, 0x3b, 0x43a, 0x430, 0x43b, 0x43e, 0x43d, 0x44c, 0x447, 0x438, 0x441, 0x442,
0x44d, 0x3b, 0x441, 0x44e, 0x43a, 0x43e, 0x43d, 0x44c, 0x447, 0x438, 0x441, 0x442,
0x44d, 0x3b, 0x448, 0x43b, 0x44f, 0x43c, 0x43e, 0x447, 0x438, 0x441, 0x442, 0x44d,
0x442, 0x430, 0x440, 0x3b, 0x430, 0x442, 0x44f, 0x3b, 0x432, 0x430, 0x441, 0x3b,
0x43a, 0x443, 0x43d, 0x3b, 0x43a, 0x430, 0x43b, 0x3b, 0x441, 0x44e, 0x43a, 0x3b,
0x448, 0x43b, 0x44f, 0x64, 0x69, 0x6d, 0x61, 0x6e, 0x109, 0x6f, 0x3b, 0x6c,
0x75, 0x6e, 0x64, 0x6f, 0x3b, 0x6d, 0x61, 0x72, 0x64, 0x6f, 0x3b, 0x6d,
0x65, 0x72, 0x6b, 0x72, 0x65, 0x64, 0x6f, 0x3b, 0x135, 0x61, 0x16d, 0x64,
0x6f, 0x3b, 0x76, 0x65, 0x6e, 0x64, 0x72, 0x65, 0x64, 0x6f, 0x3b, 0x73,
0x61, 0x62, 0x61, 0x74, 0x6f, 0x64, 0x69, 0x3b, 0x6c, 0x75, 0x3b, 0x6d,
0x61, 0x3b, 0x6d, 0x65, 0x3b, 0x135, 0x61, 0x3b, 0x76, 0x65, 0x3b, 0x73,
0x61, 0x64, 0x3b, 0x6c, 0x3b, 0x6d, 0x3b, 0x6d, 0x3b, 0x135, 0x3b, 0x76,
0x3b, 0x73, 0x70, 0xfc, 0x68, 0x61, 0x70, 0xe4, 0x65, 0x76, 0x3b, 0x65,
0x73, 0x6d, 0x61, 0x73, 0x70, 0xe4, 0x65, 0x76, 0x3b, 0x74, 0x65, 0x69,
0x73, 0x69, 0x70, 0xe4, 0x65, 0x76, 0x3b, 0x6b, 0x6f, 0x6c, 0x6d, 0x61,
0x70, 0xe4, 0x65, 0x76, 0x3b, 0x6e, 0x65, 0x6c, 0x6a, 0x61, 0x70, 0xe4,
0x65, 0x76, 0x3b, 0x72, 0x65, 0x65, 0x64, 0x65, 0x3b, 0x6c, 0x61, 0x75,
0x70, 0xe4, 0x65, 0x76, 0x50, 0x3b, 0x45, 0x3b, 0x54, 0x3b, 0x4b, 0x3b,
0x4e, 0x3b, 0x52, 0x3b, 0x4c, 0x6b, 0x254, 0x73, 0x69, 0x256, 0x61, 0x3b,
0x64, 0x7a, 0x6f, 0x256, 0x61, 0x3b, 0x62, 0x6c, 0x61, 0x256, 0x61, 0x3b,
0x6b, 0x75, 0x256, 0x61, 0x3b, 0x79, 0x61, 0x77, 0x6f, 0x256, 0x61, 0x3b,
0x66, 0x69, 0x256, 0x61, 0x3b, 0x6d, 0x65, 0x6d, 0x6c, 0x65, 0x256, 0x61,
0x6b, 0x254, 0x73, 0x3b, 0x64, 0x7a, 0x6f, 0x3b, 0x62, 0x6c, 0x61, 0x3b,
0x6b, 0x75, 0x256, 0x3b, 0x79, 0x61, 0x77, 0x3b, 0x66, 0x69, 0x256, 0x3b,
0x6d, 0x65, 0x6d, 0x6b, 0x3b, 0x64, 0x3b, 0x62, 0x3b, 0x6b, 0x3b, 0x79,
0x3b, 0x66, 0x3b, 0x6d, 0x73, 0x254, 0x301, 0x6e, 0x64, 0x254, 0x3b, 0x6d,
0x254, 0x301, 0x6e, 0x64, 0x69, 0x3b, 0x73, 0x254, 0x301, 0x6e, 0x64, 0x254,
0x20, 0x6d, 0x259, 0x6c, 0xfa, 0x20, 0x6d, 0x259, 0x301, 0x62, 0x25b, 0x30c,
0x3b, 0x73, 0x254, 0x301, 0x6e, 0x64, 0x254, 0x20, 0x6d, 0x259, 0x6c, 0xfa,
0x20, 0x6d, 0x259, 0x301, 0x6c, 0x25b, 0x301, 0x3b, 0x73, 0x254, 0x301, 0x6e,
0x64, 0x254, 0x20, 0x6d, 0x259, 0x6c, 0xfa, 0x20, 0x6d, 0x259, 0x301, 0x6e,
0x79, 0x69, 0x3b, 0x66, 0xfa, 0x6c, 0x61, 0x64, 0xe9, 0x3b, 0x73, 0xe9,
0x72, 0x61, 0x64, 0xe9, 0x73, 0x254, 0x301, 0x6e, 0x3b, 0x6d, 0x254, 0x301,
0x6e, 0x3b, 0x73, 0x6d, 0x62, 0x3b, 0x73, 0x6d, 0x6c, 0x3b, 0x73, 0x6d,
0x6e, 0x3b, 0x66, 0xfa, 0x6c, 0x3b, 0x73, 0xe9, 0x72, 0x73, 0x3b, 0x6d,
0x3b, 0x73, 0x3b, 0x73, 0x3b, 0x73, 0x3b, 0x66, 0x3b, 0x73, 0x73, 0x75,
0x6e, 0x6e, 0x75, 0x64, 0x61, 0x67, 0x75, 0x72, 0x3b, 0x6d, 0xe1, 0x6e,
0x61, 0x64, 0x61, 0x67, 0x75, 0x72, 0x3b, 0x74, 0xfd, 0x73, 0x64, 0x61,
0x67, 0x75, 0x72, 0x3b, 0x6d, 0x69, 0x6b, 0x75, 0x64, 0x61, 0x67, 0x75,
0x72, 0x3b, 0x68, 0xf3, 0x73, 0x64, 0x61, 0x67, 0x75, 0x72, 0x3b, 0x66,
0x72, 0xed, 0x67, 0x67, 0x6a, 0x61, 0x64, 0x61, 0x67, 0x75, 0x72, 0x3b,
0x6c, 0x65, 0x79, 0x67, 0x61, 0x72, 0x64, 0x61, 0x67, 0x75, 0x72, 0x73,
0x75, 0x6e, 0x3b, 0x6d, 0xe1, 0x6e, 0x3b, 0x74, 0xfd, 0x73, 0x3b, 0x6d,
0x69, 0x6b, 0x3b, 0x68, 0xf3, 0x73, 0x3b, 0x66, 0x72, 0xed, 0x3b, 0x6c,
0x65, 0x79, 0x73, 0x75, 0x6e, 0x2e, 0x3b, 0x6d, 0xe1, 0x6e, 0x2e, 0x3b,
0x74, 0xfd, 0x73, 0x2e, 0x3b, 0x6d, 0x69, 0x6b, 0x2e, 0x3b, 0x68, 0xf3,
0x73, 0x2e, 0x3b, 0x66, 0x72, 0xed, 0x2e, 0x3b, 0x6c, 0x65, 0x79, 0x2e,
0x53, 0x3b, 0x4d, 0x3b, 0x54, 0x3b, 0x4d, 0x3b, 0x48, 0x3b, 0x46, 0x3b,
0x4c, 0x4c, 0x69, 0x6e, 0x67, 0x67, 0x6f, 0x3b, 0x4c, 0x75, 0x6e, 0x65,
0x73, 0x3b, 0x4d, 0x61, 0x72, 0x74, 0x65, 0x73, 0x3b, 0x4d, 0x69, 0x79,
0x65, 0x72, 0x6b, 0x75, 0x6c, 0x65, 0x73, 0x3b, 0x48, 0x75, 0x77, 0x65,
0x62, 0x65, 0x73, 0x3b, 0x42, 0x69, 0x79, 0x65, 0x72, 0x6e, 0x65, 0x73,
0x3b, 0x53, 0x61, 0x62, 0x61, 0x64, 0x6f, 0x4c, 0x69, 0x6e, 0x3b, 0x4c,
0x75, 0x6e, 0x3b, 0x4d, 0x61, 0x72, 0x3b, 0x4d, 0x69, 0x79, 0x3b, 0x48,
0x75, 0x77, 0x3b, 0x42, 0x69, 0x79, 0x3b, 0x53, 0x61, 0x62, 0x73, 0x75,
0x6e, 0x6e, 0x75, 0x6e, 0x74, 0x61, 0x69, 0x3b, 0x6d, 0x61, 0x61, 0x6e,
0x61, 0x6e, 0x74, 0x61, 0x69, 0x3b, 0x74, 0x69, 0x69, 0x73, 0x74, 0x61,
0x69, 0x3b, 0x6b, 0x65, 0x73, 0x6b, 0x69, 0x76, 0x69, 0x69, 0x6b, 0x6b,
0x6f, 0x3b, 0x74, 0x6f, 0x72, 0x73, 0x74, 0x61, 0x69, 0x3b, 0x70, 0x65,
0x72, 0x6a, 0x61, 0x6e, 0x74, 0x61, 0x69, 0x3b, 0x6c, 0x61, 0x75, 0x61,
0x6e, 0x74, 0x61, 0x69, 0x73, 0x75, 0x6e, 0x6e, 0x75, 0x6e, 0x74, 0x61,
0x69, 0x6e, 0x61, 0x3b, 0x6d, 0x61, 0x61, 0x6e, 0x61, 0x6e, 0x74, 0x61,
0x69, 0x6e, 0x61, 0x3b, 0x74, 0x69, 0x69, 0x73, 0x74, 0x61, 0x69, 0x6e,
0x61, 0x3b, 0x6b, 0x65, 0x73, 0x6b, 0x69, 0x76, 0x69, 0x69, 0x6b, 0x6b,
0x6f, 0x6e, 0x61, 0x3b, 0x74, 0x6f, 0x72, 0x73, 0x74, 0x61, 0x69, 0x6e,
0x61, 0x3b, 0x70, 0x65, 0x72, 0x6a, 0x61, 0x6e, 0x74, 0x61, 0x69, 0x6e,
0x61, 0x3b, 0x6c, 0x61, 0x75, 0x61, 0x6e, 0x74, 0x61, 0x69, 0x6e, 0x61,
0x73, 0x75, 0x3b, 0x6d, 0x61, 0x3b, 0x74, 0x69, 0x3b, 0x6b, 0x65, 0x3b,
0x74, 0x6f, 0x3b, 0x70, 0x65, 0x3b, 0x6c, 0x61, 0x53, 0x3b, 0x4d, 0x3b,
0x54, 0x3b, 0x4b, 0x3b, 0x54, 0x3b, 0x50, 0x3b, 0x4c, 0x64, 0x69, 0x6d,
0x61, 0x6e, 0x63, 0x68, 0x65, 0x3b, 0x6c, 0x75, 0x6e, 0x64, 0x69, 0x3b,
0x6d, 0x61, 0x72, 0x64, 0x69, 0x3b, 0x6d, 0x65, 0x72, 0x63, 0x72, 0x65,
0x64, 0x69, 0x3b, 0x6a, 0x65, 0x75, 0x64, 0x69, 0x3b, 0x76, 0x65, 0x6e,
0x64, 0x72, 0x65, 0x64, 0x69, 0x3b, 0x73, 0x61, 0x6d, 0x65, 0x64, 0x69,
0x64, 0x69, 0x6d, 0x2e, 0x3b, 0x6c, 0x75, 0x6e, 0x2e, 0x3b, 0x6d, 0x61,
0x72, 0x2e, 0x3b, 0x6d, 0x65, 0x72, 0x2e, 0x3b, 0x6a, 0x65, 0x75, 0x2e,
0x3b, 0x76, 0x65, 0x6e, 0x2e, 0x3b, 0x73, 0x61, 0x6d, 0x2e, 0x44, 0x3b,
0x4c, 0x3b, 0x4d, 0x3b, 0x4d, 0x3b, 0x4a, 0x3b, 0x56, 0x3b, 0x53, 0x64,
0x6f, 0x6d, 0x65, 0x6e, 0x69, 0x65, 0x3b, 0x6c, 0x75, 0x6e, 0x69, 0x73,
0x3b, 0x6d, 0x61, 0x72, 0x74, 0x61, 0x72, 0x73, 0x3b, 0x6d, 0x69, 0x65,
0x72, 0x63, 0x75, 0x73, 0x3b, 0x6a, 0x6f, 0x69, 0x62, 0x65, 0x3b, 0x76,
0x69, 0x6e, 0x61, 0x72, 0x73, 0x3b, 0x73, 0x61, 0x62, 0x69, 0x64, 0x65,
0x64, 0x6f, 0x6d, 0x3b, 0x6c, 0x75, 0x6e, 0x3b, 0x6d, 0x61, 0x72, 0x3b,
0x6d, 0x69, 0x65, 0x3b, 0x6a, 0x6f, 0x69, 0x3b, 0x76, 0x69, 0x6e, 0x3b,
0x73, 0x61, 0x62, 0x64, 0x65, 0x77, 0x6f, 0x3b, 0x61, 0x61, 0x253, 0x6e,
0x64, 0x65, 0x3b, 0x6d, 0x61, 0x77, 0x62, 0x61, 0x61, 0x72, 0x65, 0x3b,
0x6e, 0x6a, 0x65, 0x73, 0x6c, 0x61, 0x61, 0x72, 0x65, 0x3b, 0x6e, 0x61,
0x61, 0x73, 0x61, 0x61, 0x6e, 0x64, 0x65, 0x3b, 0x6d, 0x61, 0x77, 0x6e,
0x64, 0x65, 0x3b, 0x68, 0x6f, 0x6f, 0x72, 0x65, 0x2d, 0x62, 0x69, 0x69,
0x72, 0x64, 0x65, 0x77, 0x3b, 0x61, 0x61, 0x253, 0x3b, 0x6d, 0x61, 0x77,
0x3b, 0x6e, 0x6a, 0x65, 0x3b, 0x6e, 0x61, 0x61, 0x3b, 0x6d, 0x77, 0x64,
0x3b, 0x68, 0x62, 0x69, 0x64, 0x3b, 0x61, 0x3b, 0x6d, 0x3b, 0x6e, 0x3b,
0x6e, 0x3b, 0x6d, 0x3b, 0x68, 0xd83a, 0xdd08, 0xd83a, 0xdd2b, 0xd83a, 0xdd2c, 0xd83a,
0xdd26, 0xd83a, 0xdd2d, 0xd83a, 0xdd2a, 0xd83a, 0xdd46, 0xd83a, 0xdd2b, 0x3b, 0xd83a, 0xdd00,
0xd83a, 0xdd44, 0xd83a, 0xdd29, 0xd83a, 0xdd35, 0xd83a, 0xdd32, 0xd83a, 0xdd4b, 0xd83a, 0xdd23,
0xd83a, 0xdd2b, 0x3b, 0xd83a, 0xdd03, 0xd83a, 0xdd22, 0xd83a, 0xdd31, 0xd83a, 0xdd26, 0xd83a,
0xdd22, 0xd83a, 0xdd44, 0xd83a, 0xdd2a, 0xd83a, 0xdd2b, 0x3b, 0xd83a, 0xdd10, 0xd83a, 0xdd36,
0xd83a, 0xdd2b, 0xd83a, 0xdd27, 0xd83a, 0xdd24, 0xd83a, 0xdd22, 0xd83a, 0xdd44, 0xd83a, 0xdd2a,
0xd83a, 0xdd2b, 0x3b, 0xd83a, 0xdd10, 0xd83a, 0xdd22, 0xd83a, 0xdd44, 0xd83a, 0xdd27, 0xd83a,
0xdd22, 0xd83a, 0xdd44, 0xd83a, 0xdd32, 0xd83a, 0xdd23, 0xd83a, 0xdd2b, 0x3b, 0xd83a, 0xdd03,
0xd83a, 0xdd22, 0xd83a, 0xdd31, 0xd83a, 0xdd32, 0xd83a, 0xdd23, 0xd83a, 0xdd2b, 0x3b, 0xd83a,
0xdd16, 0xd83a, 0xdd2e, 0xd83a, 0xdd2a, 0xd83a, 0xdd26, 0xd83a, 0xdd2d, 0xd83a, 0xdd2a, 0xd83a,
0xdd46, 0xd83a, 0xdd2b, 0xd83a, 0xdd08, 0xd83a, 0xdd2b, 0xd83a, 0xdd2c, 0x3b, 0xd83a, 0xdd00,
0xd83a, 0xdd44, 0xd83a, 0xdd29, 0xd83a, 0xdd35, 0x3b, 0xd83a, 0xdd03, 0xd83a, 0xdd22, 0xd83a,
0xdd26, 0x3b, 0xd83a, 0xdd14, 0xd83a, 0xdd2b, 0xd83a, 0xdd27, 0x3b, 0xd83a, 0xdd10, 0xd83a,
0xdd22, 0xd83a, 0xdd44, 0xd83a, 0xdd27, 0x3b, 0xd83a, 0xdd03, 0xd83a, 0xdd22, 0xd83a, 0xdd23,
0x3b, 0xd83a, 0xdd16, 0xd83a, 0xdd2e, 0xd83a, 0xdd2a, 0xd83a, 0xdd08, 0x3b, 0xd83a, 0xdd00,
0xd83a, 0xdd44, 0x3b, 0xd83a, 0xdd03, 0x3b, 0xd83a, 0xdd14, 0x3b, 0xd83a, 0xdd10, 0x3b,
0xd83a, 0xdd03, 0x3b, 0xd83a, 0xdd16, 0x44, 0x69, 0x44, 0xf2, 0x6d, 0x68, 0x6e,
0x61, 0x69, 0x63, 0x68, 0x3b, 0x44, 0x69, 0x4c, 0x75, 0x61, 0x69, 0x6e,
0x3b, 0x44, 0x69, 0x4d, 0xe0, 0x69, 0x72, 0x74, 0x3b, 0x44, 0x69, 0x43,
0x69, 0x61, 0x64, 0x61, 0x69, 0x6e, 0x3b, 0x44, 0x69, 0x61, 0x72, 0x44,
0x61, 0x6f, 0x69, 0x6e, 0x3b, 0x44, 0x69, 0x68, 0x41, 0x6f, 0x69, 0x6e,
0x65, 0x3b, 0x44, 0x69, 0x53, 0x61, 0x74, 0x68, 0x61, 0x69, 0x72, 0x6e,
0x65, 0x44, 0x69, 0x44, 0x3b, 0x44, 0x69, 0x4c, 0x3b, 0x44, 0x69, 0x4d,
0x3b, 0x44, 0x69, 0x43, 0x3b, 0x44, 0x69, 0x61, 0x3b, 0x44, 0x69, 0x68,
0x3b, 0x44, 0x69, 0x53, 0x44, 0x3b, 0x4c, 0x3b, 0x4d, 0x3b, 0x43, 0x3b,
0x41, 0x3b, 0x48, 0x3b, 0x53, 0x48, 0x254, 0x67, 0x62, 0x61, 0x61, 0x3b,
0x4a, 0x75, 0x3b, 0x4a, 0x75, 0x66, 0x254, 0x3b, 0x53, 0x68, 0x254, 0x3b,
0x53, 0x6f, 0x6f, 0x3b, 0x53, 0x6f, 0x68, 0x61, 0x61, 0x3b, 0x48, 0x254,
0x254, 0x48, 0x254, 0x67, 0x3b, 0x4a, 0x75, 0x3b, 0x4a, 0x75, 0x66, 0x3b,
0x53, 0x68, 0x254, 0x3b, 0x53, 0x6f, 0x6f, 0x3b, 0x53, 0x6f, 0x68, 0x3b,
0x48, 0x254, 0x254, 0x48, 0x3b, 0x4a, 0x3b, 0x4a, 0x3b, 0x53, 0x3b, 0x53,
0x3b, 0x53, 0x3b, 0x48, 0x64, 0x6f, 0x6d, 0x69, 0x6e, 0x67, 0x6f, 0x3b,
0x6c, 0x75, 0x6e, 0x73, 0x3b, 0x6d, 0x61, 0x72, 0x74, 0x65, 0x73, 0x3b,
0x6d, 0xe9, 0x72, 0x63, 0x6f, 0x72, 0x65, 0x73, 0x3b, 0x78, 0x6f, 0x76,
0x65, 0x73, 0x3b, 0x76, 0x65, 0x6e, 0x72, 0x65, 0x73, 0x3b, 0x73, 0xe1,
0x62, 0x61, 0x64, 0x6f, 0x64, 0x6f, 0x6d, 0x2e, 0x3b, 0x6c, 0x75, 0x6e,
0x73, 0x3b, 0x6d, 0x61, 0x72, 0x2e, 0x3b, 0x6d, 0xe9, 0x72, 0x2e, 0x3b,
0x78, 0x6f, 0x76, 0x2e, 0x3b, 0x76, 0x65, 0x6e, 0x2e, 0x3b, 0x73, 0xe1,
0x62, 0x2e, 0x64, 0x2e, 0x3b, 0x6c, 0x2e, 0x3b, 0x6d, 0x2e, 0x3b, 0x6d,
0x2e, 0x3b, 0x78, 0x2e, 0x3b, 0x76, 0x2e, 0x3b, 0x73, 0x2e, 0x53, 0x61,
0x62, 0x62, 0x69, 0x69, 0x74, 0x69, 0x3b, 0x42, 0x61, 0x6c, 0x61, 0x7a,
0x61, 0x3b, 0x4c, 0x77, 0x61, 0x6b, 0x75, 0x62, 0x69, 0x72, 0x69, 0x3b,
0x4c, 0x77, 0x61, 0x6b, 0x75, 0x73, 0x61, 0x74, 0x75, 0x3b, 0x4c, 0x77,
0x61, 0x6b, 0x75, 0x6e, 0x61, 0x3b, 0x4c, 0x77, 0x61, 0x6b, 0x75, 0x74,
0x61, 0x61, 0x6e, 0x6f, 0x3b, 0x4c, 0x77, 0x61, 0x6d, 0x75, 0x6b, 0x61,
0x61, 0x67, 0x61, 0x53, 0x61, 0x62, 0x3b, 0x42, 0x61, 0x6c, 0x3b, 0x4c,
0x77, 0x32, 0x3b, 0x4c, 0x77, 0x33, 0x3b, 0x4c, 0x77, 0x34, 0x3b, 0x4c,
0x77, 0x35, 0x3b, 0x4c, 0x77, 0x36, 0x53, 0x3b, 0x42, 0x3b, 0x4c, 0x3b,
0x4c, 0x3b, 0x4c, 0x3b, 0x4c, 0x3b, 0x4c, 0x12a5, 0x1281, 0x12f5, 0x3b, 0x1230,
0x1291, 0x12ed, 0x3b, 0x1220, 0x1209, 0x1235, 0x3b, 0x122b, 0x1265, 0x12d5, 0x3b, 0x1210,
0x1219, 0x1235, 0x3b, 0x12d3, 0x122d, 0x1260, 0x3b, 0x1240, 0x12f3, 0x121a, 0x1275, 0x12a5,
0x3b, 0x1230, 0x3b, 0x1220, 0x3b, 0x122b, 0x3b, 0x1210, 0x3b, 0x12d3, 0x3b, 0x1240,
0x10d9, 0x10d5, 0x10d8, 0x10e0, 0x10d0, 0x3b, 0x10dd, 0x10e0, 0x10e8, 0x10d0, 0x10d1, 0x10d0,
0x10d7, 0x10d8, 0x3b, 0x10e1, 0x10d0, 0x10db, 0x10e8, 0x10d0, 0x10d1, 0x10d0, 0x10d7, 0x10d8,
0x3b, 0x10dd, 0x10d7, 0x10ee, 0x10e8, 0x10d0, 0x10d1, 0x10d0, 0x10d7, 0x10d8, 0x3b, 0x10ee,
0x10e3, 0x10d7, 0x10e8, 0x10d0, 0x10d1, 0x10d0, 0x10d7, 0x10d8, 0x3b, 0x10de, 0x10d0, 0x10e0,
0x10d0, 0x10e1, 0x10d9, 0x10d4, 0x10d5, 0x10d8, 0x3b, 0x10e8, 0x10d0, 0x10d1, 0x10d0, 0x10d7,
0x10d8, 0x10d9, 0x10d5, 0x10d8, 0x3b, 0x10dd, 0x10e0, 0x10e8, 0x3b, 0x10e1, 0x10d0, 0x10db,
0x3b, 0x10dd, 0x10d7, 0x10ee, 0x3b, 0x10ee, 0x10e3, 0x10d7, 0x3b, 0x10de, 0x10d0, 0x10e0,
0x3b, 0x10e8, 0x10d0, 0x10d1, 0x10d9, 0x3b, 0x10dd, 0x3b, 0x10e1, 0x3b, 0x10dd, 0x3b,
0x10ee, 0x3b, 0x10de, 0x3b, 0x10e8, 0x53, 0x6f, 0x6e, 0x6e, 0x74, 0x61, 0x67,
0x3b, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x67, 0x3b, 0x44, 0x69, 0x65, 0x6e,
0x73, 0x74, 0x61, 0x67, 0x3b, 0x4d, 0x69, 0x74, 0x74, 0x77, 0x6f, 0x63,
0x68, 0x3b, 0x44, 0x6f, 0x6e, 0x6e, 0x65, 0x72, 0x73, 0x74, 0x61, 0x67,
0x3b, 0x46, 0x72, 0x65, 0x69, 0x74, 0x61, 0x67, 0x3b, 0x53, 0x61, 0x6d,
0x73, 0x74, 0x61, 0x67, 0x53, 0x6f, 0x3b, 0x4d, 0x6f, 0x3b, 0x44, 0x69,
0x3b, 0x4d, 0x69, 0x3b, 0x44, 0x6f, 0x3b, 0x46, 0x72, 0x3b, 0x53, 0x61,
0x53, 0x6f, 0x2e, 0x3b, 0x4d, 0x6f, 0x2e, 0x3b, 0x44, 0x69, 0x2e, 0x3b,
0x4d, 0x69, 0x2e, 0x3b, 0x44, 0x6f, 0x2e, 0x3b, 0x46, 0x72, 0x2e, 0x3b,
0x53, 0x61, 0x2e, 0x39a, 0x3c5, 0x3c1, 0x3b9, 0x3b1, 0x3ba, 0x3ae, 0x3b, 0x394,
0x3b5, 0x3c5, 0x3c4, 0x3ad, 0x3c1, 0x3b1, 0x3b, 0x3a4, 0x3c1, 0x3af, 0x3c4, 0x3b7,
0x3b, 0x3a4, 0x3b5, 0x3c4, 0x3ac, 0x3c1, 0x3c4, 0x3b7, 0x3b, 0x3a0, 0x3ad, 0x3bc,
0x3c0, 0x3c4, 0x3b7, 0x3b, 0x3a0, 0x3b1, 0x3c1, 0x3b1, 0x3c3, 0x3ba, 0x3b5, 0x3c5,
0x3ae, 0x3b, 0x3a3, 0x3ac, 0x3b2, 0x3b2, 0x3b1, 0x3c4, 0x3bf, 0x39a, 0x3c5, 0x3c1,
0x3b, 0x394, 0x3b5, 0x3c5, 0x3b, 0x3a4, 0x3c1, 0x3af, 0x3b, 0x3a4, 0x3b5, 0x3c4,
0x3b, 0x3a0, 0x3ad, 0x3bc, 0x3b, 0x3a0, 0x3b1, 0x3c1, 0x3b, 0x3a3, 0x3ac, 0x3b2,
0x39a, 0x3b, 0x394, 0x3b, 0x3a4, 0x3b, 0x3a4, 0x3b, 0x3a0, 0x3b, 0x3a0, 0x3b,
0x3a3, 0x41, 0x72, 0x61, 0x74, 0x65, 0x129, 0x3b, 0x41, 0x72, 0x61, 0x6b,
0xf5, 0x69, 0x3b, 0x41, 0x72, 0x61, 0x61, 0x70, 0x79, 0x3b, 0x41, 0x72,
0x61, 0x72, 0x75, 0x6e, 0x64, 0x79, 0x3b, 0x41, 0x72, 0x61, 0x70, 0x6f,
0x3b, 0x41, 0x72, 0x61, 0x70, 0x6f, 0x74, 0x65, 0x129, 0x3b, 0x41, 0x72,
0x61, 0x70, 0x6f, 0x6b, 0xf5, 0x69, 0xab0, 0xab5, 0xabf, 0xab5, 0xabe, 0xab0,
0x3b, 0xab8, 0xacb, 0xaae, 0xab5, 0xabe, 0xab0, 0x3b, 0xaae, 0xa82, 0xa97, 0xab3,
0xab5, 0xabe, 0xab0, 0x3b, 0xaac, 0xac1, 0xaa7, 0xab5, 0xabe, 0xab0, 0x3b, 0xa97,
0xac1, 0xab0, 0xac1, 0xab5, 0xabe, 0xab0, 0x3b, 0xab6, 0xac1, 0xa95, 0xacd, 0xab0,
0xab5, 0xabe, 0xab0, 0x3b, 0xab6, 0xaa8, 0xabf, 0xab5, 0xabe, 0xab0, 0xab0, 0xab5,
0xabf, 0x3b, 0xab8, 0xacb, 0xaae, 0x3b, 0xaae, 0xa82, 0xa97, 0xab3, 0x3b, 0xaac,
0xac1, 0xaa7, 0x3b, 0xa97, 0xac1, 0xab0, 0xac1, 0x3b, 0xab6, 0xac1, 0xa95, 0xacd,
0xab0, 0x3b, 0xab6, 0xaa8, 0xabf, 0xab0, 0x3b, 0xab8, 0xacb, 0x3b, 0xaae, 0xa82,
0x3b, 0xaac, 0xac1, 0x3b, 0xa97, 0xac1, 0x3b, 0xab6, 0xac1, 0x3b, 0xab6, 0x43,
0x68, 0x75, 0x6d, 0x61, 0x70, 0x69, 0x72, 0x69, 0x3b, 0x43, 0x68, 0x75,
0x6d, 0x61, 0x74, 0x61, 0x74, 0x6f, 0x3b, 0x43, 0x68, 0x75, 0x6d, 0x61,
0x69, 0x6e, 0x65, 0x3b, 0x43, 0x68, 0x75, 0x6d, 0x61, 0x74, 0x61, 0x6e,
0x6f, 0x3b, 0x41, 0x72, 0x61, 0x6d, 0x69, 0x73, 0x69, 0x3b, 0x49, 0x63,
0x68, 0x75, 0x6d, 0x61, 0x3b, 0x45, 0x73, 0x61, 0x62, 0x61, 0x74, 0x6f,
0x43, 0x70, 0x72, 0x3b, 0x43, 0x74, 0x74, 0x3b, 0x43, 0x6d, 0x6e, 0x3b,
0x43, 0x6d, 0x74, 0x3b, 0x41, 0x72, 0x73, 0x3b, 0x49, 0x63, 0x6d, 0x3b,
0x45, 0x73, 0x74, 0x43, 0x3b, 0x43, 0x3b, 0x43, 0x3b, 0x43, 0x3b, 0x41,
0x3b, 0x49, 0x3b, 0x45, 0x4c, 0x61, 0x68, 0x61, 0x64, 0x69, 0x3b, 0x4c,
0x69, 0x74, 0x69, 0x6e, 0x69, 0x6e, 0x3b, 0x54, 0x61, 0x6c, 0x61, 0x74,
0x61, 0x3b, 0x4c, 0x61, 0x72, 0x61, 0x62, 0x61, 0x3b, 0x41, 0x6c, 0x68,
0x61, 0x6d, 0x69, 0x73, 0x3b, 0x4a, 0x75, 0x6d, 0x6d, 0x61, 0x2bc, 0x61,
0x3b, 0x41, 0x73, 0x61, 0x62, 0x61, 0x72, 0x4c, 0x61, 0x68, 0x3b, 0x4c,
0x69, 0x74, 0x3b, 0x54, 0x61, 0x6c, 0x3b, 0x4c, 0x61, 0x72, 0x3b, 0x41,
0x6c, 0x68, 0x3b, 0x4a, 0x75, 0x6d, 0x3b, 0x41, 0x73, 0x61, 0x4c, 0x3b,
0x4c, 0x3b, 0x54, 0x3b, 0x4c, 0x3b, 0x41, 0x3b, 0x4a, 0x3b, 0x41, 0x644,
0x64e, 0x62d, 0x64e, 0x62f, 0x650, 0x3b, 0x644, 0x650, 0x62a, 0x650, 0x646, 0x650,
0x646, 0x652, 0x3b, 0x62a, 0x64e, 0x644, 0x64e, 0x62a, 0x64e, 0x3b, 0x644, 0x64e,
0x631, 0x64e, 0x628, 0x64e, 0x3b, 0x623, 0x64e, 0x644, 0x652, 0x62d, 0x64e, 0x645,
0x650, 0x633, 0x652, 0x3b, 0x62c, 0x64f, 0x645, 0x64e, 0x639, 0x64e, 0x3b, 0x623,
0x64e, 0x633, 0x64e, 0x628, 0x64e, 0x631, 0x652, 0x644, 0x64e, 0x62d, 0x3b, 0x644,
0x650, 0x62a, 0x3b, 0x62a, 0x64e, 0x644, 0x3b, 0x644, 0x64e, 0x631, 0x3b, 0x623,
0x64e, 0x644, 0x652, 0x62d, 0x3b, 0x62c, 0x64f, 0x645, 0x3b, 0x623, 0x64e, 0x633,
0x64e, 0x4c, 0x101, 0x70, 0x75, 0x6c, 0x65, 0x3b, 0x50, 0x6f, 0x2bb, 0x61,
0x6b, 0x61, 0x68, 0x69, 0x3b, 0x50, 0x6f, 0x2bb, 0x61, 0x6c, 0x75, 0x61,
0x3b, 0x50, 0x6f, 0x2bb, 0x61, 0x6b, 0x6f, 0x6c, 0x75, 0x3b, 0x50, 0x6f,
0x2bb, 0x61, 0x68, 0x101, 0x3b, 0x50, 0x6f, 0x2bb, 0x61, 0x6c, 0x69, 0x6d,
0x61, 0x3b, 0x50, 0x6f, 0x2bb, 0x61, 0x6f, 0x6e, 0x6f, 0x4c, 0x50, 0x3b,
0x50, 0x31, 0x3b, 0x50, 0x32, 0x3b, 0x50, 0x33, 0x3b, 0x50, 0x34, 0x3b,
0x50, 0x35, 0x3b, 0x50, 0x36, 0x5d9, 0x5d5, 0x5dd, 0x20, 0x5e8, 0x5d0, 0x5e9,
0x5d5, 0x5df, 0x3b, 0x5d9, 0x5d5, 0x5dd, 0x20, 0x5e9, 0x5e0, 0x5d9, 0x3b, 0x5d9,
0x5d5, 0x5dd, 0x20, 0x5e9, 0x5dc, 0x5d9, 0x5e9, 0x5d9, 0x3b, 0x5d9, 0x5d5, 0x5dd,
0x20, 0x5e8, 0x5d1, 0x5d9, 0x5e2, 0x5d9, 0x3b, 0x5d9, 0x5d5, 0x5dd, 0x20, 0x5d7,
0x5de, 0x5d9, 0x5e9, 0x5d9, 0x3b, 0x5d9, 0x5d5, 0x5dd, 0x20, 0x5e9, 0x5d9, 0x5e9,
0x5d9, 0x3b, 0x5d9, 0x5d5, 0x5dd, 0x20, 0x5e9, 0x5d1, 0x5ea, 0x5d9, 0x5d5, 0x5dd,
0x20, 0x5d0, 0x5f3, 0x3b, 0x5d9, 0x5d5, 0x5dd, 0x20, 0x5d1, 0x5f3, 0x3b, 0x5d9,
0x5d5, 0x5dd, 0x20, 0x5d2, 0x5f3, 0x3b, 0x5d9, 0x5d5, 0x5dd, 0x20, 0x5d3, 0x5f3,
0x3b, 0x5d9, 0x5d5, 0x5dd, 0x20, 0x5d4, 0x5f3, 0x3b, 0x5d9, 0x5d5, 0x5dd, 0x20,
0x5d5, 0x5f3, 0x3b, 0x5e9, 0x5d1, 0x5ea, 0x5d0, 0x5f3, 0x3b, 0x5d1, 0x5f3, 0x3b,
0x5d2, 0x5f3, 0x3b, 0x5d3, 0x5f3, 0x3b, 0x5d4, 0x5f3, 0x3b, 0x5d5, 0x5f3, 0x3b,
0x5e9, 0x5f3, 0x930, 0x935, 0x93f, 0x935, 0x93e, 0x930, 0x3b, 0x938, 0x94b, 0x92e,
0x935, 0x93e, 0x930, 0x3b, 0x92e, 0x902, 0x917, 0x932, 0x935, 0x93e, 0x930, 0x3b,
0x92c, 0x941, 0x927, 0x935, 0x93e, 0x930, 0x3b, 0x917, 0x941, 0x930, 0x941, 0x935,
0x93e, 0x930, 0x3b, 0x936, 0x941, 0x915, 0x94d, 0x930, 0x935, 0x93e, 0x930, 0x3b,
0x936, 0x928, 0x93f, 0x935, 0x93e, 0x930, 0x930, 0x935, 0x93f, 0x3b, 0x938, 0x94b,
0x92e, 0x3b, 0x92e, 0x902, 0x917, 0x932, 0x3b, 0x92c, 0x941, 0x927, 0x3b, 0x917,
0x941, 0x930, 0x941, 0x3b, 0x936, 0x941, 0x915, 0x94d, 0x930, 0x3b, 0x936, 0x928,
0x93f, 0x930, 0x3b, 0x938, 0x94b, 0x3b, 0x92e, 0x902, 0x3b, 0x92c, 0x941, 0x3b,
0x917, 0x941, 0x3b, 0x936, 0x941, 0x3b, 0x936, 0x52, 0x61, 0x76, 0x69, 0x77,
0x61, 0x61, 0x72, 0x3b, 0x53, 0x6f, 0x6d, 0x77, 0x61, 0x61, 0x72, 0x3b,
0x4d, 0x61, 0x6e, 0x67, 0x61, 0x6c, 0x77, 0x61, 0x61, 0x72, 0x3b, 0x42,
0x75, 0x64, 0x68, 0x77, 0x61, 0x61, 0x72, 0x3b, 0x47, 0x75, 0x72, 0x75,
0x77, 0x61, 0x61, 0x72, 0x3b, 0x53, 0x68, 0x75, 0x6b, 0x72, 0x61, 0x77,
0x61, 0x61, 0x72, 0x3b, 0x53, 0x68, 0x61, 0x6e, 0x69, 0x77, 0x61, 0x61,
0x72, 0x52, 0x61, 0x76, 0x69, 0x3b, 0x53, 0x6f, 0x6d, 0x3b, 0x4d, 0x61,
0x6e, 0x67, 0x61, 0x6c, 0x3b, 0x42, 0x75, 0x64, 0x68, 0x3b, 0x47, 0x75,
0x72, 0x75, 0x3b, 0x53, 0x68, 0x75, 0x6b, 0x72, 0x61, 0x3b, 0x53, 0x68,
0x61, 0x6e, 0x69, 0x52, 0x61, 0x3b, 0x53, 0x6f, 0x3b, 0x4d, 0x61, 0x3b,
0x42, 0x75, 0x3b, 0x47, 0x75, 0x3b, 0x53, 0x68, 0x3b, 0x53, 0x68, 0x61,
0x76, 0x61, 0x73, 0xe1, 0x72, 0x6e, 0x61, 0x70, 0x3b, 0x68, 0xe9, 0x74,
0x66, 0x151, 0x3b, 0x6b, 0x65, 0x64, 0x64, 0x3b, 0x73, 0x7a, 0x65, 0x72,
0x64, 0x61, 0x3b, 0x63, 0x73, 0xfc, 0x74, 0xf6, 0x72, 0x74, 0xf6, 0x6b,
0x3b, 0x70, 0xe9, 0x6e, 0x74, 0x65, 0x6b, 0x3b, 0x73, 0x7a, 0x6f, 0x6d,
0x62, 0x61, 0x74, 0x56, 0x3b, 0x48, 0x3b, 0x4b, 0x3b, 0x53, 0x7a, 0x65,
0x3b, 0x43, 0x73, 0x3b, 0x50, 0x3b, 0x53, 0x7a, 0x6f, 0x56, 0x3b, 0x48,
0x3b, 0x4b, 0x3b, 0x53, 0x7a, 0x3b, 0x43, 0x73, 0x3b, 0x50, 0x3b, 0x53,
0x7a, 0x73, 0x75, 0x6e, 0x6e, 0x75, 0x64, 0x61, 0x67, 0x75, 0x72, 0x3b,
0x6d, 0xe1, 0x6e, 0x75, 0x64, 0x61, 0x67, 0x75, 0x72, 0x3b, 0xfe, 0x72,
0x69, 0xf0, 0x6a, 0x75, 0x64, 0x61, 0x67, 0x75, 0x72, 0x3b, 0x6d, 0x69,
0xf0, 0x76, 0x69, 0x6b, 0x75, 0x64, 0x61, 0x67, 0x75, 0x72, 0x3b, 0x66,
0x69, 0x6d, 0x6d, 0x74, 0x75, 0x64, 0x61, 0x67, 0x75, 0x72, 0x3b, 0x66,
0xf6, 0x73, 0x74, 0x75, 0x64, 0x61, 0x67, 0x75, 0x72, 0x3b, 0x6c, 0x61,
0x75, 0x67, 0x61, 0x72, 0x64, 0x61, 0x67, 0x75, 0x72, 0x73, 0x75, 0x6e,
0x2e, 0x3b, 0x6d, 0xe1, 0x6e, 0x2e, 0x3b, 0xfe, 0x72, 0x69, 0x2e, 0x3b,
0x6d, 0x69, 0xf0, 0x2e, 0x3b, 0x66, 0x69, 0x6d, 0x2e, 0x3b, 0x66, 0xf6,
0x73, 0x2e, 0x3b, 0x6c, 0x61, 0x75, 0x2e, 0x53, 0x3b, 0x4d, 0x3b, 0xde,
0x3b, 0x4d, 0x3b, 0x46, 0x3b, 0x46, 0x3b, 0x4c, 0x53, 0x1ecd, 0x6e, 0x64,
0x65, 0x65, 0x3b, 0x4d, 0x1ecd, 0x6e, 0x64, 0x65, 0x3b, 0x54, 0x69, 0x75,
0x7a, 0x64, 0x65, 0x65, 0x3b, 0x57, 0x65, 0x6e, 0x65, 0x7a, 0x64, 0x65,
0x65, 0x3b, 0x54, 0x1ecd, 0x1ecd, 0x7a, 0x64, 0x65, 0x65, 0x3b, 0x46, 0x72,
0x61, 0x1ecb, 0x64, 0x65, 0x65, 0x3b, 0x53, 0x61, 0x74, 0x1ecd, 0x64, 0x65,
0x65, 0x53, 0x1ecd, 0x6e, 0x3b, 0x4d, 0x1ecd, 0x6e, 0x3b, 0x54, 0x69, 0x75,
0x3b, 0x57, 0x65, 0x6e, 0x3b, 0x54, 0x1ecd, 0x1ecd, 0x3b, 0x46, 0x72, 0x61,
0x1ecb, 0x3b, 0x53, 0x61, 0x74, 0x70, 0x61, 0x73, 0x65, 0x70, 0x65, 0x69,
0x76, 0x69, 0x3b, 0x76, 0x75, 0x6f, 0x73, 0x73, 0x61, 0x72, 0x67, 0xe2,
0x3b, 0x6d, 0x61, 0x6a, 0x65, 0x62, 0x61, 0x72, 0x67, 0xe2, 0x3b, 0x6b,
0x6f, 0x73, 0x6b, 0x6f, 0x6b, 0x6b, 0x6f, 0x3b, 0x74, 0x75, 0x6f, 0x72,
0xe2, 0x73, 0x74, 0xe2, 0x68, 0x3b, 0x76, 0xe1, 0x73, 0x74, 0x75, 0x70,
0x70, 0x65, 0x69, 0x76, 0x69, 0x3b, 0x6c, 0xe1, 0x76, 0x75, 0x72, 0x64,
0xe2, 0x68, 0x70, 0x61, 0x73, 0x65, 0x70, 0x65, 0x65, 0x69, 0x76, 0x69,
0x3b, 0x76, 0x75, 0x6f, 0x73, 0x73, 0x61, 0x61, 0x72, 0x67, 0xe2, 0x3b,
0x6d, 0x61, 0x6a, 0x65, 0x62, 0x61, 0x61, 0x72, 0x67, 0xe2, 0x3b, 0x6b,
0x6f, 0x73, 0x6b, 0x6f, 0x68, 0x6f, 0x3b, 0x74, 0x75, 0x6f, 0x72, 0xe2,
0x73, 0x74, 0x75, 0x76, 0x3b, 0x76, 0xe1, 0x73, 0x74, 0x75, 0x70, 0x70,
0x65, 0x65, 0x69, 0x76, 0x69, 0x3b, 0x6c, 0xe1, 0x76, 0x75, 0x72, 0x64,
0x75, 0x76, 0x70, 0x61, 0x73, 0x3b, 0x76, 0x75, 0x6f, 0x3b, 0x6d, 0x61,
0x6a, 0x3b, 0x6b, 0x6f, 0x73, 0x3b, 0x74, 0x75, 0x6f, 0x3b, 0x76, 0xe1,
0x73, 0x3b, 0x6c, 0xe1, 0x76, 0x70, 0x3b, 0x56, 0x3b, 0x4d, 0x3b, 0x4b,
0x3b, 0x54, 0x3b, 0x56, 0x3b, 0x4c, 0x4d, 0x69, 0x6e, 0x67, 0x67, 0x75,
0x3b, 0x53, 0x65, 0x6e, 0x69, 0x6e, 0x3b, 0x53, 0x65, 0x6c, 0x61, 0x73,
0x61, 0x3b, 0x52, 0x61, 0x62, 0x75, 0x3b, 0x4b, 0x61, 0x6d, 0x69, 0x73,
0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x74, 0x3b, 0x53, 0x61, 0x62, 0x74, 0x75,
0x4d, 0x69, 0x6e, 0x3b, 0x53, 0x65, 0x6e, 0x3b, 0x53, 0x65, 0x6c, 0x3b,
0x52, 0x61, 0x62, 0x3b, 0x4b, 0x61, 0x6d, 0x3b, 0x4a, 0x75, 0x6d, 0x3b,
0x53, 0x61, 0x62, 0x4d, 0x3b, 0x53, 0x3b, 0x53, 0x3b, 0x52, 0x3b, 0x4b,
0x3b, 0x4a, 0x3b, 0x53, 0x64, 0x6f, 0x6d, 0x69, 0x6e, 0x69, 0x63, 0x61,
0x3b, 0x6c, 0x75, 0x6e, 0x65, 0x64, 0x69, 0x3b, 0x6d, 0x61, 0x72, 0x74,
0x65, 0x64, 0x69, 0x3b, 0x6d, 0x65, 0x72, 0x63, 0x75, 0x72, 0x69, 0x64,
0x69, 0x3b, 0x6a, 0x6f, 0x76, 0x65, 0x64, 0x69, 0x3b, 0x76, 0x65, 0x6e,
0x65, 0x72, 0x64, 0x69, 0x3b, 0x73, 0x61, 0x62, 0x62, 0x61, 0x74, 0x6f,
0x64, 0x6f, 0x6d, 0x3b, 0x6c, 0x75, 0x6e, 0x3b, 0x6d, 0x61, 0x72, 0x3b,
0x6d, 0x65, 0x72, 0x3b, 0x6a, 0x6f, 0x76, 0x3b, 0x76, 0x65, 0x6e, 0x3b,
0x73, 0x61, 0x62, 0x64, 0x3b, 0x6c, 0x3b, 0x6d, 0x3b, 0x6d, 0x3b, 0x6a,
0x3b, 0x76, 0x3b, 0x73, 0x73, 0x6f, 0x6c, 0x65, 0x64, 0xed, 0x3b, 0x6c,
0x75, 0x6e, 0x65, 0x64, 0xed, 0x3b, 0x6d, 0x61, 0x72, 0x64, 0xed, 0x3b,
0x6d, 0x65, 0x72, 0x63, 0x75, 0x72, 0x64, 0xed, 0x3b, 0x6a, 0x6f, 0x76,
0x65, 0x64, 0xed, 0x3b, 0x76, 0x65, 0x6e, 0x65, 0x72, 0x64, 0xed, 0x3b,
0x73, 0x61, 0x74, 0x75, 0x72, 0x64, 0xed, 0x73, 0x6f, 0x6c, 0x2e, 0x3b,
0x6c, 0x75, 0x6e, 0x2e, 0x3b, 0x6d, 0x61, 0x72, 0x2e, 0x3b, 0x6d, 0x65,
0x72, 0x2e, 0x3b, 0x6a, 0x6f, 0x76, 0x2e, 0x3b, 0x76, 0x65, 0x6e, 0x2e,
0x3b, 0x73, 0x61, 0x74, 0x2e, 0x53, 0x3b, 0x4c, 0x3b, 0x4d, 0x3b, 0x4d,
0x3b, 0x4a, 0x3b, 0x56, 0x3b, 0x53, 0x14c8, 0x1466, 0x144f, 0x1591, 0x152d, 0x1585,
0x3b, 0x14c7, 0x14a1, 0x1490, 0x153e, 0x152d, 0x1405, 0x3b, 0x14c7, 0x14a1, 0x1490, 0x153e,
0x152d, 0x1405, 0x14d5, 0x1585, 0x146d, 0x3b, 0x1431, 0x1593, 0x1466, 0x14ef, 0x1585, 0x3b,
0x14ef, 0x1455, 0x14bb, 0x14a5, 0x1585, 0x3b, 0x1455, 0x14ea, 0x14d5, 0x14bb, 0x14a5, 0x1405,
0x1466, 0x3b, 0x14c8, 0x1466, 0x14f0, 0x1591, 0x152d, 0x14db, 0x1550, 0x14c2, 0x140a, 0x1585,
0x44, 0xe9, 0x20, 0x44, 0x6f, 0x6d, 0x68, 0x6e, 0x61, 0x69, 0x67, 0x68,
0x3b, 0x44, 0xe9, 0x20, 0x4c, 0x75, 0x61, 0x69, 0x6e, 0x3b, 0x44, 0xe9,
0x20, 0x4d, 0xe1, 0x69, 0x72, 0x74, 0x3b, 0x44, 0xe9, 0x20, 0x43, 0xe9,
0x61, 0x64, 0x61, 0x6f, 0x69, 0x6e, 0x3b, 0x44, 0xe9, 0x61, 0x72, 0x64,
0x61, 0x6f, 0x69, 0x6e, 0x3b, 0x44, 0xe9, 0x20, 0x68, 0x41, 0x6f, 0x69,
0x6e, 0x65, 0x3b, 0x44, 0xe9, 0x20, 0x53, 0x61, 0x74, 0x68, 0x61, 0x69,
0x72, 0x6e, 0x44, 0x6f, 0x6d, 0x68, 0x3b, 0x4c, 0x75, 0x61, 0x6e, 0x3b,
0x4d, 0xe1, 0x69, 0x72, 0x74, 0x3b, 0x43, 0xe9, 0x61, 0x64, 0x3b, 0x44,
0xe9, 0x61, 0x72, 0x3b, 0x41, 0x6f, 0x69, 0x6e, 0x65, 0x3b, 0x53, 0x61,
0x74, 0x68, 0x44, 0x3b, 0x4c, 0x3b, 0x4d, 0x3b, 0x43, 0x3b, 0x44, 0x3b,
0x41, 0x3b, 0x53, 0x64, 0x6f, 0x6d, 0x65, 0x6e, 0x69, 0x63, 0x61, 0x3b,
0x6c, 0x75, 0x6e, 0x65, 0x64, 0xec, 0x3b, 0x6d, 0x61, 0x72, 0x74, 0x65,
0x64, 0xec, 0x3b, 0x6d, 0x65, 0x72, 0x63, 0x6f, 0x6c, 0x65, 0x64, 0xec,
0x3b, 0x67, 0x69, 0x6f, 0x76, 0x65, 0x64, 0xec, 0x3b, 0x76, 0x65, 0x6e,
0x65, 0x72, 0x64, 0xec, 0x3b, 0x73, 0x61, 0x62, 0x61, 0x74, 0x6f, 0x64,
0x6f, 0x6d, 0x3b, 0x6c, 0x75, 0x6e, 0x3b, 0x6d, 0x61, 0x72, 0x3b, 0x6d,
0x65, 0x72, 0x3b, 0x67, 0x69, 0x6f, 0x3b, 0x76, 0x65, 0x6e, 0x3b, 0x73,
0x61, 0x62, 0x65e5, 0x66dc, 0x65e5, 0x3b, 0x6708, 0x66dc, 0x65e5, 0x3b, 0x706b, 0x66dc,
0x65e5, 0x3b, 0x6c34, 0x66dc, 0x65e5, 0x3b, 0x6728, 0x66dc, 0x65e5, 0x3b, 0x91d1, 0x66dc,
0x65e5, 0x3b, 0x571f, 0x66dc, 0x65e5, 0x65e5, 0x3b, 0x6708, 0x3b, 0x706b, 0x3b, 0x6c34,
0x3b, 0x6728, 0x3b, 0x91d1, 0x3b, 0x571f, 0x41, 0x68, 0x61, 0x64, 0x3b, 0x53,
0x65, 0x6e, 0x69, 0x6e, 0x3b, 0x53, 0x65, 0x6c, 0x61, 0x73, 0x61, 0x3b,
0x52, 0x61, 0x62, 0x75, 0x3b, 0x4b, 0x61, 0x6d, 0x69, 0x73, 0x3b, 0x4a,
0x75, 0x6d, 0x61, 0x74, 0x3b, 0x53, 0x61, 0x62, 0x74, 0x75, 0x41, 0x68,
0x61, 0x64, 0x3b, 0x53, 0x65, 0x6e, 0x3b, 0x53, 0x65, 0x6c, 0x3b, 0x52,
0x61, 0x62, 0x3b, 0x4b, 0x61, 0x6d, 0x3b, 0x4a, 0x75, 0x6d, 0x3b, 0x53,
0x61, 0x62, 0x41, 0x3b, 0x53, 0x3b, 0x53, 0x3b, 0x52, 0x3b, 0x4b, 0x3b,
0x4a, 0x3b, 0x53, 0x4c, 0x61, 0x64, 0x69, 0x3b, 0x4c, 0x69, 0x6e, 0x74,
0x61, 0x6e, 0x69, 0x3b, 0x54, 0x61, 0x6c, 0x61, 0x74, 0x61, 0x3b, 0x4c,
0x61, 0x72, 0x62, 0x61, 0x3b, 0x4c, 0x61, 0x6d, 0x69, 0x74, 0x3b, 0x4a,
0x75, 0x6d, 0x61, 0x3b, 0x41, 0x73, 0x61, 0x62, 0x61, 0x72, 0x4c, 0x61,
0x64, 0x3b, 0x4c, 0x69, 0x6e, 0x3b, 0x54, 0x61, 0x6c, 0x3b, 0x4c, 0x61,
0x72, 0x3b, 0x4c, 0x61, 0x6d, 0x3b, 0x4a, 0x75, 0x6d, 0x3b, 0x41, 0x73,
0x61, 0x44, 0x69, 0x6d, 0x61, 0x73, 0x3b, 0x54, 0x65, 0x6e, 0x65, 0x14b,
0x3b, 0x54, 0x61, 0x6c, 0x61, 0x74, 0x61, 0x3b, 0x41, 0x6c, 0x61, 0x72,
0x62, 0x61, 0x79, 0x3b, 0x41, 0x72, 0x61, 0x6d, 0x69, 0x73, 0x61, 0x79,
0x3b, 0x41, 0x72, 0x6a, 0x75, 0x6d, 0x61, 0x3b, 0x53, 0x69, 0x62, 0x69,
0x74, 0x69, 0x44, 0x69, 0x6d, 0x3b, 0x54, 0x65, 0x6e, 0x3b, 0x54, 0x61,
0x6c, 0x3b, 0x41, 0x6c, 0x61, 0x3b, 0x41, 0x72, 0x61, 0x3b, 0x41, 0x72,
0x6a, 0x3b, 0x53, 0x69, 0x62, 0x44, 0x3b, 0x54, 0x3b, 0x54, 0x3b, 0x41,
0x3b, 0x41, 0x3b, 0x41, 0x3b, 0x53, 0x64, 0x75, 0x6d, 0x69, 0x6e, 0x67,
0x75, 0x3b, 0x73, 0x69, 0x67, 0x75, 0x6e, 0x64, 0x61, 0x2d, 0x66, 0x65,
0x72, 0x61, 0x3b, 0x74, 0x65, 0x72, 0x73, 0x61, 0x2d, 0x66, 0x65, 0x72,
0x61, 0x3b, 0x6b, 0x75, 0x61, 0x72, 0x74, 0x61, 0x2d, 0x66, 0x65, 0x72,
0x61, 0x3b, 0x6b, 0x69, 0x6e, 0x74, 0x61, 0x2d, 0x66, 0x65, 0x72, 0x61,
0x3b, 0x73, 0x65, 0x73, 0x74, 0x61, 0x2d, 0x66, 0x65, 0x72, 0x61, 0x3b,
0x73, 0xe1, 0x62, 0x61, 0x64, 0x75, 0x64, 0x75, 0x6d, 0x3b, 0x73, 0x69,
0x67, 0x3b, 0x74, 0x65, 0x72, 0x3b, 0x6b, 0x75, 0x61, 0x3b, 0x6b, 0x69,
0x6e, 0x3b, 0x73, 0x65, 0x73, 0x3b, 0x73, 0x61, 0x62, 0x44, 0x3b, 0x53,
0x3b, 0x54, 0x3b, 0x4b, 0x3b, 0x4b, 0x3b, 0x53, 0x3b, 0x53, 0x41, 0x63,
0x65, 0x72, 0x3b, 0x41, 0x72, 0x69, 0x6d, 0x3b, 0x41, 0x72, 0x61, 0x6d,
0x3b, 0x41, 0x68, 0x61, 0x64, 0x3b, 0x41, 0x6d, 0x68, 0x61, 0x64, 0x3b,
0x53, 0x65, 0x6d, 0x3b, 0x53, 0x65, 0x64, 0x59, 0x61, 0x6e, 0x61, 0x73,
0x73, 0x3b, 0x53, 0x61, 0x6e, 0x61, 0x73, 0x73, 0x3b, 0x4b, 0x72, 0x61,
0x1e0d, 0x61, 0x73, 0x73, 0x3b, 0x4b, 0x75, 0x1e93, 0x61, 0x73, 0x73, 0x3b,
0x53, 0x61, 0x6d, 0x61, 0x73, 0x73, 0x3b, 0x53, 0x1e0d, 0x69, 0x73, 0x61,
0x73, 0x73, 0x3b, 0x53, 0x61, 0x79, 0x61, 0x73, 0x73, 0x41, 0x63, 0x65,
0x3b, 0x41, 0x72, 0x69, 0x3b, 0x41, 0x72, 0x61, 0x3b, 0x41, 0x68, 0x61,
0x3b, 0x41, 0x6d, 0x68, 0x3b, 0x53, 0x65, 0x6d, 0x3b, 0x53, 0x65, 0x64,
0x59, 0x61, 0x6e, 0x3b, 0x53, 0x61, 0x6e, 0x3b, 0x4b, 0x72, 0x61, 0x1e0d,
0x3b, 0x4b, 0x75, 0x1e93, 0x3b, 0x53, 0x61, 0x6d, 0x3b, 0x53, 0x1e0d, 0x69,
0x73, 0x3b, 0x53, 0x61, 0x79, 0x59, 0x3b, 0x53, 0x3b, 0x4b, 0x3b, 0x4b,
0x3b, 0x53, 0x3b, 0x53, 0x3b, 0x53, 0x43, 0x3b, 0x52, 0x3b, 0x52, 0x3b,
0x48, 0x3b, 0x4d, 0x3b, 0x53, 0x3b, 0x53, 0x73, 0x254, 0x6e, 0x64, 0x69,
0x3b, 0x6c, 0x75, 0x6e, 0x64, 0x69, 0x3b, 0x6d, 0x61, 0x72, 0x64, 0x69,
0x3b, 0x6d, 0x25b, 0x72, 0x6b, 0x25b, 0x72, 0x25b, 0x64, 0x69, 0x3b, 0x79,
0x65, 0x64, 0x69, 0x3b, 0x76, 0x61, 0x14b, 0x64, 0x25b, 0x72, 0x25b, 0x64,
0x69, 0x3b, 0x6d, 0x254, 0x6e, 0x254, 0x20, 0x73, 0x254, 0x6e, 0x64, 0x69,
0x73, 0x6f, 0x3b, 0x6c, 0x75, 0x3b, 0x6d, 0x61, 0x3b, 0x6d, 0x25b, 0x3b,
0x79, 0x65, 0x3b, 0x76, 0x61, 0x3b, 0x6d, 0x73, 0x73, 0x61, 0x70, 0x61,
0x61, 0x74, 0x3b, 0x61, 0x74, 0x61, 0x61, 0x73, 0x69, 0x6e, 0x6e, 0x67,
0x6f, 0x72, 0x6e, 0x65, 0x71, 0x3b, 0x6d, 0x61, 0x72, 0x6c, 0x75, 0x6e,
0x6e, 0x67, 0x6f, 0x72, 0x6e, 0x65, 0x71, 0x3b, 0x70, 0x69, 0x6e, 0x67,
0x61, 0x73, 0x75, 0x6e, 0x6e, 0x67, 0x6f, 0x72, 0x6e, 0x65, 0x71, 0x3b,
0x73, 0x69, 0x73, 0x61, 0x6d, 0x61, 0x6e, 0x6e, 0x67, 0x6f, 0x72, 0x6e,
0x65, 0x71, 0x3b, 0x74, 0x61, 0x6c, 0x6c, 0x69, 0x6d, 0x61, 0x6e, 0x6e,
0x67, 0x6f, 0x72, 0x6e, 0x65, 0x71, 0x3b, 0x61, 0x72, 0x66, 0x69, 0x6e,
0x69, 0x6e, 0x6e, 0x67, 0x6f, 0x72, 0x6e, 0x65, 0x71, 0x73, 0x61, 0x70,
0x3b, 0x61, 0x74, 0x61, 0x3b, 0x6d, 0x61, 0x72, 0x3b, 0x70, 0x69, 0x6e,
0x3b, 0x73, 0x69, 0x73, 0x3b, 0x74, 0x61, 0x6c, 0x3b, 0x61, 0x72, 0x66,
0x53, 0x3b, 0x41, 0x3b, 0x4d, 0x3b, 0x50, 0x3b, 0x53, 0x3b, 0x54, 0x3b,
0x41, 0x4b, 0x6f, 0x74, 0x69, 0x73, 0x61, 0x70, 0x3b, 0x4b, 0x6f, 0x74,
0x61, 0x61, 0x69, 0x3b, 0x4b, 0x6f, 0x61, 0x65, 0x6e, 0x67, 0x2019, 0x3b,
0x4b, 0x6f, 0x73, 0x6f, 0x6d, 0x6f, 0x6b, 0x3b, 0x4b, 0x6f, 0x61, 0x6e,
0x67, 0x2019, 0x77, 0x61, 0x6e, 0x3b, 0x4b, 0x6f, 0x6d, 0x75, 0x75, 0x74,
0x3b, 0x4b, 0x6f, 0x6c, 0x6f, 0x4b, 0x74, 0x73, 0x3b, 0x4b, 0x6f, 0x74,
0x3b, 0x4b, 0x6f, 0x6f, 0x3b, 0x4b, 0x6f, 0x73, 0x3b, 0x4b, 0x6f, 0x61,
0x3b, 0x4b, 0x6f, 0x6d, 0x3b, 0x4b, 0x6f, 0x6c, 0x54, 0x3b, 0x54, 0x3b,
0x4f, 0x3b, 0x53, 0x3b, 0x41, 0x3b, 0x4d, 0x3b, 0x4c, 0x57, 0x61, 0x20,
0x6b, 0x79, 0x75, 0x6d, 0x77, 0x61, 0x3b, 0x57, 0x61, 0x20, 0x6b, 0x77,
0x61, 0x6d, 0x62, 0x129, 0x6c, 0x129, 0x6c, 0x79, 0x61, 0x3b, 0x57, 0x61,
0x20, 0x6b, 0x65, 0x6c, 0x129, 0x3b, 0x57, 0x61, 0x20, 0x6b, 0x61, 0x74,
0x61, 0x74, 0x169, 0x3b, 0x57, 0x61, 0x20, 0x6b, 0x61, 0x6e, 0x61, 0x3b,
0x57, 0x61, 0x20, 0x6b, 0x61, 0x74, 0x61, 0x6e, 0x6f, 0x3b, 0x57, 0x61,
0x20, 0x74, 0x68, 0x61, 0x6e, 0x74, 0x68, 0x61, 0x74, 0x169, 0x57, 0x6b,
0x79, 0x3b, 0x57, 0x6b, 0x77, 0x3b, 0x57, 0x6b, 0x6c, 0x3b, 0x57, 0x74,
0x169, 0x3b, 0x57, 0x6b, 0x6e, 0x3b, 0x57, 0x74, 0x6e, 0x3b, 0x57, 0x74,
0x68, 0x59, 0x3b, 0x57, 0x3b, 0x45, 0x3b, 0x41, 0x3b, 0x41, 0x3b, 0x41,
0x3b, 0x41, 0xcad, 0xcbe, 0xca8, 0xcc1, 0xcb5, 0xcbe, 0xcb0, 0x3b, 0xcb8, 0xccb,
0xcae, 0xcb5, 0xcbe, 0xcb0, 0x3b, 0xcae, 0xc82, 0xc97, 0xcb3, 0xcb5, 0xcbe, 0xcb0,
0x3b, 0xcac, 0xcc1, 0xca7, 0xcb5, 0xcbe, 0xcb0, 0x3b, 0xc97, 0xcc1, 0xcb0, 0xcc1,
0xcb5, 0xcbe, 0xcb0, 0x3b, 0xcb6, 0xcc1, 0xc95, 0xccd, 0xcb0, 0xcb5, 0xcbe, 0xcb0,
0x3b, 0xcb6, 0xca8, 0xcbf, 0xcb5, 0xcbe, 0xcb0, 0xcad, 0xcbe, 0xca8, 0xcc1, 0x3b,
0xcb8, 0xccb, 0xcae, 0x3b, 0xcae, 0xc82, 0xc97, 0xcb3, 0x3b, 0xcac, 0xcc1, 0xca7,
0x3b, 0xc97, 0xcc1, 0xcb0, 0xcc1, 0x3b, 0xcb6, 0xcc1, 0xc95, 0xccd, 0xcb0, 0x3b,
0xcb6, 0xca8, 0xcbf, 0xcad, 0xcbe, 0x3b, 0xcb8, 0xccb, 0x3b, 0xcae, 0xc82, 0x3b,
0xcac, 0xcc1, 0x3b, 0xc97, 0xcc1, 0x3b, 0xcb6, 0xcc1, 0x3b, 0xcb6, 0x627, 0x64e,
0x62a, 0x6be, 0x648, 0x627, 0x631, 0x3b, 0x698, 0x654, 0x646, 0x62f, 0x631, 0x655,
0x631, 0x648, 0x627, 0x631, 0x3b, 0x628, 0x6c6, 0x645, 0x648, 0x627, 0x631, 0x3b,
0x628, 0x648, 0x62f, 0x648, 0x627, 0x631, 0x3b, 0x628, 0x631, 0x620, 0x633, 0x648,
0x627, 0x631, 0x3b, 0x62c, 0x64f, 0x645, 0x6c1, 0x3b, 0x628, 0x679, 0x648, 0x627,
0x631, 0x622, 0x62a, 0x6be, 0x648, 0x627, 0x631, 0x3b, 0x698, 0x654, 0x646, 0x62f,
0x655, 0x631, 0x648, 0x627, 0x631, 0x3b, 0x628, 0x6c6, 0x645, 0x648, 0x627, 0x631,
0x3b, 0x628, 0x648, 0x62f, 0x648, 0x627, 0x631, 0x3b, 0x628, 0x631, 0x620, 0x633,
0x648, 0x627, 0x631, 0x3b, 0x62c, 0x64f, 0x645, 0x6c1, 0x3b, 0x628, 0x679, 0x648,
0x627, 0x631, 0x627, 0x3b, 0x698, 0x3b, 0x628, 0x3b, 0x628, 0x3b, 0x628, 0x3b,
0x62c, 0x3b, 0x628, 0x906, 0x925, 0x935, 0x93e, 0x930, 0x3b, 0x91a, 0x902, 0x926,
0x93f, 0x930, 0x935, 0x93e, 0x930, 0x3b, 0x92c, 0x941, 0x935, 0x93e, 0x930, 0x3b,
0x92c, 0x94b, 0x926, 0x935, 0x93e, 0x930, 0x3b, 0x92c, 0x94d, 0x930, 0x947, 0x938,
0x935, 0x93e, 0x930, 0x3b, 0x91c, 0x941, 0x92e, 0x94d, 0x92e, 0x93e, 0x3b, 0x92c,
0x91f, 0x935, 0x93e, 0x930, 0x906, 0x925, 0x935, 0x93e, 0x930, 0x3b, 0x91a, 0x93c,
0x902, 0x926, 0x93f, 0x930, 0x935, 0x93e, 0x930, 0x3b, 0x92c, 0x94b, 0x92e, 0x935,
0x93e, 0x930, 0x3b, 0x92c, 0x94b, 0x926, 0x935, 0x93e, 0x930, 0x3b, 0x92c, 0x94d,
0x930, 0x947, 0x938, 0x935, 0x93e, 0x930, 0x3b, 0x91c, 0x941, 0x92e, 0x93e, 0x3b,
0x92c, 0x91f, 0x935, 0x93e, 0x930, 0x906, 0x925, 0x935, 0x93e, 0x930, 0x3b, 0x91a,
0x902, 0x926, 0x93f, 0x930, 0x935, 0x93e, 0x930, 0x3b, 0x92c, 0x941, 0x935, 0x93e,
0x930, 0x3b, 0x92c, 0x94b, 0x926, 0x935, 0x93e, 0x930, 0x3b, 0x92c, 0x94d, 0x930,
0x947, 0x938, 0x935, 0x93e, 0x930, 0x3b, 0x91c, 0x941, 0x92e, 0x93e, 0x3b, 0x92c,
0x91f, 0x935, 0x93e, 0x930, 0x905, 0x3b, 0x91a, 0x3b, 0x92c, 0x3b, 0x92c, 0x3b,
0x92c, 0x3b, 0x91c, 0x3b, 0x92c, 0x436, 0x435, 0x43a, 0x441, 0x435, 0x43d, 0x431,
0x456, 0x3b, 0x434, 0x4af, 0x439, 0x441, 0x435, 0x43d, 0x431, 0x456, 0x3b, 0x441,
0x435, 0x439, 0x441, 0x435, 0x43d, 0x431, 0x456, 0x3b, 0x441, 0x4d9, 0x440, 0x441,
0x435, 0x43d, 0x431, 0x456, 0x3b, 0x431, 0x435, 0x439, 0x441, 0x435, 0x43d, 0x431,
0x456, 0x3b, 0x436, 0x4b1, 0x43c, 0x430, 0x3b, 0x441, 0x435, 0x43d, 0x431, 0x456,
0x436, 0x441, 0x3b, 0x434, 0x441, 0x3b, 0x441, 0x441, 0x3b, 0x441, 0x440, 0x3b,
0x431, 0x441, 0x3b, 0x436, 0x43c, 0x3b, 0x441, 0x431, 0x416, 0x3b, 0x414, 0x3b,
0x421, 0x3b, 0x421, 0x3b, 0x411, 0x3b, 0x416, 0x3b, 0x421, 0x17a2, 0x17b6, 0x1791,
0x17b7, 0x178f, 0x17d2, 0x1799, 0x3b, 0x1785, 0x1793, 0x17d2, 0x1791, 0x3b, 0x17a2, 0x1784,
0x17d2, 0x1782, 0x17b6, 0x179a, 0x3b, 0x1796, 0x17bb, 0x1792, 0x3b, 0x1796, 0x17d2, 0x179a,
0x17a0, 0x179f, 0x17d2, 0x1794, 0x178f, 0x17b7, 0x17cd, 0x3b, 0x179f, 0x17bb, 0x1780, 0x17d2,
0x179a, 0x3b, 0x179f, 0x17c5, 0x179a, 0x17cd, 0x17a2, 0x17b6, 0x1791, 0x17b7, 0x178f, 0x17d2,
0x1799, 0x3b, 0x1785, 0x17d0, 0x1793, 0x17d2, 0x1791, 0x3b, 0x17a2, 0x1784, 0x17d2, 0x1782,
0x17b6, 0x179a, 0x3b, 0x1796, 0x17bb, 0x1792, 0x3b, 0x1796, 0x17d2, 0x179a, 0x17a0, 0x179f,
0x17d2, 0x1794, 0x178f, 0x17b7, 0x17cd, 0x3b, 0x179f, 0x17bb, 0x1780, 0x17d2, 0x179a, 0x3b,
0x179f, 0x17c5, 0x179a, 0x17cd, 0x17a2, 0x17b6, 0x1791, 0x17b7, 0x178f, 0x17d2, 0x1799, 0x3b,
0x1785, 0x1793, 0x17d2, 0x1791, 0x3b, 0x17a2, 0x1784, 0x17d2, 0x1782, 0x17b6, 0x179a, 0x3b,
0x1796, 0x17bb, 0x1792, 0x3b, 0x1796, 0x17d2, 0x179a, 0x17a0, 0x3b, 0x179f, 0x17bb, 0x1780,
0x17d2, 0x179a, 0x3b, 0x179f, 0x17c5, 0x179a, 0x17cd, 0x17a2, 0x3b, 0x1785, 0x3b, 0x17a2,
0x3b, 0x1796, 0x3b, 0x1796, 0x3b, 0x179f, 0x3b, 0x179f, 0x4b, 0x69, 0x75, 0x6d,
0x69, 0x61, 0x3b, 0x4e, 0x6a, 0x75, 0x6d, 0x61, 0x74, 0x61, 0x74, 0x169,
0x3b, 0x4e, 0x6a, 0x75, 0x6d, 0x61, 0x69, 0x6e, 0x65, 0x3b, 0x4e, 0x6a,
0x75, 0x6d, 0x61, 0x74, 0x61, 0x6e, 0x61, 0x3b, 0x41, 0x72, 0x61, 0x6d,
0x69, 0x74, 0x68, 0x69, 0x3b, 0x4e, 0x6a, 0x75, 0x6d, 0x61, 0x61, 0x3b,
0x4e, 0x6a, 0x75, 0x6d, 0x61, 0x6d, 0x6f, 0x74, 0x68, 0x69, 0x4b, 0x4d,
0x41, 0x3b, 0x4e, 0x54, 0x54, 0x3b, 0x4e, 0x4d, 0x4e, 0x3b, 0x4e, 0x4d,
0x54, 0x3b, 0x41, 0x52, 0x54, 0x3b, 0x4e, 0x4d, 0x41, 0x3b, 0x4e, 0x4d,
0x4d, 0x4b, 0x3b, 0x4e, 0x3b, 0x4e, 0x3b, 0x4e, 0x3b, 0x41, 0x3b, 0x4e,
0x3b, 0x4e, 0x4b, 0x75, 0x20, 0x63, 0x79, 0x75, 0x6d, 0x77, 0x65, 0x72,
0x75, 0x3b, 0x4b, 0x75, 0x77, 0x61, 0x20, 0x6d, 0x62, 0x65, 0x72, 0x65,
0x3b, 0x4b, 0x75, 0x77, 0x61, 0x20, 0x6b, 0x61, 0x62, 0x69, 0x72, 0x69,
0x3b, 0x4b, 0x75, 0x77, 0x61, 0x20, 0x67, 0x61, 0x74, 0x61, 0x74, 0x75,
0x3b, 0x4b, 0x75, 0x77, 0x61, 0x20, 0x6b, 0x61, 0x6e, 0x65, 0x3b, 0x4b,
0x75, 0x77, 0x61, 0x20, 0x67, 0x61, 0x74, 0x61, 0x6e, 0x75, 0x3b, 0x4b,
0x75, 0x77, 0x61, 0x20, 0x67, 0x61, 0x74, 0x61, 0x6e, 0x64, 0x61, 0x74,
0x75, 0x63, 0x79, 0x75, 0x2e, 0x3b, 0x6d, 0x62, 0x65, 0x2e, 0x3b, 0x6b,
0x61, 0x62, 0x2e, 0x3b, 0x67, 0x74, 0x75, 0x2e, 0x3b, 0x6b, 0x61, 0x6e,
0x2e, 0x3b, 0x67, 0x6e, 0x75, 0x2e, 0x3b, 0x67, 0x6e, 0x64, 0x2e, 0x906,
0x92f, 0x924, 0x93e, 0x930, 0x3b, 0x938, 0x94b, 0x92e, 0x93e, 0x930, 0x3b, 0x92e,
0x902, 0x917, 0x933, 0x93e, 0x930, 0x3b, 0x92c, 0x941, 0x927, 0x935, 0x93e, 0x930,
0x3b, 0x92c, 0x93f, 0x930, 0x947, 0x938, 0x94d, 0x924, 0x93e, 0x930, 0x3b, 0x936,
0x941, 0x915, 0x94d, 0x930, 0x93e, 0x930, 0x3b, 0x936, 0x947, 0x928, 0x935, 0x93e,
0x930, 0x906, 0x3b, 0x938, 0x94b, 0x3b, 0x92e, 0x902, 0x3b, 0x92c, 0x941, 0x3b,
0x92c, 0x3b, 0x936, 0x941, 0x3b, 0x936, 0x947, 0x906, 0x3b, 0x938, 0x94b, 0x3b,
0x92e, 0x902, 0x3b, 0x92c, 0x941, 0x3b, 0x92c, 0x93f, 0x3b, 0x936, 0x941, 0x3b,
0x936, 0x947, 0xc77c, 0xc694, 0xc77c, 0x3b, 0xc6d4, 0xc694, 0xc77c, 0x3b, 0xd654, 0xc694,
0xc77c, 0x3b, 0xc218, 0xc694, 0xc77c, 0x3b, 0xbaa9, 0xc694, 0xc77c, 0x3b, 0xae08, 0xc694,
0xc77c, 0x3b, 0xd1a0, 0xc694, 0xc77c, 0xc77c, 0x3b, 0xc6d4, 0x3b, 0xd654, 0x3b, 0xc218,
0x3b, 0xbaa9, 0x3b, 0xae08, 0x3b, 0xd1a0, 0x41, 0x6c, 0x68, 0x61, 0x64, 0x69,
0x3b, 0x41, 0x74, 0x69, 0x6e, 0x6e, 0x69, 0x3b, 0x41, 0x74, 0x61, 0x6c,
0x61, 0x61, 0x74, 0x61, 0x3b, 0x41, 0x6c, 0x61, 0x72, 0x62, 0x61, 0x3b,
0x41, 0x6c, 0x68, 0x61, 0x6d, 0x69, 0x69, 0x73, 0x61, 0x3b, 0x41, 0x6c,
0x7a, 0x75, 0x6d, 0x61, 0x3b, 0x41, 0x73, 0x69, 0x62, 0x74, 0x69, 0x41,
0x6c, 0x68, 0x3b, 0x41, 0x74, 0x69, 0x3b, 0x41, 0x74, 0x61, 0x3b, 0x41,
0x6c, 0x61, 0x3b, 0x41, 0x6c, 0x6d, 0x3b, 0x41, 0x6c, 0x7a, 0x3b, 0x41,
0x73, 0x69, 0x48, 0x3b, 0x54, 0x3b, 0x54, 0x3b, 0x4c, 0x3b, 0x4c, 0x3b,
0x4c, 0x3b, 0x53, 0x41, 0x6c, 0x68, 0x61, 0x64, 0x69, 0x3b, 0x41, 0x74,
0x69, 0x6e, 0x69, 0x3b, 0x41, 0x74, 0x61, 0x6c, 0x61, 0x74, 0x61, 0x3b,
0x41, 0x6c, 0x61, 0x72, 0x62, 0x61, 0x3b, 0x41, 0x6c, 0x68, 0x61, 0x6d,
0x69, 0x69, 0x73, 0x61, 0x3b, 0x41, 0x6c, 0x6a, 0x75, 0x6d, 0x61, 0x3b,
0x41, 0x73, 0x73, 0x61, 0x62, 0x64, 0x75, 0x41, 0x6c, 0x68, 0x3b, 0x41,
0x74, 0x69, 0x3b, 0x41, 0x74, 0x61, 0x3b, 0x41, 0x6c, 0x61, 0x3b, 0x41,
0x6c, 0x6d, 0x3b, 0x41, 0x6c, 0x6a, 0x3b, 0x41, 0x73, 0x73, 0x79, 0x65,
0x6b, 0x15f, 0x65, 0x6d, 0x3b, 0x64, 0x75, 0x15f, 0x65, 0x6d, 0x3b, 0x73,
0xea, 0x15f, 0x65, 0x6d, 0x3b, 0xe7, 0x61, 0x72, 0x15f, 0x65, 0x6d, 0x3b,
0x70, 0xea, 0x6e, 0x63, 0x15f, 0x65, 0x6d, 0x3b, 0xee, 0x6e, 0xee, 0x3b,
0x15f, 0x65, 0x6d, 0xee, 0x79, 0x15f, 0x6d, 0x3b, 0x64, 0x15f, 0x6d, 0x3b,
0x73, 0x15f, 0x6d, 0x3b, 0xe7, 0x15f, 0x6d, 0x3b, 0x70, 0x15f, 0x6d, 0x3b,
0xee, 0x6e, 0xee, 0x3b, 0x15f, 0x65, 0x6d, 0x59, 0x3b, 0x44, 0x3b, 0x53,
0x3b, 0xc7, 0x3b, 0x50, 0x3b, 0xce, 0x3b, 0x15e, 0x73, 0x254, 0x301, 0x6e,
0x64, 0x254, 0x3b, 0x6d, 0x254, 0x301, 0x6e, 0x64, 0x254, 0x3b, 0x73, 0x254,
0x301, 0x6e, 0x64, 0x254, 0x20, 0x6d, 0x61, 0x66, 0xfa, 0x20, 0x6d, 0xe1,
0x62, 0x61, 0x3b, 0x73, 0x254, 0x301, 0x6e, 0x64, 0x254, 0x20, 0x6d, 0x61,
0x66, 0xfa, 0x20, 0x6d, 0xe1, 0x6c, 0x61, 0x6c, 0x3b, 0x73, 0x254, 0x301,
0x6e, 0x64, 0x254, 0x20, 0x6d, 0x61, 0x66, 0xfa, 0x20, 0x6d, 0xe1, 0x6e,
0x61, 0x3b, 0x6d, 0x61, 0x62, 0xe1, 0x67, 0xe1, 0x20, 0x6d, 0xe1, 0x20,
0x73, 0x75, 0x6b, 0x75, 0x6c, 0x3b, 0x73, 0xe1, 0x73, 0x61, 0x64, 0x69,
0x73, 0x254, 0x301, 0x6e, 0x3b, 0x6d, 0x254, 0x301, 0x6e, 0x3b, 0x73, 0x6d,
0x62, 0x3b, 0x73, 0x6d, 0x6c, 0x3b, 0x73, 0x6d, 0x6e, 0x3b, 0x6d, 0x62,
0x73, 0x3b, 0x73, 0x61, 0x73, 0x73, 0x3b, 0x6d, 0x3b, 0x73, 0x3b, 0x73,
0x3b, 0x73, 0x3b, 0x6d, 0x3b, 0x73, 0x436, 0x435, 0x43a, 0x448, 0x435, 0x43c,
0x431, 0x438, 0x3b, 0x434, 0x4af, 0x439, 0x448, 0x4e9, 0x43c, 0x431, 0x4af, 0x3b,
0x448, 0x435, 0x439, 0x448, 0x435, 0x43c, 0x431, 0x438, 0x3b, 0x448, 0x430, 0x440,
0x448, 0x435, 0x43c, 0x431, 0x438, 0x3b, 0x431, 0x435, 0x439, 0x448, 0x435, 0x43c,
0x431, 0x438, 0x3b, 0x436, 0x443, 0x43c, 0x430, 0x3b, 0x438, 0x448, 0x435, 0x43c,
0x431, 0x438, 0x436, 0x435, 0x43a, 0x2e, 0x3b, 0x434, 0x4af, 0x439, 0x2e, 0x3b,
0x448, 0x435, 0x439, 0x448, 0x2e, 0x3b, 0x448, 0x430, 0x440, 0x448, 0x2e, 0x3b,
0x431, 0x435, 0x439, 0x448, 0x2e, 0x3b, 0x436, 0x443, 0x43c, 0x430, 0x3b, 0x438,
0x448, 0x43c, 0x2e, 0x416, 0x3b, 0x414, 0x3b, 0x428, 0x3b, 0x428, 0x3b, 0x411,
0x3b, 0x416, 0x3b, 0x418, 0x41, 0x14b, 0x70, 0xe9, 0x74, 0x75, 0x77, 0x61,
0x6b, 0x21f, 0x61, 0x14b, 0x3b, 0x41, 0x14b, 0x70, 0xe9, 0x74, 0x75, 0x77,
0x61, 0x14b, 0x17e, 0x69, 0x3b, 0x41, 0x14b, 0x70, 0xe9, 0x74, 0x75, 0x6e,
0x75, 0x14b, 0x70, 0x61, 0x3b, 0x41, 0x14b, 0x70, 0xe9, 0x74, 0x75, 0x79,
0x61, 0x6d, 0x6e, 0x69, 0x3b, 0x41, 0x14b, 0x70, 0xe9, 0x74, 0x75, 0x74,
0x6f, 0x70, 0x61, 0x3b, 0x41, 0x14b, 0x70, 0xe9, 0x74, 0x75, 0x7a, 0x61,
0x70, 0x74, 0x61, 0x14b, 0x3b, 0x4f, 0x77, 0xe1, 0x14b, 0x67, 0x79, 0x75,
0x17e, 0x61, 0x17e, 0x61, 0x70, 0x69, 0x41, 0x3b, 0x57, 0x3b, 0x4e, 0x3b,
0x59, 0x3b, 0x54, 0x3b, 0x5a, 0x3b, 0x4f, 0x4a, 0x75, 0x6d, 0x61, 0x70,
0xed, 0x69, 0x72, 0x69, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x74, 0xe1, 0x74,
0x75, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0xed, 0x6e, 0x65, 0x3b, 0x4a, 0x75,
0x6d, 0x61, 0x74, 0xe1, 0x61, 0x6e, 0x6f, 0x3b, 0x41, 0x6c, 0x61, 0x6d,
0xed, 0x69, 0x73, 0x69, 0x3b, 0x49, 0x6a, 0x75, 0x6d, 0xe1, 0x61, 0x3b,
0x4a, 0x75, 0x6d, 0x61, 0x6d, 0xf3, 0x6f, 0x73, 0x69, 0x50, 0xed, 0x69,
0x6c, 0x69, 0x3b, 0x54, 0xe1, 0x61, 0x74, 0x75, 0x3b, 0xcd, 0x6e, 0x65,
0x3b, 0x54, 0xe1, 0x61, 0x6e, 0x6f, 0x3b, 0x41, 0x6c, 0x68, 0x3b, 0x49,
0x6a, 0x6d, 0x3b, 0x4d, 0xf3, 0x6f, 0x73, 0x69, 0x50, 0x3b, 0x54, 0x3b,
0x45, 0x3b, 0x4f, 0x3b, 0x41, 0x3b, 0x49, 0x3b, 0x4d, 0xea7, 0xeb1, 0xe99,
0xead, 0xeb2, 0xe97, 0xeb4, 0xe94, 0x3b, 0xea7, 0xeb1, 0xe99, 0xe88, 0xeb1, 0xe99,
0x3b, 0xea7, 0xeb1, 0xe99, 0xead, 0xeb1, 0xe87, 0xe84, 0xeb2, 0xe99, 0x3b, 0xea7,
0xeb1, 0xe99, 0xe9e, 0xeb8, 0xe94, 0x3b, 0xea7, 0xeb1, 0xe99, 0xe9e, 0xeb0, 0xeab,
0xeb1, 0xe94, 0x3b, 0xea7, 0xeb1, 0xe99, 0xeaa, 0xeb8, 0xe81, 0x3b, 0xea7, 0xeb1,
0xe99, 0xec0, 0xeaa, 0xebb, 0xeb2, 0xead, 0xeb2, 0xe97, 0xeb4, 0xe94, 0x3b, 0xe88,
0xeb1, 0xe99, 0x3b, 0xead, 0xeb1, 0xe87, 0xe84, 0xeb2, 0xe99, 0x3b, 0xe9e, 0xeb8,
0xe94, 0x3b, 0xe9e, 0xeb0, 0xeab, 0xeb1, 0xe94, 0x3b, 0xeaa, 0xeb8, 0xe81, 0x3b,
0xec0, 0xeaa, 0xebb, 0xeb2, 0xead, 0xeb2, 0x3b, 0xe88, 0x3b, 0xead, 0x3b, 0xe9e,
0x3b, 0xe9e, 0xeab, 0x3b, 0xeaa, 0xeb8, 0x3b, 0xeaa, 0x64, 0x69, 0x65, 0x73,
0x20, 0x53, 0x6f, 0x6c, 0x69, 0x73, 0x3b, 0x64, 0x69, 0x65, 0x73, 0x20,
0x4c, 0x75, 0x6e, 0x61, 0x65, 0x3b, 0x64, 0x69, 0x65, 0x73, 0x20, 0x4d,
0x61, 0x72, 0x74, 0x69, 0x73, 0x3b, 0x64, 0x69, 0x65, 0x73, 0x20, 0x4d,
0x65, 0x72, 0x63, 0x75, 0x72, 0x69, 0x69, 0x3b, 0x64, 0x69, 0x65, 0x73,
0x20, 0x49, 0x6f, 0x76, 0x69, 0x73, 0x3b, 0x64, 0x69, 0x65, 0x73, 0x20,
0x56, 0x65, 0x6e, 0x65, 0x72, 0x69, 0x73, 0x3b, 0x64, 0x69, 0x65, 0x73,
0x20, 0x53, 0x61, 0x74, 0x75, 0x72, 0x6e, 0x69, 0x53, 0x6f, 0x6c, 0x3b,
0x4c, 0x75, 0x6e, 0x3b, 0x4d, 0x61, 0x72, 0x3b, 0x4d, 0x65, 0x72, 0x3b,
0x49, 0x6f, 0x76, 0x3b, 0x56, 0x65, 0x6e, 0x3b, 0x53, 0x61, 0x74, 0x53,
0x76, 0x113, 0x74, 0x64, 0x69, 0x65, 0x6e, 0x61, 0x3b, 0x50, 0x69, 0x72,
0x6d, 0x64, 0x69, 0x65, 0x6e, 0x61, 0x3b, 0x4f, 0x74, 0x72, 0x64, 0x69,
0x65, 0x6e, 0x61, 0x3b, 0x54, 0x72, 0x65, 0x161, 0x64, 0x69, 0x65, 0x6e,
0x61, 0x3b, 0x43, 0x65, 0x74, 0x75, 0x72, 0x74, 0x64, 0x69, 0x65, 0x6e,
0x61, 0x3b, 0x50, 0x69, 0x65, 0x6b, 0x74, 0x64, 0x69, 0x65, 0x6e, 0x61,
0x3b, 0x53, 0x65, 0x73, 0x74, 0x64, 0x69, 0x65, 0x6e, 0x61, 0x73, 0x76,
0x113, 0x74, 0x64, 0x69, 0x65, 0x6e, 0x61, 0x3b, 0x70, 0x69, 0x72, 0x6d,
0x64, 0x69, 0x65, 0x6e, 0x61, 0x3b, 0x6f, 0x74, 0x72, 0x64, 0x69, 0x65,
0x6e, 0x61, 0x3b, 0x74, 0x72, 0x65, 0x161, 0x64, 0x69, 0x65, 0x6e, 0x61,
0x3b, 0x63, 0x65, 0x74, 0x75, 0x72, 0x74, 0x64, 0x69, 0x65, 0x6e, 0x61,
0x3b, 0x70, 0x69, 0x65, 0x6b, 0x74, 0x64, 0x69, 0x65, 0x6e, 0x61, 0x3b,
0x73, 0x65, 0x73, 0x74, 0x64, 0x69, 0x65, 0x6e, 0x61, 0x53, 0x76, 0x113,
0x74, 0x64, 0x2e, 0x3b, 0x50, 0x69, 0x72, 0x6d, 0x64, 0x2e, 0x3b, 0x4f,
0x74, 0x72, 0x64, 0x2e, 0x3b, 0x54, 0x72, 0x65, 0x161, 0x64, 0x2e, 0x3b,
0x43, 0x65, 0x74, 0x75, 0x72, 0x74, 0x64, 0x2e, 0x3b, 0x50, 0x69, 0x65,
0x6b, 0x74, 0x64, 0x2e, 0x3b, 0x53, 0x65, 0x73, 0x74, 0x64, 0x2e, 0x73,
0x76, 0x113, 0x74, 0x64, 0x2e, 0x3b, 0x70, 0x69, 0x72, 0x6d, 0x64, 0x2e,
0x3b, 0x6f, 0x74, 0x72, 0x64, 0x2e, 0x3b, 0x74, 0x72, 0x65, 0x161, 0x64,
0x2e, 0x3b, 0x63, 0x65, 0x74, 0x75, 0x72, 0x74, 0x64, 0x2e, 0x3b, 0x70,
0x69, 0x65, 0x6b, 0x74, 0x64, 0x2e, 0x3b, 0x73, 0x65, 0x73, 0x74, 0x64,
0x2e, 0x53, 0x3b, 0x50, 0x3b, 0x4f, 0x3b, 0x54, 0x3b, 0x43, 0x3b, 0x50,
0x3b, 0x53, 0x65, 0x79, 0x65, 0x6e, 0x67, 0x61, 0x3b, 0x6d, 0x6f, 0x6b,
0x254, 0x6c, 0x254, 0x20, 0x6d, 0x77, 0x61, 0x20, 0x79, 0x61, 0x6d, 0x62,
0x6f, 0x3b, 0x6d, 0x6f, 0x6b, 0x254, 0x6c, 0x254, 0x20, 0x6d, 0x77, 0x61,
0x20, 0x6d, 0xed, 0x62, 0x61, 0x6c, 0xe9, 0x3b, 0x6d, 0x6f, 0x6b, 0x254,
0x6c, 0x254, 0x20, 0x6d, 0x77, 0x61, 0x20, 0x6d, 0xed, 0x73, 0xe1, 0x74,
0x6f, 0x3b, 0x6d, 0x6f, 0x6b, 0x254, 0x6c, 0x254, 0x20, 0x79, 0x61, 0x20,
0x6d, 0xed, 0x6e, 0xe9, 0x69, 0x3b, 0x6d, 0x6f, 0x6b, 0x254, 0x6c, 0x254,
0x20, 0x79, 0x61, 0x20, 0x6d, 0xed, 0x74, 0xe1, 0x6e, 0x6f, 0x3b, 0x6d,
0x70, 0x254, 0x301, 0x73, 0x254, 0x65, 0x79, 0x65, 0x3b, 0x79, 0x62, 0x6f,
0x3b, 0x6d, 0x62, 0x6c, 0x3b, 0x6d, 0x73, 0x74, 0x3b, 0x6d, 0x69, 0x6e,
0x3b, 0x6d, 0x74, 0x6e, 0x3b, 0x6d, 0x70, 0x73, 0x65, 0x3b, 0x79, 0x3b,
0x6d, 0x3b, 0x6d, 0x3b, 0x6d, 0x3b, 0x6d, 0x3b, 0x70, 0x73, 0x65, 0x6b,
0x6d, 0x61, 0x64, 0x69, 0x65, 0x6e, 0x69, 0x73, 0x3b, 0x70, 0x69, 0x72,
0x6d, 0x61, 0x64, 0x69, 0x65, 0x6e, 0x69, 0x73, 0x3b, 0x61, 0x6e, 0x74,
0x72, 0x61, 0x64, 0x69, 0x65, 0x6e, 0x69, 0x73, 0x3b, 0x74, 0x72, 0x65,
0x10d, 0x69, 0x61, 0x64, 0x69, 0x65, 0x6e, 0x69, 0x73, 0x3b, 0x6b, 0x65,
0x74, 0x76, 0x69, 0x72, 0x74, 0x61, 0x64, 0x69, 0x65, 0x6e, 0x69, 0x73,
0x3b, 0x70, 0x65, 0x6e, 0x6b, 0x74, 0x61, 0x64, 0x69, 0x65, 0x6e, 0x69,
0x73, 0x3b, 0x161, 0x65, 0x161, 0x74, 0x61, 0x64, 0x69, 0x65, 0x6e, 0x69,
0x73, 0x73, 0x6b, 0x3b, 0x70, 0x72, 0x3b, 0x61, 0x6e, 0x3b, 0x74, 0x72,
0x3b, 0x6b, 0x74, 0x3b, 0x70, 0x6e, 0x3b, 0x161, 0x74, 0x53, 0x3b, 0x50,
0x3b, 0x41, 0x3b, 0x54, 0x3b, 0x4b, 0x3b, 0x50, 0x3b, 0x160, 0x6e, 0x6a,
0x65, 0x17a, 0x65, 0x6c, 0x61, 0x3b, 0x70, 0xf3, 0x6e, 0x6a, 0x65, 0x17a,
0x65, 0x6c, 0x65, 0x3b, 0x77, 0x61, 0x142, 0x74, 0x6f, 0x72, 0x61, 0x3b,
0x73, 0x72, 0x6a, 0x6f, 0x64, 0x61, 0x3b, 0x73, 0x74, 0x77, 0xf3, 0x72,
0x74, 0x6b, 0x3b, 0x70, 0x11b, 0x74, 0x6b, 0x3b, 0x73, 0x6f, 0x62, 0x6f,
0x74, 0x61, 0x6e, 0x6a, 0x65, 0x3b, 0x70, 0xf3, 0x6e, 0x3b, 0x77, 0x61,
0x142, 0x3b, 0x73, 0x72, 0x6a, 0x3b, 0x73, 0x74, 0x77, 0x3b, 0x70, 0x11b,
0x74, 0x3b, 0x73, 0x6f, 0x62, 0x6e, 0x3b, 0x70, 0x3b, 0x77, 0x3b, 0x73,
0x3b, 0x73, 0x3b, 0x70, 0x3b, 0x73, 0x53, 0xfc, 0x6e, 0x6e, 0x64, 0x61,
0x67, 0x3b, 0x4d, 0x61, 0x61, 0x6e, 0x64, 0x61, 0x67, 0x3b, 0x44, 0x69,
0x6e, 0x67, 0x73, 0x64, 0x61, 0x67, 0x3b, 0x4d, 0x69, 0x64, 0x64, 0x65,
0x77, 0x65, 0x6b, 0x65, 0x6e, 0x3b, 0x44, 0x75, 0x6e, 0x6e, 0x65, 0x72,
0x73, 0x64, 0x61, 0x67, 0x3b, 0x46, 0x72, 0x65, 0x65, 0x64, 0x61, 0x67,
0x3b, 0x53, 0xfc, 0x6e, 0x6e, 0x61, 0x76, 0x65, 0x6e, 0x64, 0x53, 0xfc,
0x2e, 0x3b, 0x4d, 0x61, 0x2e, 0x3b, 0x44, 0x69, 0x2e, 0x3b, 0x4d, 0x69,
0x2e, 0x3b, 0x44, 0x75, 0x2e, 0x3b, 0x46, 0x72, 0x2e, 0x3b, 0x53, 0x61,
0x2e, 0x4c, 0x75, 0x6d, 0x69, 0x6e, 0x67, 0x75, 0x3b, 0x4e, 0x6b, 0x6f,
0x64, 0x79, 0x61, 0x3b, 0x4e, 0x64, 0xe0, 0x61, 0x79, 0xe0, 0x3b, 0x4e,
0x64, 0x61, 0x6e, 0x67, 0xf9, 0x3b, 0x4e, 0x6a, 0xf2, 0x77, 0x61, 0x3b,
0x4e, 0x67, 0xf2, 0x76, 0x79, 0x61, 0x3b, 0x4c, 0x75, 0x62, 0x69, 0x6e,
0x67, 0x75, 0x4c, 0x75, 0x6d, 0x3b, 0x4e, 0x6b, 0x6f, 0x3b, 0x4e, 0x64,
0x79, 0x3b, 0x4e, 0x64, 0x67, 0x3b, 0x4e, 0x6a, 0x77, 0x3b, 0x4e, 0x67,
0x76, 0x3b, 0x4c, 0x75, 0x62, 0x4c, 0x3b, 0x4e, 0x3b, 0x4e, 0x3b, 0x4e,
0x3b, 0x4e, 0x3b, 0x4e, 0x3b, 0x4c, 0x4a, 0x75, 0x6d, 0x61, 0x70, 0x69,
0x6c, 0x3b, 0x57, 0x75, 0x6f, 0x6b, 0x20, 0x54, 0x69, 0x63, 0x68, 0x3b,
0x54, 0x69, 0x63, 0x68, 0x20, 0x41, 0x72, 0x69, 0x79, 0x6f, 0x3b, 0x54,
0x69, 0x63, 0x68, 0x20, 0x41, 0x64, 0x65, 0x6b, 0x3b, 0x54, 0x69, 0x63,
0x68, 0x20, 0x41, 0x6e, 0x67, 0x2019, 0x77, 0x65, 0x6e, 0x3b, 0x54, 0x69,
0x63, 0x68, 0x20, 0x41, 0x62, 0x69, 0x63, 0x68, 0x3b, 0x4e, 0x67, 0x65,
0x73, 0x6f, 0x4a, 0x4d, 0x50, 0x3b, 0x57, 0x55, 0x54, 0x3b, 0x54, 0x41,
0x52, 0x3b, 0x54, 0x41, 0x44, 0x3b, 0x54, 0x41, 0x4e, 0x3b, 0x54, 0x41,
0x42, 0x3b, 0x4e, 0x47, 0x53, 0x4a, 0x3b, 0x57, 0x3b, 0x54, 0x3b, 0x54,
0x3b, 0x54, 0x3b, 0x54, 0x3b, 0x4e, 0x53, 0x6f, 0x6e, 0x6e, 0x64, 0x65,
0x67, 0x3b, 0x4d, 0xe9, 0x69, 0x6e, 0x64, 0x65, 0x67, 0x3b, 0x44, 0xeb,
0x6e, 0x73, 0x63, 0x68, 0x64, 0x65, 0x67, 0x3b, 0x4d, 0xeb, 0x74, 0x74,
0x77, 0x6f, 0x63, 0x68, 0x3b, 0x44, 0x6f, 0x6e, 0x6e, 0x65, 0x73, 0x63,
0x68, 0x64, 0x65, 0x67, 0x3b, 0x46, 0x72, 0x65, 0x69, 0x64, 0x65, 0x67,
0x3b, 0x53, 0x61, 0x6d, 0x73, 0x63, 0x68, 0x64, 0x65, 0x67, 0x53, 0x6f,
0x6e, 0x3b, 0x4d, 0xe9, 0x69, 0x3b, 0x44, 0xeb, 0x6e, 0x3b, 0x4d, 0xeb,
0x74, 0x3b, 0x44, 0x6f, 0x6e, 0x3b, 0x46, 0x72, 0x65, 0x3b, 0x53, 0x61,
0x6d, 0x53, 0x6f, 0x6e, 0x2e, 0x3b, 0x4d, 0xe9, 0x69, 0x2e, 0x3b, 0x44,
0xeb, 0x6e, 0x2e, 0x3b, 0x4d, 0xeb, 0x74, 0x2e, 0x3b, 0x44, 0x6f, 0x6e,
0x2e, 0x3b, 0x46, 0x72, 0x65, 0x2e, 0x3b, 0x53, 0x61, 0x6d, 0x2e, 0x4a,
0x75, 0x6d, 0x61, 0x70, 0x69, 0x72, 0x69, 0x3b, 0x4a, 0x75, 0x6d, 0x61,
0x74, 0x61, 0x74, 0x75, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x6e, 0x6e, 0x65,
0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x74, 0x61, 0x6e, 0x6f, 0x3b, 0x4d, 0x75,
0x72, 0x77, 0x61, 0x20, 0x77, 0x61, 0x20, 0x4b, 0x61, 0x6e, 0x6e, 0x65,
0x3b, 0x4d, 0x75, 0x72, 0x77, 0x61, 0x20, 0x77, 0x61, 0x20, 0x4b, 0x61,
0x74, 0x61, 0x6e, 0x6f, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x6d, 0x6f, 0x73,
0x69, 0x4a, 0x32, 0x3b, 0x4a, 0x33, 0x3b, 0x4a, 0x34, 0x3b, 0x4a, 0x35,
0x3b, 0x41, 0x6c, 0x3b, 0x49, 0x6a, 0x3b, 0x4a, 0x31, 0x43d, 0x435, 0x434,
0x435, 0x43b, 0x430, 0x3b, 0x43f, 0x43e, 0x43d, 0x435, 0x434, 0x435, 0x43b, 0x43d,
0x438, 0x43a, 0x3b, 0x432, 0x442, 0x43e, 0x440, 0x43d, 0x438, 0x43a, 0x3b, 0x441,
0x440, 0x435, 0x434, 0x430, 0x3b, 0x447, 0x435, 0x442, 0x432, 0x440, 0x442, 0x43e,
0x43a, 0x3b, 0x43f, 0x435, 0x442, 0x43e, 0x43a, 0x3b, 0x441, 0x430, 0x431, 0x43e,
0x442, 0x430, 0x43d, 0x435, 0x434, 0x2e, 0x3b, 0x43f, 0x43e, 0x43d, 0x2e, 0x3b,
0x432, 0x442, 0x43e, 0x2e, 0x3b, 0x441, 0x440, 0x435, 0x2e, 0x3b, 0x447, 0x435,
0x442, 0x2e, 0x3b, 0x43f, 0x435, 0x442, 0x2e, 0x3b, 0x441, 0x430, 0x431, 0x2e,
0x4a, 0x75, 0x6d, 0x61, 0x70, 0x69, 0x6c, 0x79, 0x69, 0x3b, 0x4a, 0x75,
0x6d, 0x61, 0x74, 0x61, 0x74, 0x75, 0x75, 0x3b, 0x4a, 0x75, 0x6d, 0x61,
0x6e, 0x6e, 0x65, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x74, 0x61, 0x6e, 0x75,
0x3b, 0x41, 0x6c, 0x68, 0x61, 0x6d, 0x69, 0x73, 0x69, 0x3b, 0x49, 0x6a,
0x75, 0x6d, 0x61, 0x61, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x6d, 0x6f, 0x73,
0x69, 0x4a, 0x70, 0x69, 0x3b, 0x4a, 0x74, 0x74, 0x3b, 0x4a, 0x6e, 0x6e,
0x3b, 0x4a, 0x74, 0x6e, 0x3b, 0x41, 0x6c, 0x68, 0x3b, 0x49, 0x6a, 0x75,
0x3b, 0x4a, 0x6d, 0x6f, 0x930, 0x935, 0x93f, 0x20, 0x926, 0x93f, 0x928, 0x3b,
0x938, 0x94b, 0x92e, 0x20, 0x926, 0x93f, 0x928, 0x3b, 0x92e, 0x902, 0x917, 0x932,
0x20, 0x926, 0x93f, 0x928, 0x3b, 0x92c, 0x941, 0x927, 0x20, 0x926, 0x93f, 0x928,
0x3b, 0x92c, 0x943, 0x939, 0x938, 0x94d, 0x92a, 0x924, 0x93f, 0x20, 0x926, 0x93f,
0x928, 0x3b, 0x936, 0x941, 0x915, 0x94d, 0x930, 0x20, 0x926, 0x93f, 0x928, 0x3b,
0x936, 0x928, 0x93f, 0x20, 0x926, 0x93f, 0x928, 0x53, 0x61, 0x62, 0x61, 0x74,
0x6f, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x74, 0x61, 0x74, 0x75, 0x3b, 0x4a,
0x75, 0x6d, 0x61, 0x6e, 0x6e, 0x65, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x74,
0x61, 0x6e, 0x6f, 0x3b, 0x41, 0x72, 0x61, 0x68, 0x61, 0x6d, 0x69, 0x73,
0x69, 0x3b, 0x49, 0x6a, 0x75, 0x6d, 0x61, 0x61, 0x3b, 0x4a, 0x75, 0x6d,
0x61, 0x6d, 0x6f, 0x73, 0x69, 0x53, 0x61, 0x62, 0x3b, 0x4a, 0x74, 0x74,
0x3b, 0x4a, 0x6e, 0x6e, 0x3b, 0x4a, 0x74, 0x6e, 0x3b, 0x41, 0x72, 0x61,
0x3b, 0x49, 0x6a, 0x75, 0x3b, 0x4a, 0x6d, 0x6f, 0x53, 0x3b, 0x4a, 0x3b,
0x4a, 0x3b, 0x4a, 0x3b, 0x41, 0x3b, 0x49, 0x3b, 0x4a, 0x4c, 0x69, 0x64,
0x75, 0x76, 0x61, 0x20, 0x6c, 0x79, 0x61, 0x70, 0x69, 0x6c, 0x69, 0x3b,
0x4c, 0x69, 0x64, 0x75, 0x76, 0x61, 0x20, 0x6c, 0x79, 0x61, 0x74, 0x61,
0x74, 0x75, 0x3b, 0x4c, 0x69, 0x64, 0x75, 0x76, 0x61, 0x20, 0x6c, 0x79,
0x61, 0x6e, 0x63, 0x68, 0x65, 0x63, 0x68, 0x69, 0x3b, 0x4c, 0x69, 0x64,
0x75, 0x76, 0x61, 0x20, 0x6c, 0x79, 0x61, 0x6e, 0x6e, 0x79, 0x61, 0x6e,
0x6f, 0x3b, 0x4c, 0x69, 0x64, 0x75, 0x76, 0x61, 0x20, 0x6c, 0x79, 0x61,
0x6e, 0x6e, 0x79, 0x61, 0x6e, 0x6f, 0x20, 0x6e, 0x61, 0x20, 0x6c, 0x69,
0x6e, 0x6a, 0x69, 0x3b, 0x4c, 0x69, 0x64, 0x75, 0x76, 0x61, 0x20, 0x6c,
0x79, 0x61, 0x6e, 0x6e, 0x79, 0x61, 0x6e, 0x6f, 0x20, 0x6e, 0x61, 0x20,
0x6d, 0x61, 0x76, 0x69, 0x6c, 0x69, 0x3b, 0x4c, 0x69, 0x64, 0x75, 0x76,
0x61, 0x20, 0x6c, 0x69, 0x74, 0x61, 0x6e, 0x64, 0x69, 0x4c, 0x6c, 0x32,
0x3b, 0x4c, 0x6c, 0x33, 0x3b, 0x4c, 0x6c, 0x34, 0x3b, 0x4c, 0x6c, 0x35,
0x3b, 0x4c, 0x6c, 0x36, 0x3b, 0x4c, 0x6c, 0x37, 0x3b, 0x4c, 0x6c, 0x31,
0x32, 0x3b, 0x33, 0x3b, 0x34, 0x3b, 0x35, 0x3b, 0x36, 0x3b, 0x37, 0x3b,
0x31, 0x41, 0x6c, 0x61, 0x68, 0x61, 0x64, 0x79, 0x3b, 0x41, 0x6c, 0x61,
0x74, 0x73, 0x69, 0x6e, 0x61, 0x69, 0x6e, 0x79, 0x3b, 0x54, 0x61, 0x6c,
0x61, 0x74, 0x61, 0x3b, 0x41, 0x6c, 0x61, 0x72, 0x6f, 0x62, 0x69, 0x61,
0x3b, 0x41, 0x6c, 0x61, 0x6b, 0x61, 0x6d, 0x69, 0x73, 0x79, 0x3b, 0x5a,
0x6f, 0x6d, 0x61, 0x3b, 0x41, 0x73, 0x61, 0x62, 0x6f, 0x74, 0x73, 0x79,
0x41, 0x6c, 0x61, 0x68, 0x3b, 0x41, 0x6c, 0x61, 0x74, 0x73, 0x3b, 0x54,
0x61, 0x6c, 0x3b, 0x41, 0x6c, 0x61, 0x72, 0x3b, 0x41, 0x6c, 0x61, 0x6b,
0x3b, 0x5a, 0x6f, 0x6d, 0x3b, 0x41, 0x73, 0x61, 0x62, 0x41, 0x3b, 0x41,
0x3b, 0x54, 0x3b, 0x41, 0x3b, 0x41, 0x3b, 0x5a, 0x3b, 0x41, 0xd1e, 0xd3e,
0xd2f, 0xd31, 0xd3e, 0xd34, 0xd4d, 0x200c, 0xd1a, 0x3b, 0xd24, 0xd3f, 0xd19, 0xd4d,
0xd15, 0xd33, 0xd3e, 0xd34, 0xd4d, 0x200c, 0xd1a, 0x3b, 0xd1a, 0xd4a, 0xd35, 0xd4d,
0xd35, 0xd3e, 0xd34, 0xd4d, 0x200c, 0xd1a, 0x3b, 0xd2c, 0xd41, 0xd27, 0xd28, 0xd3e,
0xd34, 0xd4d, 0x200c, 0xd1a, 0x3b, 0xd35, 0xd4d, 0xd2f, 0xd3e, 0xd34, 0xd3e, 0xd34,
0xd4d, 0x200c, 0xd1a, 0x3b, 0xd35, 0xd46, 0xd33, 0xd4d, 0xd33, 0xd3f, 0xd2f, 0xd3e,
0xd34, 0xd4d, 0x200c, 0xd1a, 0x3b, 0xd36, 0xd28, 0xd3f, 0xd2f, 0xd3e, 0xd34, 0xd4d,
0x200c, 0xd1a, 0xd1e, 0xd3e, 0xd2f, 0xd31, 0xd3e, 0xd34, 0xd4d, 0x200c, 0xd1a, 0x3b,
0xd24, 0xd3f, 0xd19, 0xd4d, 0xd15, 0xd33, 0xd3e, 0xd34, 0xd4d, 0x200c, 0xd1a, 0x3b,
0xd1a, 0xd4a, 0xd35, 0xd4d, 0xd35, 0xd3e, 0xd34, 0xd4d, 0xd1a, 0x3b, 0xd2c, 0xd41,
0xd27, 0xd28, 0xd3e, 0xd34, 0xd4d, 0x200c, 0xd1a, 0x3b, 0xd35, 0xd4d, 0xd2f, 0xd3e,
0xd34, 0xd3e, 0xd34, 0xd4d, 0x200c, 0xd1a, 0x3b, 0xd35, 0xd46, 0xd33, 0xd4d, 0xd33,
0xd3f, 0xd2f, 0xd3e, 0xd34, 0xd4d, 0x200c, 0xd1a, 0x3b, 0xd36, 0xd28, 0xd3f, 0xd2f,
0xd3e, 0xd34, 0xd4d, 0x200c, 0xd1a, 0xd1e, 0xd3e, 0xd2f, 0xd7c, 0x3b, 0xd24, 0xd3f,
0xd19, 0xd4d, 0xd15, 0xd7e, 0x3b, 0xd1a, 0xd4a, 0xd35, 0xd4d, 0xd35, 0x3b, 0xd2c,
0xd41, 0xd27, 0xd7b, 0x3b, 0xd35, 0xd4d, 0xd2f, 0xd3e, 0xd34, 0xd02, 0x3b, 0xd35,
0xd46, 0xd33, 0xd4d, 0xd33, 0xd3f, 0x3b, 0xd36, 0xd28, 0xd3f, 0xd1e, 0xd3e, 0x3b,
0xd24, 0xd3f, 0x3b, 0xd1a, 0xd4a, 0x3b, 0xd2c, 0xd41, 0x3b, 0xd35, 0xd4d, 0xd2f,
0xd3e, 0x3b, 0xd35, 0xd46, 0x3b, 0xd36, 0xd1e, 0x3b, 0xd24, 0xd3f, 0x3b, 0xd1a,
0xd4a, 0x3b, 0xd2c, 0xd41, 0x3b, 0xd35, 0xd4d, 0xd2f, 0xd3e, 0x3b, 0xd35, 0xd46,
0x3b, 0xd36, 0x41, 0x68, 0x61, 0x64, 0x3b, 0x49, 0x73, 0x6e, 0x69, 0x6e,
0x3b, 0x53, 0x65, 0x6c, 0x61, 0x73, 0x61, 0x3b, 0x52, 0x61, 0x62, 0x75,
0x3b, 0x4b, 0x68, 0x61, 0x6d, 0x69, 0x73, 0x3b, 0x4a, 0x75, 0x6d, 0x61,
0x61, 0x74, 0x3b, 0x53, 0x61, 0x62, 0x74, 0x75, 0x41, 0x68, 0x64, 0x3b,
0x49, 0x73, 0x6e, 0x3b, 0x53, 0x65, 0x6c, 0x3b, 0x52, 0x61, 0x62, 0x3b,
0x4b, 0x68, 0x61, 0x3b, 0x4a, 0x75, 0x6d, 0x3b, 0x53, 0x61, 0x62, 0x41,
0x3b, 0x49, 0x3b, 0x53, 0x3b, 0x52, 0x3b, 0x4b, 0x3b, 0x4a, 0x3b, 0x53,
0x627, 0x62d, 0x62f, 0x3b, 0x627, 0x62b, 0x646, 0x64a, 0x646, 0x3b, 0x62b, 0x644,
0x627, 0x62b, 0x3b, 0x631, 0x627, 0x628, 0x648, 0x3b, 0x62e, 0x645, 0x64a, 0x633,
0x3b, 0x62c, 0x645, 0x639, 0x629, 0x3b, 0x633, 0x628, 0x62a, 0x648, 0x49, 0x6c,
0x2d, 0x126, 0x61, 0x64, 0x64, 0x3b, 0x49, 0x74, 0x2d, 0x54, 0x6e, 0x65,
0x6a, 0x6e, 0x3b, 0x49, 0x74, 0x2d, 0x54, 0x6c, 0x69, 0x65, 0x74, 0x61,
0x3b, 0x4c, 0x2d, 0x45, 0x72, 0x62, 0x67, 0x127, 0x61, 0x3b, 0x49, 0x6c,
0x2d, 0x126, 0x61, 0x6d, 0x69, 0x73, 0x3b, 0x49, 0x6c, 0x2d, 0x120, 0x69,
0x6d, 0x67, 0x127, 0x61, 0x3b, 0x49, 0x73, 0x2d, 0x53, 0x69, 0x62, 0x74,
0x126, 0x61, 0x64, 0x3b, 0x54, 0x6e, 0x65, 0x3b, 0x54, 0x6c, 0x69, 0x3b,
0x45, 0x72, 0x62, 0x3b, 0x126, 0x61, 0x6d, 0x3b, 0x120, 0x69, 0x6d, 0x3b,
0x53, 0x69, 0x62, 0x126, 0x64, 0x3b, 0x54, 0x6e, 0x3b, 0x54, 0x6c, 0x3b,
0x45, 0x72, 0x3b, 0x126, 0x6d, 0x3b, 0x120, 0x6d, 0x3b, 0x53, 0x62, 0x126,
0x64, 0x3b, 0x54, 0x3b, 0x54, 0x6c, 0x3b, 0x45, 0x72, 0x3b, 0x126, 0x6d,
0x3b, 0x120, 0x6d, 0x3b, 0x53, 0x62, 0x9a8, 0x9cb, 0x982, 0x9ae, 0x9be, 0x987,
0x99c, 0x9bf, 0x982, 0x3b, 0x9a8, 0x9bf, 0x982, 0x9a5, 0x9cc, 0x995, 0x9be, 0x9ac,
0x9be, 0x3b, 0x9b2, 0x9c8, 0x9ac, 0x9be, 0x995, 0x9aa, 0x9cb, 0x995, 0x9aa, 0x9be,
0x3b, 0x9af, 0x9bc, 0x9c1, 0x9ae, 0x9b6, 0x995, 0x9c8, 0x9b6, 0x9be, 0x3b, 0x9b6,
0x997, 0x9cb, 0x9b2, 0x9b6, 0x9c7, 0x9a8, 0x3b, 0x987, 0x9b0, 0x9be, 0x987, 0x3b,
0x9a5, 0x9be, 0x982, 0x99c, 0x9a8, 0x9cb, 0x3b, 0x9a8, 0x9bf, 0x982, 0x3b, 0x9b2,
0x9c8, 0x3b, 0x9af, 0x9bc, 0x9c1, 0x9ae, 0x3b, 0x9b6, 0x997, 0x3b, 0x987, 0x9b0,
0x9be, 0x3b, 0x9a5, 0x9be, 0x982, 0x9a8, 0x9cb, 0x982, 0x3b, 0x9a8, 0x9bf, 0x982,
0x3b, 0x9b2, 0x9c8, 0x9ac, 0x9be, 0x3b, 0x9af, 0x9bc, 0x9c1, 0x9ae, 0x3b, 0x9b6,
0x997, 0x9cb, 0x3b, 0x987, 0x9b0, 0x9be, 0x3b, 0x9a5, 0x9be, 0x982, 0x4a, 0x65,
0x64, 0x6f, 0x6f, 0x6e, 0x65, 0x65, 0x3b, 0x4a, 0x65, 0x6c, 0x68, 0x65,
0x69, 0x6e, 0x3b, 0x4a, 0x65, 0x6d, 0x61, 0x79, 0x72, 0x74, 0x3b, 0x4a,
0x65, 0x72, 0x63, 0x65, 0x61, 0x6e, 0x3b, 0x4a, 0x65, 0x72, 0x64, 0x65,
0x69, 0x6e, 0x3b, 0x4a, 0x65, 0x68, 0x65, 0x69, 0x6e, 0x65, 0x79, 0x3b,
0x4a, 0x65, 0x73, 0x61, 0x72, 0x6e, 0x4a, 0x65, 0x64, 0x3b, 0x4a, 0x65,
0x6c, 0x3b, 0x4a, 0x65, 0x6d, 0x3b, 0x4a, 0x65, 0x72, 0x63, 0x3b, 0x4a,
0x65, 0x72, 0x64, 0x3b, 0x4a, 0x65, 0x68, 0x3b, 0x4a, 0x65, 0x73, 0x52,
0x101, 0x74, 0x61, 0x70, 0x75, 0x3b, 0x4d, 0x61, 0x6e, 0x65, 0x3b, 0x54,
0x16b, 0x72, 0x65, 0x69, 0x3b, 0x57, 0x65, 0x6e, 0x65, 0x72, 0x65, 0x69,
0x3b, 0x54, 0x101, 0x69, 0x74, 0x65, 0x3b, 0x50, 0x61, 0x72, 0x61, 0x69,
0x72, 0x65, 0x3b, 0x52, 0x101, 0x68, 0x6f, 0x72, 0x6f, 0x69, 0x52, 0x101,
0x74, 0x3b, 0x4d, 0x61, 0x6e, 0x3b, 0x54, 0x16b, 0x72, 0x3b, 0x57, 0x65,
0x6e, 0x3b, 0x54, 0x101, 0x69, 0x3b, 0x50, 0x61, 0x72, 0x3b, 0x52, 0x101,
0x68, 0x52, 0x74, 0x3b, 0x4d, 0x3b, 0x54, 0x3b, 0x57, 0x3b, 0x54, 0x3b,
0x50, 0x3b, 0x52, 0x68, 0x930, 0x935, 0x93f, 0x935, 0x93e, 0x930, 0x3b, 0x938,
0x94b, 0x92e, 0x935, 0x93e, 0x930, 0x3b, 0x92e, 0x902, 0x917, 0x933, 0x935, 0x93e,
0x930, 0x3b, 0x92c, 0x941, 0x927, 0x935, 0x93e, 0x930, 0x3b, 0x917, 0x941, 0x930,
0x941, 0x935, 0x93e, 0x930, 0x3b, 0x936, 0x941, 0x915, 0x94d, 0x930, 0x935, 0x93e,
0x930, 0x3b, 0x936, 0x928, 0x93f, 0x935, 0x93e, 0x930, 0x930, 0x935, 0x93f, 0x3b,
0x938, 0x94b, 0x92e, 0x3b, 0x92e, 0x902, 0x917, 0x933, 0x3b, 0x92c, 0x941, 0x927,
0x3b, 0x917, 0x941, 0x930, 0x941, 0x3b, 0x936, 0x941, 0x915, 0x94d, 0x930, 0x3b,
0x936, 0x928, 0x93f, 0x4a, 0x75, 0x6d, 0x61, 0x70, 0xed, 0x6c, 0xed, 0x3b,
0x4a, 0x75, 0x6d, 0x61, 0x74, 0xe1, 0x74, 0x75, 0x3b, 0x4a, 0x75, 0x6d,
0x61, 0x6e, 0x65, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x74, 0xe1, 0x6e, 0x254,
0x3b, 0x41, 0x6c, 0x61, 0xe1, 0x6d, 0x69, 0x73, 0x69, 0x3b, 0x4a, 0x75,
0x6d, 0xe1, 0x61, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x6d, 0xf3, 0x73, 0x69,
0x4b, 0x69, 0x75, 0x6d, 0x69, 0x61, 0x3b, 0x4d, 0x75, 0x72, 0x61, 0x6d,
0x75, 0x6b, 0x6f, 0x3b, 0x57, 0x61, 0x69, 0x72, 0x69, 0x3b, 0x57, 0x65,
0x74, 0x68, 0x61, 0x74, 0x75, 0x3b, 0x57, 0x65, 0x6e, 0x61, 0x3b, 0x57,
0x65, 0x74, 0x61, 0x6e, 0x6f, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x6d, 0x6f,
0x73, 0x69, 0x4b, 0x49, 0x55, 0x3b, 0x4d, 0x52, 0x41, 0x3b, 0x57, 0x41,
0x49, 0x3b, 0x57, 0x45, 0x54, 0x3b, 0x57, 0x45, 0x4e, 0x3b, 0x57, 0x54,
0x4e, 0x3b, 0x4a, 0x55, 0x4d, 0x4b, 0x3b, 0x4d, 0x3b, 0x57, 0x3b, 0x57,
0x3b, 0x57, 0x3b, 0x57, 0x3b, 0x4a, 0x41, 0x6e, 0x65, 0x67, 0x20, 0x31,
0x3b, 0x41, 0x6e, 0x65, 0x67, 0x20, 0x32, 0x3b, 0x41, 0x6e, 0x65, 0x67,
0x20, 0x33, 0x3b, 0x41, 0x6e, 0x65, 0x67, 0x20, 0x34, 0x3b, 0x41, 0x6e,
0x65, 0x67, 0x20, 0x35, 0x3b, 0x41, 0x6e, 0x65, 0x67, 0x20, 0x36, 0x3b,
0x41, 0x6e, 0x65, 0x67, 0x20, 0x37, 0x41, 0x31, 0x3b, 0x41, 0x32, 0x3b,
0x41, 0x33, 0x3b, 0x41, 0x34, 0x3b, 0x41, 0x35, 0x3b, 0x41, 0x36, 0x3b,
0x41, 0x37, 0x41d, 0x44f, 0x43c, 0x3b, 0x414, 0x430, 0x432, 0x430, 0x430, 0x3b,
0x41c, 0x44f, 0x433, 0x43c, 0x430, 0x440, 0x3b, 0x41b, 0x445, 0x430, 0x433, 0x432,
0x430, 0x3b, 0x41f, 0x4af, 0x440, 0x44d, 0x432, 0x3b, 0x411, 0x430, 0x430, 0x441,
0x430, 0x43d, 0x3b, 0x411, 0x44f, 0x43c, 0x431, 0x430, 0x43d, 0x44f, 0x43c, 0x3b,
0x434, 0x430, 0x432, 0x430, 0x430, 0x3b, 0x43c, 0x44f, 0x433, 0x43c, 0x430, 0x440,
0x3b, 0x43b, 0x445, 0x430, 0x433, 0x432, 0x430, 0x3b, 0x43f, 0x4af, 0x440, 0x44d,
0x432, 0x3b, 0x431, 0x430, 0x430, 0x441, 0x430, 0x43d, 0x3b, 0x431, 0x44f, 0x43c,
0x431, 0x430, 0x41d, 0x44f, 0x3b, 0x414, 0x430, 0x3b, 0x41c, 0x44f, 0x3b, 0x41b,
0x445, 0x3b, 0x41f, 0x4af, 0x3b, 0x411, 0x430, 0x3b, 0x411, 0x44f, 0x1828, 0x1822,
0x182e, 0x180e, 0x1820, 0x3b, 0x1833, 0x1820, 0x1838, 0x1820, 0x3b, 0x182e, 0x1822, 0x1820,
0x1820, 0x182e, 0x1820, 0x1837, 0x3b, 0x1840, 0x1820, 0x182d, 0x182a, 0x1820, 0x3b, 0x182b,
0x1826, 0x1837, 0x182a, 0x1826, 0x3b, 0x182a, 0x1820, 0x1830, 0x1820, 0x1829, 0x3b, 0x182a,
0x1822, 0x182e, 0x182a, 0x1820, 0x1828, 0x1822, 0x3b, 0x1833, 0x1820, 0x3b, 0x182e, 0x1822,
0x182d, 0x3b, 0x1840, 0x1820, 0x3b, 0x182b, 0x1826, 0x1837, 0x3b, 0x182a, 0x1820, 0x3b,
0x182a, 0x1822, 0x182e, 0x1828, 0x1822, 0x3b, 0x1832, 0x1820, 0x3b, 0x182e, 0x1822, 0x182d,
0x3b, 0x1840, 0x1820, 0x3b, 0x182b, 0x1825, 0x1837, 0x3b, 0x182a, 0x1820, 0x3b, 0x182a,
0x1822, 0x182e, 0x1828, 0x1822, 0x3b, 0x1833, 0x1820, 0x3b, 0x182e, 0x1822, 0x182d, 0x3b,
0x1840, 0x1820, 0x3b, 0x182b, 0x1825, 0x1837, 0x3b, 0x182a, 0x1820, 0x3b, 0x182a, 0x1822,
0x182e, 0x64, 0x69, 0x6d, 0x61, 0x6e, 0x73, 0x3b, 0x6c, 0x69, 0x6e, 0x64,
0x69, 0x3b, 0x6d, 0x61, 0x72, 0x64, 0x69, 0x3b, 0x6d, 0x65, 0x72, 0x6b,
0x72, 0x65, 0x64, 0x69, 0x3b, 0x7a, 0x65, 0x64, 0x69, 0x3b, 0x76, 0x61,
0x6e, 0x64, 0x72, 0x65, 0x64, 0x69, 0x3b, 0x73, 0x61, 0x6d, 0x64, 0x69,
0x64, 0x69, 0x6d, 0x3b, 0x6c, 0x69, 0x6e, 0x3b, 0x6d, 0x61, 0x72, 0x3b,
0x6d, 0x65, 0x72, 0x3b, 0x7a, 0x65, 0x3b, 0x76, 0x61, 0x6e, 0x3b, 0x73,
0x61, 0x6d, 0x64, 0x3b, 0x6c, 0x3b, 0x6d, 0x3b, 0x6d, 0x3b, 0x7a, 0x3b,
0x76, 0x3b, 0x73, 0x43, 0x6f, 0x6d, 0x2019, 0x79, 0x61, 0x6b, 0x6b, 0x65,
0x3b, 0x43, 0x6f, 0x6d, 0x6c, 0x61, 0x61, 0x257, 0x69, 0x69, 0x3b, 0x43,
0x6f, 0x6d, 0x7a, 0x79, 0x69, 0x69, 0x257, 0x69, 0x69, 0x3b, 0x43, 0x6f,
0x6d, 0x6b, 0x6f, 0x6c, 0x6c, 0x65, 0x3b, 0x43, 0x6f, 0x6d, 0x6b, 0x61,
0x6c, 0x64, 0x1dd, 0x253, 0x6c, 0x69, 0x69, 0x3b, 0x43, 0x6f, 0x6d, 0x67,
0x61, 0x69, 0x73, 0x75, 0x75, 0x3b, 0x43, 0x6f, 0x6d, 0x7a, 0x79, 0x65,
0x253, 0x73, 0x75, 0x75, 0x43, 0x79, 0x61, 0x3b, 0x43, 0x6c, 0x61, 0x3b,
0x43, 0x7a, 0x69, 0x3b, 0x43, 0x6b, 0x6f, 0x3b, 0x43, 0x6b, 0x61, 0x3b,
0x43, 0x67, 0x61, 0x3b, 0x43, 0x7a, 0x65, 0x59, 0x3b, 0x4c, 0x3b, 0x5a,
0x3b, 0x4f, 0x3b, 0x41, 0x3b, 0x47, 0x3b, 0x45, 0x4e, 0x65, 0x74, 0x74,
0x76, 0x2bc, 0x63, 0x61, 0x6b, 0x6f, 0x3b, 0x45, 0x6e, 0x68, 0x76, 0x74,
0x65, 0x63, 0x65, 0x73, 0x6b, 0x76, 0x3b, 0x45, 0x6e, 0x68, 0x76, 0x74,
0x65, 0x63, 0x65, 0x73, 0x6b, 0x76, 0x20, 0x45, 0x6e, 0x68, 0x76, 0x79,
0x76, 0x74, 0x6b, 0x65, 0x3b, 0x45, 0x6e, 0x6e, 0x76, 0x72, 0x6b, 0x76,
0x70, 0x76, 0x3b, 0x45, 0x6e, 0x6e, 0x76, 0x72, 0x6b, 0x76, 0x70, 0x76,
0x20, 0x45, 0x6e, 0x68, 0x76, 0x79, 0x76, 0x74, 0x6b, 0x65, 0x3b, 0x4e,
0x61, 0x6b, 0x20, 0x4f, 0x6b, 0x6b, 0x6f, 0x73, 0x6b, 0x76, 0x20, 0x4e,
0x65, 0x74, 0x74, 0x76, 0x3b, 0x4e, 0x65, 0x74, 0x74, 0x76, 0x20, 0x43,
0x61, 0x6b, 0x2bc, 0x63, 0x75, 0x73, 0x65, 0x53, 0x6f, 0x6e, 0x74, 0x61,
0x78, 0x74, 0x73, 0x65, 0x65, 0x73, 0x3b, 0x4d, 0x61, 0x6e, 0x74, 0x61,
0x78, 0x74, 0x73, 0x65, 0x65, 0x73, 0x3b, 0x44, 0x65, 0x6e, 0x73, 0x74,
0x61, 0x78, 0x74, 0x73, 0x65, 0x65, 0x73, 0x3b, 0x57, 0x75, 0x6e, 0x73,
0x74, 0x61, 0x78, 0x74, 0x73, 0x65, 0x65, 0x73, 0x3b, 0x44, 0x6f, 0x6e,
0x64, 0x65, 0x72, 0x74, 0x61, 0x78, 0x74, 0x73, 0x65, 0x65, 0x73, 0x3b,
0x46, 0x72, 0x61, 0x69, 0x74, 0x61, 0x78, 0x74, 0x73, 0x65, 0x65, 0x73,
0x3b, 0x53, 0x61, 0x74, 0x65, 0x72, 0x74, 0x61, 0x78, 0x74, 0x73, 0x65,
0x65, 0x73, 0x53, 0x6f, 0x6e, 0x3b, 0x4d, 0x61, 0x3b, 0x44, 0x65, 0x3b,
0x57, 0x75, 0x3b, 0x44, 0x6f, 0x3b, 0x46, 0x72, 0x3b, 0x53, 0x61, 0x74,
0x53, 0x3b, 0x4d, 0x3b, 0x45, 0x3b, 0x57, 0x3b, 0x44, 0x3b, 0x46, 0x3b,
0x41, 0x906, 0x907, 0x924, 0x92c, 0x93e, 0x930, 0x3b, 0x938, 0x94b, 0x92e, 0x92c,
0x93e, 0x930, 0x3b, 0x92e, 0x919, 0x94d, 0x917, 0x932, 0x92c, 0x93e, 0x930, 0x3b,
0x92c, 0x941, 0x927, 0x92c, 0x93e, 0x930, 0x3b, 0x92c, 0x93f, 0x939, 0x93f, 0x92c,
0x93e, 0x930, 0x3b, 0x936, 0x941, 0x915, 0x94d, 0x930, 0x92c, 0x93e, 0x930, 0x3b,
0x936, 0x928, 0x93f, 0x92c, 0x93e, 0x930, 0x906, 0x907, 0x924, 0x3b, 0x938, 0x94b,
0x92e, 0x3b, 0x92e, 0x919, 0x94d, 0x917, 0x932, 0x3b, 0x92c, 0x941, 0x927, 0x3b,
0x92c, 0x93f, 0x939, 0x93f, 0x3b, 0x936, 0x941, 0x915, 0x94d, 0x930, 0x3b, 0x936,
0x928, 0x93f, 0x906, 0x3b, 0x938, 0x94b, 0x3b, 0x92e, 0x3b, 0x92c, 0x941, 0x3b,
0x92c, 0x93f, 0x3b, 0x936, 0x941, 0x3b, 0x936, 0x6c, 0x79, 0x25b, 0x2bc, 0x25b,
0x301, 0x20, 0x73, 0x1e85, 0xed, 0x14b, 0x74, 0xe8, 0x3b, 0x6d, 0x76, 0x66,
0xf2, 0x20, 0x6c, 0x79, 0x25b, 0x30c, 0x2bc, 0x3b, 0x6d, 0x62, 0x254, 0x301,
0x254, 0x6e, 0x74, 0xe8, 0x20, 0x6d, 0x76, 0x66, 0xf2, 0x20, 0x6c, 0x79,
0x25b, 0x30c, 0x2bc, 0x3b, 0x74, 0x73, 0xe8, 0x74, 0x73, 0x25b, 0x300, 0x25b,
0x20, 0x6c, 0x79, 0x25b, 0x30c, 0x2bc, 0x3b, 0x6d, 0x62, 0x254, 0x301, 0x254,
0x6e, 0x74, 0xe8, 0x20, 0x74, 0x73, 0x65, 0x74, 0x73, 0x25b, 0x300, 0x25b,
0x20, 0x6c, 0x79, 0x25b, 0x30c, 0x2bc, 0x3b, 0x6d, 0x76, 0x66, 0xf2, 0x20,
0x6d, 0xe0, 0x67, 0x61, 0x20, 0x6c, 0x79, 0x25b, 0x30c, 0x2bc, 0x3b, 0x6d,
0xe0, 0x67, 0x61, 0x20, 0x6c, 0x79, 0x25b, 0x30c, 0x2bc, 0x53, 0x254, 0x301,
0x6e, 0x64, 0x69, 0x3b, 0x4d, 0x254, 0x301, 0x6e, 0x64, 0x69, 0x3b, 0xc1,
0x70, 0x74, 0x61, 0x20, 0x4d, 0x254, 0x301, 0x6e, 0x64, 0x69, 0x3b, 0x57,
0x25b, 0x301, 0x6e, 0x25b, 0x73, 0x25b, 0x64, 0x25b, 0x3b, 0x54, 0x254, 0x301,
0x73, 0x25b, 0x64, 0x25b, 0x3b, 0x46, 0x25b, 0x6c, 0xe2, 0x79, 0x25b, 0x64,
0x25b, 0x3b, 0x53, 0xe1, 0x73, 0x69, 0x64, 0x25b, 0x53, 0x254, 0x301, 0x3b,
0x4d, 0x254, 0x301, 0x3b, 0xc1, 0x4d, 0x3b, 0x57, 0x25b, 0x301, 0x3b, 0x54,
0x254, 0x301, 0x3b, 0x46, 0x25b, 0x3b, 0x53, 0xe1, 0x53, 0x1ecd, 0x301, 0x6e,
0x64, 0xe8, 0x3b, 0x4d, 0x1ecd, 0x301, 0x6e, 0x64, 0xe8, 0x3b, 0x54, 0x69,
0xfa, 0x7a, 0x64, 0xe8, 0x3b, 0x57, 0x1eb9, 0x301, 0x6e, 0x1eb9, 0x301, 0x7a,
0x64, 0xe8, 0x3b, 0x54, 0x1ecd, 0x301, 0x7a, 0x64, 0xe8, 0x3b, 0x46, 0x72,
0x61, 0xed, 0x64, 0xe8, 0x3b, 0x53, 0xe1, 0x74, 0x1ecd, 0x64, 0xe8, 0x53,
0x1ecd, 0x301, 0x6e, 0x3b, 0x4d, 0x1ecd, 0x301, 0x6e, 0x3b, 0x54, 0x69, 0xfa,
0x3b, 0x57, 0x1eb9, 0x301, 0x6e, 0x3b, 0x54, 0x1ecd, 0x301, 0x7a, 0x3b, 0x46,
0x72, 0x61, 0xed, 0x3b, 0x53, 0xe1, 0x74, 0x7de, 0x7ca, 0x7ef, 0x7d9, 0x7cc,
0x7df, 0x7cf, 0x7f2, 0x3b, 0x7de, 0x7d0, 0x7ec, 0x7d3, 0x7ca, 0x7ec, 0x7df, 0x7cf,
0x7f2, 0x3b, 0x7de, 0x7d0, 0x7ec, 0x7df, 0x7cf, 0x7f2, 0x3b, 0x7de, 0x7ce, 0x7e3,
0x7ce, 0x7f2, 0x7df, 0x7cf, 0x7f2, 0x3b, 0x7d3, 0x7cc, 0x7df, 0x7cf, 0x7f2, 0x3b,
0x7db, 0x7cc, 0x7ec, 0x7e3, 0x7cc, 0x7f2, 0x7ec, 0x7df, 0x7cf, 0x7f2, 0x3b, 0x7de,
0x7cd, 0x7f2, 0x7d8, 0x7cd, 0x7df, 0x7cf, 0x7f2, 0x7de, 0x7ca, 0x7ef, 0x7d9, 0x3b,
0x7de, 0x7d0, 0x7ec, 0x7d3, 0x3b, 0x7de, 0x7d0, 0x7ec, 0x7df, 0x3b, 0x7de, 0x7ce,
0x7e3, 0x3b, 0x7d3, 0x7cc, 0x7df, 0x3b, 0x7db, 0x7cc, 0x7ec, 0x7e3, 0x3b, 0x7de,
0x7cd, 0x7f2, 0x7d8, 0x7de, 0x7ca, 0x7ef, 0x7d9, 0x3b, 0x7de, 0x7d0, 0x7ec, 0x7d3,
0x3b, 0x7de, 0x7d0, 0x7ec, 0x7df, 0x7cf, 0x7f2, 0x3b, 0x7de, 0x7ce, 0x7e3, 0x3b,
0x7d3, 0x7cc, 0x7df, 0x3b, 0x7db, 0x7cc, 0x7ec, 0x7e3, 0x3b, 0x7de, 0x7cd, 0x7f2,
0x7d8, 0x7de, 0x3b, 0x7de, 0x3b, 0x7de, 0x3b, 0x7de, 0x3b, 0x7d3, 0x3b, 0x7db,
0x3b, 0x7de, 0x73, 0x6f, 0x74, 0x6e, 0x61, 0x62, 0x65, 0x61, 0x69, 0x76,
0x69, 0x3b, 0x76, 0x75, 0x6f, 0x73, 0x73, 0xe1, 0x72, 0x67, 0x61, 0x3b,
0x6d, 0x61, 0x14b, 0x14b, 0x65, 0x62, 0xe1, 0x72, 0x67, 0x61, 0x3b, 0x67,
0x61, 0x73, 0x6b, 0x61, 0x76, 0x61, 0x68, 0x6b, 0x6b, 0x75, 0x3b, 0x64,
0x75, 0x6f, 0x72, 0x61, 0x73, 0x64, 0x61, 0x74, 0x3b, 0x62, 0x65, 0x61,
0x72, 0x6a, 0x61, 0x64, 0x61, 0x74, 0x3b, 0x6c, 0xe1, 0x76, 0x76, 0x61,
0x72, 0x64, 0x61, 0x74, 0x73, 0x6f, 0x74, 0x6e, 0x3b, 0x76, 0x75, 0x6f,
0x73, 0x3b, 0x6d, 0x61, 0x14b, 0x3b, 0x67, 0x61, 0x73, 0x6b, 0x3b, 0x64,
0x75, 0x6f, 0x72, 0x3b, 0x62, 0x65, 0x61, 0x72, 0x3b, 0x6c, 0xe1, 0x76,
0x53, 0x3b, 0x56, 0x3b, 0x4d, 0x3b, 0x47, 0x3b, 0x44, 0x3b, 0x42, 0x3b,
0x4c, 0x73, 0x6f, 0x74, 0x6e, 0x61, 0x62, 0x65, 0x61, 0x69, 0x76, 0x69,
0x3b, 0x6d, 0xe1, 0x6e, 0x6e, 0x6f, 0x64, 0x61, 0x74, 0x3b, 0x64, 0x69,
0x73, 0x64, 0x61, 0x74, 0x3b, 0x67, 0x61, 0x73, 0x6b, 0x61, 0x76, 0x61,
0x68, 0x6b, 0x6b, 0x75, 0x3b, 0x64, 0x75, 0x6f, 0x72, 0x61, 0x73, 0x74,
0x61, 0x74, 0x3b, 0x62, 0x65, 0x61, 0x72, 0x6a, 0x61, 0x64, 0x61, 0x74,
0x3b, 0x6c, 0xe1, 0x76, 0x76, 0x6f, 0x72, 0x64, 0x61, 0x74, 0x73, 0x6f,
0x3b, 0x6d, 0xe1, 0x3b, 0x64, 0x69, 0x3b, 0x67, 0x61, 0x3b, 0x64, 0x75,
0x3b, 0x62, 0x65, 0x3b, 0x6c, 0xe1, 0x53, 0x3b, 0x4d, 0x3b, 0x44, 0x3b,
0x47, 0x3b, 0x44, 0x3b, 0x42, 0x3b, 0x4c, 0x4c, 0x61, 0x6d, 0x6f, 0x72,
0x65, 0x6e, 0x61, 0x3b, 0x4d, 0x75, 0x73, 0x6f, 0x70, 0x6f, 0x6c, 0x6f,
0x67, 0x6f, 0x3b, 0x4c, 0x61, 0x62, 0x6f, 0x62, 0x65, 0x64, 0x69, 0x3b,
0x4c, 0x61, 0x62, 0x6f, 0x72, 0x61, 0x72, 0x6f, 0x3b, 0x4c, 0x61, 0x62,
0x6f, 0x6e, 0x65, 0x3b, 0x4c, 0x61, 0x62, 0x6f, 0x68, 0x6c, 0x61, 0x6e,
0x6f, 0x3b, 0x4d, 0x6f, 0x6b, 0x69, 0x62, 0x65, 0x6c, 0x6f, 0x4c, 0x61,
0x6d, 0x3b, 0x4d, 0x6f, 0x73, 0x3b, 0x42, 0x65, 0x64, 0x3b, 0x52, 0x61,
0x72, 0x3b, 0x4e, 0x65, 0x3b, 0x48, 0x6c, 0x61, 0x3b, 0x4d, 0x6f, 0x6b,
0x4c, 0x3b, 0x4d, 0x3b, 0x42, 0x3b, 0x52, 0x3b, 0x4e, 0x3b, 0x48, 0x3b,
0x4d, 0x53, 0x6f, 0x6e, 0x74, 0x6f, 0x3b, 0x4d, 0x76, 0x75, 0x6c, 0x6f,
0x3b, 0x53, 0x69, 0x62, 0x69, 0x6c, 0x69, 0x3b, 0x53, 0x69, 0x74, 0x68,
0x61, 0x74, 0x68, 0x75, 0x3b, 0x53, 0x69, 0x6e, 0x65, 0x3b, 0x53, 0x69,
0x68, 0x6c, 0x61, 0x6e, 0x75, 0x3b, 0x4d, 0x67, 0x71, 0x69, 0x62, 0x65,
0x6c, 0x6f, 0x53, 0x6f, 0x6e, 0x3b, 0x4d, 0x76, 0x75, 0x3b, 0x53, 0x69,
0x62, 0x3b, 0x53, 0x69, 0x74, 0x3b, 0x53, 0x69, 0x6e, 0x3b, 0x53, 0x69,
0x68, 0x3b, 0x4d, 0x67, 0x71, 0x53, 0x3b, 0x4d, 0x3b, 0x53, 0x3b, 0x53,
0x3b, 0x53, 0x3b, 0x53, 0x3b, 0x4d, 0x73, 0xf8, 0x6e, 0x2e, 0x3b, 0x6d,
0x61, 0x6e, 0x2e, 0x3b, 0x74, 0x69, 0x72, 0x2e, 0x3b, 0x6f, 0x6e, 0x73,
0x2e, 0x3b, 0x74, 0x6f, 0x72, 0x2e, 0x3b, 0x66, 0x72, 0x65, 0x2e, 0x3b,
0x6c, 0xf8, 0x72, 0x2e, 0x73, 0xf8, 0x6e, 0x64, 0x61, 0x67, 0x3b, 0x6d,
0xe5, 0x6e, 0x64, 0x61, 0x67, 0x3b, 0x74, 0x79, 0x73, 0x64, 0x61, 0x67,
0x3b, 0x6f, 0x6e, 0x73, 0x64, 0x61, 0x67, 0x3b, 0x74, 0x6f, 0x72, 0x73,
0x64, 0x61, 0x67, 0x3b, 0x66, 0x72, 0x65, 0x64, 0x61, 0x67, 0x3b, 0x6c,
0x61, 0x75, 0x72, 0x64, 0x61, 0x67, 0x73, 0xf8, 0x6e, 0x3b, 0x6d, 0xe5,
0x6e, 0x3b, 0x74, 0x79, 0x73, 0x3b, 0x6f, 0x6e, 0x73, 0x3b, 0x74, 0x6f,
0x72, 0x3b, 0x66, 0x72, 0x65, 0x3b, 0x6c, 0x61, 0x75, 0x73, 0xf8, 0x2e,
0x3b, 0x6d, 0xe5, 0x2e, 0x3b, 0x74, 0x79, 0x2e, 0x3b, 0x6f, 0x6e, 0x2e,
0x3b, 0x74, 0x6f, 0x2e, 0x3b, 0x66, 0x72, 0x2e, 0x3b, 0x6c, 0x61, 0x2e,
0x43, 0xe4, 0x14b, 0x20, 0x6b, 0x75, 0x254, 0x74, 0x68, 0x3b, 0x4a, 0x69,
0x65, 0x63, 0x20, 0x6c, 0x61, 0x331, 0x74, 0x3b, 0x52, 0x25b, 0x77, 0x20,
0x6c, 0xe4, 0x74, 0x6e, 0x69, 0x3b, 0x44, 0x69, 0x254, 0x331, 0x6b, 0x20,
0x6c, 0xe4, 0x74, 0x6e, 0x69, 0x3b, 0x14a, 0x75, 0x61, 0x61, 0x6e, 0x20,
0x6c, 0xe4, 0x74, 0x6e, 0x69, 0x3b, 0x44, 0x68, 0x69, 0x65, 0x65, 0x63,
0x20, 0x6c, 0xe4, 0x74, 0x6e, 0x69, 0x3b, 0x42, 0xe4, 0x6b, 0x25b, 0x6c,
0x20, 0x6c, 0xe4, 0x74, 0x6e, 0x69, 0x43, 0xe4, 0x14b, 0x3b, 0x4a, 0x69,
0x65, 0x63, 0x3b, 0x52, 0x25b, 0x77, 0x3b, 0x44, 0x69, 0x254, 0x331, 0x6b,
0x3b, 0x14a, 0x75, 0x61, 0x61, 0x6e, 0x3b, 0x44, 0x68, 0x69, 0x65, 0x65,
0x63, 0x3b, 0x42, 0xe4, 0x6b, 0x25b, 0x6c, 0x43, 0x3b, 0x4a, 0x3b, 0x52,
0x3b, 0x44, 0x3b, 0x14a, 0x3b, 0x44, 0x3b, 0x42, 0x4c, 0x61, 0x6d, 0x75,
0x6c, 0x75, 0x6e, 0x67, 0x75, 0x3b, 0x4c, 0x6f, 0x6c, 0x65, 0x6d, 0x62,
0x61, 0x3b, 0x4c, 0x61, 0x63, 0x68, 0x69, 0x77, 0x69, 0x72, 0x69, 0x3b,
0x4c, 0x61, 0x63, 0x68, 0x69, 0x74, 0x61, 0x74, 0x75, 0x3b, 0x4c, 0x61,
0x63, 0x68, 0x69, 0x6e, 0x61, 0x79, 0x69, 0x3b, 0x4c, 0x61, 0x63, 0x68,
0x69, 0x73, 0x61, 0x6e, 0x75, 0x3b, 0x4c, 0x6f, 0x77, 0x65, 0x72, 0x75,
0x6b, 0x61, 0x4d, 0x75, 0x6c, 0x3b, 0x4c, 0x65, 0x6d, 0x3b, 0x57, 0x69,
0x72, 0x3b, 0x54, 0x61, 0x74, 0x3b, 0x4e, 0x61, 0x69, 0x3b, 0x53, 0x61,
0x6e, 0x3b, 0x57, 0x65, 0x72, 0x64, 0x69, 0x6d, 0x65, 0x6e, 0x67, 0x65,
0x3b, 0x64, 0x69, 0x6c, 0x75, 0x6e, 0x73, 0x3b, 0x64, 0x69, 0x6d, 0x61,
0x72, 0x73, 0x3b, 0x64, 0x69, 0x6d, 0xe8, 0x63, 0x72, 0x65, 0x73, 0x3b,
0x64, 0x69, 0x6a, 0xf2, 0x75, 0x73, 0x3b, 0x64, 0x69, 0x76, 0x65, 0x6e,
0x64, 0x72, 0x65, 0x73, 0x3b, 0x64, 0x69, 0x73, 0x73, 0x61, 0x62, 0x74,
0x65, 0x44, 0x67, 0x3b, 0x44, 0x6c, 0x3b, 0x44, 0x6d, 0x3b, 0x44, 0x63,
0x3b, 0x44, 0x6a, 0x3b, 0x44, 0x76, 0x3b, 0x44, 0x73, 0x64, 0x69, 0x6d,
0x65, 0x6e, 0x67, 0x65, 0x3b, 0x64, 0x65, 0x6c, 0x75, 0x6e, 0x73, 0x3b,
0x64, 0x69, 0x6d, 0x61, 0x72, 0x73, 0x3b, 0x64, 0x69, 0x6d, 0xe8, 0x72,
0x63, 0x6c, 0x65, 0x73, 0x3b, 0x64, 0x69, 0x6a, 0x61, 0x75, 0x73, 0x3b,
0x64, 0x69, 0x75, 0x65, 0x6e, 0x64, 0x72, 0x65, 0x73, 0x3b, 0x64, 0x69,
0x73, 0x73, 0x61, 0x62, 0x74, 0x65, 0x64, 0x69, 0x6d, 0x3b, 0x64, 0x65,
0x6c, 0x3b, 0x64, 0x6d, 0x61, 0x3b, 0x64, 0x6d, 0xe8, 0x3b, 0x64, 0x69,
0x6a, 0x3b, 0x64, 0x69, 0x75, 0x3b, 0x64, 0x69, 0x73, 0x44, 0x3b, 0x4c,
0x3b, 0x4d, 0x3b, 0x58, 0x3b, 0x4a, 0x3b, 0x55, 0x3b, 0x53, 0xb30, 0xb2c,
0xb3f, 0xb2c, 0xb3e, 0xb30, 0x3b, 0xb38, 0xb4b, 0xb2e, 0xb2c, 0xb3e, 0xb30, 0x3b,
0xb2e, 0xb19, 0xb4d, 0xb17, 0xb33, 0xb2c, 0xb3e, 0xb30, 0x3b, 0xb2c, 0xb41, 0xb27,
0xb2c, 0xb3e, 0xb30, 0x3b, 0xb17, 0xb41, 0xb30, 0xb41, 0xb2c, 0xb3e, 0xb30, 0x3b,
0xb36, 0xb41, 0xb15, 0xb4d, 0xb30, 0xb2c, 0xb3e, 0xb30, 0x3b, 0xb36, 0xb28, 0xb3f,
0xb2c, 0xb3e, 0xb30, 0xb30, 0xb2c, 0xb3f, 0x3b, 0xb38, 0xb4b, 0xb2e, 0x3b, 0xb2e,
0xb19, 0xb4d, 0xb17, 0xb33, 0x3b, 0xb2c, 0xb41, 0xb27, 0x3b, 0xb17, 0xb41, 0xb30,
0xb41, 0x3b, 0xb36, 0xb41, 0xb15, 0xb4d, 0xb30, 0x3b, 0xb36, 0xb28, 0xb3f, 0xb30,
0x3b, 0xb38, 0xb4b, 0x3b, 0xb2e, 0x3b, 0xb2c, 0xb41, 0x3b, 0xb17, 0xb41, 0x3b,
0xb36, 0xb41, 0x3b, 0xb36, 0x44, 0x69, 0x6c, 0x62, 0x61, 0x74, 0x61, 0x3b,
0x57, 0x69, 0x69, 0x78, 0x61, 0x74, 0x61, 0x3b, 0x51, 0x69, 0x62, 0x78,
0x61, 0x74, 0x61, 0x3b, 0x52, 0x6f, 0x6f, 0x62, 0x69, 0x69, 0x3b, 0x4b,
0x61, 0x6d, 0x69, 0x69, 0x73, 0x61, 0x3b, 0x4a, 0x69, 0x6d, 0x61, 0x61,
0x74, 0x61, 0x3b, 0x53, 0x61, 0x6e, 0x62, 0x61, 0x74, 0x61, 0x44, 0x69,
0x6c, 0x3b, 0x57, 0x69, 0x78, 0x3b, 0x51, 0x69, 0x62, 0x3b, 0x52, 0x6f,
0x62, 0x3b, 0x4b, 0x61, 0x6d, 0x3b, 0x4a, 0x69, 0x6d, 0x3b, 0x53, 0x61,
0x6e, 0x44, 0x3b, 0x57, 0x3b, 0x51, 0x3b, 0x52, 0x3b, 0x4b, 0x3b, 0x4a,
0x3b, 0x53, 0xd801, 0xdcb9, 0xd801, 0xdcd8, 0x358, 0xd801, 0xdcec, 0xd801, 0xdcd8, 0x20,
0xd801, 0xdccf, 0xd801, 0xdcd8, 0xd801, 0xdce4, 0xd801, 0xdcd8, 0x358, 0xd801, 0xdcf0, 0xd801,
0xdcd8, 0xd801, 0xdce4, 0xd801, 0xdce3, 0x3b, 0xd801, 0xdcb9, 0xd801, 0xdcd8, 0x358, 0xd801,
0xdcec, 0xd801, 0xdcd8, 0x20, 0xd801, 0xdcc4, 0xd801, 0xdcd8, 0xd801, 0xdce1, 0xd801, 0xdcdb,
0x358, 0xd801, 0xdce7, 0xd801, 0xdce3, 0x3b, 0xd801, 0xdcb9, 0xd801, 0xdcd8, 0x358, 0xd801,
0xdcec, 0xd801, 0xdcd8, 0x20, 0xd801, 0xdccf, 0xd801, 0xdcdf, 0xd801, 0xdcf5, 0xd801, 0xdcea,
0x358, 0xd801, 0xdcec, 0xd801, 0xdcd8, 0x3b, 0xd801, 0xdcb9, 0xd801, 0xdcd8, 0x358, 0xd801,
0xdcec, 0xd801, 0xdcd8, 0x20, 0xd801, 0xdccf, 0xd801, 0xdcdf, 0xd801, 0xdcf5, 0xd801, 0xdcd8,
0xd801, 0xdcdc, 0xd801, 0xdce3, 0x3b, 0xd801, 0xdcb9, 0xd801, 0xdcd8, 0x358, 0xd801, 0xdcec,
0xd801, 0xdcd8, 0x20, 0xd801, 0xdccf, 0xd801, 0xdcdf, 0xd801, 0xdcf0, 0xd801, 0xdcea, 0xd801,
0xdcec, 0xd801, 0xdcd8, 0x3b, 0xd801, 0xdcb9, 0xd801, 0xdcd8, 0x358, 0xd801, 0xdcec, 0xd801,
0xdcd8, 0x20, 0xd801, 0xdcc8, 0xd801, 0xdcd8, 0x20, 0xd801, 0xdcf5, 0xd801, 0xdcd8, 0xd801,
0xdcf2, 0xd801, 0xdcd8, 0x20, 0xd801, 0xdcfb, 0xd801, 0xdce3, 0x358, 0x3b, 0xd801, 0xdcb9,
0xd801, 0xdcd8, 0x358, 0xd801, 0xdcec, 0xd801, 0xdcd8, 0x20, 0xd801, 0xdcc2, 0xd801, 0xdce4,
0xd801, 0xdcd8, 0xd801, 0xdcf8, 0xd801, 0xdcdf, 0x20, 0xd801, 0xdce3, 0x358, 0xd801, 0xdce4,
0xd801, 0xdcdf, 0xd801, 0xdccf, 0x3b, 0xd801, 0xdcc4, 0x3b, 0xd801, 0xdccd, 0x3b, 0xd801,
0xdcb4, 0x3b, 0xd801, 0xdcc8, 0x3b, 0xd801, 0xdcca, 0x3b, 0xd801, 0xdcf8, 0x425, 0x443,
0x44b, 0x446, 0x430, 0x443, 0x431, 0x43e, 0x43d, 0x3b, 0x41a, 0x44a, 0x443, 0x44b,
0x440, 0x438, 0x441, 0x4d5, 0x440, 0x3b, 0x414, 0x44b, 0x446, 0x446, 0x4d5, 0x433,
0x3b, 0x4d4, 0x440, 0x442, 0x44b, 0x446, 0x446, 0x4d5, 0x433, 0x3b, 0x426, 0x44b,
0x43f, 0x43f, 0x4d5, 0x440, 0x4d5, 0x43c, 0x3b, 0x41c, 0x430, 0x439, 0x440, 0x4d5,
0x43c, 0x431, 0x43e, 0x43d, 0x3b, 0x421, 0x430, 0x431, 0x430, 0x442, 0x445, 0x443,
0x44b, 0x446, 0x430, 0x443, 0x431, 0x43e, 0x43d, 0x3b, 0x43a, 0x44a, 0x443, 0x44b,
0x440, 0x438, 0x441, 0x4d5, 0x440, 0x3b, 0x434, 0x44b, 0x446, 0x446, 0x4d5, 0x433,
0x3b, 0x4d5, 0x440, 0x442, 0x44b, 0x446, 0x446, 0x4d5, 0x433, 0x3b, 0x446, 0x44b,
0x43f, 0x43f, 0x4d5, 0x440, 0x4d5, 0x43c, 0x3b, 0x43c, 0x430, 0x439, 0x440, 0x4d5,
0x43c, 0x431, 0x43e, 0x43d, 0x3b, 0x441, 0x430, 0x431, 0x430, 0x442, 0x425, 0x446,
0x431, 0x3b, 0x41a, 0x440, 0x441, 0x3b, 0x414, 0x446, 0x433, 0x3b, 0x4d4, 0x440,
0x442, 0x3b, 0x426, 0x43f, 0x440, 0x3b, 0x41c, 0x440, 0x431, 0x3b, 0x421, 0x431,
0x442, 0x445, 0x446, 0x431, 0x3b, 0x43a, 0x440, 0x441, 0x3b, 0x434, 0x446, 0x433,
0x3b, 0x4d5, 0x440, 0x442, 0x3b, 0x446, 0x43f, 0x440, 0x3b, 0x43c, 0x440, 0x431,
0x3b, 0x441, 0x431, 0x442, 0x425, 0x3b, 0x41a, 0x3b, 0x414, 0x3b, 0x4d4, 0x3b,
0x426, 0x3b, 0x41c, 0x3b, 0x421, 0x64, 0x6a, 0x61, 0x64, 0x75, 0x6d, 0x69,
0x6e, 0x67, 0x75, 0x3b, 0x64, 0x6a, 0x61, 0x6c, 0x75, 0x6e, 0x61, 0x3b,
0x64, 0x6a, 0x61, 0x6d, 0x61, 0x72, 0x73, 0x3b, 0x64, 0x6a, 0x61, 0x72,
0x61, 0x73, 0x6f, 0x6e, 0x3b, 0x64, 0x6a, 0x61, 0x77, 0x65, 0x70, 0x73,
0x3b, 0x64, 0x6a, 0x61, 0x62, 0x69, 0xe8, 0x72, 0x6e, 0xe8, 0x3b, 0x64,
0x6a, 0x61, 0x73, 0x61, 0x62, 0x72, 0x61, 0x64, 0x64, 0x3b, 0x64, 0x6c,
0x3b, 0x64, 0x6d, 0x3b, 0x64, 0x72, 0x3b, 0x64, 0x77, 0x3b, 0x64, 0x62,
0x3b, 0x64, 0x73, 0x64a, 0x648, 0x646, 0x6cd, 0x3b, 0x62f, 0x648, 0x646, 0x6cd,
0x3b, 0x62f, 0x631, 0x6d0, 0x646, 0x6cd, 0x3b, 0x685, 0x644, 0x631, 0x646, 0x6cd,
0x3b, 0x67e, 0x64a, 0x646, 0x681, 0x646, 0x6cd, 0x3b, 0x62c, 0x645, 0x639, 0x647,
0x3b, 0x627, 0x648, 0x646, 0x6cd, 0x6cc, 0x6a9, 0x634, 0x646, 0x628, 0x647, 0x3b,
0x62f, 0x648, 0x634, 0x646, 0x628, 0x647, 0x3b, 0x633, 0x647, 0x200c, 0x634, 0x646,
0x628, 0x647, 0x3b, 0x686, 0x647, 0x627, 0x631, 0x634, 0x646, 0x628, 0x647, 0x3b,
0x67e, 0x646, 0x62c, 0x634, 0x646, 0x628, 0x647, 0x3b, 0x62c, 0x645, 0x639, 0x647,
0x3b, 0x634, 0x646, 0x628, 0x647, 0x6cc, 0x3b, 0x62f, 0x3b, 0x633, 0x3b, 0x686,
0x3b, 0x67e, 0x3b, 0x62c, 0x3b, 0x634, 0x6e, 0x69, 0x65, 0x64, 0x7a, 0x69,
0x65, 0x6c, 0x61, 0x3b, 0x70, 0x6f, 0x6e, 0x69, 0x65, 0x64, 0x7a, 0x69,
0x61, 0x142, 0x65, 0x6b, 0x3b, 0x77, 0x74, 0x6f, 0x72, 0x65, 0x6b, 0x3b,
0x15b, 0x72, 0x6f, 0x64, 0x61, 0x3b, 0x63, 0x7a, 0x77, 0x61, 0x72, 0x74,
0x65, 0x6b, 0x3b, 0x70, 0x69, 0x105, 0x74, 0x65, 0x6b, 0x3b, 0x73, 0x6f,
0x62, 0x6f, 0x74, 0x61, 0x6e, 0x69, 0x65, 0x64, 0x7a, 0x2e, 0x3b, 0x70,
0x6f, 0x6e, 0x2e, 0x3b, 0x77, 0x74, 0x2e, 0x3b, 0x15b, 0x72, 0x2e, 0x3b,
0x63, 0x7a, 0x77, 0x2e, 0x3b, 0x70, 0x74, 0x2e, 0x3b, 0x73, 0x6f, 0x62,
0x2e, 0x4e, 0x3b, 0x50, 0x3b, 0x57, 0x3b, 0x15a, 0x3b, 0x43, 0x3b, 0x50,
0x3b, 0x53, 0x6e, 0x3b, 0x70, 0x3b, 0x77, 0x3b, 0x15b, 0x3b, 0x63, 0x3b,
0x70, 0x3b, 0x73, 0x64, 0x6f, 0x6d, 0x69, 0x6e, 0x67, 0x6f, 0x3b, 0x73,
0x65, 0x67, 0x75, 0x6e, 0x64, 0x61, 0x2d, 0x66, 0x65, 0x69, 0x72, 0x61,
0x3b, 0x74, 0x65, 0x72, 0xe7, 0x61, 0x2d, 0x66, 0x65, 0x69, 0x72, 0x61,
0x3b, 0x71, 0x75, 0x61, 0x72, 0x74, 0x61, 0x2d, 0x66, 0x65, 0x69, 0x72,
0x61, 0x3b, 0x71, 0x75, 0x69, 0x6e, 0x74, 0x61, 0x2d, 0x66, 0x65, 0x69,
0x72, 0x61, 0x3b, 0x73, 0x65, 0x78, 0x74, 0x61, 0x2d, 0x66, 0x65, 0x69,
0x72, 0x61, 0x3b, 0x73, 0xe1, 0x62, 0x61, 0x64, 0x6f, 0x64, 0x6f, 0x6d,
0x2e, 0x3b, 0x73, 0x65, 0x67, 0x2e, 0x3b, 0x74, 0x65, 0x72, 0x2e, 0x3b,
0x71, 0x75, 0x61, 0x2e, 0x3b, 0x71, 0x75, 0x69, 0x2e, 0x3b, 0x73, 0x65,
0x78, 0x2e, 0x3b, 0x73, 0xe1, 0x62, 0x2e, 0x44, 0x3b, 0x53, 0x3b, 0x54,
0x3b, 0x51, 0x3b, 0x51, 0x3b, 0x53, 0x3b, 0x53, 0x64, 0x6f, 0x6d, 0x69,
0x6e, 0x67, 0x6f, 0x3b, 0x73, 0x65, 0x67, 0x75, 0x6e, 0x64, 0x61, 0x3b,
0x74, 0x65, 0x72, 0xe7, 0x61, 0x3b, 0x71, 0x75, 0x61, 0x72, 0x74, 0x61,
0x3b, 0x71, 0x75, 0x69, 0x6e, 0x74, 0x61, 0x3b, 0x73, 0x65, 0x78, 0x74,
0x61, 0x3b, 0x73, 0xe1, 0x62, 0x61, 0x64, 0x6f, 0x6e, 0x61, 0x64, 0x12b,
0x6c, 0x69, 0x3b, 0x70, 0x61, 0x6e, 0x61, 0x64, 0x12b, 0x6c, 0x69, 0x3b,
0x77, 0x69, 0x73, 0x61, 0x73, 0x12b, 0x64, 0x69, 0x73, 0x3b, 0x70, 0x75,
0x73, 0x73, 0x69, 0x73, 0x61, 0x77, 0x61, 0x69, 0x74, 0x69, 0x3b, 0x6b,
0x65, 0x74, 0x77, 0x69, 0x72, 0x74, 0x69, 0x6b, 0x73, 0x3b, 0x70, 0x113,
0x6e, 0x74, 0x6e, 0x69, 0x6b, 0x73, 0x3b, 0x73, 0x61, 0x62, 0x61, 0x74,
0x74, 0x69, 0x6b, 0x61, 0x6e, 0x61, 0x64, 0x3b, 0x70, 0x61, 0x6e, 0x3b,
0x77, 0x69, 0x73, 0x3b, 0x70, 0x75, 0x73, 0x3b, 0x6b, 0x65, 0x74, 0x3b,
0x70, 0x113, 0x6e, 0x3b, 0x73, 0x61, 0x62, 0x4e, 0x3b, 0x50, 0x3b, 0x57,
0x3b, 0x50, 0x3b, 0x4b, 0x3b, 0x50, 0x3b, 0x53, 0xa10, 0xa24, 0xa35, 0xa3e,
0xa30, 0x3b, 0xa38, 0xa4b, 0xa2e, 0xa35, 0xa3e, 0xa30, 0x3b, 0xa2e, 0xa70, 0xa17,
0xa32, 0xa35, 0xa3e, 0xa30, 0x3b, 0xa2c, 0xa41, 0xa71, 0xa27, 0xa35, 0xa3e, 0xa30,
0x3b, 0xa35, 0xa40, 0xa30, 0xa35, 0xa3e, 0xa30, 0x3b, 0xa38, 0xa3c, 0xa41, 0xa71,
0xa15, 0xa30, 0xa35, 0xa3e, 0xa30, 0x3b, 0xa38, 0xa3c, 0xa28, 0xa3f, 0xa71, 0xa1a,
0xa30, 0xa35, 0xa3e, 0xa30, 0xa10, 0xa24, 0x3b, 0xa38, 0xa4b, 0xa2e, 0x3b, 0xa2e,
0xa70, 0xa17, 0xa32, 0x3b, 0xa2c, 0xa41, 0xa71, 0xa27, 0x3b, 0xa35, 0xa40, 0xa30,
0x3b, 0xa38, 0xa3c, 0xa41, 0xa71, 0xa15, 0xa30, 0x3b, 0xa38, 0xa3c, 0xa28, 0xa3f,
0xa71, 0xa1a, 0xa30, 0xa10, 0x3b, 0xa38, 0xa4b, 0x3b, 0xa2e, 0xa70, 0x3b, 0xa2c,
0xa41, 0xa71, 0x3b, 0xa35, 0xa40, 0x3b, 0xa38, 0xa3c, 0xa41, 0xa71, 0x3b, 0xa38,
0xa3c, 0x627, 0x62a, 0x648, 0x627, 0x631, 0x3b, 0x67e, 0x6cc, 0x631, 0x3b, 0x645,
0x646, 0x6af, 0x644, 0x3b, 0x628, 0x64f, 0x62f, 0x6be, 0x3b, 0x62c, 0x645, 0x639,
0x631, 0x627, 0x62a, 0x3b, 0x62c, 0x645, 0x639, 0x6c1, 0x3b, 0x6c1, 0x641, 0x62a,
0x6c1, 0x44, 0x6f, 0x6d, 0x69, 0x6e, 0x67, 0x6f, 0x3b, 0x4c, 0x75, 0x6e,
0x65, 0x73, 0x3b, 0x4d, 0x61, 0x72, 0x74, 0x65, 0x73, 0x3b, 0x4d, 0x69,
0xe9, 0x72, 0x63, 0x6f, 0x6c, 0x65, 0x73, 0x3b, 0x4a, 0x75, 0x65, 0x76,
0x65, 0x73, 0x3b, 0x56, 0x69, 0x65, 0x72, 0x6e, 0x65, 0x73, 0x3b, 0x53,
0xe1, 0x62, 0x61, 0x64, 0x6f, 0x44, 0x6f, 0x6d, 0x3b, 0x4c, 0x75, 0x6e,
0x3b, 0x4d, 0x61, 0x72, 0x3b, 0x4d, 0x69, 0xe9, 0x3b, 0x4a, 0x75, 0x65,
0x3b, 0x56, 0x69, 0x65, 0x3b, 0x53, 0x61, 0x62, 0x44, 0x3b, 0x4c, 0x3b,
0x4d, 0x3b, 0x58, 0x3b, 0x4a, 0x3b, 0x56, 0x3b, 0x53, 0x64, 0x75, 0x6d,
0x69, 0x6e, 0x69, 0x63, 0x103, 0x3b, 0x6c, 0x75, 0x6e, 0x69, 0x3b, 0x6d,
0x61, 0x72, 0x21b, 0x69, 0x3b, 0x6d, 0x69, 0x65, 0x72, 0x63, 0x75, 0x72,
0x69, 0x3b, 0x6a, 0x6f, 0x69, 0x3b, 0x76, 0x69, 0x6e, 0x65, 0x72, 0x69,
0x3b, 0x73, 0xe2, 0x6d, 0x62, 0x103, 0x74, 0x103, 0x64, 0x75, 0x6d, 0x2e,
0x3b, 0x6c, 0x75, 0x6e, 0x2e, 0x3b, 0x6d, 0x61, 0x72, 0x2e, 0x3b, 0x6d,
0x69, 0x65, 0x2e, 0x3b, 0x6a, 0x6f, 0x69, 0x3b, 0x76, 0x69, 0x6e, 0x2e,
0x3b, 0x73, 0xe2, 0x6d, 0x2e, 0x44, 0x75, 0x6d, 0x3b, 0x4c, 0x75, 0x6e,
0x3b, 0x4d, 0x61, 0x72, 0x3b, 0x4d, 0x69, 0x65, 0x3b, 0x4a, 0x6f, 0x69,
0x3b, 0x56, 0x69, 0x6e, 0x3b, 0x53, 0xe2, 0x6d, 0x44, 0x3b, 0x4c, 0x3b,
0x4d, 0x61, 0x3b, 0x4d, 0x69, 0x3b, 0x4a, 0x3b, 0x56, 0x3b, 0x53, 0x64,
0x75, 0x6d, 0x65, 0x6e, 0x67, 0x69, 0x61, 0x3b, 0x67, 0x6c, 0x69, 0x6e,
0x64, 0x65, 0x73, 0x64, 0x69, 0x3b, 0x6d, 0x61, 0x72, 0x64, 0x69, 0x3b,
0x6d, 0x65, 0x73, 0x65, 0x6d, 0x6e, 0x61, 0x3b, 0x67, 0x69, 0x65, 0x76,
0x67, 0x69, 0x61, 0x3b, 0x76, 0x65, 0x6e, 0x64, 0x65, 0x72, 0x64, 0x69,
0x3b, 0x73, 0x6f, 0x6e, 0x64, 0x61, 0x64, 0x75, 0x3b, 0x67, 0x6c, 0x69,
0x3b, 0x6d, 0x61, 0x3b, 0x6d, 0x65, 0x3b, 0x67, 0x69, 0x65, 0x3b, 0x76,
0x65, 0x3b, 0x73, 0x6f, 0x44, 0x3b, 0x47, 0x3b, 0x4d, 0x3b, 0x4d, 0x3b,
0x47, 0x3b, 0x56, 0x3b, 0x53, 0x49, 0x6a, 0x75, 0x6d, 0x61, 0x70, 0x69,
0x6c, 0x69, 0x3b, 0x49, 0x6a, 0x75, 0x6d, 0x61, 0x74, 0x61, 0x74, 0x75,
0x3b, 0x49, 0x6a, 0x75, 0x6d, 0x61, 0x6e, 0x6e, 0x65, 0x3b, 0x49, 0x6a,
0x75, 0x6d, 0x61, 0x74, 0x61, 0x6e, 0x6f, 0x3b, 0x41, 0x6c, 0x68, 0x61,
0x6d, 0x69, 0x73, 0x69, 0x3b, 0x49, 0x6a, 0x75, 0x6d, 0x61, 0x61, 0x3b,
0x49, 0x6a, 0x75, 0x6d, 0x61, 0x6d, 0x6f, 0x73, 0x69, 0x49, 0x6a, 0x70,
0x3b, 0x49, 0x6a, 0x74, 0x3b, 0x49, 0x6a, 0x6e, 0x3b, 0x49, 0x6a, 0x74,
0x6e, 0x3b, 0x41, 0x6c, 0x68, 0x3b, 0x49, 0x6a, 0x75, 0x3b, 0x49, 0x6a,
0x6d, 0x4b, 0x75, 0x20, 0x77, 0x2019, 0x69, 0x6e, 0x64, 0x77, 0x69, 0x3b,
0x4b, 0x75, 0x20, 0x77, 0x61, 0x20, 0x6d, 0x62, 0x65, 0x72, 0x65, 0x3b,
0x4b, 0x75, 0x20, 0x77, 0x61, 0x20, 0x6b, 0x61, 0x62, 0x69, 0x72, 0x69,
0x3b, 0x4b, 0x75, 0x20, 0x77, 0x61, 0x20, 0x67, 0x61, 0x74, 0x61, 0x74,
0x75, 0x3b, 0x4b, 0x75, 0x20, 0x77, 0x61, 0x20, 0x6b, 0x61, 0x6e, 0x65,
0x3b, 0x4b, 0x75, 0x20, 0x77, 0x61, 0x20, 0x67, 0x61, 0x74, 0x61, 0x6e,
0x75, 0x3b, 0x4b, 0x75, 0x20, 0x77, 0x61, 0x20, 0x67, 0x61, 0x74, 0x61,
0x6e, 0x64, 0x61, 0x74, 0x75, 0x63, 0x75, 0x2e, 0x3b, 0x6d, 0x62, 0x65,
0x2e, 0x3b, 0x6b, 0x61, 0x62, 0x2e, 0x3b, 0x67, 0x74, 0x75, 0x2e, 0x3b,
0x6b, 0x61, 0x6e, 0x2e, 0x3b, 0x67, 0x6e, 0x75, 0x2e, 0x3b, 0x67, 0x6e,
0x64, 0x2e, 0x432, 0x43e, 0x441, 0x43a, 0x440, 0x435, 0x441, 0x435, 0x43d, 0x44c,
0x435, 0x3b, 0x43f, 0x43e, 0x43d, 0x435, 0x434, 0x435, 0x43b, 0x44c, 0x43d, 0x438,
0x43a, 0x3b, 0x432, 0x442, 0x43e, 0x440, 0x43d, 0x438, 0x43a, 0x3b, 0x441, 0x440,
0x435, 0x434, 0x430, 0x3b, 0x447, 0x435, 0x442, 0x432, 0x435, 0x440, 0x433, 0x3b,
0x43f, 0x44f, 0x442, 0x43d, 0x438, 0x446, 0x430, 0x3b, 0x441, 0x443, 0x431, 0x431,
0x43e, 0x442, 0x430, 0x432, 0x441, 0x3b, 0x43f, 0x43d, 0x3b, 0x432, 0x442, 0x3b,
0x441, 0x440, 0x3b, 0x447, 0x442, 0x3b, 0x43f, 0x442, 0x3b, 0x441, 0x431, 0x412,
0x3b, 0x41f, 0x3b, 0x412, 0x3b, 0x421, 0x3b, 0x427, 0x3b, 0x41f, 0x3b, 0x421,
0x4e, 0x61, 0x62, 0x61, 0x20, 0x53, 0x61, 0x6d, 0x62, 0x61, 0x74, 0x3b,
0x53, 0x61, 0x6e, 0x69, 0x3b, 0x53, 0x61, 0x6c, 0x75, 0x73, 0x3b, 0x52,
0x61, 0x62, 0x75, 0x71, 0x3b, 0x43, 0x61, 0x6d, 0x75, 0x73, 0x3b, 0x4a,
0x75, 0x6d, 0x71, 0x61, 0x74, 0x61, 0x3b, 0x51, 0x75, 0x6e, 0x78, 0x61,
0x20, 0x53, 0x61, 0x6d, 0x62, 0x61, 0x74, 0x4e, 0x61, 0x62, 0x3b, 0x53,
0x61, 0x6e, 0x3b, 0x53, 0x61, 0x6c, 0x3b, 0x52, 0x61, 0x62, 0x3b, 0x43,
0x61, 0x6d, 0x3b, 0x4a, 0x75, 0x6d, 0x3b, 0x51, 0x75, 0x6e, 0x4e, 0x3b,
0x53, 0x3b, 0x53, 0x3b, 0x52, 0x3b, 0x43, 0x3b, 0x4a, 0x3b, 0x51, 0x431,
0x430, 0x441, 0x43a, 0x44b, 0x4bb, 0x44b, 0x430, 0x43d, 0x43d, 0x44c, 0x430, 0x3b,
0x431, 0x44d, 0x43d, 0x438, 0x434, 0x438, 0x44d, 0x43d, 0x43d, 0x44c, 0x438, 0x43a,
0x3b, 0x43e, 0x43f, 0x442, 0x443, 0x43e, 0x440, 0x443, 0x43d, 0x43d, 0x44c, 0x443,
0x43a, 0x3b, 0x441, 0x44d, 0x440, 0x44d, 0x434, 0x44d, 0x3b, 0x447, 0x44d, 0x43f,
0x43f, 0x438, 0x44d, 0x440, 0x3b, 0x411, 0x44d, 0x44d, 0x442, 0x438, 0x4a5, 0x441,
0x44d, 0x3b, 0x441, 0x443, 0x431, 0x443, 0x43e, 0x442, 0x430, 0x431, 0x441, 0x3b,
0x431, 0x43d, 0x3b, 0x43e, 0x43f, 0x3b, 0x441, 0x44d, 0x3b, 0x447, 0x43f, 0x3b,
0x431, 0x44d, 0x3b, 0x441, 0x431, 0x411, 0x3b, 0x411, 0x3b, 0x41e, 0x3b, 0x421,
0x3b, 0x427, 0x3b, 0x411, 0x3b, 0x421, 0x4d, 0x64, 0x65, 0x72, 0x6f, 0x74,
0x20, 0x65, 0x65, 0x20, 0x61, 0x72, 0x65, 0x3b, 0x4d, 0x64, 0x65, 0x72,
0x6f, 0x74, 0x20, 0x65, 0x65, 0x20, 0x6b, 0x75, 0x6e, 0x69, 0x3b, 0x4d,
0x64, 0x65, 0x72, 0x6f, 0x74, 0x20, 0x65, 0x65, 0x20, 0x6f, 0x6e, 0x67,
0x2019, 0x77, 0x61, 0x6e, 0x3b, 0x4d, 0x64, 0x65, 0x72, 0x6f, 0x74, 0x20,
0x65, 0x65, 0x20, 0x69, 0x6e, 0x65, 0x74, 0x3b, 0x4d, 0x64, 0x65, 0x72,
0x6f, 0x74, 0x20, 0x65, 0x65, 0x20, 0x69, 0x6c, 0x65, 0x3b, 0x4d, 0x64,
0x65, 0x72, 0x6f, 0x74, 0x20, 0x65, 0x65, 0x20, 0x73, 0x61, 0x70, 0x61,
0x3b, 0x4d, 0x64, 0x65, 0x72, 0x6f, 0x74, 0x20, 0x65, 0x65, 0x20, 0x6b,
0x77, 0x65, 0x41, 0x72, 0x65, 0x3b, 0x4b, 0x75, 0x6e, 0x3b, 0x4f, 0x6e,
0x67, 0x3b, 0x49, 0x6e, 0x65, 0x3b, 0x49, 0x6c, 0x65, 0x3b, 0x53, 0x61,
0x70, 0x3b, 0x4b, 0x77, 0x65, 0x41, 0x3b, 0x4b, 0x3b, 0x4f, 0x3b, 0x49,
0x3b, 0x49, 0x3b, 0x53, 0x3b, 0x4b, 0x42, 0x69, 0x6b, 0x75, 0x61, 0x2d,
0xf4, 0x6b, 0x6f, 0x3b, 0x42, 0xef, 0x6b, 0x75, 0x61, 0x2d, 0xfb, 0x73,
0x65, 0x3b, 0x42, 0xef, 0x6b, 0x75, 0x61, 0x2d, 0x70, 0x74, 0xe2, 0x3b,
0x42, 0xef, 0x6b, 0x75, 0x61, 0x2d, 0x75, 0x73, 0xef, 0xf6, 0x3b, 0x42,
0xef, 0x6b, 0x75, 0x61, 0x2d, 0x6f, 0x6b, 0xfc, 0x3b, 0x4c, 0xe2, 0x70,
0xf4, 0x73, 0xf6, 0x3b, 0x4c, 0xe2, 0x79, 0x65, 0x6e, 0x67, 0x61, 0x42,
0x6b, 0x31, 0x3b, 0x42, 0x6b, 0x32, 0x3b, 0x42, 0x6b, 0x33, 0x3b, 0x42,
0x6b, 0x34, 0x3b, 0x42, 0x6b, 0x35, 0x3b, 0x4c, 0xe2, 0x70, 0x3b, 0x4c,
0xe2, 0x79, 0x4b, 0x3b, 0x53, 0x3b, 0x54, 0x3b, 0x53, 0x3b, 0x4b, 0x3b,
0x50, 0x3b, 0x59, 0x4d, 0x75, 0x6c, 0x75, 0x6e, 0x67, 0x75, 0x3b, 0x4a,
0x75, 0x6d, 0x61, 0x74, 0x61, 0x74, 0x75, 0x3b, 0x4a, 0x75, 0x6d, 0x61,
0x6e, 0x6e, 0x65, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x74, 0x61, 0x6e, 0x6f,
0x3b, 0x41, 0x6c, 0x61, 0x68, 0x61, 0x6d, 0x69, 0x73, 0x69, 0x3b, 0x49,
0x6a, 0x75, 0x6d, 0x61, 0x61, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x6d, 0x6f,
0x73, 0x69, 0x4d, 0x75, 0x6c, 0x3b, 0x4a, 0x74, 0x74, 0x3b, 0x4a, 0x6e,
0x6e, 0x3b, 0x4a, 0x74, 0x6e, 0x3b, 0x41, 0x6c, 0x68, 0x3b, 0x49, 0x6a,
0x75, 0x3b, 0x4a, 0x6d, 0x6f, 0x4d, 0x3b, 0x4a, 0x3b, 0x4a, 0x3b, 0x4a,
0x3b, 0x41, 0x3b, 0x49, 0x3b, 0x4a, 0x930, 0x935, 0x93f, 0x935, 0x93e, 0x938,
0x930, 0x903, 0x3b, 0x938, 0x94b, 0x92e, 0x935, 0x93e, 0x938, 0x930, 0x903, 0x3b,
0x92e, 0x902, 0x917, 0x932, 0x935, 0x93e, 0x938, 0x930, 0x903, 0x3b, 0x92c, 0x941,
0x927, 0x935, 0x93e, 0x938, 0x930, 0x903, 0x3b, 0x917, 0x941, 0x930, 0x941, 0x935,
0x93e, 0x938, 0x930, 0x3a, 0x3b, 0x936, 0x941, 0x915, 0x94d, 0x930, 0x935, 0x93e,
0x938, 0x930, 0x903, 0x3b, 0x936, 0x928, 0x93f, 0x935, 0x93e, 0x938, 0x930, 0x903,
0x1c65, 0x1c64, 0x1c78, 0x1c5c, 0x1c6e, 0x3b, 0x1c5a, 0x1c5b, 0x1c6e, 0x3b, 0x1c75, 0x1c5f,
0x1c5e, 0x1c6e, 0x3b, 0x1c65, 0x1c5f, 0x1c79, 0x1c5c, 0x1c69, 0x1c71, 0x3b, 0x1c65, 0x1c5f,
0x1c79, 0x1c68, 0x1c6b, 0x1c64, 0x3b, 0x1c61, 0x1c5f, 0x1c79, 0x1c68, 0x1c69, 0x1c62, 0x3b,
0x1c67, 0x1c69, 0x1c66, 0x1c69, 0x1c62, 0x1c65, 0x1c64, 0x1c78, 0x3b, 0x1c5a, 0x1c5b, 0x3b,
0x1c75, 0x1c5f, 0x3b, 0x1c65, 0x1c5f, 0x1c79, 0x3b, 0x1c65, 0x1c5f, 0x1c79, 0x1c68, 0x3b,
0x1c61, 0x1c5f, 0x1c79, 0x3b, 0x1c67, 0x1c69, 0x1c65, 0x3b, 0x1c5a, 0x3b, 0x1c75, 0x3b,
0x1c65, 0x3b, 0x1c65, 0x3b, 0x1c61, 0x3b, 0x1c67, 0x64, 0x6f, 0x6d, 0xec, 0x6e,
0x69, 0x67, 0x61, 0x3b, 0x6c, 0x75, 0x6e, 0x69, 0x73, 0x3b, 0x6d, 0x61,
0x72, 0x74, 0x69, 0x73, 0x3b, 0x6d, 0xe8, 0x72, 0x63, 0x75, 0x72, 0x69,
0x73, 0x3b, 0x67, 0x69, 0xf2, 0x62, 0x69, 0x61, 0x3b, 0x63, 0x68, 0x65,
0x6e, 0xe0, 0x62, 0x75, 0x72, 0x61, 0x3b, 0x73, 0xe0, 0x62, 0x61, 0x64,
0x75, 0x64, 0x6f, 0x6d, 0x3b, 0x6c, 0x75, 0x6e, 0x3b, 0x6d, 0x61, 0x72,
0x3b, 0x6d, 0xe8, 0x72, 0x3b, 0x67, 0x69, 0xf2, 0x3b, 0x63, 0x68, 0x65,
0x3b, 0x73, 0xe0, 0x62, 0x44, 0x3b, 0x4c, 0x3b, 0x4d, 0x3b, 0x4d, 0x3b,
0x47, 0x3b, 0x43, 0x3b, 0x53, 0x44, 0x69, 0x6d, 0x69, 0x6e, 0x67, 0x75,
0x3b, 0x43, 0x68, 0x69, 0x70, 0x6f, 0x73, 0x69, 0x3b, 0x43, 0x68, 0x69,
0x70, 0x69, 0x72, 0x69, 0x3b, 0x43, 0x68, 0x69, 0x74, 0x61, 0x74, 0x75,
0x3b, 0x43, 0x68, 0x69, 0x6e, 0x61, 0x69, 0x3b, 0x43, 0x68, 0x69, 0x73,
0x68, 0x61, 0x6e, 0x75, 0x3b, 0x53, 0x61, 0x62, 0x75, 0x64, 0x75, 0x44,
0x69, 0x6d, 0x3b, 0x50, 0x6f, 0x73, 0x3b, 0x50, 0x69, 0x72, 0x3b, 0x54,
0x61, 0x74, 0x3b, 0x4e, 0x61, 0x69, 0x3b, 0x53, 0x68, 0x61, 0x3b, 0x53,
0x61, 0x62, 0x44, 0x3b, 0x50, 0x3b, 0x43, 0x3b, 0x54, 0x3b, 0x4e, 0x3b,
0x53, 0x3b, 0x53, 0x43d, 0x435, 0x434, 0x435, 0x459, 0x430, 0x3b, 0x43f, 0x43e,
0x43d, 0x435, 0x434, 0x435, 0x459, 0x430, 0x43a, 0x3b, 0x443, 0x442, 0x43e, 0x440,
0x430, 0x43a, 0x3b, 0x441, 0x440, 0x435, 0x434, 0x430, 0x3b, 0x447, 0x435, 0x442,
0x432, 0x440, 0x442, 0x430, 0x43a, 0x3b, 0x43f, 0x435, 0x442, 0x430, 0x43a, 0x3b,
0x441, 0x443, 0x431, 0x43e, 0x442, 0x430, 0x43d, 0x435, 0x434, 0x3b, 0x43f, 0x43e,
0x43d, 0x3b, 0x443, 0x442, 0x43e, 0x3b, 0x441, 0x440, 0x435, 0x3b, 0x447, 0x435,
0x442, 0x3b, 0x43f, 0x435, 0x442, 0x3b, 0x441, 0x443, 0x431, 0x43d, 0x435, 0x434,
0x458, 0x435, 0x459, 0x430, 0x3b, 0x43f, 0x43e, 0x43d, 0x435, 0x434, 0x435, 0x459,
0x430, 0x43a, 0x3b, 0x443, 0x442, 0x43e, 0x440, 0x430, 0x43a, 0x3b, 0x441, 0x440,
0x438, 0x458, 0x435, 0x434, 0x430, 0x3b, 0x447, 0x435, 0x442, 0x432, 0x440, 0x442,
0x430, 0x43a, 0x3b, 0x43f, 0x435, 0x442, 0x430, 0x43a, 0x3b, 0x441, 0x443, 0x431,
0x43e, 0x442, 0x430, 0x6e, 0x65, 0x64, 0x65, 0x6c, 0x6a, 0x61, 0x3b, 0x70,
0x6f, 0x6e, 0x65, 0x64, 0x65, 0x6c, 0x6a, 0x61, 0x6b, 0x3b, 0x75, 0x74,
0x6f, 0x72, 0x61, 0x6b, 0x3b, 0x73, 0x72, 0x65, 0x64, 0x61, 0x3b, 0x10d,
0x65, 0x74, 0x76, 0x72, 0x74, 0x61, 0x6b, 0x3b, 0x70, 0x65, 0x74, 0x61,
0x6b, 0x3b, 0x73, 0x75, 0x62, 0x6f, 0x74, 0x61, 0x6e, 0x65, 0x64, 0x3b,
0x70, 0x6f, 0x6e, 0x3b, 0x75, 0x74, 0x6f, 0x3b, 0x73, 0x72, 0x65, 0x3b,
0x10d, 0x65, 0x74, 0x3b, 0x70, 0x65, 0x74, 0x3b, 0x73, 0x75, 0x62, 0x6e,
0x65, 0x64, 0x6a, 0x65, 0x6c, 0x6a, 0x61, 0x3b, 0x70, 0x6f, 0x6e, 0x65,
0x64, 0x65, 0x6c, 0x6a, 0x61, 0x6b, 0x3b, 0x75, 0x74, 0x6f, 0x72, 0x61,
0x6b, 0x3b, 0x73, 0x72, 0x69, 0x6a, 0x65, 0x64, 0x61, 0x3b, 0x10d, 0x65,
0x74, 0x76, 0x72, 0x74, 0x61, 0x6b, 0x3b, 0x70, 0x65, 0x74, 0x61, 0x6b,
0x3b, 0x73, 0x75, 0x62, 0x6f, 0x74, 0x61, 0x4a, 0x75, 0x6d, 0x61, 0x61,
0x70, 0x69, 0x69, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x61, 0x74, 0x61, 0x74,
0x75, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x61, 0x6e, 0x65, 0x3b, 0x4a, 0x75,
0x6d, 0x61, 0x61, 0x74, 0x61, 0x6e, 0x6f, 0x3b, 0x41, 0x6c, 0x68, 0x61,
0x6d, 0x69, 0x73, 0x69, 0x3b, 0x49, 0x6a, 0x75, 0x6d, 0x61, 0x61, 0x3b,
0x4a, 0x75, 0x6d, 0x61, 0x61, 0x6d, 0x6f, 0x73, 0x69, 0x4a, 0x70, 0x69,
0x3b, 0x4a, 0x74, 0x74, 0x3b, 0x4a, 0x6d, 0x6e, 0x3b, 0x4a, 0x74, 0x6e,
0x3b, 0x41, 0x6c, 0x68, 0x3b, 0x49, 0x6a, 0x75, 0x3b, 0x4a, 0x6d, 0x6f,
0x32, 0x3b, 0x33, 0x3b, 0x34, 0x3b, 0x35, 0x3b, 0x41, 0x3b, 0x49, 0x3b,
0x31, 0x53, 0x76, 0x6f, 0x6e, 0x64, 0x6f, 0x3b, 0x4d, 0x75, 0x76, 0x68,
0x75, 0x72, 0x6f, 0x3b, 0x43, 0x68, 0x69, 0x70, 0x69, 0x72, 0x69, 0x3b,
0x43, 0x68, 0x69, 0x74, 0x61, 0x74, 0x75, 0x3b, 0x43, 0x68, 0x69, 0x6e,
0x61, 0x3b, 0x43, 0x68, 0x69, 0x73, 0x68, 0x61, 0x6e, 0x75, 0x3b, 0x4d,
0x75, 0x67, 0x6f, 0x76, 0x65, 0x72, 0x61, 0x53, 0x76, 0x6f, 0x3b, 0x4d,
0x75, 0x76, 0x3b, 0x43, 0x68, 0x70, 0x3b, 0x43, 0x68, 0x74, 0x3b, 0x43,
0x68, 0x6e, 0x3b, 0x43, 0x68, 0x73, 0x3b, 0x4d, 0x75, 0x67, 0x53, 0x3b,
0x4d, 0x3b, 0x43, 0x3b, 0x43, 0x3b, 0x43, 0x3b, 0x43, 0x3b, 0x4d, 0xa46d,
0xa18f, 0xa44d, 0x3b, 0xa18f, 0xa282, 0xa2cd, 0x3b, 0xa18f, 0xa282, 0xa44d, 0x3b, 0xa18f,
0xa282, 0xa315, 0x3b, 0xa18f, 0xa282, 0xa1d6, 0x3b, 0xa18f, 0xa282, 0xa26c, 0x3b, 0xa18f,
0xa282, 0xa0d8, 0xa46d, 0xa18f, 0x3b, 0xa18f, 0xa2cd, 0x3b, 0xa18f, 0xa44d, 0x3b, 0xa18f,
0xa315, 0x3b, 0xa18f, 0xa1d6, 0x3b, 0xa18f, 0xa26c, 0x3b, 0xa18f, 0xa0d8, 0xa18f, 0x3b,
0xa2cd, 0x3b, 0xa44d, 0x3b, 0xa315, 0x3b, 0xa1d6, 0x3b, 0xa26c, 0x3b, 0xa0d8, 0x64,
0x75, 0x6d, 0xec, 0x6e, 0x69, 0x63, 0x61, 0x3b, 0x6c, 0x75, 0x6e, 0x6e,
0x69, 0x64, 0xec, 0x61, 0x3b, 0x6d, 0x61, 0x72, 0x74, 0x69, 0x64, 0xec,
0x61, 0x3b, 0x6d, 0x65, 0x72, 0x63, 0x75, 0x72, 0x69, 0x64, 0xec, 0x61,
0x3b, 0x6a, 0x6f, 0x76, 0x69, 0x64, 0xec, 0x61, 0x3b, 0x76, 0x65, 0x6e,
0x6e, 0x69, 0x64, 0xec, 0x61, 0x3b, 0x73, 0xe0, 0x62, 0x62, 0x61, 0x74,
0x75, 0x53, 0x61, 0x6d, 0x62, 0x61, 0x74, 0x61, 0x3b, 0x53, 0x61, 0x6e,
0x79, 0x6f, 0x3b, 0x4d, 0x61, 0x61, 0x6b, 0x69, 0x73, 0x61, 0x6e, 0x79,
0x6f, 0x3b, 0x52, 0x6f, 0x6f, 0x77, 0x65, 0x3b, 0x48, 0x61, 0x6d, 0x75,
0x73, 0x65, 0x3b, 0x41, 0x72, 0x62, 0x65, 0x3b, 0x51, 0x69, 0x64, 0x61,
0x61, 0x6d, 0x65, 0x53, 0x61, 0x6d, 0x3b, 0x53, 0x61, 0x6e, 0x3b, 0x4d,
0x61, 0x6b, 0x3b, 0x52, 0x6f, 0x77, 0x3b, 0x48, 0x61, 0x6d, 0x3b, 0x41,
0x72, 0x62, 0x3b, 0x51, 0x69, 0x64, 0x53, 0x3b, 0x53, 0x3b, 0x4d, 0x3b,
0x52, 0x3b, 0x48, 0x3b, 0x41, 0x3b, 0x51, 0x6e, 0x69, 0x79, 0x64, 0x7a,
0x69, 0x65, 0x6c, 0x61, 0x3b, 0x70, 0x79, 0x144, 0x64, 0x7a, 0x69, 0x61,
0x142, 0x65, 0x6b, 0x3b, 0x77, 0x74, 0x6f, 0x72, 0x65, 0x6b, 0x3b, 0x73,
0x74, 0x72, 0x7a, 0x6f, 0x64, 0x61, 0x3b, 0x73, 0x7a, 0x74, 0x77, 0x6f,
0x72, 0x74, 0x65, 0x6b, 0x3b, 0x70, 0x69, 0x14d, 0x6e, 0x74, 0x65, 0x6b,
0x3b, 0x73, 0x6f, 0x62, 0x6f, 0x74, 0x61, 0x6e, 0x69, 0x79, 0x3b, 0x70,
0x79, 0x144, 0x3b, 0x77, 0x74, 0x6f, 0x3b, 0x73, 0x74, 0x72, 0x3b, 0x73,
0x7a, 0x74, 0x3b, 0x70, 0x69, 0x14d, 0x3b, 0x73, 0x6f, 0x62, 0x622, 0x686,
0x631, 0x3b, 0x633, 0x648, 0x645, 0x631, 0x3b, 0x627, 0x6b1, 0x627, 0x631, 0x648,
0x3b, 0x627, 0x631, 0x628, 0x639, 0x3b, 0x62e, 0x645, 0x64a, 0x633, 0x3b, 0x62c,
0x645, 0x639, 0x648, 0x3b, 0x687, 0x646, 0x687, 0x631, 0x622, 0x686, 0x631, 0x3b,
0x633, 0x648, 0x3b, 0x627, 0x6b1, 0x627, 0x631, 0x648, 0x3b, 0x627, 0x631, 0x628,
0x639, 0x3b, 0x62e, 0x645, 0x3b, 0x62c, 0x645, 0x639, 0x648, 0x3b, 0x687, 0x646,
0x687, 0x631, 0x906, 0x930, 0x94d, 0x924, 0x3b, 0x938, 0x942, 0x3b, 0x92e, 0x902,
0x3b, 0x92c, 0x941, 0x952, 0x927, 0x3b, 0x935, 0x93f, 0x938, 0x3b, 0x91c, 0x941,
0x92e, 0x3b, 0x91b, 0x902, 0x91b, 0x906, 0x930, 0x94d, 0x924, 0x935, 0x93e, 0x930,
0x3b, 0x938, 0x942, 0x92e, 0x930, 0x3b, 0x92e, 0x902, 0x917, 0x932, 0x941, 0x3b,
0x92c, 0x941, 0x952, 0x927, 0x930, 0x3b, 0x935, 0x93f, 0x938, 0x94d, 0x92a, 0x924,
0x3b, 0x91c, 0x941, 0x92e, 0x94b, 0x3b, 0x91b, 0x902, 0x91b, 0x930, 0x906, 0x3b,
0x938, 0x942, 0x3b, 0x92e, 0x902, 0x3b, 0x92c, 0x941, 0x927, 0x3b, 0x935, 0x93f,
0x938, 0x3b, 0x91c, 0x941, 0x3b, 0x91b, 0x902, 0x91b, 0x906, 0x930, 0x94d, 0x924,
0x3b, 0x938, 0x942, 0x3b, 0x92e, 0x902, 0x917, 0x3b, 0x92c, 0x941, 0x952, 0x927,
0x3b, 0x935, 0x93f, 0x938, 0x3b, 0x91c, 0x941, 0x92e, 0x3b, 0x91b, 0x902, 0x91b,
0x906, 0x3b, 0x938, 0x942, 0x3b, 0x92e, 0x902, 0x3b, 0x92c, 0x941, 0x952, 0x3b,
0x935, 0x93f, 0x3b, 0x91c, 0x941, 0x3b, 0x91b, 0x902, 0xd89, 0xdbb, 0xdd2, 0xdaf,
0xdcf, 0x3b, 0xdc3, 0xdb3, 0xdd4, 0xdaf, 0xdcf, 0x3b, 0xd85, 0xd9f, 0xdc4, 0xdbb,
0xdd4, 0xdc0, 0xdcf, 0xdaf, 0xdcf, 0x3b, 0xdb6, 0xdaf, 0xdcf, 0xdaf, 0xdcf, 0x3b,
0xdb6, 0xdca, 0x200d, 0xdbb, 0xdc4, 0xdc3, 0xdca, 0xdb4, 0xdad, 0xdd2, 0xdb1, 0xdca,
0xdaf, 0xdcf, 0x3b, 0xdc3, 0xdd2, 0xd9a, 0xdd4, 0xdbb, 0xdcf, 0xdaf, 0xdcf, 0x3b,
0xdc3, 0xdd9, 0xdb1, 0xdc3, 0xdd4, 0xdbb, 0xdcf, 0xdaf, 0xdcf, 0xd89, 0xdbb, 0xdd2,
0xdaf, 0xdcf, 0x3b, 0xdc3, 0xdb3, 0xdd4, 0xdaf, 0xdcf, 0x3b, 0xd85, 0xd9f, 0xdc4,
0x3b, 0xdb6, 0xdaf, 0xdcf, 0xdaf, 0xdcf, 0x3b, 0xdb6, 0xdca, 0x200d, 0xdbb, 0xdc4,
0xdc3, 0xdca, 0x3b, 0xdc3, 0xdd2, 0xd9a, 0xdd4, 0x3b, 0xdc3, 0xdd9, 0xdb1, 0xd89,
0x3b, 0xdc3, 0x3b, 0xd85, 0x3b, 0xdb6, 0x3b, 0xdb6, 0xdca, 0x200d, 0xdbb, 0x3b,
0xdc3, 0xdd2, 0x3b, 0xdc3, 0xdd9, 0x6e, 0x65, 0x64, 0x65, 0x13e, 0x61, 0x3b,
0x70, 0x6f, 0x6e, 0x64, 0x65, 0x6c, 0x6f, 0x6b, 0x3b, 0x75, 0x74, 0x6f,
0x72, 0x6f, 0x6b, 0x3b, 0x73, 0x74, 0x72, 0x65, 0x64, 0x61, 0x3b, 0x161,
0x74, 0x76, 0x72, 0x74, 0x6f, 0x6b, 0x3b, 0x70, 0x69, 0x61, 0x74, 0x6f,
0x6b, 0x3b, 0x73, 0x6f, 0x62, 0x6f, 0x74, 0x61, 0x6e, 0x65, 0x3b, 0x70,
0x6f, 0x3b, 0x75, 0x74, 0x3b, 0x73, 0x74, 0x3b, 0x161, 0x74, 0x3b, 0x70,
0x69, 0x3b, 0x73, 0x6f, 0x6e, 0x3b, 0x70, 0x3b, 0x75, 0x3b, 0x73, 0x3b,
0x161, 0x3b, 0x70, 0x3b, 0x73, 0x6e, 0x65, 0x64, 0x65, 0x6c, 0x6a, 0x61,
0x3b, 0x70, 0x6f, 0x6e, 0x65, 0x64, 0x65, 0x6c, 0x6a, 0x65, 0x6b, 0x3b,
0x74, 0x6f, 0x72, 0x65, 0x6b, 0x3b, 0x73, 0x72, 0x65, 0x64, 0x61, 0x3b,
0x10d, 0x65, 0x74, 0x72, 0x74, 0x65, 0x6b, 0x3b, 0x70, 0x65, 0x74, 0x65,
0x6b, 0x3b, 0x73, 0x6f, 0x62, 0x6f, 0x74, 0x61, 0x6e, 0x65, 0x64, 0x2e,
0x3b, 0x70, 0x6f, 0x6e, 0x2e, 0x3b, 0x74, 0x6f, 0x72, 0x2e, 0x3b, 0x73,
0x72, 0x65, 0x2e, 0x3b, 0x10d, 0x65, 0x74, 0x2e, 0x3b, 0x70, 0x65, 0x74,
0x2e, 0x3b, 0x73, 0x6f, 0x62, 0x2e, 0x6e, 0x3b, 0x70, 0x3b, 0x74, 0x3b,
0x73, 0x3b, 0x10d, 0x3b, 0x70, 0x3b, 0x73, 0x53, 0x61, 0x62, 0x69, 0x69,
0x74, 0x69, 0x3b, 0x42, 0x61, 0x6c, 0x61, 0x7a, 0x61, 0x3b, 0x4f, 0x77,
0x6f, 0x6b, 0x75, 0x62, 0x69, 0x6c, 0x69, 0x3b, 0x4f, 0x77, 0x6f, 0x6b,
0x75, 0x73, 0x61, 0x74, 0x75, 0x3b, 0x4f, 0x6c, 0x6f, 0x6b, 0x75, 0x6e,
0x61, 0x3b, 0x4f, 0x6c, 0x6f, 0x6b, 0x75, 0x74, 0x61, 0x61, 0x6e, 0x75,
0x3b, 0x4f, 0x6c, 0x6f, 0x6d, 0x75, 0x6b, 0x61, 0x61, 0x67, 0x61, 0x53,
0x61, 0x62, 0x69, 0x3b, 0x42, 0x61, 0x6c, 0x61, 0x3b, 0x4b, 0x75, 0x62,
0x69, 0x3b, 0x4b, 0x75, 0x73, 0x61, 0x3b, 0x4b, 0x75, 0x6e, 0x61, 0x3b,
0x4b, 0x75, 0x74, 0x61, 0x3b, 0x4d, 0x75, 0x6b, 0x61, 0x53, 0x3b, 0x42,
0x3b, 0x42, 0x3b, 0x53, 0x3b, 0x4b, 0x3b, 0x4b, 0x3b, 0x4d, 0x41, 0x78,
0x61, 0x64, 0x3b, 0x49, 0x73, 0x6e, 0x69, 0x69, 0x6e, 0x3b, 0x54, 0x61,
0x6c, 0x61, 0x61, 0x64, 0x6f, 0x3b, 0x41, 0x72, 0x62, 0x61, 0x63, 0x6f,
0x3b, 0x4b, 0x68, 0x61, 0x6d, 0x69, 0x69, 0x73, 0x3b, 0x4a, 0x69, 0x6d,
0x63, 0x6f, 0x3b, 0x53, 0x61, 0x62, 0x74, 0x69, 0x41, 0x78, 0x64, 0x3b,
0x49, 0x73, 0x6e, 0x3b, 0x54, 0x6c, 0x64, 0x6f, 0x3b, 0x41, 0x72, 0x62,
0x63, 0x3b, 0x4b, 0x68, 0x6d, 0x73, 0x3b, 0x4a, 0x6d, 0x63, 0x3b, 0x53,
0x62, 0x74, 0x69, 0x41, 0x3b, 0x49, 0x3b, 0x54, 0x3b, 0x41, 0x3b, 0x4b,
0x68, 0x3b, 0x4a, 0x3b, 0x53, 0x53, 0x6f, 0x6e, 0x74, 0x61, 0x68, 0x61,
0x3b, 0x4d, 0x6d, 0x61, 0x6e, 0x74, 0x61, 0x68, 0x61, 0x3b, 0x4c, 0x61,
0x62, 0x6f, 0x62, 0x65, 0x64, 0x69, 0x3b, 0x4c, 0x61, 0x62, 0x6f, 0x72,
0x61, 0x72, 0x75, 0x3b, 0x4c, 0x61, 0x62, 0x6f, 0x6e, 0x65, 0x3b, 0x4c,
0x61, 0x62, 0x6f, 0x68, 0x6c, 0x61, 0x6e, 0x65, 0x3b, 0x4d, 0x6f, 0x71,
0x65, 0x62, 0x65, 0x6c, 0x6f, 0x53, 0x6f, 0x6e, 0x3b, 0x4d, 0x6d, 0x61,
0x3b, 0x42, 0x65, 0x64, 0x3b, 0x52, 0x61, 0x72, 0x3b, 0x4e, 0x65, 0x3b,
0x48, 0x6c, 0x61, 0x3b, 0x4d, 0x6f, 0x71, 0x75, 0x53, 0x6f, 0x6e, 0x74,
0x6f, 0x3b, 0x75, 0x4d, 0x76, 0x75, 0x6c, 0x6f, 0x3b, 0x75, 0x4c, 0x65,
0x73, 0x69, 0x62, 0x69, 0x6c, 0x69, 0x3b, 0x4c, 0x65, 0x73, 0x69, 0x74,
0x68, 0x61, 0x74, 0x68, 0x75, 0x3b, 0x75, 0x4c, 0x65, 0x73, 0x69, 0x6e,
0x65, 0x3b, 0x6e, 0x67, 0x6f, 0x4c, 0x65, 0x73, 0x69, 0x68, 0x6c, 0x61,
0x6e, 0x75, 0x3b, 0x75, 0x6d, 0x47, 0x71, 0x69, 0x62, 0x65, 0x6c, 0x6f,
0x53, 0x6f, 0x6e, 0x3b, 0x4d, 0x76, 0x75, 0x3b, 0x42, 0x69, 0x6c, 0x3b,
0x54, 0x68, 0x61, 0x3b, 0x4e, 0x65, 0x3b, 0x48, 0x6c, 0x61, 0x3b, 0x47,
0x71, 0x69, 0x64, 0x6f, 0x6d, 0x69, 0x6e, 0x67, 0x6f, 0x3b, 0x6c, 0x75,
0x6e, 0x65, 0x73, 0x3b, 0x6d, 0x61, 0x72, 0x74, 0x65, 0x73, 0x3b, 0x6d,
0x69, 0xe9, 0x72, 0x63, 0x6f, 0x6c, 0x65, 0x73, 0x3b, 0x6a, 0x75, 0x65,
0x76, 0x65, 0x73, 0x3b, 0x76, 0x69, 0x65, 0x72, 0x6e, 0x65, 0x73, 0x3b,
0x73, 0xe1, 0x62, 0x61, 0x64, 0x6f, 0x64, 0x6f, 0x6d, 0x3b, 0x6c, 0x75,
0x6e, 0x3b, 0x6d, 0x61, 0x72, 0x3b, 0x6d, 0x69, 0xe9, 0x3b, 0x6a, 0x75,
0x65, 0x3b, 0x76, 0x69, 0x65, 0x3b, 0x73, 0xe1, 0x62, 0x44, 0x3b, 0x4c,
0x3b, 0x4d, 0x3b, 0x6d, 0x3b, 0x4a, 0x3b, 0x56, 0x3b, 0x53, 0x2d30, 0x2d59,
0x2d30, 0x2d4e, 0x2d30, 0x2d59, 0x3b, 0x2d30, 0x2d62, 0x2d4f, 0x2d30, 0x2d59, 0x3b, 0x2d30,
0x2d59, 0x2d49, 0x2d4f, 0x2d30, 0x2d59, 0x3b, 0x2d30, 0x2d3d, 0x2d55, 0x2d30, 0x2d59, 0x3b,
0x2d30, 0x2d3d, 0x2d61, 0x2d30, 0x2d59, 0x3b, 0x2d30, 0x2d59, 0x2d49, 0x2d4e, 0x2d61, 0x2d30,
0x2d59, 0x3b, 0x2d30, 0x2d59, 0x2d49, 0x2d39, 0x2d62, 0x2d30, 0x2d59, 0x2d30, 0x2d59, 0x2d30,
0x3b, 0x2d30, 0x2d62, 0x2d4f, 0x3b, 0x2d30, 0x2d59, 0x2d49, 0x3b, 0x2d30, 0x2d3d, 0x2d55,
0x3b, 0x2d30, 0x2d3d, 0x2d61, 0x3b, 0x2d30, 0x2d59, 0x2d49, 0x2d4e, 0x3b, 0x2d30, 0x2d59,
0x2d49, 0x2d39, 0x4d, 0x69, 0x6e, 0x67, 0x67, 0x75, 0x3b, 0x53, 0x65, 0x6e,
0xe9, 0x6e, 0x3b, 0x53, 0x61, 0x6c, 0x61, 0x73, 0x61, 0x3b, 0x52, 0x65,
0x62, 0x6f, 0x3b, 0x4b, 0x65, 0x6d, 0x69, 0x73, 0x3b, 0x4a, 0x75, 0x6d,
0x61, 0x61, 0x68, 0x3b, 0x53, 0x61, 0x70, 0x74, 0x75, 0x4d, 0x6e, 0x67,
0x3b, 0x53, 0x65, 0x6e, 0x3b, 0x53, 0x61, 0x6c, 0x3b, 0x52, 0x65, 0x62,
0x3b, 0x4b, 0x65, 0x6d, 0x3b, 0x4a, 0x75, 0x6d, 0x3b, 0x53, 0x61, 0x70,
0x4c, 0x69, 0x73, 0x6f, 0x6e, 0x74, 0x66, 0x6f, 0x3b, 0x75, 0x4d, 0x73,
0x6f, 0x6d, 0x62, 0x75, 0x6c, 0x75, 0x6b, 0x6f, 0x3b, 0x4c, 0x65, 0x73,
0x69, 0x62, 0x69, 0x6c, 0x69, 0x3b, 0x4c, 0x65, 0x73, 0x69, 0x74, 0x73,
0x61, 0x74, 0x66, 0x75, 0x3b, 0x4c, 0x65, 0x73, 0x69, 0x6e, 0x65, 0x3b,
0x4c, 0x65, 0x73, 0x69, 0x68, 0x6c, 0x61, 0x6e, 0x75, 0x3b, 0x75, 0x4d,
0x67, 0x63, 0x69, 0x62, 0x65, 0x6c, 0x6f, 0x53, 0x6f, 0x6e, 0x3b, 0x4d,
0x73, 0x6f, 0x3b, 0x42, 0x69, 0x6c, 0x3b, 0x54, 0x73, 0x61, 0x3b, 0x4e,
0x65, 0x3b, 0x48, 0x6c, 0x61, 0x3b, 0x4d, 0x67, 0x63, 0x73, 0xf6, 0x6e,
0x64, 0x61, 0x67, 0x3b, 0x6d, 0xe5, 0x6e, 0x64, 0x61, 0x67, 0x3b, 0x74,
0x69, 0x73, 0x64, 0x61, 0x67, 0x3b, 0x6f, 0x6e, 0x73, 0x64, 0x61, 0x67,
0x3b, 0x74, 0x6f, 0x72, 0x73, 0x64, 0x61, 0x67, 0x3b, 0x66, 0x72, 0x65,
0x64, 0x61, 0x67, 0x3b, 0x6c, 0xf6, 0x72, 0x64, 0x61, 0x67, 0x73, 0xf6,
0x6e, 0x3b, 0x6d, 0xe5, 0x6e, 0x3b, 0x74, 0x69, 0x73, 0x3b, 0x6f, 0x6e,
0x73, 0x3b, 0x74, 0x6f, 0x72, 0x73, 0x3b, 0x66, 0x72, 0x65, 0x3b, 0x6c,
0xf6, 0x72, 0x53, 0x75, 0x6e, 0x6e, 0x74, 0x69, 0x67, 0x3b, 0x4d, 0xe4,
0xe4, 0x6e, 0x74, 0x69, 0x67, 0x3b, 0x5a, 0x69, 0x69, 0x73, 0x63, 0x68,
0x74, 0x69, 0x67, 0x3b, 0x4d, 0x69, 0x74, 0x74, 0x77, 0x75, 0x63, 0x68,
0x3b, 0x44, 0x75, 0x6e, 0x73, 0x63, 0x68, 0x74, 0x69, 0x67, 0x3b, 0x46,
0x72, 0x69, 0x69, 0x74, 0x69, 0x67, 0x3b, 0x53, 0x61, 0x6d, 0x73, 0x63,
0x68, 0x74, 0x69, 0x67, 0x53, 0x75, 0x2e, 0x3b, 0x4d, 0xe4, 0x2e, 0x3b,
0x5a, 0x69, 0x2e, 0x3b, 0x4d, 0x69, 0x2e, 0x3b, 0x44, 0x75, 0x2e, 0x3b,
0x46, 0x72, 0x2e, 0x3b, 0x53, 0x61, 0x2e, 0x71a, 0x715, 0x712, 0x72b, 0x712,
0x710, 0x3b, 0x72c, 0x72a, 0x71d, 0x722, 0x712, 0x72b, 0x712, 0x710, 0x3b, 0x72c,
0x720, 0x72c, 0x712, 0x72b, 0x712, 0x710, 0x3b, 0x710, 0x72a, 0x712, 0x725, 0x712,
0x72b, 0x712, 0x710, 0x3b, 0x71a, 0x721, 0x72b, 0x712, 0x72b, 0x712, 0x710, 0x3b,
0x725, 0x72a, 0x718, 0x712, 0x72c, 0x710, 0x3b, 0x72b, 0x712, 0x72c, 0x710, 0x71a,
0x715, 0x3b, 0x72c, 0x72a, 0x71d, 0x722, 0x3b, 0x72c, 0x720, 0x72c, 0x3b, 0x710,
0x72a, 0x712, 0x725, 0x3b, 0x71a, 0x721, 0x72b, 0x3b, 0x725, 0x72a, 0x718, 0x3b,
0x72b, 0x712, 0x72c, 0x710, 0x71a, 0x3b, 0x72c, 0x3b, 0x72c, 0x3b, 0x710, 0x3b,
0x71a, 0x3b, 0x725, 0x3b, 0x72b, 0x2d30, 0x2d59, 0x2d30, 0x2d4e, 0x2d30, 0x2d59, 0x3b,
0x2d30, 0x2d62, 0x2d4f, 0x2d30, 0x2d59, 0x3b, 0x2d30, 0x2d59, 0x2d49, 0x2d4f, 0x2d30, 0x2d59,
0x3b, 0x2d30, 0x2d3d, 0x2d55, 0x2d30, 0x2d59, 0x3b, 0x2d30, 0x2d3d, 0x2d61, 0x2d30, 0x2d59,
0x3b, 0x2d59, 0x2d49, 0x2d4e, 0x2d61, 0x2d30, 0x2d59, 0x3b, 0x2d30, 0x2d59, 0x2d49, 0x2d39,
0x2d62, 0x2d30, 0x2d59, 0x61, 0x73, 0x61, 0x6d, 0x61, 0x73, 0x3b, 0x61, 0x79,
0x6e, 0x61, 0x73, 0x3b, 0x61, 0x73, 0x69, 0x6e, 0x61, 0x73, 0x3b, 0x61,
0x6b, 0x1e5b, 0x61, 0x73, 0x3b, 0x61, 0x6b, 0x77, 0x61, 0x73, 0x3b, 0x61,
0x73, 0x69, 0x6d, 0x77, 0x61, 0x73, 0x3b, 0x61, 0x73, 0x69, 0x1e0d, 0x79,
0x61, 0x73, 0x61, 0x73, 0x61, 0x3b, 0x61, 0x79, 0x6e, 0x3b, 0x61, 0x73,
0x69, 0x3b, 0x61, 0x6b, 0x1e5b, 0x3b, 0x61, 0x6b, 0x77, 0x3b, 0x61, 0x73,
0x69, 0x6d, 0x3b, 0x61, 0x73, 0x69, 0x1e0d, 0x49, 0x74, 0x75, 0x6b, 0x75,
0x20, 0x6a, 0x61, 0x20, 0x6a, 0x75, 0x6d, 0x77, 0x61, 0x3b, 0x4b, 0x75,
0x72, 0x61, 0x6d, 0x75, 0x6b, 0x61, 0x20, 0x6a, 0x69, 0x6d, 0x77, 0x65,
0x72, 0x69, 0x3b, 0x4b, 0x75, 0x72, 0x61, 0x6d, 0x75, 0x6b, 0x61, 0x20,
0x6b, 0x61, 0x77, 0x69, 0x3b, 0x4b, 0x75, 0x72, 0x61, 0x6d, 0x75, 0x6b,
0x61, 0x20, 0x6b, 0x61, 0x64, 0x61, 0x64, 0x75, 0x3b, 0x4b, 0x75, 0x72,
0x61, 0x6d, 0x75, 0x6b, 0x61, 0x20, 0x6b, 0x61, 0x6e, 0x61, 0x3b, 0x4b,
0x75, 0x72, 0x61, 0x6d, 0x75, 0x6b, 0x61, 0x20, 0x6b, 0x61, 0x73, 0x61,
0x6e, 0x75, 0x3b, 0x4b, 0x69, 0x66, 0x75, 0x6c, 0x61, 0x20, 0x6e, 0x67,
0x75, 0x77, 0x6f, 0x4a, 0x75, 0x6d, 0x3b, 0x4a, 0x69, 0x6d, 0x3b, 0x4b,
0x61, 0x77, 0x3b, 0x4b, 0x61, 0x64, 0x3b, 0x4b, 0x61, 0x6e, 0x3b, 0x4b,
0x61, 0x73, 0x3b, 0x4e, 0x67, 0x75, 0x4a, 0x3b, 0x4a, 0x3b, 0x4b, 0x3b,
0x4b, 0x3b, 0x4b, 0x3b, 0x4b, 0x3b, 0x4e, 0x42f, 0x43a, 0x448, 0x430, 0x43d,
0x431, 0x435, 0x3b, 0x414, 0x443, 0x448, 0x430, 0x43d, 0x431, 0x435, 0x3b, 0x421,
0x435, 0x448, 0x430, 0x43d, 0x431, 0x435, 0x3b, 0x427, 0x43e, 0x440, 0x448, 0x430,
0x43d, 0x431, 0x435, 0x3b, 0x41f, 0x430, 0x43d, 0x4b7, 0x448, 0x430, 0x43d, 0x431,
0x435, 0x3b, 0x4b6, 0x443, 0x43c, 0x44a, 0x430, 0x3b, 0x428, 0x430, 0x43d, 0x431,
0x435, 0x42f, 0x448, 0x431, 0x3b, 0x414, 0x448, 0x431, 0x3b, 0x421, 0x448, 0x431,
0x3b, 0x427, 0x448, 0x431, 0x3b, 0x41f, 0x448, 0x431, 0x3b, 0x4b6, 0x43c, 0x44a,
0x3b, 0x428, 0x43d, 0x431, 0x42f, 0x3b, 0x414, 0x3b, 0x421, 0x3b, 0x427, 0x3b,
0x41f, 0x3b, 0x4b6, 0x3b, 0x428, 0xb9e, 0xbbe, 0xbaf, 0xbbf, 0xbb1, 0xbc1, 0x3b,
0xba4, 0xbbf, 0xb99, 0xbcd, 0xb95, 0xbb3, 0xbcd, 0x3b, 0xb9a, 0xbc6, 0xbb5, 0xbcd,
0xbb5, 0xbbe, 0xbaf, 0xbcd, 0x3b, 0xbaa, 0xbc1, 0xba4, 0xba9, 0xbcd, 0x3b, 0xbb5,
0xbbf, 0xbaf, 0xbbe, 0xbb4, 0xba9, 0xbcd, 0x3b, 0xbb5, 0xbc6, 0xbb3, 0xbcd, 0xbb3,
0xbbf, 0x3b, 0xb9a, 0xba9, 0xbbf, 0xb9e, 0xbbe, 0xbaf, 0xbbf, 0x2e, 0x3b, 0xba4,
0xbbf, 0xb99, 0xbcd, 0x2e, 0x3b, 0xb9a, 0xbc6, 0xbb5, 0xbcd, 0x2e, 0x3b, 0xbaa,
0xbc1, 0xba4, 0x2e, 0x3b, 0xbb5, 0xbbf, 0xbaf, 0xbbe, 0x2e, 0x3b, 0xbb5, 0xbc6,
0xbb3, 0xbcd, 0x2e, 0x3b, 0xb9a, 0xba9, 0xbbf, 0xb9e, 0xbbe, 0x3b, 0xba4, 0xbbf,
0x3b, 0xb9a, 0xbc6, 0x3b, 0xbaa, 0xbc1, 0x3b, 0xbb5, 0xbbf, 0x3b, 0xbb5, 0xbc6,
0x3b, 0xb9a, 0x4a, 0x69, 0x79, 0x61, 0x78, 0x20, 0x73, 0x6e, 0x67, 0x61,
0x79, 0x61, 0x6e, 0x3b, 0x74, 0x67, 0x4b, 0x69, 0x6e, 0x67, 0x61, 0x6c,
0x20, 0x6a, 0x69, 0x79, 0x61, 0x78, 0x20, 0x69, 0x79, 0x61, 0x78, 0x20,
0x73, 0x6e, 0x67, 0x61, 0x79, 0x61, 0x6e, 0x3b, 0x74, 0x67, 0x44, 0x68,
0x61, 0x20, 0x6a, 0x69, 0x79, 0x61, 0x78, 0x20, 0x69, 0x79, 0x61, 0x78,
0x20, 0x73, 0x6e, 0x67, 0x61, 0x79, 0x61, 0x6e, 0x3b, 0x74, 0x67, 0x54,
0x72, 0x75, 0x20, 0x6a, 0x69, 0x79, 0x61, 0x78, 0x20, 0x69, 0x79, 0x61,
0x78, 0x20, 0x73, 0x6e, 0x67, 0x61, 0x79, 0x61, 0x6e, 0x3b, 0x74, 0x67,
0x53, 0x70, 0x61, 0x63, 0x20, 0x6a, 0x69, 0x79, 0x61, 0x78, 0x20, 0x69,
0x79, 0x61, 0x78, 0x20, 0x73, 0x6e, 0x67, 0x61, 0x79, 0x61, 0x6e, 0x3b,
0x74, 0x67, 0x52, 0x69, 0x6d, 0x61, 0x20, 0x6a, 0x69, 0x79, 0x61, 0x78,
0x20, 0x69, 0x79, 0x61, 0x78, 0x20, 0x73, 0x6e, 0x67, 0x61, 0x79, 0x61,
0x6e, 0x3b, 0x74, 0x67, 0x4d, 0x61, 0x74, 0x61, 0x72, 0x75, 0x20, 0x6a,
0x69, 0x79, 0x61, 0x78, 0x20, 0x69, 0x79, 0x61, 0x78, 0x20, 0x73, 0x6e,
0x67, 0x61, 0x79, 0x61, 0x6e, 0x45, 0x6d, 0x70, 0x3b, 0x4b, 0x69, 0x6e,
0x3b, 0x44, 0x68, 0x61, 0x3b, 0x54, 0x72, 0x75, 0x3b, 0x53, 0x70, 0x61,
0x3b, 0x52, 0x69, 0x6d, 0x3b, 0x4d, 0x61, 0x74, 0x45, 0x3b, 0x4b, 0x3b,
0x44, 0x3b, 0x54, 0x3b, 0x53, 0x3b, 0x52, 0x3b, 0x4d, 0x44f, 0x43a, 0x448,
0x4d9, 0x43c, 0x431, 0x435, 0x3b, 0x434, 0x4af, 0x448, 0x4d9, 0x43c, 0x431, 0x435,
0x3b, 0x441, 0x438, 0x448, 0x4d9, 0x43c, 0x431, 0x435, 0x3b, 0x447, 0x4d9, 0x440,
0x448, 0x4d9, 0x43c, 0x431, 0x435, 0x3b, 0x43f, 0x4d9, 0x43d, 0x497, 0x435, 0x448,
0x4d9, 0x43c, 0x431, 0x435, 0x3b, 0x497, 0x43e, 0x43c, 0x433, 0x430, 0x3b, 0x448,
0x438, 0x43c, 0x431, 0x4d9, 0x44f, 0x43a, 0x448, 0x2e, 0x3b, 0x434, 0x4af, 0x448,
0x2e, 0x3b, 0x441, 0x438, 0x448, 0x2e, 0x3b, 0x447, 0x4d9, 0x440, 0x2e, 0x3b,
0x43f, 0x4d9, 0x43d, 0x497, 0x2e, 0x3b, 0x497, 0x43e, 0x43c, 0x2e, 0x3b, 0x448,
0x438, 0x43c, 0x2e, 0x42f, 0x3b, 0x414, 0x3b, 0x421, 0x3b, 0x427, 0x3b, 0x41f,
0x3b, 0x496, 0x3b, 0x428, 0xc06, 0xc26, 0xc3f, 0xc35, 0xc3e, 0xc30, 0xc02, 0x3b,
0xc38, 0xc4b, 0xc2e, 0xc35, 0xc3e, 0xc30, 0xc02, 0x3b, 0xc2e, 0xc02, 0xc17, 0xc33,
0xc35, 0xc3e, 0xc30, 0xc02, 0x3b, 0xc2c, 0xc41, 0xc27, 0xc35, 0xc3e, 0xc30, 0xc02,
0x3b, 0xc17, 0xc41, 0xc30, 0xc41, 0xc35, 0xc3e, 0xc30, 0xc02, 0x3b, 0xc36, 0xc41,
0xc15, 0xc4d, 0xc30, 0xc35, 0xc3e, 0xc30, 0xc02, 0x3b, 0xc36, 0xc28, 0xc3f, 0xc35,
0xc3e, 0xc30, 0xc02, 0xc06, 0xc26, 0xc3f, 0x3b, 0xc38, 0xc4b, 0xc2e, 0x3b, 0xc2e,
0xc02, 0xc17, 0xc33, 0x3b, 0xc2c, 0xc41, 0xc27, 0x3b, 0xc17, 0xc41, 0xc30, 0xc41,
0x3b, 0xc36, 0xc41, 0xc15, 0xc4d, 0xc30, 0x3b, 0xc36, 0xc28, 0xc3f, 0xc06, 0x3b,
0xc38, 0xc4b, 0x3b, 0xc2e, 0x3b, 0xc2c, 0xc41, 0x3b, 0xc17, 0xc41, 0x3b, 0xc36,
0xc41, 0x3b, 0xc36, 0x4e, 0x61, 0x6b, 0x61, 0x65, 0x6a, 0x75, 0x6d, 0x61,
0x3b, 0x4e, 0x61, 0x6b, 0x61, 0x65, 0x62, 0x61, 0x72, 0x61, 0x73, 0x61,
0x3b, 0x4e, 0x61, 0x6b, 0x61, 0x61, 0x72, 0x65, 0x3b, 0x4e, 0x61, 0x6b,
0x61, 0x75, 0x6e, 0x69, 0x3b, 0x4e, 0x61, 0x6b, 0x61, 0x75, 0x6e, 0x67,
0x2019, 0x6f, 0x6e, 0x3b, 0x4e, 0x61, 0x6b, 0x61, 0x6b, 0x61, 0x6e, 0x79,
0x3b, 0x4e, 0x61, 0x6b, 0x61, 0x73, 0x61, 0x62, 0x69, 0x74, 0x69, 0x4a,
0x75, 0x6d, 0x3b, 0x42, 0x61, 0x72, 0x3b, 0x41, 0x61, 0x72, 0x3b, 0x55,
0x6e, 0x69, 0x3b, 0x55, 0x6e, 0x67, 0x3b, 0x4b, 0x61, 0x6e, 0x3b, 0x53,
0x61, 0x62, 0x4a, 0x3b, 0x42, 0x3b, 0x41, 0x3b, 0x55, 0x3b, 0x55, 0x3b,
0x4b, 0x3b, 0x53, 0xe27, 0xe31, 0xe19, 0xe2d, 0xe32, 0xe17, 0xe34, 0xe15, 0xe22,
0xe4c, 0x3b, 0xe27, 0xe31, 0xe19, 0xe08, 0xe31, 0xe19, 0xe17, 0xe23, 0xe4c, 0x3b,
0xe27, 0xe31, 0xe19, 0xe2d, 0xe31, 0xe07, 0xe04, 0xe32, 0xe23, 0x3b, 0xe27, 0xe31,
0xe19, 0xe1e, 0xe38, 0xe18, 0x3b, 0xe27, 0xe31, 0xe19, 0xe1e, 0xe24, 0xe2b, 0xe31,
0xe2a, 0xe1a, 0xe14, 0xe35, 0x3b, 0xe27, 0xe31, 0xe19, 0xe28, 0xe38, 0xe01, 0xe23,
0xe4c, 0x3b, 0xe27, 0xe31, 0xe19, 0xe40, 0xe2a, 0xe32, 0xe23, 0xe4c, 0xe2d, 0xe32,
0x2e, 0x3b, 0xe08, 0x2e, 0x3b, 0xe2d, 0x2e, 0x3b, 0xe1e, 0x2e, 0x3b, 0xe1e,
0xe24, 0x2e, 0x3b, 0xe28, 0x2e, 0x3b, 0xe2a, 0x2e, 0xe2d, 0xe32, 0x3b, 0xe08,
0x3b, 0xe2d, 0x3b, 0xe1e, 0x3b, 0xe1e, 0xe24, 0x3b, 0xe28, 0x3b, 0xe2a, 0xf42,
0xf5f, 0xf60, 0xf0b, 0xf49, 0xf72, 0xf0b, 0xf58, 0xf0b, 0x3b, 0xf42, 0xf5f, 0xf60,
0xf0b, 0xf5f, 0xfb3, 0xf0b, 0xf56, 0xf0b, 0x3b, 0xf42, 0xf5f, 0xf60, 0xf0b, 0xf58,
0xf72, 0xf42, 0xf0b, 0xf51, 0xf58, 0xf62, 0xf0b, 0x3b, 0xf42, 0xf5f, 0xf60, 0xf0b,
0xf63, 0xfb7, 0xf42, 0xf0b, 0xf54, 0xf0b, 0x3b, 0xf42, 0xf5f, 0xf60, 0xf0b, 0xf55,
0xf74, 0xf62, 0xf0b, 0xf56, 0xf74, 0xf0b, 0x3b, 0xf42, 0xf5f, 0xf60, 0xf0b, 0xf54,
0xf0b, 0xf66, 0xf44, 0xf66, 0xf0b, 0x3b, 0xf42, 0xf5f, 0xf60, 0xf0b, 0xf66, 0xfa4,
0xf7a, 0xf53, 0xf0b, 0xf54, 0xf0b, 0xf49, 0xf72, 0xf0b, 0xf58, 0xf0b, 0x3b, 0xf5f,
0xfb3, 0xf0b, 0xf56, 0xf0b, 0x3b, 0xf58, 0xf72, 0xf42, 0xf0b, 0xf51, 0xf58, 0xf62,
0xf0b, 0x3b, 0xf63, 0xfb7, 0xf42, 0xf0b, 0xf54, 0xf0b, 0x3b, 0xf55, 0xf74, 0xf62,
0xf0b, 0xf56, 0xf74, 0xf0b, 0x3b, 0xf54, 0xf0b, 0xf66, 0xf44, 0xf66, 0xf0b, 0x3b,
0xf66, 0xfa4, 0xf7a, 0xf53, 0xf0b, 0xf54, 0xf0b, 0xf49, 0xf72, 0x3b, 0xf5f, 0xfb3,
0x3b, 0xf58, 0xf72, 0xf42, 0x3b, 0xf63, 0xfb7, 0xf42, 0x3b, 0xf55, 0xf74, 0xf62,
0x3b, 0xf66, 0xf44, 0xf66, 0x3b, 0xf66, 0xfa4, 0xf7a, 0xf53, 0x1230, 0x1295, 0x1260,
0x1275, 0x20, 0x12d3, 0x1263, 0x12ed, 0x3b, 0x1230, 0x1296, 0x3b, 0x1273, 0x120b, 0x1238,
0x1296, 0x3b, 0x12a3, 0x1228, 0x122d, 0x1263, 0x12d3, 0x3b, 0x12a8, 0x121a, 0x123d, 0x3b,
0x1305, 0x121d, 0x12d3, 0x1275, 0x3b, 0x1230, 0x1295, 0x1260, 0x1275, 0x20, 0x1295, 0x12a2,
0x123d, 0x1230, 0x2f, 0x12d3, 0x3b, 0x1230, 0x1296, 0x3b, 0x1273, 0x120b, 0x1238, 0x3b,
0x12a3, 0x1228, 0x122d, 0x3b, 0x12a8, 0x121a, 0x123d, 0x3b, 0x1305, 0x121d, 0x12d3, 0x3b,
0x1230, 0x2f, 0x1295, 0x1230, 0x3b, 0x1230, 0x3b, 0x1273, 0x3b, 0x12a3, 0x3b, 0x12a8,
0x3b, 0x1305, 0x3b, 0x1230, 0x1230, 0x1295, 0x1260, 0x1275, 0x3b, 0x1230, 0x1291, 0x12ed,
0x3b, 0x1230, 0x1209, 0x1235, 0x3b, 0x1228, 0x1261, 0x12d5, 0x3b, 0x1213, 0x1219, 0x1235,
0x3b, 0x12d3, 0x122d, 0x1262, 0x3b, 0x1240, 0x12f3, 0x121d, 0x1230, 0x1295, 0x3b, 0x1230,
0x1291, 0x3b, 0x1230, 0x1209, 0x3b, 0x1228, 0x1261, 0x3b, 0x1213, 0x1219, 0x3b, 0x12d3,
0x122d, 0x3b, 0x1240, 0x12f3, 0x1230, 0x3b, 0x1230, 0x3b, 0x1230, 0x3b, 0x1228, 0x3b,
0x1213, 0x3b, 0x12d3, 0x3b, 0x1240, 0x53, 0x61, 0x6e, 0x64, 0x65, 0x3b, 0x4d,
0x61, 0x6e, 0x64, 0x65, 0x3b, 0x54, 0x75, 0x6e, 0x64, 0x65, 0x3b, 0x54,
0x72, 0x69, 0x6e, 0x64, 0x65, 0x3b, 0x46, 0x6f, 0x6e, 0x64, 0x65, 0x3b,
0x46, 0x72, 0x61, 0x69, 0x64, 0x65, 0x3b, 0x53, 0x61, 0x72, 0x65, 0x72,
0x65, 0x53, 0x61, 0x6e, 0x3b, 0x4d, 0x61, 0x6e, 0x3b, 0x54, 0x75, 0x6e,
0x3b, 0x54, 0x72, 0x69, 0x3b, 0x46, 0x6f, 0x6e, 0x3b, 0x46, 0x72, 0x61,
0x3b, 0x53, 0x61, 0x72, 0x53, 0x101, 0x70, 0x61, 0x74, 0x65, 0x3b, 0x4d,
0x14d, 0x6e, 0x69, 0x74, 0x65, 0x3b, 0x54, 0x16b, 0x73, 0x69, 0x74, 0x65,
0x3b, 0x50, 0x75, 0x6c, 0x65, 0x6c, 0x75, 0x6c, 0x75, 0x3b, 0x54, 0x75,
0x2bb, 0x61, 0x70, 0x75, 0x6c, 0x65, 0x6c, 0x75, 0x6c, 0x75, 0x3b, 0x46,
0x61, 0x6c, 0x61, 0x69, 0x74, 0x65, 0x3b, 0x54, 0x6f, 0x6b, 0x6f, 0x6e,
0x61, 0x6b, 0x69, 0x53, 0x101, 0x70, 0x3b, 0x4d, 0x14d, 0x6e, 0x3b, 0x54,
0x16b, 0x73, 0x3b, 0x50, 0x75, 0x6c, 0x3b, 0x54, 0x75, 0x2bb, 0x61, 0x3b,
0x46, 0x61, 0x6c, 0x3b, 0x54, 0x6f, 0x6b, 0x53, 0x3b, 0x4d, 0x3b, 0x54,
0x3b, 0x50, 0x3b, 0x54, 0x3b, 0x46, 0x3b, 0x54, 0x53, 0x6f, 0x6e, 0x74,
0x61, 0x3b, 0x4d, 0x75, 0x73, 0x75, 0x6d, 0x62, 0x68, 0x75, 0x6e, 0x75,
0x6b, 0x75, 0x3b, 0x52, 0x61, 0x76, 0x75, 0x6d, 0x62, 0x69, 0x72, 0x68,
0x69, 0x3b, 0x52, 0x61, 0x76, 0x75, 0x6e, 0x68, 0x61, 0x72, 0x68, 0x75,
0x3b, 0x52, 0x61, 0x76, 0x75, 0x6d, 0x75, 0x6e, 0x65, 0x3b, 0x52, 0x61,
0x76, 0x75, 0x6e, 0x74, 0x6c, 0x68, 0x61, 0x6e, 0x75, 0x3b, 0x4d, 0x75,
0x67, 0x71, 0x69, 0x76, 0x65, 0x6c, 0x61, 0x53, 0x6f, 0x6e, 0x3b, 0x4d,
0x75, 0x73, 0x3b, 0x42, 0x69, 0x72, 0x3b, 0x48, 0x61, 0x72, 0x3b, 0x4e,
0x65, 0x3b, 0x54, 0x6c, 0x68, 0x3b, 0x4d, 0x75, 0x67, 0x54, 0x73, 0x68,
0x69, 0x70, 0x69, 0x3b, 0x4d, 0x6f, 0x73, 0x6f, 0x70, 0x75, 0x6c, 0x6f,
0x67, 0x6f, 0x3b, 0x4c, 0x61, 0x62, 0x6f, 0x62, 0x65, 0x64, 0x69, 0x3b,
0x4c, 0x61, 0x62, 0x6f, 0x72, 0x61, 0x72, 0x6f, 0x3b, 0x4c, 0x61, 0x62,
0x6f, 0x6e, 0x65, 0x3b, 0x4c, 0x61, 0x62, 0x6f, 0x74, 0x6c, 0x68, 0x61,
0x6e, 0x6f, 0x3b, 0x4d, 0x61, 0x74, 0x6c, 0x68, 0x61, 0x74, 0x73, 0x6f,
0x54, 0x73, 0x68, 0x3b, 0x4d, 0x6f, 0x73, 0x3b, 0x4c, 0x61, 0x62, 0x62,
0x3b, 0x4c, 0x61, 0x62, 0x72, 0x3b, 0x4c, 0x61, 0x62, 0x6e, 0x3b, 0x4c,
0x61, 0x62, 0x74, 0x3b, 0x4d, 0x61, 0x74, 0x50, 0x61, 0x7a, 0x61, 0x72,
0x3b, 0x50, 0x61, 0x7a, 0x61, 0x72, 0x74, 0x65, 0x73, 0x69, 0x3b, 0x53,
0x61, 0x6c, 0x131, 0x3b, 0xc7, 0x61, 0x72, 0x15f, 0x61, 0x6d, 0x62, 0x61,
0x3b, 0x50, 0x65, 0x72, 0x15f, 0x65, 0x6d, 0x62, 0x65, 0x3b, 0x43, 0x75,
0x6d, 0x61, 0x3b, 0x43, 0x75, 0x6d, 0x61, 0x72, 0x74, 0x65, 0x73, 0x69,
0x50, 0x61, 0x7a, 0x3b, 0x50, 0x7a, 0x74, 0x3b, 0x53, 0x61, 0x6c, 0x3b,
0xc7, 0x61, 0x72, 0x3b, 0x50, 0x65, 0x72, 0x3b, 0x43, 0x75, 0x6d, 0x3b,
0x43, 0x6d, 0x74, 0x50, 0x3b, 0x50, 0x3b, 0x53, 0x3b, 0xc7, 0x3b, 0x50,
0x3b, 0x43, 0x3b, 0x43, 0xdd, 0x65, 0x6b, 0x15f, 0x65, 0x6e, 0x62, 0x65,
0x3b, 0x44, 0x75, 0x15f, 0x65, 0x6e, 0x62, 0x65, 0x3b, 0x53, 0x69, 0x15f,
0x65, 0x6e, 0x62, 0x65, 0x3b, 0xc7, 0x61, 0x72, 0x15f, 0x65, 0x6e, 0x62,
0x65, 0x3b, 0x50, 0x65, 0x6e, 0x15f, 0x65, 0x6e, 0x62, 0x65, 0x3b, 0x41,
0x6e, 0x6e, 0x61, 0x3b, 0x15e, 0x65, 0x6e, 0x62, 0x65, 0xfd, 0x65, 0x6b,
0x15f, 0x65, 0x6e, 0x62, 0x65, 0x3b, 0x64, 0x75, 0x15f, 0x65, 0x6e, 0x62,
0x65, 0x3b, 0x73, 0x69, 0x15f, 0x65, 0x6e, 0x62, 0x65, 0x3b, 0xe7, 0x61,
0x72, 0x15f, 0x65, 0x6e, 0x62, 0x65, 0x3b, 0x70, 0x65, 0x6e, 0x15f, 0x65,
0x6e, 0x62, 0x65, 0x3b, 0x61, 0x6e, 0x6e, 0x61, 0x3b, 0x15f, 0x65, 0x6e,
0x62, 0x65, 0xdd, 0x65, 0x6b, 0x3b, 0x44, 0x75, 0x15f, 0x3b, 0x53, 0x69,
0x15f, 0x3b, 0xc7, 0x61, 0x72, 0x3b, 0x50, 0x65, 0x6e, 0x3b, 0x41, 0x6e,
0x6e, 0x3b, 0x15e, 0x65, 0x6e, 0xfd, 0x65, 0x6b, 0x3b, 0x64, 0x75, 0x15f,
0x3b, 0x73, 0x69, 0x15f, 0x3b, 0xe7, 0x61, 0x72, 0x3b, 0x70, 0x65, 0x6e,
0x3b, 0x61, 0x6e, 0x6e, 0x3b, 0x15f, 0x65, 0x6e, 0xdd, 0x3b, 0x44, 0x3b,
0x53, 0x3b, 0xc7, 0x3b, 0x50, 0x3b, 0x41, 0x3b, 0x15e, 0x4c, 0x61, 0x64,
0x69, 0x3b, 0x54, 0x61, 0x6e, 0x69, 0x69, 0x3b, 0x54, 0x61, 0x6c, 0x61,
0x74, 0x61, 0x3b, 0x4c, 0x61, 0x72, 0x62, 0x61, 0x3b, 0x4c, 0x61, 0x6d,
0x69, 0x74, 0x3b, 0x4a, 0x75, 0x6d, 0x61, 0x3b, 0x41, 0x73, 0x61, 0x62,
0x61, 0x74, 0x4c, 0x61, 0x64, 0x3b, 0x54, 0x61, 0x6e, 0x3b, 0x54, 0x61,
0x6c, 0x3b, 0x4c, 0x61, 0x72, 0x3b, 0x4c, 0x61, 0x6d, 0x3b, 0x4a, 0x75,
0x6d, 0x3b, 0x41, 0x73, 0x61, 0x43d, 0x435, 0x434, 0x456, 0x43b, 0x44f, 0x3b,
0x43f, 0x43e, 0x43d, 0x435, 0x434, 0x456, 0x43b, 0x43e, 0x43a, 0x3b, 0x432, 0x456,
0x432, 0x442, 0x43e, 0x440, 0x43e, 0x43a, 0x3b, 0x441, 0x435, 0x440, 0x435, 0x434,
0x430, 0x3b, 0x447, 0x435, 0x442, 0x432, 0x435, 0x440, 0x3b, 0x43f, 0x2bc, 0x44f,
0x442, 0x43d, 0x438, 0x446, 0x44f, 0x3b, 0x441, 0x443, 0x431, 0x43e, 0x442, 0x430,
0x43d, 0x435, 0x434, 0x456, 0x43b, 0x44e, 0x3b, 0x43f, 0x43e, 0x43d, 0x435, 0x434,
0x456, 0x43b, 0x43e, 0x43a, 0x3b, 0x432, 0x456, 0x432, 0x442, 0x43e, 0x440, 0x43e,
0x43a, 0x3b, 0x441, 0x435, 0x440, 0x435, 0x434, 0x443, 0x3b, 0x447, 0x435, 0x442,
0x432, 0x435, 0x440, 0x3b, 0x43f, 0x2bc, 0x44f, 0x442, 0x43d, 0x438, 0x446, 0x44e,
0x3b, 0x441, 0x443, 0x431, 0x43e, 0x442, 0x443, 0x6e, 0x6a, 0x65, 0x64, 0x17a,
0x65, 0x6c, 0x61, 0x3b, 0x70, 0xf3, 0x6e, 0x64, 0x17a, 0x65, 0x6c, 0x61,
0x3b, 0x77, 0x75, 0x74, 0x6f, 0x72, 0x61, 0x3b, 0x73, 0x72, 0x6a, 0x65,
0x64, 0x61, 0x3b, 0x161, 0x74, 0x77, 0xf3, 0x72, 0x74, 0x6b, 0x3b, 0x70,
0x6a, 0x61, 0x74, 0x6b, 0x3b, 0x73, 0x6f, 0x62, 0x6f, 0x74, 0x61, 0x6e,
0x6a, 0x65, 0x3b, 0x70, 0xf3, 0x6e, 0x3b, 0x77, 0x75, 0x74, 0x3b, 0x73,
0x72, 0x6a, 0x3b, 0x161, 0x74, 0x77, 0x3b, 0x70, 0x6a, 0x61, 0x3b, 0x73,
0x6f, 0x62, 0x6e, 0x3b, 0x70, 0x3b, 0x77, 0x3b, 0x73, 0x3b, 0x161, 0x3b,
0x70, 0x3b, 0x73, 0x627, 0x62a, 0x648, 0x627, 0x631, 0x3b, 0x67e, 0x6cc, 0x631,
0x3b, 0x645, 0x646, 0x6af, 0x644, 0x3b, 0x628, 0x62f, 0x6be, 0x3b, 0x62c, 0x645,
0x639, 0x631, 0x627, 0x62a, 0x3b, 0x62c, 0x645, 0x639, 0x6c1, 0x3b, 0x6c1, 0x641,
0x62a, 0x6c1, 0x64a, 0x6d5, 0x643, 0x634, 0x6d5, 0x646, 0x628, 0x6d5, 0x3b, 0x62f,
0x6c8, 0x634, 0x6d5, 0x646, 0x628, 0x6d5, 0x3b, 0x633, 0x6d5, 0x64a, 0x634, 0x6d5,
0x646, 0x628, 0x6d5, 0x3b, 0x686, 0x627, 0x631, 0x634, 0x6d5, 0x646, 0x628, 0x6d5,
0x3b, 0x67e, 0x6d5, 0x64a, 0x634, 0x6d5, 0x646, 0x628, 0x6d5, 0x3b, 0x62c, 0x6c8,
0x645, 0x6d5, 0x3b, 0x634, 0x6d5, 0x646, 0x628, 0x6d5, 0x64a, 0x6d5, 0x3b, 0x62f,
0x6c8, 0x3b, 0x633, 0x6d5, 0x3b, 0x686, 0x627, 0x3b, 0x67e, 0x6d5, 0x3b, 0x62c,
0x6c8, 0x3b, 0x634, 0x6d5, 0x64a, 0x3b, 0x62f, 0x3b, 0x633, 0x3b, 0x686, 0x3b,
0x67e, 0x3b, 0x62c, 0x3b, 0x634, 0x79, 0x61, 0x6b, 0x73, 0x68, 0x61, 0x6e,
0x62, 0x61, 0x3b, 0x64, 0x75, 0x73, 0x68, 0x61, 0x6e, 0x62, 0x61, 0x3b,
0x73, 0x65, 0x73, 0x68, 0x61, 0x6e, 0x62, 0x61, 0x3b, 0x63, 0x68, 0x6f,
0x72, 0x73, 0x68, 0x61, 0x6e, 0x62, 0x61, 0x3b, 0x70, 0x61, 0x79, 0x73,
0x68, 0x61, 0x6e, 0x62, 0x61, 0x3b, 0x6a, 0x75, 0x6d, 0x61, 0x3b, 0x73,
0x68, 0x61, 0x6e, 0x62, 0x61, 0x59, 0x61, 0x6b, 0x3b, 0x44, 0x75, 0x73,
0x68, 0x3b, 0x53, 0x65, 0x73, 0x68, 0x3b, 0x43, 0x68, 0x6f, 0x72, 0x3b,
0x50, 0x61, 0x79, 0x3b, 0x4a, 0x75, 0x6d, 0x3b, 0x53, 0x68, 0x61, 0x6e,
0x59, 0x3b, 0x44, 0x3b, 0x53, 0x3b, 0x43, 0x3b, 0x50, 0x3b, 0x4a, 0x3b,
0x53, 0x6cc, 0x2e, 0x3b, 0x62f, 0x2e, 0x3b, 0x633, 0x2e, 0x3b, 0x686, 0x2e,
0x3b, 0x67e, 0x2e, 0x3b, 0x62c, 0x2e, 0x3b, 0x634, 0x2e, 0x44f, 0x43a, 0x448,
0x430, 0x43d, 0x431, 0x430, 0x3b, 0x434, 0x443, 0x448, 0x430, 0x43d, 0x431, 0x430,
0x3b, 0x441, 0x435, 0x448, 0x430, 0x43d, 0x431, 0x430, 0x3b, 0x447, 0x43e, 0x440,
0x448, 0x430, 0x43d, 0x431, 0x430, 0x3b, 0x43f, 0x430, 0x439, 0x448, 0x430, 0x43d,
0x431, 0x430, 0x3b, 0x436, 0x443, 0x43c, 0x430, 0x3b, 0x448, 0x430, 0x43d, 0x431,
0x430, 0x44f, 0x43a, 0x448, 0x3b, 0x434, 0x443, 0x448, 0x3b, 0x441, 0x435, 0x448,
0x3b, 0x447, 0x43e, 0x440, 0x3b, 0x43f, 0x430, 0x439, 0x3b, 0x436, 0x443, 0x43c,
0x3b, 0x448, 0x430, 0x43d, 0x42f, 0x3b, 0x414, 0x3b, 0x421, 0x3b, 0x427, 0x3b,
0x41f, 0x3b, 0x416, 0x3b, 0x428, 0xa55e, 0xa54c, 0xa535, 0x3b, 0xa5f3, 0xa5e1, 0xa609,
0x3b, 0xa55a, 0xa55e, 0xa55a, 0x3b, 0xa549, 0xa55e, 0xa552, 0x3b, 0xa549, 0xa524, 0xa546,
0xa562, 0x3b, 0xa549, 0xa524, 0xa540, 0xa56e, 0x3b, 0xa53b, 0xa52c, 0xa533, 0x6c, 0x61,
0x68, 0x61, 0x64, 0x69, 0x3b, 0x74, 0x25b, 0x25b, 0x6e, 0x25b, 0x25b, 0x3b,
0x74, 0x61, 0x6c, 0x61, 0x74, 0x61, 0x3b, 0x61, 0x6c, 0x61, 0x62, 0x61,
0x3b, 0x61, 0x69, 0x6d, 0x69, 0x73, 0x61, 0x3b, 0x61, 0x69, 0x6a, 0x69,
0x6d, 0x61, 0x3b, 0x73, 0x69, 0x253, 0x69, 0x74, 0x69, 0x53, 0x77, 0x6f,
0x6e, 0x64, 0x61, 0x68, 0x61, 0x3b, 0x4d, 0x75, 0x73, 0x75, 0x6d, 0x62,
0x75, 0x6c, 0x75, 0x77, 0x6f, 0x3b, 0x1e3c, 0x61, 0x76, 0x68, 0x75, 0x76,
0x68, 0x69, 0x6c, 0x69, 0x3b, 0x1e3c, 0x61, 0x76, 0x68, 0x75, 0x72, 0x61,
0x72, 0x75, 0x3b, 0x1e3c, 0x61, 0x76, 0x68, 0x75, 0x1e4b, 0x61, 0x3b, 0x1e3c,
0x61, 0x76, 0x68, 0x75, 0x1e71, 0x61, 0x6e, 0x75, 0x3b, 0x4d, 0x75, 0x67,
0x69, 0x76, 0x68, 0x65, 0x6c, 0x61, 0x53, 0x77, 0x6f, 0x3b, 0x4d, 0x75,
0x73, 0x3b, 0x56, 0x68, 0x69, 0x3b, 0x52, 0x61, 0x72, 0x3b, 0x1e4a, 0x61,
0x3b, 0x1e70, 0x61, 0x6e, 0x3b, 0x4d, 0x75, 0x67, 0x43, 0x68, 0x1ee7, 0x20,
0x4e, 0x68, 0x1ead, 0x74, 0x3b, 0x54, 0x68, 0x1ee9, 0x20, 0x48, 0x61, 0x69,
0x3b, 0x54, 0x68, 0x1ee9, 0x20, 0x42, 0x61, 0x3b, 0x54, 0x68, 0x1ee9, 0x20,
0x54, 0x1b0, 0x3b, 0x54, 0x68, 0x1ee9, 0x20, 0x4e, 0x103, 0x6d, 0x3b, 0x54,
0x68, 0x1ee9, 0x20, 0x53, 0xe1, 0x75, 0x3b, 0x54, 0x68, 0x1ee9, 0x20, 0x42,
0x1ea3, 0x79, 0x43, 0x4e, 0x3b, 0x54, 0x68, 0x20, 0x32, 0x3b, 0x54, 0x68,
0x20, 0x33, 0x3b, 0x54, 0x68, 0x20, 0x34, 0x3b, 0x54, 0x68, 0x20, 0x35,
0x3b, 0x54, 0x68, 0x20, 0x36, 0x3b, 0x54, 0x68, 0x20, 0x37, 0x43, 0x4e,
0x3b, 0x54, 0x32, 0x3b, 0x54, 0x33, 0x3b, 0x54, 0x34, 0x3b, 0x54, 0x35,
0x3b, 0x54, 0x36, 0x3b, 0x54, 0x37, 0x73, 0x75, 0x64, 0x65, 0x6c, 0x3b,
0x6d, 0x75, 0x64, 0x65, 0x6c, 0x3b, 0x74, 0x75, 0x64, 0x65, 0x6c, 0x3b,
0x76, 0x65, 0x64, 0x65, 0x6c, 0x3b, 0x64, 0xf6, 0x64, 0x65, 0x6c, 0x3b,
0x66, 0x72, 0x69, 0x64, 0x65, 0x6c, 0x3b, 0x7a, 0xe4, 0x64, 0x65, 0x6c,
0x53, 0x75, 0x3b, 0x4d, 0x75, 0x3b, 0x54, 0x75, 0x3b, 0x56, 0x65, 0x3b,
0x44, 0xf6, 0x3b, 0x46, 0x72, 0x3b, 0x5a, 0xe4, 0x73, 0x75, 0x2e, 0x3b,
0x6d, 0x75, 0x2e, 0x3b, 0x74, 0x75, 0x2e, 0x3b, 0x76, 0x65, 0x2e, 0x3b,
0x64, 0xf6, 0x2e, 0x3b, 0x66, 0x72, 0x2e, 0x3b, 0x7a, 0xe4, 0x2e, 0x53,
0x3b, 0x4d, 0x3b, 0x54, 0x3b, 0x56, 0x3b, 0x44, 0x3b, 0x46, 0x3b, 0x5a,
0x53, 0x75, 0x6e, 0x6e, 0x74, 0x61, 0x67, 0x3b, 0x4d, 0xe4, 0x6e, 0x74,
0x61, 0x67, 0x3b, 0x5a, 0x69, 0x161, 0x74, 0x61, 0x67, 0x3b, 0x4d, 0x69,
0x74, 0x74, 0x77, 0x75, 0x10d, 0x3b, 0x46, 0x72, 0xf3, 0x6e, 0x74, 0x61,
0x67, 0x3b, 0x46, 0x72, 0x69, 0x74, 0x61, 0x67, 0x3b, 0x53, 0x61, 0x6d,
0x161, 0x74, 0x61, 0x67, 0x53, 0x75, 0x6e, 0x3b, 0x4d, 0xe4, 0x6e, 0x3b,
0x5a, 0x69, 0x161, 0x3b, 0x4d, 0x69, 0x74, 0x3b, 0x46, 0x72, 0xf3, 0x3b,
0x46, 0x72, 0x69, 0x3b, 0x53, 0x61, 0x6d, 0x53, 0x3b, 0x4d, 0x3b, 0x5a,
0x3b, 0x4d, 0x3b, 0x46, 0x3b, 0x46, 0x3b, 0x53, 0x44, 0x79, 0x64, 0x64,
0x20, 0x53, 0x75, 0x6c, 0x3b, 0x44, 0x79, 0x64, 0x64, 0x20, 0x4c, 0x6c,
0x75, 0x6e, 0x3b, 0x44, 0x79, 0x64, 0x64, 0x20, 0x4d, 0x61, 0x77, 0x72,
0x74, 0x68, 0x3b, 0x44, 0x79, 0x64, 0x64, 0x20, 0x4d, 0x65, 0x72, 0x63,
0x68, 0x65, 0x72, 0x3b, 0x44, 0x79, 0x64, 0x64, 0x20, 0x49, 0x61, 0x75,
0x3b, 0x44, 0x79, 0x64, 0x64, 0x20, 0x47, 0x77, 0x65, 0x6e, 0x65, 0x72,
0x3b, 0x44, 0x79, 0x64, 0x64, 0x20, 0x53, 0x61, 0x64, 0x77, 0x72, 0x6e,
0x53, 0x75, 0x6c, 0x3b, 0x4c, 0x6c, 0x75, 0x6e, 0x3b, 0x4d, 0x61, 0x77,
0x3b, 0x4d, 0x65, 0x72, 0x3b, 0x49, 0x61, 0x75, 0x3b, 0x47, 0x77, 0x65,
0x3b, 0x53, 0x61, 0x64, 0x53, 0x75, 0x6c, 0x3b, 0x4c, 0x6c, 0x75, 0x6e,
0x3b, 0x4d, 0x61, 0x77, 0x3b, 0x4d, 0x65, 0x72, 0x3b, 0x49, 0x61, 0x75,
0x3b, 0x47, 0x77, 0x65, 0x6e, 0x3b, 0x53, 0x61, 0x64, 0x53, 0x3b, 0x4c,
0x6c, 0x3b, 0x4d, 0x3b, 0x4d, 0x3b, 0x49, 0x3b, 0x47, 0x3b, 0x53, 0x73,
0x6e, 0x65, 0x69, 0x6e, 0x3b, 0x6d, 0x6f, 0x61, 0x6e, 0x64, 0x65, 0x69,
0x3b, 0x74, 0x69, 0x69, 0x73, 0x64, 0x65, 0x69, 0x3b, 0x77, 0x6f, 0x61,
0x6e, 0x73, 0x64, 0x65, 0x69, 0x3b, 0x74, 0x6f, 0x6e, 0x67, 0x65, 0x72,
0x73, 0x64, 0x65, 0x69, 0x3b, 0x66, 0x72, 0x65, 0x65, 0x64, 0x3b, 0x73,
0x6e, 0x65, 0x6f, 0x6e, 0x73, 0x69, 0x3b, 0x6d, 0x6f, 0x3b, 0x74, 0x69,
0x3b, 0x77, 0x6f, 0x3b, 0x74, 0x6f, 0x3b, 0x66, 0x72, 0x3b, 0x73, 0x6f,
0x12c8, 0x130b, 0x3b, 0x1233, 0x12ed, 0x1296, 0x3b, 0x121b, 0x1246, 0x1233, 0x129b, 0x3b,
0x12a0, 0x1229, 0x12cb, 0x3b, 0x1203, 0x1219, 0x1233, 0x3b, 0x12a0, 0x122d, 0x1263, 0x3b,
0x1244, 0x122b, 0x12c8, 0x3b, 0x1233, 0x3b, 0x121b, 0x3b, 0x12a0, 0x3b, 0x1203, 0x3b,
0x12a0, 0x3b, 0x1244, 0x44, 0x69, 0x62, 0xe9, 0x65, 0x72, 0x3b, 0x41, 0x6c,
0x74, 0x69, 0x6e, 0x65, 0x3b, 0x54, 0x61, 0x6c, 0x61, 0x61, 0x74, 0x61,
0x3b, 0xc0, 0x6c, 0x61, 0x72, 0x62, 0x61, 0x3b, 0x41, 0x6c, 0x78, 0x61,
0x6d, 0x69, 0x73, 0x3b, 0xc0, 0x6a, 0x6a, 0x75, 0x6d, 0x61, 0x3b, 0x41,
0x73, 0x65, 0x65, 0x72, 0x44, 0x69, 0x62, 0x3b, 0x41, 0x6c, 0x74, 0x3b,
0x54, 0x61, 0x6c, 0x3b, 0xc0, 0x6c, 0x61, 0x3b, 0x41, 0x6c, 0x78, 0x3b,
0xc0, 0x6a, 0x6a, 0x3b, 0x41, 0x73, 0x65, 0x43, 0x61, 0x77, 0x65, 0x3b,
0x4d, 0x76, 0x75, 0x6c, 0x6f, 0x3b, 0x4c, 0x77, 0x65, 0x73, 0x69, 0x62,
0x69, 0x6e, 0x69, 0x3b, 0x4c, 0x77, 0x65, 0x73, 0x69, 0x74, 0x68, 0x61,
0x74, 0x68, 0x75, 0x3b, 0x4c, 0x77, 0x65, 0x73, 0x69, 0x6e, 0x65, 0x3b,
0x4c, 0x77, 0x65, 0x73, 0x69, 0x68, 0x6c, 0x61, 0x6e, 0x75, 0x3b, 0x4d,
0x67, 0x71, 0x69, 0x62, 0x65, 0x6c, 0x6f, 0x43, 0x61, 0x77, 0x3b, 0x4d,
0x76, 0x75, 0x3b, 0x42, 0x69, 0x6e, 0x3b, 0x54, 0x68, 0x61, 0x3b, 0x53,
0x69, 0x6e, 0x3b, 0x48, 0x6c, 0x61, 0x3b, 0x4d, 0x67, 0x71, 0x43, 0x61,
0x77, 0x3b, 0x4d, 0x76, 0x75, 0x3b, 0x4c, 0x77, 0x65, 0x73, 0x62, 0x3b,
0x54, 0x68, 0x61, 0x3b, 0x53, 0x69, 0x6e, 0x3b, 0x48, 0x6c, 0x61, 0x3b,
0x4d, 0x67, 0x71, 0x43, 0x3b, 0x4d, 0x3b, 0x53, 0x62, 0x3b, 0x53, 0x74,
0x3b, 0x53, 0x69, 0x6e, 0x3b, 0x48, 0x6c, 0x3b, 0x4d, 0x67, 0x71, 0x43,
0x3b, 0x4d, 0x3b, 0x53, 0x62, 0x3b, 0x54, 0x68, 0x74, 0x3b, 0x53, 0x69,
0x6e, 0x3b, 0x48, 0x6c, 0x3b, 0x4d, 0x67, 0x71, 0x73, 0x254, 0x301, 0x6e,
0x64, 0x69, 0x25b, 0x3b, 0x6d, 0xf3, 0x6e, 0x64, 0x69, 0x65, 0x3b, 0x6d,
0x75, 0xe1, 0x6e, 0x79, 0xe1, 0x14b, 0x6d, 0xf3, 0x6e, 0x64, 0x69, 0x65,
0x3b, 0x6d, 0x65, 0x74, 0xfa, 0x6b, 0x70, 0xed, 0xe1, 0x70, 0x25b, 0x3b,
0x6b, 0xfa, 0x70, 0xe9, 0x6c, 0x69, 0x6d, 0x65, 0x74, 0xfa, 0x6b, 0x70,
0x69, 0x61, 0x70, 0x25b, 0x3b, 0x66, 0x65, 0x6c, 0xe9, 0x74, 0x65, 0x3b,
0x73, 0xe9, 0x73, 0x65, 0x6c, 0xe9, 0x73, 0x64, 0x3b, 0x6d, 0x64, 0x3b,
0x6d, 0x77, 0x3b, 0x65, 0x74, 0x3b, 0x6b, 0x6c, 0x3b, 0x66, 0x6c, 0x3b,
0x73, 0x73, 0x73, 0x3b, 0x6d, 0x3b, 0x6d, 0x3b, 0x65, 0x3b, 0x6b, 0x3b,
0x66, 0x3b, 0x73, 0x5d6, 0x5d5, 0x5e0, 0x5d8, 0x5d9, 0x5e7, 0x3b, 0x5de, 0x5d0,
0x5b8, 0x5e0, 0x5d8, 0x5d9, 0x5e7, 0x3b, 0x5d3, 0x5d9, 0x5e0, 0x5e1, 0x5d8, 0x5d9,
0x5e7, 0x3b, 0x5de, 0x5d9, 0x5d8, 0x5d5, 0x5d5, 0x5d0, 0x5da, 0x3b, 0x5d3, 0x5d0,
0x5e0, 0x5e2, 0x5e8, 0x5e9, 0x5d8, 0x5d9, 0x5e7, 0x3b, 0x5e4, 0x5bf, 0x5e8, 0x5f2,
0x5b7, 0x5d8, 0x5d9, 0x5e7, 0x3b, 0x5e9, 0x5d1, 0x5ea, 0xc0, 0xec, 0x6b, 0xfa,
0x3b, 0x41, 0x6a, 0xe9, 0x3b, 0xcc, 0x73, 0x1eb9, 0x301, 0x67, 0x75, 0x6e,
0x3b, 0x1ecc, 0x6a, 0x1ecd, 0x301, 0x72, 0xfa, 0x3b, 0x1ecc, 0x6a, 0x1ecd, 0x301,
0x62, 0x1ecd, 0x3b, 0x1eb8, 0x74, 0xec, 0x3b, 0xc0, 0x62, 0xe1, 0x6d, 0x1eb9,
0x301, 0x74, 0x61, 0x1ecc, 0x6a, 0x1ecd, 0x301, 0x20, 0xc0, 0xec, 0x6b, 0xfa,
0x3b, 0x1ecc, 0x6a, 0x1ecd, 0x301, 0x20, 0x41, 0x6a, 0xe9, 0x3b, 0x1ecc, 0x6a,
0x1ecd, 0x301, 0x20, 0xcc, 0x73, 0x1eb9, 0x301, 0x67, 0x75, 0x6e, 0x3b, 0x1ecc,
0x6a, 0x1ecd, 0x301, 0x72, 0xfa, 0x3b, 0x1ecc, 0x6a, 0x1ecd, 0x301, 0x62, 0x1ecd,
0x3b, 0x1ecc, 0x6a, 0x1ecd, 0x301, 0x20, 0x1eb8, 0x74, 0xec, 0x3b, 0x1ecc, 0x6a,
0x1ecd, 0x301, 0x20, 0xc0, 0x62, 0xe1, 0x6d, 0x1eb9, 0x301, 0x74, 0x61, 0xc0,
0xec, 0x6b, 0x3b, 0x41, 0x6a, 0x3b, 0xcc, 0x73, 0x1eb9, 0x301, 0x67, 0x3b,
0x1ecc, 0x6a, 0x1ecd, 0x301, 0x72, 0x3b, 0x1ecc, 0x6a, 0x1ecd, 0x301, 0x62, 0x3b,
0x1eb8, 0x74, 0x3b, 0xc0, 0x62, 0xe1, 0x6d, 0xc0, 0x3b, 0x41, 0x3b, 0xcc,
0x3b, 0x1ecc, 0x3b, 0x1ecc, 0x3b, 0x1eb8, 0x3b, 0xc0, 0xc0, 0xec, 0x6b, 0xfa,
0x3b, 0x41, 0x6a, 0xe9, 0x3b, 0xcc, 0x73, 0x25b, 0x301, 0x67, 0x75, 0x6e,
0x3b, 0x186, 0x6a, 0x254, 0x301, 0x72, 0xfa, 0x3b, 0x186, 0x6a, 0x254, 0x301,
0x62, 0x254, 0x3b, 0x190, 0x74, 0xec, 0x3b, 0xc0, 0x62, 0xe1, 0x6d, 0x25b,
0x301, 0x74, 0x61, 0x186, 0x6a, 0x254, 0x301, 0x20, 0xc0, 0xec, 0x6b, 0xfa,
0x3b, 0x186, 0x6a, 0x254, 0x301, 0x20, 0x41, 0x6a, 0xe9, 0x3b, 0x186, 0x6a,
0x254, 0x301, 0x20, 0xcc, 0x73, 0x25b, 0x301, 0x67, 0x75, 0x6e, 0x3b, 0x186,
0x6a, 0x254, 0x301, 0x72, 0xfa, 0x3b, 0x186, 0x6a, 0x254, 0x301, 0x62, 0x254,
0x3b, 0x186, 0x6a, 0x254, 0x301, 0x20, 0x190, 0x74, 0xec, 0x3b, 0x186, 0x6a,
0x254, 0x301, 0x20, 0xc0, 0x62, 0xe1, 0x6d, 0x25b, 0x301, 0x74, 0x61, 0xc0,
0xec, 0x6b, 0x3b, 0x41, 0x6a, 0x3b, 0xcc, 0x73, 0x25b, 0x301, 0x67, 0x3b,
0x186, 0x6a, 0x254, 0x301, 0x72, 0x3b, 0x186, 0x6a, 0x254, 0x301, 0x62, 0x3b,
0x190, 0x74, 0x3b, 0xc0, 0x62, 0xe1, 0x6d, 0xc0, 0x3b, 0x41, 0x3b, 0xcc,
0x3b, 0x186, 0x3b, 0x186, 0x3b, 0x190, 0x3b, 0xc0, 0x41, 0x6c, 0x68, 0x61,
0x64, 0x69, 0x3b, 0x41, 0x74, 0x69, 0x6e, 0x6e, 0x69, 0x3b, 0x41, 0x74,
0x61, 0x6c, 0x61, 0x61, 0x74, 0x61, 0x3b, 0x41, 0x6c, 0x61, 0x72, 0x62,
0x61, 0x3b, 0x41, 0x6c, 0x68, 0x61, 0x6d, 0x69, 0x73, 0x69, 0x3b, 0x41,
0x6c, 0x7a, 0x75, 0x6d, 0x61, 0x3b, 0x41, 0x73, 0x69, 0x62, 0x74, 0x69,
0x48, 0x3b, 0x54, 0x3b, 0x54, 0x3b, 0x4c, 0x3b, 0x4d, 0x3b, 0x5a, 0x3b,
0x53, 0x6e, 0x67, 0x6f, 0x65, 0x6e, 0x7a, 0x73, 0x69, 0x6e, 0x67, 0x68,
0x67, 0x69, 0x7a, 0x3b, 0x73, 0x69, 0x6e, 0x67, 0x68, 0x67, 0x69, 0x7a,
0x69, 0x74, 0x3b, 0x73, 0x69, 0x6e, 0x67, 0x68, 0x67, 0x69, 0x7a, 0x6e,
0x67, 0x65, 0x69, 0x68, 0x3b, 0x73, 0x69, 0x6e, 0x67, 0x68, 0x67, 0x69,
0x7a, 0x73, 0x61, 0x6d, 0x3b, 0x73, 0x69, 0x6e, 0x67, 0x68, 0x67, 0x69,
0x7a, 0x73, 0x65, 0x69, 0x71, 0x3b, 0x73, 0x69, 0x6e, 0x67, 0x68, 0x67,
0x69, 0x7a, 0x68, 0x61, 0x6a, 0x3b, 0x73, 0x69, 0x6e, 0x67, 0x68, 0x67,
0x69, 0x7a, 0x72, 0x6f, 0x65, 0x6b, 0x49, 0x53, 0x6f, 0x6e, 0x74, 0x6f,
0x3b, 0x55, 0x4d, 0x73, 0x6f, 0x6d, 0x62, 0x75, 0x6c, 0x75, 0x6b, 0x6f,
0x3b, 0x55, 0x4c, 0x77, 0x65, 0x73, 0x69, 0x62, 0x69, 0x6c, 0x69, 0x3b,
0x55, 0x4c, 0x77, 0x65, 0x73, 0x69, 0x74, 0x68, 0x61, 0x74, 0x68, 0x75,
0x3b, 0x55, 0x4c, 0x77, 0x65, 0x73, 0x69, 0x6e, 0x65, 0x3b, 0x55, 0x4c,
0x77, 0x65, 0x73, 0x69, 0x68, 0x6c, 0x61, 0x6e, 0x75, 0x3b, 0x55, 0x4d,
0x67, 0x71, 0x69, 0x62, 0x65, 0x6c, 0x6f, 0x53, 0x6f, 0x6e, 0x3b, 0x4d,
0x73, 0x6f, 0x3b, 0x42, 0x69, 0x6c, 0x3b, 0x54, 0x68, 0x61, 0x3b, 0x53,
0x69, 0x6e, 0x3b, 0x48, 0x6c, 0x61, 0x3b, 0x4d, 0x67, 0x71, 0x53, 0x3b,
0x4d, 0x3b, 0x42, 0x3b, 0x54, 0x3b, 0x53, 0x3b, 0x48, 0x3b, 0x4d, 0x6e,
0x75, 0x6d, 0x129, 0x67, 0x67, 0x75, 0x3b, 0x70, 0x69, 0x72, 0x2d, 0x6b,
0x75, 0x72, 0xe3, 0x2d, 0x68, 0xe1, 0x3b, 0x72, 0xe9, 0x67, 0x72, 0x65,
0x2d, 0x6b, 0x75, 0x72, 0xe3, 0x2d, 0x68, 0xe1, 0x3b, 0x74, 0x1ebd, 0x67,
0x74, 0x169, 0x2d, 0x6b, 0x75, 0x72, 0xe3, 0x2d, 0x68, 0xe1, 0x3b, 0x76,
0x1ebd, 0x6e, 0x68, 0x6b, 0xe3, 0x67, 0x72, 0x61, 0x2d, 0x6b, 0x75, 0x72,
0xe3, 0x2d, 0x68, 0xe1, 0x3b, 0x70, 0xe9, 0x6e, 0x6b, 0x61, 0x72, 0x2d,
0x6b, 0x75, 0x72, 0xe3, 0x2d, 0x68, 0xe1, 0x3b, 0x73, 0x61, 0x76, 0x6e,
0x75, 0x6e, 0x75, 0x6d, 0x2e, 0x3b, 0x70, 0x69, 0x72, 0x2e, 0x3b, 0x72,
0xe9, 0x67, 0x2e, 0x3b, 0x74, 0x1ebd, 0x67, 0x2e, 0x3b, 0x76, 0x1ebd, 0x6e,
0x2e, 0x3b, 0x70, 0xe9, 0x6e, 0x2e, 0x3b, 0x73, 0x61, 0x76, 0x2e, 0x4e,
0x2e, 0x3b, 0x50, 0x2e, 0x3b, 0x52, 0x2e, 0x3b, 0x54, 0x2e, 0x3b, 0x56,
0x2e, 0x3b, 0x50, 0x2e, 0x3b, 0x53, 0x2e, 0x6d, 0x69, 0x74, 0x75, 0xfa,
0x3b, 0x6d, 0x75, 0x72, 0x61, 0x6b, 0x69, 0x70, 0xed, 0x3b, 0x6d, 0x75,
0x72, 0x61, 0x6b, 0xed, 0x2d, 0x6d, 0x75, 0x6b, 0x169, 0x69, 0x3b, 0x6d,
0x75, 0x72, 0x61, 0x6b, 0xed, 0x2d, 0x6d, 0x75, 0x73, 0x61, 0x70, 0xed,
0x72, 0x69, 0x3b, 0x73, 0x75, 0x70, 0x61, 0x70, 0xe1, 0x3b, 0x79, 0x75,
0x6b, 0x75, 0x61, 0x6b, 0xfa, 0x3b, 0x73, 0x61, 0x75, 0x72, 0xfa, 0x6d,
0x69, 0x74, 0x3b, 0x6d, 0x75, 0x72, 0x3b, 0x6d, 0x6d, 0x6b, 0x3b, 0x6d,
0x6d, 0x73, 0x3b, 0x73, 0x75, 0x70, 0x3b, 0x79, 0x75, 0x6b, 0x3b, 0x73,
0x61, 0x75, 0x4d, 0x3b, 0x4d, 0x3b, 0x4d, 0x3b, 0x4d, 0x3b, 0x53, 0x3b,
0x59, 0x3b, 0x53, 0x910, 0x924, 0x935, 0x93e, 0x930, 0x3b, 0x938, 0x94b, 0x92e,
0x935, 0x93e, 0x930, 0x3b, 0x92e, 0x902, 0x917, 0x932, 0x935, 0x93e, 0x930, 0x3b,
0x92c, 0x941, 0x927, 0x935, 0x93e, 0x930, 0x3b, 0x92c, 0x943, 0x939, 0x938, 0x94d,
0x92a, 0x924, 0x935, 0x93e, 0x930, 0x3b, 0x936, 0x941, 0x915, 0x94d, 0x930, 0x935,
0x93e, 0x930, 0x3b, 0x936, 0x928, 0x93f, 0x935, 0x93e, 0x930, 0x53, 0xf6, 0x6e,
0x64, 0x61, 0x69, 0x3b, 0x4d, 0x75, 0x6e, 0x64, 0x61, 0x69, 0x3b, 0x54,
0x65, 0x69, 0x73, 0x64, 0x61, 0x69, 0x3b, 0x57, 0x65, 0x65, 0x64, 0x65,
0x6e, 0x73, 0x64, 0x61, 0x69, 0x3b, 0x54, 0xfc, 0xfc, 0x72, 0x73, 0x64,
0x61, 0x69, 0x3b, 0x46, 0x72, 0x65, 0x69, 0x64, 0x61, 0x69, 0x3b, 0x53,
0x61, 0x6e, 0x69, 0x6e, 0x6a, 0x53, 0xf6, 0x6e, 0x3b, 0x4d, 0x75, 0x6e,
0x3b, 0x54, 0x65, 0x69, 0x3b, 0x57, 0x65, 0x64, 0x3b, 0x54, 0xfc, 0x72,
0x3b, 0x46, 0x72, 0x65, 0x3b, 0x53, 0x61, 0x6e, 0x73, 0x75, 0x6e, 0x6f,
0x20, 0x65, 0x73, 0x75, 0x6e, 0x20, 0x23, 0x37, 0x3b, 0x73, 0x75, 0x6e,
0x6f, 0x20, 0x65, 0x73, 0x75, 0x6e, 0x20, 0x23, 0x31, 0x3b, 0x73, 0x75,
0x6e, 0x6f, 0x20, 0x65, 0x73, 0x75, 0x6e, 0x20, 0x23, 0x32, 0x3b, 0x73,
0x75, 0x6e, 0x6f, 0x20, 0x65, 0x73, 0x75, 0x6e, 0x20, 0x23, 0x33, 0x3b,
0x73, 0x75, 0x6e, 0x6f, 0x20, 0x65, 0x73, 0x75, 0x6e, 0x20, 0x23, 0x34,
0x3b, 0x73, 0x75, 0x6e, 0x6f, 0x20, 0x65, 0x73, 0x75, 0x6e, 0x20, 0x23,
0x35, 0x3b, 0x73, 0x75, 0x6e, 0x6f, 0x20, 0x65, 0x73, 0x75, 0x6e, 0x20,
0x23, 0x36, 0x53, 0x61, 0x6e, 0x64, 0x65, 0x3b, 0x4d, 0x61, 0x6e, 0x64,
0x65, 0x3b, 0x54, 0x69, 0x75, 0x73, 0x64, 0x65, 0x3b, 0x57, 0x65, 0x6e,
0x65, 0x73, 0x64, 0x65, 0x3b, 0x54, 0x6f, 0x73, 0x64, 0x65, 0x3b, 0x46,
0x72, 0x61, 0x65, 0x64, 0x65, 0x3b, 0x53, 0x61, 0x74, 0x61, 0x64, 0x65,
0x6cc, 0x6a9, 0x634, 0x645, 0x628, 0x647, 0x3b, 0x62f, 0x648, 0x634, 0x645, 0x628,
0x647, 0x3b, 0x633, 0x626, 0x6cc, 0x634, 0x645, 0x628, 0x647, 0x3b, 0x686, 0x627,
0x631, 0x634, 0x645, 0x628, 0x647, 0x3b, 0x67e, 0x646, 0x686, 0x634, 0x645, 0x628,
0x647, 0x3b, 0x62c, 0x645, 0x647, 0x3b, 0x634, 0x645, 0x628, 0x647, 0x6cc, 0x6a9,
0x3b, 0x62f, 0x648, 0x3b, 0x633, 0x626, 0x6d2, 0x3b, 0x686, 0x627, 0x631, 0x3b,
0x67e, 0x646, 0x686, 0x3b, 0x62c, 0x645, 0x647, 0x3b, 0x634, 0x645, 0x59, 0x61,
0x6b, 0x73, 0x68, 0x61, 0x6d, 0x62, 0x65, 0x68, 0x3b, 0x44, 0x6f, 0x73,
0x68, 0x61, 0x6d, 0x62, 0x65, 0x68, 0x3b, 0x53, 0x61, 0x79, 0x73, 0x68,
0x61, 0x6d, 0x62, 0x65, 0x68, 0x3b, 0x43, 0x68, 0xe1, 0x72, 0x73, 0x68,
0x61, 0x6d, 0x62, 0x65, 0x68, 0x3b, 0x50, 0x61, 0x6e, 0x63, 0x68, 0x73,
0x68, 0x61, 0x6d, 0x62, 0x65, 0x68, 0x3b, 0x4a, 0x6f, 0x6d, 0x61, 0x68,
0x3b, 0x53, 0x68, 0x61, 0x6d, 0x62, 0x65, 0x68, 0x59, 0x61, 0x6b, 0x3b,
0x44, 0x6f, 0x3b, 0x53, 0x61, 0x79, 0x3b, 0x43, 0x68, 0xe1, 0x3b, 0x50,
0x61, 0x6e, 0x3b, 0x4a, 0x6f, 0x6d, 0x3b, 0x53, 0x68, 0x61, 0x64, 0x6f,
0x6d, 0x65, 0x6e, 0x65, 0x67, 0x61, 0x3b, 0x6c, 0x75, 0x6e, 0x65, 0x73,
0x64, 0xec, 0x3b, 0x6d, 0xe4, 0x74, 0x65, 0x73, 0x64, 0xec, 0x3b, 0x6d,
0xe4, 0x63, 0x6f, 0x72, 0x64, 0xec, 0x3b, 0x7a, 0x65, 0x75, 0x67, 0x67,
0x69, 0x61, 0x3b, 0x76, 0x65, 0x6e, 0x61, 0x72, 0x64, 0xec, 0x3b, 0x73,
0x61, 0x62, 0x62, 0x6f, 0x64, 0x6f, 0x6d, 0x2e, 0x3b, 0x6c, 0x75, 0x6e,
0x2e, 0x3b, 0x6d, 0xe4, 0x74, 0x2e, 0x3b, 0x6d, 0xe4, 0x63, 0x2e, 0x3b,
0x7a, 0x65, 0x75, 0x2e, 0x3b, 0x76, 0x65, 0x6e, 0x2e, 0x3b, 0x73, 0x61,
0x62, 0x2e, 0x44, 0x3b, 0x4c, 0x3b, 0x4d, 0x3b, 0x4d, 0x3b, 0x5a, 0x3b,
0x56, 0x3b, 0x53, 0x627, 0x6cc, 0x6a9, 0x634, 0x6cc, 0x645, 0x6d2, 0x3b, 0x62f,
0x64f, 0x648, 0x634, 0x6cc, 0x645, 0x6d2, 0x3b, 0x6af, 0x6be, 0x646, 0x20, 0x622,
0x646, 0x6af, 0x627, 0x3b, 0x686, 0x627, 0x631, 0x634, 0x6cc, 0x645, 0x6d2, 0x3b,
0x67e, 0x64e, 0x626, 0x20, 0x634, 0x6cc, 0x645, 0x6d2, 0x3b, 0x634, 0x64f, 0x648,
0x6af, 0x627, 0x631, 0x3b, 0x644, 0x64e, 0x648, 0x20, 0x622, 0x646, 0x6af, 0x627,
0x627, 0x3b, 0x62f, 0x3b, 0x6af, 0x3b, 0x686, 0x3b, 0x67e, 0x3b, 0x634, 0x3b,
0x644, 0x61, 0x6c, 0x61, 0x68, 0x61, 0x256, 0x269, 0x3b, 0x61, 0x256, 0x269,
0x74, 0x25b, 0x6e, 0x25b, 0x25b, 0x3b, 0x61, 0x74, 0x61, 0x6c, 0x61, 0x61,
0x74, 0x61, 0x3b, 0x61, 0x6c, 0x61, 0x61, 0x72, 0x62, 0x61, 0x3b, 0x61,
0x6c, 0x61, 0x61, 0x6d, 0x269, 0x73, 0x68, 0x269, 0x3b, 0x61, 0x72, 0x269,
0x73, 0x1dd, 0x6d, 0x61, 0x3b, 0x61, 0x73, 0x69, 0x69, 0x62, 0x69, 0x61,
0x6c, 0x61, 0x68, 0x3b, 0x61, 0x256, 0x269, 0x74, 0x3b, 0x61, 0x74, 0x61,
0x6c, 0x3b, 0x61, 0x6c, 0x61, 0x72, 0x3b, 0x61, 0x6c, 0x61, 0x6d, 0x3b,
0x61, 0x72, 0x269, 0x73, 0x3b, 0x61, 0x73, 0x69, 0x62, 0x6c, 0x68, 0x3b,
0x256, 0x74, 0x3b, 0x74, 0x6c, 0x3b, 0x6c, 0x72, 0x3b, 0x6c, 0x6d, 0x3b,
0x72, 0x73, 0x3b, 0x73, 0x62, 0x924, 0x94b, 0x906, 0x930, 0x3b, 0x938, 0x94b,
0x906, 0x930, 0x3b, 0x92e, 0x902, 0x917, 0x932, 0x935, 0x93e, 0x930, 0x3b, 0x92c,
0x941, 0x927, 0x935, 0x93e, 0x930, 0x3b, 0x935, 0x940, 0x930, 0x935, 0x93e, 0x930,
0x3b, 0x936, 0x941, 0x915, 0x94d, 0x915, 0x930, 0x935, 0x93e, 0x930, 0x3b, 0x936,
0x928, 0x93f, 0x91a, 0x94d, 0x91a, 0x930, 0x935, 0x93e, 0x930, 0x924, 0x94b, 0x906,
0x930, 0x3b, 0x938, 0x94b, 0x906, 0x930, 0x3b, 0x92e, 0x902, 0x917, 0x932, 0x3b,
0x92c, 0x941, 0x927, 0x3b, 0x935, 0x940, 0x930, 0x3b, 0x936, 0x941, 0x915, 0x94d,
0x915, 0x930, 0x3b, 0x936, 0x928, 0x93f, 0x924, 0x3b, 0x938, 0x94b, 0x3b, 0x92e,
0x902, 0x3b, 0x92c, 0x941, 0x3b, 0x935, 0x940, 0x3b, 0x936, 0x941, 0x3b, 0x936,
0x64, 0x6f, 0x6d, 0xe9, 0x6e, 0x65, 0x67, 0x61, 0x3b, 0x6c, 0x75, 0x6e,
0x69, 0x3b, 0x6d, 0x61, 0x72, 0x74, 0x69, 0x3b, 0x6d, 0xe8, 0x72, 0x63,
0x6f, 0x72, 0x65, 0x3b, 0x7a, 0x6f, 0x62, 0x61, 0x3b, 0x76, 0xe8, 0x6e,
0x61, 0x72, 0x65, 0x3b, 0x73, 0x61, 0x62, 0x6f, 0x64, 0x6f, 0x6d, 0x3b,
0x6c, 0x75, 0x6e, 0x3b, 0x6d, 0x61, 0x72, 0x3b, 0x6d, 0x65, 0x72, 0x3b,
0x7a, 0x6f, 0x62, 0x3b, 0x76, 0xe8, 0x6e, 0x3b, 0x73, 0x61, 0x62, 0x61,
0x61, 0x64, 0x69, 0x20, 0x76, 0x61, 0x72, 0x61, 0x3b, 0x73, 0x6d, 0x62,
0x61, 0x72, 0x61, 0x3b, 0x6d, 0x61, 0x6e, 0x67, 0x61, 0x1e0d, 0x61, 0x3b,
0x70, 0x75, 0x64, 0x61, 0x72, 0x61, 0x3b, 0x6c, 0x61, 0x6b, 0x69, 0x20,
0x76, 0x61, 0x72, 0x61, 0x3b, 0x73, 0x75, 0x6b, 0x72, 0x75, 0x20, 0x76,
0x61, 0x72, 0x61, 0x3b, 0x73, 0x61, 0x6e, 0x69, 0x20, 0x76, 0x61, 0x72,
0x61, 0x61, 0x61, 0x64, 0x69, 0x3b, 0x73, 0x6d, 0x62, 0x61, 0x3b, 0x6d,
0x61, 0x6e, 0x67, 0x61, 0x3b, 0x70, 0x75, 0x64, 0x61, 0x3b, 0x6c, 0x61,
0x6b, 0x69, 0x3b, 0x73, 0x75, 0x6b, 0x72, 0x75, 0x3b, 0x73, 0x61, 0x6e,
0x69, 0x61, 0x61, 0x3b, 0x73, 0x3b, 0x6d, 0x61, 0x3b, 0x70, 0x75, 0x3b,
0x6c, 0x61, 0x3b, 0x73, 0x75, 0x3b, 0x73, 0x61, 0x906, 0x926, 0x93f, 0x20,
0x935, 0x93e, 0x930, 0x93e, 0x3b, 0x938, 0x93e, 0x945, 0x92e, 0x94d, 0x935, 0x93e,
0x930, 0x93e, 0x3b, 0x92e, 0x902, 0x917, 0x93e, 0x921, 0x93c, 0x93e, 0x3b, 0x92a,
0x941, 0x926, 0x94d, 0x926, 0x93e, 0x930, 0x93e, 0x3b, 0x932, 0x93e, 0x915, 0x94d,
0x915, 0x93f, 0x20, 0x935, 0x93e, 0x930, 0x93e, 0x3b, 0x938, 0x941, 0x915, 0x941,
0x930, 0x941, 0x20, 0x935, 0x93e, 0x930, 0x93e, 0x3b, 0x938, 0x93e, 0x928, 0x94d,
0x928, 0x93f, 0x20, 0x935, 0x93e, 0x930, 0x93e, 0xb06, 0xb26, 0xb3f, 0x20, 0xb71,
0xb3e, 0xb30, 0xb3e, 0x3b, 0xb38, 0xb2e, 0xb4d, 0xb2c, 0xb3e, 0xb30, 0xb3e, 0x3b,
0xb2e, 0xb3e, 0xb19, 0xb4d, 0xb17, 0xb3e, 0xb21, 0xb3c, 0xb3e, 0x3b, 0xb2a, 0xb41,
0xb26, 0xb3e, 0xb30, 0xb3e, 0x3b, 0xb32, 0xb3e, 0xb15, 0xb3f, 0x20, 0xb71, 0xb3e,
0xb30, 0xb3e, 0x3b, 0xb38, 0xb41, 0xb15, 0xb4d, 0xb30, 0xb41, 0x20, 0xb71, 0xb3e,
0xb30, 0xb3e, 0x3b, 0xb38, 0xb3e, 0xb28, 0xb3f, 0x20, 0xb71, 0xb3e, 0xb30, 0xb3e,
0xc35, 0xc3e, 0xc30, 0xc2e, 0xc3f, 0x3b, 0xc28, 0xc2e, 0xc3e, 0xc30, 0xc3e, 0x3b,
0xc2e, 0xc3e, 0xc02, 0xc17, 0xc3e, 0xc21, 0xc3e, 0x3b, 0xc35, 0xc41, 0xc26, 0xc3e,
0xc30, 0xc3e, 0x3b, 0xc32, 0xc3e, 0xc15, 0xc3f, 0xc35, 0xc30, 0xc3e, 0x3b, 0xc28,
0xc41, 0xc15, 0xc4d, 0x20, 0xc35, 0xc30, 0xc3e, 0x3b, 0xc38, 0xc3e, 0xc28, 0xc3f,
0xc35, 0xc30, 0xc3e
};

static constexpr char16_t byte_unit_data[] = {
0x62, 0x79, 0x74, 0x65, 0x73, 0x6b, 0x42, 0x3b, 0x4d, 0x42, 0x3b, 0x47,
0x42, 0x3b, 0x54, 0x42, 0x3b, 0x50, 0x42, 0x3b, 0x45, 0x42, 0x4b, 0x69,
0x42, 0x3b, 0x4d, 0x69, 0x42, 0x3b, 0x47, 0x69, 0x42, 0x3b, 0x54, 0x69,
0x42, 0x3b, 0x50, 0x69, 0x42, 0x3b, 0x45, 0x69, 0x42, 0x67, 0x72, 0x65,
0x65, 0x70, 0x62, 0x61, 0x6a, 0x74, 0x1263, 0x12ed, 0x1275, 0x12aa, 0x1263, 0x3b,
0x121c, 0x130b, 0x1263, 0x12ed, 0x1275, 0x3b, 0x130a, 0x1263, 0x3b, 0x1274, 0x122b, 0x1263,
0x12ed, 0x1275, 0x3b, 0x1354, 0x1263, 0x3b, 0x45, 0x42, 0x628, 0x627, 0x64a, 0x62a,
0x643, 0x64a, 0x644, 0x648, 0x628, 0x627, 0x64a, 0x62a, 0x3b, 0x645, 0x2e, 0x628,
0x3b, 0x63a, 0x2e, 0x628, 0x3b, 0x62a, 0x64a, 0x631, 0x627, 0x628, 0x627, 0x64a,
0x62a, 0x3b, 0x628, 0x64a, 0x62a, 0x627, 0x628, 0x627, 0x64a, 0x62a, 0x3b, 0x45,
0x42, 0x562, 0x561, 0x575, 0x569, 0x565, 0x580, 0x56f, 0x532, 0x3b, 0x544, 0x532,
0x3b, 0x533, 0x532, 0x3b, 0x54f, 0x532, 0x3b, 0x54a, 0x532, 0x3b, 0x45, 0x42,
0x9ac, 0x9be, 0x987, 0x99f, 0x995, 0x9bf, 0x983, 0x20, 0x9ac, 0x9be, 0x983, 0x3b,
0x9ae, 0x9c7, 0x983, 0x20, 0x9ac, 0x9be, 0x983, 0x3b, 0x997, 0x9bf, 0x983, 0x20,
0x9ac, 0x9be, 0x983, 0x3b, 0x99f, 0x9c7, 0x983, 0x20, 0x9ac, 0x9be, 0x983, 0x3b,
0x50, 0x42, 0x3b, 0x45, 0x42, 0x62, 0x61, 0x79, 0x74, 0x62, 0x79, 0x74,
0x65, 0x2d, 0x61, 0x6b, 0x431, 0x430, 0x439, 0x442, 0x44b, 0x41a, 0x411, 0x3b,
0x41c, 0x411, 0x3b, 0x413, 0x411, 0x3b, 0x422, 0x411, 0x3b, 0x41f, 0x411, 0x3b,
0x45, 0x42, 0x62, 0x61, 0x6a, 0x74, 0x6f, 0x76, 0x69, 0x6f, 0x6b, 0x74,
0x65, 0x64, 0x6f, 0xf9, 0x6b, 0x6f, 0x3b, 0x4d, 0x6f, 0x3b, 0x47, 0x6f,
0x3b, 0x54, 0x6f, 0x3b, 0x50, 0x6f, 0x3b, 0x45, 0x6f, 0x4b, 0x69, 0x6f,
0x3b, 0x4d, 0x69, 0x6f, 0x3b, 0x47, 0x69, 0x6f, 0x3b, 0x54, 0x69, 0x6f,
0x3b, 0x50, 0x69, 0x6f, 0x3b, 0x45, 0x69, 0x6f, 0x431, 0x430, 0x439, 0x442,
0x43e, 0x432, 0x435, 0x1018, 0x102d, 0x102f, 0x1000, 0x103a, 0x6d, 0x67, 0x61, 0x20,
0x62, 0x79, 0x74, 0x65, 0xd804, 0xdd1d, 0xd804, 0xdd2d, 0xd804, 0xdd16, 0xd804, 0xdd34,
0x13d7, 0x13d3, 0x13cd, 0x13a6, 0x13b5, 0x13a9, 0x5b57, 0x8282, 0x4f4d, 0x5143, 0x7d44, 0x62,
0x61, 0x6a, 0x74, 0x79, 0x62, 0x61, 0x6a, 0x74, 0x6f, 0x6a, 0x62, 0x61,
0x69, 0x64, 0x69, 0x64, 0x62, 0xfd, 0x74, 0x74, 0x61, 0x76, 0x75, 0x74,
0x6b, 0x74, 0x3b, 0x4d, 0x74, 0x3b, 0x47, 0x74, 0x3b, 0x54, 0x74, 0x3b,
0x50, 0x74, 0x3b, 0x45, 0x74, 0x4b, 0x69, 0x74, 0x3b, 0x4d, 0x69, 0x74,
0x3b, 0x47, 0x69, 0x74, 0x3b, 0x54, 0x69, 0x74, 0x3b, 0x50, 0x69, 0x74,
0x3b, 0x45, 0x69, 0x74, 0x6f, 0x63, 0x74, 0x65, 0x74, 0x73, 0xd83a, 0xdd36,
0xd83a, 0xdd2b, 0xd83a, 0xdd45, 0xd83a, 0xdd3c, 0xd83a, 0xdd46, 0xd83a, 0xdd2d, 0xd83a, 0xdd33,
0xd83a, 0xdd14, 0x3b, 0xd83a, 0xdd03, 0xd83a, 0xdd14, 0x3b, 0xd83a, 0xdd18, 0xd83a, 0xdd14,
0x3b, 0xd83a, 0xdd1a, 0xd83a, 0xdd14, 0x3b, 0xd83a, 0xdd06, 0xd83a, 0xdd14, 0x3b, 0x45,
0x42, 0x62, 0x61, 0x69, 0x64, 0x68, 0x74, 0x10d1, 0x10d0, 0x10d8, 0x10e2, 0x10d8,
0x10d9, 0x10d1, 0x10d0, 0x10d8, 0x10e2, 0x10d8, 0x3b, 0x4d, 0x42, 0x3b, 0x47, 0x42,
0x3b, 0x10e2, 0x10d1, 0x10d0, 0x10d8, 0x10e2, 0x10d8, 0x3b, 0x10de, 0x10d1, 0x10d0, 0x10d8,
0x10e2, 0x10d8, 0x3b, 0x45, 0x42, 0x42, 0x79, 0x74, 0x65, 0x73, 0xaac, 0xabe,
0xa87, 0xa9f, 0x6b, 0x42, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x4d, 0x42, 0x20,
0x7b, 0x30, 0x7d, 0x3b, 0x47, 0x42, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x54,
0x42, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x50, 0x42, 0x20, 0x7b, 0x30, 0x7d,
0x3b, 0x45, 0x42, 0x20, 0x7b, 0x30, 0x7d, 0x4b, 0x69, 0x42, 0x20, 0x7b,
0x30, 0x7d, 0x3b, 0x4d, 0x69, 0x42, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x47,
0x69, 0x42, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x54, 0x69, 0x42, 0x20, 0x7b,
0x30, 0x7d, 0x3b, 0x50, 0x69, 0x42, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x45,
0x69, 0x42, 0x20, 0x7b, 0x30, 0x7d, 0x5d1, 0x5d9, 0x5d9, 0x5d8, 0x92c, 0x93e,
0x907, 0x91f, 0x62, 0xe1, 0x6a, 0x74, 0x62, 0xe6, 0x74, 0x69, 0x6f, 0x63,
0x74, 0x65, 0x74, 0x65, 0x73, 0x62, 0x65, 0x61, 0x72, 0x74, 0x61, 0x30d0,
0x30a4, 0x30c8, 0x4b, 0x42, 0x3b, 0x4d, 0x42, 0x3b, 0x47, 0x42, 0x3b, 0x54,
0x42, 0x3b, 0x50, 0x42, 0x3b, 0x45, 0x42, 0x62, 0x69, 0x74, 0x65, 0x61,
0x1e6d, 0x61, 0x6d, 0x1e0d, 0x61, 0x6e, 0x6b, 0x41, 0x1e6c, 0x3b, 0x4d, 0x41,
0x1e6c, 0x3b, 0x47, 0x41, 0x1e6c, 0x3b, 0x54, 0x41, 0x1e6c, 0x3b, 0x50, 0x42,
0x3b, 0x45, 0x42, 0xcac, 0xcc8, 0xc9f, 0xccd, 0x200c, 0xc97, 0xcb3, 0xcc1, 0xc95,
0xcbf, 0x2e, 0xcac, 0xcc8, 0x2e, 0x3b, 0xcae, 0xcc6, 0x2e, 0xcac, 0xcc8, 0x2e,
0x3b, 0xc97, 0xcbf, 0x2e, 0xcac, 0xcc8, 0x2e, 0x3b, 0xc9f, 0xcc6, 0x2e, 0xcac,
0xcc8, 0x2e, 0x3b, 0xcaa, 0xcc6, 0xcac, 0xcc8, 0x3b, 0x45, 0x42, 0x43a, 0x411,
0x3b, 0x4d, 0x411, 0x3b, 0x413, 0x411, 0x3b, 0x54, 0x411, 0x3b, 0x41f, 0x411,
0x3b, 0x45, 0x411, 0x4b, 0x69, 0x411, 0x3b, 0x4d, 0x69, 0x411, 0x3b, 0x47,
0x69, 0x411, 0x3b, 0x54, 0x69, 0x411, 0x3b, 0x50, 0x69, 0x411, 0x3b, 0x45,
0x69, 0x411, 0x1794, 0x17c3, 0x92c, 0x93e, 0x92f, 0x91f, 0xbc14, 0xc774, 0xd2b8, 0x43a,
0x411, 0x3b, 0x41c, 0x411, 0x3b, 0x413, 0x411, 0x3b, 0x422, 0x411, 0x3b, 0x41f,
0x442, 0x431, 0x3b, 0x45, 0x42, 0x62, 0x61, 0x69, 0x74, 0x69, 0x62, 0x61,
0x69, 0x74, 0x61, 0x69, 0x431, 0x430, 0x458, 0x442, 0x438, 0xd2c, 0xd48, 0xd31,
0xd4d, 0xd31, 0xd4d, 0xd15, 0xd46, 0xd2c, 0xd3f, 0x3b, 0xd0e, 0xd02, 0xd2c, 0xd3f,
0x3b, 0xd1c, 0xd3f, 0xd2c, 0xd3f, 0x3b, 0xd1f, 0xd3f, 0xd2c, 0xd3f, 0x3b, 0xd2a,
0xd3f, 0xd2c, 0xd3f, 0x3b, 0x45, 0x42, 0x628, 0x627, 0x6cc, 0x62a, 0x6a9, 0x6cc,
0x644, 0x648, 0x628, 0x627, 0x6cc, 0x62a, 0x3b, 0x645, 0x6af, 0x627, 0x628, 0x627,
0x6cc, 0x62a, 0x3b, 0x6af, 0x6cc, 0x6af, 0x627, 0x628, 0x627, 0x6cc, 0x62a, 0x3b,
0x62a, 0x631, 0x627, 0x628, 0x627, 0x6cc, 0x62a, 0x3b, 0x50, 0x42, 0x3b, 0x45,
0x42, 0x43a, 0x411, 0x3b, 0x41c, 0x411, 0x3b, 0x413, 0x411, 0x3b, 0x422, 0x411,
0x3b, 0x41f, 0x411, 0x3b, 0x45, 0x42, 0x6b, 0x42, 0x3b, 0x4d, 0x42, 0x3b,
0x47, 0x42, 0x3b, 0x54, 0x42, 0x3b, 0x92a, 0x93f, 0x91f, 0x93e, 0x3b, 0x45,
0x42, 0x42, 0x61, 0x69, 0x74, 0x2d, 0x64, 0x1eb9, 0x6d, 0xb2c, 0xb3e, 0xb07,
0xb1f, 0xb4d, 0x628, 0x627, 0x64a, 0x67c, 0x633, 0xa2c, 0xa3e, 0xa07, 0xa1f, 0x62,
0x79, 0x21b, 0x69, 0x431, 0x430, 0x430, 0x439, 0x442, 0x43a, 0x411, 0x3b, 0x41c,
0x411, 0x3b, 0x47, 0x42, 0x3b, 0x54, 0x42, 0x3b, 0x50, 0x42, 0x3b, 0x45,
0x42, 0x431, 0x430, 0x458, 0x442, 0x43e, 0x432, 0x438, 0x628, 0x627, 0x626, 0x64a,
0x67d, 0x632, 0x6aa, 0x644, 0x648, 0x20, 0x628, 0x627, 0x626, 0x64a, 0x67d, 0x632,
0x3b, 0x4d, 0x42, 0x3b, 0x47, 0x42, 0x3b, 0x54, 0x42, 0x3b, 0x50, 0x42,
0x3b, 0x45, 0x42, 0xdb6, 0xdba, 0xdd2, 0xda7, 0xdca, 0xd9a, 0xdd2, 0xdb6, 0x20,
0x7b, 0x30, 0x7d, 0x3b, 0xdb8, 0xdd9, 0xdb6, 0x20, 0x7b, 0x30, 0x7d, 0x3b,
0xd9c, 0xdd2, 0xdb6, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0xda7, 0xdd9, 0xdb6, 0x20,
0x7b, 0x30, 0x7d, 0x3b, 0xdb4, 0xdd9, 0xdb6, 0x20, 0x7b, 0x30, 0x7d, 0x3b,
0x45, 0x42, 0x62, 0x65, 0x79, 0x74, 0x69, 0x73, 0x6b, 0x42, 0x3b, 0x4d,
0x42, 0x3b, 0x47, 0x42, 0x3b, 0x54, 0x42, 0x3b, 0x42, 0x42, 0x3b, 0x45,
0x42, 0x6b, 0x69, 0x6c, 0x6f, 0x62, 0x61, 0x69, 0x74, 0x69, 0x20, 0x7b,
0x30, 0x7d, 0x3b, 0x4d, 0x42, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x47, 0x42,
0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x74, 0x65, 0x72, 0x61, 0x62, 0x61, 0x69,
0x74, 0x69, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x50, 0x42, 0x20, 0x7b, 0x30,
0x7d, 0x3b, 0x45, 0x42, 0x712, 0x710, 0x71d, 0x72c, 0xbaa, 0xbc8, 0xb9f, 0xbcd,
0xb95, 0xbb3, 0xbcd, 0xc2c, 0xc48, 0xc1f, 0xc4d, 0x200c, 0xc32, 0xc41, 0xc15, 0xc47,
0xc2c, 0xc40, 0x3b, 0xc0e, 0xc2e, 0xc4d, 0x200c, 0xc2c, 0xc3f, 0x3b, 0xc1c, 0xc40,
0xc2c, 0xc40, 0x3b, 0xc1f, 0xc40, 0xc2c, 0xc40, 0x3b, 0xc2a, 0xc40, 0xc2c, 0xc40,
0x3b, 0x45, 0x42, 0xe44, 0xe1a, 0xe15, 0xe4c, 0x70, 0x61, 0x69, 0x74, 0x69,
0x6b, 0x42, 0x20, 0x2bb, 0x65, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x4d, 0x42,
0x20, 0x2bb, 0x65, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x47, 0x42, 0x20, 0x2bb,
0x65, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x54, 0x42, 0x20, 0x2bb, 0x65, 0x20,
0x7b, 0x30, 0x7d, 0x3b, 0x50, 0x42, 0x20, 0x2bb, 0x65, 0x20, 0x7b, 0x30,
0x7d, 0x3b, 0x45, 0x42, 0x20, 0x2bb, 0x65, 0x20, 0x7b, 0x30, 0x7d, 0x4b,
0x69, 0x42, 0x20, 0x2bb, 0x65, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x4d, 0x69,
0x42, 0x20, 0x2bb, 0x65, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x47, 0x69, 0x42,
0x20, 0x2bb, 0x65, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x54, 0x69, 0x42, 0x20,
0x2bb, 0x65, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x50, 0x69, 0x42, 0x20, 0x2bb,
0x65, 0x20, 0x7b, 0x30, 0x7d, 0x3b, 0x45, 0x69, 0x42, 0x20, 0x2bb, 0x65,
0x20, 0x7b, 0x30, 0x7d, 0x62, 0x61, 0xfd, 0x74, 0x431, 0x430, 0x439, 0x442,
0x438, 0x62, 0x79, 0x74, 0x65, 0x79, 0x628, 0x627, 0x626, 0x679, 0x6b, 0x42,
0x3b, 0x4d, 0x42, 0x3b, 0x47, 0x42, 0x3b, 0x54, 0x42, 0x3b, 0x67e, 0x6cc,
0x20, 0x628, 0x6cc, 0x3b, 0x45, 0x42, 0x62, 0x65, 0x69, 0x74, 0x69, 0x61,
0x75, 0xe0, 0x77, 0x1ecd, 0x301, 0x6e, 0x20, 0x62, 0xe1, 0xec, 0x74, 0xec,
0x6b, 0xe9, 0x62, 0xe1, 0xec, 0x74, 0xec, 0x3b, 0x4d, 0x42, 0x3b, 0x6a,
0xed, 0x62, 0xe1, 0xec, 0x74, 0xec, 0x3b, 0x54, 0xed, 0x62, 0xe1, 0xec,
0x74, 0xec, 0x3b, 0x50, 0xed, 0x62, 0xe1, 0xec, 0x74, 0xec, 0x3b, 0x45,
0x42, 0xe0, 0x77, 0x254, 0x301, 0x6e, 0x20, 0x62, 0xe1, 0xec, 0x74, 0xec,
0x62, 0x79, 0x74, 0x65, 0x20, 0x61, 0x67, 0x62, 0x79, 0x74, 0x65, 0x2d,
0x69, 0x74, 0x61
};

static constexpr char16_t am_data[] = {
0x41, 0x4d, 0x76, 0x6d, 0x2e, 0x61, 0x2e, 0x67, 0x41, 0x4e, 0x65, 0x20,
0x70, 0x61, 0x72, 0x61, 0x64, 0x69, 0x74, 0x65, 0x73, 0x1325, 0x12cb, 0x1275,
0x635, 0x9aa, 0x9c2, 0x9f0, 0x9cd, 0x9ac, 0x9be, 0x9b9, 0x9cd, 0x9a8, 0x64, 0x65,
0x20, 0x6c, 0x61, 0x20, 0x6d, 0x61, 0xf1, 0x61, 0x6e, 0x61, 0x69, 0x63,
0x68, 0x65, 0x68, 0x65, 0x61, 0x76, 0x6f, 0x410, 0x41c, 0x73, 0xe1, 0x72,
0xfa, 0x77, 0xe1, 0x49, 0x20, 0x62, 0x69, 0x6b, 0x25b, 0x302, 0x67, 0x6c,
0xe0, 0x75, 0x6c, 0x75, 0x63, 0x68, 0x65, 0x6c, 0x6f, 0x70, 0x61, 0x6d,
0x69, 0x6c, 0x61, 0x75, 0x92d, 0x94b, 0x930, 0x92b, 0x941, 0x902, 0x70, 0x72,
0x69, 0x6a, 0x65, 0x70, 0x6f, 0x64, 0x6e, 0x65, 0x43f, 0x440, 0x438, 0x458,
0x435, 0x20, 0x43f, 0x43e, 0x434, 0x43d, 0x435, 0x41, 0x2e, 0x4d, 0x2e, 0x43f,
0x440, 0x2e, 0x43e, 0x431, 0x2e, 0x1014, 0x1036, 0x1014, 0x1000, 0x103a, 0x4e0a, 0x5348,
0x61, 0x2e, 0xa0, 0x6d, 0x2e, 0x5a, 0x64, 0x61, 0x74, 0x20, 0x61, 0x7a,
0x61, 0x6c, 0x628, 0x2e, 0x646, 0x13cc, 0x13be, 0x13b4, 0x55, 0x68, 0x72, 0x20,
0x76, 0xf6, 0x72, 0x6d, 0x69, 0x64, 0x64, 0x61, 0x61, 0x63, 0x68, 0x73,
0x61, 0x2e, 0x6d, 0x2e, 0x64, 0x6f, 0x70, 0x2e, 0x938, 0x935, 0x947, 0x930,
0x69, 0x64, 0x69, 0x253, 0x61, 0xf66, 0xf94, 0xf0b, 0xf46, 0xf0b, 0x4b, 0x49,
0xd801, 0xdc08, 0xd801, 0xdc23, 0xd801, 0xdc68, 0xd801, 0xdc65, 0x61, 0x74, 0x6d, 0x14b,
0x64, 0x69, 0x6b, 0xed, 0x6b, 0xed, 0x72, 0xed, 0x67, 0x61, 0x70, 0x2e,
0x6d, 0x61, 0x74, 0x69, 0x6e, 0x73, 0x75, 0x62, 0x61, 0x6b, 0x61, 0xd83a,
0xdd00, 0xd83a, 0xdd0e, 0x3c0, 0x2e, 0x3bc, 0x2e, 0x4d, 0x61, 0x6d, 0x62, 0x69,
0x61, 0x53, 0x61, 0x66, 0x69, 0x79, 0x61, 0x5dc, 0x5e4, 0x5e0, 0x5d4, 0x5f4,
0x5e6, 0x64, 0x65, 0x2e, 0x66, 0x2e, 0x68, 0x2e, 0x4e, 0x2019, 0x1ee5, 0x74,
0x1ee5, 0x74, 0x1ee5, 0x69, 0x70, 0x2e, 0x61, 0x6e, 0x74, 0x65, 0x20, 0x6d,
0x69, 0x64, 0xed, 0x72, 0x2e, 0x6e, 0x2e, 0x5348, 0x524d, 0x49, 0x73, 0x75,
0x6b, 0x6e, 0x20, 0x74, 0x75, 0x66, 0x61, 0x74, 0x6b, 0x61, 0x72, 0x6f,
0x6f, 0x6e, 0x128, 0x79, 0x61, 0x6b, 0x77, 0x61, 0x6b, 0x79, 0x61, 0xcaa,
0xcc2, 0xcb0, 0xccd, 0xcb5, 0xcbe, 0xcb9, 0xccd, 0xca8, 0x627, 0x6d2, 0x20, 0x627,
0x6cc, 0x645, 0x92f, 0x947, 0x20, 0x90f, 0x92e, 0x4b, 0x69, 0x72, 0x6f, 0x6b,
0x6f, 0x938, 0x915, 0x93e, 0x933, 0xc624, 0xc804, 0x41, 0x64, 0x64, 0x75, 0x68,
0x61, 0x42, 0x4e, 0x6d, 0x61, 0x6e, 0xe1, 0x442, 0x430, 0x4a3, 0x43a, 0x44b,
0x54, 0x4f, 0x4f, 0xe81, 0xec8, 0xead, 0xe99, 0xe97, 0xec8, 0xebd, 0xe87, 0x70,
0x72, 0x69, 0x65, 0x6b, 0x161, 0x70, 0x75, 0x73, 0x64, 0x69, 0x65, 0x6e,
0x101, 0x6e, 0x74, 0x254, 0x301, 0x6e, 0x67, 0x254, 0x301, 0x70, 0x72, 0x69,
0x65, 0x161, 0x70, 0x69, 0x65, 0x74, 0x64, 0x6f, 0x70, 0x6f, 0x142, 0x64,
0x6e, 0x6a, 0x61, 0x44, 0x69, 0x6e, 0x64, 0x61, 0x4f, 0x44, 0x6d, 0x6f,
0x69, 0x65, 0x73, 0x43f, 0x440, 0x435, 0x442, 0x43f, 0x43b, 0x2e, 0x75, 0x74,
0x75, 0x6b, 0x6f, 0x77, 0x69, 0x63, 0x68, 0x69, 0x73, 0x68, 0x75, 0x4d,
0x75, 0x68, 0x69, 0x50, 0x47, 0x98f, 0x20, 0x98f, 0x9ae, 0x190, 0x6e, 0x6b,
0x61, 0x6b, 0x25b, 0x6e, 0x79, 0xe1, 0x52, 0x168, 0x4af, 0x2e, 0x4e9, 0x2e,
0x1826, 0x1802, 0x20, 0x1825, 0x63, 0x6f, 0x6d, 0x6d, 0x65, 0x1c1, 0x67, 0x6f,
0x61, 0x67, 0x61, 0x73, 0x92a, 0x942, 0x930, 0x94d, 0x935, 0x93e, 0x939, 0x94d,
0x928, 0x6d, 0x62, 0x61, 0x2bc, 0xe1, 0x6d, 0x62, 0x61, 0x2bc, 0x6d, 0x62,
0x61, 0xa78c, 0x6d, 0x62, 0x61, 0xa78c, 0x46, 0x1ecd, 0x20, 0x6d, 0x1ecd, 0x301,
0x6e, 0x69, 0x6e, 0x7db, 0x69, 0x111, 0x69, 0x74, 0x62, 0x65, 0x61, 0x69,
0x76, 0x65, 0x74, 0x69, 0x62, 0x66, 0x2e, 0x6d, 0x2e, 0x52, 0x57, 0x57,
0x44, 0x4d5, 0x43c, 0x431, 0x438, 0x441, 0x431, 0x43e, 0x43d, 0x44b, 0x20, 0x440,
0x430, 0x437, 0x43c, 0x4d5, 0x63a, 0x2e, 0x645, 0x2e, 0x642, 0x628, 0x644, 0x200c,
0x627, 0x632, 0x638, 0x647, 0x631, 0x64, 0x61, 0x20, 0x6d, 0x61, 0x6e, 0x68,
0xe3, 0x61, 0x6e, 0x6b, 0x73, 0x74, 0x101, 0x69, 0x6e, 0x61, 0x6e, 0xa2a,
0xa42, 0x2e, 0xa26, 0xa41, 0x2e, 0x6b, 0x61, 0x6e, 0x67, 0x2019, 0x61, 0x6d,
0x61, 0x5a, 0x2e, 0x4d, 0x55, 0x2e, 0x42d, 0x418, 0x54, 0x65, 0x73, 0x69,
0x72, 0x61, 0x6e, 0x4e, 0x44, 0x4c, 0x77, 0x61, 0x6d, 0x69, 0x6c, 0x61,
0x77, 0x75, 0x1c65, 0x1c6e, 0x1c5b, 0x1c5f, 0x1c5c, 0x70, 0x72, 0x69, 0x6a, 0x65,
0x20, 0x70, 0x6f, 0x64, 0x6e, 0x65, 0x6d, 0x61, 0x6b, 0x65, 0x6f, 0xa3b8,
0xa111, 0x64, 0x6f, 0x20, 0x70, 0x6f, 0x142, 0x65, 0x64, 0x6e, 0x69, 0x14f,
0x635, 0x628, 0x62d, 0x60c, 0x20, 0x645, 0x646, 0x62c, 0x647, 0x646, 0x62f, 0x938,
0x941, 0x92c, 0x941, 0x939, 0x20, 0x91c, 0x93e, 0xdb4, 0xdd9, 0x2e, 0xdc0, 0x2e,
0x4d, 0x75, 0x6e, 0x6b, 0x79, 0x6f, 0x47, 0x48, 0x2d5c, 0x2d49, 0x2d3c, 0x2d30,
0x2d61, 0x2d5c, 0x66, 0x6d, 0x61, 0x6d, 0x20, 0x56, 0x6f, 0x72, 0x6d, 0x69,
0x74, 0x74, 0x61, 0x67, 0x70f, 0x729, 0x71b, 0x200c, 0x74, 0x69, 0x66, 0x61,
0x77, 0x74, 0x4c, 0x75, 0x6d, 0x61, 0x20, 0x6c, 0x77, 0x61, 0x20, 0x4b,
0x53, 0x75, 0x62, 0x62, 0x61, 0x61, 0x68, 0x69, 0x54, 0x61, 0x70, 0x61,
0x72, 0x61, 0x63, 0x68, 0x75, 0xe01, 0xe48, 0xe2d, 0xe19, 0xe40, 0xe17, 0xe35,
0xe48, 0xe22, 0xe07, 0xf66, 0xf94, 0xf0b, 0xf51, 0xfb2, 0xf7c, 0xf0b, 0x1245, 0x2e,
0x1240, 0x2e, 0x68, 0x65, 0x6e, 0x67, 0x69, 0x68, 0x65, 0x6e, 0x67, 0x69,
0xd6, 0xd6, 0x67, 0xfc, 0x6e, 0x6f, 0x72, 0x74, 0x61, 0x64, 0x61, 0x6e,
0x20, 0xf6, 0x148, 0x434, 0x43f, 0x686, 0x6c8, 0x634, 0x62a, 0x649, 0x646, 0x20,
0x628, 0x6c7, 0x631, 0x6c7, 0x646, 0x422, 0x41e, 0x53, 0x41, 0x79, 0x62, 0x6b,
0x69, 0x25b, 0x6d, 0x25b, 0x301, 0x25b, 0x6d, 0x5e4, 0x5bf, 0x5d0, 0x5b7, 0x5e8,
0x5de, 0x5d9, 0x5d8, 0x5d0, 0x5b8, 0x5d2, 0xc0, 0xe1, 0x72, 0x1ecd, 0x300, 0xc0,
0xe1, 0x72, 0x254, 0x300, 0x62, 0x61, 0x6e, 0x68, 0x61, 0x65, 0x74, 0x924,
0x921, 0x915, 0x947, 0x938, 0x92c, 0x947, 0x930, 0x947, 0x70, 0x69, 0x20, 0x6f,
0x70, 0x65, 0x6e, 0x20, 0x73, 0x75, 0x6e, 0x6f, 0x28a, 0x73, 0x68, 0x69,
0x6c, 0xe8, 0x20, 0x6b, 0x28a, 0x62, 0x6f, 0x256, 0x75, 0x92d, 0x94d, 0x92f,
0x93e, 0x917, 0x93e, 0x90f, 0x20, 0x90f, 0x92e, 0xb0f, 0x20, 0xb0f, 0xb2e, 0xc0e,
0x20, 0xc0e, 0xc2e, 0xc4d
};

static constexpr char16_t pm_data[] = {
0x50, 0x4d, 0x6e, 0x6d, 0x2e, 0x61, 0x2e, 0x6b, 0x45, 0x57, 0x65, 0x20,
0x70, 0x61, 0x73, 0x64, 0x69, 0x74, 0x65, 0x73, 0x12a8, 0x1230, 0x12d3, 0x1275,
0x645, 0x985, 0x9aa, 0x9f0, 0x9be, 0x9b9, 0x9cd, 0x9a8, 0x64, 0x65, 0x20, 0x6c,
0x61, 0x20, 0x74, 0x61, 0x72, 0x64, 0x65, 0x69, 0x63, 0x68, 0x61, 0x6d,
0x74, 0x68, 0x69, 0x41f, 0x41c, 0x63, 0x25b, 0x25b, 0x301, 0x6e, 0x6b, 0x6f,
0x49, 0x20, 0x253, 0x75, 0x67, 0x61, 0x6a, 0x254, 0x70, 0x61, 0x6b, 0x61,
0x73, 0x75, 0x62, 0x61, 0x70, 0x61, 0x6d, 0x75, 0x6e, 0x79, 0x69, 0x938,
0x93e, 0x902, 0x91d, 0x92c, 0x947, 0x932, 0x93e, 0x938, 0x947, 0x70, 0x6f, 0x70,
0x6f, 0x64, 0x6e, 0x65, 0x43f, 0x43e, 0x441, 0x43b, 0x438, 0x458, 0x435, 0x20,
0x43f, 0x43e, 0x434, 0x43d, 0x435, 0x47, 0x2e, 0x4d, 0x2e, 0x441, 0x43b, 0x2e,
0x43e, 0x431, 0x2e, 0x100a, 0x1014, 0x1031, 0x4e0b, 0x5348, 0x70, 0x2e, 0xa0, 0x6d,
0x2e, 0x1e0c, 0x65, 0x66, 0x66, 0x69, 0x72, 0x20, 0x61, 0x7a, 0x61, 0x62f,
0x2e, 0x646, 0x13d2, 0x13af, 0x13f1, 0x13a2, 0x13d7, 0x13e2, 0x55, 0x68, 0x72, 0x20,
0x6e, 0x6f, 0x6d, 0x6d, 0x65, 0x6e, 0x64, 0x61, 0x61, 0x63, 0x68, 0x73,
0x70, 0x2e, 0x6d, 0x2e, 0x6f, 0x64, 0x70, 0x2e, 0x926, 0x92a, 0x948, 0x939,
0x930, 0x20, 0x92c, 0x93e, 0x926, 0x65, 0x62, 0x79, 0xe1, 0x6d, 0x75, 0xf55,
0xfb1, 0xf72, 0xf0b, 0xf46, 0xf0b, 0x55, 0x54, 0xd801, 0xdc11, 0xd801, 0xdc23, 0x70,
0x6d, 0xd801, 0xdc50, 0xd801, 0xdc65, 0x70, 0x74, 0x6d, 0x263, 0x65, 0x74, 0x72,
0x254, 0x6e, 0x67, 0x259, 0x67, 0xf3, 0x67, 0x259, 0x6c, 0x65, 0x69, 0x70,
0x2e, 0x73, 0x6f, 0x69, 0x72, 0x6b, 0x69, 0x6b, 0x69, 0x69, 0x257, 0x65,
0xd83a, 0xdd07, 0xd83a, 0xdd0e, 0x3bc, 0x2e, 0x3bc, 0x2e, 0x4d, 0x6f, 0x67, 0x59,
0x61, 0x6d, 0x6d, 0x61, 0x5d0, 0x5d7, 0x5d4, 0x5f4, 0x5e6, 0x64, 0x75, 0x2e,
0x65, 0x2e, 0x68, 0x2e, 0x4e, 0x2019, 0x61, 0x62, 0x61, 0x6c, 0x69, 0x65,
0x70, 0x2e, 0x70, 0x6f, 0x73, 0x20, 0x6d, 0x69, 0x64, 0xed, 0x69, 0x2e,
0x6e, 0x2e, 0x5348, 0x5f8c, 0x57, 0x65, 0x6e, 0x67, 0x69, 0x6e, 0x20, 0x74,
0x6d, 0x65, 0x64, 0x64, 0x69, 0x74, 0x6b, 0x6f, 0x6f, 0x73, 0x6b, 0x6f,
0x6c, 0x69, 0x6e, 0x79, 0x128, 0x79, 0x61, 0x77, 0x129, 0x6f, 0x6f, 0xc85,
0xcaa, 0xcb0, 0xcbe, 0xcb9, 0xccd, 0xca8, 0x67e, 0x6cc, 0x20, 0x627, 0x6cc, 0x645,
0x92a, 0x940, 0x20, 0x90f, 0x92e, 0x48, 0x77, 0x61, 0x129, 0x2d, 0x69, 0x6e,
0x129, 0x938, 0x93e, 0x902, 0x91c, 0xc624, 0xd6c4, 0x41, 0x6c, 0x75, 0x75, 0x6c,
0x61, 0x50, 0x4e, 0x6b, 0x75, 0x67, 0xfa, 0x442, 0x4af, 0x448, 0x442, 0x4e9,
0x43d, 0x20, 0x43a, 0x438, 0x439, 0x438, 0x43d, 0x43a, 0x438, 0x4d, 0x55, 0x55,
0xeab, 0xebc, 0xeb1, 0xe87, 0xe97, 0xec8, 0xebd, 0xe87, 0x70, 0x113, 0x63, 0x70,
0x75, 0x73, 0x64, 0x69, 0x65, 0x6e, 0x101, 0x6d, 0x70, 0xf3, 0x6b, 0x77,
0x61, 0x70, 0x6f, 0x70, 0x69, 0x65, 0x74, 0x77, 0xf3, 0x74, 0x70, 0x6f,
0x142, 0x64, 0x6e, 0x6a, 0x61, 0x44, 0x69, 0x6c, 0x6f, 0x6c, 0x6f, 0x4f,
0x54, 0x6e, 0x6f, 0x6d, 0xeb, 0x74, 0x74, 0x65, 0x73, 0x43f, 0x43e, 0x43f,
0x43b, 0x2e, 0x6b, 0x79, 0x69, 0x75, 0x6b, 0x6f, 0x6e, 0x79, 0x69, 0x6d,
0x63, 0x68, 0x6f, 0x63, 0x68, 0x69, 0x6c, 0x2019, 0x6c, 0x43, 0x68, 0x69,
0x6c, 0x6f, 0x50, 0x54, 0x47, 0x9aa, 0x9bf, 0x20, 0x98f, 0x9ae, 0x190, 0x6e,
0x64, 0xe1, 0x6d, 0xe2, 0x168, 0x47, 0x4af, 0x2e, 0x445, 0x2e, 0x1826, 0x1802,
0x20, 0x182c, 0x1823, 0x6c, 0x69, 0x6c, 0x6c, 0x69, 0x1c3, 0x75, 0x69, 0x61,
0x73, 0x905, 0x92a, 0x930, 0x93e, 0x939, 0x94d, 0x928, 0x6e, 0x63, 0x77, 0xf2,
0x6e, 0x7a, 0xe9, 0x6d, 0x14b, 0x6b, 0x61, 0x20, 0x6d, 0x62, 0x254, 0x301,
0x74, 0x20, 0x6e, 0x6a, 0x69, 0x46, 0x1ecd, 0x20, 0xed, 0x76, 0x6e, 0x69,
0x6e, 0x7e5, 0x65, 0x61, 0x68, 0x6b, 0x65, 0x74, 0x62, 0x65, 0x61, 0x69,
0x76, 0x65, 0x74, 0x65, 0x2e, 0x6d, 0x2e, 0x54, 0x14a, 0x57, 0x42, 0x4d5,
0x43c, 0x431, 0x438, 0x441, 0x431, 0x43e, 0x43d, 0x44b, 0x20, 0x444, 0x4d5, 0x441,
0x442, 0x4d5, 0x63a, 0x2e, 0x648, 0x2e, 0x628, 0x639, 0x62f, 0x627, 0x632, 0x638,
0x647, 0x631, 0x64, 0x61, 0x20, 0x74, 0x61, 0x72, 0x64, 0x65, 0x70, 0x61,
0x20, 0x70, 0x75, 0x73, 0x73, 0x69, 0x64, 0x65, 0x69, 0x6e, 0x61, 0x6e,
0xa2c, 0xa3e, 0x2e, 0xa26, 0xa41, 0x2e, 0x6b, 0x69, 0x6e, 0x67, 0x6f, 0x74,
0x6f, 0x5a, 0x2e, 0x4d, 0x57, 0x2e, 0x42d, 0x41a, 0x54, 0x65, 0x69, 0x70,
0x61, 0x4c, 0x4b, 0x50, 0x61, 0x73, 0x68, 0x61, 0x6d, 0x69, 0x68, 0x65,
0x1c67, 0x1c64, 0x1c6b, 0x1c5f, 0x1c79, 0x43f, 0x43e, 0x20, 0x43f, 0x43e, 0x434, 0x43d,
0x435, 0x70, 0x6f, 0x20, 0x70, 0x6f, 0x64, 0x6e, 0x65, 0x6e, 0x79, 0x69,
0x61, 0x67, 0x68, 0x75, 0x6f, 0xa06f, 0xa2d2, 0x70, 0x6f, 0x20, 0x70, 0x6f,
0x142, 0x65, 0x64, 0x6e, 0x69, 0x75, 0x645, 0x646, 0x62c, 0x647, 0x646, 0x62f,
0x60c, 0x20, 0x634, 0x627, 0x645, 0x936, 0x93e, 0x92e, 0x20, 0x91c, 0x93e, 0xdb4,
0x2e, 0xdc0, 0x2e, 0x70, 0x6f, 0x70, 0x2e, 0x45, 0x69, 0x67, 0x75, 0x6c,
0x6f, 0x47, 0x44, 0x2d5c, 0x2d30, 0x2d37, 0x2d33, 0x2d33, 0x2d6f, 0x2d30, 0x2d5c, 0x65,
0x6d, 0x61, 0x6d, 0x20, 0x4e, 0x61, 0x6d, 0x69, 0x74, 0x74, 0x61, 0x67,
0x70f, 0x712, 0x71b, 0x200c, 0x74, 0x61, 0x64, 0x67, 0x67, 0x2b7, 0x61, 0x74,
0x6c, 0x75, 0x6d, 0x61, 0x20, 0x6c, 0x77, 0x61, 0x20, 0x70, 0x5a, 0x61,
0x61, 0x72, 0x69, 0x6b, 0x61, 0x79, 0x20, 0x62, 0x45, 0x62, 0x6f, 0x6e,
0x67, 0x69, 0xe2b, 0xe25, 0xe31, 0xe07, 0xe40, 0xe17, 0xe35, 0xe48, 0xe22, 0xe07,
0xf55, 0xfb1, 0xf72, 0xf0b, 0xf51, 0xfb2, 0xf7c, 0xf0b, 0x12f5, 0x2e, 0x1240, 0x2e,
0x65, 0x66, 0x69, 0x61, 0x66, 0x69, 0xd6, 0x53, 0x67, 0xfc, 0x6e, 0x6f,
0x72, 0x74, 0x61, 0x64, 0x61, 0x6e, 0x20, 0x73, 0x6f, 0x148, 0x43f, 0x43f,
0x70, 0x6f, 0x70, 0x6f, 0x142, 0x64, 0x6e, 0x6a, 0x75, 0x686, 0x6c8, 0x634,
0x62a, 0x649, 0x646, 0x20, 0x643, 0x6d0, 0x64a, 0x649, 0x646, 0x54, 0x4b, 0x422,
0x41a, 0x43, 0x48, 0x79, 0x68, 0x4e, 0x67, 0x6f, 0x6b, 0x69, 0x73, 0x25b,
0x301, 0x6e, 0x64, 0x25b, 0x5e0, 0x5d0, 0x5b8, 0x5db, 0x5de, 0x5d9, 0x5d8, 0x5d0,
0x5b8, 0x5d2, 0x1ecc, 0x300, 0x73, 0xe1, 0x6e, 0x186, 0x300, 0x73, 0xe1, 0x6e,
0x62, 0x61, 0x6e, 0x72, 0x69, 0x6e, 0x67, 0x7a, 0x67, 0x76, 0x61, 0x71,
0x938, 0x93e, 0x902, 0x92e, 0x70, 0x69, 0x20, 0x70, 0x69, 0x6e, 0x69, 0x20,
0x73, 0x75, 0x6e, 0x6f, 0x28a, 0x73, 0x68, 0x69, 0x6c, 0xe8, 0x20, 0x6b,
0x28a, 0x73, 0x61, 0x73, 0x28a, 0x926, 0x92a, 0x947, 0x939, 0x930, 0x93e, 0x2f,
0x938, 0x902, 0x91c, 0x93e, 0xb2a, 0xb3f, 0x20, 0xb0f, 0xb2e, 0xc2a, 0xc3f, 0x20,
0xc0e, 0xc2e, 0xc4d
};

static constexpr char16_t currency_symbol_data[] = {
0x20be, 0x42, 0x72, 0x46, 0x64, 0x6a, 0x4e, 0x66, 0x6b, 0x52, 0x24, 0x46,
0x43, 0x46, 0x41, 0x47, 0x48, 0x20b5, 0x4c, 0x65, 0x6b, 0xeb, 0x20ac, 0x64,
0x65, 0x6e, 0x1265, 0x122d, 0x62c, 0x2e, 0x645, 0x2e, 0x200f, 0x62f, 0x2e, 0x62c,
0x2e, 0x200f, 0x62f, 0x2e, 0x628, 0x2e, 0x200f, 0x62f, 0x2e, 0x639, 0x2e, 0x200f,
0x20aa, 0x62f, 0x2e, 0x623, 0x2e, 0x200f, 0x62f, 0x2e, 0x643, 0x2e, 0x200f, 0x644,
0x2e, 0x644, 0x2e, 0x200f, 0x62f, 0x2e, 0x644, 0x2e, 0x200f, 0x623, 0x2e, 0x645,
0x2e, 0x62f, 0x2e, 0x645, 0x2e, 0x200f, 0x631, 0x2e, 0x639, 0x2e, 0x200f, 0x631,
0x2e, 0x642, 0x2e, 0x200f, 0x631, 0x2e, 0x633, 0x2e, 0x200f, 0x53, 0xa3, 0x62c,
0x2e, 0x633, 0x2e, 0x644, 0x2e, 0x633, 0x2e, 0x200f, 0x62f, 0x2e, 0x62a, 0x2e,
0x200f, 0x62f, 0x2e, 0x625, 0x2e, 0x200f, 0x631, 0x2e, 0x64a, 0x2e, 0x200f, 0x58f,
0x20b9, 0x54, 0x53, 0x68, 0x20a6, 0x20bc, 0x20ba, 0x46, 0x202f, 0x43, 0x46, 0x41,
0x9f3, 0x20bd, 0x4b, 0x4b, 0x4d, 0x41a, 0x41c, 0x43b, 0x432, 0x2e, 0x48, 0x4b,
0x24, 0xffe5, 0x20b1, 0x55, 0x53, 0x68, 0xa5, 0x4d, 0x4f, 0x50, 0x24, 0x45,
0x55, 0x52, 0x4b, 0x10d, 0x6b, 0x72, 0x2e, 0x783, 0x2e, 0x41, 0x66, 0x6c,
0x2e, 0x43, 0x67, 0x2e, 0x4e, 0x75, 0x2e, 0x4b, 0x73, 0x68, 0x55, 0x53,
0x24, 0x46, 0x42, 0x75, 0x44, 0x52, 0x70, 0x41, 0x72, 0x4d, 0x4b, 0x52,
0x4d, 0x52, 0x66, 0x52, 0x73, 0x52, 0x46, 0x57, 0x53, 0x24, 0x53, 0x52,
0x54, 0x24, 0x56, 0x54, 0x44, 0x41, 0x46, 0x43, 0x46, 0x50, 0x46, 0x47,
0x55, 0x4d, 0x4c, 0x53, 0x44, 0x54, 0xd83a, 0xdd05, 0xd83a, 0xdd0a, 0xd83a, 0xdd00,
0xd83a, 0xdd0a, 0xd83a, 0xdd05, 0xd83a, 0xdd0a, 0xd83a, 0xdd00, 0x20b2, 0x46, 0x74, 0x43,
0x41, 0x24, 0x200b, 0x20b8, 0x17db, 0x51, 0x20a9, 0x43, 0x4e, 0xa5, 0x441, 0x43e,
0x43c, 0x20ad, 0x4b, 0x7a, 0x434, 0x435, 0x43d, 0x2e, 0x4d, 0x54, 0x6e, 0x20ae,
0x928, 0x947, 0x930, 0x942, 0x7ff, 0x60b, 0x631, 0x6cc, 0x627, 0x644, 0x7a, 0x142,
0x44, 0x62, 0x53, 0x2f, 0x42, 0x73, 0x6c, 0x65, 0x69, 0x20b4, 0xdbb, 0xdd4,
0x2e, 0x20a1, 0x52, 0x44, 0x24, 0x43, 0x24, 0x42, 0x2f, 0x2e, 0x47, 0x73,
0x2e, 0x42, 0x73, 0x2e, 0x53, 0x20ab, 0x441, 0x43e, 0x43c, 0x2e, 0x52, 0x73,
0x2e, 0x4e, 0x54, 0x24, 0xe3f, 0x73, 0x6f, 0x2bb, 0x6d, 0x441, 0x45e, 0x43c
};

static constexpr char16_t currency_display_name_data[] = {
0x53, 0x75, 0x69, 0x64, 0x2d, 0x41, 0x66, 0x72, 0x69, 0x6b, 0x61, 0x61,
0x6e, 0x73, 0x65, 0x20, 0x72, 0x61, 0x6e, 0x64, 0x4e, 0x61, 0x6d, 0x69,
0x62, 0x69, 0x65, 0x73, 0x65, 0x20, 0x64, 0x6f, 0x6c, 0x6c, 0x61, 0x72,
0x43, 0x46, 0x41, 0x20, 0x46, 0xe0, 0x6c, 0xe2, 0x14b, 0x20, 0x42, 0x45,
0x41, 0x43, 0x47, 0x68, 0x61, 0x6e, 0x61, 0x20, 0x53, 0x69, 0x64, 0x69,
0x46, 0x72, 0xe2, 0x6e, 0x6b, 0x65, 0x20, 0x43, 0x46, 0x41, 0x4c, 0x65,
0x6b, 0x75, 0x20, 0x73, 0x68, 0x71, 0x69, 0x70, 0x74, 0x61, 0x72, 0x45,
0x75, 0x72, 0x6f, 0x6a, 0x61, 0x44, 0x65, 0x6e, 0x61, 0x72, 0x69, 0x20,
0x6d, 0x61, 0x71, 0x65, 0x64, 0x6f, 0x6e, 0x61, 0x73, 0x12e8, 0x12a2, 0x1275,
0x12ee, 0x1335, 0x12eb, 0x20, 0x1265, 0x122d, 0x62c, 0x646, 0x64a, 0x647, 0x20, 0x645,
0x635, 0x631, 0x64a, 0x62f, 0x64a, 0x646, 0x627, 0x631, 0x20, 0x62c, 0x632, 0x627,
0x626, 0x631, 0x64a, 0x62f, 0x64a, 0x646, 0x627, 0x631, 0x20, 0x628, 0x62d, 0x631,
0x64a, 0x646, 0x64a, 0x641, 0x631, 0x646, 0x643, 0x20, 0x648, 0x633, 0x637, 0x20,
0x623, 0x641, 0x631, 0x64a, 0x642, 0x64a, 0x641, 0x631, 0x646, 0x643, 0x20, 0x62c,
0x632, 0x631, 0x20, 0x627, 0x644, 0x642, 0x645, 0x631, 0x641, 0x631, 0x646, 0x643,
0x20, 0x62c, 0x64a, 0x628, 0x648, 0x62a, 0x64a, 0x646, 0x627, 0x643, 0x641, 0x627,
0x20, 0x623, 0x631, 0x64a, 0x62a, 0x631, 0x64a, 0x62f, 0x64a, 0x646, 0x627, 0x631,
0x20, 0x639, 0x631, 0x627, 0x642, 0x64a, 0x634, 0x64a, 0x643, 0x644, 0x20, 0x625,
0x633, 0x631, 0x627, 0x626, 0x64a, 0x644, 0x64a, 0x20, 0x62c, 0x62f, 0x64a, 0x62f,
0x62f, 0x64a, 0x646, 0x627, 0x631, 0x20, 0x623, 0x631, 0x62f, 0x646, 0x64a, 0x62f,
0x64a, 0x646, 0x627, 0x631, 0x20, 0x643, 0x648, 0x64a, 0x62a, 0x64a, 0x62c, 0x646,
0x64a, 0x647, 0x20, 0x644, 0x628, 0x646, 0x627, 0x646, 0x64a, 0x62f, 0x64a, 0x646,
0x627, 0x631, 0x20, 0x644, 0x64a, 0x628, 0x64a, 0x623, 0x648, 0x642, 0x64a, 0x629,
0x20, 0x645, 0x648, 0x631, 0x64a, 0x62a, 0x627, 0x646, 0x64a, 0x629, 0x62f, 0x631,
0x647, 0x645, 0x20, 0x645, 0x63a, 0x631, 0x628, 0x64a, 0x631, 0x64a, 0x627, 0x644,
0x20, 0x639, 0x645, 0x627, 0x646, 0x64a, 0x631, 0x64a, 0x627, 0x644, 0x20, 0x642,
0x637, 0x631, 0x64a, 0x631, 0x64a, 0x627, 0x644, 0x20, 0x633, 0x639, 0x648, 0x62f,
0x64a, 0x634, 0x644, 0x646, 0x20, 0x635, 0x648, 0x645, 0x627, 0x644, 0x64a, 0x62c,
0x646, 0x64a, 0x647, 0x20, 0x62c, 0x646, 0x648, 0x628, 0x20, 0x627, 0x644, 0x633,
0x648, 0x62f, 0x627, 0x646, 0x62c, 0x646, 0x64a, 0x647, 0x20, 0x633, 0x648, 0x62f,
0x627, 0x646, 0x64a, 0x644, 0x64a, 0x631, 0x629, 0x20, 0x633, 0x648, 0x631, 0x64a,
0x629, 0x62f, 0x64a, 0x646, 0x627, 0x631, 0x20, 0x62a, 0x648, 0x646, 0x633, 0x64a,
0x62f, 0x631, 0x647, 0x645, 0x20, 0x625, 0x645, 0x627, 0x631, 0x627, 0x62a, 0x64a,
0x631, 0x64a, 0x627, 0x644, 0x20, 0x64a, 0x645, 0x646, 0x64a, 0x65, 0x75, 0x72,
0x6f, 0x570, 0x561, 0x575, 0x56f, 0x561, 0x56f, 0x561, 0x576, 0x20, 0x564, 0x580,
0x561, 0x574, 0x9ad, 0x9be, 0x9f0, 0x9a4, 0x9c0, 0x9af, 0x9bc, 0x20, 0x9f0, 0x9c1,
0x9aa, 0x9c0, 0x73, 0x68, 0x69, 0x6c, 0x69, 0x6e, 0x67, 0x69, 0x20, 0x79,
0x61, 0x20, 0x54, 0x61, 0x6e, 0x64, 0x68, 0x61, 0x6e, 0x69, 0x61, 0x41,
0x6d, 0x61, 0x6e, 0x41, 0x7a, 0x259, 0x72, 0x62, 0x61, 0x79, 0x63, 0x61,
0x6e, 0x20, 0x4d, 0x61, 0x6e, 0x61, 0x74, 0x131, 0x43c, 0x430, 0x43d, 0x430,
0x442, 0x66, 0x72, 0xe1, 0x14b, 0x73, 0x65, 0x66, 0x61, 0x20, 0x46, 0x72,
0x61, 0x14b, 0x20, 0x28, 0x42, 0x43, 0x45, 0x41, 0x4f, 0x29, 0x9ac, 0x9be,
0x982, 0x9b2, 0x9be, 0x9a6, 0x9c7, 0x9b6, 0x9c0, 0x20, 0x99f, 0x9be, 0x995, 0x9be,
0x9ad, 0x9be, 0x9b0, 0x9a4, 0x9c0, 0x9af, 0x9bc, 0x20, 0x9b0, 0x9c1, 0x9aa, 0x9bf,
0x46, 0x72, 0x1ce, 0x14b, 0x20, 0x43, 0x46, 0x41, 0x20, 0x28, 0x42, 0x45,
0x41, 0x43, 0x29, 0x65, 0x75, 0x72, 0x6f, 0x61, 0x431, 0x435, 0x43b, 0x430,
0x440, 0x443, 0x441, 0x43a, 0x456, 0x20, 0x440, 0x443, 0x431, 0x435, 0x43b, 0x44c,
0x53, 0x68, 0x69, 0x6c, 0x69, 0x6e, 0x67, 0x69, 0x20, 0x79, 0x61, 0x20,
0x48, 0x75, 0x74, 0x61, 0x6e, 0x7a, 0x61, 0x6e, 0x69, 0x61, 0x92d, 0x93e,
0x930, 0x924, 0x928, 0x93f, 0x20, 0x930, 0x941, 0x92a, 0x940, 0x42, 0x6f, 0x73,
0x61, 0x6e, 0x73, 0x6b, 0x6f, 0x68, 0x65, 0x72, 0x63, 0x65, 0x67, 0x6f,
0x76, 0x61, 0x10d, 0x6b, 0x61, 0x20, 0x6b, 0x6f, 0x6e, 0x76, 0x65, 0x72,
0x74, 0x69, 0x62, 0x69, 0x6c, 0x6e, 0x61, 0x20, 0x6d, 0x61, 0x72, 0x6b,
0x61, 0x41a, 0x43e, 0x43d, 0x432, 0x435, 0x440, 0x442, 0x438, 0x431, 0x438, 0x43b,
0x43d, 0x430, 0x20, 0x43c, 0x430, 0x440, 0x43a, 0x430, 0x411, 0x44a, 0x43b, 0x433,
0x430, 0x440, 0x441, 0x43a, 0x438, 0x20, 0x43b, 0x435, 0x432, 0x1019, 0x103c, 0x1014,
0x103a, 0x1019, 0x102c, 0x20, 0x1000, 0x103b, 0x1015, 0x103a, 0x6e2f, 0x5e63, 0x4eba, 0x6c11,
0x5e01, 0x50, 0x68, 0x69, 0x6c, 0x69, 0x70, 0x70, 0x69, 0x6e, 0x65, 0x20,
0x50, 0x65, 0x73, 0x6f, 0x44, 0x65, 0x72, 0x68, 0x65, 0x6d, 0x20, 0x55,
0x6d, 0x65, 0x1e5b, 0x1e5b, 0x75, 0x6b, 0x69, 0x62f, 0x6cc, 0x646, 0x627, 0x631,
0x6cc, 0x20, 0x639, 0x6ce, 0x631, 0x627, 0x642, 0x6cc, 0x695, 0x6cc, 0x627, 0x6b5,
0x6cc, 0x20, 0x626, 0x6ce, 0x631, 0x627, 0x646, 0x6cc, 0xd804, 0xdd1d, 0xd804, 0xdd01,
0xd804, 0xdd23, 0xd804, 0xdd18, 0xd804, 0xdd2c, 0xd804, 0xdd25, 0xd804, 0xdd28, 0x20, 0xd804,
0xdd11, 0xd804, 0xdd2c, 0xd804, 0xdd0b, 0xd804, 0xdd03, 0xd804, 0xdd28, 0xd804, 0xdd1a, 0xd804,
0xdd34, 0xd804, 0xdd18, 0xd804, 0xdd28, 0xd804, 0xdd20, 0xd804, 0xdd1a, 0xd804, 0xdd34, 0x20,
0xd804, 0xdd22, 0xd804, 0xdd2a, 0xd804, 0xdd1b, 0xd804, 0xdd28, 0x420, 0x43e, 0x441, 0x441,
0x438, 0x439, 0x43d, 0x20, 0x441, 0x43e, 0x43c, 0x55, 0x53, 0x20, 0x13a0, 0x13d5,
0x13b3, 0x45, 0x73, 0x68, 0x69, 0x72, 0x69, 0x6e, 0x67, 0x69, 0x20, 0x79,
0x61, 0x20, 0x55, 0x67, 0x61, 0x6e, 0x64, 0x61, 0x6e2f, 0x5143, 0x6fb3, 0x95e8,
0x5e01, 0x65b0, 0x52a0, 0x5761, 0x5143, 0x6fb3, 0x9580, 0x5143, 0x65b0, 0x53f0, 0x5e63, 0x440,
0x461, 0x441, 0x441, 0x456, 0x301, 0x439, 0x441, 0x43a, 0x457, 0x439, 0x20, 0x440,
0xa64b, 0x301, 0x431, 0x43b, 0x44c, 0x420, 0x430, 0x4ab, 0x4ab, 0x435, 0x439, 0x20,
0x442, 0x435, 0x43d, 0x43a, 0x4d7, 0x10d, 0x65, 0x73, 0x6b, 0xe1, 0x20, 0x6b,
0x6f, 0x72, 0x75, 0x6e, 0x61, 0x64, 0x61, 0x6e, 0x73, 0x6b, 0x20, 0x6b,
0x72, 0x6f, 0x6e, 0x65, 0x92d, 0x93e, 0x930, 0x924, 0x940, 0x20, 0x930, 0x92a,
0x947, 0x93d, 0x41, 0x72, 0x75, 0x62, 0x61, 0x61, 0x6e, 0x73, 0x65, 0x20,
0x67, 0x75, 0x6c, 0x64, 0x65, 0x6e, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x6b,
0x61, 0x61, 0x6e, 0x73, 0x65, 0x20, 0x64, 0x6f, 0x6c, 0x6c, 0x61, 0x72,
0x43, 0x61, 0x72, 0x69, 0x62, 0x69, 0x73, 0x63, 0x68, 0x65, 0x20, 0x67,
0x75, 0x6c, 0x64, 0x65, 0x6e, 0x53, 0x75, 0x72, 0x69, 0x6e, 0x61, 0x61,
0x6d, 0x73, 0x65, 0x20, 0x64, 0x6f, 0x6c, 0x6c, 0x61, 0x72, 0xf51, 0xf44,
0xf74, 0xf63, 0xf0b, 0xf40, 0xfb2, 0xf58, 0x53, 0x68, 0x69, 0x6c, 0x69, 0x6e,
0x67, 0x69, 0x20, 0x79, 0x61, 0x20, 0x4b, 0x65, 0x6e, 0x79, 0x61, 0x55,
0x53, 0x20, 0x44, 0x6f, 0x6c, 0x6c, 0x61, 0x72, 0x45, 0x61, 0x73, 0x74,
0x20, 0x43, 0x61, 0x72, 0x69, 0x62, 0x62, 0x65, 0x61, 0x6e, 0x20, 0x44,
0x6f, 0x6c, 0x6c, 0x61, 0x72, 0x41, 0x75, 0x73, 0x74, 0x72, 0x61, 0x6c,
0x69, 0x61, 0x6e, 0x20, 0x44, 0x6f, 0x6c, 0x6c, 0x61, 0x72, 0x42, 0x61,
0x68, 0x61, 0x6d, 0x69, 0x61, 0x6e, 0x20, 0x44, 0x6f, 0x6c, 0x6c, 0x61,
0x72, 0x42, 0x61, 0x72, 0x62, 0x61, 0x64, 0x69, 0x61, 0x6e, 0x20, 0x44,
0x6f, 0x6c, 0x6c, 0x61, 0x72, 0x42, 0x65, 0x6c, 0x69, 0x7a, 0x65, 0x20,
0x44, 0x6f, 0x6c, 0x6c, 0x61, 0x72, 0x42, 0x65, 0x72, 0x6d, 0x75, 0x64,
0x69, 0x61, 0x6e, 0x20, 0x44, 0x6f, 0x6c, 0x6c, 0x61, 0x72, 0x42, 0x6f,
0x74, 0x73, 0x77, 0x61, 0x6e, 0x61, 0x6e, 0x20, 0x50, 0x75, 0x6c, 0x61,
0x42, 0x75, 0x72, 0x75, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x20, 0x46, 0x72,
0x61, 0x6e, 0x63, 0x43, 0x65, 0x6e, 0x74, 0x72, 0x61, 0x6c, 0x20, 0x41,
0x66, 0x72, 0x69, 0x63, 0x61, 0x6e, 0x20, 0x43, 0x46, 0x41, 0x20, 0x46,
0x72, 0x61, 0x6e, 0x63, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x69, 0x61, 0x6e,
0x20, 0x44, 0x6f, 0x6c, 0x6c, 0x61, 0x72, 0x43, 0x61, 0x79, 0x6d, 0x61,
0x6e, 0x20, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x20, 0x44, 0x6f,
0x6c, 0x6c, 0x61, 0x72, 0x4e, 0x65, 0x77, 0x20, 0x5a, 0x65, 0x61, 0x6c,
0x61, 0x6e, 0x64, 0x20, 0x44, 0x6f, 0x6c, 0x6c, 0x61, 0x72, 0x44, 0x61,
0x6e, 0x69, 0x73, 0x68, 0x20, 0x4b, 0x72, 0x6f, 0x6e, 0x65, 0x45, 0x72,
0x69, 0x74, 0x72, 0x65, 0x61, 0x6e, 0x20, 0x4e, 0x61, 0x6b, 0x66, 0x61,
0x53, 0x77, 0x61, 0x7a, 0x69, 0x20, 0x4c, 0x69, 0x6c, 0x61, 0x6e, 0x67,
0x65, 0x6e, 0x69, 0x46, 0x61, 0x6c, 0x6b, 0x6c, 0x61, 0x6e, 0x64, 0x20,
0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x20, 0x50, 0x6f, 0x75, 0x6e,
0x64, 0x46, 0x69, 0x6a, 0x69, 0x61, 0x6e, 0x20, 0x44, 0x6f, 0x6c, 0x6c,
0x61, 0x72, 0x47, 0x61, 0x6d, 0x62, 0x69, 0x61, 0x6e, 0x20, 0x44, 0x61,
0x6c, 0x61, 0x73, 0x69, 0x47, 0x68, 0x61, 0x6e, 0x61, 0x69, 0x61, 0x6e,
0x20, 0x43, 0x65, 0x64, 0x69, 0x47, 0x69, 0x62, 0x72, 0x61, 0x6c, 0x74,
0x61, 0x72, 0x20, 0x50, 0x6f, 0x75, 0x6e, 0x64, 0x55, 0x4b, 0x20, 0x50,
0x6f, 0x75, 0x6e, 0x64, 0x47, 0x75, 0x79, 0x61, 0x6e, 0x61, 0x65, 0x73,
0x65, 0x20, 0x44, 0x6f, 0x6c, 0x6c, 0x61, 0x72, 0x48, 0x6f, 0x6e, 0x67,
0x20, 0x4b, 0x6f, 0x6e, 0x67, 0x20, 0x44, 0x6f, 0x6c, 0x6c, 0x61, 0x72,
0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x20, 0x52, 0x75, 0x70, 0x65, 0x65,
0x49, 0x6e, 0x64, 0x6f, 0x6e, 0x65, 0x73, 0x69, 0x61, 0x6e, 0x20, 0x52,
0x75, 0x70, 0x69, 0x61, 0x68, 0x49, 0x73, 0x72, 0x61, 0x65, 0x6c, 0x69,
0x20, 0x4e, 0x65, 0x77, 0x20, 0x53, 0x68, 0x65, 0x6b, 0x65, 0x6c, 0x4a,
0x61, 0x6d, 0x61, 0x69, 0x63, 0x61, 0x6e, 0x20, 0x44, 0x6f, 0x6c, 0x6c,
0x61, 0x72, 0x4b, 0x65, 0x6e, 0x79, 0x61, 0x6e, 0x20, 0x53, 0x68, 0x69,
0x6c, 0x6c, 0x69, 0x6e, 0x67, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x20, 0x41,
0x66, 0x72, 0x69, 0x63, 0x61, 0x6e, 0x20, 0x52, 0x61, 0x6e, 0x64, 0x4c,
0x69, 0x62, 0x65, 0x72, 0x69, 0x61, 0x6e, 0x20, 0x44, 0x6f, 0x6c, 0x6c,
0x61, 0x72, 0x4d, 0x61, 0x63, 0x61, 0x6e, 0x65, 0x73, 0x65, 0x20, 0x50,
0x61, 0x74, 0x61, 0x63, 0x61, 0x4d, 0x61, 0x6c, 0x61, 0x67, 0x61, 0x73,
0x79, 0x20, 0x41, 0x72, 0x69, 0x61, 0x72, 0x79, 0x4d, 0x61, 0x6c, 0x61,
0x77, 0x69, 0x61, 0x6e, 0x20, 0x4b, 0x77, 0x61, 0x63, 0x68, 0x61, 0x4d,
0x61, 0x6c, 0x61, 0x79, 0x73, 0x69, 0x61, 0x6e, 0x20, 0x52, 0x69, 0x6e,
0x67, 0x67, 0x69, 0x74, 0x4d, 0x61, 0x6c, 0x64, 0x69, 0x76, 0x69, 0x61,
0x6e, 0x20, 0x52, 0x75, 0x66, 0x69, 0x79, 0x61, 0x61, 0x4d, 0x61, 0x75,
0x72, 0x69, 0x74, 0x69, 0x61, 0x6e, 0x20, 0x52, 0x75, 0x70, 0x65, 0x65,
0x4e, 0x61, 0x6d, 0x69, 0x62, 0x69, 0x61, 0x6e, 0x20, 0x44, 0x6f, 0x6c,
0x6c, 0x61, 0x72, 0x4e, 0x69, 0x67, 0x65, 0x72, 0x69, 0x61, 0x6e, 0x20,
0x4e, 0x61, 0x69, 0x72, 0x61, 0x50, 0x61, 0x6b, 0x69, 0x73, 0x74, 0x61,
0x6e, 0x69, 0x20, 0x52, 0x75, 0x70, 0x65, 0x65, 0x50, 0x61, 0x70, 0x75,
0x61, 0x20, 0x4e, 0x65, 0x77, 0x20, 0x47, 0x75, 0x69, 0x6e, 0x65, 0x61,
0x6e, 0x20, 0x4b, 0x69, 0x6e, 0x61, 0x52, 0x77, 0x61, 0x6e, 0x64, 0x61,
0x6e, 0x20, 0x46, 0x72, 0x61, 0x6e, 0x63, 0x53, 0x74, 0x20, 0x48, 0x65,
0x6c, 0x65, 0x6e, 0x61, 0x20, 0x50, 0x6f, 0x75, 0x6e, 0x64, 0x53, 0x61,
0x6d, 0x6f, 0x61, 0x6e, 0x20, 0x54, 0x61, 0x6c, 0x61, 0x53, 0x65, 0x79,
0x63, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0x69, 0x73, 0x20, 0x52, 0x75, 0x70,
0x65, 0x65, 0x53, 0x69, 0x65, 0x72, 0x72, 0x61, 0x20, 0x4c, 0x65, 0x6f,
0x6e, 0x65, 0x61, 0x6e, 0x20, 0x4c, 0x65, 0x6f, 0x6e, 0x65, 0x53, 0x69,
0x6e, 0x67, 0x61, 0x70, 0x6f, 0x72, 0x65, 0x20, 0x44, 0x6f, 0x6c, 0x6c,
0x61, 0x72, 0x43, 0x61, 0x72, 0x69, 0x62, 0x62, 0x65, 0x61, 0x6e, 0x20,
0x67, 0x75, 0x69, 0x6c, 0x64, 0x65, 0x72, 0x53, 0x6f, 0x6c, 0x6f, 0x6d,
0x6f, 0x6e, 0x20, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x20, 0x44,
0x6f, 0x6c, 0x6c, 0x61, 0x72, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x20, 0x53,
0x75, 0x64, 0x61, 0x6e, 0x65, 0x73, 0x65, 0x20, 0x50, 0x6f, 0x75, 0x6e,
0x64, 0x53, 0x75, 0x64, 0x61, 0x6e, 0x65, 0x73, 0x65, 0x20, 0x50, 0x6f,
0x75, 0x6e, 0x64, 0x53, 0x77, 0x65, 0x64, 0x69, 0x73, 0x68, 0x20, 0x4b,
0x72, 0x6f, 0x6e, 0x61, 0x53, 0x77, 0x69, 0x73, 0x73, 0x20, 0x46, 0x72,
0x61, 0x6e, 0x63, 0x54, 0x61, 0x6e, 0x7a, 0x61, 0x6e, 0x69, 0x61, 0x6e,
0x20, 0x53, 0x68, 0x69, 0x6c, 0x6c, 0x69, 0x6e, 0x67, 0x54, 0x6f, 0x6e,
0x67, 0x61, 0x6e, 0x20, 0x50, 0x61, 0x2bb, 0x61, 0x6e, 0x67, 0x61, 0x54,
0x72, 0x69, 0x6e, 0x69, 0x64, 0x61, 0x64, 0x20, 0x26, 0x20, 0x54, 0x6f,
0x62, 0x61, 0x67, 0x6f, 0x20, 0x44, 0x6f, 0x6c, 0x6c, 0x61, 0x72, 0x55,
0x67, 0x61, 0x6e, 0x64, 0x61, 0x6e, 0x20, 0x53, 0x68, 0x69, 0x6c, 0x6c,
0x69, 0x6e, 0x67, 0x55, 0x6e, 0x69, 0x74, 0x65, 0x64, 0x20, 0x41, 0x72,
0x61, 0x62, 0x20, 0x45, 0x6d, 0x69, 0x72, 0x61, 0x74, 0x65, 0x73, 0x20,
0x44, 0x69, 0x72, 0x68, 0x61, 0x6d, 0x42, 0x72, 0x69, 0x74, 0x69, 0x73,
0x68, 0x20, 0x50, 0x6f, 0x75, 0x6e, 0x64, 0x56, 0x61, 0x6e, 0x75, 0x61,
0x74, 0x75, 0x20, 0x56, 0x61, 0x74, 0x75, 0x5a, 0x61, 0x6d, 0x62, 0x69,
0x61, 0x6e, 0x20, 0x4b, 0x77, 0x61, 0x63, 0x68, 0x61, 0x67, 0x68, 0x61,
0x6e, 0x61, 0x20, 0x73, 0x69, 0x256, 0x69, 0x263, 0x65, 0x74, 0x6f, 0x256,
0x6f, 0x66, 0x65, 0x20, 0x61, 0x66, 0x72, 0x69, 0x6b, 0x61, 0x67, 0x61,
0x20, 0x43, 0x46, 0x41, 0x20, 0x66, 0x72, 0x61, 0x6e, 0x63, 0x20, 0x42,
0x43, 0x45, 0x41, 0x4f, 0x46, 0x259, 0x6c, 0xe1, 0x14b, 0x20, 0x43, 0x46,
0x41, 0x20, 0x28, 0x42, 0x45, 0x41, 0x43, 0x29, 0x64, 0x6f, 0x6e, 0x73,
0x6b, 0x20, 0x6b, 0x72, 0xf3, 0x6e, 0x61, 0x50, 0x69, 0x73, 0x6f, 0x20,
0x6e, 0x67, 0x20, 0x50, 0x69, 0x6c, 0x69, 0x70, 0x69, 0x6e, 0x61, 0x73,
0x64, 0x69, 0x6e, 0x61, 0x72, 0x20, 0x61, 0x6c, 0x67, 0xe9, 0x72, 0x69,
0x65, 0x6e, 0x66, 0x72, 0x61, 0x6e, 0x63, 0x20, 0x43, 0x46, 0x41, 0x20,
0x28, 0x42, 0x43, 0x45, 0x41, 0x4f, 0x29, 0x66, 0x72, 0x61, 0x6e, 0x63,
0x20, 0x62, 0x75, 0x72, 0x75, 0x6e, 0x64, 0x61, 0x69, 0x73, 0x66, 0x72,
0x61, 0x6e, 0x63, 0x20, 0x43, 0x46, 0x41, 0x20, 0x28, 0x42, 0x45, 0x41,
0x43, 0x29, 0x64, 0x6f, 0x6c, 0x6c, 0x61, 0x72, 0x20, 0x63, 0x61, 0x6e,
0x61, 0x64, 0x69, 0x65, 0x6e, 0x66, 0x72, 0x61, 0x6e, 0x63, 0x20, 0x63,
0x6f, 0x6d, 0x6f, 0x72, 0x69, 0x65, 0x6e, 0x66, 0x72, 0x61, 0x6e, 0x63,
0x20, 0x63, 0x6f, 0x6e, 0x67, 0x6f, 0x6c, 0x61, 0x69, 0x73, 0x66, 0x72,
0x61, 0x6e, 0x63, 0x20, 0x64, 0x6a, 0x69, 0x62, 0x6f, 0x75, 0x74, 0x69,
0x65, 0x6e, 0x66, 0x72, 0x61, 0x6e, 0x63, 0x20, 0x43, 0x46, 0x50, 0x66,
0x72, 0x61, 0x6e, 0x63, 0x20, 0x67, 0x75, 0x69, 0x6e, 0xe9, 0x65, 0x6e,
0x67, 0x6f, 0x75, 0x72, 0x64, 0x65, 0x20, 0x68, 0x61, 0xef, 0x74, 0x69,
0x65, 0x6e, 0x6e, 0x65, 0x61, 0x72, 0x69, 0x61, 0x72, 0x79, 0x20, 0x6d,
0x61, 0x6c, 0x67, 0x61, 0x63, 0x68, 0x65, 0x6f, 0x75, 0x67, 0x75, 0x69,
0x79, 0x61, 0x20, 0x6d, 0x61, 0x75, 0x72, 0x69, 0x74, 0x61, 0x6e, 0x69,
0x65, 0x6e, 0x72, 0x6f, 0x75, 0x70, 0x69, 0x65, 0x20, 0x6d, 0x61, 0x75,
0x72, 0x69, 0x63, 0x69, 0x65, 0x6e, 0x6e, 0x65, 0x64, 0x69, 0x72, 0x68,
0x61, 0x6d, 0x20, 0x6d, 0x61, 0x72, 0x6f, 0x63, 0x61, 0x69, 0x6e, 0x66,
0x72, 0x61, 0x6e, 0x63, 0x20, 0x72, 0x77, 0x61, 0x6e, 0x64, 0x61, 0x69,
0x73, 0x72, 0x6f, 0x75, 0x70, 0x69, 0x65, 0x20, 0x64, 0x65, 0x73, 0x20,
0x53, 0x65, 0x79, 0x63, 0x68, 0x65, 0x6c, 0x6c, 0x65, 0x73, 0x66, 0x72,
0x61, 0x6e, 0x63, 0x20, 0x73, 0x75, 0x69, 0x73, 0x73, 0x65, 0x6c, 0x69,
0x76, 0x72, 0x65, 0x20, 0x73, 0x79, 0x72, 0x69, 0x65, 0x6e, 0x6e, 0x65,
0x64, 0x69, 0x6e, 0x61, 0x72, 0x20, 0x74, 0x75, 0x6e, 0x69, 0x73, 0x69,
0x65, 0x6e, 0x76, 0x61, 0x74, 0x75, 0x20, 0x76, 0x61, 0x6e, 0x75, 0x61,
0x74, 0x75, 0x61, 0x6e, 0x4d, 0x62, 0x75, 0x75, 0x257, 0x75, 0x20, 0x53,
0x65, 0x65, 0x66, 0x61, 0x61, 0x20, 0x42, 0x43, 0x45, 0x41, 0x4f, 0xd83a,
0xdd0a, 0xd83a, 0xdd22, 0xd83a, 0xdd2a, 0xd83a, 0xdd22, 0xd83a, 0xdd32, 0x20, 0xd83a, 0xdd05,
0xd83a, 0xdd0a, 0xd83a, 0xdd00, 0x20, 0xd83a, 0xdd16, 0xd83a, 0xdd2d, 0xd83a, 0xdd45, 0xd83a,
0xdd2a, 0xd83a, 0xdd32, 0xd83a, 0xdd22, 0xd83a, 0xdd44, 0xd83a, 0xdd32, 0xd83a, 0xdd3a, 0xd83a,
0xdd2b, 0x20, 0xd83a, 0xdd00, 0xd83a, 0xdd2c, 0xd83a, 0xdd2a, 0xd83a, 0xdd2d, 0xd83a, 0xdd33,
0xd83a, 0xdd22, 0xd83a, 0xdd0a, 0xd83a, 0xdd22, 0xd83a, 0xdd2a, 0xd83a, 0xdd22, 0xd83a, 0xdd32,
0x20, 0xd83a, 0xdd1a, 0xd83a, 0xdd35, 0xd83a, 0xdd26, 0xd83a, 0xdd2e, 0xd83a, 0xdd45, 0xd83a,
0xdd2a, 0xd83a, 0xdd2d, 0x20, 0xd83a, 0xdd00, 0xd83a, 0xdd2c, 0xd83a, 0xdd2a, 0xd83a, 0xdd2d,
0xd83a, 0xdd33, 0xd83a, 0xdd2d, 0xd83a, 0xdd32, 0xd83a, 0xdd33, 0xd83a, 0xdd2e, 0xd83a, 0xdd01,
0xd83a, 0xdd22, 0xd83a, 0xdd24, 0xd83a, 0xdd22, 0xd83a, 0xdd27, 0xd83a, 0xdd2d, 0x20, 0xd83a,
0xdd18, 0xd83a, 0xdd22, 0xd83a, 0xdd25, 0xd83a, 0xdd26, 0xd83a, 0xdd2d, 0xd83a, 0xdd32, 0xd83a,
0xdd33, 0xd83a, 0xdd2e, 0xd83a, 0xdd05, 0xd83a, 0xdd2d, 0xd83a, 0xdd23, 0xd83a, 0xdd2d, 0x20,
0xd83a, 0xdd18, 0xd83a, 0xdd22, 0xd83a, 0xdd32, 0xd83a, 0xdd22, 0xd83a, 0xdd32, 0xd83a, 0xdd33,
0xd83a, 0xdd2e, 0xd83a, 0xdd0a, 0xd83a, 0xdd22, 0xd83a, 0xdd2a, 0xd83a, 0xdd22, 0xd83a, 0xdd32,
0x20, 0xd83a, 0xdd18, 0xd83a, 0xdd2d, 0xd83a, 0xdd32, 0xd83a, 0xdd2b, 0xd83a, 0xdd32, 0xd83a,
0xdd33, 0xd83a, 0xdd2e, 0xd83a, 0xdd01, 0xd83a, 0xdd22, 0xd83a, 0xdd24, 0xd83a, 0xdd22, 0x20,
0xd83a, 0xdd02, 0xd83a, 0xdd2d, 0xd83a, 0xdd26, 0xd83a, 0xdd2b, 0xd83a, 0xdd2a, 0xd83a, 0xdd2d,
0xd83a, 0xdd34, 0xd83a, 0xdd22, 0xd83a, 0xdd32, 0xd83a, 0xdd33, 0xd83a, 0xdd2e, 0xd83a, 0xdd13,
0xd83a, 0xdd3a, 0xd83a, 0xdd2d, 0xd83a, 0xdd34, 0xd83a, 0xdd22, 0x20, 0xd83a, 0xdd03, 0xd83a,
0xdd2e, 0xd83a, 0xdd2a, 0xd83a, 0xdd2d, 0xd83a, 0xdd3c, 0xd83a, 0xdd22, 0xd83a, 0xdd32, 0xd83a,
0xdd2d, 0xd83a, 0xdd34, 0xd83a, 0xdd22, 0xd83a, 0xdd32, 0xd83a, 0xdd33, 0xd83a, 0xdd2e, 0xd83a,
0xdd10, 0xd83a, 0xdd22, 0xd83a, 0xdd34, 0xd83a, 0xdd2a, 0xd83a, 0xdd22, 0x20, 0xd83a, 0xdd10,
0xd83a, 0xdd22, 0xd83a, 0xdd36, 0xd83a, 0xdd2d, 0xd83a, 0xdd2a, 0xd83a, 0xdd22, 0xd83a, 0xdd34,
0xd83a, 0xdd22, 0xd83a, 0xdd32, 0xd83a, 0xdd33, 0xd83a, 0xdd2e, 0xd83a, 0xdd02, 0xd83a, 0xdd2b,
0xd83a, 0xdd34, 0xd83a, 0xdd2e, 0xd83a, 0xdd32, 0x20, 0xd83a, 0xdd05, 0xd83a, 0xdd2b, 0xd83a,
0xdd2a, 0xd83a, 0xdd22, 0xd83a, 0xdd24, 0xd83a, 0xdd2d, 0xd83a, 0xdd34, 0xd83a, 0xdd22, 0xd83a,
0xdd32, 0xd83a, 0xdd33, 0xd83a, 0xdd2e, 0x4d, 0x62, 0x75, 0x75, 0x257, 0x69, 0x20,
0x53, 0x65, 0x65, 0x66, 0x61, 0x61, 0x20, 0x42, 0x45, 0x41, 0x43, 0x44,
0x61, 0x6c, 0x61, 0x73, 0x69, 0x20, 0x47, 0x61, 0x6d, 0x6d, 0x62, 0x69,
0x44, 0x6f, 0x6c, 0x61, 0x61, 0x72, 0x20, 0x4c, 0x69, 0x62, 0x65, 0x72,
0x69, 0x79, 0x61, 0x61, 0x55, 0x67, 0x69, 0x79, 0x79, 0x61, 0x20, 0x4d,
0x75, 0x72, 0x69, 0x74, 0x61, 0x6e, 0x69, 0x4e, 0x61, 0x79, 0x72, 0x61,
0x61, 0x20, 0x4e, 0x69, 0x6a, 0x65, 0x72, 0x69, 0x79, 0x61, 0x61, 0x4c,
0x65, 0x77, 0x6f, 0x6f, 0x6e, 0x20, 0x53, 0x65, 0x72, 0x61, 0x61, 0x20,
0x4c, 0x69, 0x79, 0x6f, 0x6e, 0x50, 0x75, 0x6e, 0x6e, 0x64, 0x20, 0x53,
0x61, 0x73, 0x61, 0x6e, 0x6e, 0x61, 0x63, 0x68, 0x53, 0x69, 0x6c, 0x69,
0x6e, 0x67, 0x69, 0x20, 0x65, 0x79, 0x61, 0x20, 0x59, 0x75, 0x67, 0x61,
0x6e, 0x64, 0x61, 0x10e5, 0x10d0, 0x10e0, 0x10d7, 0x10e3, 0x10da, 0x10d8, 0x20, 0x10da,
0x10d0, 0x10e0, 0x10d8, 0x53, 0x63, 0x68, 0x77, 0x65, 0x69, 0x7a, 0x65, 0x72,
0x20, 0x46, 0x72, 0x61, 0x6e, 0x6b, 0x65, 0x6e, 0x395, 0x3c5, 0x3c1, 0x3ce,
0xaad, 0xabe, 0xab0, 0xaa4, 0xac0, 0xaaf, 0x20, 0xab0, 0xac2, 0xaaa, 0xabf, 0xaaf,
0xabe, 0x4e, 0x61, 0x69, 0x72, 0x61, 0x72, 0x20, 0x4e, 0x61, 0x6a, 0x65,
0x72, 0x69, 0x79, 0x61, 0x646, 0x64e, 0x64a, 0x652, 0x631, 0x64e, 0x53, 0x69,
0x64, 0x69, 0x20, 0x6e, 0x61, 0x20, 0x47, 0x68, 0x61, 0x6e, 0x61, 0x4b,
0x75, 0x257, 0x69, 0x6e, 0x20, 0x53, 0x65, 0x66, 0x61, 0x20, 0x6e, 0x61,
0x20, 0x41, 0x66, 0x69, 0x72, 0x6b, 0x61, 0x20, 0x54, 0x61, 0x20, 0x59,
0x61, 0x6d, 0x6d, 0x61, 0x5e9, 0x5e7, 0x5dc, 0x20, 0x5d7, 0x5d3, 0x5e9, 0x92d,
0x93e, 0x930, 0x924, 0x940, 0x92f, 0x20, 0x930, 0x941, 0x92a, 0x92f, 0x93e, 0x6d,
0x61, 0x67, 0x79, 0x61, 0x72, 0x20, 0x66, 0x6f, 0x72, 0x69, 0x6e, 0x74,
0xed, 0x73, 0x6c, 0x65, 0x6e, 0x73, 0x6b, 0x20, 0x6b, 0x72, 0xf3, 0x6e,
0x61, 0x4e, 0x61, 0x1ecb, 0x72, 0x61, 0x52, 0x75, 0x70, 0x69, 0x61, 0x68,
0x20, 0x49, 0x6e, 0x64, 0x6f, 0x6e, 0x65, 0x73, 0x69, 0x61, 0x50, 0x75,
0x6e, 0x74, 0x20, 0x53, 0x74, 0x65, 0x69, 0x72, 0x6c, 0x69, 0x6e, 0x67,
0x66, 0x72, 0x61, 0x6e, 0x63, 0x6f, 0x20, 0x73, 0x76, 0x69, 0x7a, 0x7a,
0x65, 0x72, 0x6f, 0x65e5, 0x672c, 0x5186, 0x41, 0x331, 0x6e, 0x61, 0x69, 0x72,
0x61, 0x73, 0x65, 0x65, 0x66, 0x61, 0x20, 0x79, 0x61, 0x74, 0x69, 0x20,
0x42, 0x43, 0x45, 0x41, 0x4f, 0x53, 0x6b, 0x75, 0x64, 0x75, 0x20, 0x4b,
0x61, 0x62, 0x75, 0x76, 0x65, 0x72, 0x64, 0x69, 0x61, 0x6e, 0x75, 0x41,
0x64, 0x69, 0x6e, 0x61, 0x72, 0x20, 0x41, 0x7a, 0x7a, 0x61, 0x79, 0x72,
0x69, 0x46, 0x72, 0x61, 0x6e, 0x63, 0x20, 0x43, 0x46, 0x41, 0x64, 0x61,
0x6e, 0x6d, 0x61, 0x72, 0x6b, 0x69, 0x6d, 0x75, 0x74, 0x20, 0x6b, 0x6f,
0x72, 0x75, 0x75, 0x6e, 0x69, 0x53, 0x69, 0x6c, 0x69, 0x6e, 0x67, 0x69,
0x74, 0x61, 0x62, 0x20, 0x79, 0x61, 0x20, 0x4b, 0x65, 0x6e, 0x79, 0x61,
0x53, 0x69, 0x6c, 0x69, 0x6e, 0x67, 0x69, 0x20, 0x79, 0x61, 0x20, 0x4b,
0x65, 0x6e, 0x79, 0x61, 0xcad, 0xcbe, 0xcb0, 0xca4, 0xcc0, 0xcaf, 0x20, 0xcb0,
0xcc2, 0xcaa, 0xcbe, 0xcaf, 0xcbf, 0x6c1, 0x650, 0x646, 0x62f, 0x64f, 0x633, 0x62a,
0x672, 0x646, 0x620, 0x20, 0x631, 0x6c4, 0x67e, 0x64e, 0x6d2, 0x907, 0x902, 0x921,
0x93f, 0x92f, 0x928, 0x20, 0x930, 0x942, 0x92a, 0x940, 0x49a, 0x430, 0x437, 0x430,
0x49b, 0x441, 0x442, 0x430, 0x43d, 0x20, 0x442, 0x435, 0x4a3, 0x433, 0x435, 0x441,
0x456, 0x179a, 0x17c0, 0x179b, 0x200b, 0x1780, 0x1798, 0x17d2, 0x1796, 0x17bb, 0x1787, 0x17b6,
0x43, 0x69, 0x72, 0x69, 0x6e, 0x67, 0x69, 0x20, 0x79, 0x61, 0x20, 0x4b,
0x65, 0x6e, 0x79, 0x61, 0xb300, 0xd55c, 0xbbfc, 0xad6d, 0x20, 0xc6d0, 0xc911, 0xad6d,
0x20, 0xc704, 0xc548, 0xd654, 0xc870, 0xc120, 0x20, 0xbbfc, 0xc8fc, 0xc8fc, 0xc758, 0x20,
0xc778, 0xbbfc, 0x20, 0xacf5, 0xd654, 0xad6d, 0x20, 0xc6d0, 0x43, 0x46, 0x41, 0x20,
0x46, 0x72, 0x61, 0x14b, 0x20, 0x28, 0x42, 0x43, 0x45, 0x41, 0x4f, 0x29,
0x6c, 0xee, 0x72, 0x65, 0x79, 0xea, 0x20, 0x74, 0x69, 0x72, 0x6b, 0xee,
0x46, 0x72, 0x61, 0x14b, 0x20, 0x43, 0x46, 0x41, 0x20, 0x42, 0x45, 0x41,
0x43, 0x41a, 0x44b, 0x440, 0x433, 0x44b, 0x437, 0x441, 0x442, 0x430, 0x43d, 0x20,
0x441, 0x43e, 0x43c, 0x443, 0x53, 0x68, 0x69, 0x6c, 0xed, 0x69, 0x6e, 0x67,
0x69, 0x20, 0x79, 0x61, 0x20, 0x54, 0x61, 0x61, 0x6e, 0x73, 0x61, 0x6e,
0xed, 0x61, 0xea5, 0xeb2, 0xea7, 0x20, 0xe81, 0xeb5, 0xe9a, 0x65, 0x69, 0x72,
0x6f, 0x46, 0x61, 0x6c, 0xe1, 0x6e, 0x67, 0x61, 0x20, 0x79, 0x61, 0x20,
0x4b, 0x6f, 0x6e, 0x67, 0xf3, 0x4b, 0x77, 0x61, 0x6e, 0x7a, 0x61, 0x20,
0x79, 0x61, 0x20, 0x41, 0x6e, 0x67, 0xf3, 0x6c, 0x61, 0x46, 0x61, 0x6c,
0xe1, 0x6e, 0x67, 0x61, 0x20, 0x43, 0x46, 0x41, 0x20, 0x42, 0x45, 0x41,
0x43, 0x45, 0x75, 0x72, 0x61, 0x73, 0x4e, 0x66, 0x61, 0x6c, 0x61, 0x6e,
0x67, 0x61, 0x20, 0x77, 0x61, 0x20, 0x4b, 0x6f, 0x6e, 0x67, 0x75, 0x53,
0x69, 0x6c, 0x69, 0x6e, 0x67, 0x20, 0x6d, 0x61, 0x72, 0x20, 0x4b, 0x65,
0x6e, 0x79, 0x61, 0x53, 0x69, 0x72, 0x69, 0x6e, 0x6a, 0x69, 0x20, 0x79,
0x61, 0x20, 0x4b, 0x65, 0x6e, 0x79, 0x61, 0x41c, 0x430, 0x43a, 0x435, 0x434,
0x43e, 0x43d, 0x441, 0x43a, 0x438, 0x20, 0x434, 0x435, 0x43d, 0x430, 0x440, 0x53,
0x68, 0x69, 0x6c, 0x69, 0x6e, 0x67, 0x69, 0x20, 0x79, 0x61, 0x20, 0x54,
0x61, 0x6e, 0x7a, 0x61, 0x6e, 0x69, 0x61, 0xd07, 0xd28, 0xd4d, 0xd24, 0xd4d,
0xd2f, 0xd7b, 0x20, 0xd30, 0xd42, 0xd2a, 0x52, 0x69, 0x6e, 0x67, 0x67, 0x69,
0x74, 0x20, 0x4d, 0x61, 0x6c, 0x61, 0x79, 0x73, 0x69, 0x61, 0x62f, 0x648,
0x644, 0x631, 0x20, 0x628, 0x631, 0x648, 0x646, 0x64a, 0x631, 0x64a, 0x6a0, 0x762,
0x64a, 0x62a, 0x20, 0x645, 0x644, 0x64a, 0x633, 0x64a, 0x627, 0x44, 0x6f, 0x6c,
0x61, 0x72, 0x20, 0x42, 0x72, 0x75, 0x6e, 0x65, 0x69, 0x44, 0x6f, 0x6c,
0x61, 0x72, 0x20, 0x53, 0x69, 0x6e, 0x67, 0x61, 0x70, 0x75, 0x72, 0x61,
0x65, 0x77, 0x72, 0x6f, 0x987, 0x9a8, 0x9cd, 0x9a6, 0x9bf, 0x9af, 0x9bc, 0x9be,
0x9a8, 0x20, 0x9b0, 0x9c1, 0x9aa, 0x9c0, 0x54, 0x101, 0x72, 0x61, 0x20, 0x6f,
0x20, 0x41, 0x6f, 0x74, 0x65, 0x61, 0x72, 0x6f, 0x61, 0x49, 0x72, 0x6f,
0x70, 0x69, 0x79, 0x69, 0x61, 0x6e, 0xed, 0x20, 0x65, 0x20, 0x4b, 0x65,
0x6e, 0x79, 0x61, 0x49, 0x72, 0x6f, 0x70, 0x69, 0x79, 0x69, 0x61, 0x6e,
0xed, 0x20, 0x65, 0x20, 0x54, 0x61, 0x6e, 0x7a, 0x61, 0x6e, 0x69, 0x61,
0x627, 0x6cc, 0x631, 0x627, 0x646, 0x20, 0x631, 0x6cc, 0x627, 0x644, 0x73, 0x68,
0x69, 0x72, 0xe8, 0x41c, 0x43e, 0x43d, 0x433, 0x43e, 0x43b, 0x20, 0x442, 0x4e9,
0x433, 0x440, 0x4e9, 0x433, 0x44e, 0x430, 0x43d, 0x44c, 0x1833, 0x1825, 0x182c, 0x1825,
0x1837, 0x1822, 0x182d, 0x180c, 0x72, 0x6f, 0x75, 0x70, 0x69, 0x20, 0x6d, 0x6f,
0x72, 0x69, 0x73, 0x69, 0x65, 0x6e, 0x73, 0x6f, 0x6c, 0x61, 0x69, 0x20,
0x42, 0x45, 0x41, 0x43, 0x4e, 0x61, 0x6d, 0x69, 0x62, 0x69, 0x61, 0x20,
0x44, 0x6f, 0x6c, 0x6c, 0x61, 0x72, 0x69, 0x928, 0x947, 0x92a, 0x93e, 0x932,
0x940, 0x20, 0x930, 0x942, 0x92a, 0x948, 0x92f, 0x93e, 0x901, 0x92d, 0x93e, 0x930,
0x924, 0x940, 0x92f, 0x20, 0x930, 0x942, 0x92a, 0x93f, 0x901, 0x92f, 0x93e, 0x66,
0x65, 0x6c, 0xe1, 0x14b, 0x20, 0x43, 0x46, 0x41, 0x46, 0x25b, 0x6c, 0xe2,
0x14b, 0x4e, 0x61, 0x69, 0x6a, 0xed, 0x72, 0x69, 0xe1, 0x20, 0x4e, 0x61,
0xed, 0x72, 0x61, 0x7d6, 0x7cc, 0x7ec, 0x7e3, 0x7cd, 0x7ec, 0x7de, 0x7ca, 0x20,
0x7df, 0x7ce, 0x7ec, 0x20, 0x7df, 0x7ca, 0x7eb, 0x20, 0x7dd, 0x7ca, 0x7d9, 0x7ca,
0x7f2, 0x6e, 0x6f, 0x72, 0x67, 0x67, 0x61, 0x20, 0x6b, 0x72, 0x75, 0x76,
0x64, 0x6e, 0x6f, 0x72, 0x75, 0x6f, 0x167, 0x167, 0x61, 0x20, 0x6b, 0x72,
0x75, 0x76, 0x64, 0x6e, 0x6f, 0x44, 0x6f, 0x6c, 0x61, 0x20, 0x79, 0x61,
0x73, 0x65, 0x20, 0x41, 0x6d, 0x65, 0x6c, 0x69, 0x6b, 0x61, 0x6e, 0x6f,
0x72, 0x73, 0x6b, 0x65, 0x20, 0x6b, 0x72, 0x6f, 0x6e, 0x65, 0x72, 0xb2d,
0xb3e, 0xb30, 0xb24, 0xb40, 0xb5f, 0x20, 0xb1f, 0xb19, 0xb4d, 0xb15, 0xb3e, 0x49,
0x74, 0x6f, 0x6f, 0x70, 0x68, 0x69, 0x79, 0x61, 0x61, 0x20, 0x42, 0x69,
0x72, 0x72, 0x69, 0x69, 0x41b, 0x430, 0x440, 0x421, 0x43e, 0x43c, 0x46, 0x6c,
0x6f, 0x72, 0x69, 0x6e, 0x20, 0x64, 0x69, 0x20, 0x41, 0x72, 0x75, 0x62,
0x61, 0x627, 0x641, 0x63a, 0x627, 0x646, 0x6cd, 0x67e, 0x627, 0x6a9, 0x633, 0x62a,
0x627, 0x646, 0x6cd, 0x20, 0x6a9, 0x644, 0x62f, 0x627, 0x631, 0x647, 0x631, 0x6cc,
0x627, 0x644, 0x20, 0x627, 0x6cc, 0x631, 0x627, 0x646, 0x627, 0x641, 0x63a, 0x627,
0x646, 0x6cc, 0x20, 0x627, 0x641, 0x63a, 0x627, 0x646, 0x633, 0x62a, 0x627, 0x646,
0x7a, 0x142, 0x6f, 0x74, 0x79, 0x20, 0x70, 0x6f, 0x6c, 0x73, 0x6b, 0x69,
0x52, 0x65, 0x61, 0x6c, 0x20, 0x62, 0x72, 0x61, 0x73, 0x69, 0x6c, 0x65,
0x69, 0x72, 0x6f, 0x6b, 0x77, 0x61, 0x6e, 0x7a, 0x61, 0x20, 0x61, 0x6e,
0x67, 0x6f, 0x6c, 0x61, 0x6e, 0x6f, 0x65, 0x73, 0x63, 0x75, 0x64, 0x6f,
0x20, 0x63, 0x61, 0x62, 0x6f, 0x2d, 0x76, 0x65, 0x72, 0x64, 0x69, 0x61,
0x6e, 0x6f, 0x66, 0x72, 0x61, 0x6e, 0x63, 0x6f, 0x20, 0x43, 0x46, 0x41,
0x20, 0x28, 0x42, 0x45, 0x41, 0x43, 0x29, 0x66, 0x72, 0x61, 0x6e, 0x63,
0x6f, 0x20, 0x43, 0x46, 0x41, 0x20, 0x28, 0x42, 0x43, 0x45, 0x41, 0x4f,
0x29, 0x70, 0x61, 0x74, 0x61, 0x63, 0x61, 0x20, 0x6d, 0x61, 0x63, 0x61,
0x65, 0x6e, 0x73, 0x65, 0x6d, 0x65, 0x74, 0x69, 0x63, 0x61, 0x6c, 0x20,
0x6d, 0x6f, 0xe7, 0x61, 0x6d, 0x62, 0x69, 0x63, 0x61, 0x6e, 0x6f, 0x64,
0x6f, 0x62, 0x72, 0x61, 0x20, 0x64, 0x65, 0x20, 0x53, 0xe3, 0x6f, 0x20,
0x54, 0x6f, 0x6d, 0xe9, 0x20, 0x65, 0x20, 0x50, 0x72, 0xed, 0x6e, 0x63,
0x69, 0x70, 0x65, 0x66, 0x72, 0x61, 0x6e, 0x63, 0x6f, 0x20, 0x73, 0x75,
0xed, 0xe7, 0x6f, 0x64, 0xf3, 0x6c, 0x61, 0x72, 0x20, 0x64, 0x6f, 0x73,
0x20, 0x45, 0x73, 0x74, 0x61, 0x64, 0x6f, 0x73, 0x20, 0x55, 0x6e, 0x69,
0x64, 0x6f, 0x73, 0xa2d, 0xa3e, 0xa30, 0xa24, 0xa40, 0x20, 0xa30, 0xa41, 0xa2a,
0xa07, 0xa06, 0x631, 0x648, 0x67e, 0x626, 0x6cc, 0x6c1, 0x53, 0x6f, 0x6c, 0x20,
0x50, 0x65, 0x72, 0x75, 0x61, 0x6e, 0x6f, 0x42, 0x6f, 0x6c, 0x69, 0x76,
0x69, 0x61, 0x6e, 0x6f, 0x44, 0xf3, 0x6c, 0x61, 0x72, 0x20, 0x41, 0x6d,
0x65, 0x72, 0x69, 0x63, 0x61, 0x6e, 0x6f, 0x6c, 0x65, 0x75, 0x20, 0x72,
0x6f, 0x6d, 0xe2, 0x6e, 0x65, 0x73, 0x63, 0x6c, 0x65, 0x75, 0x20, 0x6d,
0x6f, 0x6c, 0x64, 0x6f, 0x76, 0x65, 0x6e, 0x65, 0x73, 0x63, 0x66, 0x72,
0x61, 0x6e, 0x63, 0x20, 0x73, 0x76, 0x69, 0x7a, 0x7a, 0x65, 0x72, 0x68,
0x65, 0x6c, 0x65, 0x72, 0x69, 0x20, 0x73, 0x61, 0x20, 0x54, 0x61, 0x6e,
0x7a, 0x61, 0x6e, 0x69, 0x61, 0x49, 0x66, 0x61, 0x72, 0x61, 0x6e, 0x67,
0x61, 0x20, 0x72, 0x79, 0x2019, 0x55, 0x62, 0x75, 0x72, 0x75, 0x6e, 0x64,
0x69, 0x440, 0x43e, 0x441, 0x441, 0x438, 0x439, 0x441, 0x43a, 0x438, 0x439, 0x20,
0x440, 0x443, 0x431, 0x43b, 0x44c, 0x431, 0x435, 0x43b, 0x43e, 0x440, 0x443, 0x441,
0x441, 0x43a, 0x438, 0x439, 0x20, 0x440, 0x443, 0x431, 0x43b, 0x44c, 0x43a, 0x430,
0x437, 0x430, 0x445, 0x441, 0x43a, 0x438, 0x439, 0x20, 0x442, 0x435, 0x43d, 0x433,
0x435, 0x43a, 0x438, 0x440, 0x433, 0x438, 0x437, 0x441, 0x43a, 0x438, 0x439, 0x20,
0x441, 0x43e, 0x43c, 0x43c, 0x43e, 0x43b, 0x434, 0x430, 0x432, 0x441, 0x43a, 0x438,
0x439, 0x20, 0x43b, 0x435, 0x439, 0x443, 0x43a, 0x440, 0x430, 0x438, 0x43d, 0x441,
0x43a, 0x430, 0x44f, 0x20, 0x433, 0x440, 0x438, 0x432, 0x43d, 0x430, 0x410, 0x440,
0x430, 0x441, 0x441, 0x44b, 0x44b, 0x439, 0x430, 0x20, 0x441, 0x43e, 0x43b, 0x43a,
0x443, 0x43e, 0x431, 0x430, 0x439, 0x430, 0x4e, 0x6a, 0x69, 0x6c, 0x69, 0x6e,
0x67, 0x69, 0x20, 0x65, 0x65, 0x6c, 0x20, 0x4b, 0x65, 0x6e, 0x79, 0x61,
0x66, 0x61, 0x72, 0xe2, 0x6e, 0x67, 0x61, 0x20, 0x43, 0x46, 0x41, 0x20,
0x28, 0x42, 0x45, 0x41, 0x43, 0x29, 0x49, 0x68, 0x65, 0x6c, 0x61, 0x20,
0x79, 0x61, 0x20, 0x54, 0x61, 0x6e, 0x73, 0x61, 0x6e, 0x69, 0x79, 0x61,
0x92d, 0x93e, 0x930, 0x924, 0x940, 0x92f, 0x20, 0x930, 0x942, 0x92a, 0x94d, 0x92f,
0x915, 0x92e, 0x94d, 0x1c65, 0x1c64, 0x1c67, 0x1c5a, 0x1c5b, 0x20, 0x1c68, 0x1c6e, 0x1c71,
0x1c5f, 0x1c5c, 0x20, 0x1c74, 0x1c5f, 0x1c60, 0x1c5f, 0xe8, 0x75, 0x72, 0x6f, 0x4d,
0x65, 0x74, 0x69, 0x63, 0x61, 0x6c, 0x20, 0x64, 0x65, 0x20, 0x4d, 0x6f,
0xe7, 0x61, 0x6d, 0x62, 0x69, 0x71, 0x75, 0x65, 0x441, 0x440, 0x43f, 0x441,
0x43a, 0x438, 0x20, 0x434, 0x438, 0x43d, 0x430, 0x440, 0x411, 0x43e, 0x441, 0x430,
0x43d, 0x441, 0x43a, 0x43e, 0x445, 0x435, 0x440, 0x446, 0x435, 0x433, 0x43e, 0x432,
0x430, 0x447, 0x43a, 0x430, 0x20, 0x43a, 0x43e, 0x43d, 0x432, 0x435, 0x440, 0x442,
0x438, 0x431, 0x438, 0x43b, 0x43d, 0x430, 0x20, 0x43c, 0x430, 0x440, 0x43a, 0x430,
0x415, 0x432, 0x440, 0x43e, 0x45, 0x76, 0x72, 0x6f, 0x73, 0x72, 0x70, 0x73,
0x6b, 0x69, 0x20, 0x64, 0x69, 0x6e, 0x61, 0x72, 0x73, 0x68, 0x69, 0x6c,
0x69, 0x6e, 0x67, 0x69, 0x20, 0x79, 0x61, 0x20, 0x54, 0x61, 0x6e, 0x7a,
0x61, 0x6e, 0x69, 0x61, 0x44, 0x6f, 0x72, 0x61, 0x20, 0x72, 0x65, 0x20,
0x41, 0x6d, 0x65, 0x72, 0x69, 0x6b, 0x61, 0x67e, 0x627, 0x6aa, 0x633, 0x62a,
0x627, 0x646, 0x64a, 0x20, 0x631, 0x67e, 0x64a, 0x939, 0x93f, 0x902, 0x926, 0x941,
0x938, 0x94d, 0x924, 0x93e, 0x928, 0x940, 0x20, 0x930, 0x941, 0x92a, 0x92f, 0x94b,
0xdc1, 0xdca, 0x200d, 0xdbb, 0xdd3, 0x20, 0xdbd, 0xd82, 0xd9a, 0xdcf, 0x20, 0xdbb,
0xdd4, 0xdb4, 0xdd2, 0xdba, 0xdbd, 0x65, 0x76, 0x72, 0x6f, 0x53, 0x68, 0x69,
0x6c, 0x69, 0x6e, 0x67, 0x6b, 0x61, 0x20, 0x53, 0x6f, 0x6f, 0x6d, 0x61,
0x61, 0x6c, 0x69, 0x79, 0x61, 0x46, 0x61, 0x72, 0x61, 0x6e, 0x20, 0x4a,
0x61, 0x62, 0x75, 0x75, 0x74, 0x69, 0x42, 0x69, 0x72, 0x74, 0x61, 0x20,
0x49, 0x74, 0x6f, 0x6f, 0x62, 0x62, 0x69, 0x79, 0x61, 0x53, 0x68, 0x69,
0x6c, 0x69, 0x6e, 0x67, 0x6b, 0x61, 0x20, 0x4b, 0x65, 0x6e, 0x79, 0x61,
0x70, 0x65, 0x73, 0x6f, 0x20, 0x61, 0x72, 0x67, 0x65, 0x6e, 0x74, 0x69,
0x6e, 0x6f, 0x64, 0xf3, 0x6c, 0x61, 0x72, 0x20, 0x62, 0x65, 0x6c, 0x69,
0x63, 0x65, 0xf1, 0x6f, 0x62, 0x6f, 0x6c, 0x69, 0x76, 0x69, 0x61, 0x6e,
0x6f, 0x72, 0x65, 0x61, 0x6c, 0x20, 0x62, 0x72, 0x61, 0x73, 0x69, 0x6c,
0x65, 0xf1, 0x6f, 0x50, 0x65, 0x73, 0x6f, 0x20, 0x63, 0x68, 0x69, 0x6c,
0x65, 0x6e, 0x6f, 0x70, 0x65, 0x73, 0x6f, 0x20, 0x63, 0x6f, 0x6c, 0x6f,
0x6d, 0x62, 0x69, 0x61, 0x6e, 0x6f, 0x63, 0x6f, 0x6c, 0xf3, 0x6e, 0x20,
0x63, 0x6f, 0x73, 0x74, 0x61, 0x72, 0x72, 0x69, 0x63, 0x65, 0x6e, 0x73,
0x65, 0x70, 0x65, 0x73, 0x6f, 0x20, 0x63, 0x75, 0x62, 0x61, 0x6e, 0x6f,
0x70, 0x65, 0x73, 0x6f, 0x20, 0x64, 0x6f, 0x6d, 0x69, 0x6e, 0x69, 0x63,
0x61, 0x6e, 0x6f, 0x64, 0xf3, 0x6c, 0x61, 0x72, 0x20, 0x65, 0x73, 0x74,
0x61, 0x64, 0x6f, 0x75, 0x6e, 0x69, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x66,
0x72, 0x61, 0x6e, 0x63, 0x6f, 0x20, 0x43, 0x46, 0x41, 0x20, 0x64, 0x65,
0x20, 0xc1, 0x66, 0x72, 0x69, 0x63, 0x61, 0x20, 0x43, 0x65, 0x6e, 0x74,
0x72, 0x61, 0x6c, 0x71, 0x75, 0x65, 0x74, 0x7a, 0x61, 0x6c, 0x6c, 0x65,
0x6d, 0x70, 0x69, 0x72, 0x61, 0x20, 0x68, 0x6f, 0x6e, 0x64, 0x75, 0x72,
0x65, 0xf1, 0x6f, 0x70, 0x65, 0x73, 0x6f, 0x20, 0x6d, 0x65, 0x78, 0x69,
0x63, 0x61, 0x6e, 0x6f, 0x63, 0xf3, 0x72, 0x64, 0x6f, 0x62, 0x61, 0x20,
0x6e, 0x69, 0x63, 0x61, 0x72, 0x61, 0x67, 0xfc, 0x65, 0x6e, 0x73, 0x65,
0x62, 0x61, 0x6c, 0x62, 0x6f, 0x61, 0x20, 0x70, 0x61, 0x6e, 0x61, 0x6d,
0x65, 0xf1, 0x6f, 0x67, 0x75, 0x61, 0x72, 0x61, 0x6e, 0xed, 0x20, 0x70,
0x61, 0x72, 0x61, 0x67, 0x75, 0x61, 0x79, 0x6f, 0x73, 0x6f, 0x6c, 0x20,
0x70, 0x65, 0x72, 0x75, 0x61, 0x6e, 0x6f, 0x70, 0x65, 0x73, 0x6f, 0x20,
0x66, 0x69, 0x6c, 0x69, 0x70, 0x69, 0x6e, 0x6f, 0x70, 0x65, 0x73, 0x6f,
0x20, 0x75, 0x72, 0x75, 0x67, 0x75, 0x61, 0x79, 0x6f, 0x62, 0x6f, 0x6c,
0xed, 0x76, 0x61, 0x72, 0x20, 0x73, 0x6f, 0x62, 0x65, 0x72, 0x61, 0x6e,
0x6f, 0x2d30, 0x2d37, 0x2d54, 0x2d49, 0x2d4e, 0x20, 0x2d4f, 0x20, 0x2d4d, 0x2d4e, 0x2d56,
0x2d54, 0x2d49, 0x2d31, 0x52, 0x75, 0x70, 0x65, 0x65, 0x20, 0x49, 0x6e, 0x64,
0x6f, 0x6e, 0xe9, 0x73, 0x69, 0x61, 0x46, 0x61, 0x72, 0x61, 0x6e, 0x67,
0x61, 0x20, 0x79, 0x61, 0x20, 0x4b, 0x6f, 0x6e, 0x67, 0x6f, 0x53, 0x68,
0x69, 0x6c, 0x69, 0x6e, 0x67, 0x69, 0x20, 0x79, 0x61, 0x20, 0x55, 0x67,
0x61, 0x6e, 0x64, 0x61, 0x73, 0x76, 0x65, 0x6e, 0x73, 0x6b, 0x20, 0x6b,
0x72, 0x6f, 0x6e, 0x61, 0x53, 0x63, 0x68, 0x77, 0x69, 0x69, 0x7a, 0x65,
0x72, 0x20, 0x46, 0x72, 0x61, 0x6e, 0x6b, 0x65, 0x61, 0x64, 0x72, 0x69,
0x6d, 0x20, 0x6e, 0x20, 0x6c, 0x6d, 0x263, 0x72, 0x69, 0x62, 0x421, 0x43e,
0x43c, 0x43e, 0x43d, 0x4e3, 0xb87, 0xba8, 0xbcd, 0xba4, 0xbbf, 0xbaf, 0x20, 0xbb0,
0xbc2, 0xbaa, 0xbbe, 0xbaf, 0xbcd, 0xbae, 0xbb2, 0xbc7, 0xbb7, 0xbbf, 0xbaf, 0xba9,
0xbcd, 0x20, 0xbb0, 0xbbf, 0xb99, 0xbcd, 0xb95, 0xbbf, 0xb9f, 0xbcd, 0xb9a, 0xbbf,
0xb99, 0xbcd, 0xb95, 0xbaa, 0xbcd, 0xbaa, 0xbc2, 0xbb0, 0xbcd, 0x20, 0xb9f, 0xbbe,
0xbb2, 0xbb0, 0xbcd, 0xb87, 0xbb2, 0xb99, 0xbcd, 0xb95, 0xbc8, 0x20, 0xbb0, 0xbc2,
0xbaa, 0xbbe, 0xbaf, 0xbcd, 0x70, 0x69, 0x6c, 0x61, 0x20, 0x54, 0x61, 0x69,
0x77, 0x61, 0x6e, 0x420, 0x43e, 0x441, 0x441, 0x438, 0x44f, 0x20, 0x441, 0x443,
0x43c, 0x44b, 0xc2d, 0xc3e, 0xc30, 0xc24, 0xc26, 0xc47, 0xc36, 0x20, 0xc30, 0xc42,
0xc2a, 0xc3e, 0xc2f, 0xc3f, 0x41, 0x6e, 0x67, 0x6f, 0x2019, 0x6f, 0x74, 0x6f,
0x6c, 0x20, 0x6c, 0x6f, 0x6b, 0x2019, 0x20, 0x55, 0x67, 0x61, 0x6e, 0x64,
0x61, 0x41, 0x6e, 0x67, 0x6f, 0x2019, 0x6f, 0x74, 0x6f, 0x6c, 0x20, 0x6c,
0x6f, 0x6b, 0x2019, 0x20, 0x4b, 0x65, 0x6e, 0x79, 0x61, 0xe1a, 0xe32, 0xe17,
0xf61, 0xf74, 0xf0b, 0xf68, 0xf53, 0xf0b, 0xf62, 0xf92, 0xfb1, 0xf0b, 0xf42, 0xf62,
0xf0b, 0xf66, 0xf92, 0xf7c, 0xf62, 0xf0b, 0x1293, 0x1255, 0x134b, 0x50, 0x61, 0x2bb,
0x61, 0x6e, 0x67, 0x61, 0x20, 0x66, 0x61, 0x6b, 0x61, 0x74, 0x6f, 0x6e,
0x67, 0x61, 0x54, 0xfc, 0x72, 0x6b, 0x20, 0x6c, 0x69, 0x72, 0x61, 0x73,
0x131, 0x54, 0xfc, 0x72, 0x6b, 0x6d, 0x65, 0x6e, 0x20, 0x6d, 0x61, 0x6e,
0x61, 0x64, 0x79, 0x4e, 0x65, 0x72, 0x61, 0x443, 0x43a, 0x440, 0x430, 0x457,
0x43d, 0x441, 0x44c, 0x43a, 0x430, 0x20, 0x433, 0x440, 0x438, 0x432, 0x43d, 0x44f,
0x67e, 0x627, 0x6a9, 0x633, 0x62a, 0x627, 0x646, 0x6cc, 0x20, 0x631, 0x648, 0x67e,
0x6cc, 0x6c1, 0x628, 0x6be, 0x627, 0x631, 0x62a, 0x6cc, 0x20, 0x631, 0x648, 0x67e,
0x6cc, 0x6c1, 0x62c, 0x6c7, 0x6ad, 0x6af, 0x648, 0x20, 0x64a, 0x6c8, 0x6d5, 0x646,
0x649, 0x4f, 0x2018, 0x7a, 0x62, 0x65, 0x6b, 0x69, 0x73, 0x74, 0x6f, 0x6e,
0x20, 0x73, 0x6f, 0x2018, 0x6d, 0x69, 0x40e, 0x437, 0x431, 0x435, 0x43a, 0x438,
0x441, 0x442, 0x43e, 0x43d, 0x20, 0x441, 0x45e, 0x43c, 0xa55e, 0xa524, 0xa52b, 0xa569,
0x20, 0xa55c, 0xa55e, 0xa54c, 0x4c, 0x61, 0x69, 0x62, 0x68, 0x69, 0x79, 0x61,
0x20, 0x44, 0x61, 0x6c, 0x61, 0x110, 0x1ed3, 0x6e, 0x67, 0x20, 0x56, 0x69,
0x1ec7, 0x74, 0x20, 0x4e, 0x61, 0x6d, 0x50, 0x75, 0x6e, 0x74, 0x20, 0x50,
0x72, 0x79, 0x64, 0x61, 0x69, 0x6e, 0x67e, 0x627, 0x6a9, 0x633, 0x62a, 0x627,
0x646, 0x626, 0x20, 0x631, 0x648, 0x67e, 0x6cc, 0x627, 0x648, 0x6af, 0x627, 0x646,
0x633, 0x62a, 0x627, 0x646, 0x626, 0x20, 0x627, 0x648, 0x6af, 0x627, 0x646, 0x6cc,
0x627, 0x6cc, 0x631, 0x627, 0x646, 0x626, 0x20, 0x631, 0x6cc, 0x627, 0x644, 0x46,
0x72, 0x61, 0x6e, 0x63, 0x20, 0x43, 0x46, 0x41, 0x20, 0x62, 0x75, 0x20,
0x41, 0x66, 0x72, 0x69, 0x6b, 0x20, 0x53, 0x6f, 0x77, 0x77, 0x75, 0x2d,
0x6a, 0x61, 0x6e, 0x74, 0x49, 0x52, 0x61, 0x6e, 0x64, 0x69, 0x20, 0x79,
0x61, 0x73, 0x65, 0x4d, 0x7a, 0x61, 0x6e, 0x74, 0x73, 0x69, 0x20, 0x41,
0x66, 0x72, 0x69, 0x6b, 0x61, 0x4e, 0xe1, 0xed, 0x72, 0xe0, 0x20, 0x4e,
0xe0, 0xec, 0x6a, 0xed, 0x72, 0xed, 0xe0, 0x46, 0x61, 0x72, 0x61, 0x6e,
0x73, 0xec, 0x20, 0xec, 0x77, 0x254, 0x300, 0x2d, 0x6f, 0x6f, 0x72, 0xf9,
0x6e, 0x20, 0x41, 0x66, 0xed, 0x72, 0xed, 0x6b, 0xe0, 0x79, 0x69, 0x6e,
0x7a, 0x6d, 0x69, 0x6e, 0x7a, 0x62, 0x69, 0x69, 0x2d, 0x53, 0x6f, 0x75,
0x74, 0x68, 0x20, 0x41, 0x66, 0x72, 0x69, 0x63, 0x61, 0x6e, 0x20, 0x52,
0x61, 0x6e, 0x64, 0x4d, 0x72, 0x61, 0x73, 0x69, 0x72, 0x20, 0x52, 0x65,
0x6a, 0x61, 0x72, 0x52, 0x65, 0x61, 0x75, 0x20, 0x42, 0x72, 0x61, 0x73,
0x69, 0x75, 0x77, 0x61, 0x72, 0x61, 0x50, 0x65, 0x73, 0x6f, 0x20, 0x4b,
0x75, 0x72, 0x169, 0x62, 0x69, 0x79, 0x61, 0x77, 0x61, 0x72, 0x61, 0x42,
0x75, 0x72, 0x69, 0x77, 0x61, 0x72, 0x69, 0x20, 0x57, 0x65, 0x6e, 0x65,
0x73, 0x75, 0x65, 0x72, 0x61, 0x77, 0x61, 0x72, 0x61, 0x50, 0xe1, 0x6b,
0x65, 0x73, 0x74, 0xe1, 0x6e, 0x69, 0x20, 0x72, 0x75, 0x70, 0x69, 0x47,
0x61, 0x72, 0x269, 0x256, 0x6f, 0x6e, 0x74, 0x1dd, 0x6e, 0x61, 0x20, 0x67,
0x269, 0x74, 0x65, 0x14b, 0x73, 0x68, 0x69, 0x6c, 0x65, 0x6c, 0x61, 0x14b,
0x20, 0x6b, 0x61, 0x73, 0x25b, 0x25b, 0x66, 0x61, 0x92d, 0x93e, 0x930, 0x924,
0x947, 0x20, 0x926, 0x93e, 0x20, 0x930, 0x941, 0x92a, 0x92f, 0x93e, 0x62, 0x101,
0x72, 0x61, 0x74, 0x20, 0x74, 0x69, 0x20, 0x1e6d, 0x6e, 0x6b, 0x101
};

static constexpr char16_t currency_format_data[] = {
0x25, 0x31, 0x25, 0x32, 0x25, 0x31, 0xa0, 0x25, 0x32, 0x28, 0x25, 0x32,
0x25, 0x31, 0x29, 0x25, 0x32, 0xa0, 0x25, 0x31, 0x28, 0x25, 0x31, 0xa0,
0x25, 0x32, 0x29, 0x200f, 0x25, 0x31, 0xa0, 0x25, 0x32, 0x61c, 0x25, 0x31,
0x25, 0x32, 0x28, 0x61c, 0x25, 0x31, 0x25, 0x32, 0x29, 0x28, 0x25, 0x31,
0x25, 0x32, 0x29, 0x28, 0x25, 0x31, 0x29, 0xa0, 0x25, 0x32, 0x28, 0x25,
0x32, 0xa0, 0x25, 0x31, 0x29, 0x25, 0x32, 0x2d, 0x25, 0x31, 0x200f, 0x25,
0x31, 0xa0, 0x200f, 0x25, 0x32, 0x200f, 0x200e, 0x2d, 0x25, 0x31, 0xa0, 0x200f,
0x25, 0x32, 0x25, 0x32, 0xa0, 0x2d, 0x25, 0x31, 0x25, 0x32, 0x2212, 0x25,
0x31, 0x25, 0x32, 0x2d, 0xa0, 0x25, 0x31, 0x200e, 0x25, 0x32, 0xa0, 0x25,
0x31, 0x200e, 0x28, 0x25, 0x32, 0xa0, 0x25, 0x31, 0x29, 0x25, 0x31, 0x202f,
0x25, 0x32
};

static constexpr char16_t endonyms_data[] = {
0x410, 0x525, 0x441, 0x448, 0x4d9, 0x430, 0x49a, 0x44b, 0x440, 0x4ad, 0x442, 0x4d9,
0x44b, 0x43b, 0x430, 0x51, 0x61, 0x66, 0x61, 0x72, 0x4f, 0x74, 0x6f, 0x62,
0x62, 0x69, 0x61, 0x59, 0x61, 0x62, 0x75, 0x75, 0x74, 0x69, 0x45, 0x72,
0x65, 0x74, 0x72, 0x69, 0x61, 0x41, 0x66, 0x72, 0x69, 0x6b, 0x61, 0x61,
0x6e, 0x73, 0x53, 0x75, 0x69, 0x64, 0x2d, 0x41, 0x66, 0x72, 0x69, 0x6b,
0x61, 0x4e, 0x61, 0x6d, 0x69, 0x62, 0x69, 0xeb, 0x41, 0x67, 0x68, 0x65,
0x6d, 0x4b, 0xe0, 0x6d, 0xe0, 0x6c, 0xfb, 0x14b, 0x41, 0x6b, 0x61, 0x6e,
0x47, 0x61, 0x61, 0x6e, 0x61, 0x41, 0x6b, 0x6f, 0x6f, 0x73, 0x65, 0x4b,
0x61, 0x6d, 0x65, 0x72, 0xfb, 0x6e, 0x73, 0x68, 0x71, 0x69, 0x70, 0x53,
0x68, 0x71, 0x69, 0x70, 0xeb, 0x72, 0x69, 0x4b, 0x6f, 0x73, 0x6f, 0x76,
0xeb, 0x4d, 0x61, 0x71, 0x65, 0x64, 0x6f, 0x6e, 0x69, 0x61, 0x20, 0x65,
0x20, 0x56, 0x65, 0x72, 0x69, 0x75, 0x74, 0x12a0, 0x121b, 0x122d, 0x129b, 0x12a2,
0x1275, 0x12ee, 0x1335, 0x12eb, 0x627, 0x644, 0x639, 0x631, 0x628, 0x64a, 0x629, 0x645,
0x635, 0x631, 0x627, 0x644, 0x62c, 0x632, 0x627, 0x626, 0x631, 0x627, 0x644, 0x628,
0x62d, 0x631, 0x64a, 0x646, 0x62a, 0x634, 0x627, 0x62f, 0x62c, 0x632, 0x631, 0x20,
0x627, 0x644, 0x642, 0x645, 0x631, 0x62c, 0x64a, 0x628, 0x648, 0x62a, 0x64a, 0x625,
0x631, 0x64a, 0x62a, 0x631, 0x64a, 0x627, 0x627, 0x644, 0x639, 0x631, 0x627, 0x642,
0x625, 0x633, 0x631, 0x627, 0x626, 0x64a, 0x644, 0x627, 0x644, 0x623, 0x631, 0x62f,
0x646, 0x627, 0x644, 0x643, 0x648, 0x64a, 0x62a, 0x644, 0x628, 0x646, 0x627, 0x646,
0x644, 0x64a, 0x628, 0x64a, 0x627, 0x645, 0x648, 0x631, 0x64a, 0x62a, 0x627, 0x646,
0x64a, 0x627, 0x627, 0x644, 0x645, 0x63a, 0x631, 0x628, 0x639, 0x64f, 0x645, 0x627,
0x646, 0x627, 0x644, 0x623, 0x631, 0x627, 0x636, 0x64a, 0x20, 0x627, 0x644, 0x641,
0x644, 0x633, 0x637, 0x64a, 0x646, 0x64a, 0x629, 0x642, 0x637, 0x631, 0x627, 0x644,
0x645, 0x645, 0x644, 0x643, 0x629, 0x20, 0x627, 0x644, 0x639, 0x631, 0x628, 0x64a,
0x629, 0x20, 0x627, 0x644, 0x633, 0x639, 0x648, 0x62f, 0x64a, 0x629, 0x627, 0x644,
0x635, 0x648, 0x645, 0x627, 0x644, 0x62c, 0x646, 0x648, 0x628, 0x20, 0x627, 0x644,
0x633, 0x648, 0x62f, 0x627, 0x646, 0x627, 0x644, 0x633, 0x648, 0x62f, 0x627, 0x646,
0x633, 0x648, 0x631, 0x64a, 0x627, 0x62a, 0x648, 0x646, 0x633, 0x627, 0x644, 0x625,
0x645, 0x627, 0x631, 0x627, 0x62a, 0x20, 0x627, 0x644, 0x639, 0x631, 0x628, 0x64a,
0x629, 0x20, 0x627, 0x644, 0x645, 0x62a, 0x62d, 0x62f, 0x629, 0x627, 0x644, 0x635,
0x62d, 0x631, 0x627, 0x621, 0x20, 0x627, 0x644, 0x63a, 0x631, 0x628, 0x64a, 0x629,
0x627, 0x644, 0x639, 0x631, 0x628, 0x64a, 0x629, 0x20, 0x627, 0x644, 0x641, 0x635,
0x62d, 0x649, 0x20, 0x627, 0x644, 0x62d, 0x62f, 0x64a, 0x62b, 0x629, 0x627, 0x644,
0x639, 0x627, 0x644, 0x645, 0x627, 0x644, 0x64a, 0x645, 0x646, 0x61, 0x72, 0x61,
0x67, 0x6f, 0x6e, 0xe9, 0x73, 0x45, 0x73, 0x70, 0x61, 0x6e, 0x79, 0x61,
0x570, 0x561, 0x575, 0x565, 0x580, 0x565, 0x576, 0x540, 0x561, 0x575, 0x561, 0x57d,
0x57f, 0x561, 0x576, 0x985, 0x9b8, 0x9ae, 0x9c0, 0x9af, 0x9bc, 0x9be, 0x9ad, 0x9be,
0x9f0, 0x9a4, 0x61, 0x73, 0x74, 0x75, 0x72, 0x69, 0x61, 0x6e, 0x75, 0x45,
0x73, 0x70, 0x61, 0xf1, 0x61, 0x4b, 0x69, 0x70, 0x61, 0x72, 0x65, 0x54,
0x61, 0x64, 0x68, 0x61, 0x6e, 0x69, 0x61, 0x41, 0x74, 0x73, 0x61, 0x6d,
0x61, 0x7a, 0x259, 0x72, 0x62, 0x61, 0x79, 0x63, 0x61, 0x6e, 0x41, 0x7a,
0x259, 0x72, 0x62, 0x61, 0x79, 0x63, 0x61, 0x6e, 0x62a, 0x6c6, 0x631, 0x6a9,
0x62c, 0x647, 0x430, 0x437, 0x4d9, 0x440, 0x431, 0x430, 0x458, 0x4b9, 0x430, 0x43d,
0x410, 0x437, 0x4d9, 0x440, 0x431, 0x430, 0x458, 0x4b9, 0x430, 0x43d, 0x72, 0x69,
0x6b, 0x70, 0x61, 0x6b, 0x61, 0x6d, 0x25b, 0x72, 0xfa, 0x6e, 0x62, 0x61,
0x6d, 0x61, 0x6e, 0x61, 0x6b, 0x61, 0x6e, 0x4d, 0x61, 0x6c, 0x69, 0x9ac,
0x9be, 0x982, 0x9b2, 0x9be, 0x9ac, 0x9be, 0x982, 0x9b2, 0x9be, 0x9a6, 0x9c7, 0x9b6,
0x9ad, 0x9be, 0x9b0, 0x9a4, 0x181, 0xe0, 0x73, 0xe0, 0x61, 0x4b, 0xe0, 0x6d,
0x25b, 0x300, 0x72, 0xfb, 0x6e, 0x431, 0x430, 0x448, 0x4a1, 0x43e, 0x440, 0x442,
0x20, 0x442, 0x435, 0x43b, 0x435, 0x65, 0x75, 0x73, 0x6b, 0x61, 0x72, 0x61,
0x45, 0x73, 0x70, 0x61, 0x69, 0x6e, 0x69, 0x61, 0x431, 0x435, 0x43b, 0x430,
0x440, 0x443, 0x441, 0x43a, 0x430, 0x44f, 0x411, 0x435, 0x43b, 0x430, 0x440, 0x443,
0x441, 0x44c, 0x49, 0x63, 0x68, 0x69, 0x62, 0x65, 0x6d, 0x62, 0x61, 0x5a,
0x61, 0x6d, 0x62, 0x69, 0x61, 0x48, 0x69, 0x62, 0x65, 0x6e, 0x61, 0x48,
0x75, 0x74, 0x61, 0x6e, 0x7a, 0x61, 0x6e, 0x69, 0x61, 0x92d, 0x94b, 0x91c,
0x92a, 0x941, 0x930, 0x940, 0x92d, 0x93e, 0x930, 0x924, 0x1265, 0x120a, 0x1295, 0x12a4,
0x122d, 0x1275, 0x122b, 0x92c, 0x930, 0x2019, 0x62, 0x6f, 0x73, 0x61, 0x6e, 0x73,
0x6b, 0x69, 0x42, 0x6f, 0x73, 0x6e, 0x61, 0x20, 0x69, 0x20, 0x48, 0x65,
0x72, 0x63, 0x65, 0x67, 0x6f, 0x76, 0x69, 0x6e, 0x61, 0x431, 0x43e, 0x441,
0x430, 0x43d, 0x441, 0x43a, 0x438, 0x411, 0x43e, 0x441, 0x43d, 0x430, 0x20, 0x438,
0x20, 0x425, 0x435, 0x440, 0x446, 0x435, 0x433, 0x43e, 0x432, 0x438, 0x43d, 0x430,
0x62, 0x72, 0x65, 0x7a, 0x68, 0x6f, 0x6e, 0x65, 0x67, 0x46, 0x72, 0x61,
0xf1, 0x73, 0x431, 0x44a, 0x43b, 0x433, 0x430, 0x440, 0x441, 0x43a, 0x438, 0x411,
0x44a, 0x43b, 0x433, 0x430, 0x440, 0x438, 0x44f, 0x1019, 0x103c, 0x1014, 0x103a, 0x1019,
0x102c, 0x7cb5, 0x8a9e, 0x4e2d, 0x83ef, 0x4eba, 0x6c11, 0x5171, 0x548c, 0x570b, 0x9999, 0x6e2f,
0x7279, 0x5225, 0x884c, 0x653f, 0x5340, 0x7ca4, 0x8bed, 0x4e2d, 0x534e, 0x4eba, 0x6c11, 0x5171,
0x548c, 0x56fd, 0x63, 0x61, 0x74, 0x61, 0x6c, 0xe0, 0x41, 0x6e, 0x64, 0x6f,
0x72, 0x72, 0x61, 0x46, 0x72, 0x61, 0x6e, 0xe7, 0x61, 0x49, 0x74, 0xe0,
0x6c, 0x69, 0x61, 0x43, 0x65, 0x62, 0x75, 0x61, 0x6e, 0x6f, 0x50, 0x69,
0x6c, 0x69, 0x70, 0x69, 0x6e, 0x61, 0x73, 0x54, 0x61, 0x6d, 0x61, 0x7a,
0x69, 0x263, 0x74, 0x20, 0x6e, 0x20, 0x6c, 0x61, 0x1e6d, 0x6c, 0x61, 0x1e63,
0x4d, 0x65, 0x1e5b, 0x1e5b, 0x75, 0x6b, 0x6a9, 0x648, 0x631, 0x62f, 0x6cc, 0x6cc,
0x20, 0x646, 0x627, 0x648, 0x6d5, 0x646, 0x62f, 0x6cc, 0x639, 0x6ce, 0x631, 0x627,
0x642, 0x626, 0x6ce, 0x631, 0x627, 0x646, 0xd804, 0xdd0c, 0xd804, 0xdd0b, 0xd804, 0xdd34,
0xd804, 0xdd1f, 0xd804, 0xdd33, 0xd804, 0xdd26, 0xd804, 0xdd1d, 0xd804, 0xdd01, 0xd804, 0xdd23,
0xd804, 0xdd18, 0xd804, 0xdd2c, 0xd804, 0xdd0c, 0xd804, 0xdd34, 0xd804, 0xdd1e, 0xd804, 0xdd22,
0xd804, 0xdd27, 0xd804, 0xdd16, 0xd804, 0xdd34, 0x43d, 0x43e, 0x445, 0x447, 0x438, 0x439,
0x43d, 0x420, 0x43e, 0x441, 0x441, 0x438, 0x13e3, 0x13b3, 0x13a9, 0x13cc, 0x13ca, 0x20,
0x13a2, 0x13f3, 0x13be, 0x13b5, 0x13cd, 0x13d4, 0x13c5, 0x20, 0x13cd, 0x13a6, 0x13da, 0x13a9,
0x43, 0x68, 0x69, 0x6b, 0x61, 0x73, 0x68, 0x73, 0x68, 0x61, 0x6e, 0x6f,
0x6d, 0x70, 0x61, 0x2bc, 0x55, 0x6e, 0x69, 0x74, 0x65, 0x64, 0x20, 0x53,
0x74, 0x61, 0x74, 0x65, 0x73, 0x52, 0x75, 0x6b, 0x69, 0x67, 0x61, 0x55,
0x67, 0x61, 0x6e, 0x64, 0x61, 0x7b80, 0x4f53, 0x4e2d, 0x6587, 0x4e2d, 0x56fd, 0x4e2d,
0x56fd, 0x9999, 0x6e2f, 0x7279, 0x522b, 0x884c, 0x653f, 0x533a, 0x4e2d, 0x56fd, 0x6fb3, 0x95e8,
0x7279, 0x522b, 0x884c, 0x653f, 0x533a, 0x65b0, 0x52a0, 0x5761, 0x7e41, 0x9ad4, 0x4e2d, 0x6587,
0x4e2d, 0x570b, 0x9999, 0x6e2f, 0x7279, 0x5225, 0x884c, 0x653f, 0x5340, 0x4e2d, 0x570b, 0x6fb3,
0x9580, 0x7279, 0x5225, 0x884c, 0x653f, 0x5340, 0x53f0, 0x7063, 0x446, 0x435, 0x440, 0x43a,
0x43e, 0x432, 0x43d, 0x43e, 0x441, 0x43b, 0x43e, 0x432, 0x435, 0x301, 0x43d, 0x441,
0x43a, 0x457, 0x439, 0x440, 0x461, 0x441, 0x441, 0x456, 0x301, 0x430, 0x447, 0x4d1,
0x432, 0x430, 0x448, 0x420, 0x430, 0x4ab, 0x4ab, 0x435, 0x439, 0x4b, 0xf6, 0x6c,
0x73, 0x63, 0x68, 0x44, 0x6f, 0xfc, 0x74, 0x73, 0x63, 0x68, 0x6c, 0x61,
0x6e, 0x64, 0x6b, 0x65, 0x72, 0x6e, 0x65, 0x77, 0x65, 0x6b, 0x52, 0x79,
0x77, 0x76, 0x61, 0x6e, 0x65, 0x74, 0x68, 0x20, 0x55, 0x6e, 0x79, 0x73,
0x63, 0x6f, 0x72, 0x73, 0x75, 0x46, 0x72, 0x61, 0x6e, 0x63, 0x69, 0x61,
0x68, 0x72, 0x76, 0x61, 0x74, 0x73, 0x6b, 0x69, 0x48, 0x72, 0x76, 0x61,
0x74, 0x73, 0x6b, 0x61, 0x10d, 0x65, 0x161, 0x74, 0x69, 0x6e, 0x61, 0x10c,
0x65, 0x73, 0x6b, 0x6f, 0x64, 0x61, 0x6e, 0x73, 0x6b, 0x44, 0x61, 0x6e,
0x6d, 0x61, 0x72, 0x6b, 0x47, 0x72, 0xf8, 0x6e, 0x6c, 0x61, 0x6e, 0x64,
0x78b, 0x7a8, 0x788, 0x7ac, 0x780, 0x7a8, 0x784, 0x7a6, 0x790, 0x7b0, 0x78b, 0x7a8,
0x788, 0x7ac, 0x780, 0x7a8, 0x20, 0x783, 0x7a7, 0x787, 0x7b0, 0x796, 0x7ac, 0x921,
0x94b, 0x917, 0x930, 0x940, 0x64, 0x75, 0xe1, 0x6c, 0xe1, 0x43, 0x61, 0x6d,
0x65, 0x72, 0x6f, 0x75, 0x6e, 0x4e, 0x65, 0x64, 0x65, 0x72, 0x6c, 0x61,
0x6e, 0x64, 0x73, 0x41, 0x72, 0x75, 0x62, 0x61, 0x56, 0x6c, 0x61, 0x61,
0x6d, 0x73, 0x42, 0x65, 0x6c, 0x67, 0x69, 0xeb, 0x43, 0x61, 0x72, 0x69,
0x62, 0x69, 0x73, 0x63, 0x68, 0x20, 0x4e, 0x65, 0x64, 0x65, 0x72, 0x6c,
0x61, 0x6e, 0x64, 0x43, 0x75, 0x72, 0x61, 0xe7, 0x61, 0x6f, 0x53, 0x69,
0x6e, 0x74, 0x2d, 0x4d, 0x61, 0x61, 0x72, 0x74, 0x65, 0x6e, 0x53, 0x75,
0x72, 0x69, 0x6e, 0x61, 0x6d, 0x65, 0xf62, 0xfab, 0xf7c, 0xf44, 0xf0b, 0xf41,
0xf60, 0xf56, 0xfb2, 0xf74, 0xf42, 0x4b, 0x129, 0x65, 0x6d, 0x62, 0x75, 0x4b,
0x65, 0x6e, 0x79, 0x61, 0x41, 0x6d, 0x65, 0x72, 0x69, 0x63, 0x61, 0x6e,
0x20, 0x45, 0x6e, 0x67, 0x6c, 0x69, 0x73, 0x68, 0xd801, 0xdc00, 0xd801, 0xdc4d,
0xd801, 0xdc4a, 0xd801, 0xdc2e, 0xd801, 0xdc47, 0xd801, 0xdc0f, 0xd801, 0xdc2d, 0xd801, 0xdc4c,
0xd801, 0xdc34, 0xd801, 0xdc3b, 0xd801, 0xdc32, 0xd801, 0xdc3c, 0x20, 0xd801, 0xdc1d, 0xd801,
0xdc3b, 0xd801, 0xdc29, 0xd801, 0xdc3b, 0xd801, 0xdc45, 0x41, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x6e, 0x20, 0x53, 0x61, 0x6d, 0x6f, 0x61, 0x41, 0x6e, 0x67,
0x75, 0x69, 0x6c, 0x6c, 0x61, 0x41, 0x6e, 0x74, 0x69, 0x67, 0x75, 0x61,
0x20, 0x26, 0x20, 0x42, 0x61, 0x72, 0x62, 0x75, 0x64, 0x61, 0x41, 0x75,
0x73, 0x74, 0x72, 0x61, 0x6c, 0x69, 0x61, 0x6e, 0x20, 0x45, 0x6e, 0x67,
0x6c, 0x69, 0x73, 0x68, 0x41, 0x75, 0x73, 0x74, 0x72, 0x69, 0x61, 0x42,
0x61, 0x68, 0x61, 0x6d, 0x61, 0x73, 0x42, 0x61, 0x72, 0x62, 0x61, 0x64,
0x6f, 0x73, 0x42, 0x65, 0x6c, 0x67, 0x69, 0x75, 0x6d, 0x42, 0x65, 0x6c,
0x69, 0x7a, 0x65, 0x42, 0x65, 0x72, 0x6d, 0x75, 0x64, 0x61, 0x42, 0x6f,
0x74, 0x73, 0x77, 0x61, 0x6e, 0x61, 0x42, 0x72, 0x69, 0x74, 0x69, 0x73,
0x68, 0x20, 0x49, 0x6e, 0x64, 0x69, 0x61, 0x6e, 0x20, 0x4f, 0x63, 0x65,
0x61, 0x6e, 0x20, 0x54, 0x65, 0x72, 0x72, 0x69, 0x74, 0x6f, 0x72, 0x79,
0x42, 0x72, 0x69, 0x74, 0x69, 0x73, 0x68, 0x20, 0x56, 0x69, 0x72, 0x67,
0x69, 0x6e, 0x20, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x42, 0x75,
0x72, 0x75, 0x6e, 0x64, 0x69, 0x43, 0x61, 0x6d, 0x65, 0x72, 0x6f, 0x6f,
0x6e, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x69, 0x61, 0x6e, 0x20, 0x45, 0x6e,
0x67, 0x6c, 0x69, 0x73, 0x68, 0x43, 0x61, 0x6e, 0x61, 0x64, 0x61, 0x43,
0x61, 0x79, 0x6d, 0x61, 0x6e, 0x20, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64,
0x73, 0x43, 0x68, 0x72, 0x69, 0x73, 0x74, 0x6d, 0x61, 0x73, 0x20, 0x49,
0x73, 0x6c, 0x61, 0x6e, 0x64, 0x43, 0x6f, 0x63, 0x6f, 0x73, 0x20, 0x28,
0x4b, 0x65, 0x65, 0x6c, 0x69, 0x6e, 0x67, 0x29, 0x20, 0x49, 0x73, 0x6c,
0x61, 0x6e, 0x64, 0x73, 0x43, 0x6f, 0x6f, 0x6b, 0x20, 0x49, 0x73, 0x6c,
0x61, 0x6e, 0x64, 0x73, 0x43, 0x79, 0x70, 0x72, 0x75, 0x73, 0x44, 0x65,
0x6e, 0x6d, 0x61, 0x72, 0x6b, 0x44, 0x69, 0x65, 0x67, 0x6f, 0x20, 0x47,
0x61, 0x72, 0x63, 0x69, 0x61, 0x44, 0x6f, 0x6d, 0x69, 0x6e, 0x69, 0x63,
0x61, 0x45, 0x72, 0x69, 0x74, 0x72, 0x65, 0x61, 0x45, 0x73, 0x77, 0x61,
0x74, 0x69, 0x6e, 0x69, 0x45, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x46, 0x61,
0x6c, 0x6b, 0x6c, 0x61, 0x6e, 0x64, 0x20, 0x49, 0x73, 0x6c, 0x61, 0x6e,
0x64, 0x73, 0x46, 0x69, 0x6a, 0x69, 0x46, 0x69, 0x6e, 0x6c, 0x61, 0x6e,
0x64, 0x47, 0x61, 0x6d, 0x62, 0x69, 0x61, 0x47, 0x65, 0x72, 0x6d, 0x61,
0x6e, 0x79, 0x47, 0x68, 0x61, 0x6e, 0x61, 0x47, 0x69, 0x62, 0x72, 0x61,
0x6c, 0x74, 0x61, 0x72, 0x47, 0x72, 0x65, 0x6e, 0x61, 0x64, 0x61, 0x47,
0x75, 0x61, 0x6d, 0x47, 0x75, 0x65, 0x72, 0x6e, 0x73, 0x65, 0x79, 0x47,
0x75, 0x79, 0x61, 0x6e, 0x61, 0x48, 0x6f, 0x6e, 0x67, 0x20, 0x4b, 0x6f,
0x6e, 0x67, 0x20, 0x53, 0x41, 0x52, 0x20, 0x43, 0x68, 0x69, 0x6e, 0x61,
0x49, 0x6e, 0x64, 0x6f, 0x6e, 0x65, 0x73, 0x69, 0x61, 0x49, 0x72, 0x65,
0x6c, 0x61, 0x6e, 0x64, 0x49, 0x73, 0x6c, 0x65, 0x20, 0x6f, 0x66, 0x20,
0x4d, 0x61, 0x6e, 0x49, 0x73, 0x72, 0x61, 0x65, 0x6c, 0x4a, 0x61, 0x6d,
0x61, 0x69, 0x63, 0x61, 0x4a, 0x65, 0x72, 0x73, 0x65, 0x79, 0x4b, 0x69,
0x72, 0x69, 0x62, 0x61, 0x74, 0x69, 0x4c, 0x65, 0x73, 0x6f, 0x74, 0x68,
0x6f, 0x4c, 0x69, 0x62, 0x65, 0x72, 0x69, 0x61, 0x4d, 0x61, 0x63, 0x61,
0x6f, 0x20, 0x53, 0x41, 0x52, 0x20, 0x43, 0x68, 0x69, 0x6e, 0x61, 0x4d,
0x61, 0x64, 0x61, 0x67, 0x61, 0x73, 0x63, 0x61, 0x72, 0x4d, 0x61, 0x6c,
0x61, 0x77, 0x69, 0x4d, 0x61, 0x6c, 0x61, 0x79, 0x73, 0x69, 0x61, 0x4d,
0x61, 0x6c, 0x64, 0x69, 0x76, 0x65, 0x73, 0x4d, 0x61, 0x6c, 0x74, 0x61,
0x4d, 0x61, 0x72, 0x73, 0x68, 0x61, 0x6c, 0x6c, 0x20, 0x49, 0x73, 0x6c,
0x61, 0x6e, 0x64, 0x73, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x74, 0x69, 0x75,
0x73, 0x4d, 0x69, 0x63, 0x72, 0x6f, 0x6e, 0x65, 0x73, 0x69, 0x61, 0x4d,
0x6f, 0x6e, 0x74, 0x73, 0x65, 0x72, 0x72, 0x61, 0x74, 0x4e, 0x61, 0x6d,
0x69, 0x62, 0x69, 0x61, 0x4e, 0x61, 0x75, 0x72, 0x75, 0x4e, 0x65, 0x74,
0x68, 0x65, 0x72, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x4e, 0x65, 0x77, 0x20,
0x5a, 0x65, 0x61, 0x6c, 0x61, 0x6e, 0x64, 0x4e, 0x69, 0x67, 0x65, 0x72,
0x69, 0x61, 0x4e, 0x69, 0x75, 0x65, 0x4e, 0x6f, 0x72, 0x66, 0x6f, 0x6c,
0x6b, 0x20, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x4e, 0x6f, 0x72, 0x74,
0x68, 0x65, 0x72, 0x6e, 0x20, 0x4d, 0x61, 0x72, 0x69, 0x61, 0x6e, 0x61,
0x20, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x50, 0x61, 0x6b, 0x69,
0x73, 0x74, 0x61, 0x6e, 0x50, 0x61, 0x6c, 0x61, 0x75, 0x50, 0x61, 0x70,
0x75, 0x61, 0x20, 0x4e, 0x65, 0x77, 0x20, 0x47, 0x75, 0x69, 0x6e, 0x65,
0x61, 0x50, 0x68, 0x69, 0x6c, 0x69, 0x70, 0x70, 0x69, 0x6e, 0x65, 0x73,
0x50, 0x69, 0x74, 0x63, 0x61, 0x69, 0x72, 0x6e, 0x20, 0x49, 0x73, 0x6c,
0x61, 0x6e, 0x64, 0x73, 0x50, 0x75, 0x65, 0x72, 0x74, 0x6f, 0x20, 0x52,
0x69, 0x63, 0x6f, 0x52, 0x77, 0x61, 0x6e, 0x64, 0x61, 0x53, 0x74, 0x20,
0x48, 0x65, 0x6c, 0x65, 0x6e, 0x61, 0x53, 0x74, 0x20, 0x4b, 0x69, 0x74,
0x74, 0x73, 0x20, 0x26, 0x20, 0x4e, 0x65, 0x76, 0x69, 0x73, 0x53, 0x74,
0x20, 0x4c, 0x75, 0x63, 0x69, 0x61, 0x53, 0x74, 0x20, 0x56, 0x69, 0x6e,
0x63, 0x65, 0x6e, 0x74, 0x20, 0x26, 0x20, 0x74, 0x68, 0x65, 0x20, 0x47,
0x72, 0x65, 0x6e, 0x61, 0x64, 0x69, 0x6e, 0x65, 0x73, 0x53, 0x65, 0x79,
0x63, 0x68, 0x65, 0x6c, 0x6c, 0x65, 0x73, 0x53, 0x69, 0x65, 0x72, 0x72,
0x61, 0x20, 0x4c, 0x65, 0x6f, 0x6e, 0x65, 0x53, 0x69, 0x6e, 0x67, 0x61,
0x70, 0x6f, 0x72, 0x65, 0x53, 0x69, 0x6e, 0x74, 0x20, 0x4d, 0x61, 0x61,
0x72, 0x74, 0x65, 0x6e, 0x53, 0x6c, 0x6f, 0x76, 0x65, 0x6e, 0x69, 0x61,
0x53, 0x6f, 0x6c, 0x6f, 0x6d, 0x6f, 0x6e, 0x20, 0x49, 0x73, 0x6c, 0x61,
0x6e, 0x64, 0x73, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x20, 0x41, 0x66, 0x72,
0x69, 0x63, 0x61, 0x53, 0x6f, 0x75, 0x74, 0x68, 0x20, 0x53, 0x75, 0x64,
0x61, 0x6e, 0x53, 0x75, 0x64, 0x61, 0x6e, 0x53, 0x77, 0x65, 0x64, 0x65,
0x6e, 0x53, 0x77, 0x69, 0x74, 0x7a, 0x65, 0x72, 0x6c, 0x61, 0x6e, 0x64,
0x54, 0x61, 0x6e, 0x7a, 0x61, 0x6e, 0x69, 0x61, 0x54, 0x6f, 0x6b, 0x65,
0x6c, 0x61, 0x75, 0x54, 0x6f, 0x6e, 0x67, 0x61, 0x54, 0x72, 0x69, 0x6e,
0x69, 0x64, 0x61, 0x64, 0x20, 0x26, 0x20, 0x54, 0x6f, 0x62, 0x61, 0x67,
0x6f, 0x54, 0x75, 0x72, 0x6b, 0x73, 0x20, 0x26, 0x20, 0x43, 0x61, 0x69,
0x63, 0x6f, 0x73, 0x20, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x54,
0x75, 0x76, 0x61, 0x6c, 0x75, 0x55, 0x6e, 0x69, 0x74, 0x65, 0x64, 0x20,
0x41, 0x72, 0x61, 0x62, 0x20, 0x45, 0x6d, 0x69, 0x72, 0x61, 0x74, 0x65,
0x73, 0x42, 0x72, 0x69, 0x74, 0x69, 0x73, 0x68, 0x20, 0x45, 0x6e, 0x67,
0x6c, 0x69, 0x73, 0x68, 0x55, 0x6e, 0x69, 0x74, 0x65, 0x64, 0x20, 0x4b,
0x69, 0x6e, 0x67, 0x64, 0x6f, 0x6d, 0x55, 0x2e, 0x53, 0x2e, 0x20, 0x4f,
0x75, 0x74, 0x6c, 0x79, 0x69, 0x6e, 0x67, 0x20, 0x49, 0x73, 0x6c, 0x61,
0x6e, 0x64, 0x73, 0x55, 0x2e, 0x53, 0x2e, 0x20, 0x56, 0x69, 0x72, 0x67,
0x69, 0x6e, 0x20, 0x49, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x73, 0x56, 0x61,
0x6e, 0x75, 0x61, 0x74, 0x75, 0x77, 0x6f, 0x72, 0x6c, 0x64, 0x5a, 0x69,
0x6d, 0x62, 0x61, 0x62, 0x77, 0x65, 0x44d, 0x440, 0x437, 0x44f, 0x43d, 0x44c,
0x20, 0x43a, 0x435, 0x43b, 0x44c, 0x420, 0x443, 0x437, 0x43e, 0x43d, 0x44c, 0x20,
0x43c, 0x430, 0x441, 0x442, 0x43e, 0x440, 0x45, 0x73, 0x70, 0x65, 0x72, 0x61,
0x6e, 0x74, 0x6f, 0x6d, 0x6f, 0x6e, 0x64, 0x6f, 0x65, 0x65, 0x73, 0x74,
0x69, 0x45, 0x65, 0x73, 0x74, 0x69, 0x45, 0x28b, 0x65, 0x67, 0x62, 0x65,
0x47, 0x68, 0x61, 0x6e, 0x61, 0x20, 0x6e, 0x75, 0x74, 0x6f, 0x6d, 0x65,
0x54, 0x6f, 0x67, 0x6f, 0x20, 0x6e, 0x75, 0x74, 0x6f, 0x6d, 0x65, 0x65,
0x77, 0x6f, 0x6e, 0x64, 0x6f, 0x4b, 0x61, 0x6d, 0x259, 0x72, 0xfa, 0x6e,
0x66, 0xf8, 0x72, 0x6f, 0x79, 0x73, 0x6b, 0x74, 0x46, 0xf8, 0x72, 0x6f,
0x79, 0x61, 0x72, 0x46, 0x69, 0x6c, 0x69, 0x70, 0x69, 0x6e, 0x6f, 0x73,
0x75, 0x6f, 0x6d, 0x69, 0x53, 0x75, 0x6f, 0x6d, 0x69, 0x66, 0x72, 0x61,
0x6e, 0xe7, 0x61, 0x69, 0x73, 0x46, 0x72, 0x61, 0x6e, 0x63, 0x65, 0x41,
0x6c, 0x67, 0xe9, 0x72, 0x69, 0x65, 0x42, 0x65, 0x6c, 0x67, 0x69, 0x71,
0x75, 0x65, 0x42, 0xe9, 0x6e, 0x69, 0x6e, 0x42, 0x75, 0x72, 0x6b, 0x69,
0x6e, 0x61, 0x20, 0x46, 0x61, 0x73, 0x6f, 0x66, 0x72, 0x61, 0x6e, 0xe7,
0x61, 0x69, 0x73, 0x20, 0x63, 0x61, 0x6e, 0x61, 0x64, 0x69, 0x65, 0x6e,
0x52, 0xe9, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x71, 0x75, 0x65, 0x20, 0x63,
0x65, 0x6e, 0x74, 0x72, 0x61, 0x66, 0x72, 0x69, 0x63, 0x61, 0x69, 0x6e,
0x65, 0x54, 0x63, 0x68, 0x61, 0x64, 0x43, 0x6f, 0x6d, 0x6f, 0x72, 0x65,
0x73, 0x43, 0x6f, 0x6e, 0x67, 0x6f, 0x2d, 0x42, 0x72, 0x61, 0x7a, 0x7a,
0x61, 0x76, 0x69, 0x6c, 0x6c, 0x65, 0x43, 0x6f, 0x6e, 0x67, 0x6f, 0x2d,
0x4b, 0x69, 0x6e, 0x73, 0x68, 0x61, 0x73, 0x61, 0x44, 0x6a, 0x69, 0x62,
0x6f, 0x75, 0x74, 0x69, 0x47, 0x75, 0x69, 0x6e, 0xe9, 0x65, 0x20, 0xe9,
0x71, 0x75, 0x61, 0x74, 0x6f, 0x72, 0x69, 0x61, 0x6c, 0x65, 0x47, 0x75,
0x79, 0x61, 0x6e, 0x65, 0x20, 0x66, 0x72, 0x61, 0x6e, 0xe7, 0x61, 0x69,
0x73, 0x65, 0x50, 0x6f, 0x6c, 0x79, 0x6e, 0xe9, 0x73, 0x69, 0x65, 0x20,
0x66, 0x72, 0x61, 0x6e, 0xe7, 0x61, 0x69, 0x73, 0x65, 0x47, 0x61, 0x62,
0x6f, 0x6e, 0x47, 0x75, 0x61, 0x64, 0x65, 0x6c, 0x6f, 0x75, 0x70, 0x65,
0x48, 0x61, 0xef, 0x74, 0x69, 0x43, 0xf4, 0x74, 0x65, 0x20, 0x64, 0x2019,
0x49, 0x76, 0x6f, 0x69, 0x72, 0x65, 0x4c, 0x75, 0x78, 0x65, 0x6d, 0x62,
0x6f, 0x75, 0x72, 0x67, 0x4d, 0x61, 0x72, 0x74, 0x69, 0x6e, 0x69, 0x71,
0x75, 0x65, 0x4d, 0x61, 0x75, 0x72, 0x69, 0x74, 0x61, 0x6e, 0x69, 0x65,
0x4d, 0x61, 0x75, 0x72, 0x69, 0x63, 0x65, 0x4d, 0x61, 0x79, 0x6f, 0x74,
0x74, 0x65, 0x4d, 0x6f, 0x6e, 0x61, 0x63, 0x6f, 0x4d, 0x61, 0x72, 0x6f,
0x63, 0x4e, 0x6f, 0x75, 0x76, 0x65, 0x6c, 0x6c, 0x65, 0x2d, 0x43, 0x61,
0x6c, 0xe9, 0x64, 0x6f, 0x6e, 0x69, 0x65, 0x4c, 0x61, 0x20, 0x52, 0xe9,
0x75, 0x6e, 0x69, 0x6f, 0x6e, 0x53, 0x61, 0x69, 0x6e, 0x74, 0x2d, 0x42,
0x61, 0x72, 0x74, 0x68, 0xe9, 0x6c, 0x65, 0x6d, 0x79, 0x53, 0x61, 0x69,
0x6e, 0x74, 0x2d, 0x4d, 0x61, 0x72, 0x74, 0x69, 0x6e, 0x53, 0x61, 0x69,
0x6e, 0x74, 0x2d, 0x50, 0x69, 0x65, 0x72, 0x72, 0x65, 0x2d, 0x65, 0x74,
0x2d, 0x4d, 0x69, 0x71, 0x75, 0x65, 0x6c, 0x6f, 0x6e, 0x53, 0xe9, 0x6e,
0xe9, 0x67, 0x61, 0x6c, 0x66, 0x72, 0x61, 0x6e, 0xe7, 0x61, 0x69, 0x73,
0x20, 0x73, 0x75, 0x69, 0x73, 0x73, 0x65, 0x53, 0x75, 0x69, 0x73, 0x73,
0x65, 0x53, 0x79, 0x72, 0x69, 0x65, 0x54, 0x75, 0x6e, 0x69, 0x73, 0x69,
0x65, 0x57, 0x61, 0x6c, 0x6c, 0x69, 0x73, 0x2d, 0x65, 0x74, 0x2d, 0x46,
0x75, 0x74, 0x75, 0x6e, 0x61, 0x66, 0x75, 0x72, 0x6c, 0x61, 0x6e, 0x49,
0x74, 0x61, 0x6c, 0x69, 0x65, 0x50, 0x75, 0x6c, 0x61, 0x61, 0x72, 0x53,
0x65, 0x6e, 0x65, 0x67, 0x61, 0x61, 0x6c, 0xd83a, 0xdd06, 0xd83a, 0xdd35, 0xd83a,
0xdd24, 0xd83a, 0xdd22, 0xd83a, 0xdd2a, 0xd83a, 0xdd04, 0xd83a, 0xdd35, 0xd83a, 0xdd2a, 0xd83a,
0xdd33, 0xd83a, 0xdd2d, 0xd83a, 0xdd32, 0xd83a, 0xdd22, 0x20, 0xd83a, 0xdd0a, 0xd83a, 0xdd22,
0xd83a, 0xdd27, 0xd83a, 0xdd2e, 0xd83a, 0xdd45, 0xd83a, 0xdd11, 0xd83a, 0xdd22, 0xd83a, 0xdd25,
0xd83a, 0xdd22, 0xd83a, 0xdd2a, 0xd83a, 0xdd35, 0xd83a, 0xdd45, 0xd83a, 0xdd32, 0xd83a, 0xdd18,
0xd83a, 0xdd22, 0xd83a, 0xdd25, 0xd83a, 0xdd26, 0xd83a, 0xdd2d, 0xd83a, 0xdd34, 0xd83a, 0xdd22,
0xd83a, 0xdd18, 0xd83a, 0xdd22, 0xd83a, 0xdd32, 0xd83a, 0xdd22, 0xd83a, 0xdd18, 0xd83a, 0xdd2d,
0xd83a, 0xdd32, 0xd83a, 0xdd2b, 0x2d, 0xd83a, 0xdd04, 0xd83a, 0xdd2d, 0xd83a, 0xdd27, 0xd83a,
0xdd22, 0xd83a, 0xdd31, 0xd83a, 0xdd2e, 0xd83a, 0xdd45, 0xd83a, 0xdd02, 0xd83a, 0xdd22, 0xd83a,
0xdd26, 0xd83a, 0xdd2d, 0xd83a, 0xdd2a, 0xd83a, 0xdd2d, 0xd83a, 0xdd34, 0xd83a, 0xdd22, 0xd83a,
0xdd44, 0xd83a, 0xdd03, 0xd83a, 0xdd2e, 0xd83a, 0xdd2a, 0xd83a, 0xdd3c, 0xd83a, 0xdd22, 0xd83a,
0xdd32, 0xd83a, 0xdd2d, 0xd83a, 0xdd45, 0xd83a, 0xdd10, 0xd83a, 0xdd22, 0xd83a, 0xdd36, 0xd83a,
0xdd2b, 0xd83a, 0xdd2a, 0xd83a, 0xdd2d, 0xd83a, 0xdd34, 0xd83a, 0xdd22, 0xd83a, 0xdd44, 0xd83a,
0xdd10, 0xd83a, 0xdd2d, 0xd83a, 0xdd45, 0xd83a, 0xdd36, 0xd83a, 0xdd2b, 0xd83a, 0xdd2a, 0xd83a,
0xdd05, 0xd83a, 0xdd2b, 0xd83a, 0xdd32, 0xd83a, 0xdd2b, 0xd83a, 0xdd3a, 0xd83a, 0xdd22, 0xd83a,
0xdd44, 0xd83a, 0xdd24, 0xd83a, 0xdd05, 0xd83a, 0xdd22, 0xd83a, 0xdd2a, 0xd83a, 0xdd22, 0xd83a,
0xdd24, 0xd83a, 0xdd2e, 0xd83a, 0xdd32, 0x42, 0x75, 0x72, 0x6b, 0x69, 0x62, 0x61,
0x61, 0x20, 0x46, 0x61, 0x61, 0x73, 0x6f, 0x4b, 0x61, 0x6d, 0x65, 0x72,
0x75, 0x75, 0x6e, 0x47, 0x61, 0x6d, 0x6d, 0x62, 0x69, 0x47, 0x61, 0x6e,
0x61, 0x61, 0x47, 0x69, 0x6e, 0x65, 0x2d, 0x42, 0x69, 0x73, 0x61, 0x61,
0x77, 0x6f, 0x4c, 0x69, 0x62, 0x65, 0x72, 0x69, 0x79, 0x61, 0x61, 0x4d,
0x75, 0x72, 0x69, 0x74, 0x61, 0x6e, 0x69, 0x4e, 0x69, 0x6a, 0x65, 0x72,
0x69, 0x79, 0x61, 0x61, 0x4e, 0x69, 0x6a, 0x65, 0x65, 0x72, 0x53, 0x65,
0x72, 0x61, 0x61, 0x20, 0x6c, 0x69, 0x79, 0x6f, 0x6e, 0x47, 0xe0, 0x69,
0x64, 0x68, 0x6c, 0x69, 0x67, 0x41, 0x6e, 0x20, 0x52, 0xec, 0x6f, 0x67,
0x68, 0x61, 0x63, 0x68, 0x64, 0x20, 0x41, 0x6f, 0x6e, 0x61, 0x69, 0x63,
0x68, 0x74, 0x65, 0x47, 0xe3, 0x67, 0x61, 0x6c, 0x65, 0x67, 0x6f, 0x4c,
0x75, 0x67, 0x61, 0x6e, 0x64, 0x61, 0x59, 0x75, 0x67, 0x61, 0x6e, 0x64,
0x61, 0x130d, 0x12d5, 0x12dd, 0x129b, 0x10e5, 0x10d0, 0x10e0, 0x10d7, 0x10e3, 0x10da, 0x10d8,
0x10e1, 0x10d0, 0x10e5, 0x10d0, 0x10e0, 0x10d7, 0x10d5, 0x10d4, 0x10da, 0x10dd, 0x44, 0x65,
0x75, 0x74, 0x73, 0x63, 0x68, 0x44, 0x65, 0x75, 0x74, 0x73, 0x63, 0x68,
0x6c, 0x61, 0x6e, 0x64, 0xd6, 0x73, 0x74, 0x65, 0x72, 0x72, 0x65, 0x69,
0x63, 0x68, 0x69, 0x73, 0x63, 0x68, 0x65, 0x73, 0x20, 0x44, 0x65, 0x75,
0x74, 0x73, 0x63, 0x68, 0x42, 0x65, 0x6c, 0x67, 0x69, 0x65, 0x6e, 0x49,
0x74, 0x61, 0x6c, 0x69, 0x65, 0x6e, 0x4c, 0x69, 0x65, 0x63, 0x68, 0x74,
0x65, 0x6e, 0x73, 0x74, 0x65, 0x69, 0x6e, 0x4c, 0x75, 0x78, 0x65, 0x6d,
0x62, 0x75, 0x72, 0x67, 0x53, 0x63, 0x68, 0x77, 0x65, 0x69, 0x7a, 0x65,
0x72, 0x20, 0x48, 0x6f, 0x63, 0x68, 0x64, 0x65, 0x75, 0x74, 0x73, 0x63,
0x68, 0x395, 0x3bb, 0x3bb, 0x3b7, 0x3bd, 0x3b9, 0x3ba, 0x3ac, 0x395, 0x3bb, 0x3bb,
0x3ac, 0x3b4, 0x3b1, 0x39a, 0x3cd, 0x3c0, 0x3c1, 0x3bf, 0x3c2, 0x61, 0x76, 0x61,
0xf1, 0x65, 0x2019, 0x1ebd, 0x50, 0x61, 0x72, 0x61, 0x67, 0x75, 0x61, 0x69,
0xa97, 0xac1, 0xa9c, 0xab0, 0xabe, 0xaa4, 0xac0, 0xaad, 0xabe, 0xab0, 0xaa4, 0x45,
0x6b, 0x65, 0x67, 0x75, 0x73, 0x69, 0x69, 0x48, 0x61, 0x75, 0x73, 0x61,
0x4e, 0x69, 0x6a, 0x61, 0x72, 0x2bb, 0x14c, 0x6c, 0x65, 0x6c, 0x6f, 0x20,
0x48, 0x61, 0x77, 0x61, 0x69, 0x2bb, 0x69, 0x2bb, 0x41, 0x6d, 0x65, 0x6c,
0x69, 0x6b, 0x61, 0x20, 0x48, 0x75, 0x69, 0x20, 0x50, 0x16b, 0x20, 0x2bb,
0x49, 0x61, 0x5e2, 0x5d1, 0x5e8, 0x5d9, 0x5ea, 0x5d9, 0x5e9, 0x5e8, 0x5d0, 0x5dc,
0x939, 0x93f, 0x928, 0x94d, 0x926, 0x940, 0x48, 0x69, 0x6e, 0x64, 0x69, 0x20,
0x28, 0x4c, 0x61, 0x74, 0x69, 0x6e, 0x29, 0x42, 0x68, 0x61, 0x72, 0x61,
0x74, 0x6d, 0x61, 0x67, 0x79, 0x61, 0x72, 0x4d, 0x61, 0x67, 0x79, 0x61,
0x72, 0x6f, 0x72, 0x73, 0x7a, 0xe1, 0x67, 0xed, 0x73, 0x6c, 0x65, 0x6e,
0x73, 0x6b, 0x61, 0xcd, 0x73, 0x6c, 0x61, 0x6e, 0x64, 0x49, 0x64, 0x6f,
0x49, 0x67, 0x62, 0x6f, 0x4e, 0x61, 0x1ecb, 0x6a, 0x1ecb, 0x72, 0x1ecb, 0x61,
0x61, 0x6e, 0x61, 0x72, 0xe2, 0x161, 0x6b, 0x69, 0x65, 0x6c, 0xe2, 0x53,
0x75, 0x6f, 0x6d, 0xe2, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x6c, 0x69, 0x6e,
0x67, 0x75, 0x61, 0x4d, 0x75, 0x6e, 0x64, 0x6f, 0x49, 0x6e, 0x74, 0x65,
0x72, 0x6c, 0x69, 0x6e, 0x67, 0x75, 0x65, 0x45, 0x73, 0x74, 0x6f, 0x6e,
0x69, 0x61, 0x1403, 0x14c4, 0x1483, 0x144e, 0x1450, 0x1466, 0x1472, 0x14c7, 0x1455, 0x14a5,
0x47, 0x61, 0x65, 0x69, 0x6c, 0x67, 0x65, 0xc9, 0x69, 0x72, 0x65, 0x61,
0x6e, 0x20, 0x52, 0xed, 0x6f, 0x63, 0x68, 0x74, 0x20, 0x41, 0x6f, 0x6e,
0x74, 0x61, 0x69, 0x74, 0x68, 0x65, 0x69, 0x74, 0x61, 0x6c, 0x69, 0x61,
0x6e, 0x6f, 0x49, 0x74, 0x61, 0x6c, 0x69, 0x61, 0x53, 0x61, 0x6e, 0x20,
0x4d, 0x61, 0x72, 0x69, 0x6e, 0x6f, 0x53, 0x76, 0x69, 0x7a, 0x7a, 0x65,
0x72, 0x61, 0x43, 0x69, 0x74, 0x74, 0xe0, 0x20, 0x64, 0x65, 0x6c, 0x20,
0x56, 0x61, 0x74, 0x69, 0x63, 0x61, 0x6e, 0x6f, 0x65e5, 0x672c, 0x8a9e, 0x4a,
0x61, 0x77, 0x61, 0x49, 0x6e, 0x64, 0x6f, 0x6e, 0xe9, 0x73, 0x69, 0x61,
0x4b, 0x61, 0x6a, 0x65, 0x6a, 0x6f, 0x6f, 0x6c, 0x61, 0x53, 0x65, 0x6e,
0x65, 0x67, 0x61, 0x6c, 0x6b, 0x61, 0x62, 0x75, 0x76, 0x65, 0x72, 0x64,
0x69, 0x61, 0x6e, 0x75, 0x4b, 0x61, 0x62, 0x75, 0x20, 0x56, 0x65, 0x72,
0x64, 0x69, 0x54, 0x61, 0x71, 0x62, 0x61, 0x79, 0x6c, 0x69, 0x74, 0x4c,
0x65, 0x7a, 0x7a, 0x61, 0x79, 0x65, 0x72, 0x6b, 0x61, 0x6b, 0x254, 0x4b,
0x61, 0x6d, 0x25b, 0x72, 0x75, 0x6e, 0x6b, 0x61, 0x6c, 0x61, 0x61, 0x6c,
0x6c, 0x69, 0x73, 0x75, 0x74, 0x4b, 0x61, 0x6c, 0x61, 0x61, 0x6c, 0x6c,
0x69, 0x74, 0x20, 0x4e, 0x75, 0x6e, 0x61, 0x61, 0x74, 0x4b, 0x61, 0x6c,
0x65, 0x6e, 0x6a, 0x69, 0x6e, 0x45, 0x6d, 0x65, 0x74, 0x61, 0x62, 0x20,
0x4b, 0x65, 0x6e, 0x79, 0x61, 0x4b, 0x69, 0x6b, 0x61, 0x6d, 0x62, 0x61,
0xc95, 0xca8, 0xccd, 0xca8, 0xca1, 0xcad, 0xcbe, 0xcb0, 0xca4, 0x6a9, 0x672, 0x634,
0x64f, 0x631, 0x6c1, 0x650, 0x646, 0x62f, 0x648, 0x633, 0x62a, 0x627, 0x646, 0x915,
0x949, 0x936, 0x941, 0x930, 0x939, 0x93f, 0x902, 0x926, 0x94b, 0x938, 0x94d, 0x924,
0x93e, 0x928, 0x49b, 0x430, 0x437, 0x430, 0x49b, 0x20, 0x442, 0x456, 0x43b, 0x456,
0x49a, 0x430, 0x437, 0x430, 0x49b, 0x441, 0x442, 0x430, 0x43d, 0x4b, 0x25b, 0x6e,
0x79, 0x61, 0x14b, 0x1781, 0x17d2, 0x1798, 0x17c2, 0x179a, 0x1780, 0x1798, 0x17d2, 0x1796,
0x17bb, 0x1787, 0x17b6, 0x4b, 0x2bc, 0x69, 0x63, 0x68, 0x65, 0x2bc, 0x47, 0x69,
0x6b, 0x75, 0x79, 0x75, 0x4b, 0x69, 0x6e, 0x79, 0x61, 0x72, 0x77, 0x61,
0x6e, 0x64, 0x61, 0x55, 0x20, 0x52, 0x77, 0x61, 0x6e, 0x64, 0x61, 0x915,
0x94b, 0x902, 0x915, 0x923, 0x940, 0xd55c, 0xad6d, 0xc5b4, 0xb300, 0xd55c, 0xbbfc, 0xad6d,
0xc911, 0xad6d, 0xc870, 0xc120, 0xbbfc, 0xc8fc, 0xc8fc, 0xc758, 0xc778, 0xbbfc, 0xacf5, 0xd654,
0xad6d, 0x4b, 0x6f, 0x79, 0x72, 0x61, 0x62, 0x6f, 0x72, 0x6f, 0x20, 0x73,
0x65, 0x6e, 0x6e, 0x69, 0x4d, 0x61, 0x61, 0x6c, 0x69, 0x4b, 0x6f, 0x79,
0x72, 0x61, 0x20, 0x63, 0x69, 0x69, 0x6e, 0x69, 0x4b, 0x70, 0x25b, 0x6c,
0x25b, 0x25b, 0x6b, 0x75, 0x72, 0x64, 0xee, 0x20, 0x28, 0x6b, 0x75, 0x72,
0x6d, 0x61, 0x6e, 0x63, 0xee, 0x29, 0x54, 0x69, 0x72, 0x6b, 0x69, 0x79,
0x65, 0x4b, 0x77, 0x61, 0x73, 0x69, 0x6f, 0x4b, 0x61, 0x6d, 0x65, 0x72,
0x75, 0x6e, 0x43a, 0x44b, 0x440, 0x433, 0x44b, 0x437, 0x447, 0x430, 0x41a, 0x44b,
0x440, 0x433, 0x44b, 0x437, 0x441, 0x442, 0x430, 0x43d, 0x4c, 0x61, 0x6b, 0x21f,
0xf3, 0x6c, 0x2bc, 0x69, 0x79, 0x61, 0x70, 0x69, 0x4d, 0xed, 0x6c, 0x61,
0x68, 0x61, 0x14b, 0x73, 0x6b, 0x61, 0x20, 0x54, 0x21f, 0x61, 0x6d, 0xe1,
0x6b, 0x21f, 0x6f, 0x10d, 0x68, 0x65, 0x4b, 0x268, 0x6c, 0x61, 0x61, 0x6e,
0x67, 0x69, 0x54, 0x61, 0x61, 0x6e, 0x73, 0x61, 0x6e, 0xed, 0x61, 0xea5,
0xeb2, 0xea7, 0x4c, 0x61, 0x74, 0x69, 0x6e, 0x61, 0x43, 0x69, 0x76, 0x69,
0x74, 0x61, 0x73, 0x20, 0x56, 0x61, 0x74, 0x69, 0x63, 0x61, 0x6e, 0x61,
0x6c, 0x61, 0x74, 0x76, 0x69, 0x65, 0x161, 0x75, 0x4c, 0x61, 0x74, 0x76,
0x69, 0x6a, 0x61, 0x6c, 0x69, 0x6e, 0x67, 0xe1, 0x6c, 0x61, 0x52, 0x65,
0x70, 0x75, 0x62, 0x6c, 0xed, 0x6b, 0x69, 0x20, 0x79, 0x61, 0x20, 0x4b,
0x6f, 0x6e, 0x67, 0xf3, 0x20, 0x44, 0x65, 0x6d, 0x6f, 0x6b, 0x72, 0x61,
0x74, 0xed, 0x6b, 0x69, 0x41, 0x6e, 0x67, 0xf3, 0x6c, 0x61, 0x52, 0x65,
0x70, 0x69, 0x62, 0x69, 0x6b, 0x69, 0x20, 0x79, 0x61, 0x20, 0x41, 0x66,
0x72, 0xed, 0x6b, 0x61, 0x20, 0x79, 0x61, 0x20, 0x4b, 0xe1, 0x74, 0x69,
0x4b, 0x6f, 0x6e, 0x67, 0x6f, 0x6c, 0x69, 0x65, 0x74, 0x75, 0x76, 0x69,
0x173, 0x4c, 0x69, 0x65, 0x74, 0x75, 0x76, 0x61, 0x6c, 0x61, 0x20, 0x2e,
0x6c, 0x6f, 0x6a, 0x62, 0x61, 0x6e, 0x2e, 0x64, 0x6f, 0x6c, 0x6e, 0x6f,
0x73, 0x65, 0x72, 0x62, 0x161, 0x107, 0x69, 0x6e, 0x61, 0x4e, 0x69, 0x6d,
0x73, 0x6b, 0x61, 0x4e, 0x65, 0x64, 0x64, 0x65, 0x72, 0x73, 0x61, 0x73,
0x73, 0x2019, 0x73, 0x63, 0x68, 0x44, 0xfc, 0xfc, 0x74, 0x73, 0x63, 0x68,
0x6c, 0x61, 0x6e, 0x64, 0x4e, 0x65, 0x64, 0x64, 0x65, 0x72, 0x6c, 0x61,
0x6e, 0x6e, 0x65, 0x6e, 0x54, 0x73, 0x68, 0x69, 0x6c, 0x75, 0x62, 0x61,
0x44, 0x69, 0x74, 0x75, 0x6e, 0x67, 0x61, 0x20, 0x77, 0x61, 0x20, 0x4b,
0x6f, 0x6e, 0x67, 0x75, 0x6a, 0x75, 0x6c, 0x65, 0x76, 0x73, 0xe1, 0x6d,
0x65, 0x67, 0x69, 0x65, 0x6c, 0x6c, 0x61, 0x44, 0x68, 0x6f, 0x6c, 0x75,
0x6f, 0x4c, 0xeb, 0x74, 0x7a, 0x65, 0x62, 0x75, 0x65, 0x72, 0x67, 0x65,
0x73, 0x63, 0x68, 0x4c, 0x75, 0x6c, 0x75, 0x68, 0x69, 0x61, 0x43c, 0x430,
0x43a, 0x435, 0x434, 0x43e, 0x43d, 0x441, 0x43a, 0x438, 0x421, 0x435, 0x432, 0x435,
0x440, 0x43d, 0x430, 0x20, 0x41c, 0x430, 0x43a, 0x435, 0x434, 0x43e, 0x43d, 0x438,
0x458, 0x430, 0x4b, 0x69, 0x6d, 0x61, 0x63, 0x68, 0x61, 0x6d, 0x65, 0x92e,
0x948, 0x925, 0x93f, 0x932, 0x940, 0x4d, 0x61, 0x6b, 0x75, 0x61, 0x55, 0x6d,
0x6f, 0x7a, 0x61, 0x6d, 0x62, 0x69, 0x6b, 0x69, 0x43, 0x68, 0x69, 0x6d,
0x61, 0x6b, 0x6f, 0x6e, 0x64, 0x65, 0x4d, 0x61, 0x6c, 0x61, 0x67, 0x61,
0x73, 0x79, 0x4d, 0x61, 0x64, 0x61, 0x67, 0x61, 0x73, 0x69, 0x6b, 0x61,
0x72, 0x61, 0xd2e, 0xd32, 0xd2f, 0xd3e, 0xd33, 0xd02, 0xd07, 0xd28, 0xd4d, 0xd24,
0xd4d, 0xd2f, 0x4d, 0x65, 0x6c, 0x61, 0x79, 0x75, 0x628, 0x647, 0x627, 0x633,
0x20, 0x645, 0x644, 0x627, 0x64a, 0x648, 0x628, 0x631, 0x648, 0x646, 0x64a, 0x645,
0x644, 0x64a, 0x633, 0x64a, 0x627, 0x42, 0x72, 0x75, 0x6e, 0x65, 0x69, 0x53,
0x69, 0x6e, 0x67, 0x61, 0x70, 0x75, 0x72, 0x61, 0x4d, 0x61, 0x6c, 0x74,
0x69, 0x9ae, 0x9c8, 0x9a4, 0x9c8, 0x9b2, 0x9cb, 0x9a8, 0x9cd, 0x987, 0x9a8, 0x9cd,
0x9a6, 0x9bf, 0x9af, 0x9bc, 0x9be, 0xabc3, 0xabe4, 0xabc7, 0xabe9, 0xabc2, 0xabe3, 0xabdf,
0x47, 0x61, 0x65, 0x6c, 0x67, 0x45, 0x6c, 0x6c, 0x61, 0x6e, 0x20, 0x56,
0x61, 0x6e, 0x6e, 0x69, 0x6e, 0x4d, 0x101, 0x6f, 0x72, 0x69, 0x41, 0x6f,
0x74, 0x65, 0x61, 0x72, 0x6f, 0x61, 0x4d, 0x61, 0x70, 0x75, 0x64, 0x75,
0x6e, 0x67, 0x75, 0x6e, 0x92e, 0x930, 0x93e, 0x920, 0x940, 0x54, 0x61, 0x6e,
0x73, 0x61, 0x6e, 0x69, 0x61, 0x645, 0x627, 0x632, 0x631, 0x648, 0x646, 0x6cc,
0x627, 0x6cc, 0x631, 0x627, 0x646, 0x4b, 0x129, 0x6d, 0x129, 0x72, 0x169, 0x6d,
0x65, 0x74, 0x61, 0x2bc, 0x4b, 0x61, 0x6d, 0x61, 0x6c, 0x75, 0x6e, 0x4b,
0x61, 0x6e, 0x69, 0x65, 0x6e, 0x2bc, 0x6b, 0xe9, 0x68, 0x61, 0x43c, 0x43e,
0x43d, 0x433, 0x43e, 0x43b, 0x41c, 0x43e, 0x43d, 0x433, 0x43e, 0x43b, 0x182e, 0x1823,
0x1829, 0x182d, 0x1823, 0x182f, 0x6b, 0x72, 0x65, 0x6f, 0x6c, 0x20, 0x6d, 0x6f,
0x72, 0x69, 0x73, 0x69, 0x65, 0x6e, 0x4d, 0x6f, 0x72, 0x69, 0x73, 0x4d,
0x55, 0x4e, 0x44, 0x41, 0x14a, 0x6b, 0x61, 0x6d, 0x65, 0x72, 0x75, 0x14b,
0x4d, 0x76, 0x73, 0x6b, 0x6f, 0x6b, 0x65, 0x4b, 0x68, 0x6f, 0x65, 0x6b,
0x68, 0x6f, 0x65, 0x67, 0x6f, 0x77, 0x61, 0x62, 0x4e, 0x61, 0x6d, 0x69,
0x62, 0x69, 0x61, 0x62, 0x44, 0x69, 0x6e, 0xe9, 0x20, 0x42, 0x69, 0x7a,
0x61, 0x61, 0x64, 0x928, 0x947, 0x92a, 0x93e, 0x932, 0x940, 0x53, 0x68, 0x77,
0xf3, 0x14b, 0xf2, 0x20, 0x6e, 0x67, 0x69, 0x65, 0x6d, 0x62, 0x254, 0x254,
0x6e, 0x4b, 0xe0, 0x6d, 0x61, 0x6c, 0xfb, 0x6d, 0x4e, 0x64, 0x61, 0xa78c,
0x61, 0x4b, 0x61, 0x6d, 0x25b, 0x6c, 0xfb, 0x6e, 0x4e, 0x61, 0x69, 0x6a,
0xed, 0x72, 0x69, 0xe1, 0x20, 0x50, 0xed, 0x6a, 0x69, 0x6e, 0x4e, 0x61,
0x69, 0x6a, 0xed, 0x72, 0x69, 0x61, 0x7d2, 0x7de, 0x7cf, 0x7d6, 0x7cc, 0x7ec,
0x7e3, 0x7cd, 0x7eb, 0x644, 0x6ca, 0x631, 0x6cc, 0x20, 0x634, 0x648, 0x645, 0x627,
0x644, 0x6cc, 0x64, 0x61, 0x76, 0x76, 0x69, 0x73, 0xe1, 0x6d, 0x65, 0x67,
0x69, 0x65, 0x6c, 0x6c, 0x61, 0x4e, 0x6f, 0x72, 0x67, 0x61, 0x53, 0x75,
0x6f, 0x70, 0x6d, 0x61, 0x52, 0x75, 0x6f, 0x167, 0x167, 0x61, 0x53, 0x65,
0x73, 0x6f, 0x74, 0x68, 0x6f, 0x20, 0x73, 0x61, 0x20, 0x4c, 0x65, 0x62,
0x6f, 0x61, 0x41, 0x66, 0x72, 0x69, 0x6b, 0x61, 0x20, 0x42, 0x6f, 0x72,
0x77, 0x61, 0x69, 0x73, 0x69, 0x4e, 0x64, 0x65, 0x62, 0x65, 0x6c, 0x65,
0x6e, 0x6f, 0x72, 0x73, 0x6b, 0x20, 0x62, 0x6f, 0x6b, 0x6d, 0xe5, 0x6c,
0x4e, 0x6f, 0x72, 0x67, 0x65, 0x53, 0x76, 0x61, 0x6c, 0x62, 0x61, 0x72,
0x64, 0x20, 0x6f, 0x67, 0x20, 0x4a, 0x61, 0x6e, 0x20, 0x4d, 0x61, 0x79,
0x65, 0x6e, 0x6e, 0x6f, 0x72, 0x73, 0x6b, 0x20, 0x6e, 0x79, 0x6e, 0x6f,
0x72, 0x73, 0x6b, 0x4e, 0x6f, 0x72, 0x65, 0x67, 0x54, 0x68, 0x6f, 0x6b,
0x20, 0x4e, 0x61, 0x74, 0x68, 0x4e, 0x79, 0x61, 0x6e, 0x6a, 0x61, 0x52,
0x75, 0x6e, 0x79, 0x61, 0x6e, 0x6b, 0x6f, 0x72, 0x65, 0x6f, 0x63, 0x63,
0x69, 0x74, 0x61, 0x6e, 0x45, 0x73, 0x70, 0x61, 0x6e, 0x68, 0x61, 0xb13,
0xb21, 0xb3c, 0xb3f, 0xb06, 0xb2d, 0xb3e, 0xb30, 0xb24, 0x4f, 0x72, 0x6f, 0x6d,
0x6f, 0x6f, 0x49, 0x74, 0x6f, 0x6f, 0x70, 0x68, 0x69, 0x79, 0x61, 0x61,
0x4b, 0x65, 0x65, 0x6e, 0x69, 0x79, 0x61, 0x61, 0xd801, 0xdccf, 0xd801, 0xdcd8,
0xd801, 0xdcfb, 0xd801, 0xdcd8, 0xd801, 0xdcfb, 0xd801, 0xdcdf, 0x438, 0x440, 0x43e, 0x43d,
0x413, 0x443, 0x44b, 0x440, 0x434, 0x437, 0x44b, 0x441, 0x442, 0x43e, 0x43d, 0x423,
0x4d5, 0x440, 0x4d5, 0x441, 0x435, 0x50, 0x61, 0x70, 0x69, 0x61, 0x6d, 0x65,
0x6e, 0x74, 0x75, 0x4b, 0xf2, 0x72, 0x73, 0x6f, 0x75, 0x67e, 0x69a, 0x62a,
0x648, 0x627, 0x641, 0x63a, 0x627, 0x646, 0x633, 0x62a, 0x627, 0x646, 0x67e, 0x627,
0x6a9, 0x633, 0x62a, 0x627, 0x646, 0x641, 0x627, 0x631, 0x633, 0x6cc, 0x62f, 0x631,
0x6cc, 0x70, 0x6f, 0x6c, 0x73, 0x6b, 0x69, 0x50, 0x6f, 0x6c, 0x73, 0x6b,
0x61, 0x70, 0x6f, 0x72, 0x74, 0x75, 0x67, 0x75, 0xea, 0x73, 0x42, 0x72,
0x61, 0x73, 0x69, 0x6c, 0x41, 0x6e, 0x67, 0x6f, 0x6c, 0x61, 0x43, 0x61,
0x62, 0x6f, 0x20, 0x56, 0x65, 0x72, 0x64, 0x65, 0x47, 0x75, 0x69, 0x6e,
0xe9, 0x20, 0x45, 0x71, 0x75, 0x61, 0x74, 0x6f, 0x72, 0x69, 0x61, 0x6c,
0x47, 0x75, 0x69, 0x6e, 0xe9, 0x2d, 0x42, 0x69, 0x73, 0x73, 0x61, 0x75,
0x4c, 0x75, 0x78, 0x65, 0x6d, 0x62, 0x75, 0x72, 0x67, 0x6f, 0x4d, 0x61,
0x63, 0x61, 0x75, 0x2c, 0x20, 0x52, 0x41, 0x45, 0x20, 0x64, 0x61, 0x20,
0x43, 0x68, 0x69, 0x6e, 0x61, 0x4d, 0x6f, 0xe7, 0x61, 0x6d, 0x62, 0x69,
0x71, 0x75, 0x65, 0x70, 0x6f, 0x72, 0x74, 0x75, 0x67, 0x75, 0xea, 0x73,
0x20, 0x65, 0x75, 0x72, 0x6f, 0x70, 0x65, 0x75, 0x50, 0x6f, 0x72, 0x74,
0x75, 0x67, 0x61, 0x6c, 0x53, 0xe3, 0x6f, 0x20, 0x54, 0x6f, 0x6d, 0xe9,
0x20, 0x65, 0x20, 0x50, 0x72, 0xed, 0x6e, 0x63, 0x69, 0x70, 0x65, 0x53,
0x75, 0xed, 0xe7, 0x61, 0x54, 0x69, 0x6d, 0x6f, 0x72, 0x2d, 0x4c, 0x65,
0x73, 0x74, 0x65, 0x70, 0x72, 0x16b, 0x73, 0x69, 0x73, 0x6b, 0x61, 0x6e,
0x50, 0x14d, 0x6c, 0x69, 0xa2a, 0xa70, 0xa1c, 0xa3e, 0xa2c, 0xa40, 0xa2d, 0xa3e,
0xa30, 0xa24, 0x67e, 0x646, 0x62c, 0x627, 0x628, 0x6cc, 0x52, 0x75, 0x6e, 0x61,
0x73, 0x69, 0x6d, 0x69, 0x50, 0x65, 0x72, 0xfa, 0x42, 0x6f, 0x6c, 0x69,
0x76, 0x69, 0x61, 0x45, 0x63, 0x75, 0x61, 0x64, 0x6f, 0x72, 0x72, 0x6f,
0x6d, 0xe2, 0x6e, 0x103, 0x52, 0x6f, 0x6d, 0xe2, 0x6e, 0x69, 0x61, 0x52,
0x65, 0x70, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x61, 0x20, 0x4d, 0x6f, 0x6c,
0x64, 0x6f, 0x76, 0x61, 0x72, 0x75, 0x6d, 0x61, 0x6e, 0x74, 0x73, 0x63,
0x68, 0x53, 0x76, 0x69, 0x7a, 0x72, 0x61, 0x4b, 0x69, 0x68, 0x6f, 0x72,
0x6f, 0x6d, 0x62, 0x6f, 0x49, 0x6b, 0x69, 0x72, 0x75, 0x6e, 0x64, 0x69,
0x55, 0x62, 0x75, 0x72, 0x75, 0x6e, 0x64, 0x69, 0x440, 0x443, 0x441, 0x441,
0x43a, 0x438, 0x439, 0x420, 0x43e, 0x441, 0x441, 0x438, 0x44f, 0x41a, 0x430, 0x437,
0x430, 0x445, 0x441, 0x442, 0x430, 0x43d, 0x41a, 0x438, 0x440, 0x433, 0x438, 0x437,
0x438, 0x44f, 0x41c, 0x43e, 0x43b, 0x434, 0x43e, 0x432, 0x430, 0x423, 0x43a, 0x440,
0x430, 0x438, 0x43d, 0x430, 0x4b, 0x69, 0x72, 0x75, 0x77, 0x61, 0x53, 0x61,
0x68, 0x6f, 0x441, 0x430, 0x445, 0x430, 0x20, 0x442, 0x44b, 0x43b, 0x430, 0x410,
0x440, 0x430, 0x441, 0x441, 0x44b, 0x44b, 0x439, 0x430, 0x4b, 0x69, 0x73, 0x61,
0x6d, 0x70, 0x75, 0x72, 0x53, 0xe4, 0x6e, 0x67, 0xf6, 0x4b, 0xf6, 0x64,
0xf6, 0x72, 0xf6, 0x73, 0xea, 0x73, 0x65, 0x20, 0x74, 0xee, 0x20, 0x42,
0xea, 0x61, 0x66, 0x72, 0xee, 0x6b, 0x61, 0x49, 0x73, 0x68, 0x69, 0x73,
0x61, 0x6e, 0x67, 0x75, 0x54, 0x61, 0x6e, 0x73, 0x61, 0x6e, 0x69, 0x79,
0x61, 0x938, 0x902, 0x938, 0x94d, 0x915, 0x943, 0x924, 0x20, 0x92d, 0x93e, 0x937,
0x93e, 0x92d, 0x93e, 0x930, 0x924, 0x903, 0x1c65, 0x1c5f, 0x1c71, 0x1c5b, 0x1c5f, 0x1c72,
0x1c64, 0x1c64, 0x1c71, 0x1c70, 0x1c64, 0x1c6d, 0x1c5f, 0x938, 0x93e, 0x928, 0x924, 0x93e,
0x921, 0x93c, 0x940, 0x73, 0x61, 0x72, 0x64, 0x75, 0x73, 0x65, 0x6e, 0x61,
0x441, 0x440, 0x43f, 0x441, 0x43a, 0x438, 0x421, 0x440, 0x431, 0x438, 0x458, 0x430,
0x41a, 0x43e, 0x441, 0x43e, 0x432, 0x43e, 0x426, 0x440, 0x43d, 0x430, 0x20, 0x413,
0x43e, 0x440, 0x430, 0x73, 0x72, 0x70, 0x73, 0x6b, 0x69, 0x4b, 0x6f, 0x73,
0x6f, 0x76, 0x6f, 0x43, 0x72, 0x6e, 0x61, 0x20, 0x47, 0x6f, 0x72, 0x61,
0x53, 0x72, 0x62, 0x69, 0x6a, 0x61, 0x4b, 0x69, 0x73, 0x68, 0x61, 0x6d,
0x62, 0x61, 0x61, 0x63, 0x68, 0x69, 0x53, 0x68, 0x6f, 0x6e, 0x61, 0xa188,
0xa320, 0xa259, 0xa34f, 0xa1e9, 0x73, 0x69, 0x63, 0x69, 0x6c, 0x69, 0x61, 0x6e,
0x75, 0x53, 0x69, 0x64, 0x61, 0x61, 0x6d, 0x75, 0x20, 0x41, 0x66, 0x6f,
0x49, 0x74, 0x69, 0x79, 0x6f, 0x6f, 0x70, 0x68, 0x69, 0x79, 0x61, 0x15b,
0x6c, 0x14d, 0x6e, 0x73, 0x6b, 0x69, 0x633, 0x646, 0x68c, 0x64a, 0x67e, 0x627,
0x6aa, 0x633, 0x62a, 0x627, 0x646, 0x938, 0x93f, 0x928, 0x94d, 0x927, 0x940, 0xdc3,
0xdd2, 0xd82, 0xdc4, 0xdbd, 0xdc1, 0xdca, 0x200d, 0xdbb, 0xdd3, 0x20, 0xdbd, 0xd82,
0xd9a, 0xdcf, 0xdc0, 0x73, 0xe4, 0xe4, 0x2b9, 0x6d, 0x1e9, 0x69, 0xf5, 0x6c,
0x6c, 0x4c, 0xe4, 0xe4, 0x2b9, 0x64, 0x64, 0x6a, 0xe2, 0x6e, 0x6e, 0x61,
0x6d, 0x73, 0x6c, 0x6f, 0x76, 0x65, 0x6e, 0x10d, 0x69, 0x6e, 0x61, 0x53,
0x6c, 0x6f, 0x76, 0x65, 0x6e, 0x73, 0x6b, 0x6f, 0x73, 0x6c, 0x6f, 0x76,
0x65, 0x6e, 0x161, 0x10d, 0x69, 0x6e, 0x61, 0x53, 0x6c, 0x6f, 0x76, 0x65,
0x6e, 0x69, 0x6a, 0x61, 0x4f, 0x6c, 0x75, 0x73, 0x6f, 0x67, 0x61, 0x53,
0x6f, 0x6f, 0x6d, 0x61, 0x61, 0x6c, 0x69, 0x53, 0x6f, 0x6f, 0x6d, 0x61,
0x61, 0x6c, 0x69, 0x79, 0x61, 0x4a, 0x61, 0x62, 0x75, 0x75, 0x74, 0x69,
0x49, 0x74, 0x6f, 0x6f, 0x62, 0x69, 0x79, 0x61, 0x6a9, 0x648, 0x631, 0x62f,
0x6cc, 0x20, 0x62e, 0x648, 0x627, 0x631, 0x6af, 0xc5, 0x61, 0x72, 0x6a, 0x65,
0x6c, 0x73, 0x61, 0x65, 0x6d, 0x69, 0x65, 0x6e, 0x20, 0x67, 0xef, 0x65,
0x6c, 0x65, 0x65, 0x73, 0x70, 0x61, 0xf1, 0x6f, 0x6c, 0x20, 0x64, 0x65,
0x20, 0x45, 0x73, 0x70, 0x61, 0xf1, 0x61, 0x41, 0x72, 0x67, 0x65, 0x6e,
0x74, 0x69, 0x6e, 0x61, 0x42, 0x65, 0x6c, 0x69, 0x63, 0x65, 0x43, 0x61,
0x6e, 0x61, 0x72, 0x69, 0x61, 0x73, 0x43, 0x65, 0x75, 0x74, 0x61, 0x20,
0x79, 0x20, 0x4d, 0x65, 0x6c, 0x69, 0x6c, 0x6c, 0x61, 0x43, 0x68, 0x69,
0x6c, 0x65, 0x43, 0x6f, 0x6c, 0x6f, 0x6d, 0x62, 0x69, 0x61, 0x43, 0x6f,
0x73, 0x74, 0x61, 0x20, 0x52, 0x69, 0x63, 0x61, 0x43, 0x75, 0x62, 0x61,
0x52, 0x65, 0x70, 0xfa, 0x62, 0x6c, 0x69, 0x63, 0x61, 0x20, 0x44, 0x6f,
0x6d, 0x69, 0x6e, 0x69, 0x63, 0x61, 0x6e, 0x61, 0x45, 0x6c, 0x20, 0x53,
0x61, 0x6c, 0x76, 0x61, 0x64, 0x6f, 0x72, 0x47, 0x75, 0x69, 0x6e, 0x65,
0x61, 0x20, 0x45, 0x63, 0x75, 0x61, 0x74, 0x6f, 0x72, 0x69, 0x61, 0x6c,
0x47, 0x75, 0x61, 0x74, 0x65, 0x6d, 0x61, 0x6c, 0x61, 0x48, 0x6f, 0x6e,
0x64, 0x75, 0x72, 0x61, 0x73, 0x65, 0x73, 0x70, 0x61, 0xf1, 0x6f, 0x6c,
0x20, 0x6c, 0x61, 0x74, 0x69, 0x6e, 0x6f, 0x61, 0x6d, 0x65, 0x72, 0x69,
0x63, 0x61, 0x6e, 0x6f, 0x4c, 0x61, 0x74, 0x69, 0x6e, 0x6f, 0x61, 0x6d,
0xe9, 0x72, 0x69, 0x63, 0x61, 0x65, 0x73, 0x70, 0x61, 0xf1, 0x6f, 0x6c,
0x20, 0x64, 0x65, 0x20, 0x4d, 0xe9, 0x78, 0x69, 0x63, 0x6f, 0x4d, 0xe9,
0x78, 0x69, 0x63, 0x6f, 0x4e, 0x69, 0x63, 0x61, 0x72, 0x61, 0x67, 0x75,
0x61, 0x50, 0x61, 0x6e, 0x61, 0x6d, 0xe1, 0x50, 0x61, 0x72, 0x61, 0x67,
0x75, 0x61, 0x79, 0x46, 0x69, 0x6c, 0x69, 0x70, 0x69, 0x6e, 0x61, 0x73,
0x45, 0x73, 0x74, 0x61, 0x64, 0x6f, 0x73, 0x20, 0x55, 0x6e, 0x69, 0x64,
0x6f, 0x73, 0x55, 0x72, 0x75, 0x67, 0x75, 0x61, 0x79, 0x56, 0x65, 0x6e,
0x65, 0x7a, 0x75, 0x65, 0x6c, 0x61, 0x2d5c, 0x2d30, 0x2d4e, 0x2d30, 0x2d63, 0x2d49,
0x2d56, 0x2d5c, 0x2d4d, 0x2d4e, 0x2d56, 0x2d54, 0x2d49, 0x2d31, 0x42, 0x61, 0x73, 0x61,
0x20, 0x53, 0x75, 0x6e, 0x64, 0x61, 0x4b, 0x69, 0x73, 0x77, 0x61, 0x68,
0x69, 0x6c, 0x69, 0x4a, 0x61, 0x6d, 0x68, 0x75, 0x72, 0x69, 0x20, 0x79,
0x61, 0x20, 0x4b, 0x69, 0x64, 0x65, 0x6d, 0x6f, 0x6b, 0x72, 0x61, 0x73,
0x69, 0x61, 0x20, 0x79, 0x61, 0x20, 0x4b, 0x6f, 0x6e, 0x67, 0x6f, 0x73,
0x69, 0x53, 0x77, 0x61, 0x74, 0x69, 0x65, 0x53, 0x77, 0x61, 0x74, 0x69,
0x6e, 0x69, 0x73, 0x76, 0x65, 0x6e, 0x73, 0x6b, 0x61, 0x53, 0x76, 0x65,
0x72, 0x69, 0x67, 0x65, 0xc5, 0x6c, 0x61, 0x6e, 0x64, 0x53, 0x63, 0x68,
0x77, 0x69, 0x69, 0x7a, 0x65, 0x72, 0x74, 0xfc, 0xfc, 0x74, 0x73, 0x63,
0x68, 0x46, 0x72, 0x61, 0x6e, 0x6b, 0x72, 0x69, 0x69, 0x63, 0x68, 0x4c,
0x69, 0xe4, 0x63, 0x68, 0x74, 0x65, 0x73, 0x63, 0x68, 0x74, 0xe4, 0x69,
0x723, 0x718, 0x72a, 0x71d, 0x71d, 0x710, 0x725, 0x71d, 0x72a, 0x729, 0x723, 0x718,
0x72a, 0x71d, 0x710, 0x2d5c, 0x2d30, 0x2d5b, 0x2d4d, 0x2d43, 0x2d49, 0x2d5c, 0x54, 0x61,
0x73, 0x68, 0x65, 0x6c, 0x1e25, 0x69, 0x79, 0x74, 0x6c, 0x6d, 0x263, 0x72,
0x69, 0x62, 0xaabc, 0xaa95, 0xaa92, 0xaabe, 0x4b, 0x69, 0x74, 0x61, 0x69, 0x74,
0x61, 0x442, 0x43e, 0x4b7, 0x438, 0x43a, 0x4e3, 0x422, 0x43e, 0x4b7, 0x438, 0x43a,
0x438, 0x441, 0x442, 0x43e, 0x43d, 0xba4, 0xbae, 0xbbf, 0xbb4, 0xbcd, 0xb87, 0xba8,
0xbcd, 0xba4, 0xbbf, 0xbaf, 0xbbe, 0xbae, 0xbb2, 0xbc7, 0xb9a, 0xbbf, 0xbaf, 0xbbe,
0xb9a, 0xbbf, 0xb99, 0xbcd, 0xb95, 0xbaa, 0xbcd, 0xbaa, 0xbc2, 0xbb0, 0xbcd, 0xb87,
0xbb2, 0xb99, 0xbcd, 0xb95, 0xbc8, 0x70, 0x61, 0x74, 0x61, 0x73, 0x20, 0x54,
0x61, 0x72, 0x6f, 0x6b, 0x6f, 0x54, 0x61, 0x73, 0x61, 0x77, 0x61, 0x71,
0x20, 0x73, 0x65, 0x6e, 0x6e, 0x69, 0x4e, 0x69, 0x17e, 0x65, 0x72, 0x442,
0x430, 0x442, 0x430, 0x440, 0xc24, 0xc46, 0xc32, 0xc41, 0xc17, 0xc41, 0xc2d, 0xc3e,
0xc30, 0xc24, 0xc26, 0xc47, 0xc36, 0xc02, 0x4b, 0x69, 0x74, 0x65, 0x73, 0x6f,
0x4b, 0x65, 0x6e, 0x69, 0x61, 0xe44, 0xe17, 0xe22, 0xf56, 0xf7c, 0xf51, 0xf0b,
0xf66, 0xf90, 0xf51, 0xf0b, 0xf62, 0xf92, 0xfb1, 0xf0b, 0xf53, 0xf42, 0xf62, 0xf92,
0xfb1, 0xf0b, 0xf42, 0xf62, 0xf0b, 0x1275, 0x130d, 0x1228, 0x1275, 0x130d, 0x122d, 0x129b,
0x54, 0x6f, 0x6b, 0x20, 0x50, 0x69, 0x73, 0x69, 0x6e, 0x50, 0x61, 0x70,
0x75, 0x61, 0x20, 0x4e, 0x69, 0x75, 0x67, 0x69, 0x6e, 0x69, 0x6c, 0x65,
0x61, 0x20, 0x66, 0x61, 0x6b, 0x61, 0x74, 0x6f, 0x6e, 0x67, 0x61, 0x58,
0x69, 0x74, 0x73, 0x6f, 0x6e, 0x67, 0x61, 0x53, 0x65, 0x74, 0x73, 0x77,
0x61, 0x6e, 0x61, 0x41, 0x66, 0x6f, 0x72, 0x69, 0x6b, 0x61, 0x20, 0x42,
0x6f, 0x72, 0x77, 0x61, 0x54, 0xfc, 0x72, 0x6b, 0xe7, 0x65, 0x54, 0xfc,
0x72, 0x6b, 0x69, 0x79, 0x65, 0x4b, 0x131, 0x62, 0x72, 0x131, 0x73, 0x74,
0xfc, 0x72, 0x6b, 0x6d, 0x65, 0x6e, 0x20, 0x64, 0x69, 0x6c, 0x69, 0x54,
0xfc, 0x72, 0x6b, 0x6d, 0x65, 0x6e, 0x69, 0x73, 0x74, 0x61, 0x6e, 0x4b,
0x61, 0x74, 0x61, 0x62, 0x443, 0x43a, 0x440, 0x430, 0x457, 0x43d, 0x441, 0x44c,
0x43a, 0x430, 0x423, 0x43a, 0x440, 0x430, 0x457, 0x43d, 0x430, 0x68, 0x6f, 0x72,
0x6e, 0x6a, 0x6f, 0x73, 0x65, 0x72, 0x62, 0x161, 0x107, 0x69, 0x6e, 0x61,
0x4e, 0x11b, 0x6d, 0x73, 0x6b, 0x61, 0x627, 0x631, 0x62f, 0x648, 0x628, 0x6be,
0x627, 0x631, 0x62a, 0x626, 0x6c7, 0x64a, 0x63a, 0x6c7, 0x631, 0x686, 0x6d5, 0x62c,
0x6c7, 0x6ad, 0x6af, 0x648, 0x6f, 0x2018, 0x7a, 0x62, 0x65, 0x6b, 0x4f, 0x2bb,
0x7a, 0x62, 0x65, 0x6b, 0x69, 0x73, 0x74, 0x6f, 0x6e, 0x627, 0x648, 0x632,
0x628, 0x6cc, 0x6a9, 0x45e, 0x437, 0x431, 0x435, 0x43a, 0x447, 0x430, 0x40e, 0x437,
0x431, 0x435, 0x43a, 0x438, 0x441, 0x442, 0x43e, 0x43d, 0xa559, 0xa524, 0xa55e, 0xa524,
0xa52b, 0xa569, 0x56, 0x61, 0x69, 0x4c, 0x61, 0x69, 0x62, 0x68, 0x69, 0x79,
0x61, 0x54, 0x73, 0x68, 0x69, 0x76, 0x65, 0x6e, 0x1e13, 0x61, 0x54, 0x69,
0x1ebf, 0x6e, 0x67, 0x20, 0x56, 0x69, 0x1ec7, 0x74, 0x56, 0x69, 0x1ec7, 0x74,
0x20, 0x4e, 0x61, 0x6d, 0x56, 0x6f, 0x6c, 0x61, 0x70, 0xfc, 0x6b, 0x4b,
0x79, 0x69, 0x76, 0x75, 0x6e, 0x6a, 0x6f, 0x77, 0x61, 0x6c, 0x6f, 0x6e,
0x57, 0x61, 0x6c, 0x73, 0x65, 0x72, 0x53, 0x63, 0x68, 0x77, 0x69, 0x7a,
0x57, 0x61, 0x72, 0x6c, 0x70, 0x69, 0x72, 0x69, 0x43, 0x79, 0x6d, 0x72,
0x61, 0x65, 0x67, 0x59, 0x20, 0x44, 0x65, 0x79, 0x72, 0x6e, 0x61, 0x73,
0x20, 0x55, 0x6e, 0x65, 0x64, 0x69, 0x67, 0x628, 0x644, 0x648, 0x686, 0x6cc,
0x20, 0x28, 0x631, 0x62e, 0x634, 0x627, 0x646, 0x6cc, 0x29, 0x626, 0x648, 0x645,
0x627, 0x646, 0x645, 0x62a, 0x62d, 0x62f, 0x6cc, 0x646, 0x20, 0x639, 0x631, 0x628,
0x6cc, 0x646, 0x20, 0x627, 0x645, 0x627, 0x631, 0x627, 0x62a, 0x46, 0x72, 0x79,
0x73, 0x6b, 0x4e, 0x65, 0x64, 0x65, 0x72, 0x6c, 0xe2, 0x6e, 0x12c8, 0x120b,
0x12ed, 0x1273, 0x1271, 0x57, 0x6f, 0x6c, 0x6f, 0x66, 0x49, 0x73, 0x69, 0x58,
0x68, 0x6f, 0x73, 0x61, 0x45, 0x4d, 0x7a, 0x61, 0x6e, 0x74, 0x73, 0x69,
0x20, 0x41, 0x66, 0x72, 0x69, 0x6b, 0x61, 0x6e, 0x75, 0x61, 0x73, 0x75,
0x65, 0x4b, 0x65, 0x6d, 0x65, 0x6c, 0xfa, 0x6e, 0x5d9, 0x5d9, 0x5b4, 0x5d3,
0x5d9, 0x5e9, 0x5d0, 0x5d5, 0x5e7, 0x5e8, 0x5d0, 0x5b7, 0x5d9, 0x5e0, 0x5e2, 0xc8,
0x64, 0xe8, 0x20, 0x59, 0x6f, 0x72, 0xf9, 0x62, 0xe1, 0x4e, 0xe0, 0xec,
0x6a, 0xed, 0x72, 0xed, 0xe0, 0x42, 0x25b, 0x300, 0x6e, 0x25b, 0x300, 0x5a,
0x61, 0x72, 0x6d, 0x61, 0x63, 0x69, 0x69, 0x6e, 0x65, 0x56, 0x61, 0x68,
0x63, 0x75, 0x65, 0x6e, 0x67, 0x68, 0x43, 0x75, 0x6e, 0x67, 0x68, 0x67,
0x6f, 0x7a, 0x69, 0x73, 0x69, 0x5a, 0x75, 0x6c, 0x75, 0x69, 0x4e, 0x69,
0x6e, 0x67, 0x69, 0x7a, 0x69, 0x6d, 0x75, 0x20, 0x41, 0x66, 0x72, 0x69,
0x6b, 0x61, 0x6b, 0x61, 0x6e, 0x68, 0x67, 0xe1, 0x67, 0x4d, 0x72, 0x61,
0x73, 0x69, 0x72, 0x6e, 0x68, 0x65, 0x1ebd, 0x67, 0x61, 0x74, 0x75, 0x42,
0x72, 0x61, 0x73, 0x69, 0x75, 0xf1, 0x65, 0x6e, 0x67, 0x61, 0x74, 0xfa,
0x4b, 0x75, 0x72, 0x169, 0x62, 0x69, 0x79, 0x61, 0x57, 0x65, 0x6e, 0x65,
0x73, 0x75, 0x65, 0x72, 0x61, 0x939, 0x930, 0x93f, 0x92f, 0x93e, 0x923, 0x935,
0x940, 0x4e, 0x6f, 0x72, 0x64, 0x66, 0x72, 0x69, 0x69, 0x73, 0x6b, 0x54,
0x6a, 0x69, 0x69, 0x73, 0x6b, 0x6c, 0x75, 0x6e, 0x930, 0x93e, 0x91c, 0x938,
0x94d, 0x925, 0x93e, 0x928, 0x940, 0x43c, 0x43e, 0x43a, 0x448, 0x435, 0x43d, 0x44c,
0x20, 0x43a, 0x44f, 0x43b, 0x44c, 0x74, 0x6f, 0x6b, 0x69, 0x20, 0x70, 0x6f,
0x6e, 0x61, 0x6d, 0x61, 0x20, 0x61, 0x6c, 0x65, 0x50, 0x69, 0x6a, 0x69,
0x6e, 0x53, 0x6f, 0x6c, 0x6f, 0x6d, 0x6f, 0x6e, 0x20, 0x41, 0x65, 0x6c,
0x61, 0x6e, 0x4f, 0x62, 0x6f, 0x6c, 0x6f, 0x628, 0x644, 0x6c6, 0x686, 0x6cc,
0x42, 0x61, 0x6c, 0xf3, 0x63, 0x68, 0x69, 0x50, 0xe1, 0x6b, 0x65, 0x73,
0x74, 0xe1, 0x6e, 0x6c, 0x69, 0x67, 0x75, 0x72, 0x65, 0xd803, 0xdd0c, 0xd803,
0xdd17, 0xd803, 0xdd25, 0xd803, 0xdd1d, 0xd803, 0xdd19, 0xd803, 0xdd1a, 0xd803, 0xdd12, 0xd803,
0xdd19, 0xd803, 0xdd1d, 0x62a, 0x648, 0x631, 0x648, 0x627, 0x644, 0x6cc, 0x61, 0x6e,
0x69, 0x69, 0x20, 0x6b, 0x61, 0x67, 0x269, 0x6a, 0x61, 0x42, 0x65, 0x6e,
0x25b, 0x25b, 0x915, 0x93e, 0x902, 0x917, 0x921, 0x93c, 0x940, 0x76, 0x65, 0x6e,
0x65, 0x74, 0x6f, 0x6b, 0x75, 0x76, 0x69, 0x69, 0x6e, 0x1e0d, 0x69, 0x101,
0x915, 0x941, 0x935, 0x93f, 0x92c, 0x93e, 0x930, 0x924, 0xb15, 0xb41, 0xb71, 0xb3f,
0xb2c, 0xb3e, 0xb30, 0xb24, 0xc15, 0xc41, 0xc35, 0xc3f, 0xc2c, 0xc3e, 0xc30, 0xc24,
0xc26, 0xc46, 0xc38, 0xc3e, 0xc02
};

static constexpr char language_name_list[] =
"Default\0"
"C\0"
"Abkhazian\0"
"Afar\0"
"Afrikaans\0"
"Aghem\0"
"Akan\0"
"Akkadian\0"
"Akoose\0"
"Albanian\0"
"American Sign Language\0"
"Amharic\0"
"Ancient Egyptian\0"
"Ancient Greek\0"
"Arabic\0"
"Aragonese\0"
"Aramaic\0"
"Armenian\0"
"Assamese\0"
"Asturian\0"
"Asu\0"
"Atsam\0"
"Avaric\0"
"Avestan\0"
"Aymara\0"
"Azerbaijani\0"
"Bafia\0"
"Balinese\0"
"Bambara\0"
"Bamun\0"
"Bangla\0"
"Basaa\0"
"Bashkir\0"
"Basque\0"
"Batak Toba\0"
"Belarusian\0"
"Bemba\0"
"Bena\0"
"Bhojpuri\0"
"Bislama\0"
"Blin\0"
"Bodo\0"
"Bosnian\0"
"Breton\0"
"Buginese\0"
"Bulgarian\0"
"Burmese\0"
"Cantonese\0"
"Catalan\0"
"Cebuano\0"
"Central Atlas Tamazight\0"
"Central Kurdish\0"
"Chakma\0"
"Chamorro\0"
"Chechen\0"
"Cherokee\0"
"Chickasaw\0"
"Chiga\0"
"Chinese\0"
"Church Slavic\0"
"Chuvash\0"
"Colognian\0"
"Coptic\0"
"Cornish\0"
"Corsican\0"
"Cree\0"
"Croatian\0"
"Czech\0"
"Danish\0"
"Divehi\0"
"Dogri\0"
"Duala\0"
"Dutch\0"
"Dzongkha\0"
"Embu\0"
"English\0"
"Erzya\0"
"Esperanto\0"
"Estonian\0"
"Ewe\0"
"Ewondo\0"
"Faroese\0"
"Fijian\0"
"Filipino\0"
"Finnish\0"
"French\0"
"Friulian\0"
"Fula\0" // Fulah
"Scottish Gaelic\0"
"Ga\0"
"Galician\0"
"Ganda\0"
"Geez\0"
"Georgian\0"
"German\0"
"Gothic\0"
"Greek\0"
"Guarani\0"
"Gujarati\0"
"Gusii\0"
"Haitian Creole\0"
"Hausa\0"
"Hawaiian\0"
"Hebrew\0"
"Herero\0"
"Hindi\0"
"Hiri Motu\0"
"Hungarian\0"
"Icelandic\0"
"Ido\0"
"Igbo\0"
"Inari Sami\0"
"Indonesian\0"
"Ingush\0"
"Interlingua\0"
"Interlingue\0"
"Inuktitut\0"
"Inupiaq\0"
"Irish\0"
"Italian\0"
"Japanese\0"
"Javanese\0"
"Jju\0"
"Jola-Fonyi\0"
"Kabuverdianu\0"
"Kabyle\0"
"Kako\0"
"Kalaallisut\0"
"Kalenjin\0"
"Kamba\0"
"Kannada\0"
"Kanuri\0"
"Kashmiri\0"
"Kazakh\0"
"Kenyang\0"
"Khmer\0"
"Kʼicheʼ\0"
"Kikuyu\0"
"Kinyarwanda\0"
"Komi\0"
"Kongo\0"
"Konkani\0"
"Korean\0"
"Koro\0"
"Koyraboro Senni\0"
"Koyra Chiini\0"
"Kpelle\0"
"Kuanyama\0"
"Kurdish\0"
"Kwasio\0"
"Kyrgyz\0"
"Lakota\0"
"Langi\0"
"Lao\0"
"Latin\0"
"Latvian\0"
"Lezghian\0"
"Limburgish\0"
"Lingala\0"
"Literary Chinese\0"
"Lithuanian\0"
"Lojban\0"
"Lower Sorbian\0"
"Low German\0"
"Luba-Katanga\0"
"Lule Sami\0"
"Luo\0"
"Luxembourgish\0"
"Luyia\0"
"Macedonian\0"
"Machame\0"
"Maithili\0"
"Makhuwa-Meetto\0"
"Makonde\0"
"Malagasy\0"
"Malayalam\0"
"Malay\0"
"Maltese\0"
"Mandingo\0"
"Manipuri\0"
"Manx\0"
"Māori\0"
"Mapuche\0"
"Marathi\0"
"Marshallese\0"
"Masai\0"
"Mazanderani\0"
"Mende\0"
"Meru\0"
"Metaʼ\0"
"Mohawk\0"
"Mongolian\0"
"Morisyen\0"
"Mundang\0"
"Muscogee\0"
"Nama\0"
"Nauru\0"
"Navajo\0"
"Ndonga\0"
"Nepali\0"
"Newari\0"
"Ngiemboon\0"
"Ngomba\0"
"Nigerian Pidgin\0"
"N’Ko\0"
"Northern Luri\0"
"Northern Sami\0"
"Northern Sotho\0"
"North Ndebele\0"
"Norwegian Bokmål\0"
"Norwegian Nynorsk\0"
"Nuer\0"
"Nyanja\0"
"Nyankole\0"
"Occitan\0"
"Odia\0"
"Ojibwa\0"
"Old Irish\0"
"Old Norse\0"
"Old Persian\0"
"Oromo\0"
"Osage\0"
"Ossetic\0"
"Pahlavi\0"
"Palauan\0"
"Pali\0"
"Papiamento\0"
"Pashto\0"
"Persian\0"
"Phoenician\0"
"Polish\0"
"Portuguese\0"
"Prussian\0"
"Punjabi\0"
"Quechua\0"
"Romanian\0"
"Romansh\0"
"Rombo\0"
"Rundi\0"
"Russian\0"
"Rwa\0"
"Saho\0"
"Yakut\0" // Sakha
"Samburu\0"
"Samoan\0"
"Sango\0"
"Sangu\0"
"Sanskrit\0"
"Santali\0"
"Sardinian\0"
"Saurashtra\0"
"Sena\0"
"Serbian\0"
"Shambala\0"
"Shona\0"
"Sichuan Yi\0"
"Sicilian\0"
"Sidamo\0"
"Silesian\0"
"Sindhi\0"
"Sinhala\0"
"Skolt Sami\0"
"Slovak\0"
"Slovenian\0"
"Soga\0"
"Somali\0"
"Southern Kurdish\0"
"Southern Sami\0"
"Southern Sotho\0"
"South Ndebele\0"
"Spanish\0"
"Standard Moroccan Tamazight\0"
"Sundanese\0"
"Swahili\0"
"Swati\0"
"Swedish\0"
"Swiss German\0"
"Syriac\0"
"Tachelhit\0"
"Tahitian\0"
"Tai Dam\0"
"Taita\0"
"Tajik\0"
"Tamil\0"
"Taroko\0"
"Tasawaq\0"
"Tatar\0"
"Telugu\0"
"Teso\0"
"Thai\0"
"Tibetan\0"
"Tigre\0"
"Tigrinya\0"
"Tokelau\0"
"Tok Pisin\0"
"Tongan\0"
"Tsonga\0"
"Tswana\0"
"Turkish\0"
"Turkmen\0"
"Tuvalu\0"
"Tyap\0"
"Ugaritic\0"
"Ukrainian\0"
"Upper Sorbian\0"
"Urdu\0"
"Uyghur\0"
"Uzbek\0"
"Vai\0"
"Venda\0"
"Vietnamese\0"
"Volapük\0"
"Vunjo\0"
"Walloon\0"
"Walser\0"
"Warlpiri\0"
"Welsh\0"
"Western Balochi\0"
"Western Frisian\0"
"Wolaytta\0"
"Wolof\0"
"Xhosa\0"
"Yangben\0"
"Yiddish\0"
"Yoruba\0"
"Zarma\0"
"Zhuang\0"
"Zulu\0"
"Kaingang\0"
"Nheengatu\0"
"Haryanvi\0"
"Northern Frisian\0"
"Rajasthani\0"
"Moksha\0"
"Toki Pona\0"
"Pijin\0"
"Obolo\0"
"Baluchi\0"
"Ligurian\0"
"Rohingya\0"
"Torwali\0"
"Anii\0"
"Kangri\0"
"Venetian\0"
"Kuvi\0"
;

static constexpr quint16 language_name_index[] = {
     0, // AnyLanguage
     8, // C
    10, // Abkhazian
    20, // Afar
    25, // Afrikaans
    35, // Aghem
    41, // Akan
    46, // Akkadian
    55, // Akoose
    62, // Albanian
    71, // American Sign Language
    94, // Amharic
   102, // Ancient Egyptian
   119, // Ancient Greek
   133, // Arabic
   140, // Aragonese
   150, // Aramaic
   158, // Armenian
   167, // Assamese
   176, // Asturian
   185, // Asu
   189, // Atsam
   195, // Avaric
   202, // Avestan
   210, // Aymara
   217, // Azerbaijani
   229, // Bafia
   235, // Balinese
   244, // Bambara
   252, // Bamun
   258, // Bangla
   265, // Basaa
   271, // Bashkir
   279, // Basque
   286, // Batak Toba
   297, // Belarusian
   308, // Bemba
   314, // Bena
   319, // Bhojpuri
   328, // Bislama
   336, // Blin
   341, // Bodo
   346, // Bosnian
   354, // Breton
   361, // Buginese
   370, // Bulgarian
   380, // Burmese
   388, // Cantonese
   398, // Catalan
   406, // Cebuano
   414, // Central Atlas Tamazight
   438, // Central Kurdish
   454, // Chakma
   461, // Chamorro
   470, // Chechen
   478, // Cherokee
   487, // Chickasaw
   497, // Chiga
   503, // Chinese
   511, // Church
   525, // Chuvash
   533, // Colognian
   543, // Coptic
   550, // Cornish
   558, // Corsican
   567, // Cree
   572, // Croatian
   581, // Czech
   587, // Danish
   594, // Divehi
   601, // Dogri
   607, // Duala
   613, // Dutch
   619, // Dzongkha
   628, // Embu
   633, // English
   641, // Erzya
   647, // Esperanto
   657, // Estonian
   666, // Ewe
   670, // Ewondo
   677, // Faroese
   685, // Fijian
   692, // Filipino
   701, // Finnish
   709, // French
   716, // Friulian
   725, // Fulah
   730, // Gaelic
   746, // Ga
   749, // Galician
   758, // Ganda
   764, // Geez
   769, // Georgian
   778, // German
   785, // Gothic
   792, // Greek
   798, // Guarani
   806, // Gujarati
   815, // Gusii
   821, // Haitian
   836, // Hausa
   842, // Hawaiian
   851, // Hebrew
   858, // Herero
   865, // Hindi
   871, // Hiri Motu
   881, // Hungarian
   891, // Icelandic
   901, // Ido
   905, // Igbo
   910, // Inari Sami
   921, // Indonesian
   932, // Ingush
   939, // Interlingua
   951, // Interlingue
   963, // Inuktitut
   973, // Inupiaq
   981, // Irish
   987, // Italian
   995, // Japanese
  1004, // Javanese
  1013, // Jju
  1017, // Jola-Fonyi
  1028, // Kabuverdianu
  1041, // Kabyle
  1048, // Kako
  1053, // Kalaallisut
  1065, // Kalenjin
  1074, // Kamba
  1080, // Kannada
  1088, // Kanuri
  1095, // Kashmiri
  1104, // Kazakh
  1111, // Kenyang
  1119, // Khmer
  1125, // Kiche
  1135, // Kikuyu
  1142, // Kinyarwanda
  1154, // Komi
  1159, // Kongo
  1165, // Konkani
  1173, // Korean
  1180, // Koro
  1185, // Koyraboro Senni
  1201, // Koyra Chiini
  1214, // Kpelle
  1221, // Kuanyama
  1230, // Kurdish
  1238, // Kwasio
  1245, // Kyrgyz
  1252, // Lakota
  1259, // Langi
  1265, // Lao
  1269, // Latin
  1275, // Latvian
  1283, // Lezghian
  1292, // Limburgish
  1303, // Lingala
  1311, // Literary Chinese
  1328, // Lithuanian
  1339, // Lojban
  1346, // Lower Sorbian
  1360, // Low German
  1371, // Luba-Katanga
  1384, // Lule Sami
  1394, // Luo
  1398, // Luxembourgish
  1412, // Luyia
  1418, // Macedonian
  1429, // Machame
  1437, // Maithili
  1446, // Makhuwa-Meetto
  1461, // Makonde
  1469, // Malagasy
  1478, // Malayalam
  1488, // Malay
  1494, // Maltese
  1502, // Mandingo
  1511, // Manipuri
  1520, // Manx
  1525, // Maori
  1532, // Mapuche
  1540, // Marathi
  1548, // Marshallese
  1560, // Masai
  1566, // Mazanderani
  1578, // Mende
  1584, // Meru
  1589, // Meta
  1596, // Mohawk
  1603, // Mongolian
  1613, // Morisyen
  1622, // Mundang
  1630, // Muscogee
  1639, // Nama
  1644, // Nauru
  1650, // Navajo
  1657, // Ndonga
  1664, // Nepali
  1671, // Newari
  1678, // Ngiemboon
  1688, // Ngomba
  1695, // Nigerian Pidgin
  1711, // Nko
  1718, // Northern Luri
  1732, // Northern Sami
  1746, // Northern Sotho
  1761, // North Ndebele
  1775, // Norwegian Bokmal
  1793, // Norwegian Nynorsk
  1811, // Nuer
  1816, // Nyanja
  1823, // Nyankole
  1832, // Occitan
  1840, // Odia
  1845, // Ojibwa
  1852, // Old Irish
  1862, // Old Norse
  1872, // Old Persian
  1884, // Oromo
  1890, // Osage
  1896, // Ossetic
  1904, // Pahlavi
  1912, // Palauan
  1920, // Pali
  1925, // Papiamento
  1936, // Pashto
  1943, // Persian
  1951, // Phoenician
  1962, // Polish
  1969, // Portuguese
  1980, // Prussian
  1989, // Punjabi
  1997, // Quechua
  2005, // Romanian
  2014, // Romansh
  2022, // Rombo
  2028, // Rundi
  2034, // Russian
  2042, // Rwa
  2046, // Saho
  2051, // Sakha
  2057, // Samburu
  2065, // Samoan
  2072, // Sango
  2078, // Sangu
  2084, // Sanskrit
  2093, // Santali
  2101, // Sardinian
  2111, // Saurashtra
  2122, // Sena
  2127, // Serbian
  2135, // Shambala
  2144, // Shona
  2150, // Sichuan Yi
  2161, // Sicilian
  2170, // Sidamo
  2177, // Silesian
  2186, // Sindhi
  2193, // Sinhala
  2201, // Skolt Sami
  2212, // Slovak
  2219, // Slovenian
  2229, // Soga
  2234, // Somali
  2241, // Southern Kurdish
  2258, // Southern Sami
  2272, // Southern Sotho
  2287, // South Ndebele
  2301, // Spanish
  2309, // Standard Moroccan Tamazight
  2337, // Sundanese
  2347, // Swahili
  2355, // Swati
  2361, // Swedish
  2369, // Swiss German
  2382, // Syriac
  2389, // Tachelhit
  2399, // Tahitian
  2408, // Tai Dam
  2416, // Taita
  2422, // Tajik
  2428, // Tamil
  2434, // Taroko
  2441, // Tasawaq
  2449, // Tatar
  2455, // Telugu
  2462, // Teso
  2467, // Thai
  2472, // Tibetan
  2480, // Tigre
  2486, // Tigrinya
  2495, // Tokelau
  2503, // Tok Pisin
  2513, // Tongan
  2520, // Tsonga
  2527, // Tswana
  2534, // Turkish
  2542, // Turkmen
  2550, // Tuvalu
  2557, // Tyap
  2562, // Ugaritic
  2571, // Ukrainian
  2581, // Upper Sorbian
  2595, // Urdu
  2600, // Uyghur
  2607, // Uzbek
  2613, // Vai
  2617, // Venda
  2623, // Vietnamese
  2634, // Volapuk
  2643, // Vunjo
  2649, // Walloon
  2657, // Walser
  2664, // Warlpiri
  2673, // Welsh
  2679, // Western Balochi
  2695, // Western Frisian
  2711, // Wolaytta
  2720, // Wolof
  2726, // Xhosa
  2732, // Yangben
  2740, // Yiddish
  2748, // Yoruba
  2755, // Zarma
  2761, // Zhuang
  2768, // Zulu
  2773, // Kaingang
  2782, // Nheengatu
  2792, // Haryanvi
  2801, // Northern Frisian
  2818, // Rajasthani
  2829, // Moksha
  2836, // Toki Pona
  2846, // Pijin
  2852, // Obolo
  2858, // Baluchi
  2866, // Ligurian
  2875, // Rohingya
  2884, // Torwali
  2892, // Anii
  2897, // Kangri
  2904, // Venetian
  2913, // Kuvi
};

static constexpr char script_name_list[] =
"Default\0"
"Adlam\0"
"Ahom\0"
"Anatolian Hieroglyphs\0"
"Arabic\0"
"Armenian\0"
"Avestan\0"
"Balinese\0"
"Bamum\0"
"Bangla\0"
"Bassa Vah\0"
"Batak\0"
"Bhaiksuki\0"
"Bopomofo\0"
"Brahmi\0"
"Braille\0"
"Buginese\0"
"Buhid\0"
"Unified Canadian Aboriginal Syllabics\0"
"Carian\0"
"Caucasian Albanian\0"
"Chakma\0"
"Cham\0"
"Cherokee\0"
"Coptic\0"
"Sumero-Akkadian Cuneiform\0"
"Cypriot\0"
"Cyrillic\0"
"Deseret\0"
"Devanagari\0"
"Duployan shorthand\0"
"Egyptian hieroglyphs\0"
"Elbasan\0"
"Ethiopic\0"
"Fraser\0"
"Georgian\0"
"Glagolitic\0"
"Gothic\0"
"Grantha\0"
"Greek\0"
"Gujarati\0"
"Gurmukhi\0"
"Hangul\0"
"Han\0"
"Hanunoo\0"
"Han with Bopomofo\0"
"Hatran\0"
"Hebrew\0"
"Hiragana\0"
"Imperial Aramaic\0"
"Inscriptional Pahlavi\0"
"Inscriptional Parthian\0"
"Jamo\0"
"Japanese\0"
"Javanese\0"
"Kaithi\0"
"Kannada\0"
"Katakana\0"
"Kayah Li\0"
"Kharoshthi\0"
"Khmer\0"
"Khojki\0"
"Khudawadi\0"
"Korean\0"
"Lanna\0"
"Lao\0"
"Latin\0"
"Lepcha\0"
"Limbu\0"
"Linear A\0"
"Linear B\0"
"Lycian\0"
"Lydian\0"
"Mahajani\0"
"Malayalam\0"
"Mandaean\0"
"Manichaean\0"
"Marchen\0"
"Meitei Mayek\0"
"Mende\0"
"Meroitic Cursive\0"
"Meroitic\0"
"Modi\0"
"Mongolian\0"
"Mro\0"
"Multani\0"
"Myanmar\0"
"Nabataean\0"
"Newa\0"
"New Tai Lue\0"
"N’Ko\0"
"Odia\0"
"Ogham\0"
"Ol Chiki\0"
"Old Hungarian\0"
"Old Italic\0"
"Old North Arabian\0"
"Old Permic\0"
"Old Persian\0"
"Old South Arabian\0"
"Orkhon\0"
"Osage\0"
"Osmanya\0"
"Pahawh Hmong\0"
"Palmyrene\0"
"Pau Cin Hau\0"
"Phags-pa\0"
"Phoenician\0"
"Pollard Phonetic\0"
"Psalter Pahlavi\0"
"Rejang\0"
"Runic\0"
"Samaritan\0"
"Saurashtra\0"
"Sharada\0"
"Shavian\0"
"Siddham\0"
"SignWriting\0"
"Simplified Han\0"
"Sinhala\0"
"Sora Sompeng\0"
"Sundanese\0"
"Syloti Nagri\0"
"Syriac\0"
"Tagalog\0"
"Tagbanwa\0"
"Tai Le\0"
"Tai Viet\0"
"Takri\0"
"Tamil\0"
"Tangut\0"
"Telugu\0"
"Thaana\0"
"Thai\0"
"Tibetan\0"
"Tifinagh\0"
"Tirhuta\0"
"Traditional Han\0"
"Ugaritic\0"
"Vai\0"
"Varang Kshiti\0"
"Yi\0"
"Hanifi Rohingya\0"
;

static constexpr quint16 script_name_index[] = {
     0, // AnyScript
     8, // Adlam
    14, // Ahom
    19, // Anatolian Hieroglyphs
    41, // Arabic
    48, // Armenian
    57, // Avestan
    65, // Balinese
    74, // Bamum
    80, // Bangla
    87, // Bassa Vah
    97, // Batak
   103, // Bhaiksuki
   113, // Bopomofo
   122, // Brahmi
   129, // Braille
   137, // Buginese
   146, // Buhid
   152, // Canadian Aboriginal
   190, // Carian
   197, // Caucasian Albanian
   216, // Chakma
   223, // Cham
   228, // Cherokee
   237, // Coptic
   244, // Cuneiform
   270, // Cypriot
   278, // Cyrillic
   287, // Deseret
   295, // Devanagari
   306, // Duployan
   325, // Egyptian hieroglyphs
   346, // Elbasan
   354, // Ethiopic
   363, // Fraser
   370, // Georgian
   379, // Glagolitic
   390, // Gothic
   397, // Grantha
   405, // Greek
   411, // Gujarati
   420, // Gurmukhi
   429, // Hangul
   436, // Han
   440, // Hanunoo
   448, // Han with Bopomofo
   466, // Hatran
   473, // Hebrew
   480, // Hiragana
   489, // Imperial Aramaic
   506, // Inscriptional Pahlavi
   528, // Inscriptional Parthian
   551, // Jamo
   556, // Japanese
   565, // Javanese
   574, // Kaithi
   581, // Kannada
   589, // Katakana
   598, // Kayah Li
   607, // Kharoshthi
   618, // Khmer
   624, // Khojki
   631, // Khudawadi
   641, // Korean
   648, // Lanna
   654, // Lao
   658, // Latin
   664, // Lepcha
   671, // Limbu
   677, // Linear A
   686, // Linear B
   695, // Lycian
   702, // Lydian
   709, // Mahajani
   718, // Malayalam
   728, // Mandaean
   737, // Manichaean
   748, // Marchen
   756, // Meitei Mayek
   769, // Mende
   775, // Meroitic Cursive
   792, // Meroitic
   801, // Modi
   806, // Mongolian
   816, // Mro
   820, // Multani
   828, // Myanmar
   836, // Nabataean
   846, // Newa
   851, // New Tai Lue
   863, // Nko
   870, // Odia
   875, // Ogham
   881, // Ol Chiki
   890, // Old Hungarian
   904, // Old Italic
   915, // Old North Arabian
   933, // Old Permic
   944, // Old Persian
   956, // Old South Arabian
   974, // Orkhon
   981, // Osage
   987, // Osmanya
   995, // Pahawh Hmong
  1008, // Palmyrene
  1018, // Pau Cin Hau
  1030, // Phags-pa
  1039, // Phoenician
  1050, // Pollard Phonetic
  1067, // Psalter Pahlavi
  1083, // Rejang
  1090, // Runic
  1096, // Samaritan
  1106, // Saurashtra
  1117, // Sharada
  1125, // Shavian
  1133, // Siddham
  1141, // SignWriting
  1153, // Simplified Han
  1168, // Sinhala
  1176, // Sora Sompeng
  1189, // Sundanese
  1199, // Syloti Nagri
  1212, // Syriac
  1219, // Tagalog
  1227, // Tagbanwa
  1236, // Tai Le
  1243, // Tai Viet
  1252, // Takri
  1258, // Tamil
  1264, // Tangut
  1271, // Telugu
  1278, // Thaana
  1285, // Thai
  1290, // Tibetan
  1298, // Tifinagh
  1307, // Tirhuta
  1315, // Traditional Han
  1331, // Ugaritic
  1340, // Vai
  1344, // Varang Kshiti
  1358, // Yi
  1361, // Hanifi
};

static constexpr char territory_name_list[] =
"Default\0"
"Afghanistan\0"
"Åland Islands\0"
"Albania\0"
"Algeria\0"
"American Samoa\0"
"Andorra\0"
"Angola\0"
"Anguilla\0"
"Antarctica\0"
"Antigua & Barbuda\0"
"Argentina\0"
"Armenia\0"
"Aruba\0"
"Ascension Island\0"
"Australia\0"
"Austria\0"
"Azerbaijan\0"
"Bahamas\0"
"Bahrain\0"
"Bangladesh\0"
"Barbados\0"
"Belarus\0"
"Belgium\0"
"Belize\0"
"Benin\0"
"Bermuda\0"
"Bhutan\0"
"Bolivia\0"
"Bosnia & Herzegovina\0"
"Botswana\0"
"Bouvet Island\0"
"Brazil\0"
"British Indian Ocean Territory\0"
"British Virgin Islands\0"
"Brunei\0"
"Bulgaria\0"
"Burkina Faso\0"
"Burundi\0"
"Cambodia\0"
"Cameroon\0"
"Canada\0"
"Canary Islands\0"
"Cape Verde\0"
"Caribbean Netherlands\0"
"Cayman Islands\0"
"Central African Republic\0"
"Ceuta & Melilla\0"
"Chad\0"
"Chile\0"
"China\0"
"Christmas Island\0"
"Clipperton Island\0"
"Cocos (Keeling) Islands\0"
"Colombia\0"
"Comoros\0"
"Congo - Brazzaville\0"
"Congo - Kinshasa\0"
"Cook Islands\0"
"Costa Rica\0"
"Croatia\0"
"Cuba\0"
"Curaçao\0"
"Cyprus\0"
"Czechia\0"
"Denmark\0"
"Diego Garcia\0"
"Djibouti\0"
"Dominica\0"
"Dominican Republic\0"
"Ecuador\0"
"Egypt\0"
"El Salvador\0"
"Equatorial Guinea\0"
"Eritrea\0"
"Estonia\0"
"Eswatini\0"
"Ethiopia\0"
"Europe\0"
"European Union\0"
"Falkland Islands\0"
"Faroe Islands\0"
"Fiji\0"
"Finland\0"
"France\0"
"French Guiana\0"
"French Polynesia\0"
"French Southern Territories\0"
"Gabon\0"
"Gambia\0"
"Georgia\0"
"Germany\0"
"Ghana\0"
"Gibraltar\0"
"Greece\0"
"Greenland\0"
"Grenada\0"
"Guadeloupe\0"
"Guam\0"
"Guatemala\0"
"Guernsey\0"
"Guinea-Bissau\0"
"Guinea\0"
"Guyana\0"
"Haiti\0"
"Heard & McDonald Islands\0"
"Honduras\0"
"Hong Kong SAR China\0"
"Hungary\0"
"Iceland\0"
"India\0"
"Indonesia\0"
"Iran\0"
"Iraq\0"
"Ireland\0"
"Isle of Man\0"
"Israel\0"
"Italy\0"
"Côte d’Ivoire\0" // Ivory Coast
"Jamaica\0"
"Japan\0"
"Jersey\0"
"Jordan\0"
"Kazakhstan\0"
"Kenya\0"
"Kiribati\0"
"Kosovo\0"
"Kuwait\0"
"Kyrgyzstan\0"
"Laos\0"
"Latin America\0"
"Latvia\0"
"Lebanon\0"
"Lesotho\0"
"Liberia\0"
"Libya\0"
"Liechtenstein\0"
"Lithuania\0"
"Luxembourg\0"
"Macao SAR China\0"
"North Macedonia\0"
"Madagascar\0"
"Malawi\0"
"Malaysia\0"
"Maldives\0"
"Mali\0"
"Malta\0"
"Marshall Islands\0"
"Martinique\0"
"Mauritania\0"
"Mauritius\0"
"Mayotte\0"
"Mexico\0"
"Micronesia\0"
"Moldova\0"
"Monaco\0"
"Mongolia\0"
"Montenegro\0"
"Montserrat\0"
"Morocco\0"
"Mozambique\0"
"Myanmar (Burma)\0"
"Namibia\0"
"Nauru\0"
"Nepal\0"
"Netherlands\0"
"New Caledonia\0"
"New Zealand\0"
"Nicaragua\0"
"Nigeria\0"
"Niger\0"
"Niue\0"
"Norfolk Island\0"
"Northern Mariana Islands\0"
"North Korea\0"
"Norway\0"
"Oman\0"
"Outlying Oceania\0"
"Pakistan\0"
"Palau\0"
"Palestinian Territories\0"
"Panama\0"
"Papua New Guinea\0"
"Paraguay\0"
"Peru\0"
"Philippines\0"
"Pitcairn Islands\0"
"Poland\0"
"Portugal\0"
"Puerto Rico\0"
"Qatar\0"
"Réunion\0"
"Romania\0"
"Russia\0"
"Rwanda\0"
"St. Barthélemy\0"
"St. Helena\0"
"St. Kitts & Nevis\0"
"St. Lucia\0"
"St. Martin\0"
"St. Pierre & Miquelon\0"
"St. Vincent & Grenadines\0"
"Samoa\0"
"San Marino\0"
"São Tomé & Príncipe\0"
"Saudi Arabia\0"
"Senegal\0"
"Serbia\0"
"Seychelles\0"
"Sierra Leone\0"
"Singapore\0"
"Sint Maarten\0"
"Slovakia\0"
"Slovenia\0"
"Solomon Islands\0"
"Somalia\0"
"South Africa\0"
"South Georgia & South Sandwich Islands\0"
"South Korea\0"
"South Sudan\0"
"Spain\0"
"Sri Lanka\0"
"Sudan\0"
"Suriname\0"
"Svalbard & Jan Mayen\0"
"Sweden\0"
"Switzerland\0"
"Syria\0"
"Taiwan\0"
"Tajikistan\0"
"Tanzania\0"
"Thailand\0"
"Timor-Leste\0"
"Togo\0"
"Tokelau\0"
"Tonga\0"
"Trinidad & Tobago\0"
"Tristan da Cunha\0"
"Tunisia\0"
"Türkiye\0" // Turkey
"Turkmenistan\0"
"Turks & Caicos Islands\0"
"Tuvalu\0"
"Uganda\0"
"Ukraine\0"
"United Arab Emirates\0"
"United Kingdom\0"
"U.S. Outlying Islands\0"
"United States\0"
"U.S. Virgin Islands\0"
"Uruguay\0"
"Uzbekistan\0"
"Vanuatu\0"
"Vatican City\0"
"Venezuela\0"
"Vietnam\0"
"Wallis & Futuna\0"
"Western Sahara\0"
"world\0"
"Yemen\0"
"Zambia\0"
"Zimbabwe\0"
;

static constexpr quint16 territory_name_index[] = {
     0, // AnyTerritory
     8, // Afghanistan
    20, // Aland Islands
    35, // Albania
    43, // Algeria
    51, // American Samoa
    66, // Andorra
    74, // Angola
    81, // Anguilla
    90, // Antarctica
   101, // Antigua and Barbuda
   119, // Argentina
   129, // Armenia
   137, // Aruba
   143, // Ascension Island
   160, // Australia
   170, // Austria
   178, // Azerbaijan
   189, // Bahamas
   197, // Bahrain
   205, // Bangladesh
   216, // Barbados
   225, // Belarus
   233, // Belgium
   241, // Belize
   248, // Benin
   254, // Bermuda
   262, // Bhutan
   269, // Bolivia
   277, // Bosnia and Herzegovina
   298, // Botswana
   307, // Bouvet Island
   321, // Brazil
   328, // British Indian Ocean Territory
   359, // British Virgin Islands
   382, // Brunei
   389, // Bulgaria
   398, // Burkina Faso
   411, // Burundi
   419, // Cambodia
   428, // Cameroon
   437, // Canada
   444, // Canary Islands
   459, // Cape Verde
   470, // Caribbean Netherlands
   492, // Cayman Islands
   507, // Central African Republic
   532, // Ceuta and Melilla
   548, // Chad
   553, // Chile
   559, // China
   565, // Christmas Island
   582, // Clipperton Island
   600, // Cocos Islands
   624, // Colombia
   633, // Comoros
   641, // Congo - Brazzaville
   661, // Congo - Kinshasa
   678, // Cook Islands
   691, // Costa Rica
   702, // Croatia
   710, // Cuba
   715, // Curacao
   724, // Cyprus
   731, // Czechia
   739, // Denmark
   747, // Diego Garcia
   760, // Djibouti
   769, // Dominica
   778, // Dominican Republic
   797, // Ecuador
   805, // Egypt
   811, // El Salvador
   823, // Equatorial Guinea
   841, // Eritrea
   849, // Estonia
   857, // Eswatini
   866, // Ethiopia
   875, // Europe
   882, // European Union
   897, // Falkland Islands
   914, // Faroe Islands
   928, // Fiji
   933, // Finland
   941, // France
   948, // French Guiana
   962, // French Polynesia
   979, // French Southern Territories
  1007, // Gabon
  1013, // Gambia
  1020, // Georgia
  1028, // Germany
  1036, // Ghana
  1042, // Gibraltar
  1052, // Greece
  1059, // Greenland
  1069, // Grenada
  1077, // Guadeloupe
  1088, // Guam
  1093, // Guatemala
  1103, // Guernsey
  1112, // Guinea-Bissau
  1126, // Guinea
  1133, // Guyana
  1140, // Haiti
  1146, // Heard and McDonald Islands
  1171, // Honduras
  1180, // Hong Kong
  1200, // Hungary
  1208, // Iceland
  1216, // India
  1222, // Indonesia
  1232, // Iran
  1237, // Iraq
  1242, // Ireland
  1250, // Isle of Man
  1262, // Israel
  1269, // Italy
  1275, // Ivory Coast
  1292, // Jamaica
  1300, // Japan
  1306, // Jersey
  1313, // Jordan
  1320, // Kazakhstan
  1331, // Kenya
  1337, // Kiribati
  1346, // Kosovo
  1353, // Kuwait
  1360, // Kyrgyzstan
  1371, // Laos
  1376, // Latin America
  1390, // Latvia
  1397, // Lebanon
  1405, // Lesotho
  1413, // Liberia
  1421, // Libya
  1427, // Liechtenstein
  1441, // Lithuania
  1451, // Luxembourg
  1462, // Macao
  1478, // Macedonia
  1494, // Madagascar
  1505, // Malawi
  1512, // Malaysia
  1521, // Maldives
  1530, // Mali
  1535, // Malta
  1541, // Marshall Islands
  1558, // Martinique
  1569, // Mauritania
  1580, // Mauritius
  1590, // Mayotte
  1598, // Mexico
  1605, // Micronesia
  1616, // Moldova
  1624, // Monaco
  1631, // Mongolia
  1640, // Montenegro
  1651, // Montserrat
  1662, // Morocco
  1670, // Mozambique
  1681, // Myanmar
  1697, // Namibia
  1705, // Nauru
  1711, // Nepal
  1717, // Netherlands
  1729, // New Caledonia
  1743, // New Zealand
  1755, // Nicaragua
  1765, // Nigeria
  1773, // Niger
  1779, // Niue
  1784, // Norfolk Island
  1799, // Northern Mariana Islands
  1824, // North Korea
  1836, // Norway
  1843, // Oman
  1848, // Outlying Oceania
  1865, // Pakistan
  1874, // Palau
  1880, // Palestinian Territories
  1904, // Panama
  1911, // Papua New Guinea
  1928, // Paraguay
  1937, // Peru
  1942, // Philippines
  1954, // Pitcairn
  1971, // Poland
  1978, // Portugal
  1987, // Puerto Rico
  1999, // Qatar
  2005, // Reunion
  2014, // Romania
  2022, // Russia
  2029, // Rwanda
  2036, // Saint Barthelemy
  2052, // Saint Helena
  2063, // Saint Kitts and Nevis
  2081, // Saint Lucia
  2091, // Saint Martin
  2102, // Saint Pierre and Miquelon
  2124, // Saint Vincent and Grenadines
  2149, // Samoa
  2155, // San Marino
  2166, // Sao Tome and Principe
  2189, // Saudi Arabia
  2202, // Senegal
  2210, // Serbia
  2217, // Seychelles
  2228, // Sierra Leone
  2241, // Singapore
  2251, // Sint Maarten
  2264, // Slovakia
  2273, // Slovenia
  2282, // Solomon Islands
  2298, // Somalia
  2306, // South Africa
  2319, // South Georgia and South Sandwich Islands
  2358, // South Korea
  2370, // South Sudan
  2382, // Spain
  2388, // Sri Lanka
  2398, // Sudan
  2404, // Suriname
  2413, // Svalbard and Jan Mayen
  2434, // Sweden
  2441, // Switzerland
  2453, // Syria
  2459, // Taiwan
  2466, // Tajikistan
  2477, // Tanzania
  2486, // Thailand
  2495, // Timor-Leste
  2507, // Togo
  2512, // Tokelau
  2520, // Tonga
  2526, // Trinidad and Tobago
  2544, // Tristan da Cunha
  2561, // Tunisia
  2569, // Turkey
  2578, // Turkmenistan
  2591, // Turks and Caicos Islands
  2614, // Tuvalu
  2621, // Uganda
  2628, // Ukraine
  2636, // United Arab Emirates
  2657, // United Kingdom
  2672, // United States Outlying Islands
  2694, // United States
  2708, // United States Virgin Islands
  2728, // Uruguay
  2736, // Uzbekistan
  2747, // Vanuatu
  2755, // Vatican City
  2768, // Venezuela
  2778, // Vietnam
  2786, // Wallis and Futuna
  2802, // Western Sahara
  2817, // world
  2823, // Yemen
  2829, // Zambia
  2836, // Zimbabwe
};

constexpr std::array<LanguageCodeEntry, 345> languageCodeList {
    LanguageCodeEntry {{},         {'u', 'n', 'd'}, {'u', 'n', 'd'}, {'u', 'n', 'd'}}, // AnyLanguage
    LanguageCodeEntry {{},         {'u', 'n', 'd'}, {'u', 'n', 'd'}, {'u', 'n', 'd'}}, // C
    LanguageCodeEntry {{'a', 'b'}, {'a', 'b', 'k'}, {'a', 'b', 'k'}, {'a', 'b', 'k'}}, // Abkhazian
    LanguageCodeEntry {{'a', 'a'}, {'a', 'a', 'r'}, {'a', 'a', 'r'}, {'a', 'a', 'r'}}, // Afar
    LanguageCodeEntry {{'a', 'f'}, {'a', 'f', 'r'}, {'a', 'f', 'r'}, {'a', 'f', 'r'}}, // Afrikaans
    LanguageCodeEntry {{},         {},              {},              {'a', 'g', 'q'}}, // Aghem
    LanguageCodeEntry {{'a', 'k'}, {'a', 'k', 'a'}, {'a', 'k', 'a'}, {'a', 'k', 'a'}}, // Akan
    LanguageCodeEntry {{},         {'a', 'k', 'k'}, {'a', 'k', 'k'}, {'a', 'k', 'k'}}, // Akkadian
    LanguageCodeEntry {{},         {},              {},              {'b', 's', 's'}}, // Akoose
    LanguageCodeEntry {{'s', 'q'}, {'a', 'l', 'b'}, {'s', 'q', 'i'}, {'s', 'q', 'i'}}, // Albanian
    LanguageCodeEntry {{},         {},              {},              {'a', 's', 'e'}}, // American Sign Language
    LanguageCodeEntry {{'a', 'm'}, {'a', 'm', 'h'}, {'a', 'm', 'h'}, {'a', 'm', 'h'}}, // Amharic
    LanguageCodeEntry {{},         {'e', 'g', 'y'}, {'e', 'g', 'y'}, {'e', 'g', 'y'}}, // Ancient Egyptian
    LanguageCodeEntry {{},         {'g', 'r', 'c'}, {'g', 'r', 'c'}, {'g', 'r', 'c'}}, // Ancient Greek
    LanguageCodeEntry {{'a', 'r'}, {'a', 'r', 'a'}, {'a', 'r', 'a'}, {'a', 'r', 'a'}}, // Arabic
    LanguageCodeEntry {{'a', 'n'}, {'a', 'r', 'g'}, {'a', 'r', 'g'}, {'a', 'r', 'g'}}, // Aragonese
    LanguageCodeEntry {{},         {'a', 'r', 'c'}, {'a', 'r', 'c'}, {'a', 'r', 'c'}}, // Aramaic
    LanguageCodeEntry {{'h', 'y'}, {'a', 'r', 'm'}, {'h', 'y', 'e'}, {'h', 'y', 'e'}}, // Armenian
    LanguageCodeEntry {{'a', 's'}, {'a', 's', 'm'}, {'a', 's', 'm'}, {'a', 's', 'm'}}, // Assamese
    LanguageCodeEntry {{},         {'a', 's', 't'}, {'a', 's', 't'}, {'a', 's', 't'}}, // Asturian
    LanguageCodeEntry {{},         {},              {},              {'a', 's', 'a'}}, // Asu
    LanguageCodeEntry {{},         {},              {},              {'c', 'c', 'h'}}, // Atsam
    LanguageCodeEntry {{'a', 'v'}, {'a', 'v', 'a'}, {'a', 'v', 'a'}, {'a', 'v', 'a'}}, // Avaric
    LanguageCodeEntry {{'a', 'e'}, {'a', 'v', 'e'}, {'a', 'v', 'e'}, {'a', 'v', 'e'}}, // Avestan
    LanguageCodeEntry {{'a', 'y'}, {'a', 'y', 'm'}, {'a', 'y', 'm'}, {'a', 'y', 'm'}}, // Aymara
    LanguageCodeEntry {{'a', 'z'}, {'a', 'z', 'e'}, {'a', 'z', 'e'}, {'a', 'z', 'e'}}, // Azerbaijani
    LanguageCodeEntry {{},         {},              {},              {'k', 's', 'f'}}, // Bafia
    LanguageCodeEntry {{},         {'b', 'a', 'n'}, {'b', 'a', 'n'}, {'b', 'a', 'n'}}, // Balinese
    LanguageCodeEntry {{'b', 'm'}, {'b', 'a', 'm'}, {'b', 'a', 'm'}, {'b', 'a', 'm'}}, // Bambara
    LanguageCodeEntry {{},         {},              {},              {'b', 'a', 'x'}}, // Bamun
    LanguageCodeEntry {{'b', 'n'}, {'b', 'e', 'n'}, {'b', 'e', 'n'}, {'b', 'e', 'n'}}, // Bangla
    LanguageCodeEntry {{},         {'b', 'a', 's'}, {'b', 'a', 's'}, {'b', 'a', 's'}}, // Basaa
    LanguageCodeEntry {{'b', 'a'}, {'b', 'a', 'k'}, {'b', 'a', 'k'}, {'b', 'a', 'k'}}, // Bashkir
    LanguageCodeEntry {{'e', 'u'}, {'b', 'a', 'q'}, {'e', 'u', 's'}, {'e', 'u', 's'}}, // Basque
    LanguageCodeEntry {{},         {},              {},              {'b', 'b', 'c'}}, // Batak Toba
    LanguageCodeEntry {{'b', 'e'}, {'b', 'e', 'l'}, {'b', 'e', 'l'}, {'b', 'e', 'l'}}, // Belarusian
    LanguageCodeEntry {{},         {'b', 'e', 'm'}, {'b', 'e', 'm'}, {'b', 'e', 'm'}}, // Bemba
    LanguageCodeEntry {{},         {},              {},              {'b', 'e', 'z'}}, // Bena
    LanguageCodeEntry {{},         {'b', 'h', 'o'}, {'b', 'h', 'o'}, {'b', 'h', 'o'}}, // Bhojpuri
    LanguageCodeEntry {{'b', 'i'}, {'b', 'i', 's'}, {'b', 'i', 's'}, {'b', 'i', 's'}}, // Bislama
    LanguageCodeEntry {{},         {'b', 'y', 'n'}, {'b', 'y', 'n'}, {'b', 'y', 'n'}}, // Blin
    LanguageCodeEntry {{},         {},              {},              {'b', 'r', 'x'}}, // Bodo
    LanguageCodeEntry {{'b', 's'}, {'b', 'o', 's'}, {'b', 'o', 's'}, {'b', 'o', 's'}}, // Bosnian
    LanguageCodeEntry {{'b', 'r'}, {'b', 'r', 'e'}, {'b', 'r', 'e'}, {'b', 'r', 'e'}}, // Breton
    LanguageCodeEntry {{},         {'b', 'u', 'g'}, {'b', 'u', 'g'}, {'b', 'u', 'g'}}, // Buginese
    LanguageCodeEntry {{'b', 'g'}, {'b', 'u', 'l'}, {'b', 'u', 'l'}, {'b', 'u', 'l'}}, // Bulgarian
    LanguageCodeEntry {{'m', 'y'}, {'b', 'u', 'r'}, {'m', 'y', 'a'}, {'m', 'y', 'a'}}, // Burmese
    LanguageCodeEntry {{},         {},              {},              {'y', 'u', 'e'}}, // Cantonese
    LanguageCodeEntry {{'c', 'a'}, {'c', 'a', 't'}, {'c', 'a', 't'}, {'c', 'a', 't'}}, // Catalan
    LanguageCodeEntry {{},         {'c', 'e', 'b'}, {'c', 'e', 'b'}, {'c', 'e', 'b'}}, // Cebuano
    LanguageCodeEntry {{},         {},              {},              {'t', 'z', 'm'}}, // Central Atlas Tamazight
    LanguageCodeEntry {{},         {},              {},              {'c', 'k', 'b'}}, // Central Kurdish
    LanguageCodeEntry {{},         {},              {},              {'c', 'c', 'p'}}, // Chakma
    LanguageCodeEntry {{'c', 'h'}, {'c', 'h', 'a'}, {'c', 'h', 'a'}, {'c', 'h', 'a'}}, // Chamorro
    LanguageCodeEntry {{'c', 'e'}, {'c', 'h', 'e'}, {'c', 'h', 'e'}, {'c', 'h', 'e'}}, // Chechen
    LanguageCodeEntry {{},         {'c', 'h', 'r'}, {'c', 'h', 'r'}, {'c', 'h', 'r'}}, // Cherokee
    LanguageCodeEntry {{},         {},              {},              {'c', 'i', 'c'}}, // Chickasaw
    LanguageCodeEntry {{},         {},              {},              {'c', 'g', 'g'}}, // Chiga
    LanguageCodeEntry {{'z', 'h'}, {'c', 'h', 'i'}, {'z', 'h', 'o'}, {'z', 'h', 'o'}}, // Chinese
    LanguageCodeEntry {{'c', 'u'}, {'c', 'h', 'u'}, {'c', 'h', 'u'}, {'c', 'h', 'u'}}, // Church
    LanguageCodeEntry {{'c', 'v'}, {'c', 'h', 'v'}, {'c', 'h', 'v'}, {'c', 'h', 'v'}}, // Chuvash
    LanguageCodeEntry {{},         {},              {},              {'k', 's', 'h'}}, // Colognian
    LanguageCodeEntry {{},         {'c', 'o', 'p'}, {'c', 'o', 'p'}, {'c', 'o', 'p'}}, // Coptic
    LanguageCodeEntry {{'k', 'w'}, {'c', 'o', 'r'}, {'c', 'o', 'r'}, {'c', 'o', 'r'}}, // Cornish
    LanguageCodeEntry {{'c', 'o'}, {'c', 'o', 's'}, {'c', 'o', 's'}, {'c', 'o', 's'}}, // Corsican
    LanguageCodeEntry {{'c', 'r'}, {'c', 'r', 'e'}, {'c', 'r', 'e'}, {'c', 'r', 'e'}}, // Cree
    LanguageCodeEntry {{'h', 'r'}, {'h', 'r', 'v'}, {'h', 'r', 'v'}, {'h', 'r', 'v'}}, // Croatian
    LanguageCodeEntry {{'c', 's'}, {'c', 'z', 'e'}, {'c', 'e', 's'}, {'c', 'e', 's'}}, // Czech
    LanguageCodeEntry {{'d', 'a'}, {'d', 'a', 'n'}, {'d', 'a', 'n'}, {'d', 'a', 'n'}}, // Danish
    LanguageCodeEntry {{'d', 'v'}, {'d', 'i', 'v'}, {'d', 'i', 'v'}, {'d', 'i', 'v'}}, // Divehi
    LanguageCodeEntry {{},         {'d', 'o', 'i'}, {'d', 'o', 'i'}, {'d', 'o', 'i'}}, // Dogri
    LanguageCodeEntry {{},         {'d', 'u', 'a'}, {'d', 'u', 'a'}, {'d', 'u', 'a'}}, // Duala
    LanguageCodeEntry {{'n', 'l'}, {'d', 'u', 't'}, {'n', 'l', 'd'}, {'n', 'l', 'd'}}, // Dutch
    LanguageCodeEntry {{'d', 'z'}, {'d', 'z', 'o'}, {'d', 'z', 'o'}, {'d', 'z', 'o'}}, // Dzongkha
    LanguageCodeEntry {{},         {},              {},              {'e', 'b', 'u'}}, // Embu
    LanguageCodeEntry {{'e', 'n'}, {'e', 'n', 'g'}, {'e', 'n', 'g'}, {'e', 'n', 'g'}}, // English
    LanguageCodeEntry {{},         {'m', 'y', 'v'}, {'m', 'y', 'v'}, {'m', 'y', 'v'}}, // Erzya
    LanguageCodeEntry {{'e', 'o'}, {'e', 'p', 'o'}, {'e', 'p', 'o'}, {'e', 'p', 'o'}}, // Esperanto
    LanguageCodeEntry {{'e', 't'}, {'e', 's', 't'}, {'e', 's', 't'}, {'e', 's', 't'}}, // Estonian
    LanguageCodeEntry {{'e', 'e'}, {'e', 'w', 'e'}, {'e', 'w', 'e'}, {'e', 'w', 'e'}}, // Ewe
    LanguageCodeEntry {{},         {'e', 'w', 'o'}, {'e', 'w', 'o'}, {'e', 'w', 'o'}}, // Ewondo
    LanguageCodeEntry {{'f', 'o'}, {'f', 'a', 'o'}, {'f', 'a', 'o'}, {'f', 'a', 'o'}}, // Faroese
    LanguageCodeEntry {{'f', 'j'}, {'f', 'i', 'j'}, {'f', 'i', 'j'}, {'f', 'i', 'j'}}, // Fijian
    LanguageCodeEntry {{},         {'f', 'i', 'l'}, {'f', 'i', 'l'}, {'f', 'i', 'l'}}, // Filipino
    LanguageCodeEntry {{'f', 'i'}, {'f', 'i', 'n'}, {'f', 'i', 'n'}, {'f', 'i', 'n'}}, // Finnish
    LanguageCodeEntry {{'f', 'r'}, {'f', 'r', 'e'}, {'f', 'r', 'a'}, {'f', 'r', 'a'}}, // French
    LanguageCodeEntry {{},         {'f', 'u', 'r'}, {'f', 'u', 'r'}, {'f', 'u', 'r'}}, // Friulian
    LanguageCodeEntry {{'f', 'f'}, {'f', 'u', 'l'}, {'f', 'u', 'l'}, {'f', 'u', 'l'}}, // Fulah
    LanguageCodeEntry {{'g', 'd'}, {'g', 'l', 'a'}, {'g', 'l', 'a'}, {'g', 'l', 'a'}}, // Gaelic
    LanguageCodeEntry {{},         {'g', 'a', 'a'}, {'g', 'a', 'a'}, {'g', 'a', 'a'}}, // Ga
    LanguageCodeEntry {{'g', 'l'}, {'g', 'l', 'g'}, {'g', 'l', 'g'}, {'g', 'l', 'g'}}, // Galician
    LanguageCodeEntry {{'l', 'g'}, {'l', 'u', 'g'}, {'l', 'u', 'g'}, {'l', 'u', 'g'}}, // Ganda
    LanguageCodeEntry {{},         {'g', 'e', 'z'}, {'g', 'e', 'z'}, {'g', 'e', 'z'}}, // Geez
    LanguageCodeEntry {{'k', 'a'}, {'g', 'e', 'o'}, {'k', 'a', 't'}, {'k', 'a', 't'}}, // Georgian
    LanguageCodeEntry {{'d', 'e'}, {'g', 'e', 'r'}, {'d', 'e', 'u'}, {'d', 'e', 'u'}}, // German
    LanguageCodeEntry {{},         {'g', 'o', 't'}, {'g', 'o', 't'}, {'g', 'o', 't'}}, // Gothic
    LanguageCodeEntry {{'e', 'l'}, {'g', 'r', 'e'}, {'e', 'l', 'l'}, {'e', 'l', 'l'}}, // Greek
    LanguageCodeEntry {{'g', 'n'}, {'g', 'r', 'n'}, {'g', 'r', 'n'}, {'g', 'r', 'n'}}, // Guarani
    LanguageCodeEntry {{'g', 'u'}, {'g', 'u', 'j'}, {'g', 'u', 'j'}, {'g', 'u', 'j'}}, // Gujarati
    LanguageCodeEntry {{},         {},              {},              {'g', 'u', 'z'}}, // Gusii
    LanguageCodeEntry {{'h', 't'}, {'h', 'a', 't'}, {'h', 'a', 't'}, {'h', 'a', 't'}}, // Haitian
    LanguageCodeEntry {{'h', 'a'}, {'h', 'a', 'u'}, {'h', 'a', 'u'}, {'h', 'a', 'u'}}, // Hausa
    LanguageCodeEntry {{},         {'h', 'a', 'w'}, {'h', 'a', 'w'}, {'h', 'a', 'w'}}, // Hawaiian
    LanguageCodeEntry {{'h', 'e'}, {'h', 'e', 'b'}, {'h', 'e', 'b'}, {'h', 'e', 'b'}}, // Hebrew
    LanguageCodeEntry {{'h', 'z'}, {'h', 'e', 'r'}, {'h', 'e', 'r'}, {'h', 'e', 'r'}}, // Herero
    LanguageCodeEntry {{'h', 'i'}, {'h', 'i', 'n'}, {'h', 'i', 'n'}, {'h', 'i', 'n'}}, // Hindi
    LanguageCodeEntry {{'h', 'o'}, {'h', 'm', 'o'}, {'h', 'm', 'o'}, {'h', 'm', 'o'}}, // Hiri Motu
    LanguageCodeEntry {{'h', 'u'}, {'h', 'u', 'n'}, {'h', 'u', 'n'}, {'h', 'u', 'n'}}, // Hungarian
    LanguageCodeEntry {{'i', 's'}, {'i', 'c', 'e'}, {'i', 's', 'l'}, {'i', 's', 'l'}}, // Icelandic
    LanguageCodeEntry {{'i', 'o'}, {'i', 'd', 'o'}, {'i', 'd', 'o'}, {'i', 'd', 'o'}}, // Ido
    LanguageCodeEntry {{'i', 'g'}, {'i', 'b', 'o'}, {'i', 'b', 'o'}, {'i', 'b', 'o'}}, // Igbo
    LanguageCodeEntry {{},         {'s', 'm', 'n'}, {'s', 'm', 'n'}, {'s', 'm', 'n'}}, // Inari Sami
    LanguageCodeEntry {{'i', 'd'}, {'i', 'n', 'd'}, {'i', 'n', 'd'}, {'i', 'n', 'd'}}, // Indonesian
    LanguageCodeEntry {{},         {'i', 'n', 'h'}, {'i', 'n', 'h'}, {'i', 'n', 'h'}}, // Ingush
    LanguageCodeEntry {{'i', 'a'}, {'i', 'n', 'a'}, {'i', 'n', 'a'}, {'i', 'n', 'a'}}, // Interlingua
    LanguageCodeEntry {{'i', 'e'}, {'i', 'l', 'e'}, {'i', 'l', 'e'}, {'i', 'l', 'e'}}, // Interlingue
    LanguageCodeEntry {{'i', 'u'}, {'i', 'k', 'u'}, {'i', 'k', 'u'}, {'i', 'k', 'u'}}, // Inuktitut
    LanguageCodeEntry {{'i', 'k'}, {'i', 'p', 'k'}, {'i', 'p', 'k'}, {'i', 'p', 'k'}}, // Inupiaq
    LanguageCodeEntry {{'g', 'a'}, {'g', 'l', 'e'}, {'g', 'l', 'e'}, {'g', 'l', 'e'}}, // Irish
    LanguageCodeEntry {{'i', 't'}, {'i', 't', 'a'}, {'i', 't', 'a'}, {'i', 't', 'a'}}, // Italian
    LanguageCodeEntry {{'j', 'a'}, {'j', 'p', 'n'}, {'j', 'p', 'n'}, {'j', 'p', 'n'}}, // Japanese
    LanguageCodeEntry {{'j', 'v'}, {'j', 'a', 'v'}, {'j', 'a', 'v'}, {'j', 'a', 'v'}}, // Javanese
    LanguageCodeEntry {{},         {},              {},              {'k', 'a', 'j'}}, // Jju
    LanguageCodeEntry {{},         {},              {},              {'d', 'y', 'o'}}, // Jola-Fonyi
    LanguageCodeEntry {{},         {},              {},              {'k', 'e', 'a'}}, // Kabuverdianu
    LanguageCodeEntry {{},         {'k', 'a', 'b'}, {'k', 'a', 'b'}, {'k', 'a', 'b'}}, // Kabyle
    LanguageCodeEntry {{},         {},              {},              {'k', 'k', 'j'}}, // Kako
    LanguageCodeEntry {{'k', 'l'}, {'k', 'a', 'l'}, {'k', 'a', 'l'}, {'k', 'a', 'l'}}, // Kalaallisut
    LanguageCodeEntry {{},         {},              {},              {'k', 'l', 'n'}}, // Kalenjin
    LanguageCodeEntry {{},         {'k', 'a', 'm'}, {'k', 'a', 'm'}, {'k', 'a', 'm'}}, // Kamba
    LanguageCodeEntry {{'k', 'n'}, {'k', 'a', 'n'}, {'k', 'a', 'n'}, {'k', 'a', 'n'}}, // Kannada
    LanguageCodeEntry {{'k', 'r'}, {'k', 'a', 'u'}, {'k', 'a', 'u'}, {'k', 'a', 'u'}}, // Kanuri
    LanguageCodeEntry {{'k', 's'}, {'k', 'a', 's'}, {'k', 'a', 's'}, {'k', 'a', 's'}}, // Kashmiri
    LanguageCodeEntry {{'k', 'k'}, {'k', 'a', 'z'}, {'k', 'a', 'z'}, {'k', 'a', 'z'}}, // Kazakh
    LanguageCodeEntry {{},         {},              {},              {'k', 'e', 'n'}}, // Kenyang
    LanguageCodeEntry {{'k', 'm'}, {'k', 'h', 'm'}, {'k', 'h', 'm'}, {'k', 'h', 'm'}}, // Khmer
    LanguageCodeEntry {{},         {},              {},              {'q', 'u', 'c'}}, // Kiche
    LanguageCodeEntry {{'k', 'i'}, {'k', 'i', 'k'}, {'k', 'i', 'k'}, {'k', 'i', 'k'}}, // Kikuyu
    LanguageCodeEntry {{'r', 'w'}, {'k', 'i', 'n'}, {'k', 'i', 'n'}, {'k', 'i', 'n'}}, // Kinyarwanda
    LanguageCodeEntry {{'k', 'v'}, {'k', 'o', 'm'}, {'k', 'o', 'm'}, {'k', 'o', 'm'}}, // Komi
    LanguageCodeEntry {{'k', 'g'}, {'k', 'o', 'n'}, {'k', 'o', 'n'}, {'k', 'o', 'n'}}, // Kongo
    LanguageCodeEntry {{},         {'k', 'o', 'k'}, {'k', 'o', 'k'}, {'k', 'o', 'k'}}, // Konkani
    LanguageCodeEntry {{'k', 'o'}, {'k', 'o', 'r'}, {'k', 'o', 'r'}, {'k', 'o', 'r'}}, // Korean
    LanguageCodeEntry {{},         {},              {},              {'k', 'f', 'o'}}, // Koro
    LanguageCodeEntry {{},         {},              {},              {'s', 'e', 's'}}, // Koyraboro Senni
    LanguageCodeEntry {{},         {},              {},              {'k', 'h', 'q'}}, // Koyra Chiini
    LanguageCodeEntry {{},         {'k', 'p', 'e'}, {'k', 'p', 'e'}, {'k', 'p', 'e'}}, // Kpelle
    LanguageCodeEntry {{'k', 'j'}, {'k', 'u', 'a'}, {'k', 'u', 'a'}, {'k', 'u', 'a'}}, // Kuanyama
    LanguageCodeEntry {{'k', 'u'}, {'k', 'u', 'r'}, {'k', 'u', 'r'}, {'k', 'u', 'r'}}, // Kurdish
    LanguageCodeEntry {{},         {},              {},              {'n', 'm', 'g'}}, // Kwasio
    LanguageCodeEntry {{'k', 'y'}, {'k', 'i', 'r'}, {'k', 'i', 'r'}, {'k', 'i', 'r'}}, // Kyrgyz
    LanguageCodeEntry {{},         {},              {},              {'l', 'k', 't'}}, // Lakota
    LanguageCodeEntry {{},         {},              {},              {'l', 'a', 'g'}}, // Langi
    LanguageCodeEntry {{'l', 'o'}, {'l', 'a', 'o'}, {'l', 'a', 'o'}, {'l', 'a', 'o'}}, // Lao
    LanguageCodeEntry {{'l', 'a'}, {'l', 'a', 't'}, {'l', 'a', 't'}, {'l', 'a', 't'}}, // Latin
    LanguageCodeEntry {{'l', 'v'}, {'l', 'a', 'v'}, {'l', 'a', 'v'}, {'l', 'a', 'v'}}, // Latvian
    LanguageCodeEntry {{},         {'l', 'e', 'z'}, {'l', 'e', 'z'}, {'l', 'e', 'z'}}, // Lezghian
    LanguageCodeEntry {{'l', 'i'}, {'l', 'i', 'm'}, {'l', 'i', 'm'}, {'l', 'i', 'm'}}, // Limburgish
    LanguageCodeEntry {{'l', 'n'}, {'l', 'i', 'n'}, {'l', 'i', 'n'}, {'l', 'i', 'n'}}, // Lingala
    LanguageCodeEntry {{},         {},              {},              {'l', 'z', 'h'}}, // Literary Chinese
    LanguageCodeEntry {{'l', 't'}, {'l', 'i', 't'}, {'l', 'i', 't'}, {'l', 'i', 't'}}, // Lithuanian
    LanguageCodeEntry {{},         {'j', 'b', 'o'}, {'j', 'b', 'o'}, {'j', 'b', 'o'}}, // Lojban
    LanguageCodeEntry {{},         {'d', 's', 'b'}, {'d', 's', 'b'}, {'d', 's', 'b'}}, // Lower Sorbian
    LanguageCodeEntry {{},         {'n', 'd', 's'}, {'n', 'd', 's'}, {'n', 'd', 's'}}, // Low German
    LanguageCodeEntry {{'l', 'u'}, {'l', 'u', 'b'}, {'l', 'u', 'b'}, {'l', 'u', 'b'}}, // Luba-Katanga
    LanguageCodeEntry {{},         {'s', 'm', 'j'}, {'s', 'm', 'j'}, {'s', 'm', 'j'}}, // Lule Sami
    LanguageCodeEntry {{},         {'l', 'u', 'o'}, {'l', 'u', 'o'}, {'l', 'u', 'o'}}, // Luo
    LanguageCodeEntry {{'l', 'b'}, {'l', 't', 'z'}, {'l', 't', 'z'}, {'l', 't', 'z'}}, // Luxembourgish
    LanguageCodeEntry {{},         {},              {},              {'l', 'u', 'y'}}, // Luyia
    LanguageCodeEntry {{'m', 'k'}, {'m', 'a', 'c'}, {'m', 'k', 'd'}, {'m', 'k', 'd'}}, // Macedonian
    LanguageCodeEntry {{},         {},              {},              {'j', 'm', 'c'}}, // Machame
    LanguageCodeEntry {{},         {'m', 'a', 'i'}, {'m', 'a', 'i'}, {'m', 'a', 'i'}}, // Maithili
    LanguageCodeEntry {{},         {},              {},              {'m', 'g', 'h'}}, // Makhuwa-Meetto
    LanguageCodeEntry {{},         {},              {},              {'k', 'd', 'e'}}, // Makonde
    LanguageCodeEntry {{'m', 'g'}, {'m', 'l', 'g'}, {'m', 'l', 'g'}, {'m', 'l', 'g'}}, // Malagasy
    LanguageCodeEntry {{'m', 'l'}, {'m', 'a', 'l'}, {'m', 'a', 'l'}, {'m', 'a', 'l'}}, // Malayalam
    LanguageCodeEntry {{'m', 's'}, {'m', 'a', 'y'}, {'m', 's', 'a'}, {'m', 's', 'a'}}, // Malay
    LanguageCodeEntry {{'m', 't'}, {'m', 'l', 't'}, {'m', 'l', 't'}, {'m', 'l', 't'}}, // Maltese
    LanguageCodeEntry {{},         {'m', 'a', 'n'}, {'m', 'a', 'n'}, {'m', 'a', 'n'}}, // Mandingo
    LanguageCodeEntry {{},         {'m', 'n', 'i'}, {'m', 'n', 'i'}, {'m', 'n', 'i'}}, // Manipuri
    LanguageCodeEntry {{'g', 'v'}, {'g', 'l', 'v'}, {'g', 'l', 'v'}, {'g', 'l', 'v'}}, // Manx
    LanguageCodeEntry {{'m', 'i'}, {'m', 'a', 'o'}, {'m', 'r', 'i'}, {'m', 'r', 'i'}}, // Maori
    LanguageCodeEntry {{},         {'a', 'r', 'n'}, {'a', 'r', 'n'}, {'a', 'r', 'n'}}, // Mapuche
    LanguageCodeEntry {{'m', 'r'}, {'m', 'a', 'r'}, {'m', 'a', 'r'}, {'m', 'a', 'r'}}, // Marathi
    LanguageCodeEntry {{'m', 'h'}, {'m', 'a', 'h'}, {'m', 'a', 'h'}, {'m', 'a', 'h'}}, // Marshallese
    LanguageCodeEntry {{},         {'m', 'a', 's'}, {'m', 'a', 's'}, {'m', 'a', 's'}}, // Masai
    LanguageCodeEntry {{},         {},              {},              {'m', 'z', 'n'}}, // Mazanderani
    LanguageCodeEntry {{},         {'m', 'e', 'n'}, {'m', 'e', 'n'}, {'m', 'e', 'n'}}, // Mende
    LanguageCodeEntry {{},         {},              {},              {'m', 'e', 'r'}}, // Meru
    LanguageCodeEntry {{},         {},              {},              {'m', 'g', 'o'}}, // Meta
    LanguageCodeEntry {{},         {'m', 'o', 'h'}, {'m', 'o', 'h'}, {'m', 'o', 'h'}}, // Mohawk
    LanguageCodeEntry {{'m', 'n'}, {'m', 'o', 'n'}, {'m', 'o', 'n'}, {'m', 'o', 'n'}}, // Mongolian
    LanguageCodeEntry {{},         {},              {},              {'m', 'f', 'e'}}, // Morisyen
    LanguageCodeEntry {{},         {},              {},              {'m', 'u', 'a'}}, // Mundang
    LanguageCodeEntry {{},         {'m', 'u', 's'}, {'m', 'u', 's'}, {'m', 'u', 's'}}, // Muscogee
    LanguageCodeEntry {{},         {},              {},              {'n', 'a', 'q'}}, // Nama
    LanguageCodeEntry {{'n', 'a'}, {'n', 'a', 'u'}, {'n', 'a', 'u'}, {'n', 'a', 'u'}}, // Nauru
    LanguageCodeEntry {{'n', 'v'}, {'n', 'a', 'v'}, {'n', 'a', 'v'}, {'n', 'a', 'v'}}, // Navajo
    LanguageCodeEntry {{'n', 'g'}, {'n', 'd', 'o'}, {'n', 'd', 'o'}, {'n', 'd', 'o'}}, // Ndonga
    LanguageCodeEntry {{'n', 'e'}, {'n', 'e', 'p'}, {'n', 'e', 'p'}, {'n', 'e', 'p'}}, // Nepali
    LanguageCodeEntry {{},         {'n', 'e', 'w'}, {'n', 'e', 'w'}, {'n', 'e', 'w'}}, // Newari
    LanguageCodeEntry {{},         {},              {},              {'n', 'n', 'h'}}, // Ngiemboon
    LanguageCodeEntry {{},         {},              {},              {'j', 'g', 'o'}}, // Ngomba
    LanguageCodeEntry {{},         {},              {},              {'p', 'c', 'm'}}, // Nigerian Pidgin
    LanguageCodeEntry {{},         {'n', 'q', 'o'}, {'n', 'q', 'o'}, {'n', 'q', 'o'}}, // Nko
    LanguageCodeEntry {{},         {},              {},              {'l', 'r', 'c'}}, // Northern Luri
    LanguageCodeEntry {{'s', 'e'}, {'s', 'm', 'e'}, {'s', 'm', 'e'}, {'s', 'm', 'e'}}, // Northern Sami
    LanguageCodeEntry {{},         {'n', 's', 'o'}, {'n', 's', 'o'}, {'n', 's', 'o'}}, // Northern Sotho
    LanguageCodeEntry {{'n', 'd'}, {'n', 'd', 'e'}, {'n', 'd', 'e'}, {'n', 'd', 'e'}}, // North Ndebele
    LanguageCodeEntry {{'n', 'b'}, {'n', 'o', 'b'}, {'n', 'o', 'b'}, {'n', 'o', 'b'}}, // Norwegian Bokmal
    LanguageCodeEntry {{'n', 'n'}, {'n', 'n', 'o'}, {'n', 'n', 'o'}, {'n', 'n', 'o'}}, // Norwegian Nynorsk
    LanguageCodeEntry {{},         {},              {},              {'n', 'u', 's'}}, // Nuer
    LanguageCodeEntry {{'n', 'y'}, {'n', 'y', 'a'}, {'n', 'y', 'a'}, {'n', 'y', 'a'}}, // Nyanja
    LanguageCodeEntry {{},         {'n', 'y', 'n'}, {'n', 'y', 'n'}, {'n', 'y', 'n'}}, // Nyankole
    LanguageCodeEntry {{'o', 'c'}, {'o', 'c', 'i'}, {'o', 'c', 'i'}, {'o', 'c', 'i'}}, // Occitan
    LanguageCodeEntry {{'o', 'r'}, {'o', 'r', 'i'}, {'o', 'r', 'i'}, {'o', 'r', 'i'}}, // Odia
    LanguageCodeEntry {{'o', 'j'}, {'o', 'j', 'i'}, {'o', 'j', 'i'}, {'o', 'j', 'i'}}, // Ojibwa
    LanguageCodeEntry {{},         {'s', 'g', 'a'}, {'s', 'g', 'a'}, {'s', 'g', 'a'}}, // Old Irish
    LanguageCodeEntry {{},         {'n', 'o', 'n'}, {'n', 'o', 'n'}, {'n', 'o', 'n'}}, // Old Norse
    LanguageCodeEntry {{},         {'p', 'e', 'o'}, {'p', 'e', 'o'}, {'p', 'e', 'o'}}, // Old Persian
    LanguageCodeEntry {{'o', 'm'}, {'o', 'r', 'm'}, {'o', 'r', 'm'}, {'o', 'r', 'm'}}, // Oromo
    LanguageCodeEntry {{},         {'o', 's', 'a'}, {'o', 's', 'a'}, {'o', 's', 'a'}}, // Osage
    LanguageCodeEntry {{'o', 's'}, {'o', 's', 's'}, {'o', 's', 's'}, {'o', 's', 's'}}, // Ossetic
    LanguageCodeEntry {{},         {'p', 'a', 'l'}, {'p', 'a', 'l'}, {'p', 'a', 'l'}}, // Pahlavi
    LanguageCodeEntry {{},         {'p', 'a', 'u'}, {'p', 'a', 'u'}, {'p', 'a', 'u'}}, // Palauan
    LanguageCodeEntry {{'p', 'i'}, {'p', 'l', 'i'}, {'p', 'l', 'i'}, {'p', 'l', 'i'}}, // Pali
    LanguageCodeEntry {{},         {'p', 'a', 'p'}, {'p', 'a', 'p'}, {'p', 'a', 'p'}}, // Papiamento
    LanguageCodeEntry {{'p', 's'}, {'p', 'u', 's'}, {'p', 'u', 's'}, {'p', 'u', 's'}}, // Pashto
    LanguageCodeEntry {{'f', 'a'}, {'p', 'e', 'r'}, {'f', 'a', 's'}, {'f', 'a', 's'}}, // Persian
    LanguageCodeEntry {{},         {'p', 'h', 'n'}, {'p', 'h', 'n'}, {'p', 'h', 'n'}}, // Phoenician
    LanguageCodeEntry {{'p', 'l'}, {'p', 'o', 'l'}, {'p', 'o', 'l'}, {'p', 'o', 'l'}}, // Polish
    LanguageCodeEntry {{'p', 't'}, {'p', 'o', 'r'}, {'p', 'o', 'r'}, {'p', 'o', 'r'}}, // Portuguese
    LanguageCodeEntry {{},         {},              {},              {'p', 'r', 'g'}}, // Prussian
    LanguageCodeEntry {{'p', 'a'}, {'p', 'a', 'n'}, {'p', 'a', 'n'}, {'p', 'a', 'n'}}, // Punjabi
    LanguageCodeEntry {{'q', 'u'}, {'q', 'u', 'e'}, {'q', 'u', 'e'}, {'q', 'u', 'e'}}, // Quechua
    LanguageCodeEntry {{'r', 'o'}, {'r', 'u', 'm'}, {'r', 'o', 'n'}, {'r', 'o', 'n'}}, // Romanian
    LanguageCodeEntry {{'r', 'm'}, {'r', 'o', 'h'}, {'r', 'o', 'h'}, {'r', 'o', 'h'}}, // Romansh
    LanguageCodeEntry {{},         {},              {},              {'r', 'o', 'f'}}, // Rombo
    LanguageCodeEntry {{'r', 'n'}, {'r', 'u', 'n'}, {'r', 'u', 'n'}, {'r', 'u', 'n'}}, // Rundi
    LanguageCodeEntry {{'r', 'u'}, {'r', 'u', 's'}, {'r', 'u', 's'}, {'r', 'u', 's'}}, // Russian
    LanguageCodeEntry {{},         {},              {},              {'r', 'w', 'k'}}, // Rwa
    LanguageCodeEntry {{},         {},              {},              {'s', 's', 'y'}}, // Saho
    LanguageCodeEntry {{},         {'s', 'a', 'h'}, {'s', 'a', 'h'}, {'s', 'a', 'h'}}, // Sakha
    LanguageCodeEntry {{},         {},              {},              {'s', 'a', 'q'}}, // Samburu
    LanguageCodeEntry {{'s', 'm'}, {'s', 'm', 'o'}, {'s', 'm', 'o'}, {'s', 'm', 'o'}}, // Samoan
    LanguageCodeEntry {{'s', 'g'}, {'s', 'a', 'g'}, {'s', 'a', 'g'}, {'s', 'a', 'g'}}, // Sango
    LanguageCodeEntry {{},         {},              {},              {'s', 'b', 'p'}}, // Sangu
    LanguageCodeEntry {{'s', 'a'}, {'s', 'a', 'n'}, {'s', 'a', 'n'}, {'s', 'a', 'n'}}, // Sanskrit
    LanguageCodeEntry {{},         {'s', 'a', 't'}, {'s', 'a', 't'}, {'s', 'a', 't'}}, // Santali
    LanguageCodeEntry {{'s', 'c'}, {'s', 'r', 'd'}, {'s', 'r', 'd'}, {'s', 'r', 'd'}}, // Sardinian
    LanguageCodeEntry {{},         {},              {},              {'s', 'a', 'z'}}, // Saurashtra
    LanguageCodeEntry {{},         {},              {},              {'s', 'e', 'h'}}, // Sena
    LanguageCodeEntry {{'s', 'r'}, {'s', 'r', 'p'}, {'s', 'r', 'p'}, {'s', 'r', 'p'}}, // Serbian
    LanguageCodeEntry {{},         {},              {},              {'k', 's', 'b'}}, // Shambala
    LanguageCodeEntry {{'s', 'n'}, {'s', 'n', 'a'}, {'s', 'n', 'a'}, {'s', 'n', 'a'}}, // Shona
    LanguageCodeEntry {{'i', 'i'}, {'i', 'i', 'i'}, {'i', 'i', 'i'}, {'i', 'i', 'i'}}, // Sichuan Yi
    LanguageCodeEntry {{},         {'s', 'c', 'n'}, {'s', 'c', 'n'}, {'s', 'c', 'n'}}, // Sicilian
    LanguageCodeEntry {{},         {'s', 'i', 'd'}, {'s', 'i', 'd'}, {'s', 'i', 'd'}}, // Sidamo
    LanguageCodeEntry {{},         {},              {},              {'s', 'z', 'l'}}, // Silesian
    LanguageCodeEntry {{'s', 'd'}, {'s', 'n', 'd'}, {'s', 'n', 'd'}, {'s', 'n', 'd'}}, // Sindhi
    LanguageCodeEntry {{'s', 'i'}, {'s', 'i', 'n'}, {'s', 'i', 'n'}, {'s', 'i', 'n'}}, // Sinhala
    LanguageCodeEntry {{},         {'s', 'm', 's'}, {'s', 'm', 's'}, {'s', 'm', 's'}}, // Skolt Sami
    LanguageCodeEntry {{'s', 'k'}, {'s', 'l', 'o'}, {'s', 'l', 'k'}, {'s', 'l', 'k'}}, // Slovak
    LanguageCodeEntry {{'s', 'l'}, {'s', 'l', 'v'}, {'s', 'l', 'v'}, {'s', 'l', 'v'}}, // Slovenian
    LanguageCodeEntry {{},         {},              {},              {'x', 'o', 'g'}}, // Soga
    LanguageCodeEntry {{'s', 'o'}, {'s', 'o', 'm'}, {'s', 'o', 'm'}, {'s', 'o', 'm'}}, // Somali
    LanguageCodeEntry {{},         {},              {},              {'s', 'd', 'h'}}, // Southern Kurdish
    LanguageCodeEntry {{},         {'s', 'm', 'a'}, {'s', 'm', 'a'}, {'s', 'm', 'a'}}, // Southern Sami
    LanguageCodeEntry {{'s', 't'}, {'s', 'o', 't'}, {'s', 'o', 't'}, {'s', 'o', 't'}}, // Southern Sotho
    LanguageCodeEntry {{'n', 'r'}, {'n', 'b', 'l'}, {'n', 'b', 'l'}, {'n', 'b', 'l'}}, // South Ndebele
    LanguageCodeEntry {{'e', 's'}, {'s', 'p', 'a'}, {'s', 'p', 'a'}, {'s', 'p', 'a'}}, // Spanish
    LanguageCodeEntry {{},         {'z', 'g', 'h'}, {'z', 'g', 'h'}, {'z', 'g', 'h'}}, // Standard Moroccan Tamazight
    LanguageCodeEntry {{'s', 'u'}, {'s', 'u', 'n'}, {'s', 'u', 'n'}, {'s', 'u', 'n'}}, // Sundanese
    LanguageCodeEntry {{'s', 'w'}, {'s', 'w', 'a'}, {'s', 'w', 'a'}, {'s', 'w', 'a'}}, // Swahili
    LanguageCodeEntry {{'s', 's'}, {'s', 's', 'w'}, {'s', 's', 'w'}, {'s', 's', 'w'}}, // Swati
    LanguageCodeEntry {{'s', 'v'}, {'s', 'w', 'e'}, {'s', 'w', 'e'}, {'s', 'w', 'e'}}, // Swedish
    LanguageCodeEntry {{},         {'g', 's', 'w'}, {'g', 's', 'w'}, {'g', 's', 'w'}}, // Swiss German
    LanguageCodeEntry {{},         {'s', 'y', 'r'}, {'s', 'y', 'r'}, {'s', 'y', 'r'}}, // Syriac
    LanguageCodeEntry {{},         {},              {},              {'s', 'h', 'i'}}, // Tachelhit
    LanguageCodeEntry {{'t', 'y'}, {'t', 'a', 'h'}, {'t', 'a', 'h'}, {'t', 'a', 'h'}}, // Tahitian
    LanguageCodeEntry {{},         {},              {},              {'b', 'l', 't'}}, // Tai Dam
    LanguageCodeEntry {{},         {},              {},              {'d', 'a', 'v'}}, // Taita
    LanguageCodeEntry {{'t', 'g'}, {'t', 'g', 'k'}, {'t', 'g', 'k'}, {'t', 'g', 'k'}}, // Tajik
    LanguageCodeEntry {{'t', 'a'}, {'t', 'a', 'm'}, {'t', 'a', 'm'}, {'t', 'a', 'm'}}, // Tamil
    LanguageCodeEntry {{},         {},              {},              {'t', 'r', 'v'}}, // Taroko
    LanguageCodeEntry {{},         {},              {},              {'t', 'w', 'q'}}, // Tasawaq
    LanguageCodeEntry {{'t', 't'}, {'t', 'a', 't'}, {'t', 'a', 't'}, {'t', 'a', 't'}}, // Tatar
    LanguageCodeEntry {{'t', 'e'}, {'t', 'e', 'l'}, {'t', 'e', 'l'}, {'t', 'e', 'l'}}, // Telugu
    LanguageCodeEntry {{},         {},              {},              {'t', 'e', 'o'}}, // Teso
    LanguageCodeEntry {{'t', 'h'}, {'t', 'h', 'a'}, {'t', 'h', 'a'}, {'t', 'h', 'a'}}, // Thai
    LanguageCodeEntry {{'b', 'o'}, {'t', 'i', 'b'}, {'b', 'o', 'd'}, {'b', 'o', 'd'}}, // Tibetan
    LanguageCodeEntry {{},         {'t', 'i', 'g'}, {'t', 'i', 'g'}, {'t', 'i', 'g'}}, // Tigre
    LanguageCodeEntry {{'t', 'i'}, {'t', 'i', 'r'}, {'t', 'i', 'r'}, {'t', 'i', 'r'}}, // Tigrinya
    LanguageCodeEntry {{},         {'t', 'k', 'l'}, {'t', 'k', 'l'}, {'t', 'k', 'l'}}, // Tokelau
    LanguageCodeEntry {{},         {'t', 'p', 'i'}, {'t', 'p', 'i'}, {'t', 'p', 'i'}}, // Tok Pisin
    LanguageCodeEntry {{'t', 'o'}, {'t', 'o', 'n'}, {'t', 'o', 'n'}, {'t', 'o', 'n'}}, // Tongan
    LanguageCodeEntry {{'t', 's'}, {'t', 's', 'o'}, {'t', 's', 'o'}, {'t', 's', 'o'}}, // Tsonga
    LanguageCodeEntry {{'t', 'n'}, {'t', 's', 'n'}, {'t', 's', 'n'}, {'t', 's', 'n'}}, // Tswana
    LanguageCodeEntry {{'t', 'r'}, {'t', 'u', 'r'}, {'t', 'u', 'r'}, {'t', 'u', 'r'}}, // Turkish
    LanguageCodeEntry {{'t', 'k'}, {'t', 'u', 'k'}, {'t', 'u', 'k'}, {'t', 'u', 'k'}}, // Turkmen
    LanguageCodeEntry {{},         {'t', 'v', 'l'}, {'t', 'v', 'l'}, {'t', 'v', 'l'}}, // Tuvalu
    LanguageCodeEntry {{},         {},              {},              {'k', 'c', 'g'}}, // Tyap
    LanguageCodeEntry {{},         {'u', 'g', 'a'}, {'u', 'g', 'a'}, {'u', 'g', 'a'}}, // Ugaritic
    LanguageCodeEntry {{'u', 'k'}, {'u', 'k', 'r'}, {'u', 'k', 'r'}, {'u', 'k', 'r'}}, // Ukrainian
    LanguageCodeEntry {{},         {'h', 's', 'b'}, {'h', 's', 'b'}, {'h', 's', 'b'}}, // Upper Sorbian
    LanguageCodeEntry {{'u', 'r'}, {'u', 'r', 'd'}, {'u', 'r', 'd'}, {'u', 'r', 'd'}}, // Urdu
    LanguageCodeEntry {{'u', 'g'}, {'u', 'i', 'g'}, {'u', 'i', 'g'}, {'u', 'i', 'g'}}, // Uyghur
    LanguageCodeEntry {{'u', 'z'}, {'u', 'z', 'b'}, {'u', 'z', 'b'}, {'u', 'z', 'b'}}, // Uzbek
    LanguageCodeEntry {{},         {'v', 'a', 'i'}, {'v', 'a', 'i'}, {'v', 'a', 'i'}}, // Vai
    LanguageCodeEntry {{'v', 'e'}, {'v', 'e', 'n'}, {'v', 'e', 'n'}, {'v', 'e', 'n'}}, // Venda
    LanguageCodeEntry {{'v', 'i'}, {'v', 'i', 'e'}, {'v', 'i', 'e'}, {'v', 'i', 'e'}}, // Vietnamese
    LanguageCodeEntry {{'v', 'o'}, {'v', 'o', 'l'}, {'v', 'o', 'l'}, {'v', 'o', 'l'}}, // Volapuk
    LanguageCodeEntry {{},         {},              {},              {'v', 'u', 'n'}}, // Vunjo
    LanguageCodeEntry {{'w', 'a'}, {'w', 'l', 'n'}, {'w', 'l', 'n'}, {'w', 'l', 'n'}}, // Walloon
    LanguageCodeEntry {{},         {},              {},              {'w', 'a', 'e'}}, // Walser
    LanguageCodeEntry {{},         {},              {},              {'w', 'b', 'p'}}, // Warlpiri
    LanguageCodeEntry {{'c', 'y'}, {'w', 'e', 'l'}, {'c', 'y', 'm'}, {'c', 'y', 'm'}}, // Welsh
    LanguageCodeEntry {{},         {},              {},              {'b', 'g', 'n'}}, // Western Balochi
    LanguageCodeEntry {{'f', 'y'}, {'f', 'r', 'y'}, {'f', 'r', 'y'}, {'f', 'r', 'y'}}, // Western Frisian
    LanguageCodeEntry {{},         {'w', 'a', 'l'}, {'w', 'a', 'l'}, {'w', 'a', 'l'}}, // Wolaytta
    LanguageCodeEntry {{'w', 'o'}, {'w', 'o', 'l'}, {'w', 'o', 'l'}, {'w', 'o', 'l'}}, // Wolof
    LanguageCodeEntry {{'x', 'h'}, {'x', 'h', 'o'}, {'x', 'h', 'o'}, {'x', 'h', 'o'}}, // Xhosa
    LanguageCodeEntry {{},         {},              {},              {'y', 'a', 'v'}}, // Yangben
    LanguageCodeEntry {{'y', 'i'}, {'y', 'i', 'd'}, {'y', 'i', 'd'}, {'y', 'i', 'd'}}, // Yiddish
    LanguageCodeEntry {{'y', 'o'}, {'y', 'o', 'r'}, {'y', 'o', 'r'}, {'y', 'o', 'r'}}, // Yoruba
    LanguageCodeEntry {{},         {},              {},              {'d', 'j', 'e'}}, // Zarma
    LanguageCodeEntry {{'z', 'a'}, {'z', 'h', 'a'}, {'z', 'h', 'a'}, {'z', 'h', 'a'}}, // Zhuang
    LanguageCodeEntry {{'z', 'u'}, {'z', 'u', 'l'}, {'z', 'u', 'l'}, {'z', 'u', 'l'}}, // Zulu
    LanguageCodeEntry {{},         {},              {},              {'k', 'g', 'p'}}, // Kaingang
    LanguageCodeEntry {{},         {},              {},              {'y', 'r', 'l'}}, // Nheengatu
    LanguageCodeEntry {{},         {},              {},              {'b', 'g', 'c'}}, // Haryanvi
    LanguageCodeEntry {{},         {'f', 'r', 'r'}, {'f', 'r', 'r'}, {'f', 'r', 'r'}}, // Northern Frisian
    LanguageCodeEntry {{},         {'r', 'a', 'j'}, {'r', 'a', 'j'}, {'r', 'a', 'j'}}, // Rajasthani
    LanguageCodeEntry {{},         {'m', 'd', 'f'}, {'m', 'd', 'f'}, {'m', 'd', 'f'}}, // Moksha
    LanguageCodeEntry {{},         {},              {},              {'t', 'o', 'k'}}, // Toki Pona
    LanguageCodeEntry {{},         {},              {},              {'p', 'i', 's'}}, // Pijin
    LanguageCodeEntry {{},         {},              {},              {'a', 'n', 'n'}}, // Obolo
    LanguageCodeEntry {{},         {'b', 'a', 'l'}, {'b', 'a', 'l'}, {'b', 'a', 'l'}}, // Baluchi
    LanguageCodeEntry {{},         {},              {},              {'l', 'i', 'j'}}, // Ligurian
    LanguageCodeEntry {{},         {},              {},              {'r', 'h', 'g'}}, // Rohingya
    LanguageCodeEntry {{},         {},              {},              {'t', 'r', 'w'}}, // Torwali
    LanguageCodeEntry {{},         {},              {},              {'b', 'l', 'o'}}, // Anii
    LanguageCodeEntry {{},         {},              {},              {'x', 'n', 'r'}}, // Kangri
    LanguageCodeEntry {{},         {},              {},              {'v', 'e', 'c'}}, // Venetian
    LanguageCodeEntry {{},         {},              {},              {'k', 'x', 'v'}}, // Kuvi
};

static constexpr unsigned char script_code_list[] =
"Zzzz" // AnyScript
"Adlm" // Adlam
"Ahom" // Ahom
"Hluw" // Anatolian Hieroglyphs
"Arab" // Arabic
"Armn" // Armenian
"Avst" // Avestan
"Bali" // Balinese
"Bamu" // Bamum
"Beng" // Bangla
"Bass" // Bassa Vah
"Batk" // Batak
"Bhks" // Bhaiksuki
"Bopo" // Bopomofo
"Brah" // Brahmi
"Brai" // Braille
"Bugi" // Buginese
"Buhd" // Buhid
"Cans" // Canadian Aboriginal
"Cari" // Carian
"Aghb" // Caucasian Albanian
"Cakm" // Chakma
"Cham" // Cham
"Cher" // Cherokee
"Copt" // Coptic
"Xsux" // Cuneiform
"Cprt" // Cypriot
"Cyrl" // Cyrillic
"Dsrt" // Deseret
"Deva" // Devanagari
"Dupl" // Duployan
"Egyp" // Egyptian hieroglyphs
"Elba" // Elbasan
"Ethi" // Ethiopic
"Lisu" // Fraser
"Geor" // Georgian
"Glag" // Glagolitic
"Goth" // Gothic
"Gran" // Grantha
"Grek" // Greek
"Gujr" // Gujarati
"Guru" // Gurmukhi
"Hang" // Hangul
"Hani" // Han
"Hano" // Hanunoo
"Hanb" // Han with Bopomofo
"Hatr" // Hatran
"Hebr" // Hebrew
"Hira" // Hiragana
"Armi" // Imperial Aramaic
"Phli" // Inscriptional Pahlavi
"Prti" // Inscriptional Parthian
"Jamo" // Jamo
"Jpan" // Japanese
"Java" // Javanese
"Kthi" // Kaithi
"Knda" // Kannada
"Kana" // Katakana
"Kali" // Kayah Li
"Khar" // Kharoshthi
"Khmr" // Khmer
"Khoj" // Khojki
"Sind" // Khudawadi
"Kore" // Korean
"Lana" // Lanna
"Laoo" // Lao
"Latn" // Latin
"Lepc" // Lepcha
"Limb" // Limbu
"Lina" // Linear A
"Linb" // Linear B
"Lyci" // Lycian
"Lydi" // Lydian
"Mahj" // Mahajani
"Mlym" // Malayalam
"Mand" // Mandaean
"Mani" // Manichaean
"Marc" // Marchen
"Mtei" // Meitei Mayek
"Mend" // Mende
"Merc" // Meroitic Cursive
"Mero" // Meroitic
"Modi" // Modi
"Mong" // Mongolian
"Mroo" // Mro
"Mult" // Multani
"Mymr" // Myanmar
"Nbat" // Nabataean
"Newa" // Newa
"Talu" // New Tai Lue
"Nkoo" // Nko
"Orya" // Odia
"Ogam" // Ogham
"Olck" // Ol Chiki
"Hung" // Old Hungarian
"Ital" // Old Italic
"Narb" // Old North Arabian
"Perm" // Old Permic
"Xpeo" // Old Persian
"Sarb" // Old South Arabian
"Orkh" // Orkhon
"Osge" // Osage
"Osma" // Osmanya
"Hmng" // Pahawh Hmong
"Palm" // Palmyrene
"Pauc" // Pau Cin Hau
"Phag" // Phags-pa
"Phnx" // Phoenician
"Plrd" // Pollard Phonetic
"Phlp" // Psalter Pahlavi
"Rjng" // Rejang
"Runr" // Runic
"Samr" // Samaritan
"Saur" // Saurashtra
"Shrd" // Sharada
"Shaw" // Shavian
"Sidd" // Siddham
"Sgnw" // SignWriting
"Hans" // Simplified Han
"Sinh" // Sinhala
"Sora" // Sora Sompeng
"Sund" // Sundanese
"Sylo" // Syloti Nagri
"Syrc" // Syriac
"Tglg" // Tagalog
"Tagb" // Tagbanwa
"Tale" // Tai Le
"Tavt" // Tai Viet
"Takr" // Takri
"Taml" // Tamil
"Tang" // Tangut
"Telu" // Telugu
"Thaa" // Thaana
"Thai" // Thai
"Tibt" // Tibetan
"Tfng" // Tifinagh
"Tirh" // Tirhuta
"Hant" // Traditional Han
"Ugar" // Ugaritic
"Vaii" // Vai
"Wara" // Varang Kshiti
"Yiii" // Yi
"Rohg" // Hanifi
;

static constexpr unsigned char territory_code_list[] =
"ZZ\0" // AnyTerritory
"AF\0" // Afghanistan
"AX\0" // Aland Islands
"AL\0" // Albania
"DZ\0" // Algeria
"AS\0" // American Samoa
"AD\0" // Andorra
"AO\0" // Angola
"AI\0" // Anguilla
"AQ\0" // Antarctica
"AG\0" // Antigua and Barbuda
"AR\0" // Argentina
"AM\0" // Armenia
"AW\0" // Aruba
"AC\0" // Ascension Island
"AU\0" // Australia
"AT\0" // Austria
"AZ\0" // Azerbaijan
"BS\0" // Bahamas
"BH\0" // Bahrain
"BD\0" // Bangladesh
"BB\0" // Barbados
"BY\0" // Belarus
"BE\0" // Belgium
"BZ\0" // Belize
"BJ\0" // Benin
"BM\0" // Bermuda
"BT\0" // Bhutan
"BO\0" // Bolivia
"BA\0" // Bosnia and Herzegovina
"BW\0" // Botswana
"BV\0" // Bouvet Island
"BR\0" // Brazil
"IO\0" // British Indian Ocean Territory
"VG\0" // British Virgin Islands
"BN\0" // Brunei
"BG\0" // Bulgaria
"BF\0" // Burkina Faso
"BI\0" // Burundi
"KH\0" // Cambodia
"CM\0" // Cameroon
"CA\0" // Canada
"IC\0" // Canary Islands
"CV\0" // Cape Verde
"BQ\0" // Caribbean Netherlands
"KY\0" // Cayman Islands
"CF\0" // Central African Republic
"EA\0" // Ceuta and Melilla
"TD\0" // Chad
"CL\0" // Chile
"CN\0" // China
"CX\0" // Christmas Island
"CP\0" // Clipperton Island
"CC\0" // Cocos Islands
"CO\0" // Colombia
"KM\0" // Comoros
"CG\0" // Congo - Brazzaville
"CD\0" // Congo - Kinshasa
"CK\0" // Cook Islands
"CR\0" // Costa Rica
"HR\0" // Croatia
"CU\0" // Cuba
"CW\0" // Curacao
"CY\0" // Cyprus
"CZ\0" // Czechia
"DK\0" // Denmark
"DG\0" // Diego Garcia
"DJ\0" // Djibouti
"DM\0" // Dominica
"DO\0" // Dominican Republic
"EC\0" // Ecuador
"EG\0" // Egypt
"SV\0" // El Salvador
"GQ\0" // Equatorial Guinea
"ER\0" // Eritrea
"EE\0" // Estonia
"SZ\0" // Eswatini
"ET\0" // Ethiopia
"150" // Europe
"EU\0" // European Union
"FK\0" // Falkland Islands
"FO\0" // Faroe Islands
"FJ\0" // Fiji
"FI\0" // Finland
"FR\0" // France
"GF\0" // French Guiana
"PF\0" // French Polynesia
"TF\0" // French Southern Territories
"GA\0" // Gabon
"GM\0" // Gambia
"GE\0" // Georgia
"DE\0" // Germany
"GH\0" // Ghana
"GI\0" // Gibraltar
"GR\0" // Greece
"GL\0" // Greenland
"GD\0" // Grenada
"GP\0" // Guadeloupe
"GU\0" // Guam
"GT\0" // Guatemala
"GG\0" // Guernsey
"GW\0" // Guinea-Bissau
"GN\0" // Guinea
"GY\0" // Guyana
"HT\0" // Haiti
"HM\0" // Heard and McDonald Islands
"HN\0" // Honduras
"HK\0" // Hong Kong
"HU\0" // Hungary
"IS\0" // Iceland
"IN\0" // India
"ID\0" // Indonesia
"IR\0" // Iran
"IQ\0" // Iraq
"IE\0" // Ireland
"IM\0" // Isle of Man
"IL\0" // Israel
"IT\0" // Italy
"CI\0" // Ivory Coast
"JM\0" // Jamaica
"JP\0" // Japan
"JE\0" // Jersey
"JO\0" // Jordan
"KZ\0" // Kazakhstan
"KE\0" // Kenya
"KI\0" // Kiribati
"XK\0" // Kosovo
"KW\0" // Kuwait
"KG\0" // Kyrgyzstan
"LA\0" // Laos
"419" // Latin America
"LV\0" // Latvia
"LB\0" // Lebanon
"LS\0" // Lesotho
"LR\0" // Liberia
"LY\0" // Libya
"LI\0" // Liechtenstein
"LT\0" // Lithuania
"LU\0" // Luxembourg
"MO\0" // Macao
"MK\0" // Macedonia
"MG\0" // Madagascar
"MW\0" // Malawi
"MY\0" // Malaysia
"MV\0" // Maldives
"ML\0" // Mali
"MT\0" // Malta
"MH\0" // Marshall Islands
"MQ\0" // Martinique
"MR\0" // Mauritania
"MU\0" // Mauritius
"YT\0" // Mayotte
"MX\0" // Mexico
"FM\0" // Micronesia
"MD\0" // Moldova
"MC\0" // Monaco
"MN\0" // Mongolia
"ME\0" // Montenegro
"MS\0" // Montserrat
"MA\0" // Morocco
"MZ\0" // Mozambique
"MM\0" // Myanmar
"NA\0" // Namibia
"NR\0" // Nauru
"NP\0" // Nepal
"NL\0" // Netherlands
"NC\0" // New Caledonia
"NZ\0" // New Zealand
"NI\0" // Nicaragua
"NG\0" // Nigeria
"NE\0" // Niger
"NU\0" // Niue
"NF\0" // Norfolk Island
"MP\0" // Northern Mariana Islands
"KP\0" // North Korea
"NO\0" // Norway
"OM\0" // Oman
"QO\0" // Outlying Oceania
"PK\0" // Pakistan
"PW\0" // Palau
"PS\0" // Palestinian Territories
"PA\0" // Panama
"PG\0" // Papua New Guinea
"PY\0" // Paraguay
"PE\0" // Peru
"PH\0" // Philippines
"PN\0" // Pitcairn
"PL\0" // Poland
"PT\0" // Portugal
"PR\0" // Puerto Rico
"QA\0" // Qatar
"RE\0" // Reunion
"RO\0" // Romania
"RU\0" // Russia
"RW\0" // Rwanda
"BL\0" // Saint Barthelemy
"SH\0" // Saint Helena
"KN\0" // Saint Kitts and Nevis
"LC\0" // Saint Lucia
"MF\0" // Saint Martin
"PM\0" // Saint Pierre and Miquelon
"VC\0" // Saint Vincent and Grenadines
"WS\0" // Samoa
"SM\0" // San Marino
"ST\0" // Sao Tome and Principe
"SA\0" // Saudi Arabia
"SN\0" // Senegal
"RS\0" // Serbia
"SC\0" // Seychelles
"SL\0" // Sierra Leone
"SG\0" // Singapore
"SX\0" // Sint Maarten
"SK\0" // Slovakia
"SI\0" // Slovenia
"SB\0" // Solomon Islands
"SO\0" // Somalia
"ZA\0" // South Africa
"GS\0" // South Georgia and South Sandwich Islands
"KR\0" // South Korea
"SS\0" // South Sudan
"ES\0" // Spain
"LK\0" // Sri Lanka
"SD\0" // Sudan
"SR\0" // Suriname
"SJ\0" // Svalbard and Jan Mayen
"SE\0" // Sweden
"CH\0" // Switzerland
"SY\0" // Syria
"TW\0" // Taiwan
"TJ\0" // Tajikistan
"TZ\0" // Tanzania
"TH\0" // Thailand
"TL\0" // Timor-Leste
"TG\0" // Togo
"TK\0" // Tokelau
"TO\0" // Tonga
"TT\0" // Trinidad and Tobago
"TA\0" // Tristan da Cunha
"TN\0" // Tunisia
"TR\0" // Turkey
"TM\0" // Turkmenistan
"TC\0" // Turks and Caicos Islands
"TV\0" // Tuvalu
"UG\0" // Uganda
"UA\0" // Ukraine
"AE\0" // United Arab Emirates
"GB\0" // United Kingdom
"UM\0" // United States Outlying Islands
"US\0" // United States
"VI\0" // United States Virgin Islands
"UY\0" // Uruguay
"UZ\0" // Uzbekistan
"VU\0" // Vanuatu
"VA\0" // Vatican City
"VE\0" // Venezuela
"VN\0" // Vietnam
"WF\0" // Wallis and Futuna
"EH\0" // Western Sahara
"001" // world
"YE\0" // Yemen
"ZM\0" // Zambia
"ZW\0" // Zimbabwe
;

// GENERATED PART ENDS HERE

QT_WARNING_POP // QTBUG-128930

QT_END_NAMESPACE

#endif
